// vim:expandtab:shiftwidth=2:tabstop=2:
// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "web_contents_view_oxide.h"

#include "base/logging.h"
#include "content/browser/web_contents/web_contents_impl.h"
#include "content/public/common/screen_info.h"
#include "content/public/common/screen_orientation_values.h"
#include "ui/display/display.h"
#include "ui/display/screen.h"


namespace content {

namespace {
WebContentsViewOxideFactory* g_factory;
}

WebContentsViewOxide::~WebContentsViewOxide() = default;

// static
void WebContentsViewOxide::GetScreenInfoForDisplay(
    const display::Display& display,
    ScreenInfo* result) {
  result->device_scale_factor = display.device_scale_factor();
  result->depth = 24;
  result->depth_per_component = 8;
  result->is_monochrome = false;
  result->rect = display.bounds();
  result->available_rect = display.work_area();

  int rotation = display.RotationAsDegree();

  // The rotation angle of display::Display is the clockwise screen rotation,
  // whereas the orientationAngle of blink::WebScreenInfo is the clockwise
  // content rotation
  result->orientation_angle = rotation;
  if (result->orientation_angle == 90) {
    result->orientation_angle = 270;
  } else if (result->orientation_angle == 270) {
    result->orientation_angle = 90;
  }

  gfx::Rect bounds = display.bounds();

  bool naturally_portrait;
  if (rotation == 0 || rotation == 180)
    naturally_portrait = bounds.width() <= bounds.height();
  else
    naturally_portrait = bounds.width() >= bounds.height();

  switch (rotation) {
    case 0:
      result->orientation_type =
          naturally_portrait ?
              SCREEN_ORIENTATION_VALUES_PORTRAIT_PRIMARY
              : SCREEN_ORIENTATION_VALUES_LANDSCAPE_PRIMARY;
      break;
    case 90:
      result->orientation_type =
          naturally_portrait ?
              SCREEN_ORIENTATION_VALUES_LANDSCAPE_SECONDARY
              : SCREEN_ORIENTATION_VALUES_PORTRAIT_PRIMARY;
      break;
    case 180:
      result->orientation_type =
          naturally_portrait ?
              SCREEN_ORIENTATION_VALUES_PORTRAIT_SECONDARY
              : SCREEN_ORIENTATION_VALUES_LANDSCAPE_SECONDARY;
      break;
    case 270:
      result->orientation_type =
          naturally_portrait ?
              SCREEN_ORIENTATION_VALUES_LANDSCAPE_PRIMARY
              : SCREEN_ORIENTATION_VALUES_PORTRAIT_SECONDARY;
      break;
  }
}

void SetWebContentsViewOxideFactory(WebContentsViewOxideFactory* factory) {
  g_factory = factory;
}

WebContentsView* CreateWebContentsView(
    WebContentsImpl* web_contents,
    WebContentsViewDelegate* delegate,
    RenderViewHostDelegateView** render_view_host_delegate_view) {
  DCHECK(g_factory);

  WebContentsViewOxide* rv = g_factory(web_contents);
  DCHECK(rv);

  *render_view_host_delegate_view = rv;
  return rv;
}

void WebContentsView::GetDefaultScreenInfo(ScreenInfo* result) {
  WebContentsViewOxide::GetScreenInfoForDisplay(
      display::Screen::GetScreen()->GetPrimaryDisplay(),
      result);
}

} // namespace content
