/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __QZIONIMAGE_H__
#define __QZIONIMAGE_H__

#include "qzionobject.h"

struct QZionImageBorder
{
    int left;
    int right;
    int top;
    int bottom;
    QZionImageBorder(): left(0), right(0), top(0), bottom(0) {};
    QZionImageBorder(const int left, const int right,
                     const int top, const int bottom)
        : left(left), right(right), top(top), bottom(bottom) {};
};


class QZionTiledPixmap
{

private:
    bool _is_null;
    QPixmap tiles[9];
    QZionImageBorder border;

public:
    enum qzTilePosition{
        NORTH_WEST,
        NORTH,
        NORTH_EAST,
        WEST,
        MIDDLE,
        EAST,
        SOUTH_WEST,
        SOUTH,
        SOUTH_EAST
    };

   // Start an empty TiledPixmap
    QZionTiledPixmap() : _is_null(true) {};

    QZionTiledPixmap(const QPixmap pixmap, QZionImageBorder border);

    QRect tileRectAtSize(const qzTilePosition num, QSize size);

    QPixmap tile(const qzTilePosition num) {return tiles[num];};

    bool isNull() {return _is_null;};

};


class QZionImagePrivate;

/*!
    \class QZionImage
    \brief A pixmap that uses internal image cache.

    A QZionImage is a pixmap that can be put in the canvas.
*/
class QZionImage : public QZionObject
{
    Q_OBJECT

public:
    /*!
        Constructs a QZionImage with empty pixmap.
    */
    QZionImage(QZionAbstractCanvas *canvas = NULL);

    /*!
        Constructs a QZionImage with specified image.
    */
    QZionImage(QZionAbstractCanvas *canvas, const QString &filepath);

    /*!
        Constructs a QZionImage with specified QPixmap.
    */
    QZionImage(QZionAbstractCanvas *canvas, const QPixmap &pixmap);

    /*!
        Constructs a QZionImage with specified QImage.
    */
    QZionImage(QZionAbstractCanvas *canvas, const QImage &image);

   ~QZionImage();

    QPixmap pixmap() const;
    /*!
      Load an image from a file.

        \code
            QZionImage im(canvas);
            im.load("icon.png");
        \endcode
    */
    bool load(const QString &filename);

    /*!
        Reload the image (force cache update)
    */
    bool reload();

    /*!
        Clears the contents of the image pixmap to null.
    */
    void clear();

    virtual QSize size() const;

    /*!
        Sets image size.
    */
    virtual void setSize(const QSize &size);

    /*!
        Paint image on canvas.
    */
    virtual void paint(QPainter *p);
    virtual QRect rect() const;

    virtual QColor color() const;
    virtual void setColor(const QColor &color);

    virtual void changed();

    void setBorder(const int left, const int right,
                   const int top, const int bottom);

    void setBorder(QZionImageBorder border);
    QZionImageBorder border();

protected:
    QZionImagePrivate *_QZionImage_data;

    QString _filepath;

    /*!
        Sets pixmap and mark image as changed.
    */
    void setPixmap(const QPixmap &pixmap);

    friend class QZionImagePrivate;
};

#endif
