//msg.h:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2013
 *
 *  This file is part of aroarfw, a RoarAudio framework for
 *  embedded systems (Controlers).
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  aroarfw is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef _AROARFW_MSG_H_
#define _AROARFW_MSG_H_

#include <aroarfw/types.h>
#include <aroarfw/proto.h>
#include <aroarfw/byteorder.h>

//! Maximum length of version 0 message header
#define RMSG_V0_MAX_HEADER_LEN 10

//! Maximum length of version 1 message header
#define RMSG_V1_MAX_HEADER_LEN 11

//! Maximum length of version 2 message header
#define RMSG_V2_MAX_HEADER_LEN 20

//! RoarAudio Message version 0 type
typedef unsigned char rmsg_v0_t[RMSG_V0_MAX_HEADER_LEN];

//! RoarAudio Message version 1 type
typedef unsigned char rmsg_v1_t[RMSG_V1_MAX_HEADER_LEN];

//! RoarAudio Message version 2 type
typedef unsigned char rmsg_v2_t[RMSG_V2_MAX_HEADER_LEN];

//! RoarAudio Message type, version independed.
typedef union {
 //! Version of message
 unsigned char version;

 //! Version 0 message
 rmsg_v0_t v0;

 //! Version 1 message
 rmsg_v1_t v1;

 //! Version 2 message
 rmsg_v2_t v2;
} rmsg_t;

/** \brief Create a RoarAudio version 0 message
 \param msg The Message object to create
 \param cmd The command for the new message
*/
//void   rmsg0_init  (rmsg_v0_t msg, rcmd_t cmd);
#define rmsg0_init(msg,cmd) ((msg)[1]=                \
                             ((msg)[0]=               \
                              (msg)[2]=(msg)[3]=      \
                              (msg)[4]=(msg)[5]=      \
                              (msg)[6]=(msg)[7]=      \
                              (msg)[8]=(msg)[9]= 0) + \
                             (cmd))

/** \brief Get the command of a version 0 RoarAudio message
 \param msg The Message object to be used
 \return The command
*/
#define rmsg0_getcmd(msg) (((unsigned char*)(msg))[1])

/** \brief Set the data length of a version 0 RoarAudio message
 \param msg The Message object to be used
 \param len The data length
*/
//void   rmsg0_setlen(rmsg_v0_t msg, size_t len);
#define rmsg0_setlen(msg, len) ((void)RBO_H2NB16(&(((unsigned char*)(msg))[8]), (len)))

/** \brief Get the data length of a version 0 RoarAudio message
 \param msg The Message object to be used
 \return The data length
*/
//size_t rmsg0_getlen(rmsg_v0_t msg);
#define rmsg0_getlen(msg) RBO_NB2H16(&(((unsigned char*)(msg))[8]))

/** \brief Set the stream ID of a version 0 RoarAudio message
 \param msg The Message object to be used
 \param sid The stream ID
*/
//void   rmsg0_setsid(rmsg_v0_t msg, rsid_t sid);
#define rmsg0_setsid(msg, sid) ((void)RBO_H2NB16(&(((unsigned char*)(msg))[2]), (sid)))

/** \brief Get the stream ID of a version 0 RoarAudio message
 \param msg The Message object to be used
 \return The stream ID
*/
//rsid_t rmsg0_getsid(rmsg_v0_t msg);
#define rmsg0_getsid(msg) RBO_NB2H16(&(((unsigned char*)(msg))[2]))

/** \brief Set the stream possition of a version 0 RoarAudio message
 \param msg The Message object to be used
 \param pos The stream possition
*/
//void   rmsg0_setpos(rmsg_v0_t msg, rpos_t pos);
#define rmsg0_setpos(msg, pos) ((void)RBO_H2NB32(&(((unsigned char*)(msg))[4]), (pos)))

/** \brief Get the stream possition of a version 0 RoarAudio message
 \param msg The Message object to be used
 \return The stream possition
*/
//rpos_t rmsg0_getpos(rmsg_v0_t msg);
#define rmsg0_getpos(msg) RBO_NB2H16(&(((unsigned char*)(msg))[4]))

/** \brief Get the header length of a version 0 RoarAudio message

 This function is used to get the length of the message header.

 Use This function togeter with rmsg0_header().

 \param msg The Message object to be used
 \return The header length
*/
//size_t rmsg0_len   (rmsg_v0_t msg);
#define rmsg0_len(msg) RMSG_V0_MAX_HEADER_LEN

/** \brief Get the header data of a version 0 RoarAudio message

 This function is used to get a pointer to a ready-to-send
 version of the message header.

 Use This function togeter with rmsg0_header().

 \param msg The Message object to be used
 \return The pointer to the header
*/
#define rmsg0_header(msg) (msg)


// Message version independed:

/** \brief Get the version of a RoarAudio message

 This function returns the version number of the message buffer provided.

 \param msg The Message object to be used
 \return The version number
*/
#define rmsg_version(msg) (((unsigned char*)msg)[0])

#endif

//ll
