/***************************************************************
 *
 * Copyright (C) 1990-2007, Condor Team, Computer Sciences Department,
 * University of Wisconsin-Madison, WI.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License.  You may
 * obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***************************************************************/


#ifndef CONDOR_AUTHENTICATOR_GSS
#define CONDOR_AUTHENTICATOR_GSS

#if !defined(SKIP_AUTHENTICATION) && defined(HAVE_EXT_GLOBUS)

#include "condor_auth.h"        // Condor_Auth_Base class is defined here
// In order to avoid a warning generated by including the Globus header files
// we undefine IOV_MAX. The Globus header files will redefine it for us. 
// Not only do we not use IOV_MAX in the source code, but the redefinition 
// is going to be used anyway, so we might as well get rid of the warning.
#undef IOV_MAX 
#include "globus_gss_assist.h"

#include "classad/classad_stl.h"

const char STR_GSI_DAEMON_DIRECTORY[] = "GSI_DAEMON_DIRECTORY";
const char STR_GSI_DAEMON_PROXY[]     = "GSI_DAEMON_PROXY";
const char STR_GSI_DAEMON_CERT[]      = "GSI_DAEMON_CERT";
const char STR_GSI_DAEMON_KEY[]       = "GSI_DAEMON_KEY";
const char STR_GSI_DAEMON_TRUSTED_CA_DIR[]       = "GSI_DAEMON_TRUSTED_CA_DIR";
const char STR_GSI_USER_PROXY[]       = "X509_USER_PROXY";
const char STR_GSI_CERT_DIR[]         = "X509_CERT_DIR";
const char STR_GSI_USER_CERT[]        = "X509_USER_CERT";
const char STR_GSI_USER_KEY[]         = "X509_USER_KEY";
const char STR_SSLEAY_CONF[]          = "SSLEAY_CONF";
const char STR_GSI_MAPFILE[]          = "GRIDMAP";

class Condor_Auth_X509 : public Condor_Auth_Base {
 public:
    Condor_Auth_X509(ReliSock * sock);
    //------------------------------------------
    // Constructor
    //------------------------------------------

    ~Condor_Auth_X509();
    //------------------------------------------
    // Destructor
    //------------------------------------------

    int authenticate(const char * remoteHost, CondorError* errstack, bool non_blocking);
    //------------------------------------------
    // PURPOSE: authenticate with the other side 
    // REQUIRE: hostAddr -- host to authenticate
    // RETURNS:
    //------------------------------------------

    int wrap(char* input, int input_len, char*& output, int& output_len);
    //------------------------------------------
    // PURPOSE: Wrap the buffer
    // REQUIRE: 
    // RETUNRS: TRUE -- success, FALSE -- falure
    //          May need more code later on
    //------------------------------------------
    
    int unwrap(char* input, int input_len, char*& output, int& output_len);
    //------------------------------------------
    // PURPOSE: Unwrap the buffer
    // REQUIRE: 
    // RETURNS: TRUE -- success, FALSE -- failure
    //------------------------------------------

    int nameGssToLocal( const char * GssClient );

    int isValid() const;
    //------------------------------------------
    // PURPOSE: whether the authenticator is in
    //          valid state.
    // REQUIRE: None
    // RETURNS: true or false
    //------------------------------------------

     int endTime() const;
    //------------------------------------------
    // PURPOSE: Return the expiration time for
    //          kerberos
    // REQUIRE: None
    // RETURNS: -1 -- invalid
    //          >0 -- expiration time
    //------------------------------------------

	void setFQAN(const char *fqan);
	const char *getFQAN();

 private:

	enum CondorAuthX509State {
		GetClientPre = 100,
		GSSAuth,
		GetClientPost
	};
	enum CondorAuthX509Retval {
		Fail = 0,
		Success,
		WouldBlock,
		Continue
	};

    int authenticate_self_gss(CondorError* errstack);

    int authenticate_client_gss(CondorError* errstack);

    CondorAuthX509Retval authenticate_server_gss(CondorError* errstack, bool non_blocking);
    CondorAuthX509Retval authenticate_server_pre(CondorError* errstack, bool non_blocking);
    CondorAuthX509Retval authenticate_server_gss_complete(OM_uint32, OM_uint32, CondorError* errstack);
    CondorAuthX509Retval authenticate_server_gss_post(CondorError* errstack, bool non_blocking);
	int authenticate_continue(CondorError* /*errstack*/, bool /*non_blocking*/);

    char * get_server_info();

#ifdef WIN32
	int ParseMapFile();
	int condor_gss_assist_gridmap ( char * GssClient, char *& local_user );
#endif

    /** Check whether the security context of the scoket is valid or not 
	@return TRUE if valid FALSE if not */
    bool gss_is_valid();

	/** See if host name in peer's certificate matches what we
		are connecting to.
		@return true if valid; false if not
	 */
	bool CheckServerName(char const *fqh,char const *ip,ReliSock *sock,CondorError *errstack);

    /** A specialized function that is needed for secure personal
        condor. When schedd and the user are running under the
        same userid we would still want the authentication process
        be transperant. This means that we have to maintain to
        distinct environments to processes running under the same
        id. erase\_env ensures that the X509\_USER\_PROXY which
        should \bf{NOT} be set for daemon processes is wiped out
        even if it is set, so that a daemon process does not start
        looking for a proxy which it does not need. */
    void  erase_env();
    
    void  print_log(OM_uint32 ,OM_uint32 ,int , const char*);

    //------------------------------------------
    // Data (private)
    //------------------------------------------
    gss_cred_id_t       credential_handle;
    gss_ctx_id_t        context_handle ;
    gss_name_t          m_gss_server_name;

	// These are used during the globus auth on the server side.
	gss_name_t		m_client_name;
	int			token_status;
	OM_uint32		ret_flags;
	CondorAuthX509State	m_state;
	int			m_status;

	std::string			m_fqan;
#ifdef WIN32
    typedef HashTable<MyString, MyString> Grid_Map_t;
    static Grid_Map_t * GridMap;
#endif
	static bool m_globusActivated;

	typedef struct globus_mapping_entry_s {
		std::string name;
		time_t expiry_time;
	} globus_mapping_entry_t;
	typedef classad_shared_ptr<globus_mapping_entry_t> globus_mapping_entry_ptr;
	typedef HashTable<std::string, globus_mapping_entry_ptr> GlobusMappingTable;
	static GlobusMappingTable *m_mapping;

};

#endif

#endif
