!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2021 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief Routines to calculate the minimax coefficients in order to
!>        approximate 1/x as a sum over exponential functions
!>        1/x ~ SUM_{i}^{K} w_i EXP(-a_i * x) for x belonging to [1:Rc].
!>        This module contains coefficients for minimax approximations with 1 <= k <= 15.
!> \par History
!>      11.2012 created [Mauro Del Ben]
!>      02.2015 changed module name minimax to minimax_k15 to replace this module
!>              with new minimax module, added function to get minimax error
!>              [Patrick Seewald]
! **************************************************************************************************
MODULE minimax_exp_k15
   USE kinds,                           ONLY: dp
#include "../base/base_uses.f90"

   IMPLICIT NONE

   PRIVATE

   CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'minimax_exp_k15'

   PUBLIC :: get_minimax_coeff_k15, check_range_k15, get_minimax_numerical_error

CONTAINS

! **************************************************************************************************
!> \brief Check that the range for the minimax approximation is not too small
!>        for the chosen number of integration point, or that the number of
!>        integration point is not exceeding the maximum allowed by the actual
!>        implementation.
!> \param k ...
!> \param Rc ...
!> \param ierr ...
!> \author Mauro Del Ben
! **************************************************************************************************
   SUBROUTINE check_range_k15(k, Rc, ierr)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: Rc
      INTEGER                                            :: ierr

      REAL(KIND=dp)                                      :: R_min

      ierr = 0
      R_min = 0.0_dp

      SELECT CASE (k)
      CASE (1)
         R_min = 2.0000000000000000E+00_dp
      CASE (2)
         R_min = 2.0000000000000000E+00_dp
      CASE (3)
         R_min = 2.0000000000000000E+00_dp
      CASE (4)
         R_min = 3.0000000000000000E+00_dp
      CASE (5)
         R_min = 4.0000000000000000E+00_dp
      CASE (6)
         R_min = 5.0000000000000000E+00_dp
      CASE (7)
         R_min = 7.0000000000000000E+00_dp
      CASE (8)
         R_min = 1.0000000000000000E+01_dp
      CASE (9)
         R_min = 1.0000000000000000E+01_dp
      CASE (10)
         R_min = 1.0000000000000000E+01_dp
      CASE (11)
         R_min = 1.0000000000000000E+01_dp
      CASE (12)
         R_min = 1.0000000000000000E+01_dp
      CASE (13)
         R_min = 1.0000000000000000E+01_dp
      CASE (14)
         R_min = 1.0000000000000000E+01_dp
      CASE (15)
         R_min = 2.0000000000000000E+01_dp
      CASE DEFAULT
         ierr = -1
      END SELECT

      IF (Rc < R_min .AND. ierr == 0) ierr = 1

   END SUBROUTINE check_range_k15

! **************************************************************************************************
!> \brief Sample numerical error and return its maximum.
!> \param Rc ...
!> \param aw ...
!> \return ...
!> \author Patrick Seewald
! **************************************************************************************************
   FUNCTION get_minimax_numerical_error(Rc, aw) RESULT(numerr)
      REAL(KIND=dp), INTENT(IN)                          :: Rc
      REAL(KIND=dp), DIMENSION(:), INTENT(IN)            :: aw
      REAL(KIND=dp)                                      :: numerr

      INTEGER, PARAMETER                                 :: npoints = 100

      INTEGER                                            :: j, k, np
      REAL(KIND=dp)                                      :: mm, x

      k = SIZE(aw)/2
      numerr = 0.0_dp
      DO np = 1, npoints
         x = 1.0_dp + (Rc - 1.0_dp)*(npoints - np)/(npoints - 1)
         mm = 0.0_dp
         DO j = 1, k
            mm = mm + aw(j + k)*EXP(-aw(j)*x)
         END DO
         numerr = MAX(numerr, ABS(mm - 1.0_dp/x))
      END DO

   END FUNCTION get_minimax_numerical_error

! **************************************************************************************************
!> \brief Get minimax coefficients: k15 implementation (coefficients up to k=15 terms).
!>        All a_i and w_i have been fitted with a 12th order polynomial as
!>        a function of Rc for each value of K.
!> \param k ...
!> \param Rc ...
!> \param aw ...
!> \param mm_error ...
!> \author Mauro Del Ben
! **************************************************************************************************
   SUBROUTINE get_minimax_coeff_k15(k, Rc, aw, mm_error)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: Rc
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw
      REAL(KIND=dp), INTENT(OUT), OPTIONAL               :: mm_error

      CHARACTER(LEN=*), PARAMETER :: routineN = 'get_minimax_coeff_k15'

      INTEGER                                            :: handle, icoef, kkk
      REAL(KIND=dp)                                      :: L_b, R_max, R_min, R_transf, U_b
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef

      CALL timeset(routineN, handle)
      R_min = 0.0_dp
      R_max = 0.0_dp

      SELECT CASE (k)
      CASE (1)
         R_min = 2.0000000000000000E+00_dp
         R_max = 8.6669999999999998E+00_dp
      CASE (2)
         R_min = 2.0000000000000000E+00_dp
         R_max = 4.1539999999999999E+01_dp
      CASE (3)
         R_min = 2.0000000000000000E+00_dp
         R_max = 1.4680000000000001E+02_dp
      CASE (4)
         R_min = 3.0000000000000000E+00_dp
         R_max = 4.3610000000000002E+02_dp
      CASE (5)
         R_min = 4.0000000000000000E+00_dp
         R_max = 1.1540000000000000E+03_dp
      CASE (6)
         R_min = 5.0000000000000000E+00_dp
         R_max = 2.8070000000000000E+03_dp
      CASE (7)
         R_min = 7.0000000000000000E+00_dp
         R_max = 6.3730000000000000E+03_dp
      CASE (8)
         R_min = 1.0000000000000000E+01_dp
         R_max = 1.3749000000000000E+04_dp
      CASE (9)
         R_min = 1.0000000000000000E+01_dp
         R_max = 2.8387000000000000E+04_dp
      CASE (10)
         R_min = 1.0000000000000000E+01_dp
         R_max = 5.6502000000000000E+04_dp
      CASE (11)
         R_min = 1.0000000000000000E+01_dp
         R_max = 1.0890000000000000E+05_dp
      CASE (12)
         R_min = 1.0000000000000000E+01_dp
         R_max = 2.0420000000000000E+05_dp
      CASE (13)
         R_min = 1.0000000000000000E+01_dp
         R_max = 3.7370000000000000E+05_dp
      CASE (14)
         R_min = 1.0000000000000000E+01_dp
         R_max = 6.6910000000000000E+05_dp
      CASE (15)
         R_min = 2.0000000000000000E+01_dp
         R_max = 1.1750000000000000E+06_dp
      CASE DEFAULT
         CPABORT("")
      END SELECT

      fit_coef = 0.0
      aw = 0.0
      IF (Rc < R_min) THEN
         CPABORT("")
      ELSE
         SELECT CASE (k)
         CASE (1)
            IF (Rc >= R_max) THEN
               aw(1) = 4.4649260629947779E-01_dp
               aw(2) = 1.4290997869805788E+00_dp
            ELSE
               CALL get_fit_coef_k1(k, fit_coef, L_b, U_b)
            END IF
         CASE (2)
            IF (Rc >= R_max) THEN
               aw(1) = 9.3106172568836743E-02_dp
               aw(2) = 1.0301539335400713E+00_dp
               aw(3) = 2.9765554323276328E-01_dp
               aw(4) = 1.9917443679019897E+00_dp
            ELSE
               CALL get_fit_coef_k2(k, fit_coef, L_b, U_b)
            END IF
         CASE (3)
            IF (Rc >= R_max) THEN
               aw(1) = 2.6349272848143715E-02_dp
               aw(2) = 2.8996855484815137E-01_dp
               aw(3) = 1.5533661852788561E+00_dp
               aw(4) = 8.4233696465358193E-02_dp
               aw(5) = 5.5403643315333395E-01_dp
               aw(6) = 2.3557688429224690E+00_dp
            ELSE
               CALL get_fit_coef_k3(k, fit_coef, L_b, U_b)
            END IF
         CASE (4)
            IF (Rc >= R_max) THEN
               aw(1) = 8.8685367946663256E-03_dp
               aw(2) = 9.7565835027097439E-02_dp
               aw(3) = 5.1691899901700877E-01_dp
               aw(4) = 2.0308311260118175E+00_dp
               aw(5) = 2.8350982414551848E-02_dp
               aw(6) = 1.8628933861057068E-01_dp
               aw(7) = 7.6510507307920383E-01_dp
               aw(8) = 2.6286812528458148E+00_dp
            ELSE
               CALL get_fit_coef_k4(k, fit_coef, L_b, U_b)
            END IF
         CASE (5)
            IF (Rc >= R_max) THEN
               aw(1) = 3.3529112764591387E-03_dp
               aw(2) = 3.6885523808630270E-02_dp
               aw(3) = 1.9524288957651931E-01_dp
               aw(4) = 7.5508129953934144E-01_dp
               aw(5) = 2.4730352459564653E+00_dp
               aw(6) = 1.0718601062497496E-02_dp
               aw(7) = 7.0423935703868623E-02_dp
               aw(8) = 2.8840120533084923E-01_dp
               aw(9) = 9.4333101049811663E-01_dp
               aw(10) = 2.8485313369244247E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k5_sh(k, fit_coef, L_b, U_b)
               ELSE
                  CALL get_fit_coef_k5_lo(k, fit_coef, L_b, U_b)
               END IF
            END IF
         CASE (6)
            IF (Rc >= R_max) THEN
               aw(1) = 1.3802019452414500E-03_dp
               aw(2) = 1.5183614381572768E-02_dp
               aw(3) = 8.0361511855253240E-02_dp
               aw(4) = 3.1025560863277402E-01_dp
               aw(5) = 9.9661148626887119E-01_dp
               aw(6) = 2.8871396539466740E+00_dp
               aw(7) = 4.4122353526579831E-03_dp
               aw(8) = 2.8989220321940696E-02_dp
               aw(9) = 1.1867791139299246E-01_dp
               aw(10) = 3.8613951254853507E-01_dp
               aw(11) = 1.0973265611814316E+00_dp
               aw(12) = 3.0334106111925840E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k6_sh(k, fit_coef, L_b, U_b)
               ELSE
                  CALL get_fit_coef_k6_lo(k, fit_coef, L_b, U_b)
               END IF
            END IF
         CASE (7)
            IF (Rc >= R_max) THEN
               aw(1) = 6.0680046054254941E-04_dp
               aw(2) = 6.6754145503086018E-03_dp
               aw(3) = 3.5330085311658307E-02_dp
               aw(4) = 1.3636790644450009E-01_dp
               aw(5) = 4.3692597185507065E-01_dp
               aw(6) = 1.2378541669818244E+00_dp
               aw(7) = 3.2781922981409610E+00_dp
               aw(8) = 1.9398222440315960E-03_dp
               aw(9) = 1.2744981074293062E-02_dp
               aw(10) = 5.2173800858709714E-02_dp
               aw(11) = 1.6963212116290538E-01_dp
               aw(12) = 4.7830405076036075E-01_dp
               aw(13) = 1.2328765453712285E+00_dp
               aw(14) = 3.1933984156189350E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k7_sh(k, fit_coef, L_b, U_b)
               ELSE
                  CALL get_fit_coef_k7_lo(k, fit_coef, L_b, U_b)
               END IF
            END IF
         CASE (8)
            IF (Rc >= R_max) THEN
               aw(1) = 2.8127089949768415E-04_dp
               aw(2) = 3.0942621869177655E-03_dp
               aw(3) = 1.6376552972548756E-02_dp
               aw(4) = 6.3208135461484877E-02_dp
               aw(5) = 2.0243640179262407E-01_dp
               aw(6) = 5.7158548817956678E-01_dp
               aw(7) = 1.4769970948293096E+00_dp
               aw(8) = 3.6498865034832644E+00_dp
               aw(9) = 8.9916798455210248E-04_dp
               aw(10) = 5.9076940528835931E-03_dp
               aw(11) = 2.4183992214449269E-02_dp
               aw(12) = 7.8619729999942961E-02_dp
               aw(13) = 2.2140056400660871E-01_dp
               aw(14) = 5.6481870666490019E-01_dp
               aw(15) = 1.3539633386811780E+00_dp
               aw(16) = 3.3347117690388681E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k8_sh(k, fit_coef, L_b, U_b)
               ELSE
                  CALL get_fit_coef_k8_lo(k, fit_coef, L_b, U_b)
               END IF
            END IF
         CASE (9)
            IF (Rc >= R_max) THEN
               aw(1) = 1.3620787587729222E-04_dp
               aw(2) = 1.4984233172990983E-03_dp
               aw(3) = 7.9304843136686811E-03_dp
               aw(4) = 3.0608852526075443E-02_dp
               aw(5) = 9.8023317512591052E-02_dp
               aw(6) = 2.7660071140675380E-01_dp
               aw(7) = 7.1177472290822164E-01_dp
               aw(8) = 1.7131253759530025E+00_dp
               aw(9) = 4.0050154343014466E+00_dp
               aw(10) = 4.3542990561554510E-04_dp
               aw(11) = 2.8608520316584030E-03_dp
               aw(12) = 1.1711291382515341E-02_dp
               aw(13) = 3.8071387509506595E-02_dp
               aw(14) = 1.0718758733492406E-01_dp
               aw(15) = 2.7293433084542407E-01_dp
               aw(16) = 6.4600134991152369E-01_dp
               aw(17) = 1.4634218815677373E+00_dp
               aw(18) = 3.4614670015318758E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k9_sh(k, fit_coef, L_b, U_b)
               ELSE
                  CALL get_fit_coef_k9_lo(k, fit_coef, L_b, U_b)
               END IF
            END IF
         CASE (10)
            IF (Rc >= R_max) THEN
               aw(1) = 6.8443901388305083E-05_dp
               aw(2) = 7.5295159558786751E-04_dp
               aw(3) = 3.9850355998364926E-03_dp
               aw(4) = 1.5380799076269465E-02_dp
               aw(5) = 4.9255465262658012E-02_dp
               aw(6) = 1.3897073828118570E-01_dp
               aw(7) = 3.5731192893101876E-01_dp
               aw(8) = 8.5579284635745334E-01_dp
               aw(9) = 1.9457922045681717E+00_dp
               aw(10) = 4.3457510815955152E+00_dp
               aw(11) = 2.1880174937623183E-04_dp
               aw(12) = 1.4375664500890244E-03_dp
               aw(13) = 5.8848743519528662E-03_dp
               aw(14) = 1.9130624082222252E-02_dp
               aw(15) = 5.3858487442246458E-02_dp
               aw(16) = 1.3708846821179432E-01_dp
               aw(17) = 3.2364445020328991E-01_dp
               aw(18) = 7.2229153747049757E-01_dp
               aw(19) = 1.5633339391232095E+00_dp
               aw(20) = 3.5765361929512172E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k10_sh(k, fit_coef, L_b, U_b)
               ELSE
                  CALL get_fit_coef_k10_lo(k, fit_coef, L_b, U_b)
               END IF
            END IF
         CASE (11)
            IF (Rc >= R_max) THEN
               aw(1) = 3.5503484659246813E-05_dp
               aw(2) = 3.9057395758855312E-04_dp
               aw(3) = 2.0671330057150483E-03_dp
               aw(4) = 7.9783846601041101E-03_dp
               aw(5) = 2.5549886450088238E-02_dp
               aw(6) = 7.2085122754821276E-02_dp
               aw(7) = 1.8530557881445084E-01_dp
               aw(8) = 4.4334723019604549E-01_dp
               aw(9) = 1.0024332764988158E+00_dp
               aw(10) = 2.1748065663060481E+00_dp
               aw(11) = 4.6738220578828598E+00_dp
               aw(12) = 1.1349768781191745E-04_dp
               aw(13) = 7.4570001674067511E-04_dp
               aw(14) = 3.0526245415056301E-03_dp
               aw(15) = 9.9235004487624248E-03_dp
               aw(16) = 2.7937353788559310E-02_dp
               aw(17) = 7.1104122225773142E-02_dp
               aw(18) = 1.6777050240847788E-01_dp
               aw(19) = 3.7320560837397676E-01_dp
               aw(20) = 7.9414366264691305E-01_dp
               aw(21) = 1.6552701124330567E+00_dp
               aw(22) = 3.6820059233978353E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k11_sh(k, fit_coef, L_b, U_b)
               ELSE
                  IF (Rc <= 1000.0_dp) THEN
                     CALL get_fit_coef_k11_mi(k, fit_coef, L_b, U_b)
                  ELSE
                     CALL get_fit_coef_k11_lo(k, fit_coef, L_b, U_b)
                  END IF
               END IF
            END IF
         CASE (12)
            IF (Rc >= R_max) THEN
               aw(1) = 1.8934012296256358E-05_dp
               aw(2) = 2.0829314615342741E-04_dp
               aw(3) = 1.1024023520154727E-03_dp
               aw(4) = 4.2548734477736863E-03_dp
               aw(5) = 1.3625745873543612E-02_dp
               aw(6) = 3.8442710765665453E-02_dp
               aw(7) = 9.8818287645666872E-02_dp
               aw(8) = 2.3636401719642108E-01_dp
               aw(9) = 5.3373454579640744E-01_dp
               aw(10) = 1.1508219943995057E+00_dp
               aw(11) = 2.4001213099764143E+00_dp
               aw(12) = 4.9906311509941723E+00_dp
               aw(13) = 6.0528329999281435E-05_dp
               aw(14) = 3.9768193515047327E-04_dp
               aw(15) = 1.6279651361197933E-03_dp
               aw(16) = 5.2922029903677377E-03_dp
               aw(17) = 1.4898953114611908E-02_dp
               aw(18) = 3.7918950606510460E-02_dp
               aw(19) = 8.9457745237352296E-02_dp
               aw(20) = 1.9884427477508254E-01_dp
               aw(21) = 4.2144796502526766E-01_dp
               aw(22) = 8.6198574283723861E-01_dp
               aw(23) = 1.7404432216810901E+00_dp
               aw(24) = 3.7794416490913081E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k12_sh(k, fit_coef, L_b, U_b)
               ELSE
                  IF (Rc <= 1000.0_dp) THEN
                     CALL get_fit_coef_k12_mi(k, fit_coef, L_b, U_b)
                  ELSE
                     CALL get_fit_coef_k12_lo(k, fit_coef, L_b, U_b)
                  END IF
               END IF
            END IF
         CASE (13)
            IF (Rc >= R_max) THEN
               aw(1) = 1.0347391661194427E-05_dp
               aw(2) = 1.1383169144875790E-04_dp
               aw(3) = 6.0246014985531054E-04_dp
               aw(4) = 2.3252777266436543E-03_dp
               aw(5) = 7.4464345169693259E-03_dp
               aw(6) = 2.1008804344928592E-02_dp
               aw(7) = 5.4003219260973512E-02_dp
               aw(8) = 1.2916211533151045E-01_dp
               aw(9) = 2.9156437505279720E-01_dp
               aw(10) = 6.2769461815368310E-01_dp
               aw(11) = 1.3003150067055842E+00_dp
               aw(12) = 2.6217712688276107E+00_dp
               aw(13) = 5.2973357440769391E+00_dp
               aw(14) = 3.3078584812160216E-05_dp
               aw(15) = 2.1733219160086539E-04_dp
               aw(16) = 8.8967889768721382E-04_dp
               aw(17) = 2.8921756053065669E-03_dp
               aw(18) = 8.1422349627770872E-03_dp
               aw(19) = 2.0722487658696522E-02_dp
               aw(20) = 4.8886434214697162E-02_dp
               aw(21) = 1.0864184956111259E-01_dp
               aw(22) = 2.3003323004923831E-01_dp
               aw(23) = 4.6829507330399833E-01_dp
               aw(24) = 9.2620553596664335E-01_dp
               aw(25) = 1.8198087510110939E+00_dp
               aw(26) = 3.8700490617962435E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k13_sh(k, fit_coef, L_b, U_b)
               ELSE
                  IF (Rc <= 1000.0_dp) THEN
                     CALL get_fit_coef_k13_mi(k, fit_coef, L_b, U_b)
                  ELSE
                     CALL get_fit_coef_k13_lo(k, fit_coef, L_b, U_b)
                  END IF
               END IF
            END IF
         CASE (14)
            IF (Rc >= R_max) THEN
               aw(1) = 5.7793103608868073E-06_dp
               aw(2) = 6.3578213256604128E-05_dp
               aw(3) = 3.3649100160858427E-04_dp
               aw(4) = 1.2987332474693207E-03_dp
               aw(5) = 4.1590436292552775E-03_dp
               aw(6) = 1.1734004029327220E-02_dp
               aw(7) = 3.0162212092092591E-02_dp
               aw(8) = 7.2139138815286746E-02_dp
               aw(9) = 1.6282875663814733E-01_dp
               aw(10) = 3.5040070791304617E-01_dp
               aw(11) = 7.2459730019137303E-01_dp
               aw(12) = 1.4504311395006662E+00_dp
               aw(13) = 2.8398376292459662E+00_dp
               aw(14) = 5.5949042429760905E+00_dp
               aw(15) = 1.8475323460824520E-05_dp
               aw(16) = 1.2138616450563895E-04_dp
               aw(17) = 4.9691078029321441E-04_dp
               aw(18) = 1.6153616951963146E-03_dp
               aw(19) = 4.5476671955009150E-03_dp
               aw(20) = 1.1574075510211794E-02_dp
               aw(21) = 2.7304150630042847E-02_dp
               aw(22) = 6.0675597068478228E-02_dp
               aw(23) = 1.2843669732401369E-01_dp
               aw(24) = 2.6114026945752189E-01_dp
               aw(25) = 5.1372708252492216E-01_dp
               aw(26) = 9.8714811599006669E-01_dp
               aw(27) = 1.8941325485709735E+00_dp
               aw(28) = 3.9547772395130827E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k14_sh(k, fit_coef, L_b, U_b)
               ELSE
                  IF (Rc <= 1000.0_dp) THEN
                     CALL get_fit_coef_k14_mi(k, fit_coef, L_b, U_b)
                  ELSE
                     CALL get_fit_coef_k14_lo(k, fit_coef, L_b, U_b)
                  END IF
               END IF
            END IF
         CASE (15)
            IF (Rc >= R_max) THEN
               aw(1) = 3.2916678993378631E-06_dp
               aw(2) = 3.6211649942309474E-05_dp
               aw(3) = 1.9165204138720421E-04_dp
               aw(4) = 7.3970738152952427E-04_dp
               aw(5) = 2.3688276459558682E-03_dp
               aw(6) = 6.6832262130187142E-03_dp
               aw(7) = 1.7179191903350474E-02_dp
               aw(8) = 4.1087365085774624E-02_dp
               aw(9) = 9.2737907068114855E-02_dp
               aw(10) = 1.9954458193741717E-01_dp
               aw(11) = 4.1243407495664974E-01_dp
               aw(12) = 8.2392876030355866E-01_dp
               aw(13) = 1.6008068838318577E+00_dp
               aw(14) = 3.0544270604919670E+00_dp
               aw(15) = 5.8841569972355829E+00_dp
               aw(16) = 1.0522817667026174E-05_dp
               aw(17) = 6.9136785573218804E-05_dp
               aw(18) = 2.8302083846862936E-04_dp
               aw(19) = 9.2004648989894703E-04_dp
               aw(20) = 2.5901722444491987E-03_dp
               aw(21) = 6.5921351784279962E-03_dp
               aw(22) = 1.5551321239944434E-02_dp
               aw(23) = 3.4557794033658169E-02_dp
               aw(24) = 7.3145459688388037E-02_dp
               aw(25) = 1.4866796426944323E-01_dp
               aw(26) = 2.9202494628235121E-01_dp
               aw(27) = 5.5775841334330534E-01_dp
               aw(28) = 1.0451182163084207E+00_dp
               aw(29) = 1.9640378140134063E+00_dp
               aw(30) = 4.0343871534325908E+00_dp
            ELSE
               IF (Rc <= 100.0_dp) THEN
                  CALL get_fit_coef_k15_sh(k, fit_coef, L_b, U_b)
               ELSE
                  IF (Rc <= 1000.0_dp) THEN
                     CALL get_fit_coef_k15_mi(k, fit_coef, L_b, U_b)
                  ELSE
                     CALL get_fit_coef_k15_lo(k, fit_coef, L_b, U_b)
                  END IF
               END IF
            END IF
         CASE DEFAULT
            CPABORT("")
         END SELECT
      END IF

      IF (Rc < R_max .AND. Rc >= R_min) THEN
         R_transf = SQRT(L_b/Rc)
         R_transf = SQRT(R_transf)*(SQRT(U_b/(U_b - L_b)) - SQRT(Rc/(U_b - L_b)))
         DO kkk = 1, 2*k
            aw(kkk) = fit_coef(0, kkk)
            DO icoef = 1, 12
               aw(kkk) = aw(kkk) + fit_coef(icoef, kkk)*R_transf**icoef
            END DO
         END DO
      END IF

      IF (PRESENT(mm_error)) mm_error = get_minimax_numerical_error(Rc, aw)

      CALL timestop(handle)

   END SUBROUTINE get_minimax_coeff_k15

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k1(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 8.6669999999999998E+00_dp
      fit_coef(0, 1) = 4.4649260681804581E-01_dp
      fit_coef(1, 1) = 3.0675836870877398E-06_dp
      fit_coef(2, 1) = 1.2472664651690426E+00_dp
      fit_coef(3, 1) = -1.4325378312702204E+00_dp
      fit_coef(4, 1) = -2.8852101654931545E-01_dp
      fit_coef(5, 1) = 1.3224636333703371E+01_dp
      fit_coef(6, 1) = -5.6553448268455575E+01_dp
      fit_coef(7, 1) = 1.5090215216505641E+02_dp
      fit_coef(8, 1) = -2.8212708107138189E+02_dp
      fit_coef(9, 1) = 3.6970660684742296E+02_dp
      fit_coef(10, 1) = -3.2374043561240165E+02_dp
      fit_coef(11, 1) = 1.6996735688896140E+02_dp
      fit_coef(12, 1) = -4.0395921268987330E+01_dp
      fit_coef(0, 2) = 1.4290997884622449E+00_dp
      fit_coef(1, 2) = 5.6745594674266576E-06_dp
      fit_coef(2, 2) = 2.4380748907132181E+00_dp
      fit_coef(3, 2) = -2.8016418715477878E+00_dp
      fit_coef(4, 2) = 3.7584116167213760E-01_dp
      fit_coef(5, 2) = 2.3327242171726457E+01_dp
      fit_coef(6, 2) = -1.0520185473804629E+02_dp
      fit_coef(7, 2) = 2.8832898650596110E+02_dp
      fit_coef(8, 2) = -5.4924137182200832E+02_dp
      fit_coef(9, 2) = 7.3027624158467188E+02_dp
      fit_coef(10, 2) = -6.4709409025644425E+02_dp
      fit_coef(11, 2) = 3.4309801277522899E+02_dp
      fit_coef(12, 2) = -8.2224836383626666E+01_dp

   END SUBROUTINE get_fit_coef_k1

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k2(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 4.1539999999999999E+01_dp
      fit_coef(0, 1) = 9.3106187850842653E-02_dp
      fit_coef(1, 1) = -5.0629856861437123E-05_dp
      fit_coef(2, 1) = 1.0015082423014863E+00_dp
      fit_coef(3, 1) = -3.5061593037011978E+00_dp
      fit_coef(4, 1) = 1.4483017017771591E+01_dp
      fit_coef(5, 1) = -4.5087114171411322E+01_dp
      fit_coef(6, 1) = 1.0677856101159048E+02_dp
      fit_coef(7, 1) = -1.8217614209987144E+02_dp
      fit_coef(8, 1) = 2.1105105002638726E+02_dp
      fit_coef(9, 1) = -1.5567397052444477E+02_dp
      fit_coef(10, 1) = 6.5143729915539296E+01_dp
      fit_coef(11, 1) = -1.1144201390642060E+01_dp
      fit_coef(12, 1) = -4.0413697217822347E-01_dp
      fit_coef(0, 2) = 1.0301539533889652E+00_dp
      fit_coef(1, 2) = -1.8120259734086499E-04_dp
      fit_coef(2, 2) = 3.9565096665651787E+00_dp
      fit_coef(3, 2) = -1.3814853129007110E+01_dp
      fit_coef(4, 2) = 5.7316987025660488E+01_dp
      fit_coef(5, 2) = -1.7656913969609036E+02_dp
      fit_coef(6, 2) = 4.1570794568371764E+02_dp
      fit_coef(7, 2) = -7.0182518512857257E+02_dp
      fit_coef(8, 2) = 8.0388072043543650E+02_dp
      fit_coef(9, 2) = -5.9083346503424480E+02_dp
      fit_coef(10, 2) = 2.5295462667462738E+02_dp
      fit_coef(11, 2) = -4.9543705304205133E+01_dp
      fit_coef(12, 2) = 1.0078618878729646E+00_dp
      fit_coef(0, 3) = 2.9765555120179177E-01_dp
      fit_coef(1, 3) = -1.0353208854227029E-04_dp
      fit_coef(2, 3) = 2.3095283302989413E+00_dp
      fit_coef(3, 3) = -8.0576570729831154E+00_dp
      fit_coef(4, 3) = 3.3739368092954251E+01_dp
      fit_coef(5, 3) = -1.0475419759213936E+02_dp
      fit_coef(6, 3) = 2.4672759964823950E+02_dp
      fit_coef(7, 3) = -4.1482385671796834E+02_dp
      fit_coef(8, 3) = 4.6585822070233695E+02_dp
      fit_coef(9, 3) = -3.2366345592266214E+02_dp
      fit_coef(10, 3) = 1.1849771979400440E+02_dp
      fit_coef(11, 3) = -1.0906004349060501E+01_dp
      fit_coef(12, 3) = -3.7598063728027054E+00_dp
      fit_coef(0, 4) = 1.9917444037245748E+00_dp
      fit_coef(1, 4) = -1.5897415044355091E-04_dp
      fit_coef(2, 4) = 3.2506199533242417E+00_dp
      fit_coef(3, 4) = -1.1373065869782103E+01_dp
      fit_coef(4, 4) = 4.6558059922949951E+01_dp
      fit_coef(5, 4) = -1.4224165879889640E+02_dp
      fit_coef(6, 4) = 3.3658512396870015E+02_dp
      fit_coef(7, 4) = -5.7666672132123392E+02_dp
      fit_coef(8, 4) = 6.8845868913664435E+02_dp
      fit_coef(9, 4) = -5.5227838411317748E+02_dp
      fit_coef(10, 4) = 2.8109979885115376E+02_dp
      fit_coef(11, 4) = -8.0922550511268881E+01_dp
      fit_coef(12, 4) = 9.8057271611259083E+00_dp

   END SUBROUTINE get_fit_coef_k2

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k3(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 1.4680000000000001E+02_dp
      fit_coef(0, 1) = 2.6348727105229584E-02_dp
      fit_coef(1, 1) = 2.0153944068439938E-04_dp
      fit_coef(2, 1) = 6.1530657742665373E-01_dp
      fit_coef(3, 1) = -3.1917423654295867E+00_dp
      fit_coef(4, 1) = 1.8476493141601150E+01_dp
      fit_coef(5, 1) = -7.6512288788660001E+01_dp
      fit_coef(6, 1) = 2.3572982749223701E+02_dp
      fit_coef(7, 1) = -5.2565813705739686E+02_dp
      fit_coef(8, 1) = 8.2467549297012602E+02_dp
      fit_coef(9, 1) = -8.8144565993107483E+02_dp
      fit_coef(10, 1) = 6.0887047274283691E+02_dp
      fit_coef(11, 1) = -2.4457518923875350E+02_dp
      fit_coef(12, 1) = 4.3343959760454148E+01_dp
      fit_coef(0, 2) = 2.8996604313967794E-01_dp
      fit_coef(1, 2) = 9.1966653451787525E-04_dp
      fit_coef(2, 2) = 2.9603964889982954E+00_dp
      fit_coef(3, 2) = -1.5436155715607415E+01_dp
      fit_coef(4, 2) = 8.9046449784370381E+01_dp
      fit_coef(5, 2) = -3.6819211532781554E+02_dp
      fit_coef(6, 2) = 1.1368031324897308E+03_dp
      fit_coef(7, 2) = -2.5382696326038131E+03_dp
      fit_coef(8, 2) = 3.9825571880586440E+03_dp
      fit_coef(9, 2) = -4.2549854059515601E+03_dp
      fit_coef(10, 2) = 2.9378687444731368E+03_dp
      fit_coef(11, 2) = -1.1796924244872741E+03_dp
      fit_coef(12, 2) = 2.0901178723785654E+02_dp
      fit_coef(0, 3) = 1.5533604484128483E+00_dp
      fit_coef(1, 3) = 2.0797203873888733E-03_dp
      fit_coef(2, 3) = 6.8823846111885398E+00_dp
      fit_coef(3, 3) = -3.6081265871868659E+01_dp
      fit_coef(4, 3) = 2.0382721580134501E+02_dp
      fit_coef(5, 3) = -8.3588368834778782E+02_dp
      fit_coef(6, 3) = 2.5589042797513875E+03_dp
      fit_coef(7, 3) = -5.6774399573846031E+03_dp
      fit_coef(8, 3) = 8.8818039188117691E+03_dp
      fit_coef(9, 3) = -9.4791778618478966E+03_dp
      fit_coef(10, 3) = 6.5412200642397502E+03_dp
      fit_coef(11, 3) = -2.6248240660426704E+03_dp
      fit_coef(12, 3) = 4.6461771734428112E+02_dp
      fit_coef(0, 4) = 8.4232388261126792E-02_dp
      fit_coef(1, 4) = 4.8315351483134558E-04_dp
      fit_coef(2, 4) = 1.4901688878913604E+00_dp
      fit_coef(3, 4) = -7.7278009557035778E+00_dp
      fit_coef(4, 4) = 4.5200180529539338E+01_dp
      fit_coef(5, 4) = -1.8811770415936255E+02_dp
      fit_coef(6, 4) = 5.8400962109149202E+02_dp
      fit_coef(7, 4) = -1.3122325202133370E+03_dp
      fit_coef(8, 4) = 2.0704762349353055E+03_dp
      fit_coef(9, 4) = -2.2210949812031968E+03_dp
      fit_coef(10, 4) = 1.5374220056673389E+03_dp
      fit_coef(11, 4) = -6.1817894452963594E+02_dp
      fit_coef(12, 4) = 1.0958974382960224E+02_dp
      fit_coef(0, 5) = 5.5403377016655386E-01_dp
      fit_coef(1, 5) = 9.6266405874694016E-04_dp
      fit_coef(2, 5) = 3.2463696099948760E+00_dp
      fit_coef(3, 5) = -1.7046533273203099E+01_dp
      fit_coef(4, 5) = 9.6159163168231984E+01_dp
      fit_coef(5, 5) = -3.9387879060912519E+02_dp
      fit_coef(6, 5) = 1.2057094045527570E+03_dp
      fit_coef(7, 5) = -2.6721670291204064E+03_dp
      fit_coef(8, 5) = 4.1758049164403919E+03_dp
      fit_coef(9, 5) = -4.4574085233127789E+03_dp
      fit_coef(10, 5) = 3.0806227002555916E+03_dp
      fit_coef(11, 5) = -1.2391922190720968E+03_dp
      fit_coef(12, 5) = 2.1996432808098118E+02_dp
      fit_coef(0, 6) = 2.3557652649415379E+00_dp
      fit_coef(1, 6) = 1.2866271731043475E-03_dp
      fit_coef(2, 6) = 4.2915899775667716E+00_dp
      fit_coef(3, 6) = -2.2590368917521662E+01_dp
      fit_coef(4, 6) = 1.2467969331840358E+02_dp
      fit_coef(5, 6) = -5.0586944026595319E+02_dp
      fit_coef(6, 6) = 1.5294941344789261E+03_dp
      fit_coef(7, 6) = -3.3597570045760758E+03_dp
      fit_coef(8, 6) = 5.2196087711777982E+03_dp
      fit_coef(9, 6) = -5.5365578831549756E+03_dp
      fit_coef(10, 6) = 3.7972016920254159E+03_dp
      fit_coef(11, 6) = -1.5146892655309941E+03_dp
      fit_coef(12, 6) = 2.6667582714296555E+02_dp

   END SUBROUTINE get_fit_coef_k3

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k4(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 4.3610000000000002E+02_dp
      fit_coef(0, 1) = 8.8672649711285020E-03_dp
      fit_coef(1, 1) = 5.7946595950817889E-04_dp
      fit_coef(2, 1) = 3.5675945534374653E-01_dp
      fit_coef(3, 1) = -2.0716213353930191E+00_dp
      fit_coef(4, 1) = 1.2657470729413074E+01_dp
      fit_coef(5, 1) = -4.9885844724999998E+01_dp
      fit_coef(6, 1) = 1.4055190870387446E+02_dp
      fit_coef(7, 1) = -2.8346908048131127E+02_dp
      fit_coef(8, 1) = 4.0424697790393645E+02_dp
      fit_coef(9, 1) = -3.9722796612734163E+02_dp
      fit_coef(10, 1) = 2.5527240184308903E+02_dp
      fit_coef(11, 1) = -9.6356867177556367E+01_dp
      fit_coef(12, 1) = 1.6170840941701190E+01_dp
      fit_coef(0, 2) = 9.7559203877924675E-02_dp
      fit_coef(1, 2) = 3.0161149229635965E-03_dp
      fit_coef(2, 2) = 1.8835046917379701E+00_dp
      fit_coef(3, 2) = -1.1044051251373379E+01_dp
      fit_coef(4, 2) = 6.6851851055246527E+01_dp
      fit_coef(5, 2) = -2.6245846800700389E+02_dp
      fit_coef(6, 2) = 7.3917115340985299E+02_dp
      fit_coef(7, 2) = -1.4908588623752182E+03_dp
      fit_coef(8, 2) = 2.1239922172484485E+03_dp
      fit_coef(9, 2) = -2.0828446618499129E+03_dp
      fit_coef(10, 2) = 1.3350612779873063E+03_dp
      fit_coef(11, 2) = -5.0262753942439002E+02_dp
      fit_coef(12, 2) = 8.4153648005965323E+01_dp
      fit_coef(0, 3) = 5.1690155310834918E-01_dp
      fit_coef(1, 3) = 7.9435328867281975E-03_dp
      fit_coef(2, 3) = 5.1805612337915026E+00_dp
      fit_coef(3, 3) = -3.1004834727213211E+01_dp
      fit_coef(4, 3) = 1.8433410572917765E+02_dp
      fit_coef(5, 3) = -7.2586403084370136E+02_dp
      fit_coef(6, 3) = 2.0342579068012742E+03_dp
      fit_coef(7, 3) = -4.0678145171858919E+03_dp
      fit_coef(8, 3) = 5.7545086314708260E+03_dp
      fit_coef(9, 3) = -5.6192254926256765E+03_dp
      fit_coef(10, 3) = 3.5949228753416078E+03_dp
      fit_coef(11, 3) = -1.3527284950245175E+03_dp
      fit_coef(12, 3) = 2.2651488475207483E+02_dp
      fit_coef(0, 4) = 2.0307980329330455E+00_dp
      fit_coef(1, 4) = 1.5103721465188732E-02_dp
      fit_coef(2, 4) = 1.0170027292275309E+01_dp
      fit_coef(3, 4) = -6.1629536377936731E+01_dp
      fit_coef(4, 4) = 3.6339135279187411E+02_dp
      fit_coef(5, 4) = -1.4370905260427323E+03_dp
      fit_coef(6, 4) = 4.0240409210934931E+03_dp
      fit_coef(7, 4) = -8.0281015706711596E+03_dp
      fit_coef(8, 4) = 1.1327126412565874E+04_dp
      fit_coef(9, 4) = -1.1018179056966479E+04_dp
      fit_coef(10, 4) = 7.0109171513047604E+03_dp
      fit_coef(11, 4) = -2.6210011647009460E+03_dp
      fit_coef(12, 4) = 4.3584541357166989E+02_dp
      fit_coef(0, 5) = 2.8347760489671650E-02_dp
      fit_coef(1, 5) = 1.4649670575243566E-03_dp
      fit_coef(2, 5) = 8.8299650888912973E-01_dp
      fit_coef(3, 5) = -5.0888392853266016E+00_dp
      fit_coef(4, 5) = 3.1075839685772152E+01_dp
      fit_coef(5, 5) = -1.2128602025520448E+02_dp
      fit_coef(6, 5) = 3.3937414729640119E+02_dp
      fit_coef(7, 5) = -6.8302048253288706E+02_dp
      fit_coef(8, 5) = 9.7534981752611350E+02_dp
      fit_coef(9, 5) = -9.6151742177938490E+02_dp
      fit_coef(10, 5) = 6.2037394636212662E+02_dp
      fit_coef(11, 5) = -2.3512801932664019E+02_dp
      fit_coef(12, 5) = 3.9611471687460110E+01_dp
      fit_coef(0, 6) = 1.8628153335173334E-01_dp
      fit_coef(1, 6) = 3.5520276122272556E-03_dp
      fit_coef(2, 6) = 2.3086588899079099E+00_dp
      fit_coef(3, 6) = -1.3800728356647832E+01_dp
      fit_coef(4, 6) = 8.2177612215645453E+01_dp
      fit_coef(5, 6) = -3.2302705970393248E+02_dp
      fit_coef(6, 6) = 9.0586751432164715E+02_dp
      fit_coef(7, 6) = -1.8072280789056376E+03_dp
      fit_coef(8, 6) = 2.5395651527460632E+03_dp
      fit_coef(9, 6) = -2.4571249439954981E+03_dp
      fit_coef(10, 6) = 1.5570737965484213E+03_dp
      fit_coef(11, 6) = -5.8102885219535062E+02_dp
      fit_coef(12, 6) = 9.6652055418531859E+01_dp
      fit_coef(0, 7) = 7.6509143586699357E-01_dp
      fit_coef(1, 7) = 6.2314075869334632E-03_dp
      fit_coef(2, 7) = 4.2963285111812013E+00_dp
      fit_coef(3, 7) = -2.6274891972880781E+01_dp
      fit_coef(4, 7) = 1.5420391696751719E+02_dp
      fit_coef(5, 7) = -6.1288005128817349E+02_dp
      fit_coef(6, 7) = 1.7216536923227670E+03_dp
      fit_coef(7, 7) = -3.4485704001993868E+03_dp
      fit_coef(8, 7) = 4.8977145964963656E+03_dp
      fit_coef(9, 7) = -4.8034792371771946E+03_dp
      fit_coef(10, 7) = 3.0816756873370769E+03_dp
      fit_coef(11, 7) = -1.1601064628670051E+03_dp
      fit_coef(12, 7) = 1.9390631611709998E+02_dp
      fit_coef(0, 8) = 2.6286642321590428E+00_dp
      fit_coef(1, 8) = 7.7803605643329997E-03_dp
      fit_coef(2, 8) = 5.3941542397285547E+00_dp
      fit_coef(3, 8) = -3.3087210185862091E+01_dp
      fit_coef(4, 8) = 1.9273310816176871E+02_dp
      fit_coef(5, 8) = -7.6265175017497995E+02_dp
      fit_coef(6, 8) = 2.1209378243539750E+03_dp
      fit_coef(7, 8) = -4.1903565462442602E+03_dp
      fit_coef(8, 8) = 5.8461185745331686E+03_dp
      fit_coef(9, 8) = -5.6134540471156033E+03_dp
      fit_coef(10, 8) = 3.5238957022458790E+03_dp
      fit_coef(11, 8) = -1.3009219235427045E+03_dp
      fit_coef(12, 8) = 2.1404715392962453E+02_dp

   END SUBROUTINE get_fit_coef_k4

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k5_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 1.2552725467386876E-02_dp
      fit_coef(1, 1) = 4.6649769538903360E-02_dp
      fit_coef(2, 1) = 9.2959477678781308E-02_dp
      fit_coef(3, 1) = 1.0149142608425701E-01_dp
      fit_coef(4, 1) = 2.8480683386527764E-02_dp
      fit_coef(5, 1) = 1.1113054160128813E-01_dp
      fit_coef(6, 1) = -6.0921078972010467E-01_dp
      fit_coef(7, 1) = 1.2596196983676409E+00_dp
      fit_coef(8, 1) = -1.8451880523904862E+00_dp
      fit_coef(9, 1) = 1.7957653403032408E+00_dp
      fit_coef(10, 1) = -1.0556136831977423E+00_dp
      fit_coef(11, 1) = 3.3910276380778348E-01_dp
      fit_coef(12, 1) = -4.6181724196745676E-02_dp
      fit_coef(0, 2) = 8.6549382218897089E-02_dp
      fit_coef(1, 2) = 2.4941345083475788E-01_dp
      fit_coef(2, 2) = 4.9245083501834641E-01_dp
      fit_coef(3, 2) = 5.3644207042808922E-01_dp
      fit_coef(4, 2) = 1.2428540956728855E-01_dp
      fit_coef(5, 2) = 6.7557445154066409E-01_dp
      fit_coef(6, 2) = -3.7485327413027623E+00_dp
      fit_coef(7, 2) = 8.1470783598203376E+00_dp
      fit_coef(8, 2) = -1.2089572776244864E+01_dp
      fit_coef(9, 2) = 1.1847479598929507E+01_dp
      fit_coef(10, 2) = -7.1090620100857240E+00_dp
      fit_coef(11, 2) = 2.3701867666711376E+00_dp
      fit_coef(12, 2) = -3.3987884749260155E-01_dp
      fit_coef(0, 3) = 3.3100121276962868E-01_dp
      fit_coef(1, 3) = 6.4559516531419092E-01_dp
      fit_coef(2, 3) = 1.1695370338909203E+00_dp
      fit_coef(3, 3) = 1.1909933210765451E+00_dp
      fit_coef(4, 3) = 3.2031142163339787E-01_dp
      fit_coef(5, 3) = 1.9886975419904467E+00_dp
      fit_coef(6, 3) = -9.7799471531969644E+00_dp
      fit_coef(7, 3) = 2.2252882892705870E+01_dp
      fit_coef(8, 3) = -3.7167325466273596E+01_dp
      fit_coef(9, 3) = 4.2278212835435340E+01_dp
      fit_coef(10, 3) = -2.9956267208984151E+01_dp
      fit_coef(11, 3) = 1.1968794532343871E+01_dp
      fit_coef(12, 3) = -2.0781674372557291E+00_dp
      fit_coef(0, 4) = 1.0393831770652862E+00_dp
      fit_coef(1, 4) = 1.2828501087122686E+00_dp
      fit_coef(2, 4) = 2.0825379734819873E+00_dp
      fit_coef(3, 4) = 1.7998297820064115E+00_dp
      fit_coef(4, 4) = 2.5080914149094569E-01_dp
      fit_coef(5, 4) = 4.4983607972730759E+00_dp
      fit_coef(6, 4) = -1.8966342225496536E+01_dp
      fit_coef(7, 4) = 4.5287344234453357E+01_dp
      fit_coef(8, 4) = -7.1920958793917023E+01_dp
      fit_coef(9, 4) = 6.9428467733194353E+01_dp
      fit_coef(10, 4) = -3.7831237207121376E+01_dp
      fit_coef(11, 4) = 1.0235977584153179E+01_dp
      fit_coef(12, 4) = -9.4268597782017161E-01_dp
      fit_coef(0, 5) = 2.9584698863596302E+00_dp
      fit_coef(1, 5) = 2.1184323245195742E+00_dp
      fit_coef(2, 5) = 3.1709055258739989E+00_dp
      fit_coef(3, 5) = 2.2881184795555805E+00_dp
      fit_coef(4, 5) = 1.3681290961089937E-01_dp
      fit_coef(5, 5) = 5.7626338647647000E+00_dp
      fit_coef(6, 5) = -2.2759359631501187E+01_dp
      fit_coef(7, 5) = 5.5231152856101538E+01_dp
      fit_coef(8, 5) = -8.4310336806818569E+01_dp
      fit_coef(9, 5) = 7.9823159301719699E+01_dp
      fit_coef(10, 5) = -4.7210036737799030E+01_dp
      fit_coef(11, 5) = 1.6850346492810033E+01_dp
      fit_coef(12, 5) = -2.8940063886358156E+00_dp
      fit_coef(0, 6) = 3.4210297969183165E-02_dp
      fit_coef(1, 6) = 1.1998863470153487E-01_dp
      fit_coef(2, 6) = 2.4069816123621954E-01_dp
      fit_coef(3, 6) = 2.6041301008819967E-01_dp
      fit_coef(4, 6) = 6.3759797012621147E-02_dp
      fit_coef(5, 6) = 2.9954791472086834E-01_dp
      fit_coef(6, 6) = -1.6121273118458419E+00_dp
      fit_coef(7, 6) = 3.4105969041090662E+00_dp
      fit_coef(8, 6) = -5.1176148344713379E+00_dp
      fit_coef(9, 6) = 5.1110976139035680E+00_dp
      fit_coef(10, 6) = -3.1089038444203969E+00_dp
      fit_coef(11, 6) = 1.0468712585814062E+00_dp
      fit_coef(12, 6) = -1.5183313871493764E-01_dp
      fit_coef(0, 7) = 1.3026228141842919E-01_dp
      fit_coef(1, 7) = 2.9032232823923010E-01_dp
      fit_coef(2, 7) = 5.4877493441677716E-01_dp
      fit_coef(3, 7) = 5.9286988770199878E-01_dp
      fit_coef(4, 7) = 1.6832665275575420E-01_dp
      fit_coef(5, 7) = 7.6364041194666443E-01_dp
      fit_coef(6, 7) = -4.1778527336132214E+00_dp
      fit_coef(7, 7) = 8.2782930216328303E+00_dp
      fit_coef(8, 7) = -1.0896573954399400E+01_dp
      fit_coef(9, 7) = 9.5485112538544357E+00_dp
      fit_coef(10, 7) = -5.1102322033174294E+00_dp
      fit_coef(11, 7) = 1.4792267331631863E+00_dp
      fit_coef(12, 7) = -1.7439726860494367E-01_dp
      fit_coef(0, 8) = 4.0415838257065068E-01_dp
      fit_coef(1, 8) = 5.1255268146080202E-01_dp
      fit_coef(2, 8) = 7.9992895750622572E-01_dp
      fit_coef(3, 8) = 6.6970622776170818E-01_dp
      fit_coef(4, 8) = 3.1464540387258828E-02_dp
      fit_coef(5, 8) = 2.7391878500879105E+00_dp
      fit_coef(6, 8) = -1.1671256087331150E+01_dp
      fit_coef(7, 8) = 3.2335565030282325E+01_dp
      fit_coef(8, 8) = -6.2446813951907615E+01_dp
      fit_coef(9, 8) = 7.6210993792312763E+01_dp
      fit_coef(10, 8) = -5.5813236665948722E+01_dp
      fit_coef(11, 8) = 2.2568205101339789E+01_dp
      fit_coef(12, 8) = -3.8998743959104960E+00_dp
      fit_coef(0, 9) = 1.1232181639604371E+00_dp
      fit_coef(1, 9) = 7.5524302843049163E-01_dp
      fit_coef(2, 9) = 1.0162656347519898E+00_dp
      fit_coef(3, 9) = 5.2091134380973092E-01_dp
      fit_coef(4, 9) = 6.2773571552543109E-02_dp
      fit_coef(5, 9) = 4.7009056670980359E-01_dp
      fit_coef(6, 9) = 9.6931752144443684E-01_dp
      fit_coef(7, 9) = -8.1767165915400302E+00_dp
      fit_coef(8, 9) = 3.1589453385128493E+01_dp
      fit_coef(9, 9) = -6.0876284446976854E+01_dp
      fit_coef(10, 9) = 6.0604503383962623E+01_dp
      fit_coef(11, 9) = -3.0336494827398063E+01_dp
      fit_coef(12, 9) = 6.0926762476688889E+00_dp
      fit_coef(0, 10) = 3.0638169445090688E+00_dp
      fit_coef(1, 10) = 8.9165891212009063E-01_dp
      fit_coef(2, 10) = 1.1487286532265266E+00_dp
      fit_coef(3, 10) = 5.1064617564126968E-01_dp
      fit_coef(4, 10) = -4.7446553971131300E-01_dp
      fit_coef(5, 10) = 3.6370240067130397E+00_dp
      fit_coef(6, 10) = -1.5479102145878249E+01_dp
      fit_coef(7, 10) = 4.6883317751508919E+01_dp
      fit_coef(8, 10) = -8.9069835651344420E+01_dp
      fit_coef(9, 10) = 1.1482810068630945E+02_dp
      fit_coef(10, 10) = -9.5886031308673950E+01_dp
      fit_coef(11, 10) = 4.5677182175500711E+01_dp
      fit_coef(12, 10) = -9.3187455185522623E+00_dp

   END SUBROUTINE get_fit_coef_k5_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k5_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.1540000000000000E+03_dp
      fit_coef(0, 1) = 3.3529086066115803E-03_dp
      fit_coef(1, 1) = -8.2268999028435328E-06_dp
      fit_coef(2, 1) = 3.3611781342021224E-02_dp
      fit_coef(3, 1) = -1.3832592990876919E-01_dp
      fit_coef(4, 1) = 6.8079696115130983E-01_dp
      fit_coef(5, 1) = -2.6681612815812645E+00_dp
      fit_coef(6, 1) = 8.2290959869689022E+00_dp
      fit_coef(7, 1) = -1.9074468376984107E+01_dp
      fit_coef(8, 1) = 3.2127429353482611E+01_dp
      fit_coef(9, 1) = -3.7859356041773175E+01_dp
      fit_coef(10, 1) = 2.9446783510204408E+01_dp
      fit_coef(11, 1) = -1.3538459324077703E+01_dp
      fit_coef(12, 1) = 2.7808973445415268E+00_dp
      fit_coef(0, 2) = 3.6885509656511273E-02_dp
      fit_coef(1, 2) = -4.6135404124615802E-05_dp
      fit_coef(2, 2) = 1.8694797387922529E-01_dp
      fit_coef(3, 2) = -7.7044086984549065E-01_dp
      fit_coef(4, 2) = 3.7458845491550794E+00_dp
      fit_coef(5, 2) = -1.4597914775662142E+01_dp
      fit_coef(6, 2) = 4.4865271280504238E+01_dp
      fit_coef(7, 2) = -1.0374981069891059E+02_dp
      fit_coef(8, 2) = 1.7445816708209344E+02_dp
      fit_coef(9, 2) = -2.0533721999642813E+02_dp
      fit_coef(10, 2) = 1.5956721649659403E+02_dp
      fit_coef(11, 2) = -7.3312269788822348E+01_dp
      fit_coef(12, 2) = 1.5050733022098333E+01_dp
      fit_coef(0, 3) = 1.9524285077618225E-01_dp
      fit_coef(1, 3) = -1.3983711740977594E-04_dp
      fit_coef(2, 3) = 5.5888718949926475E-01_dp
      fit_coef(3, 3) = -2.3088502072129957E+00_dp
      fit_coef(4, 3) = 1.0999754812308348E+01_dp
      fit_coef(5, 3) = -4.2542674867782623E+01_dp
      fit_coef(6, 3) = 1.2991866971851243E+02_dp
      fit_coef(7, 3) = -2.9910040841843386E+02_dp
      fit_coef(8, 3) = 5.0131493127323301E+02_dp
      fit_coef(9, 3) = -5.8857681580426390E+02_dp
      fit_coef(10, 3) = 4.5649949573815150E+02_dp
      fit_coef(11, 3) = -2.0942533223478736E+02_dp
      fit_coef(12, 3) = 4.2945558512158080E+01_dp
      fit_coef(0, 4) = 7.5508121759060887E-01_dp
      fit_coef(1, 4) = -3.1605312969997662E-04_dp
      fit_coef(2, 4) = 1.2522786699714503E+00_dp
      fit_coef(3, 4) = -5.1813437090592105E+00_dp
      fit_coef(4, 4) = 2.4364506762575651E+01_dp
      fit_coef(5, 4) = -9.3776364610573921E+01_dp
      fit_coef(6, 4) = 2.8515029307910078E+02_dp
      fit_coef(7, 4) = -6.5452025447927747E+02_dp
      fit_coef(8, 4) = 1.0946041909024659E+03_dp
      fit_coef(9, 4) = -1.2829510036669149E+03_dp
      fit_coef(10, 4) = 9.9375334913507118E+02_dp
      fit_coef(11, 4) = -4.5544368131405361E+02_dp
      fit_coef(12, 4) = 9.3324529570079108E+01_dp
      fit_coef(0, 5) = 2.4730351053385511E+00_dp
      fit_coef(1, 5) = -5.6369602817021478E-04_dp
      fit_coef(2, 5) = 2.2229968075692117E+00_dp
      fit_coef(3, 5) = -9.2054846165557898E+00_dp
      fit_coef(4, 5) = 4.2976267592818459E+01_dp
      fit_coef(5, 5) = -1.6496009719447150E+02_dp
      fit_coef(6, 5) = 5.0038928803322085E+02_dp
      fit_coef(7, 5) = -1.1466294337635131E+03_dp
      fit_coef(8, 5) = 1.9151887594969048E+03_dp
      fit_coef(9, 5) = -2.2425568445298004E+03_dp
      fit_coef(10, 5) = 1.7357499822754255E+03_dp
      fit_coef(11, 5) = -7.9505099488495580E+02_dp
      fit_coef(12, 5) = 1.6284234597227066E+02_dp
      fit_coef(0, 6) = 1.0718594277376004E-02_dp
      fit_coef(1, 6) = -2.0665714271050207E-05_dp
      fit_coef(2, 6) = 8.4582172811458273E-02_dp
      fit_coef(3, 6) = -3.4797663076359847E-01_dp
      fit_coef(4, 6) = 1.7165661645244641E+00_dp
      fit_coef(5, 6) = -6.7291693231375893E+00_dp
      fit_coef(6, 6) = 2.0768418975747345E+01_dp
      fit_coef(7, 6) = -4.8168030617751583E+01_dp
      fit_coef(8, 6) = 8.1164964031411273E+01_dp
      fit_coef(9, 6) = -9.5679411237784350E+01_dp
      fit_coef(10, 6) = 7.4441710956106434E+01_dp
      fit_coef(11, 6) = -3.4234363834212587E+01_dp
      fit_coef(12, 6) = 7.0335229296269830E+00_dp
      fit_coef(0, 7) = 7.0423918791211709E-02_dp
      fit_coef(1, 7) = -6.0238742257850590E-05_dp
      fit_coef(2, 7) = 2.4113679202510896E-01_dp
      fit_coef(3, 7) = -9.9588383189867091E-01_dp
      fit_coef(4, 7) = 4.7554782250087468E+00_dp
      fit_coef(5, 7) = -1.8401716891304922E+01_dp
      fit_coef(6, 7) = 5.6243855385486498E+01_dp
      fit_coef(7, 7) = -1.2955708976498997E+02_dp
      fit_coef(8, 7) = 2.1724259904024584E+02_dp
      fit_coef(9, 7) = -2.5514786204737450E+02_dp
      fit_coef(10, 7) = 1.9794444495810336E+02_dp
      fit_coef(11, 7) = -9.0825962860161653E+01_dp
      fit_coef(12, 7) = 1.8627265430579317E+01_dp
      fit_coef(0, 8) = 2.8840117189215214E-01_dp
      fit_coef(1, 8) = -1.3225944743723339E-04_dp
      fit_coef(2, 8) = 5.2242393022997591E-01_dp
      fit_coef(3, 8) = -2.1627434438440303E+00_dp
      fit_coef(4, 8) = 1.0121981765560314E+01_dp
      fit_coef(5, 8) = -3.8888859154975577E+01_dp
      fit_coef(6, 8) = 1.1806445276665656E+02_dp
      fit_coef(7, 8) = -2.7069811465697614E+02_dp
      fit_coef(8, 8) = 4.5233627491539119E+02_dp
      fit_coef(9, 8) = -5.2982955497378055E+02_dp
      fit_coef(10, 8) = 4.1019408642078287E+02_dp
      fit_coef(11, 8) = -1.8792358285758019E+02_dp
      fit_coef(12, 8) = 3.8496219600202558E+01_dp
      fit_coef(0, 9) = 9.4333095808344458E-01_dp
      fit_coef(1, 9) = -2.1878257569565257E-04_dp
      fit_coef(2, 9) = 8.5869507294830094E-01_dp
      fit_coef(3, 9) = -3.5589224290290429E+00_dp
      fit_coef(4, 9) = 1.6493170506220842E+01_dp
      fit_coef(5, 9) = -6.3130491079340260E+01_dp
      fit_coef(6, 9) = 1.9101930146767009E+02_dp
      fit_coef(7, 9) = -4.3694612969128224E+02_dp
      fit_coef(8, 9) = 7.2886349059237500E+02_dp
      fit_coef(9, 9) = -8.5258297326024899E+02_dp
      fit_coef(10, 9) = 6.5938598174532456E+02_dp
      fit_coef(11, 9) = -3.0184732852622824E+02_dp
      fit_coef(12, 9) = 6.1796237778410379E+01_dp
      fit_coef(0, 10) = 2.8485312741055728E+00_dp
      fit_coef(1, 10) = -2.6597242771707791E-04_dp
      fit_coef(2, 10) = 1.0421996494576409E+00_dp
      fit_coef(3, 10) = -4.3207432293774151E+00_dp
      fit_coef(4, 10) = 1.9972564302073831E+01_dp
      fit_coef(5, 10) = -7.6371765475451213E+01_dp
      fit_coef(6, 10) = 2.3088260828483425E+02_dp
      fit_coef(7, 10) = -5.2780572532914982E+02_dp
      fit_coef(8, 10) = 8.8002099877005821E+02_dp
      fit_coef(9, 10) = -1.0290334029188657E+03_dp
      fit_coef(10, 10) = 7.9563294851641638E+02_dp
      fit_coef(11, 10) = -3.6413961414797035E+02_dp
      fit_coef(12, 10) = 7.4536963875157937E+01_dp

   END SUBROUTINE get_fit_coef_k5_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k6_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 1.0105361899470566E-02_dp
      fit_coef(1, 1) = 3.9366983227627925E-02_dp
      fit_coef(2, 1) = 7.8053377439735977E-02_dp
      fit_coef(3, 1) = 8.6837580919162383E-02_dp
      fit_coef(4, 1) = 4.6771807433750277E-02_dp
      fit_coef(5, 1) = -6.6734672326170835E-02_dp
      fit_coef(6, 1) = 9.9926171459107288E-02_dp
      fit_coef(7, 1) = -4.1644827063229539E-01_dp
      fit_coef(8, 1) = 7.6706989454888019E-01_dp
      fit_coef(9, 1) = -8.7152052671237190E-01_dp
      fit_coef(10, 1) = 6.5955681044267345E-01_dp
      fit_coef(11, 1) = -2.9363049092199256E-01_dp
      fit_coef(12, 1) = 5.6414782302946012E-02_dp
      fit_coef(0, 2) = 6.3047382400045412E-02_dp
      fit_coef(1, 2) = 2.1292104950028926E-01_dp
      fit_coef(2, 2) = 4.1580501626077238E-01_dp
      fit_coef(3, 2) = 4.5381507206227650E-01_dp
      fit_coef(4, 2) = 2.3261948283202027E-01_dp
      fit_coef(5, 2) = -3.7333612007818145E-01_dp
      fit_coef(6, 2) = 5.8891901066912178E-01_dp
      fit_coef(7, 2) = -2.2435149407329531E+00_dp
      fit_coef(8, 2) = 4.0761036364131602E+00_dp
      fit_coef(9, 2) = -4.5995236732489717E+00_dp
      fit_coef(10, 2) = 3.4552630197713277E+00_dp
      fit_coef(11, 2) = -1.5313736452266173E+00_dp
      fit_coef(12, 2) = 2.9408638427612482E-01_dp
      fit_coef(0, 3) = 2.1176071064496055E-01_dp
      fit_coef(1, 3) = 5.4982332295197556E-01_dp
      fit_coef(2, 3) = 1.0068197532238754E+00_dp
      fit_coef(3, 3) = 1.0684972989083634E+00_dp
      fit_coef(4, 3) = 6.0915448276900608E-01_dp
      fit_coef(5, 3) = -1.0238203491794160E+00_dp
      fit_coef(6, 3) = 2.1584333978631443E+00_dp
      fit_coef(7, 3) = -8.0611200001081951E+00_dp
      fit_coef(8, 3) = 1.5009961267213102E+01_dp
      fit_coef(9, 3) = -1.6828339663434313E+01_dp
      fit_coef(10, 3) = 1.2012563864685774E+01_dp
      fit_coef(11, 3) = -4.9849632599680698E+00_dp
      fit_coef(12, 3) = 9.0294945051842401E-01_dp
      fit_coef(0, 4) = 5.9366013171339316E-01_dp
      fit_coef(1, 4) = 1.1012914782020269E+00_dp
      fit_coef(2, 4) = 1.8080642501136919E+00_dp
      fit_coef(3, 4) = 1.7378820099827390E+00_dp
      fit_coef(4, 4) = 9.4465841905941816E-01_dp
      fit_coef(5, 4) = -7.9254597628432621E-01_dp
      fit_coef(6, 4) = 8.0775877014080266E-01_dp
      fit_coef(7, 4) = -2.8523919108461318E+00_dp
      fit_coef(8, 4) = -5.3307961129947901E-01_dp
      fit_coef(9, 4) = 6.7453388823676228E+00_dp
      fit_coef(10, 4) = -6.9248289720281742E+00_dp
      fit_coef(11, 4) = 2.9021292958698686E+00_dp
      fit_coef(12, 4) = -4.5434573537973694E-01_dp
      fit_coef(0, 5) = 1.5164074934712080E+00_dp
      fit_coef(1, 5) = 1.8998922859528067E+00_dp
      fit_coef(2, 5) = 2.7894259870776525E+00_dp
      fit_coef(3, 5) = 2.2836065839718458E+00_dp
      fit_coef(4, 5) = 1.2416212609604089E+00_dp
      fit_coef(5, 5) = -1.9153393783150017E+00_dp
      fit_coef(6, 5) = 7.9420474748342826E+00_dp
      fit_coef(7, 5) = -2.4492358033298018E+01_dp
      fit_coef(8, 5) = 4.7450493814300060E+01_dp
      fit_coef(9, 5) = -6.6407651969522661E+01_dp
      fit_coef(10, 5) = 6.0384585557333999E+01_dp
      fit_coef(11, 5) = -3.0372584465966241E+01_dp
      fit_coef(12, 5) = 6.3523080267784335E+00_dp
      fit_coef(0, 6) = 3.7104671974427847E+00_dp
      fit_coef(1, 6) = 2.8916704696341959E+00_dp
      fit_coef(2, 6) = 3.9056283047263785E+00_dp
      fit_coef(3, 6) = 2.7310607551007684E+00_dp
      fit_coef(4, 6) = 1.4374042741229791E+00_dp
      fit_coef(5, 6) = -3.4614723670541747E+00_dp
      fit_coef(6, 6) = 1.5897273148597373E+01_dp
      fit_coef(7, 6) = -4.2970295002223452E+01_dp
      fit_coef(8, 6) = 7.7894359559436722E+01_dp
      fit_coef(9, 6) = -9.5319740659866412E+01_dp
      fit_coef(10, 6) = 7.1950730534128169E+01_dp
      fit_coef(11, 6) = -2.9556628041991047E+01_dp
      fit_coef(12, 6) = 5.0101543449317205E+00_dp
      fit_coef(0, 7) = 2.6903751790128751E-02_dp
      fit_coef(1, 7) = 1.0173916967403172E-01_dp
      fit_coef(2, 7) = 2.0130716226577558E-01_dp
      fit_coef(3, 7) = 2.2172005830916267E-01_dp
      fit_coef(4, 7) = 1.1797631841567084E-01_dp
      fit_coef(5, 7) = -1.7271060154154347E-01_dp
      fit_coef(6, 7) = 2.7144317536179635E-01_dp
      fit_coef(7, 7) = -1.1089252106255005E+00_dp
      fit_coef(8, 7) = 2.0389731343247885E+00_dp
      fit_coef(9, 7) = -2.3121180806820467E+00_dp
      fit_coef(10, 7) = 1.7413776939963503E+00_dp
      fit_coef(11, 7) = -7.7162742625008818E-01_dp
      fit_coef(12, 7) = 1.4782756356123158E-01_dp
      fit_coef(0, 8) = 8.6888406935520149E-02_dp
      fit_coef(1, 8) = 2.4888995444204018E-01_dp
      fit_coef(2, 8) = 4.7199012197419227E-01_dp
      fit_coef(3, 8) = 5.1105786073145709E-01_dp
      fit_coef(4, 8) = 2.5740296918257327E-01_dp
      fit_coef(5, 8) = -4.3538515482693113E-01_dp
      fit_coef(6, 8) = 6.0029937052834403E-01_dp
      fit_coef(7, 8) = -2.3033251784048994E+00_dp
      fit_coef(8, 8) = 4.3994596232924978E+00_dp
      fit_coef(9, 8) = -5.1923409302629455E+00_dp
      fit_coef(10, 8) = 4.0222988922179050E+00_dp
      fit_coef(11, 8) = -1.8176576142253549E+00_dp
      fit_coef(12, 8) = 3.5423906095949409E-01_dp
      fit_coef(0, 9) = 2.3221985873092740E-01_dp
      fit_coef(1, 9) = 4.3438739423022149E-01_dp
      fit_coef(2, 9) = 7.0152946783034409E-01_dp
      fit_coef(3, 9) = 6.7672001661556891E-01_dp
      fit_coef(4, 9) = 4.8622153708717586E-01_dp
      fit_coef(5, 9) = -8.1041365351394823E-01_dp
      fit_coef(6, 9) = 2.6701912318720957E+00_dp
      fit_coef(7, 9) = -8.7205715741371570E+00_dp
      fit_coef(8, 9) = 1.3607358717959904E+01_dp
      fit_coef(9, 9) = -1.1724874566877528E+01_dp
      fit_coef(10, 9) = 5.7575285125779478E+00_dp
      fit_coef(11, 9) = -1.4357902050465099E+00_dp
      fit_coef(12, 9) = 1.1555583770987149E-01_dp
      fit_coef(0, 10) = 5.7977445413011319E-01_dp
      fit_coef(1, 10) = 6.7547720201746209E-01_dp
      fit_coef(2, 10) = 8.9708210777254471E-01_dp
      fit_coef(3, 10) = 6.3516703724773627E-01_dp
      fit_coef(4, 10) = 1.3236506065579925E-01_dp
      fit_coef(5, 10) = 1.1242277490562433E+00_dp
      fit_coef(6, 10) = -4.2391903965270812E+00_dp
      fit_coef(7, 10) = 1.3658960134998143E+01_dp
      fit_coef(8, 10) = -2.7055678231066494E+01_dp
      fit_coef(9, 10) = 2.7613753748428412E+01_dp
      fit_coef(10, 10) = -1.3209392906437861E+01_dp
      fit_coef(11, 10) = 1.7306395341708665E+00_dp
      fit_coef(12, 10) = 4.1257627070756253E-01_dp
      fit_coef(0, 11) = 1.3735367200768180E+00_dp
      fit_coef(1, 11) = 9.1204205070843380E-01_dp
      fit_coef(2, 11) = 1.0566274747133615E+00_dp
      fit_coef(3, 11) = 4.5662075463077345E-01_dp
      fit_coef(4, 11) = 4.9184167220410918E-01_dp
      fit_coef(5, 11) = -3.3759011651764603E+00_dp
      fit_coef(6, 11) = 1.7420644860890455E+01_dp
      fit_coef(7, 11) = -5.0539046535940329E+01_dp
      fit_coef(8, 11) = 1.0164085731266914E+02_dp
      fit_coef(9, 11) = -1.3287580093754113E+02_dp
      fit_coef(10, 11) = 1.0416384226984417E+02_dp
      fit_coef(11, 11) = -4.4207454491849361E+01_dp
      fit_coef(12, 11) = 7.8051805666697041E+00_dp
      fit_coef(0, 12) = 3.3571561549545903E+00_dp
      fit_coef(1, 12) = 1.0554507466925072E+00_dp
      fit_coef(2, 12) = 1.1790184249005025E+00_dp
      fit_coef(3, 12) = 4.8131299480494466E-01_dp
      fit_coef(4, 12) = -1.4373971957160242E-01_dp
      fit_coef(5, 12) = 7.0579667305022653E-01_dp
      fit_coef(6, 12) = -2.8433888321215206E+00_dp
      fit_coef(7, 12) = 1.4837347864587512E+01_dp
      fit_coef(8, 12) = -3.6747200470183884E+01_dp
      fit_coef(9, 12) = 5.8955148084383744E+01_dp
      fit_coef(10, 12) = -5.7353458415070598E+01_dp
      fit_coef(11, 12) = 2.9868904883232268E+01_dp
      fit_coef(12, 12) = -6.3780881018464042E+00_dp

   END SUBROUTINE get_fit_coef_k6_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k6_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 2.8070000000000000E+03_dp
      fit_coef(0, 1) = 1.3801918030773162E-03_dp
      fit_coef(1, 1) = -1.2020528483486374E-05_dp
      fit_coef(2, 1) = 2.3317704806718915E-02_dp
      fit_coef(3, 1) = -1.1872432794031121E-01_dp
      fit_coef(4, 1) = 6.7174538480429080E-01_dp
      fit_coef(5, 1) = -2.7971289614009751E+00_dp
      fit_coef(6, 1) = 8.6652048337367535E+00_dp
      fit_coef(7, 1) = -1.9404181258270555E+01_dp
      fit_coef(8, 1) = 3.0766165134677419E+01_dp
      fit_coef(9, 1) = -3.3559777770469893E+01_dp
      fit_coef(10, 1) = 2.3905145262192065E+01_dp
      fit_coef(11, 1) = -9.9977817398860651E+00_dp
      fit_coef(12, 1) = 1.8600935931530318E+00_dp
      fit_coef(0, 2) = 1.5183557920944469E-02_dp
      fit_coef(1, 2) = -7.0300568913718165E-05_dp
      fit_coef(2, 2) = 1.3444945392494670E-01_dp
      fit_coef(3, 2) = -6.8712467440053460E-01_dp
      fit_coef(4, 2) = 3.8439092624947708E+00_dp
      fit_coef(5, 2) = -1.5943863755337579E+01_dp
      fit_coef(6, 2) = 4.9281124351364589E+01_dp
      fit_coef(7, 2) = -1.1020727704358747E+02_dp
      fit_coef(8, 2) = 1.7458545156698841E+02_dp
      fit_coef(9, 2) = -1.9032304924079617E+02_dp
      fit_coef(10, 2) = 1.3551010987771855E+02_dp
      fit_coef(11, 2) = -5.6654664500478205E+01_dp
      fit_coef(12, 2) = 1.0537734527387647E+01_dp
      fit_coef(0, 3) = 8.0361343459697018E-02_dp
      fit_coef(1, 3) = -2.2644879609720777E-04_dp
      fit_coef(2, 3) = 4.2400787838782916E-01_dp
      fit_coef(3, 3) = -2.1793513661368733E+00_dp
      fit_coef(4, 3) = 1.1986721121277252E+01_dp
      fit_coef(5, 3) = -4.9550061013597045E+01_dp
      fit_coef(6, 3) = 1.5269170262558245E+02_dp
      fit_coef(7, 3) = -3.4093496564056039E+02_dp
      fit_coef(8, 3) = 5.3967525217022001E+02_dp
      fit_coef(9, 3) = -5.8799761545490060E+02_dp
      fit_coef(10, 3) = 4.1842708147977544E+02_dp
      fit_coef(11, 3) = -1.7483665022530386E+02_dp
      fit_coef(12, 3) = 3.2499971977161401E+01_dp
      fit_coef(0, 4) = 3.1025521547243406E-01_dp
      fit_coef(1, 4) = -5.5670514020829420E-04_dp
      fit_coef(2, 4) = 1.0283519940275256E+00_dp
      fit_coef(3, 4) = -5.3051948737708079E+00_dp
      fit_coef(4, 4) = 2.8857705580976056E+01_dp
      fit_coef(5, 4) = -1.1904005396455449E+02_dp
      fit_coef(6, 4) = 3.6607040216746668E+02_dp
      fit_coef(7, 4) = -8.1652968181369386E+02_dp
      fit_coef(8, 4) = 1.2917726681162258E+03_dp
      fit_coef(9, 4) = -1.4068028264311984E+03_dp
      fit_coef(10, 4) = 1.0006527523899351E+03_dp
      fit_coef(11, 4) = -4.1792217327242378E+02_dp
      fit_coef(12, 4) = 7.7650266966660254E+01_dp
      fit_coef(0, 5) = 9.9661072333548284E-01_dp
      fit_coef(1, 5) = -1.1171889827697565E-03_dp
      fit_coef(2, 5) = 2.0461119945840913E+00_dp
      fit_coef(3, 5) = -1.0580601944377532E+01_dp
      fit_coef(4, 5) = 5.7146266925973983E+01_dp
      fit_coef(5, 5) = -2.3540296082342908E+02_dp
      fit_coef(6, 5) = 7.2293490323806668E+02_dp
      fit_coef(7, 5) = -1.6114328355080609E+03_dp
      fit_coef(8, 5) = 2.5483891682119961E+03_dp
      fit_coef(9, 5) = -2.7745316619650062E+03_dp
      fit_coef(10, 5) = 1.9729645919543370E+03_dp
      fit_coef(11, 5) = -8.2377007268050909E+02_dp
      fit_coef(12, 5) = 1.5301212059298075E+02_dp
      fit_coef(0, 6) = 2.8871384041721235E+00_dp
      fit_coef(1, 6) = -1.8663713535930727E-03_dp
      fit_coef(2, 6) = 3.4015318790079183E+00_dp
      fit_coef(3, 6) = -1.7613395948721490E+01_dp
      fit_coef(4, 6) = 9.4741249061261925E+01_dp
      fit_coef(5, 6) = -3.8994245150367010E+02_dp
      fit_coef(6, 6) = 1.1965993777329293E+03_dp
      fit_coef(7, 6) = -2.6661783300231391E+03_dp
      fit_coef(8, 6) = 4.2155136266861591E+03_dp
      fit_coef(9, 6) = -4.5888706525865791E+03_dp
      fit_coef(10, 6) = 3.2626368424723773E+03_dp
      fit_coef(11, 6) = -1.3620271028068767E+03_dp
      fit_coef(12, 6) = 2.5294961462574193E+02_dp
      fit_coef(0, 7) = 4.4122094885822010E-03_dp
      fit_coef(1, 7) = -3.0506026172807132E-05_dp
      fit_coef(2, 7) = 5.9262798671013028E-02_dp
      fit_coef(3, 7) = -3.0161167176282039E-01_dp
      fit_coef(4, 7) = 1.7086225790243108E+00_dp
      fit_coef(5, 7) = -7.1111130868609358E+00_dp
      fit_coef(6, 7) = 2.2029502145209662E+01_dp
      fit_coef(7, 7) = -4.9330933530700591E+01_dp
      fit_coef(8, 7) = 7.8203553981541233E+01_dp
      fit_coef(9, 7) = -8.5283995941844253E+01_dp
      fit_coef(10, 7) = 6.0734937581336254E+01_dp
      fit_coef(11, 7) = -2.5396331019128979E+01_dp
      fit_coef(12, 7) = 4.7244104190757472E+00_dp
      fit_coef(0, 8) = 2.8989148486487894E-02_dp
      fit_coef(1, 8) = -9.5531400887653847E-05_dp
      fit_coef(2, 8) = 1.7940835754435794E-01_dp
      fit_coef(3, 8) = -9.2139806209629660E-01_dp
      fit_coef(4, 8) = 5.0793149159385784E+00_dp
      fit_coef(5, 8) = -2.0996406588101028E+01_dp
      fit_coef(6, 8) = 6.4728578095110564E+01_dp
      fit_coef(7, 8) = -1.4454498390418735E+02_dp
      fit_coef(8, 8) = 2.2882213635238523E+02_dp
      fit_coef(9, 8) = -2.4934889854822558E+02_dp
      fit_coef(10, 8) = 1.7747811343984634E+02_dp
      fit_coef(11, 8) = -7.4175788381531561E+01_dp
      fit_coef(12, 8) = 1.3791699026484896E+01_dp
      fit_coef(0, 9) = 1.1867775101904349E-01_dp
      fit_coef(1, 9) = -2.3002545083394702E-04_dp
      fit_coef(2, 9) = 4.2351088978816531E-01_dp
      fit_coef(3, 9) = -2.1868332609774375E+00_dp
      fit_coef(4, 9) = 1.1863193969553315E+01_dp
      fit_coef(5, 9) = -4.8911381713192242E+01_dp
      fit_coef(6, 9) = 1.5033689264253886E+02_dp
      fit_coef(7, 9) = -3.3524655332010315E+02_dp
      fit_coef(8, 9) = 5.3030221224968204E+02_dp
      fit_coef(9, 9) = -5.7746463929962727E+02_dp
      fit_coef(10, 9) = 4.1070292367082004E+02_dp
      fit_coef(11, 9) = -1.7150997861389590E+02_dp
      fit_coef(12, 9) = 3.1862954979470953E+01_dp
      fit_coef(0, 10) = 3.8613921710382521E-01_dp
      fit_coef(1, 10) = -4.4141730884221628E-04_dp
      fit_coef(2, 10) = 8.0440006468030922E-01_dp
      fit_coef(3, 10) = -4.1653901689708244E+00_dp
      fit_coef(4, 10) = 2.2403190787145370E+01_dp
      fit_coef(5, 10) = -9.2211974413485876E+01_dp
      fit_coef(6, 10) = 2.8295925585741463E+02_dp
      fit_coef(7, 10) = -6.3046689253660929E+02_dp
      fit_coef(8, 10) = 9.9682217293895803E+02_dp
      fit_coef(9, 10) = -1.0850822438050727E+03_dp
      fit_coef(10, 10) = 7.7146124564531726E+02_dp
      fit_coef(11, 10) = -3.2204769431435153E+02_dp
      fit_coef(12, 10) = 5.9807896790490901E+01_dp
      fit_coef(0, 11) = 1.0973261216294632E+00_dp
      fit_coef(1, 11) = -6.7305790071552931E-04_dp
      fit_coef(2, 11) = 1.2191563472414737E+00_dp
      fit_coef(3, 11) = -6.3236622978399160E+00_dp
      fit_coef(4, 11) = 3.3838641149158832E+01_dp
      fit_coef(5, 11) = -1.3912998330249843E+02_dp
      fit_coef(6, 11) = 4.2651665499720167E+02_dp
      fit_coef(7, 11) = -9.4985408069387211E+02_dp
      fit_coef(8, 11) = 1.5014096221052871E+03_dp
      fit_coef(9, 11) = -1.6340457638756484E+03_dp
      fit_coef(10, 11) = 1.1615527639524207E+03_dp
      fit_coef(11, 11) = -4.8480157740581882E+02_dp
      fit_coef(12, 11) = 9.0015704436915854E+01_dp
      fit_coef(0, 12) = 3.0334100910855679E+00_dp
      fit_coef(1, 12) = -8.0128677103265039E-04_dp
      fit_coef(2, 12) = 1.4492945609869972E+00_dp
      fit_coef(3, 12) = -7.5204430315974555E+00_dp
      fit_coef(4, 12) = 4.0192517443549171E+01_dp
      fit_coef(5, 12) = -1.6520813224035805E+02_dp
      fit_coef(6, 12) = 5.0634186437136140E+02_dp
      fit_coef(7, 12) = -1.1274845628371224E+03_dp
      fit_coef(8, 12) = 1.7820754347486288E+03_dp
      fit_coef(9, 12) = -1.9394275221726205E+03_dp
      fit_coef(10, 12) = 1.3785794074607632E+03_dp
      fit_coef(11, 12) = -5.7535942912689143E+02_dp
      fit_coef(12, 12) = 1.0682548193028630E+02_dp

   END SUBROUTINE get_fit_coef_k6_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k7_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+00_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 8.4890821794056659E-03_dp
      fit_coef(1, 1) = 3.3909965321659449E-02_dp
      fit_coef(2, 1) = 6.7634541203882806E-02_dp
      fit_coef(3, 1) = 7.6353562644566234E-02_dp
      fit_coef(4, 1) = 3.5948563267733900E-02_dp
      fit_coef(5, 1) = -2.5771176000865662E-02_dp
      fit_coef(6, 1) = -4.0945070980132330E-02_dp
      fit_coef(7, 1) = -4.3343260753829285E-02_dp
      fit_coef(8, 1) = 1.3815432901925578E-01_dp
      fit_coef(9, 1) = -1.7538024437982933E-01_dp
      fit_coef(10, 1) = 1.6543062150575619E-01_dp
      fit_coef(11, 1) = -9.1169573964789280E-02_dp
      fit_coef(12, 1) = 2.0264349482170308E-02_dp
      fit_coef(0, 2) = 5.0116642286956788E-02_dp
      fit_coef(1, 2) = 1.8365009326344528E-01_dp
      fit_coef(2, 2) = 3.5986371694373215E-01_dp
      fit_coef(3, 2) = 3.9798008193331352E-01_dp
      fit_coef(4, 2) = 1.7835093713215566E-01_dp
      fit_coef(5, 2) = -1.3327325046872260E-01_dp
      fit_coef(6, 2) = -2.2419539361558347E-01_dp
      fit_coef(7, 2) = -1.5459559592831851E-01_dp
      fit_coef(8, 2) = 5.7498920152238775E-01_dp
      fit_coef(9, 2) = -7.2077559903241084E-01_dp
      fit_coef(10, 2) = 6.9718992787938672E-01_dp
      fit_coef(11, 2) = -3.9787065520994253E-01_dp
      fit_coef(12, 2) = 9.0654023942071268E-02_dp
      fit_coef(0, 3) = 1.5366998360970857E-01_dp
      fit_coef(1, 3) = 4.7204846669452077E-01_dp
      fit_coef(2, 3) = 8.8325935893055196E-01_dp
      fit_coef(3, 3) = 9.5894564770013224E-01_dp
      fit_coef(4, 3) = 4.2827132793975348E-01_dp
      fit_coef(5, 3) = -3.2332267433019096E-01_dp
      fit_coef(6, 3) = -6.1277314101509384E-01_dp
      fit_coef(7, 3) = -3.1904891130696167E-01_dp
      fit_coef(8, 3) = 1.7347388703763822E+00_dp
      fit_coef(9, 3) = -2.4531109689297721E+00_dp
      fit_coef(10, 3) = 2.2903712151885607E+00_dp
      fit_coef(11, 3) = -1.2225979103964606E+00_dp
      fit_coef(12, 3) = 2.6689160566637776E-01_dp
      fit_coef(0, 4) = 3.9160355914825828E-01_dp
      fit_coef(1, 4) = 9.3781039645429920E-01_dp
      fit_coef(2, 4) = 1.6046991663294892E+00_dp
      fit_coef(3, 4) = 1.6443527376353326E+00_dp
      fit_coef(4, 4) = 8.0944769705145370E-01_dp
      fit_coef(5, 4) = -5.1143473303498077E-01_dp
      fit_coef(6, 4) = -5.2679642212467515E-01_dp
      fit_coef(7, 4) = -1.8364447956089240E+00_dp
      fit_coef(8, 4) = 2.5683156855002216E+00_dp
      fit_coef(9, 4) = 1.5501948487951303E-01_dp
      fit_coef(10, 4) = -1.8725579182282932E+00_dp
      fit_coef(11, 4) = 1.2221919802984111E+00_dp
      fit_coef(12, 4) = -2.7544768102671480E-01_dp
      fit_coef(0, 5) = 9.1783363747242741E-01_dp
      fit_coef(1, 5) = 1.6309461105431255E+00_dp
      fit_coef(2, 5) = 2.4937510867743007E+00_dp
      fit_coef(3, 5) = 2.2928527582942770E+00_dp
      fit_coef(4, 5) = 9.1128547426324147E-01_dp
      fit_coef(5, 5) = 9.6182376136104375E-01_dp
      fit_coef(6, 5) = -6.3804461030632824E+00_dp
      fit_coef(7, 5) = 1.5896262996437345E+01_dp
      fit_coef(8, 5) = -3.2697959752155285E+01_dp
      fit_coef(9, 5) = 4.0204694276308771E+01_dp
      fit_coef(10, 5) = -2.6993883785520257E+01_dp
      fit_coef(11, 5) = 9.1873159371192017E+00_dp
      fit_coef(12, 5) = -1.2193412409631701E+00_dp
      fit_coef(0, 6) = 2.0447798807240689E+00_dp
      fit_coef(1, 6) = 2.5647815805469976E+00_dp
      fit_coef(2, 6) = 3.5222500643076748E+00_dp
      fit_coef(3, 6) = 2.7811523063687598E+00_dp
      fit_coef(4, 6) = 1.2018538663100924E+00_dp
      fit_coef(5, 6) = -5.4703655436328180E-01_dp
      fit_coef(6, 6) = 3.2225185493071979E+00_dp
      fit_coef(7, 6) = -1.4146455603976277E+01_dp
      fit_coef(8, 6) = 3.4172972270754862E+01_dp
      fit_coef(9, 6) = -5.7518986665041560E+01_dp
      fit_coef(10, 6) = 5.7935519876217747E+01_dp
      fit_coef(11, 6) = -3.0472542712278077E+01_dp
      fit_coef(12, 6) = 6.4684369280250085E+00_dp
      fit_coef(0, 7) = 4.4889127849133947E+00_dp
      fit_coef(1, 7) = 3.6844329124185262E+00_dp
      fit_coef(2, 7) = 4.6629325988561545E+00_dp
      fit_coef(3, 7) = 3.2053175028746703E+00_dp
      fit_coef(4, 7) = 1.1894024823027469E+00_dp
      fit_coef(5, 7) = -5.5664546189271846E-01_dp
      fit_coef(6, 7) = 4.4797254977157817E+00_dp
      fit_coef(7, 7) = -1.3162135892918478E+01_dp
      fit_coef(8, 7) = 2.6920639733012177E+01_dp
      fit_coef(9, 7) = -3.7885356014248515E+01_dp
      fit_coef(10, 7) = 3.0404958289755267E+01_dp
      fit_coef(11, 7) = -1.2048234826847173E+01_dp
      fit_coef(12, 7) = 1.7536434043357469E+00_dp
      fit_coef(0, 8) = 2.2333642667863175E-02_dp
      fit_coef(1, 8) = 8.7650336191702150E-02_dp
      fit_coef(2, 8) = 1.7407545680352357E-01_dp
      fit_coef(3, 8) = 1.9523723539373550E-01_dp
      fit_coef(4, 8) = 9.1078382735736693E-02_dp
      fit_coef(5, 8) = -6.5238131379128483E-02_dp
      fit_coef(6, 8) = -1.0790743432696713E-01_dp
      fit_coef(7, 8) = -1.0055162956331706E-01_dp
      fit_coef(8, 8) = 3.3608939631307738E-01_dp
      fit_coef(9, 8) = -4.2744783871398162E-01_dp
      fit_coef(10, 8) = 4.0569757577173882E-01_dp
      fit_coef(11, 8) = -2.2517859275370616E-01_dp
      fit_coef(12, 8) = 5.0308943083685061E-02_dp
      fit_coef(0, 9) = 6.5162011076589435E-02_dp
      fit_coef(1, 9) = 2.1474236673046285E-01_dp
      fit_coef(2, 9) = 4.1104112490299882E-01_dp
      fit_coef(3, 9) = 4.4722258512292351E-01_dp
      fit_coef(4, 9) = 1.8925685414469240E-01_dp
      fit_coef(5, 9) = -1.5238925014408933E-01_dp
      fit_coef(6, 9) = -2.8181109625687978E-01_dp
      fit_coef(7, 9) = 4.2073721801476338E-03_dp
      fit_coef(8, 9) = 2.9605389145708244E-01_dp
      fit_coef(9, 9) = -3.8389360909607761E-01_dp
      fit_coef(10, 9) = 4.5264861145927554E-01_dp
      fit_coef(11, 9) = -3.0203559346875924E-01_dp
      fit_coef(12, 9) = 7.4860217823464928E-02_dp
      fit_coef(0, 10) = 1.5346948637055580E-01_dp
      fit_coef(1, 10) = 3.6854433133610354E-01_dp
      fit_coef(2, 10) = 6.2892668627471893E-01_dp
      fit_coef(3, 10) = 6.5477471935299725E-01_dp
      fit_coef(4, 10) = 3.3294850052423303E-01_dp
      fit_coef(5, 10) = -2.7649376642801776E-01_dp
      fit_coef(6, 10) = -8.0513348586680189E-03_dp
      fit_coef(7, 10) = -2.0751899699120990E+00_dp
      fit_coef(8, 10) = 5.2487682729558243E+00_dp
      fit_coef(9, 10) = -6.5715600622772241E+00_dp
      fit_coef(10, 10) = 4.9805286462168388E+00_dp
      fit_coef(11, 10) = -2.1493517756117457E+00_dp
      fit_coef(12, 10) = 4.0028581387747814E-01_dp
      fit_coef(0, 11) = 3.4692296309076204E-01_dp
      fit_coef(1, 11) = 5.7228042968532444E-01_dp
      fit_coef(2, 11) = 8.0923016588837160E-01_dp
      fit_coef(3, 11) = 6.8825242004374665E-01_dp
      fit_coef(4, 11) = 3.0900931504334961E-01_dp
      fit_coef(5, 11) = 5.2224154947033075E-01_dp
      fit_coef(6, 11) = -2.6356703498487173E+00_dp
      fit_coef(7, 11) = 8.5966604379962970E+00_dp
      fit_coef(8, 11) = -2.2765680947013255E+01_dp
      fit_coef(9, 11) = 3.4684627103439190E+01_dp
      fit_coef(10, 11) = -2.9504573991969686E+01_dp
      fit_coef(11, 11) = 1.3352471045720314E+01_dp
      fit_coef(12, 11) = -2.5338463087754426E+00_dp
      fit_coef(0, 12) = 7.5471792316070130E-01_dp
      fit_coef(1, 12) = 8.1698667304635597E-01_dp
      fit_coef(2, 12) = 9.6453392023721507E-01_dp
      fit_coef(3, 12) = 5.8219771460039760E-01_dp
      fit_coef(4, 12) = 5.6834288665372115E-02_dp
      fit_coef(5, 12) = 9.5294461790630214E-01_dp
      fit_coef(6, 12) = -1.9967581073724616E+00_dp
      fit_coef(7, 12) = 3.9290913995308978E+00_dp
      fit_coef(8, 12) = 7.8240450388508198E-01_dp
      fit_coef(9, 12) = -1.8452411993447775E+01_dp
      fit_coef(10, 12) = 2.9544447334169273E+01_dp
      fit_coef(11, 12) = -1.9185547926257918E+01_dp
      fit_coef(12, 12) = 4.6420265475532112E+00_dp
      fit_coef(0, 13) = 1.6044221851675884E+00_dp
      fit_coef(1, 13) = 1.0402801401561750E+00_dp
      fit_coef(2, 13) = 1.0858841570371729E+00_dp
      fit_coef(3, 13) = 4.1610384890729457E-01_dp
      fit_coef(4, 13) = 3.6067842390926969E-01_dp
      fit_coef(5, 13) = -2.5175945757607381E+00_dp
      fit_coef(6, 13) = 1.3559516170001507E+01_dp
      fit_coef(7, 13) = -3.8412585801701184E+01_dp
      fit_coef(8, 13) = 7.5725440762246222E+01_dp
      fit_coef(9, 13) = -9.4629438016079902E+01_dp
      fit_coef(10, 13) = 6.8485370488181275E+01_dp
      fit_coef(11, 13) = -2.5761663952186389E+01_dp
      fit_coef(12, 13) = 3.8061288493554284E+00_dp
      fit_coef(0, 14) = 3.6234684753265820E+00_dp
      fit_coef(1, 14) = 1.1911013659903149E+00_dp
      fit_coef(2, 14) = 1.2089364306597108E+00_dp
      fit_coef(3, 14) = 4.6442125027798664E-01_dp
      fit_coef(4, 14) = -3.1777576152208559E-01_dp
      fit_coef(5, 14) = 2.0241094124978116E+00_dp
      fit_coef(6, 14) = -8.2031751444577026E+00_dp
      fit_coef(7, 14) = 2.9168093356344343E+01_dp
      fit_coef(8, 14) = -6.1746832221017200E+01_dp
      fit_coef(9, 14) = 8.7168512210629885E+01_dp
      fit_coef(10, 14) = -7.6686519988709378E+01_dp
      fit_coef(11, 14) = 3.7037473909730942E+01_dp
      fit_coef(12, 14) = -7.4610849374495922E+00_dp

   END SUBROUTINE get_fit_coef_k7_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k7_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 6.3730000000000000E+03_dp
      fit_coef(0, 1) = 6.0677583366177139E-04_dp
      fit_coef(1, 1) = 1.2203511941973786E-05_dp
      fit_coef(2, 1) = 1.5542282929846361E-02_dp
      fit_coef(3, 1) = -8.9625369583599124E-02_dp
      fit_coef(4, 1) = 5.4937244690257003E-01_dp
      fit_coef(5, 1) = -2.3465783422261945E+00_dp
      fit_coef(6, 1) = 7.2618002882762989E+00_dp
      fit_coef(7, 1) = -1.5964210654587358E+01_dp
      fit_coef(8, 1) = 2.4547214366628502E+01_dp
      fit_coef(9, 1) = -2.5732576011672748E+01_dp
      fit_coef(10, 1) = 1.7495704575487210E+01_dp
      fit_coef(11, 1) = -6.9489399028728807E+00_dp
      fit_coef(12, 1) = 1.2232329673889215E+00_dp
      fit_coef(0, 2) = 6.6752714913863925E-03_dp
      fit_coef(1, 2) = 7.0822129335160658E-05_dp
      fit_coef(2, 2) = 9.1986377885775919E-02_dp
      fit_coef(3, 2) = -5.3405428642774699E-01_dp
      fit_coef(4, 2) = 3.2354397768485041E+00_dp
      fit_coef(5, 2) = -1.3775260985114405E+01_dp
      fit_coef(6, 2) = 4.2538770413228939E+01_dp
      fit_coef(7, 2) = -9.3398515752623808E+01_dp
      fit_coef(8, 2) = 1.4350316188715925E+02_dp
      fit_coef(9, 2) = -1.5036166589456491E+02_dp
      fit_coef(10, 2) = 1.0220160925073999E+02_dp
      fit_coef(11, 2) = -4.0584863952257315E+01_dp
      fit_coef(12, 2) = 7.1433481247893695E+00_dp
      fit_coef(0, 3) = 3.5329627891532003E-02_dp
      fit_coef(1, 3) = 2.2595479303685819E-04_dp
      fit_coef(2, 3) = 3.0068670299306055E-01_dp
      fit_coef(3, 3) = -1.7613684261571489E+00_dp
      fit_coef(4, 3) = 1.0488351739421789E+01_dp
      fit_coef(5, 3) = -4.4542175766068205E+01_dp
      fit_coef(6, 3) = 1.3710066819725051E+02_dp
      fit_coef(7, 3) = -3.0052458124580761E+02_dp
      fit_coef(8, 3) = 4.6158839612526646E+02_dp
      fit_coef(9, 3) = -4.8381807226730933E+02_dp
      fit_coef(10, 3) = 3.2906228868490689E+02_dp
      fit_coef(11, 3) = -1.3076538707988345E+02_dp
      fit_coef(12, 3) = 2.3031862239615609E+01_dp
      fit_coef(0, 4) = 1.3636675797272446E-01_dp
      fit_coef(1, 4) = 5.6647704292489109E-04_dp
      fit_coef(2, 4) = 7.6618173910360343E-01_dp
      fit_coef(3, 4) = -4.5142657914579676E+00_dp
      fit_coef(4, 4) = 2.6580955043085275E+01_dp
      fit_coef(5, 4) = -1.1272540497018385E+02_dp
      fit_coef(6, 4) = 3.4623696291949358E+02_dp
      fit_coef(7, 4) = -7.5826687162428038E+02_dp
      fit_coef(8, 4) = 1.1645610278715124E+03_dp
      fit_coef(9, 4) = -1.2210276266592973E+03_dp
      fit_coef(10, 4) = 8.3085597800660651E+02_dp
      fit_coef(11, 4) = -3.3033932676724800E+02_dp
      fit_coef(12, 4) = 5.8211140143063389E+01_dp
      fit_coef(0, 5) = 4.3692354129416011E-01_dp
      fit_coef(1, 5) = 1.1976571152026670E-03_dp
      fit_coef(2, 5) = 1.6377604996306871E+00_dp
      fit_coef(3, 5) = -9.6866759364374868E+00_dp
      fit_coef(4, 5) = 5.6612700356628949E+01_dp
      fit_coef(5, 5) = -2.3984102923713382E+02_dp
      fit_coef(6, 5) = 7.3561273922776581E+02_dp
      fit_coef(7, 5) = -1.6099560282310861E+03_dp
      fit_coef(8, 5) = 2.4723564756113196E+03_dp
      fit_coef(9, 5) = -2.5927151309452665E+03_dp
      fit_coef(10, 5) = 1.7647692924297401E+03_dp
      fit_coef(11, 5) = -7.0188939174830352E+02_dp
      fit_coef(12, 5) = 1.2372418269084275E+02_dp
      fit_coef(0, 6) = 1.2378497399176813E+00_dp
      fit_coef(1, 6) = 2.1799404763566546E-03_dp
      fit_coef(2, 6) = 3.0033304053549266E+00_dp
      fit_coef(3, 6) = -1.7809339259291960E+01_dp
      fit_coef(4, 6) = 1.0356146048512531E+02_dp
      fit_coef(5, 6) = -4.3841186307805572E+02_dp
      fit_coef(6, 6) = 1.3433098010233427E+03_dp
      fit_coef(7, 6) = -2.9385308911410843E+03_dp
      fit_coef(8, 6) = 4.5121527260780776E+03_dp
      fit_coef(9, 6) = -4.7323142457902477E+03_dp
      fit_coef(10, 6) = 3.2217506991135392E+03_dp
      fit_coef(11, 6) = -1.2816488835906916E+03_dp
      fit_coef(12, 6) = 2.2596889581599186E+02_dp
      fit_coef(0, 7) = 3.2781853114633690E+00_dp
      fit_coef(1, 7) = 3.4389001435898561E-03_dp
      fit_coef(2, 7) = 4.7592429785757018E+00_dp
      fit_coef(3, 7) = -2.8265375133025159E+01_dp
      fit_coef(4, 7) = 1.6386545527527471E+02_dp
      fit_coef(5, 7) = -6.9336876574524956E+02_dp
      fit_coef(6, 7) = 2.1232162962264451E+03_dp
      fit_coef(7, 7) = -4.6431587196422133E+03_dp
      fit_coef(8, 7) = 7.1291035126709503E+03_dp
      fit_coef(9, 7) = -7.4773722512143786E+03_dp
      fit_coef(10, 7) = 5.0911609048640648E+03_dp
      fit_coef(11, 7) = -2.0255881694388165E+03_dp
      fit_coef(12, 7) = 3.5717987126526839E+02_dp
      fit_coef(0, 8) = 1.9397592535656612E-03_dp
      fit_coef(1, 8) = 3.1224136945126820E-05_dp
      fit_coef(2, 8) = 3.9794210955105026E-02_dp
      fit_coef(3, 8) = -2.2945121723050055E-01_dp
      fit_coef(4, 8) = 1.4081093618036822E+00_dp
      fit_coef(5, 8) = -6.0128883949026397E+00_dp
      fit_coef(6, 8) = 1.8615856974219142E+01_dp
      fit_coef(7, 8) = -4.0950738259707443E+01_dp
      fit_coef(8, 8) = 6.2995147189900791E+01_dp
      fit_coef(9, 8) = -6.6049282238872578E+01_dp
      fit_coef(10, 8) = 4.4906773129267386E+01_dp
      fit_coef(11, 8) = -1.7833720418073618E+01_dp
      fit_coef(12, 8) = 3.1386820276994531E+00_dp
      fit_coef(0, 9) = 1.2744789430102283E-02_dp
      fit_coef(1, 9) = 9.4709522213266069E-05_dp
      fit_coef(2, 9) = 1.2561186892205739E-01_dp
      fit_coef(3, 9) = -7.3485105104893944E-01_dp
      fit_coef(4, 9) = 4.3871454899236513E+00_dp
      fit_coef(5, 9) = -1.8627306443837497E+01_dp
      fit_coef(6, 9) = 5.7355825078700910E+01_dp
      fit_coef(7, 9) = -1.2570281346740651E+02_dp
      fit_coef(8, 9) = 1.9298761023815177E+02_dp
      fit_coef(9, 9) = -2.0218882506002689E+02_dp
      fit_coef(10, 9) = 1.3746187852198693E+02_dp
      fit_coef(11, 9) = -5.4608420832822929E+01_dp
      fit_coef(12, 9) = 9.6158489721068801E+00_dp
      fit_coef(0, 10) = 5.2173333294315491E-02_dp
      fit_coef(1, 10) = 2.3055569105486745E-04_dp
      fit_coef(2, 10) = 3.1278409064824336E-01_dp
      fit_coef(3, 10) = -1.8448833978165269E+00_dp
      fit_coef(4, 10) = 1.0840143395546759E+01_dp
      fit_coef(5, 10) = -4.5958901600709979E+01_dp
      fit_coef(6, 10) = 1.4110750881102726E+02_dp
      fit_coef(7, 10) = -3.0897514713921822E+02_dp
      fit_coef(8, 10) = 4.7453270093475732E+02_dp
      fit_coef(9, 10) = -4.9757949866356881E+02_dp
      fit_coef(10, 10) = 3.3861207241883631E+02_dp
      fit_coef(11, 10) = -1.3464036769091453E+02_dp
      fit_coef(12, 10) = 2.3727726896949317E+01_dp
      fit_coef(0, 11) = 1.6963116889689617E-01_dp
      fit_coef(1, 11) = 4.6901741492581534E-04_dp
      fit_coef(2, 11) = 6.4447000621439654E-01_dp
      fit_coef(3, 11) = -3.8181538304305924E+00_dp
      fit_coef(4, 11) = 2.2242335828744238E+01_dp
      fit_coef(5, 11) = -9.4193777329169180E+01_dp
      fit_coef(6, 11) = 2.8872166699409729E+02_dp
      fit_coef(7, 11) = -6.3173717103578508E+02_dp
      fit_coef(8, 11) = 9.7012917536585030E+02_dp
      fit_coef(9, 11) = -1.0174610499996350E+03_dp
      fit_coef(10, 11) = 6.9265148866175264E+02_dp
      fit_coef(11, 11) = -2.7552649004097742E+02_dp
      fit_coef(12, 11) = 4.8575054818192108E+01_dp
      fit_coef(0, 12) = 4.7830241431219889E-01_dp
      fit_coef(1, 12) = 8.0536633187941779E-04_dp
      fit_coef(2, 12) = 1.1161391199045863E+00_dp
      fit_coef(3, 12) = -6.6319638481849212E+00_dp
      fit_coef(4, 12) = 3.8412707149315935E+01_dp
      fit_coef(5, 12) = -1.6252589178932661E+02_dp
      fit_coef(6, 12) = 4.9759933805308168E+02_dp
      fit_coef(7, 12) = -1.0881301553940079E+03_dp
      fit_coef(8, 12) = 1.6707481111324018E+03_dp
      fit_coef(9, 12) = -1.7524442770987675E+03_dp
      fit_coef(10, 12) = 1.1932577680169172E+03_dp
      fit_coef(11, 12) = -4.7477746229497609E+02_dp
      fit_coef(12, 12) = 8.3723024185087226E+01_dp
      fit_coef(0, 13) = 1.2328742147617662E+00_dp
      fit_coef(1, 13) = 1.1464277359986964E-03_dp
      fit_coef(2, 13) = 1.5973172261127475E+00_dp
      fit_coef(3, 13) = -9.5083265310456007E+00_dp
      fit_coef(4, 13) = 5.4875950711264657E+01_dp
      fit_coef(5, 13) = -2.3203772329691336E+02_dp
      fit_coef(6, 13) = 7.0989929741104845E+02_dp
      fit_coef(7, 13) = -1.5517453595579404E+03_dp
      fit_coef(8, 13) = 2.3823114822950174E+03_dp
      fit_coef(9, 13) = -2.4989178165580306E+03_dp
      fit_coef(10, 13) = 1.7017491667373313E+03_dp
      fit_coef(11, 13) = -6.7719904341330835E+02_dp
      fit_coef(12, 13) = 1.1943636331258227E+02_dp
      fit_coef(0, 14) = 3.1933956854310037E+00_dp
      fit_coef(1, 14) = 1.3428436858287465E-03_dp
      fit_coef(2, 14) = 1.8732506344005615E+00_dp
      fit_coef(3, 14) = -1.1155445162652324E+01_dp
      fit_coef(4, 14) = 6.4329861295470778E+01_dp
      fit_coef(5, 14) = -2.7197046053760539E+02_dp
      fit_coef(6, 14) = 8.3192845835379330E+02_dp
      fit_coef(7, 14) = -1.8182993373416557E+03_dp
      fit_coef(8, 14) = 2.7914396608814409E+03_dp
      fit_coef(9, 14) = -2.9280874922129306E+03_dp
      fit_coef(10, 14) = 1.9940610145207027E+03_dp
      fit_coef(11, 14) = -7.9354809788071543E+02_dp
      fit_coef(12, 14) = 1.3996131946484607E+02_dp

   END SUBROUTINE get_fit_coef_k7_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k8_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 7.3303678789083827E-03_dp
      fit_coef(1, 1) = 1.9068147607663823E-02_dp
      fit_coef(2, 1) = 2.4523188619260956E-02_dp
      fit_coef(3, 1) = 1.7843299989374534E-02_dp
      fit_coef(4, 1) = 5.4214701717497675E-03_dp
      fit_coef(5, 1) = -2.4694578878648259E-03_dp
      fit_coef(6, 1) = -3.0284493938072380E-03_dp
      fit_coef(7, 1) = -1.6609340715173272E-05_dp
      fit_coef(8, 1) = 5.8996552393493821E-04_dp
      fit_coef(9, 1) = -5.5212702213677864E-05_dp
      fit_coef(10, 1) = 3.4862398750072344E-04_dp
      fit_coef(11, 1) = -3.9218355438884901E-04_dp
      fit_coef(12, 1) = 1.2569394142797241E-04_dp
      fit_coef(0, 2) = 4.1899841856663761E-02_dp
      fit_coef(1, 2) = 1.0302234723218924E-01_dp
      fit_coef(2, 2) = 1.3023821601879626E-01_dp
      fit_coef(3, 2) = 9.3099102625574220E-02_dp
      fit_coef(4, 2) = 2.7404221544145291E-02_dp
      fit_coef(5, 2) = -1.3085499573245986E-02_dp
      fit_coef(6, 2) = -1.5582408546316897E-02_dp
      fit_coef(7, 2) = 5.2970248514313688E-04_dp
      fit_coef(8, 2) = 2.2609393067795900E-03_dp
      fit_coef(9, 2) = 4.6279627025403949E-04_dp
      fit_coef(10, 2) = 1.4146159928236672E-03_dp
      fit_coef(11, 2) = -1.9800386255810288E-03_dp
      fit_coef(12, 2) = 6.6847497768620658E-04_dp
      fit_coef(0, 3) = 1.2090283026330555E-01_dp
      fit_coef(1, 3) = 2.6358046342322966E-01_dp
      fit_coef(2, 3) = 3.2173608677475352E-01_dp
      fit_coef(3, 3) = 2.2555617000622660E-01_dp
      fit_coef(4, 3) = 6.4242254728770876E-02_dp
      fit_coef(5, 3) = -3.3531313583337982E-02_dp
      fit_coef(6, 3) = -3.7471068968577388E-02_dp
      fit_coef(7, 3) = 3.6816061058378741E-03_dp
      fit_coef(8, 3) = 6.5687067201372204E-03_dp
      fit_coef(9, 3) = -2.6384863361432163E-03_dp
      fit_coef(10, 3) = 6.4385556221874555E-03_dp
      fit_coef(11, 3) = -6.0511747469316332E-03_dp
      fit_coef(12, 3) = 1.8525016625111832E-03_dp
      fit_coef(0, 4) = 2.8592429103323302E-01_dp
      fit_coef(1, 4) = 5.1841367625987889E-01_dp
      fit_coef(2, 4) = 5.9115684453736916E-01_dp
      fit_coef(3, 4) = 4.0105593630503783E-01_dp
      fit_coef(4, 4) = 1.1948149198856918E-01_dp
      fit_coef(5, 4) = -5.2658401069846560E-02_dp
      fit_coef(6, 4) = -7.3898713202086491E-02_dp
      fit_coef(7, 4) = 9.1566644314021242E-03_dp
      fit_coef(8, 4) = -1.1867605759483740E-02_dp
      fit_coef(9, 4) = 4.8318451940700932E-02_dp
      fit_coef(10, 4) = -3.5186576456035019E-02_dp
      fit_coef(11, 4) = 7.7443536084135598E-03_dp
      fit_coef(12, 4) = 6.1488354568711538E-04_dp
      fit_coef(0, 5) = 6.2290281260653402E-01_dp
      fit_coef(1, 5) = 8.9580902133545759E-01_dp
      fit_coef(2, 5) = 9.2609808946198835E-01_dp
      fit_coef(3, 5) = 5.8224600441757135E-01_dp
      fit_coef(4, 5) = 1.7839482993919215E-01_dp
      fit_coef(5, 5) = -4.6849100928790438E-02_dp
      fit_coef(6, 5) = -8.6070770575879779E-02_dp
      fit_coef(7, 5) = -1.8640277231677663E-02_dp
      fit_coef(8, 5) = 1.1471460387476767E-02_dp
      fit_coef(9, 5) = -4.1058254270079884E-02_dp
      fit_coef(10, 5) = 1.1306530929746011E-01_dp
      fit_coef(11, 5) = -8.8348931981328011E-02_dp
      fit_coef(12, 5) = 2.3298598302007475E-02_dp
      fit_coef(0, 6) = 1.2950331297092936E+00_dp
      fit_coef(1, 6) = 1.4208263619906618E+00_dp
      fit_coef(2, 6) = 1.3168706585134538E+00_dp
      fit_coef(3, 6) = 7.3735161688744788E-01_dp
      fit_coef(4, 6) = 2.1308021650008105E-01_dp
      fit_coef(5, 6) = -2.2140636924009825E-02_dp
      fit_coef(6, 6) = -6.5523715696762824E-02_dp
      fit_coef(7, 6) = -5.5227429757704954E-03_dp
      fit_coef(8, 6) = -3.6319163794681919E-02_dp
      fit_coef(9, 6) = -2.1057002605029421E-02_dp
      fit_coef(10, 6) = 6.7929722094043590E-02_dp
      fit_coef(11, 6) = -2.9821898442364107E-02_dp
      fit_coef(12, 6) = 2.5847668701003404E-03_dp
      fit_coef(0, 7) = 2.6150915219181412E+00_dp
      fit_coef(1, 7) = 2.0925013488168447E+00_dp
      fit_coef(2, 7) = 1.7517994308499847E+00_dp
      fit_coef(3, 7) = 8.5858648940090110E-01_dp
      fit_coef(4, 7) = 2.1877243019911102E-01_dp
      fit_coef(5, 7) = 9.2380438422983800E-03_dp
      fit_coef(6, 7) = -3.5233332041510933E-02_dp
      fit_coef(7, 7) = 5.9579661965596109E-02_dp
      fit_coef(8, 7) = -1.5530877010597718E-01_dp
      fit_coef(9, 7) = 1.1970417799859782E-01_dp
      fit_coef(10, 7) = -7.3859936805166476E-02_dp
      fit_coef(11, 7) = 3.9401452538859211E-02_dp
      fit_coef(12, 7) = -8.0701941615812720E-03_dp
      fit_coef(0, 8) = 5.2886811819505075E+00_dp
      fit_coef(1, 8) = 2.8791408561019143E+00_dp
      fit_coef(2, 8) = 2.2282496597143453E+00_dp
      fit_coef(3, 8) = 9.6463274698100221E-01_dp
      fit_coef(4, 8) = 2.0804339162869817E-01_dp
      fit_coef(5, 8) = 3.8463667107493361E-02_dp
      fit_coef(6, 8) = 6.2738138193085286E-03_dp
      fit_coef(7, 8) = 1.0100983178342585E-01_dp
      fit_coef(8, 8) = -1.8879341637967198E-01_dp
      fit_coef(9, 8) = 1.2577637243618989E-01_dp
      fit_coef(10, 8) = -4.4240946155634182E-02_dp
      fit_coef(11, 8) = -2.7654276880568853E-03_dp
      fit_coef(12, 8) = 7.1269217913952432E-03_dp
      fit_coef(0, 9) = 1.9153697185214646E-02_dp
      fit_coef(1, 9) = 4.9243996218940246E-02_dp
      fit_coef(2, 9) = 6.3063039313140626E-02_dp
      fit_coef(3, 9) = 4.5675586741478570E-02_dp
      fit_coef(4, 9) = 1.3789973749172879E-02_dp
      fit_coef(5, 9) = -6.3486543694890422E-03_dp
      fit_coef(6, 9) = -7.7438232796799697E-03_dp
      fit_coef(7, 9) = 4.2857574556254048E-05_dp
      fit_coef(8, 9) = 1.4170649293170209E-03_dp
      fit_coef(9, 9) = -6.5915991195257174E-05_dp
      fit_coef(10, 9) = 8.5657276913500048E-04_dp
      fit_coef(11, 9) = -1.0012786918386677E-03_dp
      fit_coef(12, 9) = 3.2416699208711279E-04_dp
      fit_coef(0, 10) = 5.2487499010059821E-02_dp
      fit_coef(1, 10) = 1.2020367140285117E-01_dp
      fit_coef(2, 10) = 1.4885896496539552E-01_dp
      fit_coef(3, 10) = 1.0446916703848536E-01_dp
      fit_coef(4, 10) = 2.9452247957851872E-02_dp
      fit_coef(5, 10) = -1.5096946389090963E-02_dp
      fit_coef(6, 10) = -1.6978769798053302E-02_dp
      fit_coef(7, 10) = 1.5621840217608987E-03_dp
      fit_coef(8, 10) = 1.1498062866988877E-03_dp
      fit_coef(9, 10) = 1.5932508408093571E-03_dp
      fit_coef(10, 10) = 1.2044209864296811E-03_dp
      fit_coef(11, 10) = -2.2799861036947064E-03_dp
      fit_coef(12, 10) = 8.0794913771834831E-04_dp
      fit_coef(0, 11) = 1.1214831732759308E-01_dp
      fit_coef(1, 11) = 2.0374016088705749E-01_dp
      fit_coef(2, 11) = 2.3269755141858203E-01_dp
      fit_coef(3, 11) = 1.5862165696203001E-01_dp
      fit_coef(4, 11) = 4.5536019398208713E-02_dp
      fit_coef(5, 11) = -2.4771937060547151E-02_dp
      fit_coef(6, 11) = -2.8674783757891342E-02_dp
      fit_coef(7, 11) = 3.5158594299659887E-03_dp
      fit_coef(8, 11) = 9.3535039711395342E-03_dp
      fit_coef(9, 11) = -3.0604392897587429E-03_dp
      fit_coef(10, 11) = -6.0143657962914525E-04_dp
      fit_coef(11, 11) = 4.2950330778982355E-04_dp
      fit_coef(12, 11) = 1.9255873491051108E-05_dp
      fit_coef(0, 12) = 2.3171781907279218E-01_dp
      fit_coef(1, 12) = 3.1091079731599292E-01_dp
      fit_coef(2, 12) = 3.0389179989885584E-01_dp
      fit_coef(3, 12) = 1.8532205840012622E-01_dp
      fit_coef(4, 12) = 6.2306830325449335E-02_dp
      fit_coef(5, 12) = -7.6106859087436904E-03_dp
      fit_coef(6, 12) = -3.6599453891127606E-02_dp
      fit_coef(7, 12) = 1.5509386794819603E-03_dp
      fit_coef(8, 12) = -4.0853132735310875E-02_dp
      fit_coef(9, 12) = 6.6595641401483052E-02_dp
      fit_coef(10, 12) = -1.8834300269643112E-02_dp
      fit_coef(11, 12) = -1.5460640921107466E-02_dp
      fit_coef(12, 12) = 8.1050164047865251E-03_dp
      fit_coef(0, 13) = 4.6839273718001340E-01_dp
      fit_coef(1, 13) = 4.4848685344087941E-01_dp
      fit_coef(2, 13) = 3.6456593426754630E-01_dp
      fit_coef(3, 13) = 1.7154603893334677E-01_dp
      fit_coef(4, 13) = 5.0029581620847113E-02_dp
      fit_coef(5, 13) = 1.7043310105799386E-02_dp
      fit_coef(6, 13) = 1.3624711461006474E-02_dp
      fit_coef(7, 13) = -4.0467589755335792E-02_dp
      fit_coef(8, 13) = 6.0917156869027100E-02_dp
      fit_coef(9, 13) = -1.7391850945484083E-01_dp
      fit_coef(10, 13) = 1.9540773247888954E-01_dp
      fit_coef(11, 13) = -8.8667454786059030E-02_dp
      fit_coef(12, 13) = 1.3511223411471382E-02_dp
      fit_coef(0, 14) = 9.2519886811346685E-01_dp
      fit_coef(1, 14) = 6.0160298038546722E-01_dp
      fit_coef(2, 14) = 4.1486918310874005E-01_dp
      fit_coef(3, 14) = 1.3735307260002044E-01_dp
      fit_coef(4, 14) = 1.8815781506297060E-02_dp
      fit_coef(5, 14) = 3.0299118585638541E-02_dp
      fit_coef(6, 14) = 2.2993724328518487E-02_dp
      fit_coef(7, 14) = 6.5423665107552431E-02_dp
      fit_coef(8, 14) = -1.4716829076991533E-01_dp
      fit_coef(9, 14) = 1.9132722353109541E-01_dp
      fit_coef(10, 14) = -2.2566102318270043E-01_dp
      fit_coef(11, 14) = 1.4512891962640706E-01_dp
      fit_coef(12, 14) = -3.4869303190922442E-02_dp
      fit_coef(0, 15) = 1.8180761389140441E+00_dp
      fit_coef(1, 15) = 7.3545411157338625E-01_dp
      fit_coef(2, 15) = 4.5382578616281255E-01_dp
      fit_coef(3, 15) = 1.0815904182157396E-01_dp
      fit_coef(4, 15) = -5.0564441213593594E-03_dp
      fit_coef(5, 15) = 3.1009296969077782E-02_dp
      fit_coef(6, 15) = 3.8543619111406462E-02_dp
      fit_coef(7, 15) = 4.9577607054581169E-02_dp
      fit_coef(8, 15) = -6.0901972233113555E-02_dp
      fit_coef(9, 15) = 4.4164548268671569E-02_dp
      fit_coef(10, 15) = -1.6420936751689871E-02_dp
      fit_coef(11, 15) = -1.9128788572305898E-02_dp
      fit_coef(12, 15) = 1.2610902336520378E-02_dp
      fit_coef(0, 16) = 3.8678537708718710E+00_dp
      fit_coef(1, 16) = 8.3755983422995561E-01_dp
      fit_coef(2, 16) = 5.0744880083562161E-01_dp
      fit_coef(3, 16) = 1.1252857834008548E-01_dp
      fit_coef(4, 16) = -1.3783956985565306E-02_dp
      fit_coef(5, 16) = 2.6967047539614656E-02_dp
      fit_coef(6, 16) = 4.1533124322160005E-02_dp
      fit_coef(7, 16) = 4.2761766824422134E-02_dp
      fit_coef(8, 16) = -3.1110880765649707E-02_dp
      fit_coef(9, 16) = 1.0013876986386597E-02_dp
      fit_coef(10, 16) = 2.7350239856939598E-02_dp
      fit_coef(11, 16) = -3.3885317157732221E-02_dp
      fit_coef(12, 16) = 1.0044298911104206E-02_dp

   END SUBROUTINE get_fit_coef_k8_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k8_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.3749000000000000E+04_dp
      fit_coef(0, 1) = 2.8123802981039521E-04_dp
      fit_coef(1, 1) = 1.5926645040140255E-05_dp
      fit_coef(2, 1) = 1.0329030933196252E-02_dp
      fit_coef(3, 1) = -6.4007908632844474E-02_dp
      fit_coef(4, 1) = 4.1161308446326278E-01_dp
      fit_coef(5, 1) = -1.7624808185774974E+00_dp
      fit_coef(6, 1) = 5.3986045573131776E+00_dp
      fit_coef(7, 1) = -1.1672753089064145E+01_dp
      fit_coef(8, 1) = 1.7591571465892009E+01_dp
      fit_coef(9, 1) = -1.8028596170855813E+01_dp
      fit_coef(10, 1) = 1.1956816057777168E+01_dp
      fit_coef(11, 1) = -4.6228922203591667E+00_dp
      fit_coef(12, 1) = 7.9068650631410775E-01_dp
      fit_coef(0, 2) = 3.0940667380651658E-03_dp
      fit_coef(1, 2) = 9.4603372213690319E-05_dp
      fit_coef(2, 2) = 6.2371593714423038E-02_dp
      fit_coef(3, 2) = -3.9061079256549203E-01_dp
      fit_coef(4, 2) = 2.4800148089165819E+00_dp
      fit_coef(5, 2) = -1.0590097222199683E+01_dp
      fit_coef(6, 2) = 3.2358288023437467E+01_dp
      fit_coef(7, 2) = -6.9843326031928697E+01_dp
      fit_coef(8, 2) = 1.0512898764379077E+02_dp
      fit_coef(9, 2) = -1.0764631696630022E+02_dp
      fit_coef(10, 2) = 7.1348387712368961E+01_dp
      fit_coef(11, 2) = -2.7573670968312566E+01_dp
      fit_coef(12, 2) = 4.7147055876112587E+00_dp
      fit_coef(0, 3) = 1.6375908158103483E-02_dp
      fit_coef(1, 3) = 3.1160349661731832E-04_dp
      fit_coef(2, 3) = 2.0927456992109450E-01_dp
      fit_coef(3, 3) = -1.3273785824948792E+00_dp
      fit_coef(4, 3) = 8.2688701711323898E+00_dp
      fit_coef(5, 3) = -3.5220567469598883E+01_dp
      fit_coef(6, 3) = 1.0701331897299248E+02_dp
      fit_coef(7, 3) = -2.2989760166900771E+02_dp
      fit_coef(8, 3) = 3.4493893446724235E+02_dp
      fit_coef(9, 3) = -3.5248615129301214E+02_dp
      fit_coef(10, 3) = 2.3334466251650863E+02_dp
      fit_coef(11, 3) = -9.0116552857752083E+01_dp
      fit_coef(12, 3) = 1.5403005349166568E+01_dp
      fit_coef(0, 4) = 6.3206454085883088E-02_dp
      fit_coef(1, 4) = 8.1154192163640840E-04_dp
      fit_coef(2, 4) = 5.5144861056801797E-01_dp
      fit_coef(3, 4) = -3.5260842601451339E+00_dp
      fit_coef(4, 4) = 2.1693237689818268E+01_dp
      fit_coef(5, 4) = -9.2258467452491558E+01_dp
      fit_coef(6, 4) = 2.7922864280490649E+02_dp
      fit_coef(7, 4) = -5.9806470556712168E+02_dp
      fit_coef(8, 4) = 8.9553395470219289E+02_dp
      fit_coef(9, 4) = -9.1394179906769205E+02_dp
      fit_coef(10, 4) = 6.0454016322906523E+02_dp
      fit_coef(11, 4) = -2.3336202357129835E+02_dp
      fit_coef(12, 4) = 3.9877696019475444E+01_dp
      fit_coef(0, 5) = 2.0243266526935697E-01_dp
      fit_coef(1, 5) = 1.8020515157672840E-03_dp
      fit_coef(2, 5) = 1.2344900509541614E+00_dp
      fit_coef(3, 5) = -7.9367169391085666E+00_dp
      fit_coef(4, 5) = 4.8422935474594851E+01_dp
      fit_coef(5, 5) = -2.0571215987444512E+02_dp
      fit_coef(6, 5) = 6.2097865514974285E+02_dp
      fit_coef(7, 5) = -1.3272491727851836E+03_dp
      fit_coef(8, 5) = 1.9845109548223063E+03_dp
      fit_coef(9, 5) = -2.0233745953849727E+03_dp
      fit_coef(10, 5) = 1.3376052915228147E+03_dp
      fit_coef(11, 5) = -5.1616417908155108E+02_dp
      fit_coef(12, 5) = 8.8189346087826550E+01_dp
      fit_coef(0, 6) = 5.7157821118623775E-01_dp
      fit_coef(1, 6) = 3.5076317937191783E-03_dp
      fit_coef(2, 6) = 2.4171981047624476E+00_dp
      fit_coef(3, 6) = -1.5600811633970423E+01_dp
      fit_coef(4, 6) = 9.4627104824817778E+01_dp
      fit_coef(5, 6) = -4.0167179845150463E+02_dp
      fit_coef(6, 6) = 1.2102930856782566E+03_dp
      fit_coef(7, 6) = -2.5828405060244945E+03_dp
      fit_coef(8, 6) = 3.8576577717746441E+03_dp
      fit_coef(9, 6) = -3.9303584111835289E+03_dp
      fit_coef(10, 6) = 2.5970876357091988E+03_dp
      fit_coef(11, 6) = -1.0019128608117558E+03_dp
      fit_coef(12, 6) = 1.7115745096843858E+02_dp
      fit_coef(0, 7) = 1.4769845791046401E+00_dp
      fit_coef(1, 7) = 6.0305390650928353E-03_dp
      fit_coef(2, 7) = 4.1735972990944301E+00_dp
      fit_coef(3, 7) = -2.7010363741735947E+01_dp
      fit_coef(4, 7) = 1.6316505318502820E+02_dp
      fit_coef(5, 7) = -6.9218832414970177E+02_dp
      fit_coef(6, 7) = 2.0829974158076584E+03_dp
      fit_coef(7, 7) = -4.4403073409985855E+03_dp
      fit_coef(8, 7) = 6.6265908711188122E+03_dp
      fit_coef(9, 7) = -6.7478402025357373E+03_dp
      fit_coef(10, 7) = 4.4572806839765526E+03_dp
      fit_coef(11, 7) = -1.7191849878725106E+03_dp
      fit_coef(12, 7) = 2.9365492248421521E+02_dp
      fit_coef(0, 8) = 3.6498674343702047E+00_dp
      fit_coef(1, 8) = 9.1861328914861536E-03_dp
      fit_coef(2, 8) = 6.3746908376074938E+00_dp
      fit_coef(3, 8) = -4.1325382231508740E+01_dp
      fit_coef(4, 8) = 2.4901070747707607E+02_dp
      fit_coef(5, 8) = -1.0559668335879119E+03_dp
      fit_coef(6, 8) = 3.1752073241924800E+03_dp
      fit_coef(7, 8) = -6.7638081975788209E+03_dp
      fit_coef(8, 8) = 1.0088934681089724E+04_dp
      fit_coef(9, 8) = -1.0269971775238855E+04_dp
      fit_coef(10, 8) = 6.7823044591615362E+03_dp
      fit_coef(11, 8) = -2.6155916262455330E+03_dp
      fit_coef(12, 8) = 4.4673488306700750E+02_dp
      fit_coef(0, 9) = 8.9908324754226225E-04_dp
      fit_coef(1, 9) = 4.1044297500840920E-05_dp
      fit_coef(2, 9) = 2.6592960271921522E-02_dp
      fit_coef(3, 9) = -1.6475343310038174E-01_dp
      fit_coef(4, 9) = 1.0601282924777382E+00_dp
      fit_coef(5, 9) = -4.5352614218846243E+00_dp
      fit_coef(6, 9) = 1.3894117699928159E+01_dp
      fit_coef(7, 9) = -3.0070697693720529E+01_dp
      fit_coef(8, 9) = 4.5371751202142313E+01_dp
      fit_coef(9, 9) = -4.6549283112947627E+01_dp
      fit_coef(10, 9) = 3.0899823469725881E+01_dp
      fit_coef(11, 9) = -1.1955338603808340E+01_dp
      fit_coef(12, 9) = 2.0459285416163961E+00_dp
      fit_coef(0, 10) = 5.9074268743020538E-03_dp
      fit_coef(1, 10) = 1.2917982248413781E-04_dp
      fit_coef(2, 10) = 8.6643119919001477E-02_dp
      fit_coef(3, 10) = -5.4867858026705008E-01_dp
      fit_coef(4, 10) = 3.4301753057411557E+00_dp
      fit_coef(5, 10) = -1.4614938748597057E+01_dp
      fit_coef(6, 10) = 4.4473633864726359E+01_dp
      fit_coef(7, 10) = -9.5626734151356388E+01_dp
      fit_coef(8, 10) = 1.4351698350787132E+02_dp
      fit_coef(9, 10) = -1.4665263177842959E+02_dp
      fit_coef(10, 10) = 9.7069655720786628E+01_dp
      fit_coef(11, 10) = -3.7481250052554522E+01_dp
      fit_coef(12, 10) = 6.4052741628428116E+00_dp
      fit_coef(0, 11) = 2.4183310977353760E-02_dp
      fit_coef(1, 11) = 3.2874969051224556E-04_dp
      fit_coef(2, 11) = 2.2379265645203872E-01_dp
      fit_coef(3, 11) = -1.4327470925490047E+00_dp
      fit_coef(4, 11) = 8.7976488483593407E+00_dp
      fit_coef(5, 11) = -3.7406648187178646E+01_dp
      fit_coef(6, 11) = 1.1314765967557075E+02_dp
      fit_coef(7, 11) = -2.4223242835221623E+02_dp
      fit_coef(8, 11) = 3.6262537126551155E+02_dp
      fit_coef(9, 11) = -3.7003484902257611E+02_dp
      fit_coef(10, 11) = 2.4474926980111050E+02_dp
      fit_coef(11, 11) = -9.4473072063769564E+01_dp
      fit_coef(12, 11) = 1.6143361281904095E+01_dp
      fit_coef(0, 12) = 7.8618264891410383E-02_dp
      fit_coef(1, 12) = 7.0637070558397471E-04_dp
      fit_coef(2, 12) = 4.8525758636061883E-01_dp
      fit_coef(3, 12) = -3.1258030882321890E+00_dp
      fit_coef(4, 12) = 1.9013088969838954E+01_dp
      fit_coef(5, 12) = -8.0742838543906203E+01_dp
      fit_coef(6, 12) = 2.4349690163060086E+02_dp
      fit_coef(7, 12) = -5.2005299814464468E+02_dp
      fit_coef(8, 12) = 7.7720399577073124E+02_dp
      fit_coef(9, 12) = -7.9217171146497253E+02_dp
      fit_coef(10, 12) = 5.2358376975962551E+02_dp
      fit_coef(11, 12) = -2.0202246440009665E+02_dp
      fit_coef(12, 12) = 3.4515024598323109E+01_dp
      fit_coef(0, 13) = 2.2139783616846259E-01_dp
      fit_coef(1, 13) = 1.3144046612244267E-03_dp
      fit_coef(2, 13) = 9.0893314563834471E-01_dp
      fit_coef(3, 13) = -5.8797685805649715E+00_dp
      fit_coef(4, 13) = 3.5538337045891197E+01_dp
      fit_coef(5, 13) = -1.5078315089899843E+02_dp
      fit_coef(6, 13) = 4.5381740391672264E+02_dp
      fit_coef(7, 13) = -9.6759884283658630E+02_dp
      fit_coef(8, 13) = 1.4442703065970643E+03_dp
      fit_coef(9, 13) = -1.4708851169077066E+03_dp
      fit_coef(10, 13) = 9.7168186776942412E+02_dp
      fit_coef(11, 13) = -3.7480558358376271E+02_dp
      fit_coef(12, 13) = 6.4023927170066514E+01_dp
      fit_coef(0, 14) = 5.6481431024685758E-01_dp
      fit_coef(1, 14) = 2.1175592913405812E-03_dp
      fit_coef(2, 14) = 1.4715067019515555E+00_dp
      fit_coef(3, 14) = -9.5481410751299673E+00_dp
      fit_coef(4, 14) = 5.7450639914728498E+01_dp
      fit_coef(5, 14) = -2.4358417666644485E+02_dp
      fit_coef(6, 14) = 7.3209348863097784E+02_dp
      fit_coef(7, 14) = -1.5589336783178264E+03_dp
      fit_coef(8, 14) = 2.3247369231211455E+03_dp
      fit_coef(9, 14) = -2.3660752817146231E+03_dp
      fit_coef(10, 14) = 1.5624107971537085E+03_dp
      fit_coef(11, 14) = -6.0251181283667290E+02_dp
      fit_coef(12, 14) = 1.0290483462401868E+02_dp
      fit_coef(0, 15) = 1.3539573260602098E+00_dp
      fit_coef(1, 15) = 2.8953150568411467E-03_dp
      fit_coef(2, 15) = 2.0183462102822127E+00_dp
      fit_coef(3, 15) = -1.3121809745507548E+01_dp
      fit_coef(4, 15) = 7.8730592243301871E+01_dp
      fit_coef(5, 15) = -3.3365664063421633E+02_dp
      fit_coef(6, 15) = 1.0019300382950731E+03_dp
      fit_coef(7, 15) = -2.1317727140515958E+03_dp
      fit_coef(8, 15) = 3.1770143228209445E+03_dp
      fit_coef(9, 15) = -3.2321383186557791E+03_dp
      fit_coef(10, 15) = 2.1337098849426625E+03_dp
      fit_coef(11, 15) = -8.2267510237563181E+02_dp
      fit_coef(12, 15) = 1.4049188918686588E+02_dp
      fit_coef(0, 16) = 3.3347047868343913E+00_dp
      fit_coef(1, 16) = 3.3620771239965489E-03_dp
      fit_coef(2, 16) = 2.3453111222189276E+00_dp
      fit_coef(3, 16) = -1.5253659417099389E+01_dp
      fit_coef(4, 16) = 9.1468857073866545E+01_dp
      fit_coef(5, 16) = -3.8760249737899488E+02_dp
      fit_coef(6, 16) = 1.1637180704439147E+03_dp
      fit_coef(7, 16) = -2.4755718153726775E+03_dp
      fit_coef(8, 16) = 3.6888887814157529E+03_dp
      fit_coef(9, 16) = -3.7525428895000127E+03_dp
      fit_coef(10, 16) = 2.4771015308842470E+03_dp
      fit_coef(11, 16) = -9.5503423745582688E+02_dp
      fit_coef(12, 16) = 1.6309118629666474E+02_dp

   END SUBROUTINE get_fit_coef_k8_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k9_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 6.4552940952308244E-03_dp
      fit_coef(1, 1) = 1.6981103140462521E-02_dp
      fit_coef(2, 1) = 2.1954315188636421E-02_dp
      fit_coef(3, 1) = 1.6041342676653195E-02_dp
      fit_coef(4, 1) = 4.8873501014382193E-03_dp
      fit_coef(5, 1) = -2.2607549832820450E-03_dp
      fit_coef(6, 1) = -2.5816889517990111E-03_dp
      fit_coef(7, 1) = -3.3177472147924311E-04_dp
      fit_coef(8, 1) = 6.7633586217191212E-04_dp
      fit_coef(9, 1) = 5.0154491931341720E-04_dp
      fit_coef(10, 1) = -7.5477037926869911E-04_dp
      fit_coef(11, 1) = 4.3750564946997106E-04_dp
      fit_coef(12, 1) = -1.0884574145659750E-04_dp
      fit_coef(0, 2) = 3.6158523047046677E-02_dp
      fit_coef(1, 2) = 9.1448302892550112E-02_dp
      fit_coef(2, 2) = 1.1643393031728949E-01_dp
      fit_coef(3, 2) = 8.3846218207491621E-02_dp
      fit_coef(4, 2) = 2.4929838698097138E-02_dp
      fit_coef(5, 2) = -1.2000438048214228E-02_dp
      fit_coef(6, 2) = -1.3233969051869923E-02_dp
      fit_coef(7, 2) = -1.6781088808427976E-03_dp
      fit_coef(8, 2) = 3.7175799646608039E-03_dp
      fit_coef(9, 2) = 2.3200967656010554E-03_dp
      fit_coef(10, 2) = -3.7296821680115493E-03_dp
      fit_coef(11, 2) = 2.1900296497794772E-03_dp
      fit_coef(12, 2) = -5.4966296541064880E-04_dp
      fit_coef(0, 3) = 1.0021175251987058E-01_dp
      fit_coef(1, 3) = 2.3290658157250768E-01_dp
      fit_coef(2, 3) = 2.8826132533071419E-01_dp
      fit_coef(3, 3) = 2.0349781078404069E-01_dp
      fit_coef(4, 3) = 5.8186389475606060E-02_dp
      fit_coef(5, 3) = -3.0264996373773254E-02_dp
      fit_coef(6, 3) = -3.0958058136578551E-02_dp
      fit_coef(7, 3) = -3.3385124438745062E-03_dp
      fit_coef(8, 3) = 9.8293593154800095E-03_dp
      fit_coef(9, 3) = 3.3092123173594926E-03_dp
      fit_coef(10, 3) = -6.9541522350748128E-03_dp
      fit_coef(11, 3) = 4.3124239191782493E-03_dp
      fit_coef(12, 3) = -1.1315599631624136E-03_dp
      fit_coef(0, 4) = 2.2411697245587953E-01_dp
      fit_coef(1, 4) = 4.5426701810076259E-01_dp
      fit_coef(2, 4) = 5.3412584957488496E-01_dp
      fit_coef(3, 4) = 3.6826576404725364E-01_dp
      fit_coef(4, 4) = 1.0602701865363849E-01_dp
      fit_coef(5, 4) = -5.4577502966425870E-02_dp
      fit_coef(6, 4) = -5.8813664835258211E-02_dp
      fit_coef(7, 4) = -6.1835169673331971E-03_dp
      fit_coef(8, 4) = 1.9342657781255358E-02_dp
      fit_coef(9, 4) = 1.5715487208569516E-02_dp
      fit_coef(10, 4) = -2.9256796795427198E-02_dp
      fit_coef(11, 4) = 1.8078340581926571E-02_dp
      fit_coef(12, 4) = -4.4135712366918118E-03_dp
      fit_coef(0, 5) = 4.5965700239101925E-01_dp
      fit_coef(1, 5) = 7.7718614768161642E-01_dp
      fit_coef(2, 5) = 8.4369237846467748E-01_dp
      fit_coef(3, 5) = 5.5382304325752407E-01_dp
      fit_coef(4, 5) = 1.6629027091834961E-01_dp
      fit_coef(5, 5) = -6.3214899288962637E-02_dp
      fit_coef(6, 5) = -8.7377144006966334E-02_dp
      fit_coef(7, 5) = -1.9259962748352861E-02_dp
      fit_coef(8, 5) = 5.7845664647801637E-03_dp
      fit_coef(9, 5) = 4.9430272241004450E-02_dp
      fit_coef(10, 5) = -3.0204735992589018E-02_dp
      fit_coef(11, 5) = 1.1560750816998883E-03_dp
      fit_coef(12, 5) = 2.2620193192966114E-03_dp
      fit_coef(0, 6) = 9.0150523193514287E-01_dp
      fit_coef(1, 6) = 1.2281870858285731E+00_dp
      fit_coef(2, 6) = 1.2075268204406777E+00_dp
      fit_coef(3, 6) = 7.2753542801068738E-01_dp
      fit_coef(4, 6) = 2.1703768564838269E-01_dp
      fit_coef(5, 6) = -4.9393451753158686E-02_dp
      fit_coef(6, 6) = -7.2910241488968047E-02_dp
      fit_coef(7, 6) = -7.2551142234446420E-02_dp
      fit_coef(8, 6) = 8.0982810239361422E-02_dp
      fit_coef(9, 6) = -1.2121381989765857E-01_dp
      fit_coef(10, 6) = 1.5600379928593311E-01_dp
      fit_coef(11, 6) = -8.4966639309547370E-02_dp
      fit_coef(12, 6) = 1.6060557175561714E-02_dp
      fit_coef(0, 7) = 1.7180040458559993E+00_dp
      fit_coef(1, 7) = 1.8243250896998631E+00_dp
      fit_coef(2, 7) = 1.6163577010542527E+00_dp
      fit_coef(3, 7) = 8.7029777481360915E-01_dp
      fit_coef(4, 7) = 2.4164335157245592E-01_dp
      fit_coef(5, 7) = -2.1288880285781901E-02_dp
      fit_coef(6, 7) = -4.1346282161013680E-02_dp
      fit_coef(7, 7) = -5.2892337289649802E-02_dp
      fit_coef(8, 7) = 1.2566897761471612E-02_dp
      fit_coef(9, 7) = -3.1238115025798451E-02_dp
      fit_coef(10, 7) = 8.8197509236568605E-03_dp
      fit_coef(11, 7) = 3.4268320342607482E-02_dp
      fit_coef(12, 7) = -1.7500888643553550E-02_dp
      fit_coef(0, 8) = 3.2202910229369826E+00_dp
      fit_coef(1, 8) = 2.5584863603841099E+00_dp
      fit_coef(2, 8) = 2.0603420885505477E+00_dp
      fit_coef(3, 8) = 9.8186936771714084E-01_dp
      fit_coef(4, 8) = 2.4292352348458562E-01_dp
      fit_coef(5, 8) = 1.0777326484119814E-02_dp
      fit_coef(6, 8) = 3.4596476543145072E-03_dp
      fit_coef(7, 8) = -2.5878994552358304E-02_dp
      fit_coef(8, 8) = -2.8942176969688937E-02_dp
      fit_coef(9, 8) = 1.7377801484123011E-02_dp
      fit_coef(10, 8) = -5.3614378940648612E-02_dp
      fit_coef(11, 8) = 5.9949568881798150E-02_dp
      fit_coef(12, 8) = -1.7982770689698030E-02_dp
      fit_coef(0, 9) = 6.1060543796063369E+00_dp
      fit_coef(1, 9) = 3.4046538736762950E+00_dp
      fit_coef(2, 9) = 2.5438587661473857E+00_dp
      fit_coef(3, 9) = 1.0825640947785635E+00_dp
      fit_coef(4, 9) = 2.3188547903431525E-01_dp
      fit_coef(5, 9) = 4.0838177926133055E-02_dp
      fit_coef(6, 9) = 5.8580319059672786E-02_dp
      fit_coef(7, 9) = -2.5983271132225795E-02_dp
      fit_coef(8, 9) = 1.2322432042298232E-02_dp
      fit_coef(9, 9) = -5.5917635514737024E-02_dp
      fit_coef(10, 9) = 2.7316598068796355E-02_dp
      fit_coef(11, 9) = 1.8442499628175768E-03_dp
      fit_coef(12, 9) = -1.7781863984523483E-03_dp
      fit_coef(0, 10) = 1.6794417808730849E-02_dp
      fit_coef(1, 10) = 4.3812238879122306E-02_dp
      fit_coef(2, 10) = 5.6433388526163281E-02_dp
      fit_coef(3, 10) = 4.1089213098926332E-02_dp
      fit_coef(4, 10) = 1.2450115917928132E-02_dp
      fit_coef(5, 10) = -5.8158386932097572E-03_dp
      fit_coef(6, 10) = -6.5845058729652479E-03_dp
      fit_coef(7, 10) = -8.4117835741441777E-04_dp
      fit_coef(8, 10) = 1.7550704199386239E-03_dp
      fit_coef(9, 10) = 1.2453478991127647E-03_dp
      fit_coef(10, 10) = -1.9039470025569718E-03_dp
      fit_coef(11, 10) = 1.1072558306069514E-03_dp
      fit_coef(12, 10) = -2.7613956295636030E-04_dp
      fit_coef(0, 11) = 4.4208436478371291E-02_dp
      fit_coef(1, 11) = 1.0636068480303659E-01_dp
      fit_coef(2, 11) = 1.3297982738437528E-01_dp
      fit_coef(3, 11) = 9.4166685284241089E-02_dp
      fit_coef(4, 11) = 2.7104946950991506E-02_dp
      fit_coef(5, 11) = -1.3725955515035670E-02_dp
      fit_coef(6, 11) = -1.4441472385392985E-02_dp
      fit_coef(7, 11) = -1.8001873375527177E-03_dp
      fit_coef(8, 11) = 4.3120736020466057E-03_dp
      fit_coef(9, 11) = 2.3403116157563499E-03_dp
      fit_coef(10, 11) = -3.9978958575902293E-03_dp
      fit_coef(11, 11) = 2.3671743796162751E-03_dp
      fit_coef(12, 11) = -5.9737357121890616E-04_dp
      fit_coef(0, 12) = 8.7964729281669957E-02_dp
      fit_coef(1, 12) = 1.7861597760308465E-01_dp
      fit_coef(2, 12) = 2.1020751169557236E-01_dp
      fit_coef(3, 12) = 1.4439439121052172E-01_dp
      fit_coef(4, 12) = 3.9618119968394774E-02_dp
      fit_coef(5, 12) = -2.3084280501379426E-02_dp
      fit_coef(6, 12) = -2.1408162571924363E-02_dp
      fit_coef(7, 12) = -1.6462884108872455E-03_dp
      fit_coef(8, 12) = 1.0736589853496247E-02_dp
      fit_coef(9, 12) = -4.2541437088059438E-03_dp
      fit_coef(10, 12) = -9.3740060359603919E-05_dp
      fit_coef(11, 12) = 1.2300444443144750E-03_dp
      fit_coef(12, 12) = -5.1803647309730399E-04_dp
      fit_coef(0, 13) = 1.6822075290435659E-01_dp
      fit_coef(1, 13) = 2.6777049222376398E-01_dp
      fit_coef(2, 13) = 2.7956089737268164E-01_dp
      fit_coef(3, 13) = 1.8086459490735671E-01_dp
      fit_coef(4, 13) = 5.6336421826288820E-02_dp
      fit_coef(5, 13) = -2.0705833559352838E-02_dp
      fit_coef(6, 13) = -3.4790488764663681E-02_dp
      fit_coef(7, 13) = -2.0331621831735812E-03_dp
      fit_coef(8, 13) = -1.1735723784168657E-02_dp
      fit_coef(9, 13) = 6.1673712195088486E-02_dp
      fit_coef(10, 13) = -6.7206330987665763E-02_dp
      fit_coef(11, 13) = 3.3093275404480781E-02_dp
      fit_coef(12, 13) = -6.5493688196066237E-03_dp
      fit_coef(0, 14) = 3.1816462977746751E-01_dp
      fit_coef(1, 14) = 3.8275682615881462E-01_dp
      fit_coef(2, 14) = 3.3806848941519918E-01_dp
      fit_coef(3, 14) = 1.8434916370789287E-01_dp
      fit_coef(4, 14) = 5.9808120500728569E-02_dp
      fit_coef(5, 14) = 4.0170881027690826E-03_dp
      fit_coef(6, 14) = -9.8506182208028918E-03_dp
      fit_coef(7, 14) = -4.1552347574310020E-02_dp
      fit_coef(8, 14) = 4.0058352212536501E-02_dp
      fit_coef(9, 14) = -1.0194992831605759E-01_dp
      fit_coef(10, 14) = 1.6261296510646620E-01_dp
      fit_coef(11, 14) = -1.0779408129071036E-01_dp
      fit_coef(12, 14) = 2.6181546202421097E-02_dp
      fit_coef(0, 15) = 5.9260717069203162E-01_dp
      fit_coef(1, 15) = 5.2236714064676937E-01_dp
      fit_coef(2, 15) = 3.8810235742485544E-01_dp
      fit_coef(3, 15) = 1.5987501622242634E-01_dp
      fit_coef(4, 15) = 3.8676464470426337E-02_dp
      fit_coef(5, 15) = 2.2397530474472072E-02_dp
      fit_coef(6, 15) = 2.9241369156544045E-02_dp
      fit_coef(7, 15) = -2.8237942949318490E-02_dp
      fit_coef(8, 15) = 2.5790590509430979E-02_dp
      fit_coef(9, 15) = -7.6958891581302161E-02_dp
      fit_coef(10, 15) = 3.2763833296732159E-02_dp
      fit_coef(11, 15) = 3.0093663681386516E-02_dp
      fit_coef(12, 15) = -1.8130008571038608E-02_dp
      fit_coef(0, 16) = 1.0894766158803928E+00_dp
      fit_coef(1, 16) = 6.6875957514002238E-01_dp
      fit_coef(2, 16) = 4.2769216255296599E-01_dp
      fit_coef(3, 16) = 1.2558051545388288E-01_dp
      fit_coef(4, 16) = 1.1302801231878518E-02_dp
      fit_coef(5, 16) = 3.1745607467806851E-02_dp
      fit_coef(6, 16) = 3.5806902619106150E-02_dp
      fit_coef(7, 16) = 4.4306027813559613E-02_dp
      fit_coef(8, 16) = -1.0289320673076535E-01_dp
      fit_coef(9, 16) = 1.4936934424649737E-01_dp
      fit_coef(10, 16) = -1.9115632729633894E-01_dp
      fit_coef(11, 16) = 1.1798136645873131E-01_dp
      fit_coef(12, 16) = -2.5669389351849693E-02_dp
      fit_coef(0, 17) = 2.0166575300282217E+00_dp
      fit_coef(1, 17) = 7.9409079746788347E-01_dp
      fit_coef(2, 17) = 4.6030756808902995E-01_dp
      fit_coef(3, 17) = 1.0086447786948728E-01_dp
      fit_coef(4, 17) = -6.7793884282790666E-03_dp
      fit_coef(5, 17) = 3.1537489875332711E-02_dp
      fit_coef(6, 17) = 5.2729459288171666E-02_dp
      fit_coef(7, 17) = 6.4791698755050928E-03_dp
      fit_coef(8, 17) = 2.1264307513768375E-02_dp
      fit_coef(9, 17) = -5.0214257290051294E-02_dp
      fit_coef(10, 17) = 5.5541231917168123E-02_dp
      fit_coef(11, 17) = -4.9625274322895069E-02_dp
      fit_coef(12, 17) = 1.7390208326760141E-02_dp
      fit_coef(0, 18) = 4.0940777457341033E+00_dp
      fit_coef(1, 18) = 9.0230847484016108E-01_dp
      fit_coef(2, 18) = 5.1722017559623490E-01_dp
      fit_coef(3, 18) = 1.0860069828810702E-01_dp
      fit_coef(4, 18) = -1.3407150393913490E-02_dp
      fit_coef(5, 18) = 2.8202598978792404E-02_dp
      fit_coef(6, 18) = 5.5571528490456060E-02_dp
      fit_coef(7, 18) = 1.1439449932901843E-03_dp
      fit_coef(8, 18) = 4.1437472839587285E-02_dp
      fit_coef(9, 18) = -6.3654618071805502E-02_dp
      fit_coef(10, 18) = 6.9641516666139436E-02_dp
      fit_coef(11, 18) = -4.3730228367233441E-02_dp
      fit_coef(12, 18) = 9.7082624411000788E-03_dp

   END SUBROUTINE get_fit_coef_k9_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k9_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 2.8387000000000000E+04_dp
      fit_coef(0, 1) = 1.3617241890792169E-04_dp
      fit_coef(1, 1) = 1.7628977768292345E-05_dp
      fit_coef(2, 1) = 6.8456013917834527E-03_dp
      fit_coef(3, 1) = -4.3895761081829160E-02_dp
      fit_coef(4, 1) = 2.9175828400673420E-01_dp
      fit_coef(5, 1) = -1.2306713809247385E+00_dp
      fit_coef(6, 1) = 3.6801538863592786E+00_dp
      fit_coef(7, 1) = -7.7402871595381262E+00_dp
      fit_coef(8, 1) = 1.1345092910279265E+01_dp
      fit_coef(9, 1) = -1.1320431537786655E+01_dp
      fit_coef(10, 1) = 7.3192031213325555E+00_dp
      fit_coef(11, 1) = -2.7615646170479575E+00_dp
      fit_coef(12, 1) = 4.6126151831636680E-01_dp
      fit_coef(0, 2) = 1.4982089284441154E-03_dp
      fit_coef(1, 2) = 1.0653572616093308E-04_dp
      fit_coef(2, 2) = 4.2011508058427197E-02_dp
      fit_coef(3, 2) = -2.7370993918916198E-01_dp
      fit_coef(4, 2) = 1.7939751803412998E+00_dp
      fit_coef(5, 2) = -7.5589803739294439E+00_dp
      fit_coef(6, 2) = 2.2557089057857102E+01_dp
      fit_coef(7, 2) = -4.7359259599571885E+01_dp
      fit_coef(8, 2) = 6.9309690166002625E+01_dp
      fit_coef(9, 2) = -6.9067026631983438E+01_dp
      fit_coef(10, 2) = 4.4604006663335653E+01_dp
      fit_coef(11, 2) = -1.6812960687505747E+01_dp
      fit_coef(12, 2) = 2.8059777999526059E+00_dp
      fit_coef(0, 3) = 7.9297670024187777E-03_dp
      fit_coef(1, 3) = 3.5670578026461845E-04_dp
      fit_coef(2, 3) = 1.4395693117221828E-01_dp
      fit_coef(3, 3) = -9.5715933171411527E-01_dp
      fit_coef(4, 3) = 6.1575946587200887E+00_dp
      fit_coef(5, 3) = -2.6009500954650985E+01_dp
      fit_coef(6, 3) = 7.7302143694085373E+01_dp
      fit_coef(7, 3) = -1.6146363554826667E+02_dp
      fit_coef(8, 3) = 2.3520423885679529E+02_dp
      fit_coef(9, 3) = -2.3348174715185502E+02_dp
      fit_coef(10, 3) = 1.5031746608470456E+02_dp
      fit_coef(11, 3) = -5.6519157967148246E+01_dp
      fit_coef(12, 3) = 9.4136465031151371E+00_dp
      fit_coef(0, 4) = 3.0606944390022209E-02_dp
      fit_coef(1, 4) = 9.4936819479453946E-04_dp
      fit_coef(2, 4) = 3.8897471198892203E-01_dp
      fit_coef(3, 4) = -2.6197251988016608E+00_dp
      fit_coef(4, 4) = 1.6650781475997896E+01_dp
      fit_coef(5, 4) = -7.0453305665193994E+01_dp
      fit_coef(6, 4) = 2.0877294584192904E+02_dp
      fit_coef(7, 4) = -4.3466952414824391E+02_dp
      fit_coef(8, 4) = 6.3140792713339783E+02_dp
      fit_coef(9, 4) = -6.2527262377050522E+02_dp
      fit_coef(10, 4) = 4.0172259183111106E+02_dp
      fit_coef(11, 4) = -1.5078038623522440E+02_dp
      fit_coef(12, 4) = 2.5075879780026515E+01_dp
      fit_coef(0, 5) = 9.8018953522175933E-02_dp
      fit_coef(1, 5) = 2.1720322644464903E-03_dp
      fit_coef(2, 5) = 8.9930042879493677E-01_dp
      fit_coef(3, 5) = -6.1094240846085883E+00_dp
      fit_coef(4, 5) = 3.8522442672428134E+01_dp
      fit_coef(5, 5) = -1.6318345003883445E+02_dp
      fit_coef(6, 5) = 4.8266158600585260E+02_dp
      fit_coef(7, 5) = -1.0027451259770581E+03_dp
      fit_coef(8, 5) = 1.4536819184010467E+03_dp
      fit_coef(9, 5) = -1.4370066077396380E+03_dp
      fit_coef(10, 5) = 9.2184153436145550E+02_dp
      fit_coef(11, 5) = -3.4555746263788461E+02_dp
      fit_coef(12, 5) = 5.7407542232062980E+01_dp
      fit_coef(0, 6) = 2.7659185448391987E-01_dp
      fit_coef(1, 6) = 4.4094406641574943E-03_dp
      fit_coef(2, 6) = 1.8398642403823151E+00_dp
      fit_coef(3, 6) = -1.2577480695844354E+01_dp
      fit_coef(4, 6) = 7.8862889072705556E+01_dp
      fit_coef(5, 6) = -3.3434171658322168E+02_dp
      fit_coef(6, 6) = 9.8773755594540955E+02_dp
      fit_coef(7, 6) = -2.0489450065104556E+03_dp
      fit_coef(8, 6) = 2.9660415791608848E+03_dp
      fit_coef(9, 6) = -2.9282468772949642E+03_dp
      fit_coef(10, 6) = 1.8764113820048669E+03_dp
      fit_coef(11, 6) = -7.0273375035027368E+02_dp
      fit_coef(12, 6) = 1.1665511108284656E+02_dp
      fit_coef(0, 7) = 7.1175857336521264E-01_dp
      fit_coef(1, 7) = 8.0417831942619766E-03_dp
      fit_coef(2, 7) = 3.3752364445702137E+00_dp
      fit_coef(3, 7) = -2.3180805628186413E+01_dp
      fit_coef(4, 7) = 1.4475705309197897E+02_dp
      fit_coef(5, 7) = -6.1407495752819284E+02_dp
      fit_coef(6, 7) = 1.8127278994439107E+03_dp
      fit_coef(7, 7) = -3.7561578028747299E+03_dp
      fit_coef(8, 7) = 5.4315325157499246E+03_dp
      fit_coef(9, 7) = -5.3571860391096889E+03_dp
      fit_coef(10, 7) = 3.4300556791845847E+03_dp
      fit_coef(11, 7) = -1.2837010342914398E+03_dp
      fit_coef(12, 7) = 2.1297330497738872E+02_dp
      fit_coef(0, 8) = 1.7130988759895882E+00_dp
      fit_coef(1, 8) = 1.3197929578399978E-02_dp
      fit_coef(2, 8) = 5.5632864218169500E+00_dp
      fit_coef(3, 8) = -3.8336822165302195E+01_dp
      fit_coef(4, 8) = 2.3871040314392795E+02_dp
      fit_coef(5, 8) = -1.0130802214079603E+03_dp
      fit_coef(6, 8) = 2.9890590755362450E+03_dp
      fit_coef(7, 8) = -6.1888221737685317E+03_dp
      fit_coef(8, 8) = 8.9422907047984954E+03_dp
      fit_coef(9, 8) = -8.8137663823846488E+03_dp
      fit_coef(10, 8) = 5.6398489234791505E+03_dp
      fit_coef(11, 8) = -2.1096575748430814E+03_dp
      fit_coef(12, 8) = 3.4985649609391197E+02_dp
      fit_coef(0, 9) = 4.0049762380285738E+00_dp
      fit_coef(1, 9) = 1.9523136227240855E-02_dp
      fit_coef(2, 9) = 8.2524358089376726E+00_dp
      fit_coef(3, 9) = -5.6989983733349561E+01_dp
      fit_coef(4, 9) = 3.5421093521195178E+02_dp
      fit_coef(5, 9) = -1.5036827242000363E+03_dp
      fit_coef(6, 9) = 4.4352294689617192E+03_dp
      fit_coef(7, 9) = -9.1786236202633227E+03_dp
      fit_coef(8, 9) = 1.3255721756847372E+04_dp
      fit_coef(9, 9) = -1.3059405760710591E+04_dp
      fit_coef(10, 9) = 8.3534120377636300E+03_dp
      fit_coef(11, 9) = -3.1236948984785458E+03_dp
      fit_coef(12, 9) = 5.1787990413580712E+02_dp
      fit_coef(0, 10) = 4.3533793103490108E-04_dp
      fit_coef(1, 10) = 4.5700924266841888E-05_dp
      fit_coef(2, 10) = 1.7699021404652904E-02_dp
      fit_coef(3, 10) = -1.1339163743096678E-01_dp
      fit_coef(4, 10) = 7.5317562249438585E-01_dp
      fit_coef(5, 10) = -3.1670388958213715E+00_dp
      fit_coef(6, 10) = 9.4488857516948261E+00_dp
      fit_coef(7, 10) = -1.9855290060320243E+01_dp
      fit_coef(8, 10) = 2.9102458261293140E+01_dp
      fit_coef(9, 10) = -2.9053092441811895E+01_dp
      fit_coef(10, 10) = 1.8796923562697987E+01_dp
      fit_coef(11, 10) = -7.0973082087264441E+00_dp
      fit_coef(12, 10) = 1.1862930929920559E+00_dp
      fit_coef(0, 11) = 2.8605561196639808E-03_dp
      fit_coef(1, 11) = 1.4711961159333449E-04_dp
      fit_coef(2, 11) = 5.9181533881525895E-02_dp
      fit_coef(3, 11) = -3.9234994207703527E-01_dp
      fit_coef(4, 11) = 2.5337109677677274E+00_dp
      fit_coef(5, 11) = -1.0698664293959414E+01_dp
      fit_coef(6, 11) = 3.1866930157362180E+01_dp
      fit_coef(7, 11) = -6.6687974631903131E+01_dp
      fit_coef(8, 11) = 9.7258475471561397E+01_dp
      fit_coef(9, 11) = -9.6610121349263864E+01_dp
      fit_coef(10, 11) = 6.2222012749124062E+01_dp
      fit_coef(11, 11) = -2.3400982843737399E+01_dp
      fit_coef(12, 11) = 3.8982563637438763E+00_dp
      fit_coef(0, 12) = 1.1710521653280541E-02_dp
      fit_coef(1, 12) = 3.8298132150440530E-04_dp
      fit_coef(2, 12) = 1.5720490880478075E-01_dp
      fit_coef(3, 12) = -1.0604827594360497E+00_dp
      fit_coef(4, 12) = 6.7293547068829866E+00_dp
      fit_coef(5, 12) = -2.8478199973116869E+01_dp
      fit_coef(6, 12) = 8.4349514492679333E+01_dp
      fit_coef(7, 12) = -1.7552622863849189E+02_dp
      fit_coef(8, 12) = 2.5489178159805036E+02_dp
      fit_coef(9, 12) = -2.5238005448515145E+02_dp
      fit_coef(10, 12) = 1.6214051051889587E+02_dp
      fit_coef(11, 12) = -6.0856023343944223E+01_dp
      fit_coef(12, 12) = 1.0120675732982292E+01_dp
      fit_coef(0, 13) = 3.8069679293470479E-02_dp
      fit_coef(1, 13) = 8.5030453244577039E-04_dp
      fit_coef(2, 13) = 3.5319383125255671E-01_dp
      fit_coef(3, 13) = -2.4058326640152856E+00_dp
      fit_coef(4, 13) = 1.5130832210336226E+01_dp
      fit_coef(5, 13) = -6.4118678617210037E+01_dp
      fit_coef(6, 13) = 1.8951627589384418E+02_dp
      fit_coef(7, 13) = -3.9344807722056498E+02_dp
      fit_coef(8, 13) = 5.7003275418870396E+02_dp
      fit_coef(9, 13) = -5.6318587947825517E+02_dp
      fit_coef(10, 13) = 3.6110912251064480E+02_dp
      fit_coef(11, 13) = -1.3530674224009485E+02_dp
      fit_coef(12, 13) = 2.2470490007659631E+01_dp
      fit_coef(0, 14) = 1.0718425215372540E-01_dp
      fit_coef(1, 14) = 1.6606498612547411E-03_dp
      fit_coef(2, 14) = 6.9558041141151850E-01_dp
      fit_coef(3, 14) = -4.7698051853919008E+00_dp
      fit_coef(4, 14) = 2.9821463877897131E+01_dp
      fit_coef(5, 14) = -1.2648112443458319E+02_dp
      fit_coef(6, 14) = 3.7339814874668650E+02_dp
      fit_coef(7, 14) = -7.7395557459000918E+02_dp
      fit_coef(8, 14) = 1.1195459460253383E+03_dp
      fit_coef(9, 14) = -1.1045564613814279E+03_dp
      fit_coef(10, 14) = 7.0739903710729982E+02_dp
      fit_coef(11, 14) = -2.6480250102979971E+02_dp
      fit_coef(12, 14) = 4.3940376198319392E+01_dp
      fit_coef(0, 15) = 2.7292854447821768E-01_dp
      fit_coef(1, 15) = 2.8818089651899471E-03_dp
      fit_coef(2, 15) = 1.2148125826442298E+00_dp
      fit_coef(3, 15) = -8.3719688971021089E+00_dp
      fit_coef(4, 15) = 5.2119215804340925E+01_dp
      fit_coef(5, 15) = -2.2119562023064080E+02_dp
      fit_coef(6, 15) = 6.5252980054868908E+02_dp
      fit_coef(7, 15) = -1.3509643415136143E+03_dp
      fit_coef(8, 15) = 1.9519497444270371E+03_dp
      fit_coef(9, 15) = -1.9238403235944149E+03_dp
      fit_coef(10, 15) = 1.2310185427991041E+03_dp
      fit_coef(11, 15) = -4.6046906916991361E+02_dp
      fit_coef(12, 15) = 7.6360956106162789E+01_dp
      fit_coef(0, 16) = 6.4599249663823077E-01_dp
      fit_coef(1, 16) = 4.4100315779845725E-03_dp
      fit_coef(2, 16) = 1.8680668872908257E+00_dp
      fit_coef(3, 16) = -1.2921923578119841E+01_dp
      fit_coef(4, 16) = 8.0194113618089176E+01_dp
      fit_coef(5, 16) = -3.4051136254025738E+02_dp
      fit_coef(6, 16) = 1.0040346645389629E+03_dp
      fit_coef(7, 16) = -2.0769701859400898E+03_dp
      fit_coef(8, 16) = 2.9983623465977071E+03_dp
      fit_coef(9, 16) = -2.9529215447188330E+03_dp
      fit_coef(10, 16) = 1.8882605454973300E+03_dp
      fit_coef(11, 16) = -7.0592135980760236E+02_dp
      fit_coef(12, 16) = 1.1701028544478041E+02_dp
      fit_coef(0, 17) = 1.4634101645424438E+00_dp
      fit_coef(1, 17) = 5.8372190193024661E-03_dp
      fit_coef(2, 17) = 2.4803142380725367E+00_dp
      fit_coef(3, 17) = -1.7197461327455997E+01_dp
      fit_coef(4, 17) = 1.0652223495438764E+02_dp
      fit_coef(5, 17) = -4.5244169342907389E+02_dp
      fit_coef(6, 17) = 1.3337313060706883E+03_dp
      fit_coef(7, 17) = -2.7575829556743033E+03_dp
      fit_coef(8, 17) = 3.9787771328072104E+03_dp
      fit_coef(9, 17) = -3.9165784027159407E+03_dp
      fit_coef(10, 17) = 2.5034305875991531E+03_dp
      fit_coef(11, 17) = -9.3557062065768866E+02_dp
      fit_coef(12, 17) = 1.5502979180126792E+02_dp
      fit_coef(0, 18) = 3.4614534817764171E+00_dp
      fit_coef(1, 18) = 6.7354426924417570E-03_dp
      fit_coef(2, 18) = 2.8636831995229377E+00_dp
      fit_coef(3, 18) = -1.9864414118527225E+01_dp
      fit_coef(4, 18) = 1.2299824240049257E+02_dp
      fit_coef(5, 18) = -5.2245197249800015E+02_dp
      fit_coef(6, 18) = 1.5400592407663903E+03_dp
      fit_coef(7, 18) = -3.1838942159350613E+03_dp
      fit_coef(8, 18) = 4.5934235919896164E+03_dp
      fit_coef(9, 18) = -4.5212045254281238E+03_dp
      fit_coef(10, 18) = 2.8896730980666093E+03_dp
      fit_coef(11, 18) = -1.0798434840197956E+03_dp
      fit_coef(12, 18) = 1.7892665893271254E+02_dp

   END SUBROUTINE get_fit_coef_k9_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k10_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 5.7695951023970724E-03_dp
      fit_coef(1, 1) = 1.5304265823073267E-02_dp
      fit_coef(2, 1) = 1.9872484454133494E-02_dp
      fit_coef(3, 1) = 1.4567559207979195E-02_dp
      fit_coef(4, 1) = 4.4496400713523688E-03_dp
      fit_coef(5, 1) = -2.0505040635942838E-03_dp
      fit_coef(6, 1) = -2.4017738586966987E-03_dp
      fit_coef(7, 1) = -2.1712350568040649E-04_dp
      fit_coef(8, 1) = 5.6984531809305442E-04_dp
      fit_coef(9, 1) = 3.6792198394363707E-04_dp
      fit_coef(10, 1) = -4.8212959819146314E-04_dp
      fit_coef(11, 1) = 2.3577845786711992E-04_dp
      fit_coef(12, 1) = -5.1828587653862264E-05_dp
      fit_coef(0, 2) = 3.1885037358175702E-02_dp
      fit_coef(1, 2) = 8.2163350099799248E-02_dp
      fit_coef(2, 2) = 1.0528892661444159E-01_dp
      fit_coef(3, 2) = 7.6258768872869365E-02_dp
      fit_coef(4, 2) = 2.2829853937072226E-02_dp
      fit_coef(5, 2) = -1.0880783303710033E-02_dp
      fit_coef(6, 2) = -1.2408166352033598E-02_dp
      fit_coef(7, 2) = -9.9928746104626666E-04_dp
      fit_coef(8, 2) = 2.9811486982521352E-03_dp
      fit_coef(9, 2) = 1.8293482458015533E-03_dp
      fit_coef(10, 2) = -2.4355182872439814E-03_dp
      fit_coef(11, 2) = 1.1814205683589125E-03_dp
      fit_coef(12, 2) = -2.5780560125925923E-04_dp
      fit_coef(0, 3) = 8.5985451133225102E-02_dp
      fit_coef(1, 3) = 2.0835675405554938E-01_dp
      fit_coef(2, 3) = 2.6074125375875318E-01_dp
      fit_coef(3, 3) = 1.8534648507356308E-01_dp
      fit_coef(4, 3) = 5.3562405566639951E-02_dp
      fit_coef(5, 3) = -2.7120042620104229E-02_dp
      fit_coef(6, 3) = -2.9327190422043962E-02_dp
      fit_coef(7, 3) = -1.7334511896438729E-03_dp
      fit_coef(8, 3) = 7.0440054029393588E-03_dp
      fit_coef(9, 3) = 4.0411619935981176E-03_dp
      fit_coef(10, 3) = -5.4238824013001187E-03_dp
      fit_coef(11, 3) = 2.5591744878845188E-03_dp
      fit_coef(12, 3) = -5.4685946652433383E-04_dp
      fit_coef(0, 4) = 1.8459848405035967E-01_dp
      fit_coef(1, 4) = 4.0371818510223983E-01_dp
      fit_coef(2, 4) = 4.8557108030340618E-01_dp
      fit_coef(3, 4) = 3.3806374668738176E-01_dp
      fit_coef(4, 4) = 9.5858897409524330E-02_dp
      fit_coef(5, 4) = -5.0962592261426340E-02_dp
      fit_coef(6, 4) = -5.3586938181428435E-02_dp
      fit_coef(7, 4) = -2.2127455334925599E-03_dp
      fit_coef(8, 4) = 1.6831832652833605E-02_dp
      fit_coef(9, 4) = 3.0255279721402920E-03_dp
      fit_coef(10, 4) = -8.9664127863715425E-03_dp
      fit_coef(11, 4) = 5.2862341907787443E-03_dp
      fit_coef(12, 4) = -1.2820544869880796E-03_dp
      fit_coef(0, 5) = 3.6074777273994402E-01_dp
      fit_coef(1, 5) = 6.8428202854170017E-01_dp
      fit_coef(2, 5) = 7.7269339647894841E-01_dp
      fit_coef(3, 5) = 5.2037619710849736E-01_dp
      fit_coef(4, 5) = 1.5184339437662456E-01_dp
      fit_coef(5, 5) = -6.9927558231047474E-02_dp
      fit_coef(6, 5) = -8.7838357618212293E-02_dp
      fit_coef(7, 5) = -2.6261703222456140E-03_dp
      fit_coef(8, 5) = 7.4641639246942270E-04_dp
      fit_coef(9, 5) = 6.0318746256968377E-02_dp
      fit_coef(10, 5) = -6.1544498098345557E-02_dp
      fit_coef(11, 5) = 2.6700632837816051E-02_dp
      fit_coef(12, 5) = -4.6151870101819167E-03_dp
      fit_coef(0, 6) = 6.7343634143757625E-01_dp
      fit_coef(1, 6) = 1.0727120778542409E+00_dp
      fit_coef(2, 6) = 1.1129907008803974E+00_dp
      fit_coef(3, 6) = 7.0510865936118905E-01_dp
      fit_coef(4, 6) = 2.1018587810883321E-01_dp
      fit_coef(5, 6) = -6.7282163797883557E-02_dp
      fit_coef(6, 6) = -1.0027943290324723E-01_dp
      fit_coef(7, 6) = -3.5384540450618285E-02_dp
      fit_coef(8, 6) = 2.7298363384797509E-02_dp
      fit_coef(9, 6) = -1.7363812432330001E-02_dp
      fit_coef(10, 6) = 7.1528999902988041E-02_dp
      fit_coef(11, 6) = -6.3212218372623016E-02_dp
      fit_coef(12, 6) = 1.7361608127688759E-02_dp
      fit_coef(0, 7) = 1.2230817482271914E+00_dp
      fit_coef(1, 7) = 1.5911587218644585E+00_dp
      fit_coef(2, 7) = 1.4985802851816239E+00_dp
      fit_coef(3, 7) = 8.6838937807453775E-01_dp
      fit_coef(4, 7) = 2.5143747366687763E-01_dp
      fit_coef(5, 7) = -4.6498700536608512E-02_dp
      fit_coef(6, 7) = -7.6008644969091751E-02_dp
      fit_coef(7, 7) = -6.7454078584185465E-02_dp
      fit_coef(8, 7) = 6.4705093608466005E-02_dp
      fit_coef(9, 7) = -1.2642157251641012E-01_dp
      fit_coef(10, 7) = 1.5991737393743985E-01_dp
      fit_coef(11, 7) = -7.4991121735080518E-02_dp
      fit_coef(12, 7) = 1.0660115126825225E-02_dp
      fit_coef(0, 8) = 2.1806860083417416E+00_dp
      fit_coef(1, 8) = 2.2497241492451168E+00_dp
      fit_coef(2, 8) = 1.9203525211709502E+00_dp
      fit_coef(3, 8) = 9.9969616591345500E-01_dp
      fit_coef(4, 8) = 2.6834037626429591E-01_dp
      fit_coef(5, 8) = -1.4414826654982551E-02_dp
      fit_coef(6, 8) = -4.5077253602132578E-02_dp
      fit_coef(7, 8) = -2.2179502828276729E-02_dp
      fit_coef(8, 8) = -4.7039179966872090E-02_dp
      fit_coef(9, 8) = 2.8411736628542274E-02_dp
      fit_coef(10, 8) = -4.1123712302139126E-02_dp
      fit_coef(11, 8) = 5.9491575141153316E-02_dp
      fit_coef(12, 8) = -2.2126040786265226E-02_dp
      fit_coef(0, 9) = 3.8549835579963414E+00_dp
      fit_coef(1, 9) = 3.0378827588214725E+00_dp
      fit_coef(2, 9) = 2.3710751843734088E+00_dp
      fit_coef(3, 9) = 1.1032804253025836E+00_dp
      fit_coef(4, 9) = 2.6679454698903354E-01_dp
      fit_coef(5, 9) = 2.0134331862044223E-02_dp
      fit_coef(6, 9) = -3.9522591450155079E-04_dp
      fit_coef(7, 9) = 7.4771871672822615E-03_dp
      fit_coef(8, 9) = -8.2968039659936169E-02_dp
      fit_coef(9, 9) = 5.7925277454710461E-02_dp
      fit_coef(10, 9) = -6.7746771231294078E-02_dp
      fit_coef(11, 9) = 5.6397993380196895E-02_dp
      fit_coef(12, 9) = -1.4614807655446010E-02_dp
      fit_coef(0, 10) = 6.9382542461265118E+00_dp
      fit_coef(1, 10) = 3.9362780471139174E+00_dp
      fit_coef(2, 10) = 2.8610007550879040E+00_dp
      fit_coef(3, 10) = 1.1997673683708834E+00_dp
      fit_coef(4, 10) = 2.5581466127865921E-01_dp
      fit_coef(5, 10) = 5.3143801132552647E-02_dp
      fit_coef(6, 10) = 5.1417759773000900E-02_dp
      fit_coef(7, 10) = 1.7678646974295765E-02_dp
      fit_coef(8, 10) = -5.9075047650367432E-02_dp
      fit_coef(9, 10) = 3.3432194957975334E-03_dp
      fit_coef(10, 10) = 3.6827535376516749E-03_dp
      fit_coef(11, 10) = 1.1242355647481125E-03_dp
      fit_coef(12, 10) = 9.1270977158394349E-04_dp
      fit_coef(0, 11) = 1.4966733342014741E-02_dp
      fit_coef(1, 11) = 3.9453275484996823E-02_dp
      fit_coef(2, 11) = 5.1068715597025638E-02_dp
      fit_coef(3, 11) = 3.7329091135728701E-02_dp
      fit_coef(4, 11) = 1.1348488247113217E-02_dp
      fit_coef(5, 11) = -5.2730834502581489E-03_dp
      fit_coef(6, 11) = -6.1356690848579278E-03_dp
      fit_coef(7, 11) = -5.3965409284480723E-04_dp
      fit_coef(8, 11) = 1.4598866172702007E-03_dp
      fit_coef(9, 11) = 9.3119411997548570E-04_dp
      fit_coef(10, 11) = -1.2246113300432057E-03_dp
      fit_coef(11, 11) = 5.9768922566995949E-04_dp
      fit_coef(12, 11) = -1.3116685473996465E-04_dp
      fit_coef(0, 12) = 3.8347028757142743E-02_dp
      fit_coef(1, 12) = 9.5250205763913104E-02_dp
      fit_coef(2, 12) = 1.2014345228064902E-01_dp
      fit_coef(3, 12) = 8.5770121052551718E-02_dp
      fit_coef(4, 12) = 2.5026676932284449E-02_dp
      fit_coef(5, 12) = -1.2425272424457500E-02_dp
      fit_coef(6, 12) = -1.3711933166334719E-02_dp
      fit_coef(7, 12) = -9.4665591048253780E-04_dp
      fit_coef(8, 12) = 3.3417995277827857E-03_dp
      fit_coef(9, 12) = 1.9446347870976362E-03_dp
      fit_coef(10, 12) = -2.6369332046548466E-03_dp
      fit_coef(11, 12) = 1.2654540755567044E-03_dp
      fit_coef(12, 12) = -2.7319897896357627E-04_dp
      fit_coef(0, 13) = 7.2491464313657286E-02_dp
      fit_coef(1, 13) = 1.5872975379092427E-01_dp
      fit_coef(2, 13) = 1.9076646279106174E-01_dp
      fit_coef(3, 13) = 1.3193323939107945E-01_dp
      fit_coef(4, 13) = 3.6136743102133911E-02_dp
      fit_coef(5, 13) = -2.0313520615877108E-02_dp
      fit_coef(6, 13) = -1.9992174365017833E-02_dp
      fit_coef(7, 13) = -1.5921233024662168E-04_dp
      fit_coef(8, 13) = 5.1319901147546939E-03_dp
      fit_coef(9, 13) = 1.3342159647477779E-03_dp
      fit_coef(10, 13) = -2.2525420201435871E-03_dp
      fit_coef(11, 13) = 9.7269009421198751E-04_dp
      fit_coef(12, 13) = -2.0108283608299456E-04_dp
      fit_coef(0, 14) = 1.3008928188788060E-01_dp
      fit_coef(1, 14) = 2.3463488136486438E-01_dp
      fit_coef(2, 14) = 2.5759089406615687E-01_dp
      fit_coef(3, 14) = 1.7130271768667638E-01_dp
      fit_coef(4, 14) = 4.9321273379861225E-02_dp
      fit_coef(5, 14) = -2.5370622060839100E-02_dp
      fit_coef(6, 14) = -2.9809532415184900E-02_dp
      fit_coef(7, 14) = -1.5417020754737089E-03_dp
      fit_coef(8, 14) = 1.0746594499561295E-02_dp
      fit_coef(9, 14) = 9.4386979195149809E-03_dp
      fit_coef(10, 14) = -1.9447726689701503E-02_dp
      fit_coef(11, 14) = 1.2247210595444594E-02_dp
      fit_coef(12, 14) = -2.9200822587582783E-03_dp
      fit_coef(0, 15) = 2.3180186914323608E-01_dp
      fit_coef(1, 15) = 3.3040402783496398E-01_dp
      fit_coef(2, 15) = 3.1506693954203840E-01_dp
      fit_coef(3, 15) = 1.8838521756146479E-01_dp
      fit_coef(4, 15) = 6.0573038457003757E-02_dp
      fit_coef(5, 15) = -8.8248929607627029E-03_dp
      fit_coef(6, 15) = -3.2842705061116521E-02_dp
      fit_coef(7, 15) = -3.8281459849134332E-03_dp
      fit_coef(8, 15) = -3.5375767464361246E-02_dp
      fit_coef(9, 15) = 6.9092174166339268E-02_dp
      fit_coef(10, 15) = -3.1110614611766781E-02_dp
      fit_coef(11, 15) = -4.5781884002109160E-03_dp
      fit_coef(12, 15) = 4.8446541181232632E-03_dp
      fit_coef(0, 16) = 4.0986431246574251E-01_dp
      fit_coef(1, 16) = 4.5035445709902655E-01_dp
      fit_coef(2, 16) = 3.6429713371972472E-01_dp
      fit_coef(3, 16) = 1.7696011433627806E-01_dp
      fit_coef(4, 16) = 5.2443940811353028E-02_dp
      fit_coef(5, 16) = 1.2731147310719145E-02_dp
      fit_coef(6, 16) = 1.2109883658619791E-02_dp
      fit_coef(7, 16) = -5.9511768374893713E-02_dp
      fit_coef(8, 16) = 9.3282598295030036E-02_dp
      fit_coef(9, 16) = -2.1484669589298355E-01_dp
      fit_coef(10, 16) = 2.5063354943196953E-01_dp
      fit_coef(11, 16) = -1.3015721063956481E-01_dp
      fit_coef(12, 16) = 2.5163533574069471E-02_dp
      fit_coef(0, 17) = 7.1698706857622740E-01_dp
      fit_coef(1, 17) = 5.8839597496590568E-01_dp
      fit_coef(2, 17) = 4.0531879096886647E-01_dp
      fit_coef(3, 17) = 1.4784602258417653E-01_dp
      fit_coef(4, 17) = 2.8867088347120023E-02_dp
      fit_coef(5, 17) = 2.7968490127980788E-02_dp
      fit_coef(6, 17) = 2.8408348764650926E-02_dp
      fit_coef(7, 17) = 1.6630805867207835E-02_dp
      fit_coef(8, 17) = -6.5625993058929091E-02_dp
      fit_coef(9, 17) = 7.6969150282882134E-02_dp
      fit_coef(10, 17) = -1.3660437598545347E-01_dp
      fit_coef(11, 17) = 1.2266105788436464E-01_dp
      fit_coef(12, 17) = -3.7327615955604555E-02_dp
      fit_coef(0, 18) = 1.2469010615236285E+00_dp
      fit_coef(1, 18) = 7.2690110706329647E-01_dp
      fit_coef(2, 18) = 4.3679155042183249E-01_dp
      fit_coef(3, 18) = 1.1545454220850887E-01_dp
      fit_coef(4, 18) = 6.1457460689058442E-03_dp
      fit_coef(5, 18) = 3.4446450178461388E-02_dp
      fit_coef(6, 18) = 3.6803538637622653E-02_dp
      fit_coef(7, 18) = 4.9357342985472417E-02_dp
      fit_coef(8, 18) = -9.8393813847182057E-02_dp
      fit_coef(9, 18) = 1.3063016463318536E-01_dp
      fit_coef(10, 18) = -1.5004086198844574E-01_dp
      fit_coef(11, 18) = 7.8940423149777589E-02_dp
      fit_coef(12, 18) = -1.3100508732995670E-02_dp
      fit_coef(0, 19) = 2.2020857144301220E+00_dp
      fit_coef(1, 19) = 8.4496482669483042E-01_dp
      fit_coef(2, 19) = 4.6567133382072923E-01_dp
      fit_coef(3, 19) = 9.5046757527473213E-02_dp
      fit_coef(4, 19) = -7.6836222188503209E-03_dp
      fit_coef(5, 19) = 3.4125967755404704E-02_dp
      fit_coef(6, 19) = 5.0617003282733740E-02_dp
      fit_coef(7, 19) = 1.2669228101081176E-02_dp
      fit_coef(8, 19) = 1.4598367047566483E-02_dp
      fit_coef(9, 19) = -4.8146800081324943E-02_dp
      fit_coef(10, 19) = 6.4386938465572141E-02_dp
      fit_coef(11, 19) = -5.7857218864180007E-02_dp
      fit_coef(12, 19) = 1.9235315360114507E-02_dp
      fit_coef(0, 20) = 4.3050004435836158E+00_dp
      fit_coef(1, 20) = 9.5976454358976238E-01_dp
      fit_coef(2, 20) = 5.2630087854492191E-01_dp
      fit_coef(3, 20) = 1.0553767791618235E-01_dp
      fit_coef(4, 20) = -1.2890279709804050E-02_dp
      fit_coef(5, 20) = 3.1631245951617148E-02_dp
      fit_coef(6, 20) = 5.1846721167766188E-02_dp
      fit_coef(7, 20) = 1.3931333624011479E-02_dp
      fit_coef(8, 20) = 1.8289419861101722E-02_dp
      fit_coef(9, 20) = -3.6941691804435302E-02_dp
      fit_coef(10, 20) = 5.2037604092826852E-02_dp
      fit_coef(11, 20) = -3.6753728963893388E-02_dp
      fit_coef(12, 20) = 8.3623158625230273E-03_dp

   END SUBROUTINE get_fit_coef_k10_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k10_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 5.6502000000000000E+04_dp
      fit_coef(0, 1) = 6.8411577604394062E-05_dp
      fit_coef(1, 1) = 1.6187597727541771E-05_dp
      fit_coef(2, 1) = 4.5871906583389663E-03_dp
      fit_coef(3, 1) = -3.0179373289852002E-02_dp
      fit_coef(4, 1) = 2.0974369374411136E-01_dp
      fit_coef(5, 1) = -8.7935873765434858E-01_dp
      fit_coef(6, 1) = 2.5878525249126527E+00_dp
      fit_coef(7, 1) = -5.3179060500186415E+00_dp
      fit_coef(8, 1) = 7.5892153487240241E+00_dp
      fit_coef(9, 1) = -7.3662068168070789E+00_dp
      fit_coef(10, 1) = 4.6344768125153601E+00_dp
      fit_coef(11, 1) = -1.7032106343424485E+00_dp
      fit_coef(12, 1) = 2.7741991549383826E-01_dp
      fit_coef(0, 2) = 7.5275323624865811E-04_dp
      fit_coef(1, 2) = 9.9232481710528875E-05_dp
      fit_coef(2, 2) = 2.8524678297215784E-02_dp
      fit_coef(3, 2) = -1.9184050122553747E-01_dp
      fit_coef(4, 2) = 1.3126341229840768E+00_dp
      fit_coef(5, 2) = -5.5140690139707527E+00_dp
      fit_coef(6, 2) = 1.6223500597815761E+01_dp
      fit_coef(7, 2) = -3.3331549263061540E+01_dp
      fit_coef(8, 2) = 4.7557073952296840E+01_dp
      fit_coef(9, 2) = -4.6143966985722287E+01_dp
      fit_coef(10, 2) = 2.9018553303790554E+01_dp
      fit_coef(11, 2) = -1.0658983662222605E+01_dp
      fit_coef(12, 2) = 1.7351738260550609E+00_dp
      fit_coef(0, 3) = 3.9843606366604320E-03_dp
      fit_coef(1, 3) = 3.3764501333670899E-04_dp
      fit_coef(2, 3) = 9.9383477377645593E-02_dp
      fit_coef(3, 3) = -6.8774784004509004E-01_dp
      fit_coef(4, 3) = 4.6197752367269072E+00_dp
      fit_coef(5, 3) = -1.9617538089460556E+01_dp
      fit_coef(6, 3) = 5.7821554084121672E+01_dp
      fit_coef(7, 3) = -1.1866755198936438E+02_dp
      fit_coef(8, 3) = 1.6899851790071165E+02_dp
      fit_coef(9, 3) = -1.6365529688728941E+02_dp
      fit_coef(10, 3) = 1.0272818227085014E+02_dp
      fit_coef(11, 3) = -3.7670599569470930E+01_dp
      fit_coef(12, 3) = 6.1231647953439321E+00_dp
      fit_coef(0, 4) = 1.5378969066141557E-02_dp
      fit_coef(1, 4) = 9.1548502345901116E-04_dp
      fit_coef(2, 4) = 2.7372675256892554E-01_dp
      fit_coef(3, 4) = -1.9285608309876876E+00_dp
      fit_coef(4, 4) = 1.2803726769725465E+01_dp
      fit_coef(5, 4) = -5.4755620729323439E+01_dp
      fit_coef(6, 4) = 1.6149496474022160E+02_dp
      fit_coef(7, 4) = -3.3133212972051763E+02_dp
      fit_coef(8, 4) = 4.7168386050627021E+02_dp
      fit_coef(9, 4) = -4.5657552176306115E+02_dp
      fit_coef(10, 4) = 2.8644979600872671E+02_dp
      fit_coef(11, 4) = -1.0497888439910341E+02_dp
      fit_coef(12, 4) = 1.7052642506098298E+01_dp
      fit_coef(0, 5) = 4.9251180618047154E-02_dp
      fit_coef(1, 5) = 2.1434954065410114E-03_dp
      fit_coef(2, 5) = 6.4789716785013773E-01_dp
      fit_coef(3, 5) = -4.6201666914403452E+00_dp
      fit_coef(4, 5) = 3.0440812558042978E+01_dp
      fit_coef(5, 5) = -1.3079799887515426E+02_dp
      fit_coef(6, 5) = 3.8607450191762678E+02_dp
      fit_coef(7, 5) = -7.9209798040256953E+02_dp
      fit_coef(8, 5) = 1.1273243950495682E+03_dp
      fit_coef(9, 5) = -1.0907316178904814E+03_dp
      fit_coef(10, 5) = 6.8394210358544069E+02_dp
      fit_coef(11, 5) = -2.5050774766365097E+02_dp
      fit_coef(12, 5) = 4.0668458941697637E+01_dp
      fit_coef(0, 6) = 1.3896177117430439E-01_dp
      fit_coef(1, 6) = 4.4858622820344507E-03_dp
      fit_coef(2, 6) = 1.3666220193198582E+00_dp
      fit_coef(3, 6) = -9.8301126837720361E+00_dp
      fit_coef(4, 6) = 6.4421339807998407E+01_dp
      fit_coef(5, 6) = -2.7772758192096933E+02_dp
      fit_coef(6, 6) = 8.2039176215178543E+02_dp
      fit_coef(7, 6) = -1.6832836477463704E+03_dp
      fit_coef(8, 6) = 2.3951662567438020E+03_dp
      fit_coef(9, 6) = -2.3166358021581445E+03_dp
      fit_coef(10, 6) = 1.4520770947679368E+03_dp
      fit_coef(11, 6) = -5.3163751269775196E+02_dp
      fit_coef(12, 6) = 8.6273624073151339E+01_dp
      fit_coef(0, 7) = 3.5729488480195148E-01_dp
      fit_coef(1, 7) = 8.5259570236280655E-03_dp
      fit_coef(2, 7) = 2.6129406592915587E+00_dp
      fit_coef(3, 7) = -1.8917534003546230E+01_dp
      fit_coef(4, 7) = 1.2348529805386671E+02_dp
      fit_coef(5, 7) = -5.3366943392175176E+02_dp
      fit_coef(6, 7) = 1.5775492634594218E+03_dp
      fit_coef(7, 7) = -3.2371204369651055E+03_dp
      fit_coef(8, 7) = 4.6054333227458374E+03_dp
      fit_coef(9, 7) = -4.4533030686941875E+03_dp
      fit_coef(10, 7) = 2.7905356324814979E+03_dp
      fit_coef(11, 7) = -1.0213693901858213E+03_dp
      fit_coef(12, 7) = 1.6569764375256605E+02_dp
      fit_coef(0, 8) = 8.5576326888027310E-01_dp
      fit_coef(1, 8) = 1.4794510243735135E-02_dp
      fit_coef(2, 8) = 4.5547377508589388E+00_dp
      fit_coef(3, 8) = -3.3139938275450689E+01_dp
      fit_coef(4, 8) = 2.1567954334326745E+02_dp
      fit_coef(5, 8) = -9.3383264595716162E+02_dp
      fit_coef(6, 8) = 2.7621813661246701E+03_dp
      fit_coef(7, 8) = -5.6685491647807048E+03_dp
      fit_coef(8, 8) = 8.0637217679143905E+03_dp
      fit_coef(9, 8) = -7.7958512685636942E+03_dp
      fit_coef(10, 8) = 4.8839651073288114E+03_dp
      fit_coef(11, 8) = -1.7871820510264045E+03_dp
      fit_coef(12, 8) = 2.8987122244652051E+02_dp
      fit_coef(0, 9) = 1.9457453939982912E+00_dp
      fit_coef(1, 9) = 2.3412967111721320E-02_dp
      fit_coef(2, 9) = 7.2322818888009728E+00_dp
      fit_coef(3, 9) = -5.2813697786989756E+01_dp
      fit_coef(4, 9) = 3.4297595366907478E+02_dp
      fit_coef(5, 9) = -1.4869895685428958E+03_dp
      fit_coef(6, 9) = 4.4006178698965341E+03_dp
      fit_coef(7, 9) = -9.0318062878397541E+03_dp
      fit_coef(8, 9) = 1.2847105139427764E+04_dp
      fit_coef(9, 9) = -1.2418582634546168E+04_dp
      fit_coef(10, 9) = 7.7787564640697719E+03_dp
      fit_coef(11, 9) = -2.8459924565854521E+03_dp
      fit_coef(12, 9) = 4.6152903795316814E+02_dp
      fit_coef(0, 10) = 4.3456834390820589E+00_dp
      fit_coef(1, 10) = 3.3830817174057760E-02_dp
      fit_coef(2, 10) = 1.0473175894252773E+01_dp
      fit_coef(3, 10) = -7.6661420311861534E+01_dp
      fit_coef(4, 10) = 4.9715082005031553E+02_dp
      fit_coef(5, 10) = -2.1572931305206675E+03_dp
      fit_coef(6, 10) = 6.3865758669205397E+03_dp
      fit_coef(7, 10) = -1.3108704020006919E+04_dp
      fit_coef(8, 10) = 1.8645335870199138E+04_dp
      fit_coef(9, 10) = -1.8021767528763816E+04_dp
      fit_coef(10, 10) = 1.1287284370105295E+04_dp
      fit_coef(11, 10) = -4.1291992041477679E+03_dp
      fit_coef(12, 10) = 6.6955320372977019E+02_dp
      fit_coef(0, 11) = 2.1871793237239991E-04_dp
      fit_coef(1, 11) = 4.1996695354716680E-05_dp
      fit_coef(2, 11) = 1.1909044603489627E-02_dp
      fit_coef(3, 11) = -7.8370180707781900E-02_dp
      fit_coef(4, 11) = 5.4437980679476261E-01_dp
      fit_coef(5, 11) = -2.2731288629006801E+00_dp
      fit_coef(6, 11) = 6.6595389300613057E+00_dp
      fit_coef(7, 11) = -1.3637699933400350E+01_dp
      fit_coef(8, 11) = 1.9415873433968578E+01_dp
      fit_coef(9, 11) = -1.8815612032886889E+01_dp
      fit_coef(10, 11) = 1.1826000388483891E+01_dp
      fit_coef(11, 11) = -4.3434643915415521E+00_dp
      fit_coef(12, 11) = 7.0720998747701980E-01_dp
      fit_coef(0, 12) = 1.4372890424941905E-03_dp
      fit_coef(1, 12) = 1.3870980955592494E-04_dp
      fit_coef(2, 12) = 4.0623488722695424E-02_dp
      fit_coef(3, 12) = -2.7976182205365724E-01_dp
      fit_coef(4, 12) = 1.8859471674798522E+00_dp
      fit_coef(5, 12) = -7.9899664094381135E+00_dp
      fit_coef(6, 12) = 2.3590888671742746E+01_dp
      fit_coef(7, 12) = -4.8520694758407572E+01_dp
      fit_coef(8, 12) = 6.9210085374286393E+01_dp
      fit_coef(9, 12) = -6.7087759398956806E+01_dp
      fit_coef(10, 12) = 4.2134889802249333E+01_dp
      fit_coef(11, 12) = -1.5455367957037941E+01_dp
      fit_coef(12, 12) = 2.5125419036554626E+00_dp
      fit_coef(0, 13) = 5.8841380648918286E-03_dp
      fit_coef(1, 13) = 3.6831095166302676E-04_dp
      fit_coef(2, 13) = 1.1027978007247614E-01_dp
      fit_coef(3, 13) = -7.7842683175317762E-01_dp
      fit_coef(4, 13) = 5.1600304775061447E+00_dp
      fit_coef(5, 13) = -2.2078165304729261E+01_dp
      fit_coef(6, 13) = 6.5091858701369759E+01_dp
      fit_coef(7, 13) = -1.3345586681486435E+02_dp
      fit_coef(8, 13) = 1.8988777441894680E+02_dp
      fit_coef(9, 13) = -1.8375507399027742E+02_dp
      fit_coef(10, 13) = 1.1527676579024002E+02_dp
      fit_coef(11, 13) = -4.2248714108699794E+01_dp
      fit_coef(12, 13) = 6.8635207622576706E+00_dp
      fit_coef(0, 14) = 1.9128947829258891E-02_dp
      fit_coef(1, 14) = 8.3862411750444427E-04_dp
      fit_coef(2, 14) = 2.5428661374901718E-01_dp
      fit_coef(3, 14) = -1.8194680686167399E+00_dp
      fit_coef(4, 14) = 1.1961590588350870E+01_dp
      fit_coef(5, 14) = -5.1467768848618732E+01_dp
      fit_coef(6, 14) = 1.5192487004386140E+02_dp
      fit_coef(7, 14) = -3.1168857906892379E+02_dp
      fit_coef(8, 14) = 4.4359497200294106E+02_dp
      fit_coef(9, 14) = -4.2917874847834634E+02_dp
      fit_coef(10, 14) = 2.6909129512594421E+02_dp
      fit_coef(11, 14) = -9.8546840419864964E+01_dp
      fit_coef(12, 14) = 1.5995921552846303E+01_dp
      fit_coef(0, 15) = 5.3855099007751053E-02_dp
      fit_coef(1, 15) = 1.6951202894427357E-03_dp
      fit_coef(2, 15) = 5.1828197057808256E-01_dp
      fit_coef(3, 15) = -3.7423907453765741E+00_dp
      fit_coef(4, 15) = 2.4465983174552605E+01_dp
      fit_coef(5, 15) = -1.0563582904908276E+02_dp
      fit_coef(6, 15) = 3.1210473924282360E+02_dp
      fit_coef(7, 15) = -6.4036928643586384E+02_dp
      fit_coef(8, 15) = 9.1110483535702542E+02_dp
      fit_coef(9, 15) = -8.8110226224210760E+02_dp
      fit_coef(10, 15) = 5.5218119224018108E+02_dp
      fit_coef(11, 15) = -2.0212810880984674E+02_dp
      fit_coef(12, 15) = 3.2795085750230733E+01_dp
      fit_coef(0, 16) = 1.3708229195964663E-01_dp
      fit_coef(1, 16) = 3.0894965681102504E-03_dp
      fit_coef(2, 16) = 9.5049659850978763E-01_dp
      fit_coef(3, 16) = -6.9100459255840390E+00_dp
      fit_coef(4, 16) = 4.4990748234228938E+01_dp
      fit_coef(5, 16) = -1.9474683678177831E+02_dp
      fit_coef(6, 16) = 5.7588070574709627E+02_dp
      fit_coef(7, 16) = -1.1817334181055951E+03_dp
      fit_coef(8, 16) = 1.6810743266760862E+03_dp
      fit_coef(9, 16) = -1.6252829227456073E+03_dp
      fit_coef(10, 16) = 1.0182493640037462E+03_dp
      fit_coef(11, 16) = -3.7261997804392706E+02_dp
      fit_coef(12, 16) = 6.0438944235281710E+01_dp
      fit_coef(0, 17) = 3.2363428278830764E-01_dp
      fit_coef(1, 17) = 5.0854788099187554E-03_dp
      fit_coef(2, 17) = 1.5721133863814079E+00_dp
      fit_coef(3, 17) = -1.1489215181637956E+01_dp
      fit_coef(4, 17) = 7.4573534244656926E+01_dp
      fit_coef(5, 17) = -3.2342087496454349E+02_dp
      fit_coef(6, 17) = 9.5710734682470525E+02_dp
      fit_coef(7, 17) = -1.9642973854914649E+03_dp
      fit_coef(8, 17) = 2.7939950801758901E+03_dp
      fit_coef(9, 17) = -2.7007178237082067E+03_dp
      fit_coef(10, 17) = 1.6916202883905507E+03_dp
      fit_coef(11, 17) = -6.1888646191941052E+02_dp
      fit_coef(12, 17) = 1.0035990724886408E+02_dp
      fit_coef(0, 18) = 7.2227658357344648E-01_dp
      fit_coef(1, 18) = 7.4788858723735070E-03_dp
      fit_coef(2, 18) = 2.3204406383068310E+00_dp
      fit_coef(3, 18) = -1.7025391312911438E+01_dp
      fit_coef(4, 18) = 1.1025201945927834E+02_dp
      fit_coef(5, 18) = -4.7884421266123894E+02_dp
      fit_coef(6, 18) = 1.4179656218358291E+03_dp
      fit_coef(7, 18) = -2.9105197632238601E+03_dp
      fit_coef(8, 18) = 4.1395786832377917E+03_dp
      fit_coef(9, 18) = -4.0007691450093103E+03_dp
      fit_coef(10, 18) = 2.5054759345520029E+03_dp
      fit_coef(11, 18) = -9.1647339178452830E+02_dp
      fit_coef(12, 18) = 1.4859116561702967E+02_dp
      fit_coef(0, 19) = 1.5633146329155199E+00_dp
      fit_coef(1, 19) = 9.6549690016911830E-03_dp
      fit_coef(2, 19) = 3.0024364887949786E+00_dp
      fit_coef(3, 19) = -2.2083956048519831E+01_dp
      fit_coef(4, 19) = 1.4280565323321272E+02_dp
      fit_coef(5, 19) = -6.2078293004938610E+02_dp
      fit_coef(6, 19) = 1.8390901346312414E+03_dp
      fit_coef(7, 19) = -3.7753165401987344E+03_dp
      fit_coef(8, 19) = 5.3693456762596288E+03_dp
      fit_coef(9, 19) = -5.1888078370584299E+03_dp
      fit_coef(10, 19) = 3.2491199407176041E+03_dp
      fit_coef(11, 19) = -1.1883539144960864E+03_dp
      fit_coef(12, 19) = 1.9265088532272543E+02_dp
      fit_coef(0, 20) = 3.5765140128766371E+00_dp
      fit_coef(1, 20) = 1.1092017818854312E-02_dp
      fit_coef(2, 20) = 3.4506220465727910E+00_dp
      fit_coef(3, 20) = -2.5391055124716122E+01_dp
      fit_coef(4, 20) = 1.6415262153898755E+02_dp
      fit_coef(5, 20) = -7.1368338873477956E+02_dp
      fit_coef(6, 20) = 2.1144976599358906E+03_dp
      fit_coef(7, 20) = -4.3407867181744632E+03_dp
      fit_coef(8, 20) = 6.1735437097235817E+03_dp
      fit_coef(9, 20) = -5.9658726159362077E+03_dp
      fit_coef(10, 20) = 3.7356297419849939E+03_dp
      fit_coef(11, 20) = -1.3662663971818399E+03_dp
      fit_coef(12, 20) = 2.2148914181168524E+02_dp

   END SUBROUTINE get_fit_coef_k10_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k11_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 5.2171158533482745E-03_dp
      fit_coef(1, 1) = 1.3928278169011961E-02_dp
      fit_coef(2, 1) = 1.8150859480949404E-02_dp
      fit_coef(3, 1) = 1.3341207416860837E-02_dp
      fit_coef(4, 1) = 4.0847907407787400E-03_dp
      fit_coef(5, 1) = -1.8862057580171247E-03_dp
      fit_coef(6, 1) = -2.2005927300723840E-03_dp
      fit_coef(7, 1) = -2.1569598773101882E-04_dp
      fit_coef(8, 1) = 5.3374114714500355E-04_dp
      fit_coef(9, 1) = 3.6786248990713488E-04_dp
      fit_coef(10, 1) = -4.9716456588384794E-04_dp
      fit_coef(11, 1) = 2.5364063357271070E-04_dp
      fit_coef(12, 1) = -5.7560771366250216E-05_dp
      fit_coef(0, 2) = 2.8561321333081039E-02_dp
      fit_coef(1, 2) = 7.4577106385596906E-02_dp
      fit_coef(2, 2) = 9.6095992956011567E-02_dp
      fit_coef(3, 2) = 6.9921004219163585E-02_dp
      fit_coef(4, 2) = 2.1046006027983205E-02_dp
      fit_coef(5, 2) = -1.0007180795562516E-02_dp
      fit_coef(6, 2) = -1.1398359734540125E-02_dp
      fit_coef(7, 2) = -1.0344153575843251E-03_dp
      fit_coef(8, 2) = 2.8187351548754256E-03_dp
      fit_coef(9, 2) = 1.8266346026223723E-03_dp
      fit_coef(10, 2) = -2.5228015867621856E-03_dp
      fit_coef(11, 2) = 1.2855087989079601E-03_dp
      fit_coef(12, 2) = -2.9136730780610450E-04_dp
      fit_coef(0, 3) = 7.5565672474912859E-02_dp
      fit_coef(1, 3) = 1.8837838893012882E-01_dp
      fit_coef(2, 3) = 2.3788031159177950E-01_dp
      fit_coef(3, 3) = 1.7019649338979928E-01_dp
      fit_coef(4, 3) = 4.9702597603960480E-02_dp
      fit_coef(5, 3) = -2.4843028083216520E-02_dp
      fit_coef(6, 3) = -2.7107463499384012E-02_dp
      fit_coef(7, 3) = -2.0998400355075556E-03_dp
      fit_coef(8, 3) = 6.8537414053824698E-03_dp
      fit_coef(9, 3) = 4.1037972870635875E-03_dp
      fit_coef(10, 3) = -5.8343544319931773E-03_dp
      fit_coef(11, 3) = 2.9563811258876472E-03_dp
      fit_coef(12, 3) = -6.6659895709530450E-04_dp
      fit_coef(0, 4) = 1.5746184306040770E-01_dp
      fit_coef(1, 4) = 3.6307100829710298E-01_dp
      fit_coef(2, 4) = 4.4411067220455136E-01_dp
      fit_coef(3, 4) = 3.1156735023866838E-01_dp
      fit_coef(4, 4) = 8.8318376917750713E-02_dp
      fit_coef(5, 4) = -4.6905172790974697E-02_dp
      fit_coef(6, 4) = -4.8633790525923246E-02_dp
      fit_coef(7, 4) = -2.6374461756591546E-03_dp
      fit_coef(8, 4) = 1.4152819458865665E-02_dp
      fit_coef(9, 4) = 4.0713403471031064E-03_dp
      fit_coef(10, 4) = -8.1648840133065614E-03_dp
      fit_coef(11, 4) = 4.4172781177938061E-03_dp
      fit_coef(12, 4) = -1.0468849549327278E-03_dp
      fit_coef(0, 5) = 2.9628095459795201E-01_dp
      fit_coef(1, 5) = 6.1067548081671130E-01_dp
      fit_coef(2, 5) = 7.1073485839444905E-01_dp
      fit_coef(3, 5) = 4.8623515872109696E-01_dp
      fit_coef(4, 5) = 1.3873344982657515E-01_dp
      fit_coef(5, 5) = -7.1183381958692823E-02_dp
      fit_coef(6, 5) = -7.9018683634448778E-02_dp
      fit_coef(7, 5) = -4.1849522345858052E-03_dp
      fit_coef(8, 5) = 1.8669075914354103E-02_dp
      fit_coef(9, 5) = 2.5640549363637480E-02_dp
      fit_coef(10, 5) = -3.7238167104347006E-02_dp
      fit_coef(11, 5) = 2.0235521804447935E-02_dp
      fit_coef(12, 5) = -4.4633644085237409E-03_dp
      fit_coef(0, 6) = 5.3079425354928900E-01_dp
      fit_coef(1, 6) = 9.4909599974912540E-01_dp
      fit_coef(2, 6) = 1.0298933488572886E+00_dp
      fit_coef(3, 6) = 6.7464315041251843E-01_dp
      fit_coef(4, 6) = 1.9785397712598238E-01_dp
      fit_coef(5, 6) = -8.0277433825610117E-02_dp
      fit_coef(6, 6) = -1.0862148180963889E-01_dp
      fit_coef(7, 6) = -1.2190533137530357E-02_dp
      fit_coef(8, 6) = -4.3235690156727307E-03_dp
      fit_coef(9, 6) = 7.1406154351795634E-02_dp
      fit_coef(10, 6) = -4.7520764363596951E-02_dp
      fit_coef(11, 6) = 6.9200248215417976E-03_dp
      fit_coef(12, 6) = 1.7348958867823125E-03_dp
      fit_coef(0, 7) = 9.2513564519354430E-01_dp
      fit_coef(1, 7) = 1.3995257053484436E+00_dp
      fit_coef(2, 7) = 1.3941989426918036E+00_dp
      fit_coef(3, 7) = 8.5298558301770655E-01_dp
      fit_coef(4, 7) = 2.5029080019290989E-01_dp
      fit_coef(5, 7) = -6.9418886584503764E-02_dp
      fit_coef(6, 7) = -1.0048676489518746E-01_dp
      fit_coef(7, 7) = -6.5140068180983265E-02_dp
      fit_coef(8, 7) = 7.1961159826162302E-02_dp
      fit_coef(9, 7) = -1.0845933619226010E-01_dp
      fit_coef(10, 7) = 1.7018602738764302E-01_dp
      fit_coef(11, 7) = -1.0954176694571671E-01_dp
      fit_coef(12, 7) = 2.4968149530473070E-02_dp
      fit_coef(0, 8) = 1.5835046294778308E+00_dp
      fit_coef(1, 8) = 1.9791294738794438E+00_dp
      fit_coef(2, 8) = 1.7961980138940286E+00_dp
      fit_coef(3, 8) = 1.0051925409840725E+00_dp
      fit_coef(4, 8) = 2.8272786197343069E-01_dp
      fit_coef(5, 8) = -4.3458147169371041E-02_dp
      fit_coef(6, 8) = -7.1388547183826534E-02_dp
      fit_coef(7, 8) = -6.9105034008714997E-02_dp
      fit_coef(8, 8) = 4.9061162554567488E-02_dp
      fit_coef(9, 8) = -1.0568300110809625E-01_dp
      fit_coef(10, 8) = 1.1802618075924728E-01_dp
      fit_coef(11, 8) = -3.4740368047083123E-02_dp
      fit_coef(12, 8) = -2.0070688342111883E-03_dp
      fit_coef(0, 9) = 2.6780451266380028E+00_dp
      fit_coef(1, 9) = 2.6926034999618897E+00_dp
      fit_coef(2, 9) = 2.2273953918391149E+00_dp
      fit_coef(3, 9) = 1.1263680392434321E+00_dp
      fit_coef(4, 9) = 2.9395288063392006E-01_dp
      fit_coef(5, 9) = -8.9510678035254702E-03_dp
      fit_coef(6, 9) = -3.6852206543507286E-02_dp
      fit_coef(7, 9) = -2.0316323307418317E-02_dp
      fit_coef(8, 9) = -6.2348519142117052E-02_dp
      fit_coef(9, 9) = 5.0424313584454393E-02_dp
      fit_coef(10, 9) = -7.4016849780124558E-02_dp
      fit_coef(11, 9) = 8.1891720384978092E-02_dp
      fit_coef(12, 9) = -2.7202540553906623E-02_dp
      fit_coef(0, 10) = 4.5149548116067084E+00_dp
      fit_coef(1, 10) = 3.5279902941181556E+00_dp
      fit_coef(2, 10) = 2.6834197969535900E+00_dp
      fit_coef(3, 10) = 1.2234297318240759E+00_dp
      fit_coef(4, 10) = 2.9061691203379159E-01_dp
      fit_coef(5, 10) = 2.6998851164527955E-02_dp
      fit_coef(6, 10) = 1.1503151087891473E-02_dp
      fit_coef(7, 10) = -2.9263858448428439E-04_dp
      fit_coef(8, 10) = -7.4172023234369305E-02_dp
      fit_coef(9, 10) = 4.4507509039308939E-02_dp
      fit_coef(10, 10) = -6.0661296025507475E-02_dp
      fit_coef(11, 10) = 5.4037676664233973E-02_dp
      fit_coef(12, 10) = -1.3914979544265946E-02_dp
      fit_coef(0, 11) = 7.7831432490395818E+00_dp
      fit_coef(1, 11) = 4.4728899281150136E+00_dp
      fit_coef(2, 11) = 3.1793306487074551E+00_dp
      fit_coef(3, 11) = 1.3166282129264584E+00_dp
      fit_coef(4, 11) = 2.7989649807784495E-01_dp
      fit_coef(5, 11) = 6.1709167955964331E-02_dp
      fit_coef(6, 11) = 6.5030571914413512E-02_dp
      fit_coef(7, 11) = 5.8652910860924629E-03_dp
      fit_coef(8, 11) = -4.3479817133538171E-02_dp
      fit_coef(9, 11) = -1.5946703925855100E-02_dp
      fit_coef(10, 11) = 1.4840762855674984E-02_dp
      fit_coef(11, 11) = -1.6764580405545060E-03_dp
      fit_coef(12, 11) = 1.0361409280594497E-03_dp
      fit_coef(0, 12) = 1.3505623056792662E-02_dp
      fit_coef(1, 12) = 3.5881778901904388E-02_dp
      fit_coef(2, 12) = 4.6635585408641191E-02_dp
      fit_coef(3, 12) = 3.4196109564738703E-02_dp
      fit_coef(4, 12) = 1.0427820008519861E-02_dp
      fit_coef(5, 12) = -4.8496513706743740E-03_dp
      fit_coef(6, 12) = -5.6249874188544387E-03_dp
      fit_coef(7, 12) = -5.4152291061643394E-04_dp
      fit_coef(8, 12) = 1.3707635274310744E-03_dp
      fit_coef(9, 12) = 9.3081024123118999E-04_dp
      fit_coef(10, 12) = -1.2643999344426028E-03_dp
      fit_coef(11, 12) = 6.4488398114361249E-04_dp
      fit_coef(12, 12) = -1.4631445736861331E-04_dp
      fit_coef(0, 13) = 3.3953336933937019E-02_dp
      fit_coef(1, 13) = 8.6200605204820452E-02_dp
      fit_coef(2, 13) = 1.0956860661445041E-01_dp
      fit_coef(3, 13) = 7.8748352145278117E-02_dp
      fit_coef(4, 13) = 2.3201917648793789E-02_dp
      fit_coef(5, 13) = -1.1428692068356564E-02_dp
      fit_coef(6, 13) = -1.2646975774724427E-02_dp
      fit_coef(7, 13) = -1.0364501016809282E-03_dp
      fit_coef(8, 13) = 3.2018633266795151E-03_dp
      fit_coef(9, 13) = 1.9223995549073674E-03_dp
      fit_coef(10, 13) = -2.7320063801979958E-03_dp
      fit_coef(11, 13) = 1.3899117897301057E-03_dp
      fit_coef(12, 13) = -3.1441281031980347E-04_dp
      fit_coef(0, 14) = 6.1851572757860851E-02_dp
      fit_coef(1, 14) = 1.4267393289529001E-01_dp
      fit_coef(2, 14) = 1.7418539035135686E-01_dp
      fit_coef(3, 14) = 1.2141235983496969E-01_dp
      fit_coef(4, 14) = 3.3706874354447809E-02_dp
      fit_coef(5, 14) = -1.8360609527326543E-02_dp
      fit_coef(6, 14) = -1.8541339179191114E-02_dp
      fit_coef(7, 14) = -8.8529750610819111E-04_dp
      fit_coef(8, 14) = 4.7747853958221521E-03_dp
      fit_coef(9, 14) = 2.4689097936542491E-03_dp
      fit_coef(10, 14) = -3.6719910751692379E-03_dp
      fit_coef(11, 14) = 1.8252247936231616E-03_dp
      fit_coef(12, 14) = -4.0633639834620388E-04_dp
      fit_coef(0, 15) = 1.0551618877519962E-01_dp
      fit_coef(1, 15) = 2.0866982892328420E-01_dp
      fit_coef(2, 15) = 2.3758645562105263E-01_dp
      fit_coef(3, 15) = 1.6022711660180100E-01_dp
      fit_coef(4, 15) = 4.3995832787546506E-02_dp
      fit_coef(5, 15) = -2.5306890576195295E-02_dp
      fit_coef(6, 15) = -2.5168288399177011E-02_dp
      fit_coef(7, 15) = -9.3194538735970800E-04_dp
      fit_coef(8, 15) = 1.1886232926624151E-02_dp
      fit_coef(9, 15) = -3.1847717078905507E-03_dp
      fit_coef(10, 15) = -2.3213252827938664E-03_dp
      fit_coef(11, 15) = 2.4083825205789490E-03_dp
      fit_coef(12, 15) = -7.2529680267051182E-04_dp
      fit_coef(0, 16) = 1.7844852042063208E-01_dp
      fit_coef(1, 16) = 2.8956711047461353E-01_dp
      fit_coef(2, 16) = 2.9424320012524147E-01_dp
      fit_coef(3, 16) = 1.8566958335038000E-01_dp
      fit_coef(4, 16) = 5.6552619188388630E-02_dp
      fit_coef(5, 16) = -1.9536083908459775E-02_dp
      fit_coef(6, 16) = -3.5122564007741566E-02_dp
      fit_coef(7, 16) = -1.5160502263432738E-04_dp
      fit_coef(8, 16) = -1.8782648209524570E-02_dp
      fit_coef(9, 16) = 6.7898727329487735E-02_dp
      fit_coef(10, 16) = -6.6952173441897606E-02_dp
      fit_coef(11, 16) = 3.0244889278910881E-02_dp
      fit_coef(12, 16) = -5.4548300942684411E-03_dp
      fit_coef(0, 17) = 3.0103516162173788E-01_dp
      fit_coef(1, 17) = 3.9093832828366182E-01_dp
      fit_coef(2, 17) = 3.4287403235055175E-01_dp
      fit_coef(3, 17) = 1.8694079224619659E-01_dp
      fit_coef(4, 17) = 5.8759826809024841E-02_dp
      fit_coef(5, 17) = 1.8203291052277864E-03_dp
      fit_coef(6, 17) = -1.4847774478906702E-02_dp
      fit_coef(7, 17) = -3.0225726948560502E-02_dp
      fit_coef(8, 17) = 1.7580090339776970E-02_dp
      fit_coef(9, 17) = -6.3682901560076013E-02_dp
      fit_coef(10, 17) = 1.2826078376018740E-01_dp
      fit_coef(11, 17) = -9.3634524667592312E-02_dp
      fit_coef(12, 17) = 2.4194956180134675E-02_dp
      fit_coef(0, 18) = 5.0459047494440223E-01_dp
      fit_coef(1, 18) = 5.1293218387604667E-01_dp
      fit_coef(2, 18) = 3.8452848514240090E-01_dp
      fit_coef(3, 18) = 1.6703912013390276E-01_dp
      fit_coef(4, 18) = 4.3712588651811067E-02_dp
      fit_coef(5, 18) = 1.9051249953627954E-02_dp
      fit_coef(6, 18) = 2.5883902146128887E-02_dp
      fit_coef(7, 18) = -5.0770600600935537E-02_dp
      fit_coef(8, 18) = 7.7584684197698039E-02_dp
      fit_coef(9, 18) = -1.7926872234928357E-01_dp
      fit_coef(10, 18) = 1.6678298865222213E-01_dp
      fit_coef(11, 18) = -5.5645139244720254E-02_dp
      fit_coef(12, 18) = 2.7420024112079135E-03_dp
      fit_coef(0, 19) = 8.3992248227121313E-01_dp
      fit_coef(1, 19) = 6.4715555436372529E-01_dp
      fit_coef(2, 19) = 4.1796785083991977E-01_dp
      fit_coef(3, 19) = 1.3654955607272076E-01_dp
      fit_coef(4, 19) = 2.1070840652137535E-02_dp
      fit_coef(5, 19) = 3.1743033314222624E-02_dp
      fit_coef(6, 19) = 2.9784087155825711E-02_dp
      fit_coef(7, 19) = 4.1081836375772660E-02_dp
      fit_coef(8, 19) = -1.1439519939475226E-01_dp
      fit_coef(9, 19) = 1.6305361885018524E-01_dp
      fit_coef(10, 19) = -2.2346150468788073E-01_dp
      fit_coef(11, 19) = 1.6036296981154594E-01_dp
      fit_coef(12, 19) = -4.2164356684994284E-02_dp
      fit_coef(0, 20) = 1.3973915504793548E+00_dp
      fit_coef(1, 20) = 7.7764466691628698E-01_dp
      fit_coef(2, 20) = 4.4348607986397731E-01_dp
      fit_coef(3, 20) = 1.0694241439192023E-01_dp
      fit_coef(4, 20) = 2.6327294220292721E-03_dp
      fit_coef(5, 20) = 3.5966741485323687E-02_dp
      fit_coef(6, 20) = 4.1705374001031803E-02_dp
      fit_coef(7, 20) = 3.8907029803020722E-02_dp
      fit_coef(8, 20) = -6.8120058033489070E-02_dp
      fit_coef(9, 20) = 8.2786813906914350E-02_dp
      fit_coef(10, 20) = -9.1368844284995504E-02_dp
      fit_coef(11, 20) = 3.7752943929935098E-02_dp
      fit_coef(12, 20) = -1.9232870385022373E-03_dp
      fit_coef(0, 21) = 2.3760047811723362E+00_dp
      fit_coef(1, 21) = 8.8976406037305356E-01_dp
      fit_coef(2, 21) = 4.7037419059997421E-01_dp
      fit_coef(3, 21) = 9.0428943264851042E-02_dp
      fit_coef(4, 21) = -8.0951525285273343E-03_dp
      fit_coef(5, 21) = 3.5586564490451805E-02_dp
      fit_coef(6, 21) = 5.2837656508520334E-02_dp
      fit_coef(7, 21) = 6.4161552882951054E-03_dp
      fit_coef(8, 21) = 2.7554689120903888E-02_dp
      fit_coef(9, 21) = -6.3909420973179670E-02_dp
      fit_coef(10, 21) = 7.9771394321258737E-02_dp
      fit_coef(11, 21) = -6.5034308453826573E-02_dp
      fit_coef(12, 21) = 2.0094355264141887E-02_dp
      fit_coef(0, 22) = 4.5028432526721405E+00_dp
      fit_coef(1, 22) = 1.0114482591044458E+00_dp
      fit_coef(2, 22) = 5.3486119037036706E-01_dp
      fit_coef(3, 22) = 1.0317932221763447E-01_dp
      fit_coef(4, 22) = -1.2314024715229079E-02_dp
      fit_coef(5, 22) = 3.3690522473128488E-02_dp
      fit_coef(6, 22) = 5.3663513056946451E-02_dp
      fit_coef(7, 22) = 1.0785244432659307E-02_dp
      fit_coef(8, 22) = 2.2642841476459809E-02_dp
      fit_coef(9, 22) = -3.9218764804072916E-02_dp
      fit_coef(10, 22) = 5.2080967974292493E-02_dp
      fit_coef(11, 22) = -3.5217909102649027E-02_dp
      fit_coef(12, 22) = 7.5721478110433137E-03_dp

   END SUBROUTINE get_fit_coef_k11_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k11_mi(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.1533853671352736E+03_dp
      fit_coef(0, 1) = 7.3828894587718401E-04_dp
      fit_coef(1, 1) = 2.0775080037103563E-03_dp
      fit_coef(2, 1) = 2.9605504652653850E-03_dp
      fit_coef(3, 1) = 2.5239761597293463E-03_dp
      fit_coef(4, 1) = 1.1723403873630633E-03_dp
      fit_coef(5, 1) = 4.6491653525586301E-05_dp
      fit_coef(6, 1) = -2.5819644062415552E-04_dp
      fit_coef(7, 1) = -3.9831509458327039E-05_dp
      fit_coef(8, 1) = -4.0413534671757017E-05_dp
      fit_coef(9, 1) = 2.5700229767237757E-04_dp
      fit_coef(10, 1) = -2.8744580127669869E-04_dp
      fit_coef(11, 1) = 1.4858790649392180E-04_dp
      fit_coef(12, 1) = -3.1394148877043504E-05_dp
      fit_coef(0, 2) = 4.3620891975219853E-03_dp
      fit_coef(1, 2) = 1.1465965739592934E-02_dp
      fit_coef(2, 2) = 1.6004195686742342E-02_dp
      fit_coef(3, 2) = 1.3390829030018066E-02_dp
      fit_coef(4, 2) = 6.0723492355871950E-03_dp
      fit_coef(5, 2) = 1.7271227012111563E-04_dp
      fit_coef(6, 2) = -1.3295013600067558E-03_dp
      fit_coef(7, 2) = -2.0819503017025333E-04_dp
      fit_coef(8, 2) = -1.9485312280291003E-04_dp
      fit_coef(9, 2) = 1.3461436514495772E-03_dp
      fit_coef(10, 2) = -1.5381432319053760E-03_dp
      fit_coef(11, 2) = 8.0871429267979347E-04_dp
      fit_coef(12, 2) = -1.7336433357221408E-04_dp
      fit_coef(0, 3) = 1.3393965217340013E-02_dp
      fit_coef(1, 3) = 3.0758021534147448E-02_dp
      fit_coef(2, 3) = 4.0863508679567187E-02_dp
      fit_coef(3, 3) = 3.3137344564780624E-02_dp
      fit_coef(4, 3) = 1.4642382183959756E-02_dp
      fit_coef(5, 3) = 1.8546563333903454E-04_dp
      fit_coef(6, 3) = -3.2981507356429299E-03_dp
      fit_coef(7, 3) = -5.0371329478405473E-04_dp
      fit_coef(8, 3) = -3.3526812040834221E-04_dp
      fit_coef(9, 3) = 3.3638986135397790E-03_dp
      fit_coef(10, 3) = -4.0699164251672606E-03_dp
      fit_coef(11, 3) = 2.2132500698992790E-03_dp
      fit_coef(12, 3) = -4.8530598400547189E-04_dp
      fit_coef(0, 4) = 3.4178972479544684E-02_dp
      fit_coef(1, 4) = 6.5546031770227484E-02_dp
      fit_coef(2, 4) = 8.0042248964935656E-02_dp
      fit_coef(3, 4) = 6.0702076580615758E-02_dp
      fit_coef(4, 4) = 2.5719498592455581E-02_dp
      fit_coef(5, 4) = 6.3119790847243661E-04_dp
      fit_coef(6, 4) = -5.4349702628550432E-03_dp
      fit_coef(7, 4) = -1.2659291521318623E-03_dp
      fit_coef(8, 4) = -1.2216416836694759E-03_dp
      fit_coef(9, 4) = 6.6635331234771687E-03_dp
      fit_coef(10, 4) = -7.2400923612061709E-03_dp
      fit_coef(11, 4) = 3.7136157185037903E-03_dp
      fit_coef(12, 4) = -7.8916627439796453E-04_dp
      fit_coef(0, 5) = 8.0104328961022445E-02_dp
      fit_coef(1, 5) = 1.2586958566619899E-01_dp
      fit_coef(2, 5) = 1.3877416617876437E-01_dp
      fit_coef(3, 5) = 9.4435072367624009E-02_dp
      fit_coef(4, 5) = 3.5574728421505095E-02_dp
      fit_coef(5, 5) = 4.6709813565982229E-04_dp
      fit_coef(6, 5) = -5.5823066584693949E-03_dp
      fit_coef(7, 5) = -2.2508074433607930E-03_dp
      fit_coef(8, 5) = -4.8236909977057691E-04_dp
      fit_coef(9, 5) = 5.8248465053641401E-03_dp
      fit_coef(10, 5) = -7.4300192099394369E-03_dp
      fit_coef(11, 5) = 4.5059815374626708E-03_dp
      fit_coef(12, 5) = -1.1055416581292031E-03_dp
      fit_coef(0, 6) = 1.7731760423288304E-01_dp
      fit_coef(1, 6) = 2.2598655727257688E-01_dp
      fit_coef(2, 6) = 2.2430968436915458E-01_dp
      fit_coef(3, 6) = 1.3311060621110593E-01_dp
      fit_coef(4, 6) = 4.0488867018282172E-02_dp
      fit_coef(5, 6) = -2.1669989611095342E-03_dp
      fit_coef(6, 6) = -3.8187866091584140E-03_dp
      fit_coef(7, 6) = -2.1306428597222633E-03_dp
      fit_coef(8, 6) = 7.7683777527477076E-04_dp
      fit_coef(9, 6) = 4.9822370243334966E-03_dp
      fit_coef(10, 6) = -9.5726170222928900E-03_dp
      fit_coef(11, 6) = 6.8634964686467978E-03_dp
      fit_coef(12, 6) = -1.7999298724748951E-03_dp
      fit_coef(0, 7) = 3.7517258865761766E-01_dp
      fit_coef(1, 7) = 3.8333519695677720E-01_dp
      fit_coef(2, 7) = 3.4375707015157120E-01_dp
      fit_coef(3, 7) = 1.7423153407071543E-01_dp
      fit_coef(4, 7) = 3.7337456774139702E-02_dp
      fit_coef(5, 7) = -7.4660271581285850E-03_dp
      fit_coef(6, 7) = 5.8400107303188187E-04_dp
      fit_coef(7, 7) = -2.5244872448215655E-03_dp
      fit_coef(8, 7) = 4.2884385157431048E-03_dp
      fit_coef(9, 7) = 1.0453917543656458E-03_dp
      fit_coef(10, 7) = -9.2913505401427225E-03_dp
      fit_coef(11, 7) = 8.1349139258150831E-03_dp
      fit_coef(12, 7) = -2.3025720564300821E-03_dp
      fit_coef(0, 8) = 7.6460515920837613E-01_dp
      fit_coef(1, 8) = 6.1549142645707966E-01_dp
      fit_coef(2, 8) = 5.0182800509604020E-01_dp
      fit_coef(3, 8) = 2.1325320085467817E-01_dp
      fit_coef(4, 8) = 2.3360017105729571E-02_dp
      fit_coef(5, 8) = -1.4054384598267826E-02_dp
      fit_coef(6, 8) = 9.0661378414736218E-03_dp
      fit_coef(7, 8) = -4.7433494326856486E-03_dp
      fit_coef(8, 8) = 1.0160041902537189E-02_dp
      fit_coef(9, 8) = -6.9427342461860627E-03_dp
      fit_coef(10, 8) = -5.4907207434525202E-03_dp
      fit_coef(11, 8) = 8.0763791261621969E-03_dp
      fit_coef(12, 8) = -2.6484224072421108E-03_dp
      fit_coef(0, 9) = 1.5114021780363067E+00_dp
      fit_coef(1, 9) = 9.3368349741314893E-01_dp
      fit_coef(2, 9) = 6.9854171505894413E-01_dp
      fit_coef(3, 9) = 2.4498257864430506E-01_dp
      fit_coef(4, 9) = -2.7117345776915117E-03_dp
      fit_coef(5, 9) = -1.9684397407321690E-02_dp
      fit_coef(6, 9) = 2.2750102305246520E-02_dp
      fit_coef(7, 9) = -9.9596165022990610E-03_dp
      fit_coef(8, 9) = 1.8166084113835305E-02_dp
      fit_coef(9, 9) = -1.9043752557357005E-02_dp
      fit_coef(10, 9) = 2.2864257000369307E-03_dp
      fit_coef(11, 9) = 6.3623059846125004E-03_dp
      fit_coef(12, 9) = -2.7468174322941873E-03_dp
      fit_coef(0, 10) = 2.9272838679658890E+00_dp
      fit_coef(1, 10) = 1.3344095709119908E+00_dp
      fit_coef(2, 10) = 9.2872428481763736E-01_dp
      fit_coef(3, 10) = 2.6731240098857367E-01_dp
      fit_coef(4, 10) = -3.9323628602554263E-02_dp
      fit_coef(5, 10) = -2.2664729979783912E-02_dp
      fit_coef(6, 10) = 4.1418459538349613E-02_dp
      fit_coef(7, 10) = -1.8765822971615966E-02_dp
      fit_coef(8, 10) = 2.8023480739935883E-02_dp
      fit_coef(9, 10) = -3.4574149384271238E-02_dp
      fit_coef(10, 10) = 1.3734227321286851E-02_dp
      fit_coef(11, 10) = 2.9343888906998264E-03_dp
      fit_coef(12, 10) = -2.5433341475355697E-03_dp
      fit_coef(0, 11) = 5.7131838450036696E+00_dp
      fit_coef(1, 11) = 1.8005538920264632E+00_dp
      fit_coef(2, 11) = 1.1880536006356908E+00_dp
      fit_coef(3, 11) = 2.8559106021602204E-01_dp
      fit_coef(4, 11) = -8.2639115843313488E-02_dp
      fit_coef(5, 11) = -2.3761780940649874E-02_dp
      fit_coef(6, 11) = 6.3493180678645345E-02_dp
      fit_coef(7, 11) = -3.0273355255164173E-02_dp
      fit_coef(8, 11) = 3.9310329422917428E-02_dp
      fit_coef(9, 11) = -5.2371339180397693E-02_dp
      fit_coef(10, 11) = 2.7484059561085201E-02_dp
      fit_coef(11, 11) = -1.5309209297749292E-03_dp
      fit_coef(12, 11) = -2.1659478188002048E-03_dp
      fit_coef(0, 12) = 1.9426697312926121E-03_dp
      fit_coef(1, 12) = 5.3902355339823542E-03_dp
      fit_coef(2, 12) = 7.6460050852952361E-03_dp
      fit_coef(3, 12) = 6.4861449445603606E-03_dp
      fit_coef(4, 12) = 2.9975041609363725E-03_dp
      fit_coef(5, 12) = 1.1267866675204334E-04_dp
      fit_coef(6, 12) = -6.6052089732986108E-04_dp
      fit_coef(7, 12) = -1.0429593050340101E-04_dp
      fit_coef(8, 12) = -9.6642089460867778E-05_dp
      fit_coef(9, 12) = 6.5468723131356925E-04_dp
      fit_coef(10, 12) = -7.3909338726213167E-04_dp
      fit_coef(11, 12) = 3.8434400387220237E-04_dp
      fit_coef(12, 12) = -8.1587630115757833E-05_dp
      fit_coef(0, 13) = 5.6766583157203338E-03_dp
      fit_coef(1, 13) = 1.3746861086398525E-02_dp
      fit_coef(2, 13) = 1.8677286574891313E-02_dp
      fit_coef(3, 13) = 1.5318320571135682E-02_dp
      fit_coef(4, 13) = 6.7566786656071713E-03_dp
      fit_coef(5, 13) = 7.1919811741634156E-05_dp
      fit_coef(6, 13) = -1.4743272069088146E-03_dp
      fit_coef(7, 13) = -2.0331439016954778E-04_dp
      fit_coef(8, 13) = -1.9066551850151903E-04_dp
      fit_coef(9, 13) = 1.4614366852185527E-03_dp
      fit_coef(10, 13) = -1.7182804418019637E-03_dp
      fit_coef(11, 13) = 9.2385541265890814E-04_dp
      fit_coef(12, 13) = -2.0177461300662419E-04_dp
      fit_coef(0, 14) = 1.3396971882867550E-02_dp
      fit_coef(1, 14) = 2.5745664924886561E-02_dp
      fit_coef(2, 14) = 3.1409108695941872E-02_dp
      fit_coef(3, 14) = 2.3978671080145342E-02_dp
      fit_coef(4, 14) = 1.0268221460486631E-02_dp
      fit_coef(5, 14) = 1.2930798224983239E-04_dp
      fit_coef(6, 14) = -2.4415003996848300E-03_dp
      fit_coef(7, 14) = -4.3479841388968817E-04_dp
      fit_coef(8, 14) = -5.2954762444991550E-04_dp
      fit_coef(9, 14) = 3.3920573298805374E-03_dp
      fit_coef(10, 14) = -3.9601620536706839E-03_dp
      fit_coef(11, 14) = 2.1033319510085288E-03_dp
      fit_coef(12, 14) = -4.5128467093780031E-04_dp
      fit_coef(0, 15) = 3.0313612771117847E-02_dp
      fit_coef(1, 15) = 4.5511722981043855E-02_dp
      fit_coef(2, 15) = 4.7843577097949709E-02_dp
      fit_coef(3, 15) = 3.0848847485647288E-02_dp
      fit_coef(4, 15) = 1.1173391444445794E-02_dp
      fit_coef(5, 15) = 5.3673413927424196E-04_dp
      fit_coef(6, 15) = -1.2788112460830441E-03_dp
      fit_coef(7, 15) = -1.2106504102044236E-03_dp
      fit_coef(8, 15) = -1.3614299287656570E-04_dp
      fit_coef(9, 15) = 8.4193596534928113E-04_dp
      fit_coef(10, 15) = -4.9266376268823403E-04_dp
      fit_coef(11, 15) = 2.4264482966524924E-04_dp
      fit_coef(12, 15) = -8.0800915180974188E-05_dp
      fit_coef(0, 16) = 6.5727773851375626E-02_dp
      fit_coef(1, 16) = 7.7524625497591287E-02_dp
      fit_coef(2, 16) = 7.0866135281020856E-02_dp
      fit_coef(3, 16) = 3.6455648551271543E-02_dp
      fit_coef(4, 16) = 7.9253539051012834E-03_dp
      fit_coef(5, 16) = -1.2114748812317041E-03_dp
      fit_coef(6, 16) = 8.2015105160121449E-04_dp
      fit_coef(7, 16) = -3.0470170403903876E-04_dp
      fit_coef(8, 16) = 8.4675959307711432E-04_dp
      fit_coef(9, 16) = -6.9307175814985243E-04_dp
      fit_coef(10, 16) = -1.7074186347168978E-03_dp
      fit_coef(11, 16) = 2.1004252363773243E-03_dp
      fit_coef(12, 16) = -6.5954156111989743E-04_dp
      fit_coef(0, 17) = 1.3663204308698063E-01_dp
      fit_coef(1, 17) = 1.2566212501641080E-01_dp
      fit_coef(2, 17) = 1.0142604274144125E-01_dp
      fit_coef(3, 17) = 4.0492590765185132E-02_dp
      fit_coef(4, 17) = 1.3893599733611069E-03_dp
      fit_coef(5, 17) = -4.0829840345847287E-03_dp
      fit_coef(6, 17) = 2.8641937242071414E-03_dp
      fit_coef(7, 17) = 1.4034997283466152E-04_dp
      fit_coef(8, 17) = 2.2130109369625853E-03_dp
      fit_coef(9, 17) = -1.9780643203207446E-03_dp
      fit_coef(10, 17) = -1.3761639970619644E-03_dp
      fit_coef(11, 17) = 1.9654825472193074E-03_dp
      fit_coef(12, 17) = -6.0411821444792237E-04_dp
      fit_coef(0, 18) = 2.7371713350977883E-01_dp
      fit_coef(1, 18) = 1.9207380047459294E-01_dp
      fit_coef(2, 18) = 1.3829663189483482E-01_dp
      fit_coef(3, 18) = 4.0969959876443064E-02_dp
      fit_coef(4, 18) = -8.1744595653702960E-03_dp
      fit_coef(5, 18) = -6.2758002950377006E-03_dp
      fit_coef(6, 18) = 6.1933985486500821E-03_dp
      fit_coef(7, 18) = -1.1137257661482085E-03_dp
      fit_coef(8, 18) = 4.7491704854226244E-03_dp
      fit_coef(9, 18) = -5.9366942385150516E-03_dp
      fit_coef(10, 18) = 1.9770890986731604E-03_dp
      fit_coef(11, 18) = 6.2460690633304441E-04_dp
      fit_coef(12, 18) = -4.2628349475002297E-04_dp
      fit_coef(0, 19) = 5.3176214769076025E-01_dp
      fit_coef(1, 19) = 2.7427114346332437E-01_dp
      fit_coef(2, 19) = 1.7786670623161183E-01_dp
      fit_coef(3, 19) = 3.6141984545340726E-02_dp
      fit_coef(4, 19) = -2.0019481045233233E-02_dp
      fit_coef(5, 19) = -6.5029597037300265E-03_dp
      fit_coef(6, 19) = 1.0978881646568977E-02_dp
      fit_coef(7, 19) = -3.5326165348373130E-03_dp
      fit_coef(8, 19) = 6.9664331496620468E-03_dp
      fit_coef(9, 19) = -1.0072536637310938E-02_dp
      fit_coef(10, 19) = 5.7316495954974223E-03_dp
      fit_coef(11, 19) = -8.1951094019536695E-04_dp
      fit_coef(12, 19) = -2.4352440718963261E-04_dp
      fit_coef(0, 20) = 1.0110078977345236E+00_dp
      fit_coef(1, 20) = 3.6158811802096996E-01_dp
      fit_coef(2, 20) = 2.1462317218559554E-01_dp
      fit_coef(3, 20) = 2.6882407712661614E-02_dp
      fit_coef(4, 20) = -3.1860390881090057E-02_dp
      fit_coef(5, 20) = -4.4365197992393256E-03_dp
      fit_coef(6, 20) = 1.6356522282533974E-02_dp
      fit_coef(7, 20) = -7.0191772084550030E-03_dp
      fit_coef(8, 20) = 9.0001998656736130E-03_dp
      fit_coef(9, 20) = -1.4010401343075048E-02_dp
      fit_coef(10, 20) = 9.7773201744224356E-03_dp
      fit_coef(11, 20) = -2.6263089272930114E-03_dp
      fit_coef(12, 20) = 5.8079667692654174E-05_dp
      fit_coef(0, 21) = 1.9229902544572308E+00_dp
      fit_coef(1, 21) = 4.3639570130727606E-01_dp
      fit_coef(2, 21) = 2.4465083184076966E-01_dp
      fit_coef(3, 21) = 1.8632226021993891E-02_dp
      fit_coef(4, 21) = -4.0654823075276181E-02_dp
      fit_coef(5, 21) = -1.6438863848394413E-03_dp
      fit_coef(6, 21) = 2.0679654468894963E-02_dp
      fit_coef(7, 21) = -1.0433989825777933E-02_dp
      fit_coef(8, 21) = 1.0631019238151107E-02_dp
      fit_coef(9, 21) = -1.6823436574756146E-02_dp
      fit_coef(10, 21) = 1.2861656736955324E-02_dp
      fit_coef(11, 21) = -4.1063260520761953E-03_dp
      fit_coef(12, 21) = 3.2620599079153290E-04_dp
      fit_coef(0, 22) = 3.9876478956152801E+00_dp
      fit_coef(1, 22) = 4.9699973939632702E-01_dp
      fit_coef(2, 22) = 2.7726332870042114E-01_dp
      fit_coef(3, 22) = 2.0557461247107046E-02_dp
      fit_coef(4, 22) = -4.5589994159230353E-02_dp
      fit_coef(5, 22) = -1.1953594192753122E-03_dp
      fit_coef(6, 22) = 2.3317546922334211E-02_dp
      fit_coef(7, 22) = -1.2287704350934838E-02_dp
      fit_coef(8, 22) = 1.1987190724120066E-02_dp
      fit_coef(9, 22) = -1.8798325292619430E-02_dp
      fit_coef(10, 22) = 1.4466982948530036E-02_dp
      fit_coef(11, 22) = -4.6627474463343845E-03_dp
      fit_coef(12, 22) = 3.8343338955871208E-04_dp

   END SUBROUTINE get_fit_coef_k11_mi

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k11_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.0890000000000000E+05_dp
      fit_coef(0, 1) = 3.5476589277685659E-05_dp
      fit_coef(1, 1) = 1.4120924025631340E-05_dp
      fit_coef(2, 1) = 3.1076536118129984E-03_dp
      fit_coef(3, 1) = -2.0920445403620737E-02_dp
      fit_coef(4, 1) = 1.5503824842219388E-01_dp
      fit_coef(5, 1) = -6.5946634106067126E-01_dp
      fit_coef(6, 1) = 1.9567718308121806E+00_dp
      fit_coef(7, 1) = -4.0223034504541495E+00_dp
      fit_coef(8, 1) = 5.7073753184657754E+00_dp
      fit_coef(9, 1) = -5.4854200724345779E+00_dp
      fit_coef(10, 1) = 3.4084108118688103E+00_dp
      fit_coef(11, 1) = -1.2351133915217054E+00_dp
      fit_coef(12, 1) = 1.9818096546374844E-01_dp
      fit_coef(0, 2) = 3.9040597969308965E-04_dp
      fit_coef(1, 2) = 8.7958193593917847E-05_dp
      fit_coef(2, 2) = 1.9518122056059333E-02_dp
      fit_coef(3, 2) = -1.3498019724834617E-01_dp
      fit_coef(4, 2) = 9.8104943789195032E-01_dp
      fit_coef(5, 2) = -4.1896253590373966E+00_dp
      fit_coef(6, 2) = 1.2439540817050382E+01_dp
      fit_coef(7, 2) = -2.5590801859617393E+01_dp
      fit_coef(8, 2) = 3.6347431140247103E+01_dp
      fit_coef(9, 2) = -3.4967385668188783E+01_dp
      fit_coef(10, 2) = 2.1745312479785735E+01_dp
      fit_coef(11, 2) = -7.8852461430485148E+00_dp
      fit_coef(12, 2) = 1.2659121491127978E+00_dp
      fit_coef(0, 3) = 2.0665460609606825E-03_dp
      fit_coef(1, 3) = 3.0623598182770673E-04_dp
      fit_coef(2, 3) = 6.8781938169008475E-02_dp
      fit_coef(3, 3) = -4.9168954382575458E-01_dp
      fit_coef(4, 3) = 3.4930362601920923E+00_dp
      fit_coef(5, 3) = -1.5167014240164407E+01_dp
      fit_coef(6, 3) = 4.5295440057571241E+01_dp
      fit_coef(7, 3) = -9.3377006161805866E+01_dp
      fit_coef(8, 3) = 1.3271885253969418E+02_dp
      fit_coef(9, 3) = -1.2770077780864555E+02_dp
      fit_coef(10, 3) = 7.9411953123430422E+01_dp
      fit_coef(11, 3) = -2.8793930151510978E+01_dp
      fit_coef(12, 3) = 4.6222166428791578E+00_dp
      fit_coef(0, 4) = 7.9767528386571820E-03_dp
      fit_coef(1, 4) = 8.4910081412428783E-04_dp
      fit_coef(2, 4) = 1.9200935759540066E-01_dp
      fit_coef(3, 4) = -1.4007008225024304E+00_dp
      fit_coef(4, 4) = 9.8035873307907426E+00_dp
      fit_coef(5, 4) = -4.2999162330329014E+01_dp
      fit_coef(6, 4) = 1.2867236226296961E+02_dp
      fit_coef(7, 4) = -2.6546890298183149E+02_dp
      fit_coef(8, 4) = 3.7764422948289194E+02_dp
      fit_coef(9, 4) = -3.6371847327447421E+02_dp
      fit_coef(10, 4) = 2.2639866330975411E+02_dp
      fit_coef(11, 4) = -8.2161179736409451E+01_dp
      fit_coef(12, 4) = 1.3199047691946380E+01_dp
      fit_coef(0, 5) = 2.5545969787194346E-02_dp
      fit_coef(1, 5) = 2.0342070978811074E-03_dp
      fit_coef(2, 5) = 4.6216853849385225E-01_dp
      fit_coef(3, 5) = -3.4177493858793202E+00_dp
      fit_coef(4, 5) = 2.3691045823517857E+01_dp
      fit_coef(5, 5) = -1.0461938927709393E+02_dp
      fit_coef(6, 5) = 3.1370386858745627E+02_dp
      fit_coef(7, 5) = -6.4798973211308271E+02_dp
      fit_coef(8, 5) = 9.2266003511934855E+02_dp
      fit_coef(9, 5) = -8.8927522399842849E+02_dp
      fit_coef(10, 5) = 5.5383510055109002E+02_dp
      fit_coef(11, 5) = -2.0107166636215129E+02_dp
      fit_coef(12, 5) = 3.2311819146896454E+01_dp
      fit_coef(0, 6) = 7.2076697268762174E-02_dp
      fit_coef(1, 6) = 4.3697938638884411E-03_dp
      fit_coef(2, 6) = 9.9608072915738466E-01_dp
      fit_coef(3, 6) = -7.4383652629013977E+00_dp
      fit_coef(4, 6) = 5.1208554327170454E+01_dp
      fit_coef(5, 6) = -2.2721354633426219E+02_dp
      fit_coef(6, 6) = 6.8250267908846160E+02_dp
      fit_coef(7, 6) = -1.4111404970254437E+03_dp
      fit_coef(8, 6) = 2.0105465406328890E+03_dp
      fit_coef(9, 6) = -1.9386324350975158E+03_dp
      fit_coef(10, 6) = 1.2077495454590323E+03_dp
      fit_coef(11, 6) = -4.3858368013964258E+02_dp
      fit_coef(12, 6) = 7.0492945521981113E+01_dp
      fit_coef(0, 7) = 1.8528902440326048E-01_dp
      fit_coef(1, 7) = 8.5761877628560846E-03_dp
      fit_coef(2, 7) = 1.9596400785319805E+00_dp
      fit_coef(3, 7) = -1.4742195445863704E+01_dp
      fit_coef(4, 7) = 1.0097053276986436E+02_dp
      fit_coef(5, 7) = -4.4958224203673950E+02_dp
      fit_coef(6, 7) = 1.3525059945742282E+03_dp
      fit_coef(7, 7) = -2.7986628560219651E+03_dp
      fit_coef(8, 7) = 3.9893201483629923E+03_dp
      fit_coef(9, 7) = -3.8478121346267285E+03_dp
      fit_coef(10, 7) = 2.3976749678349697E+03_dp
      fit_coef(11, 7) = -8.7083742454840910E+02_dp
      fit_coef(12, 7) = 1.3998610570704449E+02_dp
      fit_coef(0, 8) = 4.4331726198419791E-01_dp
      fit_coef(1, 8) = 1.5511309832479217E-02_dp
      fit_coef(2, 8) = 3.5507741823714190E+00_dp
      fit_coef(3, 8) = -2.6865693016984693E+01_dp
      fit_coef(4, 8) = 1.8327560251896824E+02_dp
      fit_coef(5, 8) = -8.1823836532717723E+02_dp
      fit_coef(6, 8) = 2.4647768105667869E+03_dp
      fit_coef(7, 8) = -5.1036331419270782E+03_dp
      fit_coef(8, 8) = 7.2776337187971394E+03_dp
      fit_coef(9, 8) = -7.0211118090420277E+03_dp
      fit_coef(10, 8) = 4.3757325625709809E+03_dp
      fit_coef(11, 8) = -1.5894501984451088E+03_dp
      fit_coef(12, 8) = 2.5552396179932322E+02_dp
      fit_coef(0, 9) = 1.0023832039292297E+00_dp
      fit_coef(1, 9) = 2.5898302934575013E-02_dp
      fit_coef(2, 9) = 5.9367320140376343E+00_dp
      fit_coef(3, 9) = -4.5119163651256045E+01_dp
      fit_coef(4, 9) = 3.0685214719931554E+02_dp
      fit_coef(5, 9) = -1.3727570869476449E+03_dp
      fit_coef(6, 9) = 4.1397102479948817E+03_dp
      fit_coef(7, 9) = -8.5765932664076063E+03_dp
      fit_coef(8, 9) = 1.2233664390647617E+04_dp
      fit_coef(9, 9) = -1.1804562465563917E+04_dp
      fit_coef(10, 9) = 7.3577527032167036E+03_dp
      fit_coef(11, 9) = -2.6728608608039476E+03_dp
      fit_coef(12, 9) = 4.2972086937637658E+02_dp
      fit_coef(0, 10) = 2.1747295228003427E+00_dp
      fit_coef(1, 10) = 3.9825856965749605E-02_dp
      fit_coef(2, 10) = 9.1386038693541103E+00_dp
      fit_coef(3, 10) = -6.9684875108374484E+01_dp
      fit_coef(4, 10) = 4.7283781908541010E+02_dp
      fit_coef(5, 10) = -2.1185130397402854E+03_dp
      fit_coef(6, 10) = 6.3941973699549608E+03_dp
      fit_coef(7, 10) = -1.3253289025658554E+04_dp
      fit_coef(8, 10) = 1.8908933653764056E+04_dp
      fit_coef(9, 10) = -1.8248136599625253E+04_dp
      fit_coef(10, 10) = 1.1374965884570694E+04_dp
      fit_coef(11, 10) = -4.1324304505338878E+03_dp
      fit_coef(12, 10) = 6.6440472838840810E+02_dp
      fit_coef(0, 11) = 4.6737127719821085E+00_dp
      fit_coef(1, 11) = 5.6471682863233792E-02_dp
      fit_coef(2, 11) = 1.2966744420426899E+01_dp
      fit_coef(3, 11) = -9.9092986263943558E+01_dp
      fit_coef(4, 11) = 6.7136997890125622E+02_dp
      fit_coef(5, 11) = -3.0109895935607155E+03_dp
      fit_coef(6, 11) = 9.0933208665348502E+03_dp
      fit_coef(7, 11) = -1.8853514883165644E+04_dp
      fit_coef(8, 11) = 2.6903247532779991E+04_dp
      fit_coef(9, 11) = -2.5965395036313621E+04_dp
      fit_coef(10, 11) = 1.6186393937419709E+04_dp
      fit_coef(11, 11) = -5.8805907336782202E+03_dp
      fit_coef(12, 11) = 9.4549381336496526E+02_dp
      fit_coef(0, 12) = 1.1342816745586836E-04_dp
      fit_coef(1, 12) = 3.6576968324737305E-05_dp
      fit_coef(2, 12) = 8.1035507833661872E-03_dp
      fit_coef(3, 12) = -5.4754734634044767E-02_dp
      fit_coef(4, 12) = 4.0635660271826546E-01_dp
      fit_coef(5, 12) = -1.7252011567424357E+00_dp
      fit_coef(6, 12) = 5.0991785559775868E+00_dp
      fit_coef(7, 12) = -1.0443089131832354E+01_dp
      fit_coef(8, 12) = 1.4772607027520992E+01_dp
      fit_coef(9, 12) = -1.4163309607497387E+01_dp
      fit_coef(10, 12) = 8.7833832339673332E+00_dp
      fit_coef(11, 12) = -3.1779319684136769E+00_dp
      fit_coef(12, 12) = 5.0928471655731977E-01_dp
      fit_coef(0, 13) = 7.4546004903846121E-04_dp
      fit_coef(1, 13) = 1.2519052310895946E-04_dp
      fit_coef(2, 13) = 2.7990742019097205E-02_dp
      fit_coef(3, 13) = -1.9878830068107795E-01_dp
      fit_coef(4, 13) = 1.4175816998714001E+00_dp
      fit_coef(5, 13) = -6.1266447654072609E+00_dp
      fit_coef(6, 13) = 1.8316162032568503E+01_dp
      fit_coef(7, 13) = -3.7851604364859433E+01_dp
      fit_coef(8, 13) = 5.3922831469585589E+01_dp
      fit_coef(9, 13) = -5.1975301110470006E+01_dp
      fit_coef(10, 13) = 3.2361327142046470E+01_dp
      fit_coef(11, 13) = -1.1743599901510185E+01_dp
      fit_coef(12, 13) = 1.8861805928420110E+00_dp
      fit_coef(0, 14) = 3.0519693802336465E-03_dp
      fit_coef(1, 14) = 3.4087975510808114E-04_dp
      fit_coef(2, 14) = 7.7167269766524146E-02_dp
      fit_coef(3, 14) = -5.6411267648401553E-01_dp
      fit_coef(4, 14) = 3.9421610586910765E+00_dp
      fit_coef(5, 14) = -1.7305564712941042E+01_dp
      fit_coef(6, 14) = 5.1769587328365290E+01_dp
      fit_coef(7, 14) = -1.0670184531108517E+02_dp
      fit_coef(8, 14) = 1.5162660340890758E+02_dp
      fit_coef(9, 14) = -1.4590770106048981E+02_dp
      fit_coef(10, 14) = 9.0764737150408479E+01_dp
      fit_coef(11, 14) = -3.2925715085693440E+01_dp
      fit_coef(12, 14) = 5.2881884424153274E+00_dp
      fit_coef(0, 15) = 9.9219657650822073E-03_dp
      fit_coef(1, 15) = 7.9669846000388241E-04_dp
      fit_coef(2, 15) = 1.8124096946957050E-01_dp
      fit_coef(3, 15) = -1.3450166921157356E+00_dp
      fit_coef(4, 15) = 9.2993195019520840E+00_dp
      fit_coef(5, 15) = -4.1141390542063078E+01_dp
      fit_coef(6, 15) = 1.2339197373801343E+02_dp
      fit_coef(7, 15) = -2.5492312646946198E+02_dp
      fit_coef(8, 15) = 3.6308859133318100E+02_dp
      fit_coef(9, 15) = -3.5006957300050141E+02_dp
      fit_coef(10, 15) = 2.1808828321735942E+02_dp
      fit_coef(11, 15) = -7.9197426579451133E+01_dp
      fit_coef(12, 15) = 1.2729290836050698E+01_dp
      fit_coef(0, 16) = 2.7934155736110411E-02_dp
      fit_coef(1, 16) = 1.6577256632979880E-03_dp
      fit_coef(2, 16) = 3.7842445866661195E-01_dp
      fit_coef(3, 16) = -2.8373077002350295E+00_dp
      fit_coef(4, 16) = 1.9477723226227106E+01_dp
      fit_coef(5, 16) = -8.6598640916868774E+01_dp
      fit_coef(6, 16) = 2.6025925292176282E+02_dp
      fit_coef(7, 16) = -5.3828360062495688E+02_dp
      fit_coef(8, 16) = 7.6712698683802557E+02_dp
      fit_coef(9, 16) = -7.3983293131025414E+02_dp
      fit_coef(10, 16) = 4.6097553662229262E+02_dp
      fit_coef(11, 16) = -1.6741742993909008E+02_dp
      fit_coef(12, 16) = 2.6911040347084413E+01_dp
      fit_coef(0, 17) = 7.1098068080604329E-02_dp
      fit_coef(1, 17) = 3.1344747909274207E-03_dp
      fit_coef(2, 17) = 7.1727119916633186E-01_dp
      fit_coef(3, 17) = -5.4188366020725613E+00_dp
      fit_coef(4, 17) = 3.7004431896986794E+01_dp
      fit_coef(5, 17) = -1.6510685856094676E+02_dp
      fit_coef(6, 17) = 4.9706150634055501E+02_dp
      fit_coef(7, 17) = -1.0289510132785595E+03_dp
      fit_coef(8, 17) = 1.4670886352439184E+03_dp
      fit_coef(9, 17) = -1.4153076846774175E+03_dp
      fit_coef(10, 17) = 8.8203709163857422E+02_dp
      fit_coef(11, 17) = -3.2039085416412871E+02_dp
      fit_coef(12, 17) = 5.1506838572455884E+01_dp
      fit_coef(0, 18) = 1.6776001286461931E-01_dp
      fit_coef(1, 18) = 5.4256243545789283E-03_dp
      fit_coef(2, 18) = 1.2438301964496239E+00_dp
      fit_coef(3, 18) = -9.4525365959341521E+00_dp
      fit_coef(4, 18) = 6.4287530580546374E+01_dp
      fit_coef(5, 18) = -2.8761451534054112E+02_dp
      fit_coef(6, 18) = 8.6715581047564865E+02_dp
      fit_coef(7, 18) = -1.7963912098780600E+03_dp
      fit_coef(8, 18) = 2.5623050325486606E+03_dp
      fit_coef(9, 18) = -2.4724333743568150E+03_dp
      fit_coef(10, 18) = 1.5410800293083496E+03_dp
      fit_coef(11, 18) = -5.5983897666404175E+02_dp
      fit_coef(12, 18) = 9.0007748058412503E+01_dp
      fit_coef(0, 19) = 3.7318901847591823E-01_dp
      fit_coef(1, 19) = 8.5741894158780207E-03_dp
      fit_coef(2, 19) = 1.9684022818381477E+00_dp
      fit_coef(3, 19) = -1.5029124759115470E+01_dp
      fit_coef(4, 19) = 1.0188619253386351E+02_dp
      fit_coef(5, 19) = -4.5678606319996516E+02_dp
      fit_coef(6, 19) = 1.3789467638499050E+03_dp
      fit_coef(7, 19) = -2.8584140258154093E+03_dp
      fit_coef(8, 19) = 4.0784585124208406E+03_dp
      fit_coef(9, 19) = -3.9361263103870356E+03_dp
      fit_coef(10, 19) = 2.4536814783734530E+03_dp
      fit_coef(11, 19) = -8.9143170846733369E+02_dp
      fit_coef(12, 19) = 1.4332667121452164E+02_dp
      fit_coef(0, 20) = 7.9411999812197465E-01_dp
      fit_coef(1, 20) = 1.2222970002666296E-02_dp
      fit_coef(2, 20) = 2.8089778868006130E+00_dp
      fit_coef(3, 20) = -2.1523724383534741E+01_dp
      fit_coef(4, 20) = 1.4555880546046066E+02_dp
      fit_coef(5, 20) = -6.5361423760505409E+02_dp
      fit_coef(6, 20) = 1.9751646032271344E+03_dp
      fit_coef(7, 20) = -4.0964475425550763E+03_dp
      fit_coef(8, 20) = 5.8464616672416569E+03_dp
      fit_coef(9, 20) = -5.6432337456166642E+03_dp
      fit_coef(10, 20) = 3.5181407559250561E+03_dp
      fit_coef(11, 20) = -1.2782197709057280E+03_dp
      fit_coef(12, 20) = 2.0552243167477678E+02_dp
      fit_coef(0, 21) = 1.6552401388491180E+00_dp
      fit_coef(1, 21) = 1.5475484985637764E-02_dp
      fit_coef(2, 21) = 3.5586577312092178E+00_dp
      fit_coef(3, 21) = -2.7328423530790474E+01_dp
      fit_coef(4, 21) = 1.8453595318851117E+02_dp
      fit_coef(5, 21) = -8.2943501604533219E+02_dp
      fit_coef(6, 21) = 2.5081783317335353E+03_dp
      fit_coef(7, 21) = -5.2037296899464609E+03_dp
      fit_coef(8, 21) = 7.4280891345363261E+03_dp
      fit_coef(9, 21) = -7.1705455707166202E+03_dp
      fit_coef(10, 21) = 4.4705361605949456E+03_dp
      fit_coef(11, 21) = -1.6242951789399854E+03_dp
      fit_coef(12, 21) = 2.6117201512280747E+02_dp
      fit_coef(0, 22) = 3.6819715912926032E+00_dp
      fit_coef(1, 22) = 1.7724765507747240E-02_dp
      fit_coef(2, 22) = 4.0762275279679621E+00_dp
      fit_coef(3, 22) = -3.1313218015726726E+01_dp
      fit_coef(4, 22) = 2.1139684580404636E+02_dp
      fit_coef(5, 22) = -9.5029615455661258E+02_dp
      fit_coef(6, 22) = 2.8739893385614673E+03_dp
      fit_coef(7, 22) = -5.9630507500813519E+03_dp
      fit_coef(8, 22) = 8.5122490258366070E+03_dp
      fit_coef(9, 22) = -8.2172415036707516E+03_dp
      fit_coef(10, 22) = 5.1231461949378099E+03_dp
      fit_coef(11, 22) = -1.8614171279369891E+03_dp
      fit_coef(12, 22) = 2.9929960798773016E+02_dp

   END SUBROUTINE get_fit_coef_k11_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k12_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 4.7621173842280597E-03_dp
      fit_coef(1, 1) = 1.2779124179361450E-02_dp
      fit_coef(2, 1) = 1.6703410753614655E-02_dp
      fit_coef(3, 1) = 1.2304881409550545E-02_dp
      fit_coef(4, 1) = 3.7749685318705707E-03_dp
      fit_coef(5, 1) = -1.7434526435310602E-03_dp
      fit_coef(6, 1) = -2.0397823012582900E-03_dp
      fit_coef(7, 1) = -1.9667552585414541E-04_dp
      fit_coef(8, 1) = 4.9733991832847789E-04_dp
      fit_coef(9, 1) = 3.3438174942184307E-04_dp
      fit_coef(10, 1) = -4.5000255284931892E-04_dp
      fit_coef(11, 1) = 2.2568471250347118E-04_dp
      fit_coef(12, 1) = -5.0451245801427364E-05_dp
      fit_coef(0, 2) = 2.5892231310748540E-02_dp
      fit_coef(1, 2) = 6.8272349594299786E-02_dp
      fit_coef(2, 2) = 8.8380776497241700E-02_dp
      fit_coef(3, 2) = 6.4547463586032264E-02_dp
      fit_coef(4, 2) = 1.9512355251434594E-02_dp
      fit_coef(5, 2) = -9.2451673534127178E-03_dp
      fit_coef(6, 2) = -1.0592986518352070E-02_dp
      fit_coef(7, 2) = -9.5129033192623013E-04_dp
      fit_coef(8, 2) = 2.6222990485926024E-03_dp
      fit_coef(9, 2) = 1.6749335092584598E-03_dp
      fit_coef(10, 2) = -2.2956689993050795E-03_dp
      fit_coef(11, 2) = 1.1496135531876962E-03_dp
      fit_coef(12, 2) = -2.5655891388629779E-04_dp
      fit_coef(0, 3) = 6.7567184232783226E-02_dp
      fit_coef(1, 3) = 1.7186061352294887E-01_dp
      fit_coef(2, 3) = 2.1865383385874831E-01_dp
      fit_coef(3, 3) = 1.5733674301194298E-01_dp
      fit_coef(4, 3) = 4.6341817240479470E-02_dp
      fit_coef(5, 3) = -2.2920074974133117E-02_dp
      fit_coef(6, 3) = -2.5337146772633146E-02_dp
      fit_coef(7, 3) = -1.9861858355440082E-03_dp
      fit_coef(8, 3) = 6.4137367343312197E-03_dp
      fit_coef(9, 3) = 3.7881539374578659E-03_dp
      fit_coef(10, 3) = -5.3502220389785920E-03_dp
      fit_coef(11, 3) = 2.6697483796138425E-03_dp
      fit_coef(12, 3) = -5.9317689227171935E-04_dp
      fit_coef(0, 4) = 1.3773723556451323E-01_dp
      fit_coef(1, 4) = 3.2976491191003016E-01_dp
      fit_coef(2, 4) = 4.0862257137690161E-01_dp
      fit_coef(3, 4) = 2.8864406700326678E-01_dp
      fit_coef(4, 4) = 8.2368391636866115E-02_dp
      fit_coef(5, 4) = -4.3117011034440302E-02_dp
      fit_coef(6, 4) = -4.5352842098712653E-02_dp
      fit_coef(7, 4) = -2.5963365652366087E-03_dp
      fit_coef(8, 4) = 1.2026849031951965E-02_dp
      fit_coef(9, 4) = 5.5083059863503094E-03_dp
      fit_coef(10, 4) = -8.5515536696519658E-03_dp
      fit_coef(11, 4) = 4.2807103276345489E-03_dp
      fit_coef(12, 4) = -9.5472120253031029E-04_dp
      fit_coef(0, 5) = 2.5166827573794959E-01_dp
      fit_coef(1, 5) = 5.5126096550816395E-01_dp
      fit_coef(2, 5) = 6.5643230451689671E-01_dp
      fit_coef(3, 5) = 4.5390238114819420E-01_dp
      fit_coef(4, 5) = 1.2801802686296021E-01_dp
      fit_coef(5, 5) = -6.8438409988043361E-02_dp
      fit_coef(6, 5) = -7.1938494310622186E-02_dp
      fit_coef(7, 5) = -3.7818617497551153E-03_dp
      fit_coef(8, 5) = 2.2624574898233631E-02_dp
      fit_coef(9, 5) = 7.8302540970164088E-03_dp
      fit_coef(10, 5) = -1.7136424214551770E-02_dp
      fit_coef(11, 5) = 9.9150152558899352E-03_dp
      fit_coef(12, 5) = -2.3421367209343525E-03_dp
      fit_coef(0, 6) = 4.3589892832098903E-01_dp
      fit_coef(1, 6) = 8.5015779653011903E-01_dp
      fit_coef(2, 6) = 9.5612932461007116E-01_dp
      fit_coef(3, 6) = 6.4033701313877844E-01_dp
      fit_coef(4, 6) = 1.8409747302118074E-01_dp
      fit_coef(5, 6) = -8.6870444790686202E-02_dp
      fit_coef(6, 6) = -1.0566638795756129E-01_dp
      fit_coef(7, 6) = -4.6671972769434591E-03_dp
      fit_coef(8, 6) = 3.7389422606596834E-03_dp
      fit_coef(9, 6) = 6.9425181522878909E-02_dp
      fit_coef(10, 6) = -7.1984629461529587E-02_dp
      fit_coef(11, 6) = 3.1646309655079689E-02_dp
      fit_coef(12, 6) = -5.6112243049874127E-03_dp
      fit_coef(0, 7) = 7.3351111106633227E-01_dp
      fit_coef(1, 7) = 1.2445631654341636E+00_dp
      fit_coef(2, 7) = 1.3007750122826698E+00_dp
      fit_coef(3, 7) = 8.2747315487381601E-01_dp
      fit_coef(4, 7) = 2.4170033735886270E-01_dp
      fit_coef(5, 7) = -8.6239790315945988E-02_dp
      fit_coef(6, 7) = -1.2071103017063417E-01_dp
      fit_coef(7, 7) = -3.2654987483749273E-02_dp
      fit_coef(8, 7) = 1.8998066989690098E-02_dp
      fit_coef(9, 7) = 1.4613127895351082E-02_dp
      fit_coef(10, 7) = 4.0069718709076961E-02_dp
      fit_coef(11, 7) = -5.0009415078673951E-02_dp
      fit_coef(12, 7) = 1.5360303269937313E-02_dp
      fit_coef(0, 8) = 1.2123305472983175E+00_dp
      fit_coef(1, 8) = 1.7528922824057094E+00_dp
      fit_coef(2, 8) = 1.6839617270590670E+00_dp
      fit_coef(3, 8) = 9.9693839370582804E-01_dp
      fit_coef(4, 8) = 2.8670772642350745E-01_dp
      fit_coef(5, 8) = -6.8905744965899102E-02_dp
      fit_coef(6, 8) = -9.8654503643879704E-02_dp
      fit_coef(7, 8) = -8.3833733804509133E-02_dp
      fit_coef(8, 8) = 9.6718797026550299E-02_dp
      fit_coef(9, 8) = -1.6741156687903433E-01_dp
      fit_coef(10, 8) = 2.2372278410722946E-01_dp
      fit_coef(11, 8) = -1.2395029372556110E-01_dp
      fit_coef(12, 8) = 2.4118562336799924E-02_dp
      fit_coef(0, 9) = 1.9789993531060730E+00_dp
      fit_coef(1, 9) = 2.3876726133812909E+00_dp
      fit_coef(2, 9) = 2.0983662040251230E+00_dp
      fit_coef(3, 9) = 1.1384504548954626E+00_dp
      fit_coef(4, 9) = 3.1166074671817184E-01_dp
      fit_coef(5, 9) = -3.8607395105363457E-02_dp
      fit_coef(6, 9) = -6.8592782585259429E-02_dp
      fit_coef(7, 9) = -5.9599143650154687E-02_dp
      fit_coef(8, 9) = 1.5691989539307449E-02_dp
      fit_coef(9, 9) = -6.2725923481489740E-02_dp
      fit_coef(10, 9) = 6.0324082207683011E-02_dp
      fit_coef(11, 9) = 8.7062543323561756E-03_dp
      fit_coef(12, 9) = -1.3873325190108787E-02_dp
      fit_coef(0, 10) = 3.2058934682668765E+00_dp
      fit_coef(1, 10) = 3.1496633677047572E+00_dp
      fit_coef(2, 10) = 2.5365793238247520E+00_dp
      fit_coef(3, 10) = 1.2508497217707608E+00_dp
      fit_coef(4, 10) = 3.1872392758788848E-01_dp
      fit_coef(5, 10) = -2.0600061181166207E-03_dp
      fit_coef(6, 10) = -3.1041845624322214E-02_dp
      fit_coef(7, 10) = -1.2321496532646232E-02_dp
      fit_coef(8, 10) = -8.3121266244633590E-02_dp
      fit_coef(9, 10) = 7.0964635310385341E-02_dp
      fit_coef(10, 10) = -9.6237663309511903E-02_dp
      fit_coef(11, 10) = 9.4035522256218335E-02_dp
      fit_coef(12, 10) = -2.9123579370706108E-02_dp
      fit_coef(0, 11) = 5.1968506439191477E+00_dp
      fit_coef(1, 11) = 4.0268304189360205E+00_dp
      fit_coef(2, 11) = 2.9970061410634696E+00_dp
      fit_coef(3, 11) = 1.3425771299145335E+00_dp
      fit_coef(4, 11) = 3.1423213151024637E-01_dp
      fit_coef(5, 11) = 3.5373029368206500E-02_dp
      fit_coef(6, 11) = 1.9061738352272550E-02_dp
      fit_coef(7, 11) = 2.6409588889696359E-03_dp
      fit_coef(8, 11) = -8.0417831524322025E-02_dp
      fit_coef(9, 11) = 4.2275831208402226E-02_dp
      fit_coef(10, 11) = -5.5418627131360906E-02_dp
      fit_coef(11, 11) = 4.9224692889938519E-02_dp
      fit_coef(12, 11) = -1.2095050837781235E-02_dp
      fit_coef(0, 12) = 8.6390361849437944E+00_dp
      fit_coef(1, 12) = 5.0136507853028878E+00_dp
      fit_coef(2, 12) = 3.4986136939910475E+00_dp
      fit_coef(3, 12) = 1.4331799180952967E+00_dp
      fit_coef(4, 12) = 3.0383064834623563E-01_dp
      fit_coef(5, 12) = 7.1865533368803650E-02_dp
      fit_coef(6, 12) = 7.2720154429931433E-02_dp
      fit_coef(7, 12) = 9.0385929014071293E-03_dp
      fit_coef(8, 12) = -5.0215078840340335E-02_dp
      fit_coef(9, 12) = -1.6285869510121675E-02_dp
      fit_coef(10, 12) = 1.8810174768165218E-02_dp
      fit_coef(11, 12) = -4.9947183805966527E-03_dp
      fit_coef(12, 12) = 2.0646826965866082E-03_dp
      fit_coef(0, 13) = 1.2309045987914696E-02_dp
      fit_coef(1, 13) = 3.2903305474272386E-02_dp
      fit_coef(2, 13) = 4.2910210824443497E-02_dp
      fit_coef(3, 13) = 3.1546310458948443E-02_dp
      fit_coef(4, 13) = 9.6441395261976804E-03_dp
      fit_coef(5, 13) = -4.4817476203074733E-03_dp
      fit_coef(6, 13) = -5.2171109042265578E-03_dp
      fit_coef(7, 13) = -4.9480341958360466E-04_dp
      fit_coef(8, 13) = 1.2767786392278010E-03_dp
      fit_coef(9, 13) = 8.4783300412091171E-04_dp
      fit_coef(10, 13) = -1.1459374668372458E-03_dp
      fit_coef(11, 13) = 5.7451555318326379E-04_dp
      fit_coef(12, 13) = -1.2838651639065425E-04_dp
      fit_coef(0, 14) = 3.0520606650124064E-02_dp
      fit_coef(1, 14) = 7.8713058722533771E-02_dp
      fit_coef(2, 14) = 1.0070665977979393E-01_dp
      fit_coef(3, 14) = 7.2776550328791670E-02_dp
      fit_coef(4, 14) = 2.1600668660132882E-02_dp
      fit_coef(5, 14) = -1.0555564054529464E-02_dp
      fit_coef(6, 14) = -1.1793835693071972E-02_dp
      fit_coef(7, 14) = -9.6297146505358621E-04_dp
      fit_coef(8, 14) = 2.9732450457490918E-03_dp
      fit_coef(9, 14) = 1.7819794140024070E-03_dp
      fit_coef(10, 14) = -2.5012216511924486E-03_dp
      fit_coef(11, 14) = 1.2501076056785740E-03_dp
      fit_coef(12, 14) = -2.7830976323044436E-04_dp
      fit_coef(0, 15) = 5.4104615066817303E-02_dp
      fit_coef(1, 15) = 1.2949473893963137E-01_dp
      fit_coef(2, 15) = 1.6007411910969396E-01_dp
      fit_coef(3, 15) = 1.1246986879057709E-01_dp
      fit_coef(4, 15) = 3.1687096013120898E-02_dp
      fit_coef(5, 15) = -1.6843831926749306E-02_dp
      fit_coef(6, 15) = -1.7499787344842720E-02_dp
      fit_coef(7, 15) = -1.0003535746501569E-03_dp
      fit_coef(8, 15) = 4.5492268626677436E-03_dp
      fit_coef(9, 15) = 2.3968134835377171E-03_dp
      fit_coef(10, 15) = -3.5437690918133768E-03_dp
      fit_coef(11, 15) = 1.7505580597451528E-03_dp
      fit_coef(12, 15) = -3.8471361244187636E-04_dp
      fit_coef(0, 16) = 8.8713962386069239E-02_dp
      fit_coef(1, 16) = 1.8782619870062861E-01_dp
      fit_coef(2, 16) = 2.1958966435206734E-01_dp
      fit_coef(3, 16) = 1.4951615375763633E-01_dp
      fit_coef(4, 16) = 4.0416134448872314E-02_dp
      fit_coef(5, 16) = -2.3556358508589833E-02_dp
      fit_coef(6, 16) = -2.2705784163903314E-02_dp
      fit_coef(7, 16) = -2.4239170766668074E-04_dp
      fit_coef(8, 16) = 7.8042834525613135E-03_dp
      fit_coef(9, 16) = -4.9113258408041134E-04_dp
      fit_coef(10, 16) = -1.9479282277976330E-03_dp
      fit_coef(11, 16) = 1.2196593353575873E-03_dp
      fit_coef(12, 16) = -3.1694142020982456E-04_dp
      fit_coef(0, 17) = 1.4348763712172333E-01_dp
      fit_coef(1, 17) = 2.5746461675937976E-01_dp
      fit_coef(2, 17) = 2.7495919513020706E-01_dp
      fit_coef(3, 17) = 1.7887972591601542E-01_dp
      fit_coef(4, 17) = 5.1311785540495640E-02_dp
      fit_coef(5, 17) = -2.4925938907640350E-02_dp
      fit_coef(6, 17) = -3.1142444913188724E-02_dp
      fit_coef(7, 17) = -2.1097798874644825E-03_dp
      fit_coef(8, 17) = 6.5586673591109508E-03_dp
      fit_coef(9, 17) = 2.0478515204581191E-02_dp
      fit_coef(10, 17) = -2.9823701948863770E-02_dp
      fit_coef(11, 17) = 1.6872186230723108E-02_dp
      fit_coef(12, 17) = -3.7548105037558685E-03_dp
      fit_coef(0, 18) = 2.3202358256485947E-01_dp
      fit_coef(1, 18) = 3.4341973939523573E-01_dp
      fit_coef(2, 18) = 3.2325120093458798E-01_dp
      fit_coef(3, 18) = 1.9035817858464060E-01_dp
      fit_coef(4, 18) = 5.9225378877332230E-02_dp
      fit_coef(5, 18) = -9.4096717658205004E-03_dp
      fit_coef(6, 18) = -3.1716501677007125E-02_dp
      fit_coef(7, 18) = -3.4342725566089858E-03_dp
      fit_coef(8, 18) = -3.6117207531663649E-02_dp
      fit_coef(9, 18) = 7.0489947329726826E-02_dp
      fit_coef(10, 18) = -3.4005775663369868E-02_dp
      fit_coef(11, 18) = -2.1420043382904190E-03_dp
      fit_coef(12, 18) = 4.1297341956319060E-03_dp
      fit_coef(0, 19) = 3.7428328098642150E-01_dp
      fit_coef(1, 19) = 4.4856859975928659E-01_dp
      fit_coef(2, 19) = 3.6501433131086519E-01_dp
      fit_coef(3, 19) = 1.8076014515005295E-01_dp
      fit_coef(4, 19) = 5.3394895281397445E-02_dp
      fit_coef(5, 19) = 9.7271953924831494E-03_dp
      fit_coef(6, 19) = 6.5024998766183614E-03_dp
      fit_coef(7, 19) = -5.6807667997414525E-02_dp
      fit_coef(8, 19) = 8.5060380392462415E-02_dp
      fit_coef(9, 19) = -2.0020496460057130E-01_dp
      fit_coef(10, 19) = 2.5161185176172424E-01_dp
      fit_coef(11, 19) = -1.4235287380012143E-01_dp
      fit_coef(12, 19) = 3.0493092100125858E-02_dp
      fit_coef(0, 20) = 6.0072363160772879E-01_dp
      fit_coef(1, 20) = 5.7028422329589301E-01_dp
      fit_coef(2, 20) = 4.0013007703573727E-01_dp
      fit_coef(3, 20) = 1.5645364840008563E-01_dp
      fit_coef(4, 20) = 3.5325876772107849E-02_dp
      fit_coef(5, 20) = 2.4573063136551693E-02_dp
      fit_coef(6, 20) = 3.0002223471025873E-02_dp
      fit_coef(7, 20) = -1.9881304587976378E-02_dp
      fit_coef(8, 20) = 1.0115048553542312E-02_dp
      fit_coef(9, 20) = -5.4992791578093891E-02_dp
      fit_coef(10, 20) = 7.6777053071169160E-03_dp
      fit_coef(11, 20) = 4.6072145627107347E-02_dp
      fit_coef(12, 20) = -2.2183512517221421E-02_dp
      fit_coef(0, 21) = 9.6043920592225018E-01_dp
      fit_coef(1, 21) = 6.9944535349071602E-01_dp
      fit_coef(2, 21) = 4.2734572555968520E-01_dp
      fit_coef(3, 21) = 1.2636666009694217E-01_dp
      fit_coef(4, 21) = 1.5064634695905944E-02_dp
      fit_coef(5, 21) = 3.4716238870572014E-02_dp
      fit_coef(6, 21) = 3.1317370170347261E-02_dp
      fit_coef(7, 21) = 5.3593201219119903E-02_dp
      fit_coef(8, 21) = -1.3301585636639335E-01_dp
      fit_coef(9, 21) = 1.9267279137188889E-01_dp
      fit_coef(10, 21) = -2.4206546149930569E-01_dp
      fit_coef(11, 21) = 1.5617787264354296E-01_dp
      fit_coef(12, 21) = -3.7036502375757153E-02_dp
      fit_coef(0, 22) = 1.5411563268918131E+00_dp
      fit_coef(1, 22) = 8.2232573092114225E-01_dp
      fit_coef(2, 22) = 4.4862113308179175E-01_dp
      fit_coef(3, 22) = 9.9831855190677568E-02_dp
      fit_coef(4, 22) = 2.0555253555204072E-04_dp
      fit_coef(5, 22) = 3.7473053552323943E-02_dp
      fit_coef(6, 22) = 4.4849690487304791E-02_dp
      fit_coef(7, 22) = 3.0975652422523699E-02_dp
      fit_coef(8, 22) = -4.3734383490695965E-02_dp
      fit_coef(9, 22) = 4.2410222686447063E-02_dp
      fit_coef(10, 22) = -4.1372075255841384E-02_dp
      fit_coef(11, 22) = 3.9840636595015544E-03_dp
      fit_coef(12, 22) = 6.7429310166998080E-03_dp
      fit_coef(0, 23) = 2.5398045384820884E+00_dp
      fit_coef(1, 23) = 9.2972371375931584E-01_dp
      fit_coef(2, 23) = 4.7467813354768001E-01_dp
      fit_coef(3, 23) = 8.6733603768484421E-02_dp
      fit_coef(4, 23) = -8.2527901024651990E-03_dp
      fit_coef(5, 23) = 3.7146900277764211E-02_dp
      fit_coef(6, 23) = 5.3317290655855602E-02_dp
      fit_coef(7, 23) = 4.9026643386959707E-03_dp
      fit_coef(8, 23) = 3.1627402676112949E-02_dp
      fit_coef(9, 23) = -6.8971662666575367E-02_dp
      fit_coef(10, 23) = 8.6156199337311154E-02_dp
      fit_coef(11, 23) = -6.7821612909471307E-02_dp
      fit_coef(12, 23) = 2.0108151024555578E-02_dp
      fit_coef(0, 24) = 4.6893655829446335E+00_dp
      fit_coef(1, 24) = 1.0584609550444721E+00_dp
      fit_coef(2, 24) = 5.4301054603506849E-01_dp
      fit_coef(3, 24) = 1.0134661869424059E-01_dp
      fit_coef(4, 24) = -1.1773405982811336E-02_dp
      fit_coef(5, 24) = 3.5814305836818772E-02_dp
      fit_coef(6, 24) = 5.3896955561631099E-02_dp
      fit_coef(7, 24) = 1.1794245491552634E-02_dp
      fit_coef(8, 24) = 2.0345857593909816E-02_dp
      fit_coef(9, 24) = -3.4998963970591324E-02_dp
      fit_coef(10, 24) = 4.8444694673587890E-02_dp
      fit_coef(11, 24) = -3.2817041946018727E-02_dp
      fit_coef(12, 24) = 6.8092769866170090E-03_dp

   END SUBROUTINE get_fit_coef_k12_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k12_mi(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.1533853671352736E+03_dp
      fit_coef(0, 1) = 6.6326907649095958E-04_dp
      fit_coef(1, 1) = 1.8894159031673920E-03_dp
      fit_coef(2, 1) = 2.7089355355061177E-03_dp
      fit_coef(3, 1) = 2.3224757764888754E-03_dp
      fit_coef(4, 1) = 1.0863117203975862E-03_dp
      fit_coef(5, 1) = 4.5151540703808010E-05_dp
      fit_coef(6, 1) = -2.3515149052107272E-04_dp
      fit_coef(7, 1) = -8.2929268974049080E-05_dp
      fit_coef(8, 1) = 1.1614919956202285E-04_dp
      fit_coef(9, 1) = -2.5284724218147597E-05_dp
      fit_coef(10, 1) = -1.3858258705778020E-05_dp
      fit_coef(11, 1) = 8.0009405004549032E-06_dp
      fit_coef(12, 1) = -1.0748701295750969E-06_dp
      fit_coef(0, 2) = 3.8227101929591994E-03_dp
      fit_coef(1, 2) = 1.0364280264194323E-02_dp
      fit_coef(2, 2) = 1.4590446568112637E-02_dp
      fit_coef(3, 2) = 1.2309386146930817E-02_dp
      fit_coef(4, 2) = 5.6468571531384185E-03_dp
      fit_coef(5, 2) = 1.8293346115620324E-04_dp
      fit_coef(6, 2) = -1.2260374384726667E-03_dp
      fit_coef(7, 2) = -4.2679063675895478E-04_dp
      fit_coef(8, 2) = 6.4562440386666691E-04_dp
      fit_coef(9, 2) = -2.0425912250358465E-04_dp
      fit_coef(10, 2) = -1.0718250746829363E-05_dp
      fit_coef(11, 2) = 1.3450401493497087E-05_dp
      fit_coef(12, 2) = 3.6903867124824597E-08_dp
      fit_coef(0, 3) = 1.1208738658104969E-02_dp
      fit_coef(1, 3) = 2.7433467450107527E-02_dp
      fit_coef(2, 3) = 3.7122781203738041E-02_dp
      fit_coef(3, 3) = 3.0505000476205493E-02_dp
      fit_coef(4, 3) = 1.3613669155986628E-02_dp
      fit_coef(5, 3) = 2.0805962290796248E-04_dp
      fit_coef(6, 3) = -3.0074089272987834E-03_dp
      fit_coef(7, 3) = -1.0066712315667174E-03_dp
      fit_coef(8, 3) = 1.8121892538119837E-03_dp
      fit_coef(9, 3) = -8.6670466868856387E-04_dp
      fit_coef(10, 3) = 2.3031588351142570E-04_dp
      fit_coef(11, 3) = -6.1244719299360145E-05_dp
      fit_coef(12, 3) = 1.5021644001300220E-05_dp
      fit_coef(0, 4) = 2.7060109838528964E-02_dp
      fit_coef(1, 4) = 5.7181276021668774E-02_dp
      fit_coef(2, 4) = 7.2163197996693393E-02_dp
      fit_coef(3, 4) = 5.6401492360043291E-02_dp
      fit_coef(4, 4) = 2.4397958649191483E-02_dp
      fit_coef(5, 4) = 4.1542953141707141E-04_dp
      fit_coef(6, 4) = -5.4082092785117776E-03_dp
      fit_coef(7, 4) = -1.9248439551360079E-03_dp
      fit_coef(8, 4) = 2.9200980290588186E-03_dp
      fit_coef(9, 4) = -9.7952619465276037E-04_dp
      fit_coef(10, 4) = 1.6789215463763337E-04_dp
      fit_coef(11, 4) = -1.4147604636971736E-04_dp
      fit_coef(12, 4) = 6.0667969636576076E-05_dp
      fit_coef(0, 5) = 6.0162354447464560E-02_dp
      fit_coef(1, 5) = 1.0720429058942169E-01_dp
      fit_coef(2, 5) = 1.2343759565688829E-01_dp
      fit_coef(3, 5) = 8.8531585067086430E-02_dp
      fit_coef(4, 5) = 3.5364287351100372E-02_dp
      fit_coef(5, 5) = 6.0629507679625032E-04_dp
      fit_coef(6, 5) = -6.6115157196993245E-03_dp
      fit_coef(7, 5) = -3.1579068142273776E-03_dp
      fit_coef(8, 5) = 5.0366482632659849E-03_dp
      fit_coef(9, 5) = -5.0447339830031095E-03_dp
      fit_coef(10, 5) = 4.5014753449279963E-03_dp
      fit_coef(11, 5) = -2.3546672147255598E-03_dp
      fit_coef(12, 5) = 5.0845127538551175E-04_dp
      fit_coef(0, 6) = 1.2715764725238982E-01_dp
      fit_coef(1, 6) = 1.8881057094422343E-01_dp
      fit_coef(2, 6) = 1.9672779228400020E-01_dp
      fit_coef(3, 6) = 1.2577665875739202E-01_dp
      fit_coef(4, 6) = 4.3061470368101699E-02_dp
      fit_coef(5, 6) = -9.4453422287111893E-04_dp
      fit_coef(6, 6) = -6.1901712508776350E-03_dp
      fit_coef(7, 6) = -2.9123796915168040E-03_dp
      fit_coef(8, 6) = 7.5406628009243720E-03_dp
      fit_coef(9, 6) = -1.0878759759659724E-02_dp
      fit_coef(10, 6) = 9.0511431231051973E-03_dp
      fit_coef(11, 6) = -3.9841133010950968E-03_dp
      fit_coef(12, 6) = 7.4476527248770687E-04_dp
      fit_coef(0, 7) = 2.5849300626981397E-01_dp
      fit_coef(1, 7) = 3.1665344505552956E-01_dp
      fit_coef(2, 7) = 2.9840462942608115E-01_dp
      fit_coef(3, 7) = 1.6655149775226027E-01_dp
      fit_coef(4, 7) = 4.4635943287943819E-02_dp
      fit_coef(5, 7) = -5.0840338248153878E-03_dp
      fit_coef(6, 7) = -4.1935958573150935E-03_dp
      fit_coef(7, 7) = -1.2238069004985115E-03_dp
      fit_coef(8, 7) = 1.0355993093508651E-02_dp
      fit_coef(9, 7) = -1.8364631667690871E-02_dp
      fit_coef(10, 7) = 1.5416014403986910E-02_dp
      fit_coef(11, 7) = -6.8470052029501608E-03_dp
      fit_coef(12, 7) = 1.3249810031683059E-03_dp
      fit_coef(0, 8) = 5.0864125871117427E-01_dp
      fit_coef(1, 8) = 5.0748696881076827E-01_dp
      fit_coef(2, 8) = 4.3378556875785329E-01_dp
      fit_coef(3, 8) = 2.0761241946020209E-01_dp
      fit_coef(4, 8) = 3.7494223843576979E-02_dp
      fit_coef(5, 8) = -1.1199453066603072E-02_dp
      fit_coef(6, 8) = 2.6211861457654248E-04_dp
      fit_coef(7, 8) = 1.0181856045809953E-03_dp
      fit_coef(8, 8) = 1.3617506208780007E-02_dp
      fit_coef(9, 8) = -2.9083172474373836E-02_dp
      fit_coef(10, 8) = 2.6093940387631132E-02_dp
      fit_coef(11, 8) = -1.2187538547457615E-02_dp
      fit_coef(12, 8) = 2.4442839760494672E-03_dp
      fit_coef(0, 9) = 9.7351269086421022E-01_dp
      fit_coef(1, 9) = 7.7682804453441712E-01_dp
      fit_coef(2, 9) = 6.0537341804741274E-01_dp
      fit_coef(3, 9) = 2.4417103093525630E-01_dp
      fit_coef(4, 9) = 1.9643095133629673E-02_dp
      fit_coef(5, 9) = -1.7552027242727909E-02_dp
      fit_coef(6, 9) = 8.2268681071634708E-03_dp
      fit_coef(7, 9) = 3.3800170464644732E-03_dp
      fit_coef(8, 9) = 1.5951520448433745E-02_dp
      fit_coef(9, 9) = -4.2082858702406162E-02_dp
      fit_coef(10, 9) = 4.0832752772189854E-02_dp
      fit_coef(11, 9) = -1.9914993594640506E-02_dp
      fit_coef(12, 9) = 4.0828025408897019E-03_dp
      fit_coef(0, 10) = 1.8218150409063079E+00_dp
      fit_coef(1, 10) = 1.1328940612144838E+00_dp
      fit_coef(2, 10) = 8.1145165005204600E-01_dp
      fit_coef(3, 10) = 2.7169894714361381E-01_dp
      fit_coef(4, 10) = -9.2391365403536683E-03_dp
      fit_coef(5, 10) = -2.1969808519198136E-02_dp
      fit_coef(6, 10) = 2.0246181533686430E-02_dp
      fit_coef(7, 10) = 5.2443671348299629E-03_dp
      fit_coef(8, 10) = 1.6294437997198600E-02_dp
      fit_coef(9, 10) = -5.6113566142827496E-02_dp
      fit_coef(10, 10) = 5.9108716717708169E-02_dp
      fit_coef(11, 10) = -3.0017787615102216E-02_dp
      fit_coef(12, 10) = 6.2851355640176588E-03_dp
      fit_coef(0, 11) = 3.3638892281286350E+00_dp
      fit_coef(1, 11) = 1.5701436592256290E+00_dp
      fit_coef(2, 11) = 1.0468794820440899E+00_dp
      fit_coef(3, 11) = 2.8932244841622851E-01_dp
      fit_coef(4, 11) = -4.7147330695069850E-02_dp
      fit_coef(5, 11) = -2.3231305463748575E-02_dp
      fit_coef(6, 11) = 3.5814349205865105E-02_dp
      fit_coef(7, 11) = 6.1681105190574941E-03_dp
      fit_coef(8, 11) = 1.4380388978420344E-02_dp
      fit_coef(9, 11) = -7.0072767054186372E-02_dp
      fit_coef(10, 11) = 7.9925293980236525E-02_dp
      fit_coef(11, 11) = -4.2106160715496485E-02_dp
      fit_coef(12, 11) = 8.9934277746186089E-03_dp
      fit_coef(0, 12) = 6.2958157177735696E+00_dp
      fit_coef(1, 12) = 2.0736278555613925E+00_dp
      fit_coef(2, 12) = 1.3100027104859728E+00_dp
      fit_coef(3, 12) = 3.0309841684283917E-01_dp
      fit_coef(4, 12) = -9.0874734045887973E-02_dp
      fit_coef(5, 12) = -2.2511990033847892E-02_dp
      fit_coef(6, 12) = 5.3762327238007911E-02_dp
      fit_coef(7, 12) = 6.4301275560599072E-03_dp
      fit_coef(8, 12) = 1.1130561807738493E-02_dp
      fit_coef(9, 12) = -8.4343774419336590E-02_dp
      fit_coef(10, 12) = 1.0282039809053546E-01_dp
      fit_coef(11, 12) = -5.5735942843285882E-02_dp
      fit_coef(12, 12) = 1.2089953955292771E-02_dp
      fit_coef(0, 13) = 1.7361176995326563E-03_dp
      fit_coef(1, 13) = 4.8950464700643088E-03_dp
      fit_coef(2, 13) = 6.9887889000087690E-03_dp
      fit_coef(3, 13) = 5.9681592147681978E-03_dp
      fit_coef(4, 13) = 2.7797694969593904E-03_dp
      fit_coef(5, 13) = 1.0974079237570183E-04_dp
      fit_coef(6, 13) = -6.0276261358583726E-04_dp
      fit_coef(7, 13) = -2.1159566479273911E-04_dp
      fit_coef(8, 13) = 3.0294248744461766E-04_dp
      fit_coef(9, 13) = -7.4399992147188479E-05_dp
      fit_coef(10, 13) = -2.7468045558828552E-05_dp
      fit_coef(11, 13) = 1.6865141251068427E-05_dp
      fit_coef(12, 13) = -2.0377510281004492E-06_dp
      fit_coef(0, 14) = 4.8355092096145566E-03_dp
      fit_coef(1, 14) = 1.2333932959289709E-02_dp
      fit_coef(2, 14) = 1.6972825457174837E-02_dp
      fit_coef(3, 14) = 1.4060909191254103E-02_dp
      fit_coef(4, 14) = 6.2955683700635671E-03_dp
      fit_coef(5, 14) = 1.2177301196491723E-04_dp
      fit_coef(6, 14) = -1.3677107176344808E-03_dp
      fit_coef(7, 14) = -4.6095049437820380E-04_dp
      fit_coef(8, 14) = 7.6937933826695876E-04_dp
      fit_coef(9, 14) = -3.3164200459717756E-04_dp
      fit_coef(10, 14) = 8.1724364865420408E-05_dp
      fit_coef(11, 14) = -2.9312146170772606E-05_dp
      fit_coef(12, 14) = 8.9970999683901846E-06_dp
      fit_coef(0, 15) = 1.0612378818881555E-02_dp
      fit_coef(1, 15) = 2.2459336506101473E-02_dp
      fit_coef(2, 15) = 2.8371129792457078E-02_dp
      fit_coef(3, 15) = 2.2261852601842439E-02_dp
      fit_coef(4, 15) = 9.5999731643326893E-03_dp
      fit_coef(5, 15) = -1.8558563529812452E-05_dp
      fit_coef(6, 15) = -2.2357421491240670E-03_dp
      fit_coef(7, 15) = -7.4423534393391617E-04_dp
      fit_coef(8, 15) = 1.5057122702193376E-03_dp
      fit_coef(9, 15) = -7.0297537707829122E-04_dp
      fit_coef(10, 15) = 7.1959722147797053E-05_dp
      fit_coef(11, 15) = 4.6639800520364974E-05_dp
      fit_coef(12, 15) = -1.2660883153173911E-05_dp
      fit_coef(0, 16) = 2.2502651094608313E-02_dp
      fit_coef(1, 16) = 3.8302915711040345E-02_dp
      fit_coef(2, 16) = 4.2327776536488830E-02_dp
      fit_coef(3, 16) = 2.9242971747304312E-02_dp
      fit_coef(4, 16) = 1.1483493368870591E-02_dp
      fit_coef(5, 16) = 4.4372686263633332E-04_dp
      fit_coef(6, 16) = -2.1409365324177557E-03_dp
      fit_coef(7, 16) = -1.1499475056719786E-03_dp
      fit_coef(8, 16) = 9.9305143157658372E-04_dp
      fit_coef(9, 16) = -8.6357250324013148E-04_dp
      fit_coef(10, 16) = 1.4327807191970657E-03_dp
      fit_coef(11, 16) = -1.0670863396023819E-03_dp
      fit_coef(12, 16) = 2.8011891218022886E-04_dp
      fit_coef(0, 17) = 4.6379852772014826E-02_dp
      fit_coef(1, 17) = 6.3632440293261802E-02_dp
      fit_coef(2, 17) = 6.1214298124389097E-02_dp
      fit_coef(3, 17) = 3.4856680478514791E-02_dp
      fit_coef(4, 17) = 9.8511055688256591E-03_dp
      fit_coef(5, 17) = -4.1408424159966774E-04_dp
      fit_coef(6, 17) = -2.7116403636660304E-04_dp
      fit_coef(7, 17) = -8.3345412989501344E-04_dp
      fit_coef(8, 17) = 2.8388169463227923E-03_dp
      fit_coef(9, 17) = -6.0896330652201345E-03_dp
      fit_coef(10, 17) = 5.4431369422386086E-03_dp
      fit_coef(11, 17) = -2.2497263945134662E-03_dp
      fit_coef(12, 17) = 3.6399313731776849E-04_dp
      fit_coef(0, 18) = 9.2521202823536430E-02_dp
      fit_coef(1, 18) = 1.0203973195050642E-01_dp
      fit_coef(2, 18) = 8.6453931343429755E-02_dp
      fit_coef(3, 18) = 3.9391412232875435E-02_dp
      fit_coef(4, 18) = 5.0815288567403350E-03_dp
      fit_coef(5, 18) = -2.8315021323427666E-03_dp
      fit_coef(6, 18) = 1.1044933352897194E-03_dp
      fit_coef(7, 18) = 1.2026713128526101E-03_dp
      fit_coef(8, 18) = 2.3687414607481347E-03_dp
      fit_coef(9, 18) = -5.9882091197473600E-03_dp
      fit_coef(10, 18) = 4.5247728538859366E-03_dp
      fit_coef(11, 18) = -1.7071153745195653E-03_dp
      fit_coef(12, 18) = 2.9758468692744693E-04_dp
      fit_coef(0, 19) = 1.7895123701529778E-01_dp
      fit_coef(1, 19) = 1.5649170004661517E-01_dp
      fit_coef(2, 19) = 1.1781626034461101E-01_dp
      fit_coef(3, 19) = 4.1616478405439042E-02_dp
      fit_coef(4, 19) = -2.3717162520277809E-03_dp
      fit_coef(5, 19) = -5.3386440882515086E-03_dp
      fit_coef(6, 19) = 3.0469549759779353E-03_dp
      fit_coef(7, 19) = 2.0334444707206275E-03_dp
      fit_coef(8, 19) = 3.2479855073872382E-03_dp
      fit_coef(9, 19) = -9.1986286972097365E-03_dp
      fit_coef(10, 19) = 8.5242332274741671E-03_dp
      fit_coef(11, 19) = -4.1329263555985006E-03_dp
      fit_coef(12, 19) = 8.6616145809027139E-04_dp
      fit_coef(0, 20) = 3.3684611079928783E-01_dp
      fit_coef(1, 20) = 2.2786610373174537E-01_dp
      fit_coef(2, 20) = 1.5339071016684352E-01_dp
      fit_coef(3, 20) = 3.9677854839472344E-02_dp
      fit_coef(4, 20) = -1.2264564209904332E-02_dp
      fit_coef(5, 20) = -6.5927521643919379E-03_dp
      fit_coef(6, 20) = 6.0512561434243684E-03_dp
      fit_coef(7, 20) = 2.3902542529467478E-03_dp
      fit_coef(8, 20) = 3.0195138462315793E-03_dp
      fit_coef(9, 20) = -1.2050135993152236E-02_dp
      fit_coef(10, 20) = 1.2762387029635221E-02_dp
      fit_coef(11, 20) = -6.5239286773536758E-03_dp
      fit_coef(12, 20) = 1.3694468177831019E-03_dp
      fit_coef(0, 21) = 6.2004160201378677E-01_dp
      fit_coef(1, 21) = 3.1231452836704038E-01_dp
      fit_coef(2, 21) = 1.8949289978076236E-01_dp
      fit_coef(3, 21) = 3.2623505491414392E-02_dp
      fit_coef(4, 21) = -2.3505999675060181E-02_dp
      fit_coef(5, 21) = -5.7260158711334771E-03_dp
      fit_coef(6, 21) = 9.9807237729862951E-03_dp
      fit_coef(7, 21) = 2.2222797839610000E-03_dp
      fit_coef(8, 21) = 1.4664130053888486E-03_dp
      fit_coef(9, 21) = -1.3740779304315174E-02_dp
      fit_coef(10, 21) = 1.6640549191351679E-02_dp
      fit_coef(11, 21) = -8.9416960852337218E-03_dp
      fit_coef(12, 21) = 1.9187092435646925E-03_dp
      fit_coef(0, 22) = 1.1255181435117496E+00_dp
      fit_coef(1, 22) = 3.9889498776431231E-01_dp
      fit_coef(2, 22) = 2.2171584629075095E-01_dp
      fit_coef(3, 22) = 2.2225826529528901E-02_dp
      fit_coef(4, 22) = -3.3907556115484502E-02_dp
      fit_coef(5, 22) = -2.8764770890914323E-03_dp
      fit_coef(6, 22) = 1.3968131777336038E-02_dp
      fit_coef(7, 22) = 1.4214928484319333E-03_dp
      fit_coef(8, 22) = -8.3453566281261021E-04_dp
      fit_coef(9, 22) = -1.4109958929870586E-02_dp
      fit_coef(10, 22) = 1.9726438913848971E-02_dp
      fit_coef(11, 22) = -1.1193142315748616E-02_dp
      fit_coef(12, 22) = 2.4755702682764356E-03_dp
      fit_coef(0, 23) = 2.0598610404318460E+00_dp
      fit_coef(1, 23) = 4.7243372474599793E-01_dp
      fit_coef(2, 23) = 2.4849210820527318E-01_dp
      fit_coef(3, 23) = 1.4006813172765054E-02_dp
      fit_coef(4, 23) = -4.1308949825231950E-02_dp
      fit_coef(5, 23) = 1.5690842161415170E-04_dp
      fit_coef(6, 23) = 1.6942861921432139E-02_dp
      fit_coef(7, 23) = 4.7152452086483685E-04_dp
      fit_coef(8, 23) = -2.8341187428894861E-03_dp
      fit_coef(9, 23) = -1.3846753524951650E-02_dp
      fit_coef(10, 23) = 2.1790170949455349E-02_dp
      fit_coef(11, 23) = -1.2886556590999076E-02_dp
      fit_coef(12, 23) = 2.9162146561646620E-03_dp
      fit_coef(0, 24) = 4.1434250198174896E+00_dp
      fit_coef(1, 24) = 5.3740865227175982E-01_dp
      fit_coef(2, 24) = 2.8197846608440591E-01_dp
      fit_coef(3, 24) = 1.6092456027878326E-02_dp
      fit_coef(4, 24) = -4.6040710751708042E-02_dp
      fit_coef(5, 24) = 6.5042394570127837E-04_dp
      fit_coef(6, 24) = 1.8932659149369899E-02_dp
      fit_coef(7, 24) = 2.4082473716184700E-04_dp
      fit_coef(8, 24) = -3.3484787514316723E-03_dp
      fit_coef(9, 24) = -1.5284105573369426E-02_dp
      fit_coef(10, 24) = 2.4407280609082902E-02_dp
      fit_coef(11, 24) = -1.4513713351654377E-02_dp
      fit_coef(12, 24) = 3.2970008560632581E-03_dp

   END SUBROUTINE get_fit_coef_k12_mi

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k12_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.0890000000000000E+05_dp
      fit_coef(0, 1) = 2.3129808933247259E-05_dp
      fit_coef(1, 1) = 1.3229791040971861E-04_dp
      fit_coef(2, 1) = 8.9078224043542527E-04_dp
      fit_coef(3, 1) = 3.5906725339575205E-04_dp
      fit_coef(4, 1) = 1.4513983588735334E-02_dp
      fit_coef(5, 1) = -5.2233430638158315E-02_dp
      fit_coef(6, 1) = 1.6441632828707831E-01_dp
      fit_coef(7, 1) = -3.6058840741824150E-01_dp
      fit_coef(8, 1) = 5.3428676974850686E-01_dp
      fit_coef(9, 1) = -5.2731672647478189E-01_dp
      fit_coef(10, 1) = 3.3265094311193522E-01_dp
      fit_coef(11, 1) = -1.2145741283688967E-01_dp
      fit_coef(12, 1) = 1.9537487536713263E-02_dp
      fit_coef(0, 2) = 2.3455247909237882E-04_dp
      fit_coef(1, 2) = 8.2160398869974953E-04_dp
      fit_coef(2, 2) = 5.3319821419640272E-03_dp
      fit_coef(3, 2) = 6.4575814983609142E-04_dp
      fit_coef(4, 2) = 8.7822372114111175E-02_dp
      fit_coef(5, 2) = -3.2631392232557216E-01_dp
      fit_coef(6, 2) = 1.0208260875409669E+00_dp
      fit_coef(7, 2) = -2.2300026314422441E+00_dp
      fit_coef(8, 2) = 3.3057374126568657E+00_dp
      fit_coef(9, 2) = -3.2713127430468321E+00_dp
      fit_coef(10, 2) = 2.0709853451512608E+00_dp
      fit_coef(11, 2) = -7.5901118153106462E-01_dp
      fit_coef(12, 2) = 1.2254511188136619E-01_dp
      fit_coef(0, 3) = 1.1944107046876868E-03_dp
      fit_coef(1, 3) = 2.8490802943664383E-03_dp
      fit_coef(2, 3) = 1.7507904217475573E-02_dp
      fit_coef(3, 3) = -6.4522716917725272E-03_dp
      fit_coef(4, 3) = 2.8224130041588374E-01_dp
      fit_coef(5, 3) = -1.1757205798515102E+00_dp
      fit_coef(6, 3) = 3.7822941249117266E+00_dp
      fit_coef(7, 3) = -8.2843383024642065E+00_dp
      fit_coef(8, 3) = 1.2238481082417001E+01_dp
      fit_coef(9, 3) = -1.2055852436649243E+01_dp
      fit_coef(10, 3) = 7.6007782878436343E+00_dp
      fit_coef(11, 3) = -2.7765219925860354E+00_dp
      fit_coef(12, 3) = 4.4718773429739378E-01_dp
      fit_coef(0, 4) = 4.5120683484607405E-03_dp
      fit_coef(1, 4) = 7.9095288831978021E-03_dp
      fit_coef(2, 4) = 4.6697868261753268E-02_dp
      fit_coef(3, 4) = -3.3544522740173804E-02_dp
      fit_coef(4, 4) = 7.2425137774094894E-01_dp
      fit_coef(5, 4) = -3.2633003302313521E+00_dp
      fit_coef(6, 4) = 1.0450673822567490E+01_dp
      fit_coef(7, 4) = -2.2594031368355793E+01_dp
      fit_coef(8, 4) = 3.3109462208134374E+01_dp
      fit_coef(9, 4) = -3.2506292757603717E+01_dp
      fit_coef(10, 4) = 2.0487989626895530E+01_dp
      fit_coef(11, 4) = -7.4948539272232253E+00_dp
      fit_coef(12, 4) = 1.2099196036738793E+00_dp
      fit_coef(0, 5) = 1.4248952254027877E-02_dp
      fit_coef(1, 5) = 1.9074798942523639E-02_dp
      fit_coef(2, 5) = 1.0943167990343619E-01_dp
      fit_coef(3, 5) = -1.0671670298403642E-01_dp
      fit_coef(4, 5) = 1.6526275275600326E+00_dp
      fit_coef(5, 5) = -7.8775178049382024E+00_dp
      fit_coef(6, 5) = 2.5305646502994179E+01_dp
      fit_coef(7, 5) = -5.4666053461159009E+01_dp
      fit_coef(8, 5) = 8.0167059237163343E+01_dp
      fit_coef(9, 5) = -7.8825322292091528E+01_dp
      fit_coef(10, 5) = 4.9764390489009351E+01_dp
      fit_coef(11, 5) = -1.8233568839424560E+01_dp
      fit_coef(12, 5) = 2.9478228744193333E+00_dp
      fit_coef(0, 6) = 3.9801596054429601E-02_dp
      fit_coef(1, 6) = 4.1447021761176796E-02_dp
      fit_coef(2, 6) = 2.3270073234920230E-01_dp
      fit_coef(3, 6) = -2.7292288542272475E-01_dp
      fit_coef(4, 6) = 3.4506136511101664E+00_dp
      fit_coef(5, 6) = -1.7124854042561445E+01_dp
      fit_coef(6, 6) = 5.5335626634127287E+01_dp
      fit_coef(7, 6) = -1.1966893675071402E+02_dp
      fit_coef(8, 6) = 1.7554731067692325E+02_dp
      fit_coef(9, 6) = -1.7262672563793396E+02_dp
      fit_coef(10, 6) = 1.0899761858660371E+02_dp
      fit_coef(11, 6) = -3.9946406626328411E+01_dp
      fit_coef(12, 6) = 6.4605650754876507E+00_dp
      fit_coef(0, 7) = 1.0153848301405891E-01_dp
      fit_coef(1, 7) = 8.2744575115798746E-02_dp
      fit_coef(2, 7) = 4.5675281550800451E-01_dp
      fit_coef(3, 7) = -6.0755960983591895E-01_dp
      fit_coef(4, 7) = 6.6840425945230333E+00_dp
      fit_coef(5, 7) = -3.4181153764390729E+01_dp
      fit_coef(6, 7) = 1.1120663926638191E+02_dp
      fit_coef(7, 7) = -2.4092641084144972E+02_dp
      fit_coef(8, 7) = 3.5355143552413938E+02_dp
      fit_coef(9, 7) = -3.4765931274190069E+02_dp
      fit_coef(10, 7) = 2.1950666813842750E+02_dp
      fit_coef(11, 7) = -8.0451830347244339E+01_dp
      fit_coef(12, 7) = 1.3013672905602384E+01_dp
      fit_coef(0, 8) = 2.4141648240665145E-01_dp
      fit_coef(1, 8) = 1.5335953037887173E-01_dp
      fit_coef(2, 8) = 8.3504832050626976E-01_dp
      fit_coef(3, 8) = -1.2178157194978934E+00_dp
      fit_coef(4, 8) = 1.2099585014816819E+01_dp
      fit_coef(5, 8) = -6.3313617826158982E+01_dp
      fit_coef(6, 8) = 2.0741722770815861E+02_dp
      fit_coef(7, 8) = -4.5029790616194964E+02_dp
      fit_coef(8, 8) = 6.6110515192856462E+02_dp
      fit_coef(9, 8) = -6.5007509234254940E+02_dp
      fit_coef(10, 8) = 4.1040648800886731E+02_dp
      fit_coef(11, 8) = -1.5041026682712433E+02_dp
      fit_coef(12, 8) = 2.4330155922976292E+01_dp
      fit_coef(0, 9) = 5.4247999033308114E-01_dp
      fit_coef(1, 9) = 2.6499445034705121E-01_dp
      fit_coef(2, 9) = 1.4268709589831525E+00_dp
      fit_coef(3, 9) = -2.2313886617958429E+00_dp
      fit_coef(4, 9) = 2.0518890229005105E+01_dp
      fit_coef(5, 9) = -1.0930947078396949E+02_dp
      fit_coef(6, 9) = 3.6045833696584418E+02_dp
      fit_coef(7, 9) = -7.8423527312908436E+02_dp
      fit_coef(8, 9) = 1.1520122421721351E+03_dp
      fit_coef(9, 9) = -1.1328174992310501E+03_dp
      fit_coef(10, 9) = 7.1509445979181294E+02_dp
      fit_coef(11, 9) = -2.6204807849728383E+02_dp
      fit_coef(12, 9) = 4.2385855620433802E+01_dp
      fit_coef(0, 10) = 1.1649235984334310E+00_dp
      fit_coef(1, 10) = 4.2669860434201973E-01_dp
      fit_coef(2, 10) = 2.2769218254498678E+00_dp
      fit_coef(3, 10) = -3.7556598199553419E+00_dp
      fit_coef(4, 10) = 3.2554151177465044E+01_dp
      fit_coef(5, 10) = -1.7584595871324311E+02_dp
      fit_coef(6, 10) = 5.8329724780519552E+02_dp
      fit_coef(7, 10) = -1.2717006032060506E+03_dp
      fit_coef(8, 10) = 1.8691949694429402E+03_dp
      fit_coef(9, 10) = -1.8381765181945018E+03_dp
      fit_coef(10, 10) = 1.1602510916037054E+03_dp
      fit_coef(11, 10) = -4.2512958735964088E+02_dp
      fit_coef(12, 10) = 6.8758003199532453E+01_dp
      fit_coef(0, 11) = 2.4212426991212057E+00_dp
      fit_coef(1, 11) = 6.3843606877537262E-01_dp
      fit_coef(2, 11) = 3.3832944732267647E+00_dp
      fit_coef(3, 11) = -5.8032061767073824E+00_dp
      fit_coef(4, 11) = 4.8167947114080192E+01_dp
      fit_coef(5, 11) = -2.6286759432487781E+02_dp
      fit_coef(6, 11) = 8.7620572915548814E+02_dp
      fit_coef(7, 11) = -1.9137694452215110E+03_dp
      fit_coef(8, 11) = 2.8145274374678247E+03_dp
      fit_coef(9, 11) = -2.7680919699076530E+03_dp
      fit_coef(10, 11) = 1.7471091763340671E+03_dp
      fit_coef(11, 11) = -6.4010183684923209E+02_dp
      fit_coef(12, 11) = 1.0351773205889742E+02_dp
      fit_coef(0, 12) = 5.0200579894610868E+00_dp
      fit_coef(1, 12) = 8.8885387561015605E-01_dp
      fit_coef(2, 12) = 4.6883210461508416E+00_dp
      fit_coef(3, 12) = -8.2512193578220483E+00_dp
      fit_coef(4, 12) = 6.6560605008919623E+01_dp
      fit_coef(5, 12) = -3.6572207879590201E+02_dp
      fit_coef(6, 12) = 1.2232199700982205E+03_dp
      fit_coef(7, 12) = -2.6752268847851810E+03_dp
      fit_coef(8, 12) = 3.9360821093174841E+03_dp
      fit_coef(9, 12) = -3.8714869910206562E+03_dp
      fit_coef(10, 12) = 2.4434517488097472E+03_dp
      fit_coef(11, 12) = -8.9517007084930697E+02_dp
      fit_coef(12, 12) = 1.4475878519184542E+02_dp
      fit_coef(0, 13) = 7.1480283160792286E-05_dp
      fit_coef(1, 13) = 3.4522475460442413E-04_dp
      fit_coef(2, 13) = 2.3289358018025357E-03_dp
      fit_coef(3, 13) = 8.6564456187697154E-04_dp
      fit_coef(4, 13) = 3.9444759289228873E-02_dp
      fit_coef(5, 13) = -1.4607809825480644E-01_dp
      fit_coef(6, 13) = 4.5801849703119135E-01_dp
      fit_coef(7, 13) = -9.9551749039632609E-01_dp
      fit_coef(8, 13) = 1.4633991701280600E+00_dp
      fit_coef(9, 13) = -1.4351206048684466E+00_dp
      fit_coef(10, 13) = 9.0075138053758774E-01_dp
      fit_coef(11, 13) = -3.2755040336927682E-01_dp
      fit_coef(12, 13) = 5.2516019203549216E-02_dp
      fit_coef(0, 14) = 4.3512255060467424E-04_dp
      fit_coef(1, 14) = 1.1619032009340215E-03_dp
      fit_coef(2, 14) = 7.2125439844170202E-03_dp
      fit_coef(3, 14) = -1.5997165186082474E-03_dp
      fit_coef(4, 14) = 1.1580962215615681E-01_dp
      fit_coef(5, 14) = -4.5588347118168826E-01_dp
      fit_coef(6, 14) = 1.4668513671508041E+00_dp
      fit_coef(7, 14) = -3.2658979984403209E+00_dp
      fit_coef(8, 14) = 4.9101588767499029E+00_dp
      fit_coef(9, 14) = -4.9103950358935151E+00_dp
      fit_coef(10, 14) = 3.1335239775801118E+00_dp
      fit_coef(11, 14) = -1.1555673477193260E+00_dp
      fit_coef(12, 14) = 1.8749662399467892E-01_dp
      fit_coef(0, 15) = 1.7310205801983041E-03_dp
      fit_coef(1, 15) = 3.1669171624321819E-03_dp
      fit_coef(2, 15) = 1.8644195870254832E-02_dp
      fit_coef(3, 15) = -1.4397262238866691E-02_dp
      fit_coef(4, 15) = 2.9169977531123992E-01_dp
      fit_coef(5, 15) = -1.3313481914883316E+00_dp
      fit_coef(6, 15) = 4.2756251478512226E+00_dp
      fit_coef(7, 15) = -9.1958972503690166E+00_dp
      fit_coef(8, 15) = 1.3362193435862070E+01_dp
      fit_coef(9, 15) = -1.3004184796302695E+01_dp
      fit_coef(10, 15) = 8.1326698622082620E+00_dp
      fit_coef(11, 15) = -2.9559537831500120E+00_dp
      fit_coef(12, 15) = 4.7473220658578025E-01_dp
      fit_coef(0, 16) = 5.5360416394316771E-03_dp
      fit_coef(1, 16) = 7.4544925165475080E-03_dp
      fit_coef(2, 16) = 4.2442503620852397E-02_dp
      fit_coef(3, 16) = -4.4215606043728234E-02_dp
      fit_coef(4, 16) = 6.3386466456182333E-01_dp
      fit_coef(5, 16) = -3.0715020425513488E+00_dp
      fit_coef(6, 16) = 9.8361982985015253E+00_dp
      fit_coef(7, 16) = -2.1180667612485266E+01_dp
      fit_coef(8, 16) = 3.1043088235838166E+01_dp
      fit_coef(9, 16) = -3.0559569250108336E+01_dp
      fit_coef(10, 16) = 1.9330505471141983E+01_dp
      fit_coef(11, 16) = -7.0975069725987057E+00_dp
      fit_coef(12, 16) = 1.1497097136190815E+00_dp
      fit_coef(0, 17) = 1.5414916387788672E-02_dp
      fit_coef(1, 17) = 1.5715678851517083E-02_dp
      fit_coef(2, 17) = 8.7468870899845605E-02_dp
      fit_coef(3, 17) = -1.0975140358097349E-01_dp
      fit_coef(4, 17) = 1.2852966018847800E+00_dp
      fit_coef(5, 17) = -6.4984894179776092E+00_dp
      fit_coef(6, 17) = 2.1000940295849389E+01_dp
      fit_coef(7, 17) = -4.5396783297216288E+01_dp
      fit_coef(8, 17) = 6.6608738269027413E+01_dp
      fit_coef(9, 17) = -6.5525352583359691E+01_dp
      fit_coef(10, 17) = 4.1386945120960981E+01_dp
      fit_coef(11, 17) = -1.5171963717870183E+01_dp
      fit_coef(12, 17) = 2.4543585537353447E+00_dp
      fit_coef(0, 18) = 3.8916898079420920E-02_dp
      fit_coef(1, 18) = 3.0312084028360865E-02_dp
      fit_coef(2, 18) = 1.6576784827897006E-01_dp
      fit_coef(3, 18) = -2.3516121046359606E-01_dp
      fit_coef(4, 18) = 2.4050118552826443E+00_dp
      fit_coef(5, 18) = -1.2525237294764612E+01_dp
      fit_coef(6, 18) = 4.0835140514931439E+01_dp
      fit_coef(7, 18) = -8.8493887923230830E+01_dp
      fit_coef(8, 18) = 1.2986858749858015E+02_dp
      fit_coef(9, 18) = -1.2770555910039072E+02_dp
      fit_coef(10, 18) = 8.0636021175329191E+01_dp
      fit_coef(11, 18) = -2.9558075550791788E+01_dp
      fit_coef(12, 18) = 4.7822089064411317E+00_dp
      fit_coef(0, 19) = 9.1240612545516228E-02_dp
      fit_coef(1, 19) = 5.4034493382966092E-02_dp
      fit_coef(2, 19) = 2.9135349130073102E-01_dp
      fit_coef(3, 19) = -4.5212301023387952E-01_dp
      fit_coef(4, 19) = 4.1878151793085188E+00_dp
      fit_coef(5, 19) = -2.2306599564702282E+01_dp
      fit_coef(6, 19) = 7.3348603292660968E+01_dp
      fit_coef(7, 19) = -1.5938432150171050E+02_dp
      fit_coef(8, 19) = 2.3403743136945869E+02_dp
      fit_coef(9, 19) = -2.3012589740957122E+02_dp
      fit_coef(10, 19) = 1.4528076150095299E+02_dp
      fit_coef(11, 19) = -5.3246238298596822E+01_dp
      fit_coef(12, 19) = 8.6138613725490618E+00_dp
      fit_coef(0, 20) = 2.0179304564503545E-01_dp
      fit_coef(1, 20) = 8.9210315804133342E-02_dp
      fit_coef(2, 20) = 4.7545204864558721E-01_dp
      fit_coef(3, 20) = -7.9032122760761392E-01_dp
      fit_coef(4, 20) = 6.7848792396969912E+00_dp
      fit_coef(5, 20) = -3.6782770745377228E+01_dp
      fit_coef(6, 20) = 1.2191045579708324E+02_dp
      fit_coef(7, 20) = -2.6562927550077575E+02_dp
      fit_coef(8, 20) = 3.9032415807076654E+02_dp
      fit_coef(9, 20) = -3.8381139592009845E+02_dp
      fit_coef(10, 20) = 2.4226291326011409E+02_dp
      fit_coef(11, 20) = -8.8773520901146185E+01_dp
      fit_coef(12, 20) = 1.4358956500859366E+01_dp
      fit_coef(0, 21) = 4.2593952773320098E-01_dp
      fit_coef(1, 21) = 1.3568653118311338E-01_dp
      fit_coef(2, 21) = 7.1623822743042354E-01_dp
      fit_coef(3, 21) = -1.2560727864754864E+00_dp
      fit_coef(4, 21) = 1.0163959042284175E+01_dp
      fit_coef(5, 21) = -5.5870348063658447E+01_dp
      fit_coef(6, 21) = 1.8650175155799977E+02_dp
      fit_coef(7, 21) = -4.0744671452434335E+02_dp
      fit_coef(8, 21) = 5.9919650717200875E+02_dp
      fit_coef(9, 21) = -5.8926562705630113E+02_dp
      fit_coef(10, 21) = 3.7190276855784708E+02_dp
      fit_coef(11, 21) = -1.3625501508902701E+02_dp
      fit_coef(12, 21) = 2.2035594424138168E+01_dp
      fit_coef(0, 22) = 8.6820953800091549E-01_dp
      fit_coef(1, 22) = 1.8780376015732225E-01_dp
      fit_coef(2, 22) = 9.8395493031048986E-01_dp
      fit_coef(3, 22) = -1.7957997667101941E+00_dp
      fit_coef(4, 22) = 1.3905769981465895E+01_dp
      fit_coef(5, 22) = -7.7228503961529029E+01_dp
      fit_coef(6, 22) = 2.5935228694562625E+02_dp
      fit_coef(7, 22) = -5.6796297694678856E+02_dp
      fit_coef(8, 22) = 8.3593005414822915E+02_dp
      fit_coef(9, 22) = -8.2221938772784824E+02_dp
      fit_coef(10, 22) = 5.1889603350223433E+02_dp
      fit_coef(11, 22) = -1.9008494460566484E+02_dp
      fit_coef(12, 22) = 3.0737084830973959E+01_dp
      fit_coef(0, 23) = 1.7481856703690022E+00_dp
      fit_coef(1, 23) = 2.3346689762517508E-01_dp
      fit_coef(2, 23) = 1.2175326565887574E+00_dp
      fit_coef(3, 23) = -2.2759160332795787E+00_dp
      fit_coef(4, 23) = 1.7165218972872630E+01_dp
      fit_coef(5, 23) = -9.5910398735102618E+01_dp
      fit_coef(6, 23) = 3.2337485666569609E+02_dp
      fit_coef(7, 23) = -7.0937150908633851E+02_dp
      fit_coef(8, 23) = 1.0447103125794226E+03_dp
      fit_coef(9, 23) = -1.0277535716241930E+03_dp
      fit_coef(10, 23) = 6.4860244114600755E+02_dp
      fit_coef(11, 23) = -2.3758314620598588E+02_dp
      fit_coef(12, 23) = 3.8414439806414237E+01_dp
      fit_coef(0, 24) = 3.7882878590945706E+00_dp
      fit_coef(1, 24) = 2.6672643511994515E-01_dp
      fit_coef(2, 24) = 1.3901597873378935E+00_dp
      fit_coef(3, 24) = -2.6062582764354025E+00_dp
      fit_coef(4, 24) = 1.9594239409959744E+01_dp
      fit_coef(5, 24) = -1.0955141547937794E+02_dp
      fit_coef(6, 24) = 3.6959856539692777E+02_dp
      fit_coef(7, 24) = -8.1102106918375466E+02_dp
      fit_coef(8, 24) = 1.1945701938386849E+03_dp
      fit_coef(9, 24) = -1.1752376758852338E+03_dp
      fit_coef(10, 24) = 7.4168531728428854E+02_dp
      fit_coef(11, 24) = -2.7167776208361096E+02_dp
      fit_coef(12, 24) = 4.3926613983789913E+01_dp

   END SUBROUTINE get_fit_coef_k12_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k13_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 4.3806959251380109E-03_dp
      fit_coef(1, 1) = 1.1805095681240434E-02_dp
      fit_coef(2, 1) = 1.5469511812292528E-02_dp
      fit_coef(3, 1) = 1.1417683853936199E-02_dp
      fit_coef(4, 1) = 3.5089172870450503E-03_dp
      fit_coef(5, 1) = -1.6212006877380839E-03_dp
      fit_coef(6, 1) = -1.8991808677819510E-03_dp
      fit_coef(7, 1) = -1.8247535956842715E-04_dp
      fit_coef(8, 1) = 4.6296227124327700E-04_dp
      fit_coef(9, 1) = 3.1714051554687601E-04_dp
      fit_coef(10, 1) = -4.2555794580904834E-04_dp
      fit_coef(11, 1) = 2.1260104216928313E-04_dp
      fit_coef(12, 1) = -4.7302059547381066E-05_dp
      fit_coef(0, 2) = 2.3696032787686165E-02_dp
      fit_coef(1, 2) = 6.2953090652548949E-02_dp
      fit_coef(2, 2) = 8.1812508977763682E-02_dp
      fit_coef(3, 2) = 5.9935099274848913E-02_dp
      fit_coef(4, 2) = 1.8183698651738394E-02_dp
      fit_coef(5, 2) = -8.5926086043909885E-03_dp
      fit_coef(6, 2) = -9.8822906424356552E-03_dp
      fit_coef(7, 2) = -8.9212086720690355E-04_dp
      fit_coef(8, 2) = 2.4413793189226370E-03_dp
      fit_coef(9, 2) = 1.5997746303520928E-03_dp
      fit_coef(10, 2) = -2.1816355097527334E-03_dp
      fit_coef(11, 2) = 1.0890520189793466E-03_dp
      fit_coef(12, 2) = -2.4201086567176617E-04_dp
      fit_coef(0, 3) = 6.1206818631275799E-02_dp
      fit_coef(1, 3) = 1.5800300818436122E-01_dp
      fit_coef(2, 3) = 2.0228324543170234E-01_dp
      fit_coef(3, 3) = 1.4627045924471369E-01_dp
      fit_coef(4, 3) = 4.3383951728004834E-02_dp
      fit_coef(5, 3) = -2.1290904984850527E-02_dp
      fit_coef(6, 3) = -2.3732647813102206E-02_dp
      fit_coef(7, 3) = -1.9047854738674234E-03_dp
      fit_coef(8, 3) = 5.9890949427493156E-03_dp
      fit_coef(9, 3) = 3.6484212780910821E-03_dp
      fit_coef(10, 3) = -5.1182607229190253E-03_dp
      fit_coef(11, 3) = 2.5506650132351419E-03_dp
      fit_coef(12, 3) = -5.6515120961717785E-04_dp
      fit_coef(0, 4) = 1.2273976907574904E-01_dp
      fit_coef(1, 4) = 3.0202610531899604E-01_dp
      fit_coef(2, 4) = 3.7810599512667076E-01_dp
      fit_coef(3, 4) = 2.6877539512343529E-01_dp
      fit_coef(4, 4) = 7.7369990738750394E-02_dp
      fit_coef(5, 4) = -3.9931781197404183E-02_dp
      fit_coef(6, 4) = -4.2604215569189550E-02_dp
      fit_coef(7, 4) = -2.7223623128689494E-03_dp
      fit_coef(8, 4) = 1.1010550845635398E-02_dp
      fit_coef(9, 4) = 5.9806779753877508E-03_dp
      fit_coef(10, 4) = -8.7651481516437067E-03_dp
      fit_coef(11, 4) = 4.3537113139725109E-03_dp
      fit_coef(12, 4) = -9.6064235434420908E-04_dp
      fit_coef(0, 5) = 2.1922884952130442E-01_dp
      fit_coef(1, 5) = 5.0239277267534854E-01_dp
      fit_coef(2, 5) = 6.0878683083947316E-01_dp
      fit_coef(3, 5) = 4.2445143636733651E-01_dp
      fit_coef(4, 5) = 1.1949507494490993E-01_dp
      fit_coef(5, 5) = -6.4343269144918072E-02_dp
      fit_coef(6, 5) = -6.6581789191562657E-02_dp
      fit_coef(7, 5) = -3.2621357858438767E-03_dp
      fit_coef(8, 5) = 2.0013098730936332E-02_dp
      fit_coef(9, 5) = 5.6869475340117050E-03_dp
      fit_coef(10, 5) = -1.2059838120484294E-02_dp
      fit_coef(11, 5) = 6.5584888368600807E-03_dp
      fit_coef(12, 5) = -1.5261431175938014E-03_dp
      fit_coef(0, 6) = 3.6944915773408882E-01_dp
      fit_coef(1, 6) = 7.6979797793437166E-01_dp
      fit_coef(2, 6) = 8.9033669564805873E-01_dp
      fit_coef(3, 6) = 6.0532992150591614E-01_dp
      fit_coef(4, 6) = 1.7130397954922461E-01_dp
      fit_coef(5, 6) = -8.8192182985475279E-02_dp
      fit_coef(6, 6) = -9.7966070392074389E-02_dp
      fit_coef(7, 6) = -4.8436829553671259E-03_dp
      fit_coef(8, 6) = 2.0833002694907763E-02_dp
      fit_coef(9, 6) = 3.5063121112200099E-02_dp
      fit_coef(10, 6) = -4.7438965646280404E-02_dp
      fit_coef(11, 6) = 2.4703796048967528E-02_dp
      fit_coef(12, 6) = -5.2577610515280489E-03_dp
      fit_coef(0, 7) = 6.0348926202620468E-01_dp
      fit_coef(1, 7) = 1.1189609399540368E+00_dp
      fit_coef(2, 7) = 1.2166554041301936E+00_dp
      fit_coef(3, 7) = 7.9564460112103685E-01_dp
      fit_coef(4, 7) = 2.2942989193522059E-01_dp
      fit_coef(5, 7) = -9.7714358227207448E-02_dp
      fit_coef(6, 7) = -1.2746711453160517E-01_dp
      fit_coef(7, 7) = -1.1047808976133488E-02_dp
      fit_coef(8, 7) = -6.1793196597745570E-03_dp
      fit_coef(9, 7) = 8.7267129421221348E-02_dp
      fit_coef(10, 7) = -6.4166653350011871E-02_dp
      fit_coef(11, 7) = 1.4747313101772477E-02_dp
      fit_coef(12, 7) = 2.3977301657934416E-04_dp
      fit_coef(0, 8) = 9.6768723967411674E-01_dp
      fit_coef(1, 8) = 1.5668848161581139E+00_dp
      fit_coef(2, 8) = 1.5818658236819807E+00_dp
      fit_coef(3, 8) = 9.7733015664802736E-01_dp
      fit_coef(4, 8) = 2.8255279451553161E-01_dp
      fit_coef(5, 8) = -8.9391803088038346E-02_dp
      fit_coef(6, 8) = -1.2421016153669295E-01_dp
      fit_coef(7, 8) = -6.1021516900378492E-02_dp
      fit_coef(8, 8) = 6.2677519441489124E-02_dp
      fit_coef(9, 8) = -8.0250399433389449E-02_dp
      fit_coef(10, 8) = 1.5369427952945630E-01_dp
      fit_coef(11, 8) = -1.1092684421625504E-01_dp
      fit_coef(12, 8) = 2.7549923051188644E-02_dp
      fit_coef(0, 9) = 1.5325340819017725E+00_dp
      fit_coef(1, 9) = 2.1287996982316222E+00_dp
      fit_coef(2, 9) = 1.9799152367840203E+00_dp
      fit_coef(3, 9) = 1.1370787116857501E+00_dp
      fit_coef(4, 9) = 3.2008697898446903E-01_dp
      fit_coef(5, 9) = -6.6640990523988475E-02_dp
      fit_coef(6, 9) = -9.5466135919243425E-02_dp
      fit_coef(7, 9) = -9.2042888056781805E-02_dp
      fit_coef(8, 9) = 9.7360896464160462E-02_dp
      fit_coef(9, 9) = -1.8127317675763444E-01_dp
      fit_coef(10, 9) = 2.2214526244390012E-01_dp
      fit_coef(11, 9) = -1.0494335632769292E-01_dp
      fit_coef(12, 9) = 1.5481275818011413E-02_dp
      fit_coef(0, 10) = 2.4061866675798735E+00_dp
      fit_coef(1, 10) = 2.8132877393110070E+00_dp
      fit_coef(2, 10) = 2.4037430753636428E+00_dp
      fit_coef(3, 10) = 1.2687163022362127E+00_dp
      fit_coef(4, 10) = 3.3899108614443485E-01_dp
      fit_coef(5, 10) = -3.2980562806816459E-02_dp
      fit_coef(6, 10) = -6.4495629707069638E-02_dp
      fit_coef(7, 10) = -4.9196501013761162E-02_dp
      fit_coef(8, 10) = -1.8973498191644747E-02_dp
      fit_coef(9, 10) = -1.6127741943174356E-02_dp
      fit_coef(10, 10) = 1.0778655156935451E-03_dp
      fit_coef(11, 10) = 4.9661103678829296E-02_dp
      fit_coef(12, 10) = -2.4208904460265530E-02_dp
      fit_coef(0, 11) = 3.7607303248227026E+00_dp
      fit_coef(1, 11) = 3.6183981611913807E+00_dp
      fit_coef(2, 11) = 2.8473208041198181E+00_dp
      fit_coef(3, 11) = 1.3735956585457292E+00_dp
      fit_coef(4, 11) = 3.4303721265244597E-01_dp
      fit_coef(5, 11) = 5.1365790260944994E-03_dp
      fit_coef(6, 11) = -2.3683792375462247E-02_dp
      fit_coef(7, 11) = -7.7157824160531199E-03_dp
      fit_coef(8, 11) = -9.7425777898074148E-02_dp
      fit_coef(9, 11) = 8.3518836722114576E-02_dp
      fit_coef(10, 11) = -1.1044052160199369E-01_dp
      fit_coef(11, 11) = 1.0133376735634138E-01_dp
      fit_coef(12, 11) = -2.9906470245636543E-02_dp
      fit_coef(0, 12) = 5.8979582300825903E+00_dp
      fit_coef(1, 12) = 4.5329131993702125E+00_dp
      fit_coef(2, 12) = 3.3115813786557839E+00_dp
      fit_coef(3, 12) = 1.4609520855498697E+00_dp
      fit_coef(4, 12) = 3.3783414333423767E-01_dp
      fit_coef(5, 12) = 4.3826302981445429E-02_dp
      fit_coef(6, 12) = 2.7923917468353154E-02_dp
      fit_coef(7, 12) = 2.7776299343854542E-03_dp
      fit_coef(8, 12) = -8.3036915110656417E-02_dp
      fit_coef(9, 12) = 3.7742211424302625E-02_dp
      fit_coef(10, 12) = -4.9885068616704199E-02_dp
      fit_coef(11, 12) = 4.5143513558608316E-02_dp
      fit_coef(12, 12) = -1.0663317459135431E-02_dp
      fit_coef(0, 13) = 9.5045784867374294E+00_dp
      fit_coef(1, 13) = 5.5579146894547398E+00_dp
      fit_coef(2, 13) = 3.8186762276220274E+00_dp
      fit_coef(3, 13) = 1.5494988454410836E+00_dp
      fit_coef(4, 13) = 3.2780283598581783E-01_dp
      fit_coef(5, 13) = 8.1835332761182517E-02_dp
      fit_coef(6, 13) = 8.1960790462737673E-02_dp
      fit_coef(7, 13) = 8.9017910412122984E-03_dp
      fit_coef(8, 13) = -5.2756529738479566E-02_dp
      fit_coef(9, 13) = -1.9132069822996558E-02_dp
      fit_coef(10, 13) = 2.2550262525116004E-02_dp
      fit_coef(11, 13) = -7.1174631455651191E-03_dp
      fit_coef(12, 13) = 2.6153386964695999E-03_dp
      fit_coef(0, 14) = 1.1310131998968936E-02_dp
      fit_coef(1, 14) = 3.0381882917892972E-02_dp
      fit_coef(2, 14) = 3.9735561262063152E-02_dp
      fit_coef(3, 14) = 2.9276417146478988E-02_dp
      fit_coef(4, 14) = 8.9698683531803033E-03_dp
      fit_coef(5, 14) = -4.1668138620200067E-03_dp
      fit_coef(6, 14) = -4.8597312635549313E-03_dp
      fit_coef(7, 14) = -4.6021605223065968E-04_dp
      fit_coef(8, 14) = 1.1885176206365188E-03_dp
      fit_coef(9, 14) = 8.0548775761279056E-04_dp
      fit_coef(10, 14) = -1.0849907036440899E-03_dp
      fit_coef(11, 14) = 5.4194768198807949E-04_dp
      fit_coef(12, 14) = -1.2054873767366686E-04_dp
      fit_coef(0, 15) = 2.7754333434518740E-02_dp
      fit_coef(1, 15) = 7.2425221320615574E-02_dp
      fit_coef(2, 15) = 9.3171796368255483E-02_dp
      fit_coef(3, 15) = 6.7635165400501615E-02_dp
      fit_coef(4, 15) = 2.0195124261742557E-02_dp
      fit_coef(5, 15) = -9.8067712377566344E-03_dp
      fit_coef(6, 15) = -1.1030330619304156E-02_dp
      fit_coef(7, 15) = -9.1656688079229311E-04_dp
      fit_coef(8, 15) = 2.7689224656143080E-03_dp
      fit_coef(9, 15) = 1.7173968423888279E-03_dp
      fit_coef(10, 15) = -2.3913024611684291E-03_dp
      fit_coef(11, 15) = 1.1924633353927081E-03_dp
      fit_coef(12, 15) = -2.6453134445543388E-04_dp
      fit_coef(0, 16) = 4.8205145888088087E-02_dp
      fit_coef(1, 16) = 1.1851851999761799E-01_dp
      fit_coef(2, 16) = 1.4800321928791688E-01_dp
      fit_coef(3, 16) = 1.0475875377187537E-01_dp
      fit_coef(4, 16) = 2.9891263684815553E-02_dp
      fit_coef(5, 16) = -1.5617974138582616E-02_dp
      fit_coef(6, 16) = -1.6518323399901058E-02_dp
      fit_coef(7, 16) = -1.0386542240169942E-03_dp
      fit_coef(8, 16) = 4.3031804541452875E-03_dp
      fit_coef(9, 16) = 2.3296942214223468E-03_dp
      fit_coef(10, 16) = -3.4336179918521620E-03_dp
      fit_coef(11, 16) = 1.7042723642156064E-03_dp
      fit_coef(12, 16) = -3.7515429160435323E-04_dp
      fit_coef(0, 17) = 7.6637000090091376E-02_dp
      fit_coef(1, 17) = 1.7073623076407599E-01_dp
      fit_coef(2, 17) = 2.0360358978335744E-01_dp
      fit_coef(3, 17) = 1.3981725357818653E-01_dp
      fit_coef(4, 17) = 3.7911755072739847E-02_dp
      fit_coef(5, 17) = -2.1730826974108042E-02_dp
      fit_coef(6, 17) = -2.1165853445559209E-02_dp
      fit_coef(7, 17) = -4.3396413995129904E-04_dp
      fit_coef(8, 17) = 5.9819715180725796E-03_dp
      fit_coef(9, 17) = 1.7458780018996711E-03_dp
      fit_coef(10, 17) = -3.3770780241614367E-03_dp
      fit_coef(11, 17) = 1.6967821161902551E-03_dp
      fit_coef(12, 17) = -3.7810404338462888E-04_dp
      fit_coef(0, 18) = 1.1941431468383547E-01_dp
      fit_coef(1, 18) = 2.3176044822900177E-01_dp
      fit_coef(2, 18) = 2.5706191693734187E-01_dp
      fit_coef(3, 18) = 1.7029046402780762E-01_dp
      fit_coef(4, 18) = 4.6762674262981342E-02_dp
      fit_coef(5, 18) = -2.6181781674092602E-02_dp
      fit_coef(6, 18) = -2.7357912314188080E-02_dp
      fit_coef(7, 18) = -1.4374470171699591E-03_dp
      fit_coef(8, 18) = 1.2160163836047608E-02_dp
      fit_coef(9, 18) = 7.2851082934719002E-04_dp
      fit_coef(10, 18) = -7.7359567257111982E-03_dp
      fit_coef(11, 18) = 5.4157372236747318E-03_dp
      fit_coef(12, 18) = -1.3750847828525607E-03_dp
      fit_coef(0, 19) = 1.8596241285720252E-01_dp
      fit_coef(1, 19) = 3.0553364801134103E-01_dp
      fit_coef(2, 19) = 3.0496827359436179E-01_dp
      fit_coef(3, 19) = 1.8871536236922606E-01_dp
      fit_coef(4, 19) = 5.6375245286351912E-02_dp
      fit_coef(5, 19) = -1.8700350814395448E-02_dp
      fit_coef(6, 19) = -3.4527217392530681E-02_dp
      fit_coef(7, 19) = -6.3872694617221477E-05_dp
      fit_coef(8, 19) = -2.2205235124562510E-02_dp
      fit_coef(9, 19) = 7.0816767471594697E-02_dp
      fit_coef(10, 19) = -6.6120085860146768E-02_dp
      fit_coef(11, 19) = 2.8289249488746906E-02_dp
      fit_coef(12, 19) = -4.7843132337791906E-03_dp
      fit_coef(0, 20) = 2.8971434006808466E-01_dp
      fit_coef(1, 20) = 3.9569789839598735E-01_dp
      fit_coef(2, 20) = 3.4670622356516184E-01_dp
      fit_coef(3, 20) = 1.8876868004229172E-01_dp
      fit_coef(4, 20) = 5.7788519919382296E-02_dp
      fit_coef(5, 20) = 5.4887363554269302E-05_dp
      fit_coef(6, 20) = -1.7484482268241279E-02_dp
      fit_coef(7, 20) = -2.3812061294207602E-02_dp
      fit_coef(8, 20) = 3.6673498428854141E-03_dp
      fit_coef(9, 20) = -3.6033615474019322E-02_dp
      fit_coef(10, 20) = 9.9463288944652603E-02_dp
      fit_coef(11, 20) = -7.9311225375931868E-02_dp
      fit_coef(12, 20) = 2.1453780314514743E-02_dp
      fit_coef(0, 21) = 4.5021808525283785E-01_dp
      fit_coef(1, 21) = 5.0277275639459118E-01_dp
      fit_coef(2, 21) = 3.8272861573261618E-01_dp
      fit_coef(3, 21) = 1.7235807446302881E-01_dp
      fit_coef(4, 21) = 4.6572363862520262E-02_dp
      fit_coef(5, 21) = 1.5881707525432769E-02_dp
      fit_coef(6, 21) = 2.2025322210179048E-02_dp
      fit_coef(7, 21) = -6.2308984089511119E-02_dp
      fit_coef(8, 21) = 1.0309535046393066E-01_dp
      fit_coef(9, 21) = -2.2967091111578883E-01_dp
      fit_coef(10, 21) = 2.4313367642185343E-01_dp
      fit_coef(11, 21) = -1.1137814444194188E-01_dp
      fit_coef(12, 21) = 1.7887958008893388E-02_dp
      fit_coef(0, 22) = 6.9710873795862005E-01_dp
      fit_coef(1, 22) = 6.2256041357045133E-01_dp
      fit_coef(2, 22) = 4.1216050290755374E-01_dp
      fit_coef(3, 22) = 1.4610725607133404E-01_dp
      fit_coef(4, 22) = 2.8009636014951746E-02_dp
      fit_coef(5, 22) = 2.9171197185645873E-02_dp
      fit_coef(6, 22) = 3.0389459142154723E-02_dp
      fit_coef(7, 22) = 1.2605729292193118E-02_dp
      fit_coef(8, 22) = -6.0485512980266130E-02_dp
      fit_coef(9, 22) = 7.0433726753735382E-02_dp
      fit_coef(10, 22) = -1.3375453829923470E-01_dp
      fit_coef(11, 22) = 1.2456403150558767E-01_dp
      fit_coef(12, 22) = -3.8728626230412060E-02_dp
      fit_coef(0, 23) = 1.0779765313876117E+00_dp
      fit_coef(1, 23) = 7.4609279169328480E-01_dp
      fit_coef(2, 23) = 4.3439321318574559E-01_dp
      fit_coef(3, 23) = 1.1738611843507005E-01_dp
      fit_coef(4, 23) = 1.0530309563700853E-02_dp
      fit_coef(5, 23) = 3.6897191893439256E-02_dp
      fit_coef(6, 23) = 3.4044088146621569E-02_dp
      fit_coef(7, 23) = 5.5410057699209495E-02_dp
      fit_coef(8, 23) = -1.2792045350588788E-01_dp
      fit_coef(9, 23) = 1.8262592506787398E-01_dp
      fit_coef(10, 23) = -2.1867054150540904E-01_dp
      fit_coef(11, 23) = 1.2980623399937813E-01_dp
      fit_coef(12, 23) = -2.7516040284682932E-02_dp
      fit_coef(0, 24) = 1.6785411350231978E+00_dp
      fit_coef(1, 24) = 8.6201990862233702E-01_dp
      fit_coef(2, 24) = 4.5273971353495590E-01_dp
      fit_coef(3, 24) = 9.3905664009676981E-02_dp
      fit_coef(4, 24) = -1.4517367291959843E-03_dp
      fit_coef(5, 24) = 3.8726538172099936E-02_dp
      fit_coef(6, 24) = 4.7579808176484301E-02_dp
      fit_coef(7, 24) = 2.3323849168230236E-02_dp
      fit_coef(8, 24) = -2.2238667006060801E-02_dp
      fit_coef(9, 24) = 8.3019689769436942E-03_dp
      fit_coef(10, 24) = -8.7391379765216180E-04_dp
      fit_coef(11, 24) = -2.1892298772107112E-02_dp
      fit_coef(12, 24) = 1.2948019279576950E-02_dp
      fit_coef(0, 25) = 2.6946568599370697E+00_dp
      fit_coef(1, 25) = 9.6576339244071330E-01_dp
      fit_coef(2, 25) = 4.7873508727946390E-01_dp
      fit_coef(3, 25) = 8.3762183149891992E-02_dp
      fit_coef(4, 25) = -8.2300319329272915E-03_dp
      fit_coef(5, 25) = 3.8469555958125901E-02_dp
      fit_coef(6, 25) = 5.3859007054592130E-02_dp
      fit_coef(7, 25) = 3.4301845918780008E-03_dp
      fit_coef(8, 25) = 3.4783546363171880E-02_dp
      fit_coef(9, 25) = -7.1844324491727757E-02_dp
      fit_coef(10, 25) = 8.9320542108673212E-02_dp
      fit_coef(11, 25) = -6.8466536447383028E-02_dp
      fit_coef(12, 25) = 1.9631577406052880E-02_dp
      fit_coef(0, 26) = 4.8659851910149703E+00_dp
      fit_coef(1, 26) = 1.1016222687932846E+00_dp
      fit_coef(2, 26) = 5.5082141072412572E-01_dp
      fit_coef(3, 26) = 9.9924055011494373E-02_dp
      fit_coef(4, 26) = -1.1242348443643024E-02_dp
      fit_coef(5, 26) = 3.7638525869382183E-02_dp
      fit_coef(6, 26) = 5.4458588514965843E-02_dp
      fit_coef(7, 26) = 1.1975138580003340E-02_dp
      fit_coef(8, 26) = 1.9387184067595927E-02_dp
      fit_coef(9, 26) = -3.2120751380981701E-02_dp
      fit_coef(10, 26) = 4.5502034364108004E-02_dp
      fit_coef(11, 26) = -3.0627416778773675E-02_dp
      fit_coef(12, 26) = 6.0865046197086294E-03_dp

   END SUBROUTINE get_fit_coef_k13_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k13_mi(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.1693251854461644E+03_dp
      fit_coef(0, 1) = 5.9571247085285603E-04_dp
      fit_coef(1, 1) = 1.7200118003776515E-03_dp
      fit_coef(2, 1) = 2.4894492611009689E-03_dp
      fit_coef(3, 1) = 2.1537471215151561E-03_dp
      fit_coef(4, 1) = 1.0174139924949508E-03_dp
      fit_coef(5, 1) = 4.6357691642043961E-05_dp
      fit_coef(6, 1) = -2.2922455590426252E-04_dp
      fit_coef(7, 1) = -4.5027856096292727E-05_dp
      fit_coef(8, 1) = 3.2368140787386204E-05_dp
      fit_coef(9, 1) = 7.8015031488317424E-05_dp
      fit_coef(10, 1) = -9.1410611282450338E-05_dp
      fit_coef(11, 1) = 4.0455359969903499E-05_dp
      fit_coef(12, 1) = -6.8554946194150892E-06_dp
      fit_coef(0, 2) = 3.3749978736579119E-03_dp
      fit_coef(1, 2) = 9.3865303071796868E-03_dp
      fit_coef(2, 2) = 1.3370050944546014E-02_dp
      fit_coef(3, 2) = 1.1408817834538747E-02_dp
      fit_coef(4, 2) = 5.3018599368705056E-03_dp
      fit_coef(5, 2) = 1.9895208136664570E-04_dp
      fit_coef(6, 2) = -1.2013959538153690E-03_dp
      fit_coef(7, 2) = -2.1522942347644450E-04_dp
      fit_coef(8, 2) = 1.6227159652722640E-04_dp
      fit_coef(9, 2) = 4.0937759938008522E-04_dp
      fit_coef(10, 2) = -4.8203337203911678E-04_dp
      fit_coef(11, 2) = 2.1503525076000546E-04_dp
      fit_coef(12, 2) = -3.6767133200206401E-05_dp
      fit_coef(0, 3) = 9.5695858129820257E-03_dp
      fit_coef(1, 3) = 2.4594246941685691E-02_dp
      fit_coef(2, 3) = 3.3909912181295111E-02_dp
      fit_coef(3, 3) = 2.8275272197259928E-02_dp
      fit_coef(4, 3) = 1.2793089131376839E-02_dp
      fit_coef(5, 3) = 2.8137691410500335E-04_dp
      fit_coef(6, 3) = -2.9348509030753785E-03_dp
      fit_coef(7, 3) = -4.3154495298106305E-04_dp
      fit_coef(8, 3) = 4.1400164345487900E-04_dp
      fit_coef(9, 3) = 9.1497967805617892E-04_dp
      fit_coef(10, 3) = -1.1243533488522371E-03_dp
      fit_coef(11, 3) = 5.1331094158916764E-04_dp
      fit_coef(12, 3) = -8.9543140456745687E-05_dp
      fit_coef(0, 4) = 2.2102062421629307E-02_dp
      fit_coef(1, 4) = 5.0388239734107464E-02_dp
      fit_coef(2, 4) = 6.5597474400466413E-02_dp
      fit_coef(3, 4) = 5.2593837264420067E-02_dp
      fit_coef(4, 4) = 2.3138245186338603E-02_dp
      fit_coef(5, 4) = 3.5363359503456162E-04_dp
      fit_coef(6, 4) = -5.4308630565349861E-03_dp
      fit_coef(7, 4) = -7.1690573641355025E-04_dp
      fit_coef(8, 4) = 4.5525716627285349E-04_dp
      fit_coef(9, 4) = 2.3355643424946655E-03_dp
      fit_coef(10, 4) = -2.6681070004726240E-03_dp
      fit_coef(11, 4) = 1.2090694842315898E-03_dp
      fit_coef(12, 4) = -2.1156836663214308E-04_dp
      fit_coef(0, 5) = 4.6953984288651295E-02_dp
      fit_coef(1, 5) = 9.2513506061771736E-02_dp
      fit_coef(2, 5) = 1.1114827337176617E-01_dp
      fit_coef(3, 5) = 8.3233678613110429E-02_dp
      fit_coef(4, 5) = 3.4608571546062436E-02_dp
      fit_coef(5, 5) = 6.2863207858912962E-04_dp
      fit_coef(6, 5) = -7.4478035609513958E-03_dp
      fit_coef(7, 5) = -1.1406622073555459E-03_dp
      fit_coef(8, 5) = 6.1142108219932139E-05_dp
      fit_coef(9, 5) = 2.8713082420645268E-03_dp
      fit_coef(10, 5) = -2.3152373063920332E-03_dp
      fit_coef(11, 5) = 6.9898905114079072E-04_dp
      fit_coef(12, 5) = -6.0162701114697736E-05_dp
      fit_coef(0, 6) = 9.5182645385563286E-02_dp
      fit_coef(1, 6) = 1.5985393660033628E-01_dp
      fit_coef(2, 6) = 1.7511073340807867E-01_dp
      fit_coef(3, 6) = 1.1909736973485001E-01_dp
      fit_coef(4, 6) = 4.4162492084356325E-02_dp
      fit_coef(5, 6) = -1.7481218668970254E-04_dp
      fit_coef(6, 6) = -7.8382486311276943E-03_dp
      fit_coef(7, 6) = -6.5498122365884547E-04_dp
      fit_coef(8, 6) = 7.2110889562901611E-05_dp
      fit_coef(9, 6) = 9.5978488993906188E-04_dp
      fit_coef(10, 6) = -6.1518613580109974E-04_dp
      fit_coef(11, 6) = 2.4316501239763139E-04_dp
      fit_coef(12, 6) = -5.3529912569972472E-05_dp
      fit_coef(0, 7) = 1.8645165311079259E-01_dp
      fit_coef(1, 7) = 2.6436160380537138E-01_dp
      fit_coef(2, 7) = 2.6302083875944771E-01_dp
      fit_coef(3, 7) = 1.5910774510414938E-01_dp
      fit_coef(4, 7) = 4.9073653325355709E-02_dp
      fit_coef(5, 7) = -3.2795187276607025E-03_dp
      fit_coef(6, 7) = -6.8810055747994857E-03_dp
      fit_coef(7, 7) = 2.0021507448155459E-03_dp
      fit_coef(8, 7) = -1.4027814134567758E-03_dp
      fit_coef(9, 7) = 6.9098043550271603E-04_dp
      fit_coef(10, 7) = -1.0637655745511332E-03_dp
      fit_coef(11, 7) = 9.2536193146607883E-04_dp
      fit_coef(12, 7) = -2.4940185896462312E-04_dp
      fit_coef(0, 8) = 3.5501666895531042E-01_dp
      fit_coef(1, 8) = 4.2062390191703830E-01_dp
      fit_coef(2, 8) = 3.8014762337012359E-01_dp
      fit_coef(3, 8) = 2.0109387264368472E-01_dp
      fit_coef(4, 8) = 4.6954945508850141E-02_dp
      fit_coef(5, 8) = -8.7244019551801829E-03_dp
      fit_coef(6, 8) = -4.0487338235444901E-03_dp
      fit_coef(7, 8) = 5.9512521905139611E-03_dp
      fit_coef(8, 8) = -3.7498978634030503E-03_dp
      fit_coef(9, 8) = 2.1099763933573197E-04_dp
      fit_coef(10, 8) = -4.3311783252141623E-04_dp
      fit_coef(11, 8) = 7.0199947379022344E-04_dp
      fit_coef(12, 8) = -2.0527979943177016E-04_dp
      fit_coef(0, 9) = 6.5960294052628099E-01_dp
      fit_coef(1, 9) = 6.4434244249393713E-01_dp
      fit_coef(2, 9) = 5.3018928176744540E-01_dp
      fit_coef(3, 9) = 2.4130639214143917E-01_dp
      fit_coef(4, 9) = 3.5653141239669908E-02_dp
      fit_coef(5, 9) = -1.5407439897727408E-02_dp
      fit_coef(6, 9) = 1.6157905391085510E-03_dp
      fit_coef(7, 9) = 1.0708866394811738E-02_dp
      fit_coef(8, 9) = -8.0705877921707705E-03_dp
      fit_coef(9, 9) = 1.9585451901869654E-04_dp
      fit_coef(10, 9) = 1.1606320581042068E-03_dp
      fit_coef(11, 9) = -2.3231700639256246E-04_dp
      fit_coef(12, 9) = -1.9173218066136157E-05_dp
      fit_coef(0, 10) = 1.1999125471871359E+00_dp
      fit_coef(1, 10) = 9.4887754086202536E-01_dp
      fit_coef(2, 10) = 7.1392148306788861E-01_dp
      fit_coef(3, 10) = 2.7502418030970321E-01_dp
      fit_coef(4, 10) = 1.3864789499781904E-02_dp
      fit_coef(5, 10) = -2.1424737017769674E-02_dp
      fit_coef(6, 10) = 1.0990606216211212E-02_dp
      fit_coef(7, 10) = 1.5754376025894169E-02_dp
      fit_coef(8, 10) = -1.5720284237712618E-02_dp
      fit_coef(9, 10) = 2.0987333422688541E-03_dp
      fit_coef(10, 10) = 2.9263737858274525E-03_dp
      fit_coef(11, 10) = -1.6119558060080142E-03_dp
      fit_coef(12, 10) = 2.7700177106853099E-04_dp
      fit_coef(0, 11) = 2.1464589921277430E+00_dp
      fit_coef(1, 11) = 1.3398008082135153E+00_dp
      fit_coef(2, 11) = 9.2855004111736639E-01_dp
      fit_coef(3, 11) = 2.9850369229739804E-01_dp
      fit_coef(4, 11) = -1.8024501913163959E-02_dp
      fit_coef(5, 11) = -2.4792994220460485E-02_dp
      fit_coef(6, 11) = 2.4309270311746085E-02_dp
      fit_coef(7, 11) = 2.0376050200980579E-02_dp
      fit_coef(8, 11) = -2.7413278630826550E-02_dp
      fit_coef(9, 11) = 7.1577405110793838E-03_dp
      fit_coef(10, 11) = 4.1376588773677659E-03_dp
      fit_coef(11, 11) = -3.2837333351403799E-03_dp
      fit_coef(12, 11) = 6.9015069153665373E-04_dp
      fit_coef(0, 12) = 3.8079327315676679E+00_dp
      fit_coef(1, 12) = 1.8102136985271529E+00_dp
      fit_coef(2, 12) = 1.1693768610866579E+00_dp
      fit_coef(3, 12) = 3.1192154530912264E-01_dp
      fit_coef(4, 12) = -5.7801420614916721E-02_dp
      fit_coef(5, 12) = -2.4725077881519446E-02_dp
      fit_coef(6, 12) = 4.0887008920651363E-02_dp
      fit_coef(7, 12) = 2.4169421898620948E-02_dp
      fit_coef(8, 12) = -4.2916907464272722E-02_dp
      fit_coef(9, 12) = 1.5836520381657740E-02_dp
      fit_coef(10, 12) = 4.2153911444231572E-03_dp
      fit_coef(11, 12) = -5.0069163439141369E-03_dp
      fit_coef(12, 12) = 1.1850222139382642E-03_dp
      fit_coef(0, 13) = 6.8767988424609685E+00_dp
      fit_coef(1, 13) = 2.3477041904049458E+00_dp
      fit_coef(2, 13) = 1.4371470680965179E+00_dp
      fit_coef(3, 13) = 3.2177727032276354E-01_dp
      fit_coef(4, 13) = -1.0285806633893342E-01_dp
      fit_coef(5, 13) = -2.2677975661685234E-02_dp
      fit_coef(6, 13) = 5.9703880119426990E-02_dp
      fit_coef(7, 13) = 2.7586476735575311E-02_dp
      fit_coef(8, 13) = -6.1182232779440265E-02_dp
      fit_coef(9, 13) = 2.7103387955214536E-02_dp
      fit_coef(10, 13) = 3.4260870398828006E-03_dp
      fit_coef(11, 13) = -6.7215676437135359E-03_dp
      fit_coef(12, 13) = 1.7275038459318248E-03_dp
      fit_coef(0, 14) = 1.5536180830079045E-03_dp
      fit_coef(1, 14) = 4.4506140115349617E-03_dp
      fit_coef(2, 14) = 6.4178048727277781E-03_dp
      fit_coef(3, 14) = 5.5342602456232345E-03_dp
      fit_coef(4, 14) = 2.6046871795835742E-03_dp
      fit_coef(5, 14) = 1.1368344431687511E-04_dp
      fit_coef(6, 14) = -5.8781627121643181E-04_dp
      fit_coef(7, 14) = -1.1285527376227048E-04_dp
      fit_coef(8, 14) = 8.2159134979200638E-05_dp
      fit_coef(9, 14) = 1.9994801604583968E-04_dp
      fit_coef(10, 14) = -2.3463016476463693E-04_dp
      fit_coef(11, 14) = 1.0405614708917490E-04_dp
      fit_coef(12, 14) = -1.7674175132111243E-05_dp
      fit_coef(0, 15) = 4.1834098874360473E-03_dp
      fit_coef(1, 15) = 1.1102877368401057E-02_dp
      fit_coef(2, 15) = 1.5508145773336595E-02_dp
      fit_coef(3, 15) = 1.3020239177249622E-02_dp
      fit_coef(4, 15) = 5.9265544529938010E-03_dp
      fit_coef(5, 15) = 1.6028851543557779E-04_dp
      fit_coef(6, 15) = -1.3496676957738614E-03_dp
      fit_coef(7, 15) = -2.1194238008936958E-04_dp
      fit_coef(8, 15) = 1.6858962619466225E-04_dp
      fit_coef(9, 15) = 4.6274241631720910E-04_dp
      fit_coef(10, 15) = -5.4510567016319549E-04_dp
      fit_coef(11, 15) = 2.4471543070832161E-04_dp
      fit_coef(12, 15) = -4.2146500465828684E-05_dp
      fit_coef(0, 16) = 8.6718931437848486E-03_dp
      fit_coef(1, 16) = 1.9798273846496010E-02_dp
      fit_coef(2, 16) = 2.5805909237192746E-02_dp
      fit_coef(3, 16) = 2.0709762497138948E-02_dp
      fit_coef(4, 16) = 9.0267828044289580E-03_dp
      fit_coef(5, 16) = -8.2030242003155534E-06_dp
      fit_coef(6, 16) = -2.1466923201882496E-03_dp
      fit_coef(7, 16) = -2.3839418666152536E-04_dp
      fit_coef(8, 16) = 3.9713557969496511E-04_dp
      fit_coef(9, 16) = 5.4418611978137037E-04_dp
      fit_coef(10, 16) = -7.9800986012929951E-04_dp
      fit_coef(11, 16) = 3.9897082718071900E-04_dp
      fit_coef(12, 16) = -7.5484271874357470E-05_dp
      fit_coef(0, 17) = 1.7358581030722377E-02_dp
      fit_coef(1, 17) = 3.2741569114568887E-02_dp
      fit_coef(2, 17) = 3.8005765925330802E-02_dp
      fit_coef(3, 17) = 2.7705507640870534E-02_dp
      fit_coef(4, 17) = 1.1375410029678385E-02_dp
      fit_coef(5, 17) = 2.5826247104955189E-04_dp
      fit_coef(6, 17) = -2.6224716007508839E-03_dp
      fit_coef(7, 17) = -3.1373038771318260E-04_dp
      fit_coef(8, 17) = -6.4482138467522902E-04_dp
      fit_coef(9, 17) = 2.3188017066251647E-03_dp
      fit_coef(10, 17) = -1.8974564925171979E-03_dp
      fit_coef(11, 17) = 6.6385240884256846E-04_dp
      fit_coef(12, 17) = -8.2217281900096977E-05_dp
      fit_coef(0, 18) = 3.4131291193668846E-02_dp
      fit_coef(1, 18) = 5.2991311639389929E-02_dp
      fit_coef(2, 18) = 5.3873497163302024E-02_dp
      fit_coef(3, 18) = 3.3395673340616244E-02_dp
      fit_coef(4, 18) = 1.1018355439494237E-02_dp
      fit_coef(5, 18) = 1.9602178372730269E-05_dp
      fit_coef(6, 18) = -1.1688901927899766E-03_dp
      fit_coef(7, 18) = -4.0116811266019766E-04_dp
      fit_coef(8, 18) = 3.2072961899086736E-04_dp
      fit_coef(9, 18) = -1.9201145783602021E-03_dp
      fit_coef(10, 18) = 2.5996845486259772E-03_dp
      fit_coef(11, 18) = -1.3613851440031023E-03_dp
      fit_coef(12, 18) = 2.6284577266591210E-04_dp
      fit_coef(0, 19) = 6.5506726065987148E-02_dp
      fit_coef(1, 19) = 8.3686618695792928E-02_dp
      fit_coef(2, 19) = 7.4994836901648418E-02_dp
      fit_coef(3, 19) = 3.8175402517462660E-02_dp
      fit_coef(4, 19) = 7.6429339054575234E-03_dp
      fit_coef(5, 19) = -1.8419060705385749E-03_dp
      fit_coef(6, 19) = 2.7308690208177329E-04_dp
      fit_coef(7, 19) = 1.6627660280614655E-03_dp
      fit_coef(8, 19) = -8.3731649392740858E-04_dp
      fit_coef(9, 19) = -8.4805699699116934E-04_dp
      fit_coef(10, 19) = 1.7342012245697370E-05_dp
      fit_coef(11, 19) = 6.1766339244338580E-04_dp
      fit_coef(12, 19) = -2.3138634064074576E-04_dp
      fit_coef(0, 20) = 1.2258176989295384E-01_dp
      fit_coef(1, 20) = 1.2778305763600906E-01_dp
      fit_coef(2, 20) = 1.0172491446523661E-01_dp
      fit_coef(3, 20) = 4.1491871675887949E-02_dp
      fit_coef(4, 20) = 1.7844232528150912E-03_dp
      fit_coef(5, 20) = -4.3718842072007049E-03_dp
      fit_coef(6, 20) = 1.7506621259053797E-03_dp
      fit_coef(7, 20) = 3.4297516381714774E-03_dp
      fit_coef(8, 20) = -1.8677852767092760E-03_dp
      fit_coef(9, 20) = -2.5672855213340693E-04_dp
      fit_coef(10, 20) = -1.3122155242537307E-04_dp
      fit_coef(11, 20) = 3.0827036993633989E-04_dp
      fit_coef(12, 20) = -7.5397667904590979E-05_dp
      fit_coef(0, 21) = 2.2407341908539469E-01_dp
      fit_coef(1, 21) = 1.8741517537369246E-01_dp
      fit_coef(2, 21) = 1.3316931550316194E-01_dp
      fit_coef(3, 21) = 4.1846304068269342E-02_dp
      fit_coef(4, 21) = -6.3923892927403468E-03_dp
      fit_coef(5, 21) = -6.3290486135616480E-03_dp
      fit_coef(6, 21) = 4.0785090266698392E-03_dp
      fit_coef(7, 21) = 4.4129649982871406E-03_dp
      fit_coef(8, 21) = -3.0701955863932482E-03_dp
      fit_coef(9, 21) = -5.0155854005366068E-04_dp
      fit_coef(10, 21) = 1.2677059967610719E-03_dp
      fit_coef(11, 21) = -6.5728561506278344E-04_dp
      fit_coef(12, 21) = 1.3236733395452350E-04_dp
      fit_coef(0, 22) = 4.0125849549125764E-01_dp
      fit_coef(1, 22) = 2.6234441978160788E-01_dp
      fit_coef(2, 22) = 1.6705808644309805E-01_dp
      fit_coef(3, 22) = 3.7759081921101723E-02_dp
      fit_coef(4, 22) = -1.6444808051945672E-02_dp
      fit_coef(5, 22) = -6.7086735514358875E-03_dp
      fit_coef(6, 22) = 7.4079840573032128E-03_dp
      fit_coef(7, 22) = 5.0153636055701280E-03_dp
      fit_coef(8, 22) = -5.8034207254719751E-03_dp
      fit_coef(9, 22) = 7.1487044000835261E-04_dp
      fit_coef(10, 22) = 1.7935699565438628E-03_dp
      fit_coef(11, 22) = -1.1756448775298205E-03_dp
      fit_coef(12, 22) = 2.3788942753303003E-04_dp
      fit_coef(0, 23) = 7.0683650900241923E-01_dp
      fit_coef(1, 23) = 3.4775269742355053E-01_dp
      fit_coef(2, 23) = 1.9991809868316918E-01_dp
      fit_coef(3, 23) = 2.8998979248009617E-02_dp
      fit_coef(4, 23) = -2.7072247921548855E-02_dp
      fit_coef(5, 23) = -4.9720416004814215E-03_dp
      fit_coef(6, 23) = 1.1386666180451437E-02_dp
      fit_coef(7, 23) = 4.9476484479521700E-03_dp
      fit_coef(8, 23) = -9.6210622467224182E-03_dp
      fit_coef(9, 23) = 3.3069098482828903E-03_dp
      fit_coef(10, 23) = 1.6164601539057236E-03_dp
      fit_coef(11, 23) = -1.5619532900798120E-03_dp
      fit_coef(12, 23) = 3.5694695591620168E-04_dp
      fit_coef(0, 24) = 1.2349305749929025E+00_dp
      fit_coef(1, 24) = 4.3289986648386525E-01_dp
      fit_coef(2, 24) = 2.2845857393019400E-01_dp
      fit_coef(3, 24) = 1.7946385370009432E-02_dp
      fit_coef(4, 24) = -3.6314829777361717E-02_dp
      fit_coef(5, 24) = -1.6120574652244938E-03_dp
      fit_coef(6, 24) = 1.5127251697612880E-02_dp
      fit_coef(7, 24) = 4.2179377246450477E-03_dp
      fit_coef(8, 24) = -1.3727271503534512E-02_dp
      fit_coef(9, 24) = 6.9063396049124318E-03_dp
      fit_coef(10, 24) = 6.9366312151399967E-04_dp
      fit_coef(11, 24) = -1.7353304432909385E-03_dp
      fit_coef(12, 24) = 4.6246138244891298E-04_dp
      fit_coef(0, 25) = 2.1884594395954422E+00_dp
      fit_coef(1, 25) = 5.0512490214011563E-01_dp
      fit_coef(2, 25) = 2.5294331999124109E-01_dp
      fit_coef(3, 25) = 9.9552977533875799E-03_dp
      fit_coef(4, 25) = -4.2741646955111831E-02_dp
      fit_coef(5, 25) = 1.4996753556095149E-03_dp
      fit_coef(6, 25) = 1.7828818970966834E-02_dp
      fit_coef(7, 25) = 3.4400974982679682E-03_dp
      fit_coef(8, 25) = -1.7077573476609768E-02_dp
      fit_coef(9, 25) = 1.0264977475261727E-02_dp
      fit_coef(10, 25) = -4.9655867540557698E-04_dp
      fit_coef(11, 25) = -1.6960953173867819E-03_dp
      fit_coef(12, 25) = 5.2021831684559222E-04_dp
      fit_coef(0, 26) = 4.2896812115374434E+00_dp
      fit_coef(1, 26) = 5.7441806238555526E-01_dp
      fit_coef(2, 26) = 2.8764315098250082E-01_dp
      fit_coef(3, 26) = 1.2205511996262715E-02_dp
      fit_coef(4, 26) = -4.7500863719757744E-02_dp
      fit_coef(5, 26) = 1.9853897004354611E-03_dp
      fit_coef(6, 26) = 1.9866889885557001E-02_dp
      fit_coef(7, 26) = 3.6718675673585712E-03_dp
      fit_coef(8, 26) = -1.9405833778650398E-02_dp
      fit_coef(9, 26) = 1.1968919361045174E-02_dp
      fit_coef(10, 26) = -8.6190027805330149E-04_dp
      fit_coef(11, 26) = -1.8010240017088701E-03_dp
      fit_coef(12, 26) = 5.7196032777695038E-04_dp

   END SUBROUTINE get_fit_coef_k13_mi

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k13_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 3.7370000000000000E+05_dp
      fit_coef(0, 1) = 1.0328373040080946E-05_dp
      fit_coef(1, 1) = 1.0231661765573473E-05_dp
      fit_coef(2, 1) = 1.4274421628573503E-03_dp
      fit_coef(3, 1) = -9.2218261005801595E-03_dp
      fit_coef(4, 1) = 7.9453452208337924E-02_dp
      fit_coef(5, 1) = -3.4395924855303062E-01_dp
      fit_coef(6, 1) = 1.0503011873064374E+00_dp
      fit_coef(7, 1) = -2.2107082564421301E+00_dp
      fit_coef(8, 1) = 3.1904625274566456E+00_dp
      fit_coef(9, 1) = -3.1005928635963018E+00_dp
      fit_coef(10, 1) = 1.9388301331438162E+00_dp
      fit_coef(11, 1) = -7.0437721043920587E-01_dp
      fit_coef(12, 1) = 1.1297211307192120E-01_dp
      fit_coef(0, 2) = 1.1370944761350369E-04_dp
      fit_coef(1, 2) = 6.5461398441333884E-05_dp
      fit_coef(2, 2) = 9.0857762128942112E-03_dp
      fit_coef(3, 2) = -6.1037659291954947E-02_dp
      fit_coef(4, 2) = 5.0701553071696137E-01_dp
      fit_coef(5, 2) = -2.2094585548616172E+00_dp
      fit_coef(6, 2) = 6.7357245459730972E+00_dp
      fit_coef(7, 2) = -1.4152000667009908E+01_dp
      fit_coef(8, 2) = 2.0402127724921442E+01_dp
      fit_coef(9, 2) = -1.9817215032938513E+01_dp
      fit_coef(10, 2) = 1.2389779868005123E+01_dp
      fit_coef(11, 2) = -4.5013495035293962E+00_dp
      fit_coef(12, 2) = 7.2205686510589684E-01_dp
      fit_coef(0, 3) = 6.0201360833921547E-04_dp
      fit_coef(1, 3) = 2.3713142412530957E-04_dp
      fit_coef(2, 3) = 3.2408075864146658E-02_dp
      fit_coef(3, 3) = -2.2633769436834197E-01_dp
      fit_coef(4, 3) = 1.7944183579800783E+00_dp
      fit_coef(5, 3) = -8.0322858690413170E+00_dp
      fit_coef(6, 3) = 2.4758947246186221E+01_dp
      fit_coef(7, 3) = -5.2297882977111676E+01_dp
      fit_coef(8, 3) = 7.5599300021495196E+01_dp
      fit_coef(9, 3) = -7.3529871053182731E+01_dp
      fit_coef(10, 3) = 4.5999098353208382E+01_dp
      fit_coef(11, 3) = -1.6715853539005085E+01_dp
      fit_coef(12, 3) = 2.6814682156594003E+00_dp
      fit_coef(0, 4) = 2.3239931463444295E-03_dp
      fit_coef(1, 4) = 6.7915483564538726E-04_dp
      fit_coef(2, 4) = 9.2065143976354222E-02_dp
      fit_coef(3, 4) = -6.5920735047309242E-01_dp
      fit_coef(4, 4) = 5.0701238208447936E+00_dp
      fit_coef(5, 4) = -2.3078149685475569E+01_dp
      fit_coef(6, 4) = 7.1252230915973797E+01_dp
      fit_coef(7, 4) = -1.5026332268406557E+02_dp
      fit_coef(8, 4) = 2.1682358395745959E+02_dp
      fit_coef(9, 4) = -2.1059821196596170E+02_dp
      fit_coef(10, 4) = 1.3162443597110655E+02_dp
      fit_coef(11, 4) = -4.7803836464227409E+01_dp
      fit_coef(12, 4) = 7.6658373170675951E+00_dp
      fit_coef(0, 5) = 7.4432405183934578E-03_dp
      fit_coef(1, 5) = 1.6828152409194407E-03_dp
      fit_coef(2, 5) = 2.2645498811467052E-01_dp
      fit_coef(3, 5) = -1.6470414947742027E+00_dp
      fit_coef(4, 5) = 1.2406213814972343E+01_dp
      fit_coef(5, 5) = -5.7052515747077990E+01_dp
      fit_coef(6, 5) = 1.7648148382559137E+02_dp
      fit_coef(7, 5) = -3.7241542082575245E+02_dp
      fit_coef(8, 5) = 5.3766501171349421E+02_dp
      fit_coef(9, 5) = -5.2248797832997366E+02_dp
      fit_coef(10, 5) = 3.2669873907570917E+02_dp
      fit_coef(11, 5) = -1.1869478309408113E+02_dp
      fit_coef(12, 5) = 1.9039603955629481E+01_dp
      fit_coef(0, 6) = 2.1001669881294191E-02_dp
      fit_coef(1, 6) = 3.7493611984017344E-03_dp
      fit_coef(2, 6) = 5.0174562015489310E-01_dp
      fit_coef(3, 6) = -3.6903350557230272E+00_dp
      fit_coef(4, 6) = 2.7380167524745204E+01_dp
      fit_coef(5, 6) = -1.2680883651700998E+02_dp
      fit_coef(6, 6) = 3.9315723208637161E+02_dp
      fit_coef(7, 6) = -8.3061013874245850E+02_dp
      fit_coef(8, 6) = 1.2000077154718190E+03_dp
      fit_coef(9, 6) = -1.1666285380811528E+03_dp
      fit_coef(10, 6) = 7.2965528810276930E+02_dp
      fit_coef(11, 6) = -2.6514064826477511E+02_dp
      fit_coef(12, 6) = 4.2535848707897181E+01_dp
      fit_coef(0, 7) = 5.3988578814712126E-02_dp
      fit_coef(1, 7) = 7.6787040162035062E-03_dp
      fit_coef(2, 7) = 1.0229613253807799E+00_dp
      fit_coef(3, 7) = -7.5874025191500545E+00_dp
      fit_coef(4, 7) = 5.5644821860203081E+01_dp
      fit_coef(5, 7) = -2.5903592714657515E+02_dp
      fit_coef(6, 7) = 8.0491520561775883E+02_dp
      fit_coef(7, 7) = -1.7025740847831944E+03_dp
      fit_coef(8, 7) = 2.4614597158125571E+03_dp
      fit_coef(9, 7) = -2.3939670515607563E+03_dp
      fit_coef(10, 7) = 1.4976634167179416E+03_dp
      fit_coef(11, 7) = -5.4431186678214578E+02_dp
      fit_coef(12, 7) = 8.7333538775326986E+01_dp
      fit_coef(0, 8) = 1.2913416270542125E-01_dp
      fit_coef(1, 8) = 1.4637308467888006E-02_dp
      fit_coef(2, 8) = 1.9427142860381650E+00_dp
      fit_coef(3, 8) = -1.4504186559531496E+01_dp
      fit_coef(4, 8) = 1.0539283356781200E+02_dp
      fit_coef(5, 8) = -4.9250973009160060E+02_dp
      fit_coef(6, 8) = 1.5335961739523964E+03_dp
      fit_coef(7, 8) = -3.2477036056116422E+03_dp
      fit_coef(8, 8) = 4.6984634605620395E+03_dp
      fit_coef(9, 8) = -4.5714722547915217E+03_dp
      fit_coef(10, 8) = 2.8606360597296798E+03_dp
      fit_coef(11, 8) = -1.0398487637572459E+03_dp
      fit_coef(12, 8) = 1.6686131315440755E+02_dp
      fit_coef(0, 9) = 2.9151438803227880E-01_dp
      fit_coef(1, 9) = 2.6141059594544370E-02_dp
      fit_coef(2, 9) = 3.4585779341218914E+00_dp
      fit_coef(3, 9) = -2.5957371800387559E+01_dp
      fit_coef(4, 9) = 1.8719902735916295E+02_dp
      fit_coef(5, 9) = -8.7738902954943887E+02_dp
      fit_coef(6, 9) = 2.7372194924276941E+03_dp
      fit_coef(7, 9) = -5.8030086010828445E+03_dp
      fit_coef(8, 9) = 8.4006232674596267E+03_dp
      fit_coef(9, 9) = -8.1767388699954290E+03_dp
      fit_coef(10, 9) = 5.1179187652543951E+03_dp
      fit_coef(11, 9) = -1.8606812820509087E+03_dp
      fit_coef(12, 9) = 2.9861178044260612E+02_dp
      fit_coef(0, 10) = 6.2761071501945276E-01_dp
      fit_coef(1, 10) = 4.3829697389904873E-02_dp
      fit_coef(2, 10) = 5.7833019860251502E+00_dp
      fit_coef(3, 10) = -4.3589104975069581E+01_dp
      fit_coef(4, 10) = 3.1240972247660102E+02_dp
      fit_coef(5, 10) = -1.4676188400419255E+03_dp
      fit_coef(6, 10) = 4.5862715204341948E+03_dp
      fit_coef(7, 10) = -9.7328856974721111E+03_dp
      fit_coef(8, 10) = 1.4098068171267152E+04_dp
      fit_coef(9, 10) = -1.3727352353131695E+04_dp
      fit_coef(10, 10) = 8.5941131147490141E+03_dp
      fit_coef(11, 10) = -3.1249775864789372E+03_dp
      fit_coef(12, 10) = 5.0156651089860543E+02_dp
      fit_coef(0, 11) = 1.3001830139334465E+00_dp
      fit_coef(1, 11) = 6.8889203557683909E-02_dp
      fit_coef(2, 11) = 9.0695201480430416E+00_dp
      fit_coef(3, 11) = -6.8588523830477001E+01_dp
      fit_coef(4, 11) = 4.8911175818178219E+02_dp
      fit_coef(5, 11) = -2.3017939006101778E+03_dp
      fit_coef(6, 11) = 7.2034170829290761E+03_dp
      fit_coef(7, 11) = -1.5300571255050740E+04_dp
      fit_coef(8, 11) = 2.2174747250984612E+04_dp
      fit_coef(9, 11) = -2.1598799766875782E+04_dp
      fit_coef(10, 11) = 1.3524962200868669E+04_dp
      fit_coef(11, 11) = -4.9186216136818812E+03_dp
      fit_coef(12, 11) = 7.8952838092425554E+02_dp
      fit_coef(0, 12) = 2.6215772490899263E+00_dp
      fit_coef(1, 12) = 1.0119187877095515E-01_dp
      fit_coef(2, 12) = 1.3298790450785793E+01_dp
      fit_coef(3, 12) = -1.0082948191003250E+02_dp
      fit_coef(4, 12) = 7.1624051882694800E+02_dp
      fit_coef(5, 12) = -3.3750615698310812E+03_dp
      fit_coef(6, 12) = 1.0574350973373779E+04_dp
      fit_coef(7, 12) = -2.2477084129138388E+04_dp
      fit_coef(8, 12) = 3.2590012647996798E+04_dp
      fit_coef(9, 12) = -3.1752355473368389E+04_dp
      fit_coef(10, 12) = 1.9886579242865319E+04_dp
      fit_coef(11, 12) = -7.2330170769743845E+03_dp
      fit_coef(12, 12) = 1.1611278007055782E+03_dp
      fit_coef(0, 13) = 5.2970688415539806E+00_dp
      fit_coef(1, 13) = 1.3913815821587258E-01_dp
      fit_coef(2, 13) = 1.8263424659829006E+01_dp
      fit_coef(3, 13) = -1.3870952790580026E+02_dp
      fit_coef(4, 13) = 9.8271245072947818E+02_dp
      fit_coef(5, 13) = -4.6347205083298186E+03_dp
      fit_coef(6, 13) = 1.4532606648412351E+04_dp
      fit_coef(7, 13) = -3.0906783800217447E+04_dp
      fit_coef(8, 13) = 4.4826644713674119E+04_dp
      fit_coef(9, 13) = -4.3683152384513414E+04_dp
      fit_coef(10, 13) = 2.7362388150846749E+04_dp
      fit_coef(11, 13) = -9.9529300467619232E+03_dp
      fit_coef(12, 13) = 1.5978560147630030E+03_dp
      fit_coef(0, 14) = 3.3029079169995746E-05_dp
      fit_coef(1, 14) = 2.6643361318081685E-05_dp
      fit_coef(2, 14) = 3.7453854295855472E-03_dp
      fit_coef(3, 14) = -2.4461573064643052E-02_dp
      fit_coef(4, 14) = 2.1192677272898949E-01_dp
      fit_coef(5, 14) = -9.2298813642259792E-01_dp
      fit_coef(6, 14) = 2.8202954491441887E+00_dp
      fit_coef(7, 14) = -5.9307601899579083E+00_dp
      fit_coef(8, 14) = 8.5478350235106983E+00_dp
      fit_coef(9, 14) = -8.2958924766363573E+00_dp
      fit_coef(10, 14) = 5.1810926707806537E+00_dp
      fit_coef(11, 14) = -1.8802314403008740E+00_dp
      fit_coef(12, 14) = 3.0127397290756769E-01_dp
      fit_coef(0, 15) = 2.1715120936896851E-04_dp
      fit_coef(1, 15) = 9.6158838084655870E-05_dp
      fit_coef(2, 15) = 1.3102183838061880E-02_dp
      fit_coef(3, 15) = -9.0442571305710165E-02_dp
      fit_coef(4, 15) = 7.2073640737136146E-01_dp
      fit_coef(5, 15) = -3.1779329684848223E+00_dp
      fit_coef(6, 15) = 9.7478690828954182E+00_dp
      fit_coef(7, 15) = -2.0582241665854877E+01_dp
      fit_coef(8, 15) = 2.9794991676852089E+01_dp
      fit_coef(9, 15) = -2.9037927118237850E+01_dp
      fit_coef(10, 15) = 1.8203528955433185E+01_dp
      fit_coef(11, 15) = -6.6278898854648514E+00_dp
      fit_coef(12, 15) = 1.0650411372773607E+00_dp
      fit_coef(0, 16) = 8.8916557085975359E-04_dp
      fit_coef(1, 16) = 2.7147003112010167E-04_dp
      fit_coef(2, 16) = 3.6886711972600503E-02_dp
      fit_coef(3, 16) = -2.6522390438189170E-01_dp
      fit_coef(4, 16) = 2.0383361789339589E+00_dp
      fit_coef(5, 16) = -9.3159312869424529E+00_dp
      fit_coef(6, 16) = 2.8836230573430583E+01_dp
      fit_coef(7, 16) = -6.0862901561188608E+01_dp
      fit_coef(8, 16) = 8.7796437530891168E+01_dp
      fit_coef(9, 16) = -8.5203527246658652E+01_dp
      fit_coef(10, 16) = 5.3197256397040526E+01_dp
      fit_coef(11, 16) = -1.9300241563707537E+01_dp
      fit_coef(12, 16) = 3.0920034178994409E+00_dp
      fit_coef(0, 17) = 2.8909222386173440E-03_dp
      fit_coef(1, 17) = 6.5986499705299726E-04_dp
      fit_coef(2, 17) = 8.8661109522972403E-02_dp
      fit_coef(3, 17) = -6.4720090249728535E-01_dp
      fit_coef(4, 17) = 4.8509444516645317E+00_dp
      fit_coef(5, 17) = -2.2365482960360204E+01_dp
      fit_coef(6, 17) = 6.9142722370552491E+01_dp
      fit_coef(7, 17) = -1.4576807130419496E+02_dp
      fit_coef(8, 17) = 2.1031616948122442E+02_dp
      fit_coef(9, 17) = -2.0432151206915759E+02_dp
      fit_coef(10, 17) = 1.2775135126432421E+02_dp
      fit_coef(11, 17) = -4.6418220663487915E+01_dp
      fit_coef(12, 17) = 7.4470095221066224E+00_dp
      fit_coef(0, 18) = 8.1395124276193651E-03_dp
      fit_coef(1, 18) = 1.4295080919928922E-03_dp
      fit_coef(2, 18) = 1.9095038913922552E-01_dp
      fit_coef(3, 18) = -1.4103276835369478E+00_dp
      fit_coef(4, 18) = 1.0406506856630042E+01_dp
      fit_coef(5, 18) = -4.8337660891823411E+01_dp
      fit_coef(6, 18) = 1.4991827465085015E+02_dp
      fit_coef(7, 18) = -3.1676057422657112E+02_dp
      fit_coef(8, 18) = 4.5769739592136420E+02_dp
      fit_coef(9, 18) = -4.4502674091606121E+02_dp
      fit_coef(10, 18) = 2.7836644169317543E+02_dp
      fit_coef(11, 18) = -1.0115946637317019E+02_dp
      fit_coef(12, 18) = 1.6229500712924121E+01_dp
      fit_coef(0, 19) = 2.0717076029345759E-02_dp
      fit_coef(1, 19) = 2.8356072593395381E-03_dp
      fit_coef(2, 19) = 3.7697026585720034E-01_dp
      fit_coef(3, 19) = -2.8080135576835485E+00_dp
      fit_coef(4, 19) = 2.0474949195105673E+01_dp
      fit_coef(5, 19) = -9.5582095227962725E+01_dp
      fit_coef(6, 19) = 2.9724241292709326E+02_dp
      fit_coef(7, 19) = -6.2897377141618983E+02_dp
      fit_coef(8, 19) = 9.0952762860883638E+02_dp
      fit_coef(9, 19) = -8.8470183512709411E+02_dp
      fit_coef(10, 19) = 5.5351047071161520E+02_dp
      fit_coef(11, 19) = -2.0117820626571816E+02_dp
      fit_coef(12, 19) = 3.2279767274864902E+01_dp
      fit_coef(0, 20) = 4.8876460483642845E-02_dp
      fit_coef(1, 20) = 5.2174808763164821E-03_dp
      fit_coef(2, 20) = 6.9090200030553239E-01_dp
      fit_coef(3, 20) = -5.1804295291616596E+00_dp
      fit_coef(4, 20) = 3.7420371100773927E+01_dp
      fit_coef(5, 20) = -1.7533661861799825E+02_dp
      fit_coef(6, 20) = 5.4658265990827033E+02_dp
      fit_coef(7, 20) = -1.1581864472600616E+03_dp
      fit_coef(8, 20) = 1.6761108303901358E+03_dp
      fit_coef(9, 20) = -1.6311266679849095E+03_dp
      fit_coef(10, 20) = 1.0208160076430613E+03_dp
      fit_coef(11, 20) = -3.7110037560401321E+02_dp
      fit_coef(12, 20) = 5.9552981560922035E+01_dp
      fit_coef(0, 21) = 1.0862470653291458E-01_dp
      fit_coef(1, 21) = 8.9557210597009419E-03_dp
      fit_coef(2, 21) = 1.1819761188507145E+00_dp
      fit_coef(3, 21) = -8.9092575248149508E+00_dp
      fit_coef(4, 21) = 6.3862158642253107E+01_dp
      fit_coef(5, 21) = -3.0008394292824050E+02_dp
      fit_coef(6, 21) = 9.3746815308631687E+02_dp
      fit_coef(7, 21) = -1.9889858954576416E+03_dp
      fit_coef(8, 21) = 2.8805735949038526E+03_dp
      fit_coef(9, 21) = -2.8045287776281298E+03_dp
      fit_coef(10, 21) = 1.7556741628884376E+03_dp
      fit_coef(11, 21) = -6.3836601287630617E+02_dp
      fit_coef(12, 21) = 1.0245612268571801E+02_dp
      fit_coef(0, 22) = 2.3000575937054105E-01_dp
      fit_coef(1, 22) = 1.4334607293667794E-02_dp
      fit_coef(2, 22) = 1.8864545891903439E+00_dp
      fit_coef(3, 22) = -1.4280352788842627E+01_dp
      fit_coef(4, 22) = 1.0170803299679834E+02_dp
      fit_coef(5, 22) = -4.7898216088621695E+02_dp
      fit_coef(6, 22) = 1.4991893689445949E+03_dp
      fit_coef(7, 22) = -3.1844797903553772E+03_dp
      fit_coef(8, 22) = 4.6151825676230128E+03_dp
      fit_coef(9, 22) = -4.4952731839222251E+03_dp
      fit_coef(10, 22) = 2.8148739884496081E+03_dp
      fit_coef(11, 22) = -1.0236785820690926E+03_dp
      fit_coef(12, 22) = 1.6431836962570071E+02_dp
      fit_coef(0, 23) = 4.6825429870000890E-01_dp
      fit_coef(1, 23) = 2.1256443046165384E-02_dp
      fit_coef(2, 23) = 2.7905151165222755E+00_dp
      fit_coef(3, 23) = -2.1197457436160700E+01_dp
      fit_coef(4, 23) = 1.5017192005334016E+02_dp
      fit_coef(5, 23) = -7.0843773023664141E+02_dp
      fit_coef(6, 23) = 2.2210314991220407E+03_dp
      fit_coef(7, 23) = -4.7227286038286647E+03_dp
      fit_coef(8, 23) = 6.8489146048021994E+03_dp
      fit_coef(9, 23) = -6.6736306428096368E+03_dp
      fit_coef(10, 23) = 4.1800022473986091E+03_dp
      fit_coef(11, 23) = -1.5203921664836910E+03_dp
      fit_coef(12, 23) = 2.4407860279057451E+02_dp
      fit_coef(0, 24) = 9.2615015167881409E-01_dp
      fit_coef(1, 24) = 2.8851029357722529E-02_dp
      fit_coef(2, 24) = 3.7800720586882299E+00_dp
      fit_coef(3, 24) = -2.8790145745264784E+01_dp
      fit_coef(4, 24) = 2.0311830055185206E+02_dp
      fit_coef(5, 24) = -9.5941022550967875E+02_dp
      fit_coef(6, 24) = 3.0118661422758710E+03_dp
      fit_coef(7, 24) = -6.4099639194822603E+03_dp
      fit_coef(8, 24) = 9.3008315617510980E+03_dp
      fit_coef(9, 24) = -9.0659183663546519E+03_dp
      fit_coef(10, 24) = 5.6796728484798878E+03_dp
      fit_coef(11, 24) = -2.0661781228629306E+03_dp
      fit_coef(12, 24) = 3.3173195728147749E+02_dp
      fit_coef(0, 25) = 1.8197406882188483E+00_dp
      fit_coef(1, 25) = 3.5439117125219140E-02_dp
      fit_coef(2, 25) = 4.6374741775348376E+00_dp
      fit_coef(3, 25) = -3.5376154233726169E+01_dp
      fit_coef(4, 25) = 2.4894862911423743E+02_dp
      fit_coef(5, 25) = -1.1767196794716788E+03_dp
      fit_coef(6, 25) = 3.6971703023290502E+03_dp
      fit_coef(7, 25) = -7.8729723077281906E+03_dp
      fit_coef(8, 25) = 1.1427813147275856E+04_dp
      fit_coef(9, 25) = -1.1141769814002466E+04_dp
      fit_coef(10, 25) = 6.9812302395124789E+03_dp
      fit_coef(11, 25) = -2.5399272897012484E+03_dp
      fit_coef(12, 25) = 4.0782349862907824E+02_dp
      fit_coef(0, 26) = 3.8699714083339569E+00_dp
      fit_coef(1, 26) = 4.0430697875251408E-02_dp
      fit_coef(2, 26) = 5.2898404083820747E+00_dp
      fit_coef(3, 26) = -4.0359075619765512E+01_dp
      fit_coef(4, 26) = 2.8393314652682147E+02_dp
      fit_coef(5, 26) = -1.3421543359055597E+03_dp
      fit_coef(6, 26) = 4.2173994053467277E+03_dp
      fit_coef(7, 26) = -8.9814986542291372E+03_dp
      fit_coef(8, 26) = 1.3037566274562520E+04_dp
      fit_coef(9, 26) = -1.2711681344703467E+04_dp
      fit_coef(10, 26) = 7.9650982603751236E+03_dp
      fit_coef(11, 26) = -2.8979280791052265E+03_dp
      fit_coef(12, 26) = 4.6531132823912742E+02_dp

   END SUBROUTINE get_fit_coef_k13_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k14_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 4.0562245241676015E-03_dp
      fit_coef(1, 1) = 1.0969040206552460E-02_dp
      fit_coef(2, 1) = 1.4405191275649688E-02_dp
      fit_coef(3, 1) = 1.0649635686233358E-02_dp
      fit_coef(4, 1) = 3.2778746107621219E-03_dp
      fit_coef(5, 1) = -1.5147809086133006E-03_dp
      fit_coef(6, 1) = -1.7768378644379805E-03_dp
      fit_coef(7, 1) = -1.7040745050157837E-04_dp
      fit_coef(8, 1) = 4.3464032657914310E-04_dp
      fit_coef(9, 1) = 2.9772666014833774E-04_dp
      fit_coef(10, 1) = -3.9919353211016935E-04_dp
      fit_coef(11, 1) = 1.9832458267342404E-04_dp
      fit_coef(12, 1) = -4.3867764193471502E-05_dp
      fit_coef(0, 2) = 2.1853974229517521E-02_dp
      fit_coef(1, 2) = 5.8406086298247388E-02_dp
      fit_coef(2, 2) = 7.6152918249589960E-02_dp
      fit_coef(3, 2) = 5.5933948505163415E-02_dp
      fit_coef(4, 2) = 1.7021801576352837E-02_dp
      fit_coef(5, 2) = -8.0247279189159457E-03_dp
      fit_coef(6, 2) = -9.2606257683723776E-03_dp
      fit_coef(7, 2) = -8.4039046464910933E-04_dp
      fit_coef(8, 2) = 2.2920465602860878E-03_dp
      fit_coef(9, 2) = 1.5096114542523550E-03_dp
      fit_coef(10, 2) = -2.0536902896596300E-03_dp
      fit_coef(11, 2) = 1.0198430679936156E-03_dp
      fit_coef(12, 2) = -2.2535568541917208E-04_dp
      fit_coef(0, 3) = 5.6010245407281063E-02_dp
      fit_coef(1, 3) = 1.4622279778697206E-01_dp
      fit_coef(2, 3) = 1.8818537023914636E-01_dp
      fit_coef(3, 3) = 1.3664268008941158E-01_dp
      fit_coef(4, 3) = 4.0762241157569830E-02_dp
      fit_coef(5, 3) = -1.9875823087429162E-02_dp
      fit_coef(6, 3) = -2.2308862126762034E-02_dp
      fit_coef(7, 3) = -1.8247566176568901E-03_dp
      fit_coef(8, 3) = 5.6284318752787164E-03_dp
      fit_coef(9, 3) = 3.4697564866730192E-03_dp
      fit_coef(10, 3) = -4.8438240685035059E-03_dp
      fit_coef(11, 3) = 2.4030731708201193E-03_dp
      fit_coef(12, 3) = -5.2978518261630908E-04_dp
      fit_coef(0, 4) = 1.1092328992892893E-01_dp
      fit_coef(1, 4) = 2.7859680338891873E-01_dp
      fit_coef(2, 4) = 3.5169352998000031E-01_dp
      fit_coef(3, 4) = 2.5142478390455381E-01_dp
      fit_coef(4, 4) = 7.2990020531424102E-02_dp
      fit_coef(5, 4) = -3.7221206950039602E-02_dp
      fit_coef(6, 4) = -4.0210444834685513E-02_dp
      fit_coef(7, 4) = -2.7577403917926737E-03_dp
      fit_coef(8, 4) = 1.0362436751946298E-02_dp
      fit_coef(9, 4) = 5.8637315949354504E-03_dp
      fit_coef(10, 4) = -8.4785224237194489E-03_dp
      fit_coef(11, 4) = 4.1965838668479400E-03_dp
      fit_coef(12, 4) = -9.2121396633372532E-04_dp
      fit_coef(0, 5) = 1.9465685621238632E-01_dp
      fit_coef(1, 5) = 4.6152338108270852E-01_dp
      fit_coef(2, 5) = 5.6693470576329441E-01_dp
      fit_coef(3, 5) = 3.9807380880317028E-01_dp
      fit_coef(4, 5) = 1.1253953929433713E-01_dp
      fit_coef(5, 5) = -6.0165003179139975E-02_dp
      fit_coef(6, 5) = -6.2492236340615785E-02_dp
      fit_coef(7, 5) = -3.1516877242773393E-03_dp
      fit_coef(8, 5) = 1.7371050989592705E-02_dp
      fit_coef(9, 5) = 6.8885779131374857E-03_dp
      fit_coef(10, 5) = -1.1712499657108166E-02_dp
      fit_coef(11, 5) = 5.9558759087716033E-03_dp
      fit_coef(12, 5) = -1.3302080306339819E-03_dp
      fit_coef(0, 6) = 3.2087033276941335E-01_dp
      fit_coef(1, 6) = 7.0343427128207259E-01_dp
      fit_coef(2, 6) = 8.3155424687895485E-01_dp
      fit_coef(3, 6) = 5.7158103074002242E-01_dp
      fit_coef(4, 6) = 1.6033482878158128E-01_dp
      fit_coef(5, 6) = -8.5970624138830110E-02_dp
      fit_coef(6, 6) = -9.0709492543919121E-02_dp
      fit_coef(7, 6) = -4.6906764752169330E-03_dp
      fit_coef(8, 6) = 2.7054261624413843E-02_dp
      fit_coef(9, 6) = 1.4112534091895116E-02_dp
      fit_coef(10, 6) = -2.5828598206741545E-02_dp
      fit_coef(11, 6) = 1.4364571586594627E-02_dp
      fit_coef(12, 6) = -3.2730781292866435E-03_dp
      fit_coef(0, 7) = 5.1123719677822987E-01_dp
      fit_coef(1, 7) = 1.0160745695891158E+00_dp
      fit_coef(2, 7) = 1.1406376033091816E+00_dp
      fit_coef(3, 7) = 7.6072431018364839E-01_dp
      fit_coef(4, 7) = 2.1623548132157380E-01_dp
      fit_coef(5, 7) = -1.0385662211042419E-01_dp
      fit_coef(6, 7) = -1.2398076049296762E-01_dp
      fit_coef(7, 7) = -5.5258832288643421E-03_dp
      fit_coef(8, 7) = 5.4749667731452062E-03_dp
      fit_coef(9, 7) = 7.8756125030946533E-02_dp
      fit_coef(10, 7) = -8.1450501282583770E-02_dp
      fit_coef(11, 7) = 3.5582766973436604E-02_dp
      fit_coef(12, 7) = -6.2851848529127960E-03_dp
      fit_coef(0, 8) = 7.9862454256130755E-01_dp
      fit_coef(1, 8) = 1.4140917795956391E+00_dp
      fit_coef(2, 8) = 1.4887439734322301E+00_dp
      fit_coef(3, 8) = 9.4959856751799310E-01_dp
      fit_coef(4, 8) = 2.7310813416061092E-01_dp
      fit_coef(5, 8) = -1.0471167639057280E-01_dp
      fit_coef(6, 8) = -1.4117735675446755E-01_dp
      fit_coef(7, 8) = -2.8720200153060829E-02_dp
      fit_coef(8, 8) = 9.9988658384949844E-03_dp
      fit_coef(9, 8) = 4.4994288100147821E-02_dp
      fit_coef(10, 8) = 9.8296298107774047E-03_dp
      fit_coef(11, 8) = -3.6510317397394888E-02_dp
      fit_coef(12, 8) = 1.3048706811534378E-02_dp
      fit_coef(0, 9) = 1.2318540903074773E+00_dp
      fit_coef(1, 9) = 1.9126535284122799E+00_dp
      fit_coef(2, 9) = 1.8706375498146159E+00_dp
      fit_coef(3, 9) = 1.1237046593985318E+00_dp
      fit_coef(4, 9) = 3.2033236887832522E-01_dp
      fit_coef(5, 9) = -9.0264012269780411E-02_dp
      fit_coef(6, 9) = -1.2294629287092433E-01_dp
      fit_coef(7, 9) = -8.7215860683106217E-02_dp
      fit_coef(8, 9) = 1.0323515038061323E-01_dp
      fit_coef(9, 9) = -1.6774909880232289E-01_dp
      fit_coef(10, 9) = 2.4446069536217355E-01_dp
      fit_coef(11, 9) = -1.4978114145464044E-01_dp
      fit_coef(12, 9) = 3.2845034249025538E-02_dp
      fit_coef(0, 10) = 1.8833267667765345E+00_dp
      fit_coef(1, 10) = 2.5238922534191865E+00_dp
      fit_coef(2, 10) = 2.2803847971690017E+00_dp
      fit_coef(3, 10) = 1.2737526428801398E+00_dp
      fit_coef(4, 10) = 3.5105123224520401E-01_dp
      fit_coef(5, 10) = -6.2857188281637041E-02_dp
      fit_coef(6, 10) = -9.2210310288645597E-02_dp
      fit_coef(7, 10) = -9.0722133116613465E-02_dp
      fit_coef(8, 10) = 7.7837218665255559E-02_dp
      fit_coef(9, 10) = -1.6015129533752295E-01_dp
      fit_coef(10, 10) = 1.8283995908504222E-01_dp
      fit_coef(11, 10) = -6.6325970667744996E-02_dp
      fit_coef(12, 10) = 3.0820539987197176E-03_dp
      fit_coef(0, 11) = 2.8620729078671019E+00_dp
      fit_coef(1, 11) = 3.2531972126791606E+00_dp
      fit_coef(2, 11) = 2.7114279585314924E+00_dp
      fit_coef(3, 11) = 1.3964902193137803E+00_dp
      fit_coef(4, 11) = 3.6517639119142786E-01_dp
      fit_coef(5, 11) = -2.6597116040773625E-02_dp
      fit_coef(6, 11) = -5.9694762140569908E-02_dp
      fit_coef(7, 11) = -3.8758341163376515E-02_dp
      fit_coef(8, 11) = -5.1332218924274670E-02_dp
      fit_coef(9, 11) = 2.5767378796536407E-02_dp
      fit_coef(10, 11) = -4.9870158292428322E-02_dp
      fit_coef(11, 11) = 8.2792987390825273E-02_dp
      fit_coef(12, 11) = -3.1947335725901589E-02_dp
      fit_coef(0, 12) = 4.3396108413641770E+00_dp
      fit_coef(1, 12) = 4.0968755914648138E+00_dp
      fit_coef(2, 12) = 3.1592385203472810E+00_dp
      fit_coef(3, 12) = 1.4949577953660931E+00_dp
      fit_coef(4, 12) = 3.6703288840113468E-01_dp
      fit_coef(5, 12) = 1.2827934117818336E-02_dp
      fit_coef(6, 12) = -1.5966885082492444E-02_dp
      fit_coef(7, 12) = -4.1769471548348015E-03_dp
      fit_coef(8, 12) = -1.0829887623228632E-01_dp
      fit_coef(9, 12) = 9.0063525913958845E-02_dp
      fit_coef(10, 12) = -1.1718852204865197E-01_dp
      fit_coef(11, 12) = 1.0379976023591850E-01_dp
      fit_coef(12, 12) = -2.9513628601409953E-02_dp
      fit_coef(0, 13) = 6.6160518775236428E+00_dp
      fit_coef(1, 13) = 5.0450926097390525E+00_dp
      fit_coef(2, 13) = 3.6269712810913406E+00_dp
      fit_coef(3, 13) = 1.5787210925267479E+00_dp
      fit_coef(4, 13) = 3.6140765183946116E-01_dp
      fit_coef(5, 13) = 5.2660795733846105E-02_dp
      fit_coef(6, 13) = 3.6679818035805827E-02_dp
      fit_coef(7, 13) = 3.0817238462290870E-03_dp
      fit_coef(8, 13) = -8.5513616434734735E-02_dp
      fit_coef(9, 13) = 3.2480437301242795E-02_dp
      fit_coef(10, 13) = -4.3097395900725401E-02_dp
      fit_coef(11, 13) = 4.0285499498477921E-02_dp
      fit_coef(12, 13) = -9.0795169799425725E-03_dp
      fit_coef(0, 14) = 1.0378663082522372E+01_dp
      fit_coef(1, 14) = 6.1051748119732530E+00_dp
      fit_coef(2, 14) = 4.1393929927065685E+00_dp
      fit_coef(3, 14) = 1.6656362128427478E+00_dp
      fit_coef(4, 14) = 3.5175946384380213E-01_dp
      fit_coef(5, 14) = 9.2053766494033781E-02_dp
      fit_coef(6, 14) = 9.0977018487087655E-02_dp
      fit_coef(7, 14) = 9.2641513651703887E-03_dp
      fit_coef(8, 14) = -5.5705669679808022E-02_dp
      fit_coef(9, 14) = -2.2165364299652109E-02_dp
      fit_coef(10, 14) = 2.7050501330544607E-02_dp
      fit_coef(11, 14) = -9.8544712706888447E-03_dp
      fit_coef(12, 14) = 3.3393463606462780E-03_dp
      fit_coef(0, 15) = 1.0463061146787979E-02_dp
      fit_coef(1, 15) = 2.8219919737545576E-02_dp
      fit_coef(2, 15) = 3.6997970832918815E-02_dp
      fit_coef(3, 15) = 2.7310442581655728E-02_dp
      fit_coef(4, 15) = 8.3833807447771629E-03_dp
      fit_coef(5, 15) = -3.8927429936348136E-03_dp
      fit_coef(6, 15) = -4.5483815262270738E-03_dp
      fit_coef(7, 15) = -4.3064267744574074E-04_dp
      fit_coef(8, 15) = 1.1157764240345824E-03_dp
      fit_coef(9, 15) = 7.5712564333480544E-04_dp
      fit_coef(10, 15) = -1.0186473443121876E-03_dp
      fit_coef(11, 15) = 5.0602854051267157E-04_dp
      fit_coef(12, 15) = -1.1190592947238425E-04_dp
      fit_coef(0, 16) = 2.5471251770259992E-02_dp
      fit_coef(1, 16) = 6.7073734523031128E-02_dp
      fit_coef(2, 16) = 8.6686390148806164E-02_dp
      fit_coef(3, 16) = 6.3163726138437945E-02_dp
      fit_coef(4, 16) = 1.8954077181435283E-02_dp
      fit_coef(5, 16) = -9.1546883000010616E-03_dp
      fit_coef(6, 16) = -1.0357597642743745E-02_dp
      fit_coef(7, 16) = -8.7389112118697681E-04_dp
      fit_coef(8, 16) = 2.5999493392185090E-03_dp
      fit_coef(9, 16) = 1.6312506180299385E-03_dp
      fit_coef(10, 16) = -2.2608110574895506E-03_dp
      fit_coef(11, 16) = 1.1220090537728696E-03_dp
      fit_coef(12, 16) = -2.4758294844337260E-04_dp
      fit_coef(0, 17) = 4.3551694119749178E-02_dp
      fit_coef(1, 17) = 1.0925569190527253E-01_dp
      fit_coef(2, 17) = 1.3759390563727888E-01_dp
      fit_coef(3, 17) = 9.8026280776137426E-02_dp
      fit_coef(4, 17) = 2.8264281628926104E-02_dp
      fit_coef(5, 17) = -1.4570182357127179E-02_dp
      fit_coef(6, 17) = -1.5618085831617121E-02_dp
      fit_coef(7, 17) = -1.0386286860505113E-03_dp
      fit_coef(8, 17) = 4.0623648670812191E-03_dp
      fit_coef(9, 17) = 2.2393521384181845E-03_dp
      fit_coef(10, 17) = -3.2758370140128020E-03_dp
      fit_coef(11, 17) = 1.6218936224125669E-03_dp
      fit_coef(12, 17) = -3.5585220637753326E-04_dp
      fit_coef(0, 18) = 6.7585710312796060E-02_dp
      fit_coef(1, 18) = 1.5648129031717098E-01_dp
      fit_coef(2, 18) = 1.8950121838341139E-01_dp
      fit_coef(3, 18) = 1.3119730563134613E-01_dp
      fit_coef(4, 18) = 3.5948846183710491E-02_dp
      fit_coef(5, 18) = -2.0164654606078806E-02_dp
      fit_coef(6, 18) = -2.0055622446027831E-02_dp
      fit_coef(7, 18) = -6.9996789003826023E-04_dp
      fit_coef(8, 18) = 5.4059447836637305E-03_dp
      fit_coef(9, 18) = 2.3753238258316855E-03_dp
      fit_coef(10, 18) = -3.8252579297800944E-03_dp
      fit_coef(11, 18) = 1.8835729864486405E-03_dp
      fit_coef(12, 18) = -4.0977704401502611E-04_dp
      fit_coef(0, 19) = 1.0211824263133877E-01_dp
      fit_coef(1, 19) = 2.1075731039578760E-01_dp
      fit_coef(2, 19) = 2.4059034712033525E-01_dp
      fit_coef(3, 19) = 1.6134884680771011E-01_dp
      fit_coef(4, 19) = 4.3350649255430605E-02_dp
      fit_coef(5, 19) = -2.5429945851545656E-02_dp
      fit_coef(6, 19) = -2.4808194209897237E-02_dp
      fit_coef(7, 19) = -4.5717313943595194E-04_dp
      fit_coef(8, 19) = 9.7364264196386652E-03_dp
      fit_coef(9, 19) = -9.7009624494576196E-04_dp
      fit_coef(10, 19) = -2.9110931341643355E-03_dp
      fit_coef(11, 19) = 2.0554438206790630E-03_dp
      fit_coef(12, 19) = -5.3799541981056737E-04_dp
      fit_coef(0, 20) = 1.5387007707293665E-01_dp
      fit_coef(1, 20) = 2.7502702312847982E-01_dp
      fit_coef(2, 20) = 2.8778273930986864E-01_dp
      fit_coef(3, 20) = 1.8376769806731383E-01_dp
      fit_coef(4, 20) = 5.2375278020144772E-02_dp
      fit_coef(5, 20) = -2.4188675683434760E-02_dp
      fit_coef(6, 20) = -3.1878082611254147E-02_dp
      fit_coef(7, 20) = -2.0440498451142863E-03_dp
      fit_coef(8, 20) = 1.7743467113242066E-03_dp
      fit_coef(9, 20) = 2.9993028484423083E-02_dp
      fit_coef(10, 20) = -3.7493397222034638E-02_dp
      fit_coef(11, 20) = 1.9771570359339716E-02_dp
      fit_coef(12, 20) = -4.1700973428850313E-03_dp
      fit_coef(0, 21) = 2.3226494166758732E-01_dp
      fit_coef(1, 21) = 3.5273736189295163E-01_dp
      fit_coef(2, 21) = 3.2946524583838299E-01_dp
      fit_coef(3, 21) = 1.9174149746173408E-01_dp
      fit_coef(4, 21) = 5.8141887677596829E-02_dp
      fit_coef(5, 21) = -9.7825016932211791E-03_dp
      fit_coef(6, 21) = -3.0972124419450422E-02_dp
      fit_coef(7, 21) = -2.8877801446827007E-03_dp
      fit_coef(8, 21) = -3.6764584475810547E-02_dp
      fit_coef(9, 21) = 7.0959072404975718E-02_dp
      fit_coef(10, 21) = -3.5178291993696188E-02_dp
      fit_coef(11, 21) = -9.8088379200671617E-04_dp
      fit_coef(12, 21) = 3.7618284525752117E-03_dp
      fit_coef(0, 22) = 3.5052467022994122E-01_dp
      fit_coef(1, 22) = 4.4588641535984419E-01_dp
      fit_coef(2, 22) = 3.6589831022304731E-01_dp
      fit_coef(3, 22) = 1.8352935077226612E-01_dp
      fit_coef(4, 22) = 5.3697669457182857E-02_dp
      fit_coef(5, 22) = 7.4149644860502300E-03_dp
      fit_coef(6, 22) = 1.2817157120917000E-03_dp
      fit_coef(7, 22) = -5.0033726005623343E-02_dp
      fit_coef(8, 22) = 6.8349818056791897E-02_dp
      fit_coef(9, 22) = -1.7000421175238506E-01_dp
      fit_coef(10, 22) = 2.3036729897077399E-01_dp
      fit_coef(11, 22) = -1.3857914112066141E-01_dp
      fit_coef(12, 22) = 3.1478942190343064E-02_dp
      fit_coef(0, 23) = 5.2779750339667775E-01_dp
      fit_coef(1, 23) = 5.5332389225129053E-01_dp
      fit_coef(2, 23) = 3.9690230471520443E-01_dp
      fit_coef(3, 23) = 1.6312883734215308E-01_dp
      fit_coef(4, 23) = 3.9552650126779464E-02_dp
      fit_coef(5, 23) = 2.1230303801123101E-02_dp
      fit_coef(6, 23) = 2.9571154052233411E-02_dp
      fit_coef(7, 23) = -4.5494947806276824E-02_dp
      fit_coef(8, 23) = 6.6833147701713960E-02_dp
      fit_coef(9, 23) = -1.5861734029387040E-01_dp
      fit_coef(10, 23) = 1.3362195160908977E-01_dp
      fit_coef(11, 23) = -3.0781412640455877E-02_dp
      fit_coef(12, 23) = -4.0950711211332350E-03_dp
      fit_coef(0, 24) = 7.9293469396592242E-01_dp
      fit_coef(1, 24) = 6.7009858113053966E-01_dp
      fit_coef(2, 24) = 4.2145481861873185E-01_dp
      fit_coef(3, 24) = 1.3642048218640376E-01_dp
      fit_coef(4, 24) = 2.1922752252294477E-02_dp
      fit_coef(5, 24) = 3.2874744499942446E-02_dp
      fit_coef(6, 24) = 3.0227410874256264E-02_dp
      fit_coef(7, 24) = 3.7127938956607313E-02_dp
      fit_coef(8, 24) = -1.1080272293058828E-01_dp
      fit_coef(9, 24) = 1.5696651468543765E-01_dp
      fit_coef(10, 24) = -2.2158444445106978E-01_dp
      fit_coef(11, 24) = 1.6507639589038958E-01_dp
      fit_coef(12, 24) = -4.4931540554666242E-02_dp
      fit_coef(0, 25) = 1.1922412385770986E+00_dp
      fit_coef(1, 25) = 7.8786890944540700E-01_dp
      fit_coef(2, 25) = 4.3978600997964423E-01_dp
      fit_coef(3, 25) = 1.0955812452827124E-01_dp
      fit_coef(4, 25) = 7.1243000175333282E-03_dp
      fit_coef(5, 25) = 3.8586652562380884E-02_dp
      fit_coef(6, 25) = 3.7152220150416283E-02_dp
      fit_coef(7, 25) = 5.1558439446380777E-02_dp
      fit_coef(8, 25) = -1.1083564016687726E-01_dp
      fit_coef(9, 25) = 1.5290802796167138E-01_dp
      fit_coef(10, 25) = -1.7624313473286510E-01_dp
      fit_coef(11, 25) = 9.5327501416994614E-02_dp
      fit_coef(12, 25) = -1.7028005023958050E-02_dp
      fit_coef(0, 26) = 1.8099472486487456E+00_dp
      fit_coef(1, 26) = 8.9758639724651268E-01_dp
      fit_coef(2, 26) = 4.5619407404374801E-01_dp
      fit_coef(3, 26) = 8.8963048208584819E-02_dp
      fit_coef(4, 26) = -2.5842727308316919E-03_dp
      fit_coef(5, 26) = 3.9852736978965124E-02_dp
      fit_coef(6, 26) = 4.9637846366422017E-02_dp
      fit_coef(7, 26) = 1.7104716626654703E-02_dp
      fit_coef(8, 26) = -5.1404130029074147E-03_dp
      fit_coef(9, 26) = -1.8271352476782278E-02_dp
      fit_coef(10, 26) = 3.0152636564646153E-02_dp
      fit_coef(11, 26) = -4.0913238884136026E-02_dp
      fit_coef(12, 26) = 1.7209300973137127E-02_dp
      fit_coef(0, 27) = 2.8415520058389325E+00_dp
      fit_coef(1, 27) = 9.9858013711246008E-01_dp
      fit_coef(2, 27) = 4.8263553345863242E-01_dp
      fit_coef(3, 27) = 8.1362436826961093E-02_dp
      fit_coef(4, 27) = -8.1050564154871527E-03_dp
      fit_coef(5, 27) = 3.9674059022610446E-02_dp
      fit_coef(6, 27) = 5.4176205880144122E-02_dp
      fit_coef(7, 27) = 2.6408258437705550E-03_dp
      fit_coef(8, 27) = 3.6418581121829920E-02_dp
      fit_coef(9, 27) = -7.2590963148842197E-02_dp
      fit_coef(10, 27) = 9.0190833590845856E-02_dp
      fit_coef(11, 27) = -6.7850000466319185E-02_dp
      fit_coef(12, 27) = 1.8925208943418582E-02_dp
      fit_coef(0, 28) = 5.0338617688312999E+00_dp
      fit_coef(1, 28) = 1.1415572014123594E+00_dp
      fit_coef(2, 28) = 5.5834539482924794E-01_dp
      fit_coef(3, 28) = 9.8824807093161326E-02_dp
      fit_coef(4, 28) = -1.0740041647079411E-02_dp
      fit_coef(5, 28) = 3.9309450063690091E-02_dp
      fit_coef(6, 28) = 5.4929203884089291E-02_dp
      fit_coef(7, 28) = 1.2322580840140561E-02_dp
      fit_coef(8, 28) = 1.8323929549560497E-02_dp
      fit_coef(9, 28) = -2.9383377972038723E-02_dp
      fit_coef(10, 28) = 4.2820470862085666E-02_dp
      fit_coef(11, 28) = -2.8686857835350920E-02_dp
      fit_coef(12, 28) = 5.4565805278581868E-03_dp

   END SUBROUTINE get_fit_coef_k14_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k14_mi(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.1722808202408223E+03_dp
      fit_coef(0, 1) = 5.4467895546803416E-04_dp
      fit_coef(1, 1) = 1.5857771854287006E-03_dp
      fit_coef(2, 1) = 2.3073428933440593E-03_dp
      fit_coef(3, 1) = 2.0056780241026078E-03_dp
      fit_coef(4, 1) = 9.5252806530394737E-04_dp
      fit_coef(5, 1) = 4.5225480425719547E-05_dp
      fit_coef(6, 1) = -2.1217629613097376E-04_dp
      fit_coef(7, 1) = -5.0216895355491796E-05_dp
      fit_coef(8, 1) = 3.8287705248810476E-05_dp
      fit_coef(9, 1) = 7.2535948309621214E-05_dp
      fit_coef(10, 1) = -9.2252835912099736E-05_dp
      fit_coef(11, 1) = 4.3983534472290816E-05_dp
      fit_coef(12, 1) = -8.2237204784032551E-06_dp
      fit_coef(0, 2) = 3.0464535721029173E-03_dp
      fit_coef(1, 2) = 8.6149108928923676E-03_dp
      fit_coef(2, 2) = 1.2362165768730629E-02_dp
      fit_coef(3, 2) = 1.0619459122609453E-02_dp
      fit_coef(4, 2) = 4.9727394573446972E-03_dp
      fit_coef(5, 2) = 2.0040047900640227E-04_dp
      fit_coef(6, 2) = -1.1121250175071501E-03_dp
      fit_coef(7, 2) = -2.5253190910672047E-04_dp
      fit_coef(8, 2) = 2.0786942223541925E-04_dp
      fit_coef(9, 2) = 3.6647453679821519E-04_dp
      fit_coef(10, 2) = -4.7725200991211036E-04_dp
      fit_coef(11, 2) = 2.3017003059152598E-04_dp
      fit_coef(12, 2) = -4.3440212796964786E-05_dp
      fit_coef(0, 3) = 8.4179094658504636E-03_dp
      fit_coef(1, 3) = 2.2382112298387765E-02_dp
      fit_coef(2, 3) = 3.1254689785869226E-02_dp
      fit_coef(3, 3) = 2.6305836769925709E-02_dp
      fit_coef(4, 3) = 1.2019230418388028E-02_dp
      fit_coef(5, 3) = 3.2210723317093755E-04_dp
      fit_coef(6, 3) = -2.7056207855165560E-03_dp
      fit_coef(7, 3) = -5.6040022077270704E-04_dp
      fit_coef(8, 3) = 5.4121141114160339E-04_dp
      fit_coef(9, 3) = 8.1004167044932199E-04_dp
      fit_coef(10, 3) = -1.1112740690897719E-03_dp
      fit_coef(11, 3) = 5.4779892766252693E-04_dp
      fit_coef(12, 3) = -1.0512838684553561E-04_dp
      fit_coef(0, 4) = 1.8743574617056585E-02_dp
      fit_coef(1, 4) = 4.5210340778486291E-02_dp
      fit_coef(2, 4) = 6.0224002984973317E-02_dp
      fit_coef(3, 4) = 4.9094876694850402E-02_dp
      fit_coef(4, 4) = 2.1831501314762986E-02_dp
      fit_coef(5, 4) = 3.2839489812930983E-04_dp
      fit_coef(6, 4) = -5.0144416787406901E-03_dp
      fit_coef(7, 4) = -9.7843840294365748E-04_dp
      fit_coef(8, 4) = 1.0342060306701268E-03_dp
      fit_coef(9, 4) = 1.5910763103707255E-03_dp
      fit_coef(10, 4) = -2.2573252364893278E-03_dp
      fit_coef(11, 4) = 1.1445941503911742E-03_dp
      fit_coef(12, 4) = -2.2506139447365695E-04_dp
      fit_coef(0, 5) = 3.8243937154880572E-02_dp
      fit_coef(1, 5) = 8.1467829047918827E-02_dp
      fit_coef(2, 5) = 1.0129786520932144E-01_dp
      fit_coef(3, 5) = 7.8216152968887814E-02_dp
      fit_coef(4, 5) = 3.3350941393280653E-02_dp
      fit_coef(5, 5) = 5.2823533556460426E-04_dp
      fit_coef(6, 5) = -7.3749832466795265E-03_dp
      fit_coef(7, 5) = -1.5646554141616070E-03_dp
      fit_coef(8, 5) = 9.1282587135457208E-04_dp
      fit_coef(9, 5) = 2.8481769249969957E-03_dp
      fit_coef(10, 5) = -3.1292839864729078E-03_dp
      fit_coef(11, 5) = 1.3433122010827140E-03_dp
      fit_coef(12, 5) = -2.2591769996675308E-04_dp
      fit_coef(0, 6) = 7.4571737250878262E-02_dp
      fit_coef(1, 6) = 1.3811565890497141E-01_dp
      fit_coef(2, 6) = 1.5798315551672285E-01_dp
      fit_coef(3, 6) = 1.1260753901599996E-01_dp
      fit_coef(4, 6) = 4.4100222118730312E-02_dp
      fit_coef(5, 6) = 2.1773862891778066E-04_dp
      fit_coef(6, 6) = -8.3087822491922161E-03_dp
      fit_coef(7, 6) = -2.1043028241427880E-03_dp
      fit_coef(8, 6) = 1.8792681898943972E-03_dp
      fit_coef(9, 6) = 3.1606237510210035E-04_dp
      fit_coef(10, 6) = -5.4242874432157808E-04_dp
      fit_coef(11, 6) = 2.2538605364452114E-04_dp
      fit_coef(12, 6) = -5.0219363398529674E-05_dp
      fit_coef(0, 7) = 1.4098571554579828E-01_dp
      fit_coef(1, 7) = 2.2478771441370507E-01_dp
      fit_coef(2, 7) = 2.3494329165976915E-01_dp
      fit_coef(3, 7) = 1.5140646411833683E-01_dp
      fit_coef(4, 7) = 5.1480514391188670E-02_dp
      fit_coef(5, 7) = -1.9136093640078735E-03_dp
      fit_coef(6, 7) = -7.8257777325216969E-03_dp
      fit_coef(7, 7) = -9.9807084346081507E-04_dp
      fit_coef(8, 7) = 2.2963464957662357E-03_dp
      fit_coef(9, 7) = -1.9447302259471774E-03_dp
      fit_coef(10, 7) = 4.3017240890955453E-04_dp
      fit_coef(11, 7) = 4.7144332953281682E-04_dp
      fit_coef(12, 7) = -2.1997791857150025E-04_dp
      fit_coef(0, 8) = 2.5999858418452071E-01_dp
      fit_coef(1, 8) = 3.5368499241454637E-01_dp
      fit_coef(2, 8) = 3.3704821274394664E-01_dp
      fit_coef(3, 8) = 1.9318903274447125E-01_dp
      fit_coef(4, 8) = 5.3280944838378592E-02_dp
      fit_coef(5, 8) = -6.3861100195045095E-03_dp
      fit_coef(6, 8) = -5.8003810200407740E-03_dp
      fit_coef(7, 8) = 1.4075796032845642E-03_dp
      fit_coef(8, 8) = 2.4434092283670862E-03_dp
      fit_coef(9, 8) = -4.5616600603175456E-03_dp
      fit_coef(10, 8) = 1.8926356543785563E-03_dp
      fit_coef(11, 8) = 2.2201716818935917E-04_dp
      fit_coef(12, 8) = -2.2032045330927993E-04_dp
      fit_coef(0, 9) = 4.6927925194228570E-01_dp
      fit_coef(1, 9) = 5.3896976247930295E-01_dp
      fit_coef(2, 9) = 4.6838489154168161E-01_dp
      fit_coef(3, 9) = 2.3526229136436330E-01_dp
      fit_coef(4, 9) = 4.7454952056688732E-02_dp
      fit_coef(5, 9) = -1.2685981031239016E-02_dp
      fit_coef(6, 9) = -1.4744220873732451E-03_dp
      fit_coef(7, 9) = 4.2905695407292228E-03_dp
      fit_coef(8, 9) = 2.4711991371951058E-03_dp
      fit_coef(9, 9) = -8.6550441967466144E-03_dp
      fit_coef(10, 9) = 5.5989569069497076E-03_dp
      fit_coef(11, 9) = -1.3463966560621509E-03_dp
      fit_coef(12, 9) = 5.6917396181052615E-05_dp
      fit_coef(0, 10) = 8.3113745246756776E-01_dp
      fit_coef(1, 10) = 7.9501654600703975E-01_dp
      fit_coef(2, 10) = 6.3118071249392171E-01_dp
      fit_coef(3, 10) = 2.7364530371013518E-01_dp
      fit_coef(4, 10) = 3.2333401581906181E-02_dp
      fit_coef(5, 10) = -1.9422043063657940E-02_dp
      fit_coef(6, 10) = 6.0392759901066601E-03_dp
      fit_coef(7, 10) = 7.1646103557214402E-03_dp
      fit_coef(8, 10) = 1.3209371123133807E-03_dp
      fit_coef(9, 10) = -1.3329493616005855E-02_dp
      fit_coef(10, 10) = 1.1118391120775293E-02_dp
      fit_coef(11, 10) = -3.9802162789608605E-03_dp
      fit_coef(12, 10) = 5.4198727348417862E-04_dp
      fit_coef(0, 11) = 1.4482023274879392E+00_dp
      fit_coef(1, 11) = 1.1330048924955463E+00_dp
      fit_coef(2, 11) = 8.2485522761522434E-01_dp
      fit_coef(3, 11) = 3.0394925510152571E-01_dp
      fit_coef(4, 11) = 7.2638847029286640E-03_dp
      fit_coef(5, 11) = -2.4671422910119499E-02_dp
      fit_coef(6, 11) = 1.7366475605224755E-02_dp
      fit_coef(7, 11) = 9.3584526548286841E-03_dp
      fit_coef(8, 11) = -1.8731252836985278E-03_dp
      fit_coef(9, 11) = -1.7532663892185343E-02_dp
      fit_coef(10, 11) = 1.7972283371407362E-02_dp
      fit_coef(11, 11) = -7.6180802914050756E-03_dp
      fit_coef(12, 11) = 1.2561263859800901E-03_dp
      fit_coef(0, 12) = 2.4920360801417596E+00_dp
      fit_coef(1, 12) = 1.5562052067140830E+00_dp
      fit_coef(2, 12) = 1.0460115631746023E+00_dp
      fit_coef(3, 12) = 3.2329508963929127E-01_dp
      fit_coef(4, 12) = -2.6843427897215052E-02_dp
      fit_coef(5, 12) = -2.6728379775572533E-02_dp
      fit_coef(6, 12) = 3.2440057594826148E-02_dp
      fit_coef(7, 12) = 1.0203339477948091E-02_dp
      fit_coef(8, 12) = -7.5116389543029823E-03_dp
      fit_coef(9, 12) = -2.0251551518009794E-02_dp
      fit_coef(10, 12) = 2.5495522854135564E-02_dp
      fit_coef(11, 12) = -1.2104045379629679E-02_dp
      fit_coef(12, 12) = 2.2002184184182066E-03_dp
      fit_coef(0, 13) = 4.2695725351678870E+00_dp
      fit_coef(1, 13) = 2.0569142521947663E+00_dp
      fit_coef(2, 13) = 1.2906701222000276E+00_dp
      fit_coef(3, 13) = 3.3274287972587252E-01_dp
      fit_coef(4, 13) = -6.7760240413352227E-02_dp
      fit_coef(5, 13) = -2.5210323779882499E-02_dp
      fit_coef(6, 13) = 5.0470840558373135E-02_dp
      fit_coef(7, 13) = 9.5160539695092316E-03_dp
      fit_coef(8, 13) = -1.5360600121122914E-02_dp
      fit_coef(9, 13) = -2.1060084216392275E-02_dp
      fit_coef(10, 13) = 3.3061137892312159E-02_dp
      fit_coef(11, 13) = -1.7135171327692631E-02_dp
      fit_coef(12, 13) = 3.3198097857474905E-03_dp
      fit_coef(0, 14) = 7.4706366451203330E+00_dp
      fit_coef(1, 14) = 2.6255475168075386E+00_dp
      fit_coef(2, 14) = 1.5617856992687866E+00_dp
      fit_coef(3, 14) = 3.3898821868503676E-01_dp
      fit_coef(4, 14) = -1.1349423621812627E-01_dp
      fit_coef(5, 14) = -2.1752415544934678E-02_dp
      fit_coef(6, 14) = 7.0659941513993574E-02_dp
      fit_coef(7, 14) = 7.9261305614328585E-03_dp
      fit_coef(8, 14) = -2.4678202093857821E-02_dp
      fit_coef(9, 14) = -2.0791238512333610E-02_dp
      fit_coef(10, 14) = 4.0834605352408383E-02_dp
      fit_coef(11, 14) = -2.2601414883140436E-02_dp
      fit_coef(12, 14) = 4.5678225526306271E-03_dp
      fit_coef(0, 15) = 1.4166144548081244E-03_dp
      fit_coef(1, 15) = 4.0988329863504650E-03_dp
      fit_coef(2, 15) = 5.9448187504450257E-03_dp
      fit_coef(3, 15) = 5.1533064863913264E-03_dp
      fit_coef(4, 15) = 2.4394418491756548E-03_dp
      fit_coef(5, 15) = 1.1163214246871172E-04_dp
      fit_coef(6, 15) = -5.4395417380967090E-04_dp
      fit_coef(7, 15) = -1.2744617101115058E-04_dp
      fit_coef(8, 15) = 9.9020248580942340E-05_dp
      fit_coef(9, 15) = 1.8429376434903194E-04_dp
      fit_coef(10, 15) = -2.3572941138030507E-04_dp
      fit_coef(11, 15) = 1.1270257688728590E-04_dp
      fit_coef(12, 15) = -2.1120306082703953E-05_dp
      fit_coef(0, 16) = 3.7180060821504178E-03_dp
      fit_coef(1, 16) = 1.0136250443655282E-02_dp
      fit_coef(2, 16) = 1.4300959605183702E-02_dp
      fit_coef(3, 16) = 1.2111006697402966E-02_dp
      fit_coef(4, 16) = 5.5715231524265116E-03_dp
      fit_coef(5, 16) = 1.7441962760050931E-04_dp
      fit_coef(6, 16) = -1.2509495746884976E-03_dp
      fit_coef(7, 16) = -2.6932600471792030E-04_dp
      fit_coef(8, 16) = 2.4202856508010202E-04_dp
      fit_coef(9, 16) = 3.9544082532111499E-04_dp
      fit_coef(10, 16) = -5.2934481355649469E-04_dp
      fit_coef(11, 16) = 2.5870684688921450E-04_dp
      fit_coef(12, 16) = -4.9352748645601205E-05_dp
      fit_coef(0, 17) = 7.3573792852102415E-03_dp
      fit_coef(1, 17) = 1.7769053739351975E-02_dp
      fit_coef(2, 17) = 2.3686656634361428E-02_dp
      fit_coef(3, 17) = 1.9285714809028087E-02_dp
      fit_coef(4, 17) = 8.4872745881098346E-03_dp
      fit_coef(5, 17) = 3.4696339797671155E-05_dp
      fit_coef(6, 17) = -1.9464426054708341E-03_dp
      fit_coef(7, 17) = -3.4174035665184455E-04_dp
      fit_coef(8, 17) = 4.7554753790861559E-04_dp
      fit_coef(9, 17) = 4.2925599708472817E-04_dp
      fit_coef(10, 17) = -7.0959124228041825E-04_dp
      fit_coef(11, 17) = 3.7293802063667668E-04_dp
      fit_coef(12, 17) = -7.4872640261499496E-05_dp
      fit_coef(0, 18) = 1.3977177746865988E-02_dp
      fit_coef(1, 18) = 2.8606023538180633E-02_dp
      fit_coef(2, 18) = 3.4571612966699722E-02_dp
      fit_coef(3, 18) = 2.6149166759159050E-02_dp
      fit_coef(4, 18) = 1.0984913521363774E-02_dp
      fit_coef(5, 18) = 7.7055787642742803E-05_dp
      fit_coef(6, 18) = -2.5922755331103330E-03_dp
      fit_coef(7, 18) = -4.8489977776779012E-04_dp
      fit_coef(8, 18) = 1.5912901418092180E-04_dp
      fit_coef(9, 18) = 1.5881868365226204E-03_dp
      fit_coef(10, 18) = -1.8151464311114490E-03_dp
      fit_coef(11, 18) = 8.5592821993186205E-04_dp
      fit_coef(12, 18) = -1.5937383356627851E-04_dp
      fit_coef(0, 19) = 2.6264101814506675E-02_dp
      fit_coef(1, 19) = 4.5069161923578932E-02_dp
      fit_coef(2, 19) = 4.8169189713700107E-02_dp
      fit_coef(3, 19) = 3.1904492882075079E-02_dp
      fit_coef(4, 19) = 1.1603766261612461E-02_dp
      fit_coef(5, 19) = 1.8449908333587032E-04_dp
      fit_coef(6, 19) = -1.7877726122463406E-03_dp
      fit_coef(7, 19) = -7.5764206916605059E-04_dp
      fit_coef(8, 19) = 2.8343231172872694E-04_dp
      fit_coef(9, 19) = -5.9914897854961122E-04_dp
      fit_coef(10, 19) = 1.3517055951379382E-03_dp
      fit_coef(11, 19) = -9.4046015820634316E-04_dp
      fit_coef(12, 19) = 2.2350581726021887E-04_dp
      fit_coef(0, 20) = 4.8542207101173400E-02_dp
      fit_coef(1, 20) = 6.9832820033889687E-02_dp
      fit_coef(2, 20) = 6.5995740079255238E-02_dp
      fit_coef(3, 20) = 3.6752596227584124E-02_dp
      fit_coef(4, 20) = 9.4571244340008204E-03_dp
      fit_coef(5, 20) = -1.0484402051494153E-03_dp
      fit_coef(6, 20) = -1.5752390773495777E-04_dp
      fit_coef(7, 20) = 1.0876735653687303E-04_dp
      fit_coef(8, 20) = 1.0073470936970389E-03_dp
      fit_coef(9, 20) = -3.0335098186363291E-03_dp
      fit_coef(10, 20) = 2.1855940575857963E-03_dp
      fit_coef(11, 20) = -4.5502298284867644E-04_dp
      fit_coef(12, 20) = -3.4790305788945764E-05_dp
      fit_coef(0, 21) = 8.7942456610255598E-02_dp
      fit_coef(1, 21) = 1.0555324383884353E-01_dp
      fit_coef(2, 21) = 8.8752093517154698E-02_dp
      fit_coef(3, 21) = 4.0620455348294761E-02_dp
      fit_coef(4, 21) = 4.9418858666300082E-03_dp
      fit_coef(5, 21) = -3.3032510647944919E-03_dp
      fit_coef(6, 21) = 1.1546343697416441E-03_dp
      fit_coef(7, 21) = 1.9549734572505361E-03_dp
      fit_coef(8, 21) = 5.6213884018718063E-05_dp
      fit_coef(9, 21) = -1.9404012109824945E-03_dp
      fit_coef(10, 21) = 5.3863092062487572E-04_dp
      fit_coef(11, 21) = 3.8099987718174182E-04_dp
      fit_coef(12, 21) = -1.6063218378014254E-04_dp
      fit_coef(0, 22) = 1.5620880999122425E-01_dp
      fit_coef(1, 22) = 1.5462937480065686E-01_dp
      fit_coef(2, 22) = 1.1615728166593994E-01_dp
      fit_coef(3, 22) = 4.2496601438037490E-02_dp
      fit_coef(4, 22) = -1.6862132065151886E-03_dp
      fit_coef(5, 22) = -5.5527712863935753E-03_dp
      fit_coef(6, 22) = 3.0288231280584822E-03_dp
      fit_coef(7, 22) = 2.7239270206899919E-03_dp
      fit_coef(8, 22) = 2.3923207067007299E-04_dp
      fit_coef(9, 22) = -3.4467494238847617E-03_dp
      fit_coef(10, 22) = 2.6143123206984532E-03_dp
      fit_coef(11, 22) = -9.5623661147149601E-04_dp
      fit_coef(12, 22) = 1.5884529743136178E-04_dp
      fit_coef(0, 23) = 2.7248883355523357E-01_dp
      fit_coef(1, 23) = 2.1839288076669880E-01_dp
      fit_coef(2, 23) = 1.4690508945871661E-01_dp
      fit_coef(3, 23) = 4.0972405769053262E-02_dp
      fit_coef(4, 23) = -1.0255836491498508E-02_dp
      fit_coef(5, 23) = -6.8103039861294205E-03_dp
      fit_coef(6, 23) = 5.7774256001662866E-03_dp
      fit_coef(7, 23) = 2.9716307657906005E-03_dp
      fit_coef(8, 23) = -3.8415214985403008E-04_dp
      fit_coef(9, 23) = -4.5597005257117969E-03_dp
      fit_coef(10, 23) = 4.6944833426410494E-03_dp
      fit_coef(11, 23) = -2.1231572186145029E-03_dp
      fit_coef(12, 23) = 3.8196379187575797E-04_dp
      fit_coef(0, 24) = 4.6789642997790454E-01_dp
      fit_coef(1, 24) = 2.9564830559342764E-01_dp
      fit_coef(2, 24) = 1.7859318172912267E-01_dp
      fit_coef(3, 24) = 3.5029171279950352E-02_dp
      fit_coef(4, 24) = -2.0106998533844407E-02_dp
      fit_coef(5, 24) = -6.3290051255165099E-03_dp
      fit_coef(6, 24) = 9.3627910136797072E-03_dp
      fit_coef(7, 24) = 2.6599389408438201E-03_dp
      fit_coef(8, 24) = -2.0615332759583711E-03_dp
      fit_coef(9, 24) = -4.6173560793899586E-03_dp
      fit_coef(10, 24) = 6.2747096995400117E-03_dp
      fit_coef(11, 24) = -3.1434465968595239E-03_dp
      fit_coef(12, 24) = 5.9448478612450029E-04_dp
      fit_coef(0, 25) = 7.9390404143601678E-01_dp
      fit_coef(1, 25) = 3.8094350576553354E-01_dp
      fit_coef(2, 25) = 2.0815634677459935E-01_dp
      fit_coef(3, 25) = 2.5054416108850112E-02_dp
      fit_coef(4, 25) = -2.9877418297843340E-02_dp
      fit_coef(5, 25) = -3.8648566065354233E-03_dp
      fit_coef(6, 25) = 1.3281740014574807E-02_dp
      fit_coef(7, 25) = 1.6073039898392629E-03_dp
      fit_coef(8, 25) = -4.3975022763734321E-03_dp
      fit_coef(9, 25) = -3.6092937145507167E-03_dp
      fit_coef(10, 25) = 7.3177671091434133E-03_dp
      fit_coef(11, 25) = -4.1069025396153367E-03_dp
      fit_coef(12, 25) = 8.3442522049861077E-04_dp
      fit_coef(0, 26) = 1.3421299372909123E+00_dp
      fit_coef(1, 26) = 4.6409313440446603E-01_dp
      fit_coef(2, 26) = 2.3339441406477554E-01_dp
      fit_coef(3, 26) = 1.3785053308326188E-02_dp
      fit_coef(4, 26) = -3.7941211960007708E-02_dp
      fit_coef(5, 26) = -1.7132484132012456E-04_dp
      fit_coef(6, 26) = 1.6722419910236191E-02_dp
      fit_coef(7, 26) = 4.5287606479249242E-05_dp
      fit_coef(8, 26) = -6.8396430449082519E-03_dp
      fit_coef(9, 26) = -1.7433447611092249E-03_dp
      fit_coef(10, 26) = 7.6094558147861901E-03_dp
      fit_coef(11, 26) = -4.8080400030221532E-03_dp
      fit_coef(12, 26) = 1.0435040492483968E-03_dp
      fit_coef(0, 27) = 2.3127647897779702E+00_dp
      fit_coef(1, 27) = 5.3494805569190185E-01_dp
      fit_coef(2, 27) = 2.5603491110440707E-01_dp
      fit_coef(3, 27) = 6.2124469115487774E-03_dp
      fit_coef(4, 27) = -4.3500571617098645E-02_dp
      fit_coef(5, 27) = 2.9267102310614004E-03_dp
      fit_coef(6, 27) = 1.9176839114778386E-02_dp
      fit_coef(7, 27) = -1.3114239457504276E-03_dp
      fit_coef(8, 27) = -8.7196471933934649E-03_dp
      fit_coef(9, 27) = -1.4236987601663990E-05_dp
      fit_coef(10, 27) = 7.5377946011018363E-03_dp
      fit_coef(11, 27) = -5.2272346094548762E-03_dp
      fit_coef(12, 27) = 1.1857786912012739E-03_dp
      fit_coef(0, 28) = 4.4310556767314351E+00_dp
      fit_coef(1, 28) = 6.0851379611857248E-01_dp
      fit_coef(2, 28) = 2.9194514151733547E-01_dp
      fit_coef(3, 28) = 8.6001943701959644E-03_dp
      fit_coef(4, 28) = -4.8295972382001021E-02_dp
      fit_coef(5, 28) = 3.4189484756573089E-03_dp
      fit_coef(6, 28) = 2.1364281048084820E-02_dp
      fit_coef(7, 28) = -1.6460148108292097E-03_dp
      fit_coef(8, 28) = -9.8467342034871362E-03_dp
      fit_coef(9, 28) = 1.9090555066223413E-04_dp
      fit_coef(10, 28) = 8.3032333199491645E-03_dp
      fit_coef(11, 28) = -5.8101678643582262E-03_dp
      fit_coef(12, 28) = 1.3244384992621098E-03_dp

   END SUBROUTINE get_fit_coef_k14_mi

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k14_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 9.4708858956711737E+01_dp
      U_b = 6.6910000000000000E+05_dp
      fit_coef(0, 1) = 5.7644508135195325E-06_dp
      fit_coef(1, 1) = 8.8059899546543633E-06_dp
      fit_coef(2, 1) = 9.8246367093007174E-04_dp
      fit_coef(3, 1) = -5.7370459929060305E-03_dp
      fit_coef(4, 1) = 5.4909691175756163E-02_dp
      fit_coef(5, 1) = -2.3201875112604708E-01_dp
      fit_coef(6, 1) = 7.0724150950356890E-01_dp
      fit_coef(7, 1) = -1.4925604266418715E+00_dp
      fit_coef(8, 1) = 2.1603279291789366E+00_dp
      fit_coef(9, 1) = -2.1043657951755335E+00_dp
      fit_coef(10, 1) = 1.3179096181212082E+00_dp
      fit_coef(11, 1) = -4.7916040323369041E-01_dp
      fit_coef(12, 1) = 7.6853944770310323E-02_dp
      fit_coef(0, 2) = 6.3482217934141788E-05_dp
      fit_coef(1, 2) = 5.6715849587820812E-05_dp
      fit_coef(2, 2) = 6.2902935744076748E-03_dp
      fit_coef(3, 2) = -3.8825646732371183E-02_dp
      fit_coef(4, 2) = 3.5215367034308359E-01_dp
      fit_coef(5, 2) = -1.5068239503633283E+00_dp
      fit_coef(6, 2) = 4.5854281378525714E+00_dp
      fit_coef(7, 2) = -9.6460393022334507E+00_dp
      fit_coef(8, 2) = 1.3925086681528807E+01_dp
      fit_coef(9, 2) = -1.3538171863016146E+01_dp
      fit_coef(10, 2) = 8.4668043138765512E+00_dp
      fit_coef(11, 2) = -3.0752413149585256E+00_dp
      fit_coef(12, 2) = 4.9288840241270471E-01_dp
      fit_coef(0, 3) = 3.3613588297925877E-04_dp
      fit_coef(1, 3) = 2.0810718441537009E-04_dp
      fit_coef(2, 3) = 2.2496352429605227E-02_dp
      fit_coef(3, 3) = -1.4508680125471160E-01_dp
      fit_coef(4, 3) = 1.2240878955617038E+00_dp
      fit_coef(5, 3) = -5.4269306653344804E+00_dp
      fit_coef(6, 3) = 1.6783138844933998E+01_dp
      fit_coef(7, 3) = -3.5628143752717627E+01_dp
      fit_coef(8, 3) = 5.1720421721162715E+01_dp
      fit_coef(9, 3) = -5.0458114469889409E+01_dp
      fit_coef(10, 3) = 3.1625635386312588E+01_dp
      fit_coef(11, 3) = -1.1502746293672999E+01_dp
      fit_coef(12, 3) = 1.8452512921088244E+00_dp
      fit_coef(0, 4) = 1.2977058344000711E-03_dp
      fit_coef(1, 4) = 6.0051493044492425E-04_dp
      fit_coef(2, 4) = 6.4379175573630529E-02_dp
      fit_coef(3, 4) = -4.2944616668718744E-01_dp
      fit_coef(4, 4) = 3.4645304257752727E+00_dp
      fit_coef(5, 4) = -1.5790291716256178E+01_dp
      fit_coef(6, 4) = 4.9078910409345937E+01_dp
      fit_coef(7, 4) = -1.0406732095355643E+02_dp
      fit_coef(8, 4) = 1.5069319179330122E+02_dp
      fit_coef(9, 4) = -1.4664743032705482E+02_dp
      fit_coef(10, 4) = 9.1717900592693681E+01_dp
      fit_coef(11, 4) = -3.3301586691674665E+01_dp
      fit_coef(12, 4) = 5.3348395481918729E+00_dp
      fit_coef(0, 5) = 4.1564666328203293E-03_dp
      fit_coef(1, 5) = 1.5024272029496644E-03_dp
      fit_coef(2, 5) = 1.5961690430876282E-01_dp
      fit_coef(3, 5) = -1.0856048657785689E+00_dp
      fit_coef(4, 5) = 8.4819759793848739E+00_dp
      fit_coef(5, 5) = -3.9236853866944621E+01_dp
      fit_coef(6, 5) = 1.2225270389761864E+02_dp
      fit_coef(7, 5) = -2.5932080520941867E+02_dp
      fit_coef(8, 5) = 3.7560378396472731E+02_dp
      fit_coef(9, 5) = -3.6564399683386677E+02_dp
      fit_coef(10, 5) = 2.2876919730710458E+02_dp
      fit_coef(11, 5) = -8.3091860520998807E+01_dp
      fit_coef(12, 5) = 1.3315159702194723E+01_dp
      fit_coef(0, 6) = 1.1728185806559138E-02_dp
      fit_coef(1, 6) = 3.3854671547705814E-03_dp
      fit_coef(2, 6) = 3.5708079041567914E-01_dp
      fit_coef(3, 6) = -2.4616715243590788E+00_dp
      fit_coef(4, 6) = 1.8784735712246597E+01_dp
      fit_coef(5, 6) = -8.7758450553219319E+01_dp
      fit_coef(6, 6) = 2.7430420638340945E+02_dp
      fit_coef(7, 6) = -5.8284545441781790E+02_dp
      fit_coef(8, 6) = 8.4515131830014900E+02_dp
      fit_coef(9, 6) = -8.2334556633746092E+02_dp
      fit_coef(10, 6) = 5.1537716566398308E+02_dp
      fit_coef(11, 6) = -1.8724712324686436E+02_dp
      fit_coef(12, 6) = 3.0011313266344054E+01_dp
      fit_coef(0, 7) = 3.0150116177462289E-02_dp
      fit_coef(1, 7) = 7.0271320856721093E-03_dp
      fit_coef(2, 7) = 7.3674268137292198E-01_dp
      fit_coef(3, 7) = -5.1296976804148287E+00_dp
      fit_coef(4, 7) = 3.8429636567626652E+01_dp
      fit_coef(5, 7) = -1.8076150868535993E+02_dp
      fit_coef(6, 7) = 5.6676422159052879E+02_dp
      fit_coef(7, 7) = -1.2064911489924480E+03_dp
      fit_coef(8, 7) = 1.7514726329556534E+03_dp
      fit_coef(9, 7) = -1.7075068700281913E+03_dp
      fit_coef(10, 7) = 1.0693144394521294E+03_dp
      fit_coef(11, 7) = -3.8862358061084939E+02_dp
      fit_coef(12, 7) = 6.2300629367049794E+01_dp
      fit_coef(0, 8) = 7.2115669678109895E-02_dp
      fit_coef(1, 8) = 1.3616506829774023E-02_dp
      fit_coef(2, 8) = 1.4203294247117737E+00_dp
      fit_coef(3, 8) = -9.9648479573798241E+00_dp
      fit_coef(4, 8) = 7.3547304516631314E+01_dp
      fit_coef(5, 8) = -3.4763570685939845E+02_dp
      fit_coef(6, 8) = 1.0931366005010798E+03_dp
      fit_coef(7, 8) = -2.3311962044304441E+03_dp
      fit_coef(8, 8) = 3.3880559926822129E+03_dp
      fit_coef(9, 8) = -3.3054142191968185E+03_dp
      fit_coef(10, 8) = 2.0709975619174938E+03_dp
      fit_coef(11, 8) = -7.5292079976886191E+02_dp
      fit_coef(12, 8) = 1.2073055461067925E+02_dp
      fit_coef(0, 9) = 1.6278592399460171E-01_dp
      fit_coef(1, 9) = 2.4823462702639722E-02_dp
      fit_coef(2, 9) = 2.5779844651162875E+00_dp
      fit_coef(3, 9) = -1.8195793409897270E+01_dp
      fit_coef(4, 9) = 1.3264400733833980E+02_dp
      fit_coef(5, 9) = -6.2921784847028903E+02_dp
      fit_coef(6, 9) = 1.9837274033715200E+03_dp
      fit_coef(7, 9) = -4.2376252580133068E+03_dp
      fit_coef(8, 9) = 6.1654716911349278E+03_dp
      fit_coef(9, 9) = -6.0193635625448478E+03_dp
      fit_coef(10, 9) = 3.7732320110600213E+03_dp
      fit_coef(11, 9) = -1.3722389889916640E+03_dp
      fit_coef(12, 9) = 2.2009155940297515E+02_dp
      fit_coef(0, 10) = 3.5032689539851275E-01_dp
      fit_coef(1, 10) = 4.2737381729215401E-02_dp
      fit_coef(2, 10) = 4.4215419106194087E+00_dp
      fit_coef(3, 10) = -3.1358951004261929E+01_dp
      fit_coef(4, 10) = 2.2622710383931096E+02_dp
      fit_coef(5, 10) = -1.0760101708193520E+03_dp
      fit_coef(6, 10) = 3.4001822298289899E+03_dp
      fit_coef(7, 10) = -7.2748054353872876E+03_dp
      fit_coef(8, 10) = 1.0595181287227540E+04_dp
      fit_coef(9, 10) = -1.0351115266133067E+04_dp
      fit_coef(10, 10) = 6.4916005963247544E+03_dp
      fit_coef(11, 10) = -2.3616229307375006E+03_dp
      fit_coef(12, 10) = 3.7886792111335615E+02_dp
      fit_coef(0, 11) = 7.2447713579898398E-01_dp
      fit_coef(1, 11) = 6.9519327565965100E-02_dp
      fit_coef(2, 11) = 7.1687589104301583E+00_dp
      fit_coef(3, 11) = -5.1041793387291833E+01_dp
      fit_coef(4, 11) = 3.6499013684520196E+02_dp
      fit_coef(5, 11) = -1.7394623733769292E+03_dp
      fit_coef(6, 11) = 5.5078086466424802E+03_dp
      fit_coef(7, 11) = -1.1800797526080492E+04_dp
      fit_coef(8, 11) = 1.7203067323847517E+04_dp
      fit_coef(9, 11) = -1.6817396268950357E+04_dp
      fit_coef(10, 11) = 1.0551469125979243E+04_dp
      fit_coef(11, 11) = -3.8397827922054616E+03_dp
      fit_coef(12, 11) = 6.1614478300189171E+02_dp
      fit_coef(0, 12) = 1.4502467390425808E+00_dp
      fit_coef(1, 12) = 1.0661107753602478E-01_dp
      fit_coef(2, 12) = 1.0963185897291746E+01_dp
      fit_coef(3, 12) = -7.8300692460550920E+01_dp
      fit_coef(4, 12) = 5.5584223144558530E+02_dp
      fit_coef(5, 12) = -2.6528572306059414E+03_dp
      fit_coef(6, 12) = 8.4143371492021706E+03_dp
      fit_coef(7, 12) = -1.8050418749368942E+04_dp
      fit_coef(8, 12) = 2.6335573301758028E+04_dp
      fit_coef(9, 12) = -2.5759713754102195E+04_dp
      fit_coef(10, 12) = 1.6168369713625771E+04_dp
      fit_coef(11, 12) = -5.8854889464635708E+03_dp
      fit_coef(12, 12) = 9.4460223869839535E+02_dp
      fit_coef(0, 13) = 2.8395717023886369E+00_dp
      fit_coef(1, 13) = 1.5366537352589704E-01_dp
      fit_coef(2, 13) = 1.5767280021363437E+01_dp
      fit_coef(3, 13) = -1.1287621706718255E+02_dp
      fit_coef(4, 13) = 7.9673183153700211E+02_dp
      fit_coef(5, 13) = -3.8063687624389750E+03_dp
      fit_coef(6, 13) = 1.2089346236379877E+04_dp
      fit_coef(7, 13) = -2.5959938226844133E+04_dp
      fit_coef(8, 13) = 3.7901426581098414E+04_dp
      fit_coef(9, 13) = -3.7090020172809200E+04_dp
      fit_coef(10, 13) = 2.3287606333954678E+04_dp
      fit_coef(11, 13) = -8.4789859745242975E+03_dp
      fit_coef(12, 13) = 1.3610885265396084E+03_dp
      fit_coef(0, 14) = 5.5945431735079278E+00_dp
      fit_coef(1, 14) = 2.0856787151109613E-01_dp
      fit_coef(2, 14) = 2.1367891445332752E+01_dp
      fit_coef(3, 14) = -1.5321402782127791E+02_dp
      fit_coef(4, 14) = 1.0771822980689246E+03_dp
      fit_coef(5, 14) = -5.1495462635748599E+03_dp
      fit_coef(6, 14) = 1.6370767484953010E+04_dp
      fit_coef(7, 14) = -3.5178430776828289E+04_dp
      fit_coef(8, 14) = 5.1385383840132439E+04_dp
      fit_coef(9, 14) = -5.0302167209417865E+04_dp
      fit_coef(10, 14) = 3.1590534978260897E+04_dp
      fit_coef(11, 14) = -1.1504033487227012E+04_dp
      fit_coef(12, 14) = 1.8469177824866308E+03_dp
      fit_coef(0, 15) = 1.8436235820019048E-05_dp
      fit_coef(1, 15) = 2.3107371865340239E-05_dp
      fit_coef(2, 15) = 2.5779879028853276E-03_dp
      fit_coef(3, 15) = -1.5220660379968103E-02_dp
      fit_coef(4, 15) = 1.4674238877197790E-01_dp
      fit_coef(5, 15) = -6.2720720593806745E-01_dp
      fit_coef(6, 15) = 1.9196496962860754E+00_dp
      fit_coef(7, 15) = -4.0563204420089054E+00_dp
      fit_coef(8, 15) = 5.8719543662268974E+00_dp
      fit_coef(9, 15) = -5.7179931070241752E+00_dp
      fit_coef(10, 15) = 3.5792099623773965E+00_dp
      fit_coef(11, 15) = -1.3005810138247331E+00_dp
      fit_coef(12, 15) = 2.0848698369596702E-01_dp
      fit_coef(0, 16) = 1.2124348449172285E-04_dp
      fit_coef(1, 16) = 8.3813269418456530E-05_dp
      fit_coef(2, 16) = 9.0897102638036447E-03_dp
      fit_coef(3, 16) = -5.7911668892308865E-02_dp
      fit_coef(4, 16) = 4.9294344453578842E-01_dp
      fit_coef(5, 16) = -2.1314596717186927E+00_dp
      fit_coef(6, 16) = 6.5094911565878411E+00_dp
      fit_coef(7, 16) = -1.3738226450588236E+01_dp
      fit_coef(8, 16) = 1.9897297045097488E+01_dp
      fit_coef(9, 16) = -1.9402396684702406E+01_dp
      fit_coef(10, 16) = 1.2166199138851475E+01_dp
      fit_coef(11, 16) = -4.4288923635919755E+00_dp
      fit_coef(12, 16) = 7.1121697870477651E-01_dp
      fit_coef(0, 17) = 4.9649968043970563E-04_dp
      fit_coef(1, 17) = 2.4013320731623524E-04_dp
      fit_coef(2, 17) = 2.5742527096276644E-02_dp
      fit_coef(3, 17) = -1.7243873414770514E-01_dp
      fit_coef(4, 17) = 1.3895762899290705E+00_dp
      fit_coef(5, 17) = -6.3761453454497428E+00_dp
      fit_coef(6, 17) = 1.9936814180089332E+01_dp
      fit_coef(7, 17) = -4.2440835139836899E+01_dp
      fit_coef(8, 17) = 6.1588254511673313E+01_dp
      fit_coef(9, 17) = -5.9991544128378500E+01_dp
      fit_coef(10, 17) = 3.7528872717238954E+01_dp
      fit_coef(11, 17) = -1.3623706398668176E+01_dp
      fit_coef(12, 17) = 2.1816169197074027E+00_dp
      fit_coef(0, 18) = 1.6143519027172607E-03_dp
      fit_coef(1, 18) = 5.8859169571603552E-04_dp
      fit_coef(2, 18) = 6.2478735725773553E-02_dp
      fit_coef(3, 18) = -4.2711342899465948E-01_dp
      fit_coef(4, 18) = 3.3147135916470036E+00_dp
      fit_coef(5, 18) = -1.5401373294982442E+01_dp
      fit_coef(6, 18) = 4.7954953793067048E+01_dp
      fit_coef(7, 18) = -1.0153978302120807E+02_dp
      fit_coef(8, 18) = 1.4682909652330889E+02_dp
      fit_coef(9, 18) = -1.4276036864090156E+02_dp
      fit_coef(10, 18) = 8.9245840751766565E+01_dp
      fit_coef(11, 18) = -3.2398300584951542E+01_dp
      fit_coef(12, 18) = 5.1900908130178340E+00_dp
      fit_coef(0, 19) = 4.5454454347221614E-03_dp
      fit_coef(1, 19) = 1.2919916988711280E-03_dp
      fit_coef(2, 19) = 1.3597866025632654E-01_dp
      fit_coef(3, 19) = -9.4213191119816730E-01_dp
      fit_coef(4, 19) = 7.1313517958816419E+00_dp
      fit_coef(5, 19) = -3.3458248656849491E+01_dp
      fit_coef(6, 19) = 1.0462115349553649E+02_dp
      fit_coef(7, 19) = -2.2230013365840125E+02_dp
      fit_coef(8, 19) = 3.2237995602287663E+02_dp
      fit_coef(9, 19) = -3.1411722628668201E+02_dp
      fit_coef(10, 19) = 1.9665741705687799E+02_dp
      fit_coef(11, 19) = -7.1459843200825901E+01_dp
      fit_coef(12, 19) = 1.1454531272981891E+01_dp
      fit_coef(0, 20) = 1.1569593836045351E-02_dp
      fit_coef(1, 20) = 2.6018407470241437E-03_dp
      fit_coef(2, 20) = 2.7208902236045929E-01_dp
      fit_coef(3, 20) = -1.9039272645212668E+00_dp
      fit_coef(4, 20) = 1.4140811044678967E+01_dp
      fit_coef(5, 20) = -6.6772929853793315E+01_dp
      fit_coef(6, 20) = 2.0958553375037602E+02_dp
      fit_coef(7, 20) = -4.4639362838727067E+02_dp
      fit_coef(8, 20) = 6.4825457307516922E+02_dp
      fit_coef(9, 20) = -6.3211066495078410E+02_dp
      fit_coef(10, 20) = 3.9589944611561242E+02_dp
      fit_coef(11, 20) = -1.4389110171301371E+02_dp
      fit_coef(12, 20) = 2.3067990020945253E+01_dp
      fit_coef(0, 21) = 2.7295736154530334E-02_dp
      fit_coef(1, 21) = 4.8782968254850083E-03_dp
      fit_coef(2, 21) = 5.0739031138062951E-01_dp
      fit_coef(3, 21) = -3.5769970240701987E+00_dp
      fit_coef(4, 21) = 2.6164736431475280E+01_dp
      fit_coef(5, 21) = -1.2410237515226034E+02_dp
      fit_coef(6, 21) = 3.9082320961974159E+02_dp
      fit_coef(7, 21) = -8.3417851759708878E+02_dp
      fit_coef(8, 21) = 1.2129937424978116E+03_dp
      fit_coef(9, 21) = -1.1837877342560605E+03_dp
      fit_coef(10, 21) = 7.4185106985661810E+02_dp
      fit_coef(11, 21) = -2.6974056955563321E+02_dp
      fit_coef(12, 21) = 4.3256996209012456E+01_dp
      fit_coef(0, 22) = 6.0660781563425317E-02_dp
      fit_coef(1, 22) = 8.5794040217948859E-03_dp
      fit_coef(2, 22) = 8.8821388130041634E-01_dp
      fit_coef(3, 22) = -6.2985610532830956E+00_dp
      fit_coef(4, 22) = 4.5492645494685874E+01_dp
      fit_coef(5, 22) = -2.1647261584955578E+02_dp
      fit_coef(6, 22) = 6.8370044474800454E+02_dp
      fit_coef(7, 22) = -1.4621335320610413E+03_dp
      fit_coef(8, 22) = 2.1287754889346256E+03_dp
      fit_coef(9, 22) = -2.0792410286940094E+03_dp
      fit_coef(10, 22) = 1.3037501245395274E+03_dp
      fit_coef(11, 22) = -4.7423981275141949E+02_dp
      fit_coef(12, 22) = 7.6073607209164223E+01_dp
      fit_coef(0, 23) = 1.2841217227424742E-01_dp
      fit_coef(1, 23) = 1.4188057337444604E-02_dp
      fit_coef(2, 23) = 1.4629508408620973E+00_dp
      fit_coef(3, 23) = -1.0423327628190115E+01_dp
      fit_coef(4, 23) = 7.4479996989370790E+01_dp
      fit_coef(5, 23) = -3.5524319107213654E+02_dp
      fit_coef(6, 23) = 1.1248481685847394E+03_dp
      fit_coef(7, 23) = -2.4098075097537289E+03_dp
      fit_coef(8, 23) = 3.5126267297370300E+03_dp
      fit_coef(9, 23) = -3.4335802646233810E+03_dp
      fit_coef(10, 23) = 2.1541281893481446E+03_dp
      fit_coef(11, 23) = -7.8386608937552410E+02_dp
      fit_coef(12, 23) = 1.2577649211754711E+02_dp
      fit_coef(0, 24) = 2.6110218583026157E-01_dp
      fit_coef(1, 24) = 2.2013210715212957E-02_dp
      fit_coef(2, 24) = 2.2618021820392888E+00_dp
      fit_coef(3, 24) = -1.6177325764415730E+01_dp
      fit_coef(4, 24) = 1.1453533241330872E+02_dp
      fit_coef(5, 24) = -5.4723833721116853E+02_dp
      fit_coef(6, 24) = 1.7366429607985401E+03_dp
      fit_coef(7, 24) = -3.7264526069498952E+03_dp
      fit_coef(8, 24) = 5.4376949325389724E+03_dp
      fit_coef(9, 24) = -5.3192362305534161E+03_dp
      fit_coef(10, 24) = 3.3388433630606642E+03_dp
      fit_coef(11, 24) = -1.2154181459321210E+03_dp
      fit_coef(12, 24) = 1.9507456139845061E+02_dp
      fit_coef(0, 25) = 5.1367200823804149E-01_dp
      fit_coef(1, 25) = 3.1811354680997252E-02_dp
      fit_coef(2, 25) = 3.2585980335370284E+00_dp
      fit_coef(3, 25) = -2.3379379060808798E+01_dp
      fit_coef(4, 25) = 1.6424152452481010E+02_dp
      fit_coef(5, 25) = -7.8569152121234890E+02_dp
      fit_coef(6, 25) = 2.4981080171697954E+03_dp
      fit_coef(7, 25) = -5.3680206518529076E+03_dp
      fit_coef(8, 25) = 7.8407527495523063E+03_dp
      fit_coef(9, 25) = -7.6750999224540392E+03_dp
      fit_coef(10, 25) = 4.8198767192823361E+03_dp
      fit_coef(11, 25) = -1.7551476130824467E+03_dp
      fit_coef(12, 25) = 2.8177229346472944E+02_dp
      fit_coef(0, 26) = 9.8707480111071055E-01_dp
      fit_coef(1, 26) = 4.2323366844794610E-02_dp
      fit_coef(2, 26) = 4.3248620683488594E+00_dp
      fit_coef(3, 26) = -3.1101768723183273E+01_dp
      fit_coef(4, 26) = 2.1715876476583264E+02_dp
      fit_coef(5, 26) = -1.0396442548695900E+03_dp
      fit_coef(6, 26) = 3.3104927069027003E+03_dp
      fit_coef(7, 26) = -7.1219751261209803E+03_dp
      fit_coef(8, 26) = 1.0411115847803250E+04_dp
      fit_coef(9, 26) = -1.0196927796670321E+04_dp
      fit_coef(10, 26) = 6.4061259733442739E+03_dp
      fit_coef(11, 26) = -2.3334570806260454E+03_dp
      fit_coef(12, 26) = 3.7469580771614551E+02_dp
      fit_coef(0, 27) = 1.8940435475450061E+00_dp
      fit_coef(1, 27) = 5.1360926179740848E-02_dp
      fit_coef(2, 27) = 5.2404454262865769E+00_dp
      fit_coef(3, 27) = -3.7737331118803013E+01_dp
      fit_coef(4, 27) = 2.6250426447967351E+02_dp
      fit_coef(5, 27) = -1.2571969272125693E+03_dp
      fit_coef(6, 27) = 4.0068677681386607E+03_dp
      fit_coef(7, 27) = -8.6264349397777878E+03_dp
      fit_coef(8, 27) = 1.2616976771304368E+04_dp
      fit_coef(9, 27) = -1.2361959020546034E+04_dp
      fit_coef(10, 27) = 7.7683322759890580E+03_dp
      fit_coef(11, 27) = -2.8301914681580724E+03_dp
      fit_coef(12, 27) = 4.5452476900845880E+02_dp
      fit_coef(0, 28) = 3.9546758238329791E+00_dp
      fit_coef(1, 28) = 5.8523148761978797E-02_dp
      fit_coef(2, 28) = 5.9702778386920183E+00_dp
      fit_coef(3, 28) = -4.2997287866308362E+01_dp
      fit_coef(4, 28) = 2.9898597699426614E+02_dp
      fit_coef(5, 28) = -1.4318903472750299E+03_dp
      fit_coef(6, 28) = 4.5640259927402913E+03_dp
      fit_coef(7, 28) = -9.8267478597299942E+03_dp
      fit_coef(8, 28) = 1.4373446674375520E+04_dp
      fit_coef(9, 28) = -1.4083574897310356E+04_dp
      fit_coef(10, 28) = 8.8505067497423279E+03_dp
      fit_coef(11, 28) = -3.2245357392636670E+03_dp
      fit_coef(12, 28) = 5.1786594922938832E+02_dp

   END SUBROUTINE get_fit_coef_k14_lo

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k15_sh(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 2.0000000000000000E+01_dp
      U_b = 1.0000000000000000E+02_dp
      fit_coef(0, 1) = 3.7767636658328917E-03_dp
      fit_coef(1, 1) = 8.1211740094389931E-03_dp
      fit_coef(2, 1) = 8.4713080613072721E-03_dp
      fit_coef(3, 1) = 4.9722625039685525E-03_dp
      fit_coef(4, 1) = 1.2149725765861214E-03_dp
      fit_coef(5, 1) = -4.4528795735146096E-04_dp
      fit_coef(6, 1) = -4.1397080234777437E-04_dp
      fit_coef(7, 1) = -3.3426994813408613E-05_dp
      fit_coef(8, 1) = 6.8763616446013435E-05_dp
      fit_coef(9, 1) = 2.6807372741525578E-05_dp
      fit_coef(10, 1) = -2.8714180956136556E-05_dp
      fit_coef(11, 1) = 9.6272015928166124E-06_dp
      fit_coef(12, 1) = -1.2744413786197213E-06_dp
      fit_coef(0, 2) = 2.0284783343273730E-02_dp
      fit_coef(1, 2) = 4.3187845019499065E-02_dp
      fit_coef(2, 2) = 4.4768123513328789E-02_dp
      fit_coef(3, 2) = 2.6126720474645704E-02_dp
      fit_coef(4, 2) = 6.3201115983603541E-03_dp
      fit_coef(5, 2) = -2.3578759036246063E-03_dp
      fit_coef(6, 2) = -2.1606459505341194E-03_dp
      fit_coef(7, 2) = -1.6580374853951958E-04_dp
      fit_coef(8, 2) = 3.6074887455186000E-04_dp
      fit_coef(9, 2) = 1.3781750573407730E-04_dp
      fit_coef(10, 2) = -1.4977975033955343E-04_dp
      fit_coef(11, 2) = 5.0490744097337301E-05_dp
      fit_coef(12, 2) = -6.7504814416040953E-06_dp
      fit_coef(0, 3) = 5.1673260780627950E-02_dp
      fit_coef(1, 3) = 1.0789295508104972E-01_dp
      fit_coef(2, 3) = 1.1057351883667553E-01_dp
      fit_coef(3, 3) = 6.3878269616968825E-02_dp
      fit_coef(4, 3) = 1.5180871067702629E-02_dp
      fit_coef(5, 3) = -5.8375073829049018E-03_dp
      fit_coef(6, 3) = -5.2188107945168550E-03_dp
      fit_coef(7, 3) = -3.6420812718932399E-04_dp
      fit_coef(8, 3) = 8.7905419776698594E-04_dp
      fit_coef(9, 3) = 3.2415893028994188E-04_dp
      fit_coef(10, 3) = -3.6139234804234916E-04_dp
      fit_coef(11, 3) = 1.2287995757801328E-04_dp
      fit_coef(12, 3) = -1.6675660997752087E-05_dp
      fit_coef(0, 4) = 1.0134621267461434E-01_dp
      fit_coef(1, 4) = 2.0498960034213551E-01_dp
      fit_coef(2, 4) = 2.0657812668218858E-01_dp
      fit_coef(3, 4) = 1.1767379558549464E-01_dp
      fit_coef(4, 4) = 2.7288178413390367E-02_dp
      fit_coef(5, 4) = -1.0924148745488705E-02_dp
      fit_coef(6, 4) = -9.4432740413890777E-03_dp
      fit_coef(7, 4) = -5.6769752225927254E-04_dp
      fit_coef(8, 4) = 1.6082317938650511E-03_dp
      fit_coef(9, 4) = 5.6496352365318649E-04_dp
      fit_coef(10, 4) = -6.5204939503955980E-04_dp
      fit_coef(11, 4) = 2.2378537926777479E-04_dp
      fit_coef(12, 4) = -3.0821862755447693E-05_dp
      fit_coef(0, 5) = 1.7540646948915822E-01_dp
      fit_coef(1, 5) = 3.3841107332002812E-01_dp
      fit_coef(2, 5) = 3.3317753537976630E-01_dp
      fit_coef(3, 5) = 1.8664213256865744E-01_dp
      fit_coef(4, 5) = 4.2122973931643735E-02_dp
      fit_coef(5, 5) = -1.7653496250856538E-02_dp
      fit_coef(6, 5) = -1.4677820339754412E-02_dp
      fit_coef(7, 5) = -6.7172992100736459E-04_dp
      fit_coef(8, 5) = 2.5463509524145020E-03_dp
      fit_coef(9, 5) = 7.9689495762103719E-04_dp
      fit_coef(10, 5) = -9.9176850710568391E-04_dp
      fit_coef(11, 5) = 3.5040533908774847E-04_dp
      fit_coef(12, 5) = -5.0509200990651491E-05_dp
      fit_coef(0, 6) = 2.8403858696731338E-01_dp
      fit_coef(1, 6) = 5.1354379131383487E-01_dp
      fit_coef(2, 6) = 4.8963667336603639E-01_dp
      fit_coef(3, 6) = 2.6912774693427816E-01_dp
      fit_coef(4, 6) = 5.9724553513996380E-02_dp
      fit_coef(5, 6) = -2.5703789609353513E-02_dp
      fit_coef(6, 6) = -2.1114838016090463E-02_dp
      fit_coef(7, 6) = -7.0228514229756346E-04_dp
      fit_coef(8, 6) = 3.8581969229036879E-03_dp
      fit_coef(9, 6) = 1.2194767346868753E-03_dp
      fit_coef(10, 6) = -1.8021363838819021E-03_dp
      fit_coef(11, 6) = 7.5507648660773613E-04_dp
      fit_coef(12, 6) = -1.3075406318307264E-04_dp
      fit_coef(0, 7) = 4.4325614880929831E-01_dp
      fit_coef(1, 7) = 7.3781666079304498E-01_dp
      fit_coef(2, 7) = 6.7367679967887084E-01_dp
      fit_coef(3, 7) = 3.6132751402347024E-01_dp
      fit_coef(4, 7) = 8.0499595178773156E-02_dp
      fit_coef(5, 7) = -3.3029532312549834E-02_dp
      fit_coef(6, 7) = -2.8790475820923794E-02_dp
      fit_coef(7, 7) = -1.7795611862018949E-03_dp
      fit_coef(8, 7) = 5.1379466412685825E-03_dp
      fit_coef(9, 7) = 3.1353774828616044E-03_dp
      fit_coef(10, 7) = -3.3341189829340133E-03_dp
      fit_coef(11, 7) = 9.5780853222126921E-04_dp
      fit_coef(12, 7) = -4.7564006414347488E-05_dp
      fit_coef(0, 8) = 6.7707705919780792E-01_dp
      fit_coef(1, 8) = 1.0208946341910239E+00_dp
      fit_coef(2, 8) = 8.8225292727415439E-01_dp
      fit_coef(3, 8) = 4.5683990983568840E-01_dp
      fit_coef(4, 8) = 1.0315634645104643E-01_dp
      fit_coef(5, 8) = -3.6302702410040397E-02_dp
      fit_coef(6, 8) = -3.5178064788216744E-02_dp
      fit_coef(7, 8) = -5.4614672692391321E-03_dp
      fit_coef(8, 8) = 5.9947422484209678E-03_dp
      fit_coef(9, 8) = 2.7277018733018010E-03_dp
      fit_coef(10, 8) = 9.2005661112863006E-04_dp
      fit_coef(11, 8) = -2.7491424308858183E-03_dp
      fit_coef(12, 8) = 9.3417034494363172E-04_dp
      fit_coef(0, 9) = 1.0205594171746224E+00_dp
      fit_coef(1, 9) = 1.3737393203947723E+00_dp
      fit_coef(2, 9) = 1.1123746840237188E+00_dp
      fit_coef(3, 9) = 5.4861957447712506E-01_dp
      fit_coef(4, 9) = 1.2420055508662378E-01_dp
      fit_coef(5, 9) = -3.4071192988448372E-02_dp
      fit_coef(6, 9) = -3.6752879111842843E-02_dp
      fit_coef(7, 9) = -9.8874781786285233E-03_dp
      fit_coef(8, 9) = 5.0195565707678762E-03_dp
      fit_coef(9, 9) = -7.6508018734261539E-04_dp
      fit_coef(10, 9) = 7.1843501967062814E-03_dp
      fit_coef(11, 9) = -4.9459397204347057E-03_dp
      fit_coef(12, 9) = 8.9782856131910235E-04_dp
      fit_coef(0, 10) = 1.5244258156390766E+00_dp
      fit_coef(1, 10) = 1.8066993429217002E+00_dp
      fit_coef(2, 10) = 1.3609242838483797E+00_dp
      fit_coef(3, 10) = 6.3115404377443018E-01_dp
      fit_coef(4, 10) = 1.4030025341356947E-01_dp
      fit_coef(5, 10) = -2.7329038046514077E-02_dp
      fit_coef(6, 10) = -3.2676873494217773E-02_dp
      fit_coef(7, 10) = -1.1415330583361451E-02_dp
      fit_coef(8, 10) = -4.9009191168826780E-06_dp
      fit_coef(9, 10) = 1.3306667731994193E-04_dp
      fit_coef(10, 10) = 3.7437558621727469E-03_dp
      fit_coef(11, 10) = 4.8589108656243562E-04_dp
      fit_coef(12, 10) = -1.0469727509729387E-03_dp
      fit_coef(0, 11) = 2.2624235303155098E+00_dp
      fit_coef(1, 11) = 2.3271837597685869E+00_dp
      fit_coef(2, 11) = 1.6242573707008741E+00_dp
      fit_coef(3, 11) = 7.0131249760739722E-01_dp
      fit_coef(4, 11) = 1.5012768567411283E-01_dp
      fit_coef(5, 11) = -1.7685800296799118E-02_dp
      fit_coef(6, 11) = -2.4133853082992485E-02_dp
      fit_coef(7, 11) = -1.0124353385573369E-02_dp
      fit_coef(8, 11) = -5.4053355656654414E-03_dp
      fit_coef(9, 11) = 2.0601323692594885E-03_dp
      fit_coef(10, 11) = -1.9168010184212245E-03_dp
      fit_coef(11, 11) = 5.2635221561462978E-03_dp
      fit_coef(12, 11) = -2.0650943061245367E-03_dp
      fit_coef(0, 12) = 3.3440185138398908E+00_dp
      fit_coef(1, 12) = 2.9374869303015658E+00_dp
      fit_coef(2, 12) = 1.8986970103463769E+00_dp
      fit_coef(3, 12) = 7.5852460527436116E-01_dp
      fit_coef(4, 12) = 1.5431087315065109E-01_dp
      fit_coef(5, 12) = -6.2637714258166168E-03_dp
      fit_coef(6, 12) = -1.2867661430609648E-02_dp
      fit_coef(7, 12) = -7.7297513116621266E-03_dp
      fit_coef(8, 12) = -7.9684475195776500E-03_dp
      fit_coef(9, 12) = 7.0191640613392089E-04_dp
      fit_coef(10, 12) = -2.3205524009468916E-03_dp
      fit_coef(11, 12) = 4.7115703272364142E-03_dp
      fit_coef(12, 12) = -1.3939780575372204E-03_dp
      fit_coef(0, 13) = 4.9400406076482417E+00_dp
      fit_coef(1, 13) = 3.6338865240810336E+00_dp
      fit_coef(2, 13) = 2.1823364183861980E+00_dp
      fit_coef(3, 13) = 8.0486953703551811E-01_dp
      fit_coef(4, 13) = 1.5444706443796333E-01_dp
      fit_coef(5, 13) = 6.2245841342081663E-03_dp
      fit_coef(6, 13) = -4.5752093352805732E-04_dp
      fit_coef(7, 13) = -4.9804550111796943E-03_dp
      fit_coef(8, 13) = -8.4778512610873564E-03_dp
      fit_coef(9, 13) = -2.0214510332599842E-03_dp
      fit_coef(10, 13) = 2.4374313233334989E-04_dp
      fit_coef(11, 13) = 1.9195383981977568E-03_dp
      fit_coef(12, 13) = -3.7228313108121918E-04_dp
      fit_coef(0, 14) = 7.3492824088289943E+00_dp
      fit_coef(1, 14) = 4.4099530097240827E+00_dp
      fit_coef(2, 14) = 2.4783620449758916E+00_dp
      fit_coef(3, 14) = 8.4513384310546469E-01_dp
      fit_coef(4, 14) = 1.5211923704319483E-01_dp
      fit_coef(5, 14) = 1.9150817795865041E-02_dp
      fit_coef(6, 14) = 1.2175087669328011E-02_dp
      fit_coef(7, 14) = -1.9876602775734512E-03_dp
      fit_coef(8, 14) = -8.3964504059626105E-03_dp
      fit_coef(9, 14) = -3.7517484520995536E-03_dp
      fit_coef(10, 14) = 2.5098290732908429E-03_dp
      fit_coef(11, 14) = -2.2174814827198228E-04_dp
      fit_coef(12, 14) = 1.7762122632969973E-04_dp
      fit_coef(0, 15) = 1.1260372250288922E+01_dp
      fit_coef(1, 15) = 5.2761367256188780E+00_dp
      fit_coef(2, 15) = 2.8037057941841899E+00_dp
      fit_coef(3, 15) = 8.8780102116211801E-01_dp
      fit_coef(4, 15) = 1.4843982592096647E-01_dp
      fit_coef(5, 15) = 3.2027815482417170E-02_dp
      fit_coef(6, 15) = 2.5031908188376327E-02_dp
      fit_coef(7, 15) = 1.3905318224525349E-03_dp
      fit_coef(8, 15) = -8.2768851858847706E-03_dp
      fit_coef(9, 15) = -4.1010656916587729E-03_dp
      fit_coef(10, 15) = 3.7595505954445777E-03_dp
      fit_coef(11, 15) = -1.2076253125456195E-03_dp
      fit_coef(12, 15) = 2.7539917666371817E-04_dp
      fit_coef(0, 16) = 9.7353035224829500E-03_dp
      fit_coef(1, 16) = 2.0886976749800733E-02_dp
      fit_coef(2, 16) = 2.1755665531631045E-02_dp
      fit_coef(3, 16) = 1.2752382400370278E-02_dp
      fit_coef(4, 16) = 3.1086335160659716E-03_dp
      fit_coef(5, 16) = -1.1441703553915466E-03_dp
      fit_coef(6, 16) = -1.0600438281376386E-03_dp
      fit_coef(7, 16) = -8.4585212227881147E-05_dp
      fit_coef(8, 16) = 1.7629742631278603E-04_dp
      fit_coef(9, 16) = 6.8399637396386302E-05_dp
      fit_coef(10, 16) = -7.3517685522867944E-05_dp
      fit_coef(11, 16) = 2.4681600354643785E-05_dp
      fit_coef(12, 16) = -3.2754157510708014E-06_dp
      fit_coef(0, 17) = 2.3550935618358620E-02_dp
      fit_coef(1, 17) = 4.9522562600574506E-02_dp
      fit_coef(2, 17) = 5.0940261314895138E-02_dp
      fit_coef(3, 17) = 2.9520151355165270E-02_dp
      fit_coef(4, 17) = 7.0526431743143485E-03_dp
      fit_coef(5, 17) = -2.6886627324933282E-03_dp
      fit_coef(6, 17) = -2.4209379054065576E-03_dp
      fit_coef(7, 17) = -1.7381016208850244E-04_dp
      fit_coef(8, 17) = 4.0675731507373121E-04_dp
      fit_coef(9, 17) = 1.5152633434160025E-04_dp
      fit_coef(10, 17) = -1.6769671036439022E-04_dp
      fit_coef(11, 17) = 5.6896776687533419E-05_dp
      fit_coef(12, 17) = -7.6947045683246862E-06_dp
      fit_coef(0, 18) = 3.9777578377675531E-02_dp
      fit_coef(1, 18) = 8.0346278051875858E-02_dp
      fit_coef(2, 18) = 8.0791839568255266E-02_dp
      fit_coef(3, 18) = 4.5890106592349850E-02_dp
      fit_coef(4, 18) = 1.0581783818106148E-02_dp
      fit_coef(5, 18) = -4.2772167663816868E-03_dp
      fit_coef(6, 18) = -3.6711065186785822E-03_dp
      fit_coef(7, 18) = -2.1295067951996251E-04_dp
      fit_coef(8, 18) = 6.2748074852499950E-04_dp
      fit_coef(9, 18) = 2.1773104607889315E-04_dp
      fit_coef(10, 18) = -2.5378512352868573E-04_dp
      fit_coef(11, 18) = 8.7481966065534113E-05_dp
      fit_coef(12, 18) = -1.2141070019913610E-05_dp
      fit_coef(0, 19) = 6.0562131081817058E-02_dp
      fit_coef(1, 19) = 1.1449873938476936E-01_dp
      fit_coef(2, 19) = 1.1129944008292268E-01_dp
      fit_coef(3, 19) = 6.1560590847477989E-02_dp
      fit_coef(4, 19) = 1.3534891528073901E-02_dp
      fit_coef(5, 19) = -5.9033543536541871E-03_dp
      fit_coef(6, 19) = -4.7450366895801120E-03_dp
      fit_coef(7, 19) = -1.7764449860279051E-04_dp
      fit_coef(8, 19) = 8.2604161016353668E-04_dp
      fit_coef(9, 19) = 2.5651846839713858E-04_dp
      fit_coef(10, 19) = -3.2185438255892302E-04_dp
      fit_coef(11, 19) = 1.1225233677891509E-04_dp
      fit_coef(12, 19) = -1.5831921058481165E-05_dp
      fit_coef(0, 20) = 8.9226696029307800E-02_dp
      fit_coef(1, 20) = 1.5322995807536874E-01_dp
      fit_coef(2, 20) = 1.4178721914749884E-01_dp
      fit_coef(3, 20) = 7.6128391563680162E-02_dp
      fit_coef(4, 20) = 1.6136643158648808E-02_dp
      fit_coef(5, 20) = -7.5330493292977262E-03_dp
      fit_coef(6, 20) = -5.7529238965828419E-03_dp
      fit_coef(7, 20) = -1.1707762217534482E-05_dp
      fit_coef(8, 20) = 1.0894518853190246E-03_dp
      fit_coef(9, 20) = 1.8904025635640227E-04_dp
      fit_coef(10, 20) = -3.8503303956792745E-04_dp
      fit_coef(11, 20) = 1.6892465284271444E-04_dp
      fit_coef(12, 20) = -3.1503784546561519E-05_dp
      fit_coef(0, 21) = 1.3067133728179103E-01_dp
      fit_coef(1, 21) = 1.9826805579875501E-01_dp
      fit_coef(2, 21) = 1.7073403778885513E-01_dp
      fit_coef(3, 21) = 8.8223081054500926E-02_dp
      fit_coef(4, 21) = 1.9182613501886863E-02_dp
      fit_coef(5, 21) = -8.2278999968844989E-03_dp
      fit_coef(6, 21) = -7.1938427008996124E-03_dp
      fit_coef(7, 21) = -2.3630795339020511E-04_dp
      fit_coef(8, 21) = 1.4678849246082021E-03_dp
      fit_coef(9, 21) = 1.0431379287022221E-03_dp
      fit_coef(10, 21) = -1.5357712204035895E-03_dp
      fit_coef(11, 21) = 7.1343365682177630E-04_dp
      fit_coef(12, 21) = -1.2573122208893727E-04_dp
      fit_coef(0, 22) = 1.9171789962238633E-01_dp
      fit_coef(1, 22) = 2.5190501212777222E-01_dp
      fit_coef(2, 22) = 1.9682158960117535E-01_dp
      fit_coef(3, 22) = 9.5068036046292548E-02_dp
      fit_coef(4, 22) = 2.2164921662634080E-02_dp
      fit_coef(5, 22) = -5.7919264320229974E-03_dp
      fit_coef(6, 22) = -7.7045178800915467E-03_dp
      fit_coef(7, 22) = -2.2724175760734976E-03_dp
      fit_coef(8, 22) = 1.0118193318505186E-03_dp
      fit_coef(9, 22) = 2.0169812523148983E-03_dp
      fit_coef(10, 22) = -9.2577594680840871E-06_dp
      fit_coef(11, 22) = -1.3014429778002868E-03_dp
      fit_coef(12, 22) = 5.2474898728580737E-04_dp
      fit_coef(0, 23) = 2.8169225019506716E-01_dp
      fit_coef(1, 23) = 3.1611118986493281E-01_dp
      fit_coef(2, 23) = 2.1983832131612513E-01_dp
      fit_coef(3, 23) = 9.4742330833316193E-02_dp
      fit_coef(4, 23) = 2.2513794210120210E-02_dp
      fit_coef(5, 23) = -5.0847893755973886E-04_dp
      fit_coef(6, 23) = -4.3926536784528831E-03_dp
      fit_coef(7, 23) = -4.7592264209054850E-03_dp
      fit_coef(8, 23) = 5.4373390930081153E-04_dp
      fit_coef(9, 23) = -3.2404033623214612E-03_dp
      fit_coef(10, 23) = 7.9533405554355376E-03_dp
      fit_coef(11, 23) = -5.0278147899489754E-03_dp
      fit_coef(12, 23) = 1.0211564000482632E-03_dp
      fit_coef(0, 24) = 4.1360870667704458E-01_dp
      fit_coef(1, 24) = 3.9134279925186166E-01_dp
      fit_coef(2, 24) = 2.3989520225411959E-01_dp
      fit_coef(3, 24) = 8.7887165884511398E-02_dp
      fit_coef(4, 24) = 1.8986346126332081E-02_dp
      fit_coef(5, 24) = 4.7658422103979536E-03_dp
      fit_coef(6, 24) = 1.3492788485409764E-03_dp
      fit_coef(7, 24) = -3.3357808918414787E-03_dp
      fit_coef(8, 24) = -3.1088669483680088E-03_dp
      fit_coef(9, 24) = -1.8511419844152608E-03_dp
      fit_coef(10, 24) = 1.7736202675366283E-03_dp
      fit_coef(11, 24) = 2.1114330344624466E-03_dp
      fit_coef(12, 24) = -1.3226303250553409E-03_dp
      fit_coef(0, 25) = 6.0621802446427508E-01_dp
      fit_coef(1, 25) = 4.7585064174951042E-01_dp
      fit_coef(2, 25) = 2.5658915757225498E-01_dp
      fit_coef(3, 25) = 7.6658354342588539E-02_dp
      fit_coef(4, 25) = 1.2995988704122139E-02_dp
      fit_coef(5, 25) = 8.4318408777111463E-03_dp
      fit_coef(6, 25) = 6.5820665925556696E-03_dp
      fit_coef(7, 25) = 1.8845523065198023E-04_dp
      fit_coef(8, 25) = -6.1170321098400040E-03_dp
      fit_coef(9, 25) = 2.6963328854067922E-03_dp
      fit_coef(10, 25) = -6.7464598603185540E-03_dp
      fit_coef(11, 25) = 6.8191427136023554E-03_dp
      fit_coef(12, 25) = -1.9275464800632919E-03_dp
      fit_coef(0, 26) = 8.8764155033206149E-01_dp
      fit_coef(1, 26) = 5.6552140990150657E-01_dp
      fit_coef(2, 26) = 2.6943393742267280E-01_dp
      fit_coef(3, 26) = 6.3561620296985116E-02_dp
      fit_coef(4, 26) = 6.7821546029915858E-03_dp
      fit_coef(5, 26) = 1.0671885911308030E-02_dp
      fit_coef(6, 26) = 1.0202617186318210E-02_dp
      fit_coef(7, 26) = 2.0623230180061023E-03_dp
      fit_coef(8, 26) = -4.0999080350069333E-03_dp
      fit_coef(9, 26) = 3.3757569259654344E-04_dp
      fit_coef(10, 26) = -3.2168362434274113E-03_dp
      fit_coef(11, 26) = 1.9561977675803606E-03_dp
      fit_coef(12, 26) = -1.0127854359087637E-05_dp
      fit_coef(0, 27) = 1.3031121163620227E+00_dp
      fit_coef(1, 27) = 6.5442627957957644E-01_dp
      fit_coef(2, 27) = 2.7907544789512745E-01_dp
      fit_coef(3, 27) = 5.1209194718773846E-02_dp
      fit_coef(4, 27) = 1.8680237447576146E-03_dp
      fit_coef(5, 27) = 1.2060225663826541E-02_dp
      fit_coef(6, 27) = 1.2057587702851754E-02_dp
      fit_coef(7, 27) = 2.6179284359982300E-03_dp
      fit_coef(8, 27) = -1.2404837715336242E-03_dp
      fit_coef(9, 27) = -2.5831650098316326E-03_dp
      fit_coef(10, 27) = 1.8086181493242913E-03_dp
      fit_coef(11, 27) = -2.3645920962560019E-03_dp
      fit_coef(12, 27) = 1.0746225263111922E-03_dp
      fit_coef(0, 28) = 1.9357875429827360E+00_dp
      fit_coef(1, 28) = 7.3707936760259829E-01_dp
      fit_coef(2, 28) = 2.8863411383043175E-01_dp
      fit_coef(3, 28) = 4.2272421989736957E-02_dp
      fit_coef(4, 28) = -1.3236760602961552E-03_dp
      fit_coef(5, 28) = 1.2811295424819810E-02_dp
      fit_coef(6, 28) = 1.2606618885406514E-02_dp
      fit_coef(7, 28) = 2.8686367951981208E-03_dp
      fit_coef(8, 28) = -4.2949161661710804E-05_dp
      fit_coef(9, 28) = -2.4420917937843715E-03_dp
      fit_coef(10, 28) = 2.7346339142362607E-03_dp
      fit_coef(11, 28) = -2.7192205830173727E-03_dp
      fit_coef(12, 28) = 8.3334329375208966E-04_dp
      fit_coef(0, 29) = 2.9813305217729247E+00_dp
      fit_coef(1, 29) = 8.1556707900662084E-01_dp
      fit_coef(2, 29) = 3.0574250813204679E-01_dp
      fit_coef(3, 29) = 3.9576605747462988E-02_dp
      fit_coef(4, 29) = -3.1516191139201481E-03_dp
      fit_coef(5, 29) = 1.2947548079823706E-02_dp
      fit_coef(6, 29) = 1.2648514261782698E-02_dp
      fit_coef(7, 29) = 3.1343334418898880E-03_dp
      fit_coef(8, 29) = 1.2305058520847782E-04_dp
      fit_coef(9, 29) = -9.8020686398801802E-04_dp
      fit_coef(10, 29) = 1.7095022610177765E-03_dp
      fit_coef(11, 29) = -1.5105150611297968E-03_dp
      fit_coef(12, 29) = 2.8195377606949853E-04_dp
      fit_coef(0, 30) = 5.1939564822259765E+00_dp
      fit_coef(1, 30) = 9.3452037903374208E-01_dp
      fit_coef(2, 30) = 3.5551447159259092E-01_dp
      fit_coef(3, 30) = 4.8826933534516080E-02_dp
      fit_coef(4, 30) = -4.0672239419310479E-03_dp
      fit_coef(5, 30) = 1.2876916535847991E-02_dp
      fit_coef(6, 30) = 1.3347608970308627E-02_dp
      fit_coef(7, 30) = 3.7439290651003427E-03_dp
      fit_coef(8, 30) = 1.3798569341874731E-04_dp
      fit_coef(9, 30) = 1.7467286573212931E-04_dp
      fit_coef(10, 30) = 9.2180740758796703E-04_dp
      fit_coef(11, 30) = -5.6981792340525463E-04_dp
      fit_coef(12, 30) = -3.5378278861525159E-05_dp

   END SUBROUTINE get_fit_coef_k15_sh

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k15_mi(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 1.0000000000000000E+02_dp
      U_b = 1.1738988511192356E+03_dp
      fit_coef(0, 1) = 5.0212735786207190E-04_dp
      fit_coef(1, 1) = 1.4715598436560518E-03_dp
      fit_coef(2, 1) = 2.1503966610512443E-03_dp
      fit_coef(3, 1) = 1.8763892029773810E-03_dp
      fit_coef(4, 1) = 8.9509317267188646E-04_dp
      fit_coef(5, 1) = 4.4252413515572752E-05_dp
      fit_coef(6, 1) = -1.9929035057866966E-04_dp
      fit_coef(7, 1) = -5.0485804346307633E-05_dp
      fit_coef(8, 1) = 4.3968375574232470E-05_dp
      fit_coef(9, 1) = 5.5154425679931483E-05_dp
      fit_coef(10, 1) = -7.3825809523082769E-05_dp
      fit_coef(11, 1) = 3.4418190236750298E-05_dp
      fit_coef(12, 1) = -6.1654048128061428E-06_dp
      fit_coef(0, 2) = 2.7813466099976822E-03_dp
      fit_coef(1, 2) = 7.9639349588492489E-03_dp
      fit_coef(2, 2) = 1.1498418447714263E-02_dp
      fit_coef(3, 2) = 9.9309054844448665E-03_dp
      fit_coef(4, 2) = 4.6793464572880547E-03_dp
      fit_coef(5, 2) = 2.0134378481332076E-04_dp
      fit_coef(6, 2) = -1.0461830621068451E-03_dp
      fit_coef(7, 2) = -2.5547396359220325E-04_dp
      fit_coef(8, 2) = 2.3777699827707836E-04_dp
      fit_coef(9, 2) = 2.7502067371517350E-04_dp
      fit_coef(10, 2) = -3.7882475853089022E-04_dp
      fit_coef(11, 2) = 1.7838098098998129E-04_dp
      fit_coef(12, 2) = -3.2173474626321743E-05_dp
      fit_coef(0, 3) = 7.5356589508119715E-03_dp
      fit_coef(1, 3) = 2.0547011268364480E-02_dp
      fit_coef(2, 3) = 2.8986632022747129E-02_dp
      fit_coef(3, 3) = 2.4585742824888547E-02_dp
      fit_coef(4, 3) = 1.1332324461894762E-02_dp
      fit_coef(5, 3) = 3.5501376638767630E-04_dp
      fit_coef(6, 3) = -2.5488497267062661E-03_dp
      fit_coef(7, 3) = -5.7872409900516482E-04_dp
      fit_coef(8, 3) = 6.0733088515269604E-04_dp
      fit_coef(9, 3) = 6.0352400231591973E-04_dp
      fit_coef(10, 3) = -8.7889897017369044E-04_dp
      fit_coef(11, 3) = 4.2107155921709870E-04_dp
      fit_coef(12, 3) = -7.6791853388456923E-05_dp
      fit_coef(0, 4) = 1.6293654129500710E-02_dp
      fit_coef(1, 4) = 4.1034183058244827E-02_dp
      fit_coef(2, 4) = 5.5669340450143955E-02_dp
      fit_coef(3, 4) = 4.5958232995003026E-02_dp
      fit_coef(4, 4) = 2.0631055086044563E-02_dp
      fit_coef(5, 4) = 3.6554527705558160E-04_dp
      fit_coef(6, 4) = -4.7143452832445311E-03_dp
      fit_coef(7, 4) = -9.9264691382352382E-04_dp
      fit_coef(8, 4) = 1.2322009627532241E-03_dp
      fit_coef(9, 4) = 9.8185260435618648E-04_dp
      fit_coef(10, 4) = -1.5959947544011608E-03_dp
      fit_coef(11, 4) = 7.9825857947420098E-04_dp
      fit_coef(12, 4) = -1.5065823877361689E-04_dp
      fit_coef(0, 5) = 3.2116880043710744E-02_dp
      fit_coef(1, 5) = 7.2781942537050448E-02_dp
      fit_coef(2, 5) = 9.3125631841889350E-02_dp
      fit_coef(3, 5) = 7.3591074240215937E-02_dp
      fit_coef(4, 5) = 3.1935152140736278E-02_dp
      fit_coef(5, 5) = 4.5831309250835472E-04_dp
      fit_coef(6, 5) = -7.2579368033050861E-03_dp
      fit_coef(7, 5) = -1.5242456005729315E-03_dp
      fit_coef(8, 5) = 1.3880778676825922E-03_dp
      fit_coef(9, 5) = 2.2376670079413724E-03_dp
      fit_coef(10, 5) = -2.8229103749250520E-03_dp
      fit_coef(11, 5) = 1.2683701333535493E-03_dp
      fit_coef(12, 5) = -2.1655233738863679E-04_dp
      fit_coef(0, 6) = 6.0477196624071472E-02_dp
      fit_coef(1, 6) = 1.2125095461882505E-01_dp
      fit_coef(2, 6) = 1.4404594415379110E-01_dp
      fit_coef(3, 6) = 1.0653676582778267E-01_dp
      fit_coef(4, 6) = 4.3372431118558187E-02_dp
      fit_coef(5, 6) = 4.1288606887333705E-04_dp
      fit_coef(6, 6) = -8.8579280488838095E-03_dp
      fit_coef(7, 6) = -2.2179852026296800E-03_dp
      fit_coef(8, 6) = 2.0958278133874500E-03_dp
      fit_coef(9, 6) = 6.9556540802001902E-04_dp
      fit_coef(10, 6) = -7.7943766158600528E-04_dp
      fit_coef(11, 6) = 1.1446740869183900E-04_dp
      fit_coef(12, 6) = 2.7632689805845764E-05_dp
      fit_coef(0, 7) = 1.1064399319041990E-01_dp
      fit_coef(1, 7) = 1.9418438863145457E-01_dp
      fit_coef(2, 7) = 2.1228094292888233E-01_dp
      fit_coef(3, 7) = 1.4399775783971039E-01_dp
      fit_coef(4, 7) = 5.2541209241390302E-02_dp
      fit_coef(5, 7) = -9.2537655514377752E-04_dp
      fit_coef(6, 7) = -9.0102109734644890E-03_dp
      fit_coef(7, 7) = -1.8871157279751268E-03_dp
      fit_coef(8, 7) = 3.1309099680042760E-03_dp
      fit_coef(9, 7) = -2.5004066194694607E-03_dp
      fit_coef(10, 7) = 1.5656511931140620E-03_dp
      fit_coef(11, 7) = -4.6135496029970502E-04_dp
      fit_coef(12, 7) = 3.2764384769675960E-05_dp
      fit_coef(0, 8) = 1.9798263302865093E-01_dp
      fit_coef(1, 8) = 3.0162627763842398E-01_dp
      fit_coef(2, 8) = 3.0217808443088900E-01_dp
      fit_coef(3, 8) = 1.8500065299157431E-01_dp
      fit_coef(4, 8) = 5.7340548130963523E-02_dp
      fit_coef(5, 8) = -4.3765501944754721E-03_dp
      fit_coef(6, 8) = -7.8869300696153017E-03_dp
      fit_coef(7, 8) = 1.3835188624175719E-04_dp
      fit_coef(8, 8) = 3.3750918634805581E-03_dp
      fit_coef(9, 8) = -5.0244120026273721E-03_dp
      fit_coef(10, 8) = 2.8097352620888730E-03_dp
      fit_coef(11, 8) = -4.8749664373366714E-04_dp
      fit_coef(12, 8) = -3.4446077687337908E-05_dp
      fit_coef(0, 9) = 3.4762598832467834E-01_dp
      fit_coef(1, 9) = 4.5573634191690915E-01_dp
      fit_coef(2, 9) = 4.1780164140817255E-01_dp
      fit_coef(3, 9) = 2.2768957404533277E-01_dp
      fit_coef(4, 9) = 5.5879258865385507E-02_dp
      fit_coef(5, 9) = -9.9081094692702303E-03_dp
      fit_coef(6, 9) = -5.0068528611206149E-03_dp
      fit_coef(7, 9) = 3.0160271964011468E-03_dp
      fit_coef(8, 9) = 3.6222265118699710E-03_dp
      fit_coef(9, 9) = -8.8397638670103182E-03_dp
      fit_coef(10, 9) = 5.9259674015836239E-03_dp
      fit_coef(11, 9) = -1.7478385304278991E-03_dp
      fit_coef(12, 9) = 1.9988838059233530E-04_dp
      fit_coef(0, 10) = 6.0023140973611755E-01_dp
      fit_coef(1, 10) = 6.6999181383267858E-01_dp
      fit_coef(2, 10) = 5.6210771610729648E-01_dp
      fit_coef(3, 10) = 2.6900784986520249E-01_dp
      fit_coef(4, 10) = 4.6435418682624782E-02_dp
      fit_coef(5, 10) = -1.6642885693320342E-02_dp
      fit_coef(6, 10) = 3.8014327565738168E-04_dp
      fit_coef(7, 10) = 6.2610253146593775E-03_dp
      fit_coef(8, 10) = 3.2146802606417453E-03_dp
      fit_coef(9, 10) = -1.3711754453203115E-02_dp
      fit_coef(10, 10) = 1.1099834233825179E-02_dp
      fit_coef(11, 10) = -4.2111321446896338E-03_dp
      fit_coef(12, 10) = 6.7715733841587470E-04_dp
      fit_coef(0, 11) = 1.0210655880657196E+00_dp
      fit_coef(1, 11) = 9.5730192228839006E-01_dp
      fit_coef(2, 11) = 7.3607353846396317E-01_dp
      fit_coef(3, 11) = 3.0496319982420517E-01_dp
      fit_coef(4, 11) = 2.7825470684213698E-02_dp
      fit_coef(5, 11) = -2.3036487360505412E-02_dp
      fit_coef(6, 11) = 9.0002346429021024E-03_dp
      fit_coef(7, 11) = 9.4071699260299454E-03_dp
      fit_coef(8, 11) = 1.1246072531079006E-03_dp
      fit_coef(9, 11) = -1.8600382884449367E-02_dp
      fit_coef(10, 11) = 1.7805450549655721E-02_dp
      fit_coef(11, 11) = -7.6910226916182545E-03_dp
      fit_coef(12, 11) = 1.3712121437792250E-03_dp
      fit_coef(0, 12) = 1.7149561293339615E+00_dp
      fit_coef(1, 12) = 1.3267744256882403E+00_dp
      fit_coef(2, 12) = 9.3812895165305032E-01_dp
      fit_coef(3, 12) = 3.3164771999234482E-01_dp
      fit_coef(4, 12) = -4.9178162808578741E-05_dp
      fit_coef(5, 12) = -2.7249300178168019E-02_dp
      fit_coef(6, 12) = 2.1205458082480019E-02_dp
      fit_coef(7, 12) = 1.1807990512194026E-02_dp
      fit_coef(8, 12) = -3.3661868556830038E-03_dp
      fit_coef(9, 12) = -2.2430236127638940E-02_dp
      fit_coef(10, 12) = 2.5486188136087912E-02_dp
      fit_coef(11, 12) = -1.2102581018395429E-02_dp
      fit_coef(12, 12) = 2.3004381707239473E-03_dp
      fit_coef(0, 13) = 2.8539608561967360E+00_dp
      fit_coef(1, 13) = 1.7797688368810891E+00_dp
      fit_coef(2, 13) = 1.1646443061164560E+00_dp
      fit_coef(3, 13) = 3.4701820688332419E-01_dp
      fit_coef(4, 13) = -3.5929878683565182E-02_dp
      fit_coef(5, 13) = -2.7887790242899075E-02_dp
      fit_coef(6, 13) = 3.6702245498739489E-02_dp
      fit_coef(7, 13) = 1.2908675248958822E-02_dp
      fit_coef(8, 13) = -1.0507576982805505E-02_dp
      fit_coef(9, 13) = -2.4256099909163161E-02_dp
      fit_coef(10, 13) = 3.3404768253646827E-02_dp
      fit_coef(11, 13) = -1.7212687576065637E-02_dp
      fit_coef(12, 13) = 3.4428516812536531E-03_dp
      fit_coef(0, 14) = 4.7432936777438393E+00_dp
      fit_coef(1, 14) = 2.3082382103412251E+00_dp
      fit_coef(2, 14) = 1.4124938166206138E+00_dp
      fit_coef(3, 14) = 3.5283333252051396E-01_dp
      fit_coef(4, 14) = -7.7696363994240056E-02_dp
      fit_coef(5, 14) = -2.4914689387708116E-02_dp
      fit_coef(6, 14) = 5.4698262510218663E-02_dp
      fit_coef(7, 14) = 1.2640616949106807E-02_dp
      fit_coef(8, 14) = -1.9941323208816249E-02_dp
      fit_coef(9, 14) = -2.3878392941469071E-02_dp
      fit_coef(10, 14) = 4.1055603040417131E-02_dp
      fit_coef(11, 14) = -2.2734874408993563E-02_dp
      fit_coef(12, 14) = 4.7400588393736497E-03_dp
      fit_coef(0, 15) = 8.0710112216342775E+00_dp
      fit_coef(1, 15) = 2.9055595870824633E+00_dp
      fit_coef(2, 15) = 1.6866060964430423E+00_dp
      fit_coef(3, 15) = 3.5581815059695143E-01_dp
      fit_coef(4, 15) = -1.2393245100043256E-01_dp
      fit_coef(5, 15) = -2.0058383552301023E-02_dp
      fit_coef(6, 15) = 7.4640422271172679E-02_dp
      fit_coef(7, 15) = 1.1638820236828815E-02_dp
      fit_coef(8, 15) = -3.0928127527058424E-02_dp
      fit_coef(9, 15) = -2.2331476973823049E-02_dp
      fit_coef(10, 15) = 4.8851141745373840E-02_dp
      fit_coef(11, 15) = -2.8681346469043273E-02_dp
      fit_coef(12, 15) = 6.1686347014912515E-03_dp
      fit_coef(0, 16) = 1.3032066476244642E-03_dp
      fit_coef(1, 16) = 3.8001657920981851E-03_dp
      fit_coef(2, 16) = 5.5378064823689123E-03_dp
      fit_coef(3, 16) = 4.8206577899485006E-03_dp
      fit_coef(4, 16) = 2.2930144045863621E-03_dp
      fit_coef(5, 16) = 1.0986261495124486E-04_dp
      fit_coef(6, 16) = -5.1106328320878528E-04_dp
      fit_coef(7, 16) = -1.2833639292522648E-04_dp
      fit_coef(8, 16) = 1.1356979226852716E-04_dp
      fit_coef(9, 16) = 1.3973942184720303E-04_dp
      fit_coef(10, 16) = -1.8830360701542777E-04_dp
      fit_coef(11, 16) = 8.7997162572425538E-05_dp
      fit_coef(12, 16) = -1.5788992466518323E-05_dp
      fit_coef(0, 17) = 3.3544869799214628E-03_dp
      fit_coef(1, 17) = 9.3282083632741895E-03_dp
      fit_coef(2, 17) = 1.3271370703308300E-02_dp
      fit_coef(3, 17) = 1.1319835500098886E-02_dp
      fit_coef(4, 17) = 5.2524178309963383E-03_dp
      fit_coef(5, 17) = 1.8435087221054173E-04_dp
      fit_coef(6, 17) = -1.1797404048471125E-03_dp
      fit_coef(7, 17) = -2.7516373364939583E-04_dp
      fit_coef(8, 17) = 2.7734414812233385E-04_dp
      fit_coef(9, 17) = 2.9073439175898594E-04_dp
      fit_coef(10, 17) = -4.1550528469613449E-04_dp
      fit_coef(11, 17) = 1.9807893770328183E-04_dp
      fit_coef(12, 17) = -3.6025500379600989E-05_dp
      fit_coef(0, 18) = 6.3980511048867201E-03_dp
      fit_coef(1, 18) = 1.6128813328333793E-02_dp
      fit_coef(2, 18) = 2.1881033183460945E-02_dp
      fit_coef(3, 18) = 1.8021417293275506E-02_dp
      fit_coef(4, 18) = 8.0169003498435417E-03_dp
      fit_coef(5, 18) = 8.7835720128096682E-05_dp
      fit_coef(6, 18) = -1.8255106100472426E-03_dp
      fit_coef(7, 18) = -3.5831804749509092E-04_dp
      fit_coef(8, 18) = 4.8285099003593436E-04_dp
      fit_coef(9, 18) = 3.2465264639707240E-04_dp
      fit_coef(10, 18) = -5.5521076066674020E-04_dp
      fit_coef(11, 18) = 2.7719905231653717E-04_dp
      fit_coef(12, 18) = -5.1820236344809990E-05_dp
      fit_coef(0, 19) = 1.1614376264492066E-02_dp
      fit_coef(1, 19) = 2.5394323464202898E-02_dp
      fit_coef(2, 19) = 3.1733081124309684E-02_dp
      fit_coef(3, 19) = 2.4647015498881791E-02_dp
      fit_coef(4, 19) = 1.0500322632906088E-02_dp
      fit_coef(5, 19) = -1.2365877291301032E-05_dp
      fit_coef(6, 19) = -2.4904038430234622E-03_dp
      fit_coef(7, 19) = -4.8516369798151703E-04_dp
      fit_coef(8, 19) = 6.1030277836899595E-04_dp
      fit_coef(9, 19) = 7.4056213117855896E-04_dp
      fit_coef(10, 19) = -1.1330531422737624E-03_dp
      fit_coef(11, 19) = 5.8028785292689455E-04_dp
      fit_coef(12, 19) = -1.1292938481673382E-04_dp
      fit_coef(0, 20) = 2.0921907643278354E-02_dp
      fit_coef(1, 20) = 3.9009588349526253E-02_dp
      fit_coef(2, 20) = 4.3627398050961781E-02_dp
      fit_coef(3, 20) = 3.0443034676393681E-02_dp
      fit_coef(4, 20) = 1.1774589070303968E-02_dp
      fit_coef(5, 20) = 1.7741503302715373E-04_dp
      fit_coef(6, 20) = -2.3187359164888160E-03_dp
      fit_coef(7, 20) = -6.1637368205092694E-04_dp
      fit_coef(8, 20) = -5.3221714109514493E-05_dp
      fit_coef(9, 20) = 8.1942339340144803E-04_dp
      fit_coef(10, 20) = -2.2354982629820732E-04_dp
      fit_coef(11, 20) = -2.0305492301268794E-04_dp
      fit_coef(12, 20) = 9.4193850816527892E-05_dp
      fit_coef(0, 21) = 3.7306315930286466E-02_dp
      fit_coef(1, 21) = 5.9218420616905736E-02_dp
      fit_coef(2, 21) = 5.8868131527264646E-02_dp
      fit_coef(3, 21) = 3.5324612409341802E-02_dp
      fit_coef(4, 21) = 1.0684265753661753E-02_dp
      fit_coef(5, 21) = -4.9371106930299301E-04_dp
      fit_coef(6, 21) = -8.1292928450316614E-04_dp
      fit_coef(7, 21) = -5.1619725924012061E-04_dp
      fit_coef(8, 21) = 1.3804573307375741E-03_dp
      fit_coef(9, 21) = -3.4663308659499629E-03_dp
      fit_coef(10, 21) = 3.3745183791109305E-03_dp
      fit_coef(11, 21) = -1.4291379899595120E-03_dp
      fit_coef(12, 21) = 2.2454897381750800E-04_dp
      fit_coef(0, 22) = 6.5528592554674012E-02_dp
      fit_coef(1, 22) = 8.8324441729528691E-02_dp
      fit_coef(2, 22) = 7.8339185211631426E-02_dp
      fit_coef(3, 22) = 3.9448552741652813E-02_dp
      fit_coef(4, 22) = 7.3083551980621516E-03_dp
      fit_coef(5, 22) = -2.3247267042822940E-03_dp
      fit_coef(6, 22) = 4.5294755854382923E-04_dp
      fit_coef(7, 22) = 1.2969150143151570E-03_dp
      fit_coef(8, 22) = 4.6541067893071160E-04_dp
      fit_coef(9, 22) = -2.5032569665157803E-03_dp
      fit_coef(10, 22) = 1.1684561968462262E-03_dp
      fit_coef(11, 22) = 1.5370128957771531E-04_dp
      fit_coef(12, 22) = -1.4620664545141759E-04_dp
      fit_coef(0, 23) = 1.1324428651781197E-01_dp
      fit_coef(1, 23) = 1.2859806173609753E-01_dp
      fit_coef(2, 23) = 1.0214609707305304E-01_dp
      fit_coef(3, 23) = 4.2253398459926365E-02_dp
      fit_coef(4, 23) = 1.9762536371264882E-03_dp
      fit_coef(5, 23) = -4.5714996217850632E-03_dp
      fit_coef(6, 23) = 1.8868500721209416E-03_dp
      fit_coef(7, 23) = 2.5695589529597719E-03_dp
      fit_coef(8, 23) = 2.3850425890078301E-04_dp
      fit_coef(9, 23) = -3.0200640644662698E-03_dp
      fit_coef(10, 23) = 1.9503667616499671E-03_dp
      fit_coef(11, 23) = -5.5257674937112412E-04_dp
      fit_coef(12, 23) = 7.8113401323294423E-05_dp
      fit_coef(0, 24) = 1.9267245469427555E-01_dp
      fit_coef(1, 24) = 1.8189882938781768E-01_dp
      fit_coef(2, 24) = 1.2960901455552834E-01_dp
      fit_coef(3, 24) = 4.2611091038268031E-02_dp
      fit_coef(4, 24) = -5.1926218101019609E-03_dp
      fit_coef(5, 24) = -6.3425612842204224E-03_dp
      fit_coef(6, 24) = 4.0010538098000379E-03_dp
      fit_coef(7, 24) = 3.1214642561276981E-03_dp
      fit_coef(8, 24) = 1.0563824093820107E-04_dp
      fit_coef(9, 24) = -4.5110524003026762E-03_dp
      fit_coef(10, 24) = 4.2402114420073420E-03_dp
      fit_coef(11, 24) = -1.9163626754183359E-03_dp
      fit_coef(12, 24) = 3.6969348039378312E-04_dp
      fit_coef(0, 25) = 3.2317976642569807E-01_dp
      fit_coef(1, 25) = 2.4881850229416846E-01_dp
      fit_coef(2, 25) = 1.5917736470901570E-01_dp
      fit_coef(3, 25) = 3.9343328717200927E-02_dp
      fit_coef(4, 25) = -1.3902519808734082E-02_dp
      fit_coef(5, 25) = -6.8640926105066162E-03_dp
      fit_coef(6, 25) = 6.9006071006695079E-03_dp
      fit_coef(7, 25) = 3.2927740290179990E-03_dp
      fit_coef(8, 25) = -1.0983902176266030E-03_dp
      fit_coef(9, 25) = -5.0555007528979004E-03_dp
      fit_coef(10, 25) = 6.0190910606148324E-03_dp
      fit_coef(11, 25) = -2.9821080457428841E-03_dp
      fit_coef(12, 25) = 5.8221139171923042E-04_dp
      fit_coef(0, 26) = 5.3554089995575782E-01_dp
      fit_coef(1, 26) = 3.2739155636224210E-01_dp
      fit_coef(2, 26) = 1.8848689258384349E-01_dp
      fit_coef(3, 26) = 3.1884980883658924E-02_dp
      fit_coef(4, 26) = -2.3339754433360369E-02_dp
      fit_coef(5, 26) = -5.6037573210389264E-03_dp
      fit_coef(6, 26) = 1.0403698986274485E-02_dp
      fit_coef(7, 26) = 2.8865920265869838E-03_dp
      fit_coef(8, 26) = -3.2033768610669569E-03_dp
      fit_coef(9, 26) = -4.5416607006448602E-03_dp
      fit_coef(10, 26) = 7.3043246118188162E-03_dp
      fit_coef(11, 26) = -3.9689298970288006E-03_dp
      fit_coef(12, 26) = 8.1056551195733142E-04_dp
      fit_coef(0, 27) = 8.8000603036746139E-01_dp
      fit_coef(1, 27) = 4.1182532361787932E-01_dp
      fit_coef(2, 27) = 2.1497937217091584E-01_dp
      fit_coef(3, 27) = 2.1108047064593377E-02_dp
      fit_coef(4, 27) = -3.2191882260831459E-02_dp
      fit_coef(5, 27) = -2.5724974235411323E-03_dp
      fit_coef(6, 27) = 1.3955680448193437E-02_dp
      fit_coef(7, 27) = 1.8152463536498431E-03_dp
      fit_coef(8, 27) = -5.8233926497395817E-03_dp
      fit_coef(9, 27) = -2.9482077294913736E-03_dp
      fit_coef(10, 27) = 7.9261411558925292E-03_dp
      fit_coef(11, 27) = -4.8132854344172145E-03_dp
      fit_coef(12, 27) = 1.0440601147544259E-03_dp
      fit_coef(0, 28) = 1.4460627242259527E+00_dp
      fit_coef(1, 28) = 4.9271419953901929E-01_dp
      fit_coef(2, 28) = 2.3743318711644951E-01_dp
      fit_coef(3, 28) = 9.9076815053936729E-03_dp
      fit_coef(4, 28) = -3.9191794449869852E-02_dp
      fit_coef(5, 28) = 1.3099534113565326E-03_dp
      fit_coef(6, 28) = 1.6894034568497723E-02_dp
      fit_coef(7, 28) = 3.7043982265874470E-04_dp
      fit_coef(8, 28) = -8.3968062748629741E-03_dp
      fit_coef(9, 28) = -6.6028855186588131E-04_dp
      fit_coef(10, 28) = 7.8140284723942129E-03_dp
      fit_coef(11, 28) = -5.3528674153974880E-03_dp
      fit_coef(12, 28) = 1.2297837544111832E-03_dp
      fit_coef(0, 29) = 2.4319559958545311E+00_dp
      fit_coef(1, 29) = 5.6229813567117626E-01_dp
      fit_coef(2, 29) = 2.5870101698993758E-01_dp
      fit_coef(3, 29) = 2.8306941439785287E-03_dp
      fit_coef(4, 29) = -4.4047429705758694E-02_dp
      fit_coef(5, 29) = 4.3384122634558238E-03_dp
      fit_coef(6, 29) = 1.8978692315301807E-02_dp
      fit_coef(7, 29) = -7.9790741322960021E-04_dp
      fit_coef(8, 29) = -1.0330971785215522E-02_dp
      fit_coef(9, 29) = 1.2390333347493327E-03_dp
      fit_coef(10, 29) = 7.5464073935277154E-03_dp
      fit_coef(11, 29) = -5.6851817992854548E-03_dp
      fit_coef(12, 29) = 1.3586302192394280E-03_dp
      fit_coef(0, 30) = 4.5666853091127635E+00_dp
      fit_coef(1, 30) = 6.4009785574659228E-01_dp
      fit_coef(2, 30) = 2.9587958616654764E-01_dp
      fit_coef(3, 30) = 5.3258996536484551E-03_dp
      fit_coef(4, 30) = -4.8921450862390199E-02_dp
      fit_coef(5, 30) = 4.8519904409378041E-03_dp
      fit_coef(6, 30) = 2.1138966562008546E-02_dp
      fit_coef(7, 30) = -9.9509468782633297E-04_dp
      fit_coef(8, 30) = -1.1615586429846661E-02_dp
      fit_coef(9, 30) = 1.4899160811089988E-03_dp
      fit_coef(10, 30) = 8.3912654254730754E-03_dp
      fit_coef(11, 30) = -6.3525637249113569E-03_dp
      fit_coef(12, 30) = 1.5228141862253625E-03_dp

   END SUBROUTINE get_fit_coef_k15_mi

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param fit_coef ...
!> \param L_b ...
!> \param U_b ...
! **************************************************************************************************
   SUBROUTINE get_fit_coef_k15_lo(k, fit_coef, L_b, U_b)
      INTEGER                                            :: k
      REAL(KIND=dp), DIMENSION(0:12, 2*k)                :: fit_coef
      REAL(KIND=dp)                                      :: L_b, U_b

      L_b = 9.3527642781376642E+01_dp
      U_b = 1.1750000000000000E+06_dp
      fit_coef(0, 1) = 3.2793279985494519E-06_dp
      fit_coef(1, 1) = 7.2761221861298733E-06_dp
      fit_coef(2, 1) = 6.6879597248076629E-04_dp
      fit_coef(3, 1) = -3.3377387242640721E-03_dp
      fit_coef(4, 1) = 3.6932190908574580E-02_dp
      fit_coef(5, 1) = -1.4859062400069553E-01_dp
      fit_coef(6, 1) = 4.4629627390711790E-01_dp
      fit_coef(7, 1) = -9.3682363031214910E-01_dp
      fit_coef(8, 1) = 1.3525709274164297E+00_dp
      fit_coef(9, 1) = -1.3157143856436124E+00_dp
      fit_coef(10, 1) = 8.2321722743838222E-01_dp
      fit_coef(11, 1) = -2.9905777894458391E-01_dp
      fit_coef(12, 1) = 4.7927187586119996E-02_dp
      fit_coef(0, 2) = 3.6131623637423831E-05_dp
      fit_coef(1, 2) = 4.7106962333648227E-05_dp
      fit_coef(2, 2) = 4.3052245932532632E-03_dp
      fit_coef(3, 2) = -2.3337397086091179E-02_dp
      fit_coef(4, 2) = 2.3805631448383252E-01_dp
      fit_coef(5, 2) = -9.8141547342868762E-01_dp
      fit_coef(6, 2) = 2.9514193901676489E+00_dp
      fit_coef(7, 2) = -6.1764264357918330E+00_dp
      fit_coef(8, 2) = 8.8882482060089814E+00_dp
      fit_coef(9, 2) = -8.6216924305612981E+00_dp
      fit_coef(10, 2) = 5.3820183799751744E+00_dp
      fit_coef(11, 2) = -1.9515484948325068E+00_dp
      fit_coef(12, 2) = 3.1228748667302658E-01_dp
      fit_coef(0, 3) = 1.9135313464109315E-04_dp
      fit_coef(1, 3) = 1.7460510061267730E-04_dp
      fit_coef(2, 3) = 1.5413483600038124E-02_dp
      fit_coef(3, 3) = -8.7803353742691062E-02_dp
      fit_coef(4, 3) = 8.0147578597001090E-01_dp
      fit_coef(5, 3) = -3.4506170283047766E+00_dp
      fit_coef(6, 3) = 1.0597559824370764E+01_dp
      fit_coef(7, 3) = -2.2472933987251839E+01_dp
      fit_coef(8, 3) = 3.2637358630788739E+01_dp
      fit_coef(9, 3) = -3.1863935196267601E+01_dp
      fit_coef(10, 3) = 1.9983399971045149E+01_dp
      fit_coef(11, 3) = -7.2707478368141611E+00_dp
      fit_coef(12, 3) = 1.1664152307259732E+00_dp
      fit_coef(0, 4) = 7.3883732820766948E-04_dp
      fit_coef(1, 4) = 5.0733833618914323E-04_dp
      fit_coef(2, 4) = 4.4370796269257232E-02_dp
      fit_coef(3, 4) = -2.6481196747393215E-01_dp
      fit_coef(4, 4) = 2.2595142950561815E+00_dp
      fit_coef(5, 4) = -1.0181630118704184E+01_dp
      fit_coef(6, 4) = 3.1679590172936724E+01_dp
      fit_coef(7, 4) = -6.7336030116199026E+01_dp
      fit_coef(8, 4) = 9.7677953648704644E+01_dp
      fit_coef(9, 4) = -9.5149794831930734E+01_dp
      fit_coef(10, 4) = 5.9530024414743906E+01_dp
      fit_coef(11, 4) = -2.1611138637657191E+01_dp
      fit_coef(12, 4) = 3.4601757819555536E+00_dp
      fit_coef(0, 5) = 2.3666313203208780E-03_dp
      fit_coef(1, 5) = 1.2786885886632092E-03_dp
      fit_coef(2, 5) = 1.1076208254179676E-01_dp
      fit_coef(3, 5) = -6.7839594619191412E-01_dp
      fit_coef(4, 5) = 5.5114285011328183E+00_dp
      fit_coef(5, 5) = -2.5395097951060229E+01_dp
      fit_coef(6, 5) = 7.9307253014224088E+01_dp
      fit_coef(7, 5) = -1.6859211837607822E+02_dp
      fit_coef(8, 5) = 2.4452261050037967E+02_dp
      fit_coef(9, 5) = -2.3820141119829208E+02_dp
      fit_coef(10, 5) = 1.4905893976672397E+02_dp
      fit_coef(11, 5) = -5.4127909713308576E+01_dp
      fit_coef(12, 5) = 8.6691001852426552E+00_dp
      fit_coef(0, 6) = 6.6782258468515501E-03_dp
      fit_coef(1, 6) = 2.9071241132673617E-03_dp
      fit_coef(2, 6) = 2.4975031064405542E-01_dp
      fit_coef(3, 6) = -1.5562125797528150E+00_dp
      fit_coef(4, 6) = 1.2185079476552678E+01_dp
      fit_coef(5, 6) = -5.6926325045922916E+01_dp
      fit_coef(6, 6) = 1.7855104053066674E+02_dp
      fit_coef(7, 6) = -3.8050660514368747E+02_dp
      fit_coef(8, 6) = 5.5288905790672220E+02_dp
      fit_coef(9, 6) = -5.3932598394454135E+02_dp
      fit_coef(10, 6) = 3.3782483380582988E+02_dp
      fit_coef(11, 6) = -1.2276125245641836E+02_dp
      fit_coef(12, 6) = 1.9671382604642698E+01_dp
      fit_coef(0, 7) = 1.7168691373333111E-02_dp
      fit_coef(1, 7) = 6.0978405270416770E-03_dp
      fit_coef(2, 7) = 5.2025306168908569E-01_dp
      fit_coef(3, 7) = -3.2821022955670651E+00_dp
      fit_coef(4, 7) = 2.4958349575372374E+01_dp
      fit_coef(5, 7) = -1.1765023272930763E+02_dp
      fit_coef(6, 7) = 3.7056008209422993E+02_dp
      fit_coef(7, 7) = -7.9184187030173371E+02_dp
      fit_coef(8, 7) = 1.1527200840436421E+03_dp
      fit_coef(9, 7) = -1.1258727391755413E+03_dp
      fit_coef(10, 7) = 7.0584825509402651E+02_dp
      fit_coef(11, 7) = -2.5665682670985007E+02_dp
      fit_coef(12, 7) = 4.1145879034693607E+01_dp
      fit_coef(0, 8) = 4.1066740219417780E-02_dp
      fit_coef(1, 8) = 1.1965752322210791E-02_dp
      fit_coef(2, 8) = 1.0148791234031664E+00_dp
      fit_coef(3, 8) = -6.4624725502258276E+00_dp
      fit_coef(4, 8) = 4.7976535067054421E+01_dp
      fit_coef(5, 8) = -2.2750659978981074E+02_dp
      fit_coef(6, 8) = 7.1929276146624795E+02_dp
      fit_coef(7, 8) = -1.5410667800156759E+03_dp
      fit_coef(8, 8) = 2.2474354642090384E+03_dp
      fit_coef(9, 8) = -2.1978239495400421E+03_dp
      fit_coef(10, 8) = 1.3791127707578614E+03_dp
      fit_coef(11, 8) = -5.0179288602294997E+02_dp
      fit_coef(12, 8) = 8.0484577079207980E+01_dp
      fit_coef(0, 9) = 9.2699687800883265E-02_dp
      fit_coef(1, 9) = 2.2155471885187787E-02_dp
      fit_coef(2, 9) = 1.8695902697850499E+00_dp
      fit_coef(3, 9) = -1.1991899383860448E+01_dp
      fit_coef(4, 9) = 8.7242978410239118E+01_dp
      fit_coef(5, 9) = -4.1536938867885857E+02_dp
      fit_coef(6, 9) = 1.3176623596141608E+03_dp
      fit_coef(7, 9) = -2.8299350356509945E+03_dp
      fit_coef(8, 9) = 4.1340902247522199E+03_dp
      fit_coef(9, 9) = -4.0476690586057271E+03_dp
      fit_coef(10, 9) = 2.5420687410683036E+03_dp
      fit_coef(11, 9) = -9.2553103476606759E+02_dp
      fit_coef(12, 9) = 1.4852229538367158E+02_dp
      fit_coef(0, 10) = 1.9947743279902694E-01_dp
      fit_coef(1, 10) = 3.8899312951373388E-02_dp
      fit_coef(2, 10) = 3.2680572849945402E+00_dp
      fit_coef(3, 10) = -2.1084081716614804E+01_dp
      fit_coef(4, 10) = 1.5075557198241123E+02_dp
      fit_coef(5, 10) = -7.1967411375827533E+02_dp
      fit_coef(6, 10) = 2.2897159192976455E+03_dp
      fit_coef(7, 10) = -4.9285687810081845E+03_dp
      fit_coef(8, 10) = 7.2112684192795978E+03_dp
      fit_coef(9, 10) = -7.0684961344198391E+03_dp
      fit_coef(10, 10) = 4.4429042040785398E+03_dp
      fit_coef(11, 10) = -1.6185943473816480E+03_dp
      fit_coef(12, 10) = 2.5986231052596014E+02_dp
      fit_coef(0, 11) = 4.1232198870536146E-01_dp
      fit_coef(1, 11) = 6.4893474864739994E-02_dp
      fit_coef(2, 11) = 5.4310004234894391E+00_dp
      fit_coef(3, 11) = -3.5204004815099751E+01_dp
      fit_coef(4, 11) = 2.4798012213498384E+02_dp
      fit_coef(5, 11) = -1.1858431117066064E+03_dp
      fit_coef(6, 11) = 3.7825064907174756E+03_dp
      fit_coef(7, 11) = -8.1581937192205814E+03_dp
      fit_coef(8, 11) = 1.1954131106568948E+04_dp
      fit_coef(9, 11) = -1.1729786611671518E+04_dp
      fit_coef(10, 11) = 7.3784592386581926E+03_dp
      fit_coef(11, 11) = -2.6896086736495204E+03_dp
      fit_coef(12, 11) = 4.3200462908615089E+02_dp
      fit_coef(0, 12) = 8.2375109973285454E-01_dp
      fit_coef(1, 12) = 1.0280784827845055E-01_dp
      fit_coef(2, 12) = 8.5755836263465604E+00_dp
      fit_coef(3, 12) = -5.5800839220514590E+01_dp
      fit_coef(4, 12) = 3.8804437171318665E+02_dp
      fit_coef(5, 12) = -1.8575547029234676E+03_dp
      fit_coef(6, 12) = 5.9379886844425691E+03_dp
      fit_coef(7, 12) = -1.2830180852831514E+04_dp
      fit_coef(8, 12) = 1.8824783357291468E+04_dp
      fit_coef(9, 12) = -1.8489266233281007E+04_dp
      fit_coef(10, 12) = 1.1638691611847828E+04_dp
      fit_coef(11, 12) = -4.2448366640433251E+03_dp
      fit_coef(12, 12) = 6.8208814863958401E+02_dp
      fit_coef(0, 13) = 1.6005401729604098E+00_dp
      fit_coef(1, 13) = 1.5427671373341209E-01_dp
      fit_coef(2, 13) = 1.2832947943920599E+01_dp
      fit_coef(3, 13) = -8.3759357589093241E+01_dp
      fit_coef(4, 13) = 5.7621942259264915E+02_dp
      fit_coef(5, 13) = -2.7598336160779349E+03_dp
      fit_coef(6, 13) = 8.8381623335643399E+03_dp
      fit_coef(7, 13) = -1.9126056849476197E+04_dp
      fit_coef(8, 13) = 2.8094638893959345E+04_dp
      fit_coef(9, 13) = -2.7617301955762628E+04_dp
      fit_coef(10, 13) = 1.7395609257161428E+04_dp
      fit_coef(11, 13) = -6.3475238166994641E+03_dp
      fit_coef(12, 13) = 1.0203400395817024E+03_dp
      fit_coef(0, 14) = 3.0540489317639166E+00_dp
      fit_coef(1, 14) = 2.1865700868129184E-01_dp
      fit_coef(2, 14) = 1.8148202718663860E+01_dp
      fit_coef(3, 14) = -1.1872710055025075E+02_dp
      fit_coef(4, 14) = 8.0984788084197976E+02_dp
      fit_coef(5, 14) = -3.8796556939052798E+03_dp
      fit_coef(6, 14) = 1.2441638354776058E+04_dp
      fit_coef(7, 14) = -2.6957427016371610E+04_dp
      fit_coef(8, 14) = 3.9635425931543767E+04_dp
      fit_coef(9, 14) = -3.8989060841935432E+04_dp
      fit_coef(10, 14) = 2.4571211983659101E+04_dp
      fit_coef(11, 14) = -8.9693909543608097E+03_dp
      fit_coef(12, 14) = 1.4422380562945843E+03_dp
      fit_coef(0, 15) = 5.8836496088258903E+00_dp
      fit_coef(1, 15) = 2.9333916159882772E-01_dp
      fit_coef(2, 15) = 2.4309148431266522E+01_dp
      fit_coef(3, 15) = -1.5928663857255199E+02_dp
      fit_coef(4, 15) = 1.0800144511418321E+03_dp
      fit_coef(5, 15) = -5.1742775717293125E+03_dp
      fit_coef(6, 15) = 1.6609418151702128E+04_dp
      fit_coef(7, 15) = -3.6019371990459913E+04_dp
      fit_coef(8, 15) = 5.2994637899369998E+04_dp
      fit_coef(9, 15) = -5.2156366955415026E+04_dp
      fit_coef(10, 15) = 3.2881619526078182E+04_dp
      fit_coef(11, 15) = -1.2006412191447664E+04_dp
      fit_coef(12, 15) = 1.9310054306610489E+03_dp
      fit_coef(0, 16) = 1.0490015508948160E-05_dp
      fit_coef(1, 16) = 1.9245918297480574E-05_dp
      fit_coef(2, 16) = 1.7514019628420299E-03_dp
      fit_coef(3, 16) = -8.7825475436631283E-03_dp
      fit_coef(4, 16) = 9.8132135498064849E-02_dp
      fit_coef(5, 16) = -4.0139357711585455E-01_dp
      fit_coef(6, 16) = 1.2151507236457464E+00_dp
      fit_coef(7, 16) = -2.5608071661985146E+00_dp
      fit_coef(8, 16) = 3.7049308475400640E+00_dp
      fit_coef(9, 16) = -3.6079085144794738E+00_dp
      fit_coef(10, 16) = 2.2586411450421502E+00_dp
      fit_coef(11, 16) = -8.2071784242647206E-01_dp
      fit_coef(12, 16) = 1.3153768317036563E-01_dp
      fit_coef(0, 17) = 6.9018219767429348E-05_dp
      fit_coef(1, 17) = 6.9648188439671224E-05_dp
      fit_coef(2, 17) = 6.2459333601416062E-03_dp
      fit_coef(3, 17) = -3.5441557914769072E-02_dp
      fit_coef(4, 17) = 3.2906922405173999E-01_dp
      fit_coef(5, 17) = -1.3703072930999050E+00_dp
      fit_coef(6, 17) = 4.1206399805320482E+00_dp
      fit_coef(7, 17) = -8.6245400433435595E+00_dp
      fit_coef(8, 17) = 1.2424942794117596E+01_dp
      fit_coef(9, 17) = -1.2071284705492808E+01_dp
      fit_coef(10, 17) = 7.5482859678406982E+00_dp
      fit_coef(11, 17) = -2.7416411252527824E+00_dp
      fit_coef(12, 17) = 4.3940704181003898E-01_dp
      fit_coef(0, 18) = 2.8267117174337328E-04_dp
      fit_coef(1, 18) = 2.0340536143980756E-04_dp
      fit_coef(2, 18) = 1.7678726727191417E-02_dp
      fit_coef(3, 18) = -1.0545298250093989E-01_dp
      fit_coef(4, 18) = 8.9650844672191476E-01_dp
      fit_coef(5, 18) = -4.0700645201639842E+00_dp
      fit_coef(6, 18) = 1.2778228701375353E+01_dp
      fit_coef(7, 18) = -2.7365134031898545E+01_dp
      fit_coef(8, 18) = 3.9913427510994921E+01_dp
      fit_coef(9, 18) = -3.9026449660683376E+01_dp
      fit_coef(10, 18) = 2.4477897493830259E+01_dp
      fit_coef(11, 18) = -8.9007828894477790E+00_dp
      fit_coef(12, 18) = 1.4266184467905938E+00_dp
      fit_coef(0, 19) = 9.1918738520005492E-04_dp
      fit_coef(1, 19) = 5.0032919937804867E-04_dp
      fit_coef(2, 19) = 4.3364999825556638E-02_dp
      fit_coef(3, 19) = -2.6781408638943244E-01_dp
      fit_coef(4, 19) = 2.1566125289358995E+00_dp
      fit_coef(5, 19) = -1.0022123878208944E+01_dp
      fit_coef(6, 19) = 3.1317396694353242E+01_dp
      fit_coef(7, 19) = -6.6447554440483501E+01_dp
      fit_coef(8, 19) = 9.6141728832817350E+01_dp
      fit_coef(9, 19) = -9.3449718908958914E+01_dp
      fit_coef(10, 19) = 5.8372166225769881E+01_dp
      fit_coef(11, 19) = -2.1166765338703282E+01_dp
      fit_coef(12, 19) = 3.3863816402153009E+00_dp
      fit_coef(0, 20) = 2.5882618681224243E-03_dp
      fit_coef(1, 20) = 1.1101963800374212E-03_dp
      fit_coef(2, 20) = 9.5153090359631570E-02_dp
      fit_coef(3, 20) = -5.9667092259469967E-01_dp
      fit_coef(4, 20) = 4.6158300825488254E+00_dp
      fit_coef(5, 20) = -2.1702960038752170E+01_dp
      fit_coef(6, 20) = 6.8104607605264903E+01_dp
      fit_coef(7, 20) = -1.4509864087700180E+02_dp
      fit_coef(8, 20) = 2.1082070081369434E+02_dp
      fit_coef(9, 20) = -2.0567820908774044E+02_dp
      fit_coef(10, 20) = 1.2886369297377382E+02_dp
      fit_coef(11, 20) = -4.6839217031726335E+01_dp
      fit_coef(12, 20) = 7.5072963372495414E+00_dp
      fit_coef(0, 21) = 6.5882379082633398E-03_dp
      fit_coef(1, 21) = 2.2621691339110277E-03_dp
      fit_coef(2, 21) = 1.9246964638130989E-01_dp
      fit_coef(3, 21) = -1.2217276753530955E+00_dp
      fit_coef(4, 21) = 9.1698242992620767E+00_dp
      fit_coef(5, 21) = -4.3465513365240064E+01_dp
      fit_coef(6, 21) = 1.3710631049681336E+02_dp
      fit_coef(7, 21) = -2.9321690733250756E+02_dp
      fit_coef(8, 21) = 4.2710092920695109E+02_dp
      fit_coef(9, 21) = -4.1732542801681706E+02_dp
      fit_coef(10, 21) = 2.6170639243954514E+02_dp
      fit_coef(11, 21) = -9.5176513369112968E+01_dp
      fit_coef(12, 21) = 1.5259801532746318E+01_dp
      fit_coef(0, 22) = 1.5543900572102644E-02_dp
      fit_coef(1, 22) = 4.3029374070070658E-03_dp
      fit_coef(2, 22) = 3.6379929804956312E-01_dp
      fit_coef(3, 22) = -2.3299690981198706E+00_dp
      fit_coef(4, 22) = 1.7060773313313661E+01_dp
      fit_coef(5, 22) = -8.1273230384261126E+01_dp
      fit_coef(6, 22) = 2.5746624440333989E+02_dp
      fit_coef(7, 22) = -5.5230172621897168E+02_dp
      fit_coef(8, 22) = 8.0612603717585921E+02_dp
      fit_coef(9, 22) = -7.8876694465068897E+02_dp
      fit_coef(10, 22) = 4.9512974619778498E+02_dp
      fit_coef(11, 22) = -1.8020197227270759E+02_dp
      fit_coef(12, 22) = 2.8909017650939049E+01_dp
      fit_coef(0, 23) = 3.4544493883799224E-02_dp
      fit_coef(1, 23) = 7.7057465719973899E-03_dp
      fit_coef(2, 23) = 6.4801783632712762E-01_dp
      fit_coef(3, 23) = -4.1791996544114722E+00_dp
      fit_coef(4, 23) = 2.9972887604114607E+01_dp
      fit_coef(5, 23) = -1.4323654947892643E+02_dp
      fit_coef(6, 23) = 4.5543253426208781E+02_dp
      fit_coef(7, 23) = -9.7965395620159916E+02_dp
      fit_coef(8, 23) = 1.4326296354673486E+03_dp
      fit_coef(9, 23) = -1.4036958339933367E+03_dp
      fit_coef(10, 23) = 8.8201906011468520E+02_dp
      fit_coef(11, 23) = -3.2125145658138058E+02_dp
      fit_coef(12, 23) = 5.1566695258306126E+01_dp
      fit_coef(0, 24) = 7.3122924052638857E-02_dp
      fit_coef(1, 24) = 1.3047345576674729E-02_dp
      fit_coef(2, 24) = 1.0921563698572587E+00_dp
      fit_coef(3, 24) = -7.0828159501016552E+00_dp
      fit_coef(4, 24) = 4.9900195235442546E+01_dp
      fit_coef(5, 24) = -2.3893035920690767E+02_dp
      fit_coef(6, 24) = 7.6209216068295359E+02_dp
      fit_coef(7, 24) = -1.6433368571765945E+03_dp
      fit_coef(8, 24) = 2.4074464958116155E+03_dp
      fit_coef(9, 24) = -2.3618333223611826E+03_dp
      fit_coef(10, 24) = 1.4854588983048657E+03_dp
      fit_coef(11, 24) = -5.4141801556510109E+02_dp
      fit_coef(12, 24) = 8.6954202781474834E+01_dp
      fit_coef(0, 25) = 1.4863184827325007E-01_dp
      fit_coef(1, 25) = 2.0897369993486958E-02_dp
      fit_coef(2, 25) = 1.7422013268701704E+00_dp
      fit_coef(3, 25) = -1.1349892975305522E+01_dp
      fit_coef(4, 25) = 7.8731198009107700E+01_dp
      fit_coef(5, 25) = -3.7738777230483873E+02_dp
      fit_coef(6, 25) = 1.2069682708983435E+03_dp
      fit_coef(7, 25) = -2.6084306788827607E+03_dp
      fit_coef(8, 25) = 3.8275166346901069E+03_dp
      fit_coef(9, 25) = -3.7594561324724668E+03_dp
      fit_coef(10, 25) = 2.3665593844071400E+03_dp
      fit_coef(11, 25) = -8.6313167371072871E+02_dp
      fit_coef(12, 25) = 1.3869371175524410E+02_dp
      fit_coef(0, 26) = 2.9197038671042685E-01_dp
      fit_coef(1, 26) = 3.1553064146021624E-02_dp
      fit_coef(2, 26) = 2.6212810102963688E+00_dp
      fit_coef(3, 26) = -1.7140795787714239E+01_dp
      fit_coef(4, 26) = 1.1729936023843838E+02_dp
      fit_coef(5, 26) = -5.6251944415470587E+02_dp
      fit_coef(6, 26) = 1.8032102744423644E+03_dp
      fit_coef(7, 26) = -3.9047643960741279E+03_dp
      fit_coef(8, 26) = 5.7382582448223930E+03_dp
      fit_coef(9, 26) = -5.6424075077151520E+03_dp
      fit_coef(10, 26) = 3.5547702376861953E+03_dp
      fit_coef(11, 26) = -1.2973000963117572E+03_dp
      fit_coef(12, 26) = 2.0855898009649175E+02_dp
      fit_coef(0, 27) = 5.5768130937544869E-01_dp
      fit_coef(1, 27) = 4.4572128359602628E-02_dp
      fit_coef(2, 27) = 3.6916856472540940E+00_dp
      fit_coef(3, 27) = -2.4213381713827740E+01_dp
      fit_coef(4, 27) = 1.6378750132966567E+02_dp
      fit_coef(5, 27) = -7.8545555955911163E+02_dp
      fit_coef(6, 27) = 2.5226651741402102E+03_dp
      fit_coef(7, 27) = -5.4722074342493534E+03_dp
      fit_coef(8, 27) = 8.0523685521856414E+03_dp
      fit_coef(9, 27) = -7.9256723613351096E+03_dp
      fit_coef(10, 27) = 4.9969395283607182E+03_dp
      fit_coef(11, 27) = -1.8246427440641146E+03_dp
      fit_coef(12, 27) = 2.9346534721488899E+02_dp
      fit_coef(0, 28) = 1.0450173925119002E+00_dp
      fit_coef(1, 28) = 5.8265207981760117E-02_dp
      fit_coef(2, 28) = 4.8143926353121813E+00_dp
      fit_coef(3, 28) = -3.1648523514665637E+01_dp
      fit_coef(4, 28) = 2.1213334922148832E+02_dp
      fit_coef(5, 28) = -1.0169903912741421E+03_dp
      fit_coef(6, 28) = 3.2709939445720015E+03_dp
      fit_coef(7, 28) = -7.1053218328636922E+03_dp
      fit_coef(8, 28) = 1.0466780738282137E+04_dp
      fit_coef(9, 28) = -1.0310448956212835E+04_dp
      fit_coef(10, 28) = 6.5044600754848825E+03_dp
      fit_coef(11, 28) = -2.3762315782073229E+03_dp
      fit_coef(12, 28) = 3.8231992824865711E+02_dp
      fit_coef(0, 29) = 1.9639167282286836E+00_dp
      fit_coef(1, 29) = 6.9961149923850899E-02_dp
      fit_coef(2, 29) = 5.7725439445353004E+00_dp
      fit_coef(3, 29) = -3.7996769718433761E+01_dp
      fit_coef(4, 29) = 2.5327571805969885E+02_dp
      fit_coef(5, 29) = -1.2137919470627476E+03_dp
      fit_coef(6, 29) = 3.9071985303035226E+03_dp
      fit_coef(7, 29) = -8.4944480629188183E+03_dp
      fit_coef(8, 29) = 1.2521478684961125E+04_dp
      fit_coef(9, 29) = -1.2340746128541903E+04_dp
      fit_coef(10, 29) = 7.7883138533160081E+03_dp
      fit_coef(11, 29) = -2.8461039477634558E+03_dp
      fit_coef(12, 29) = 4.5802666012412590E+02_dp
      fit_coef(0, 30) = 4.0342493016500347E+00_dp
      fit_coef(1, 30) = 7.9647259264362835E-02_dp
      fit_coef(2, 30) = 6.5710181771750698E+00_dp
      fit_coef(3, 30) = -4.3255572360835060E+01_dp
      fit_coef(4, 30) = 2.8820713216838652E+02_dp
      fit_coef(5, 30) = -1.3810306152021003E+03_dp
      fit_coef(6, 30) = 4.4457135475846326E+03_dp
      fit_coef(7, 30) = -9.6658445565099519E+03_dp
      fit_coef(8, 30) = 1.4249062004096166E+04_dp
      fit_coef(9, 30) = -1.4044077033410062E+04_dp
      fit_coef(10, 30) = 8.8636369152484167E+03_dp
      fit_coef(11, 30) = -3.2391609850813516E+03_dp
      fit_coef(12, 30) = 5.2129451486951780E+02_dp

   END SUBROUTINE get_fit_coef_k15_lo

END MODULE minimax_exp_k15
