﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.
#nullable disable

using System;
using System.Collections;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Diagnostics;
using System.Linq;
using Microsoft.CodeAnalysis.CSharp.Symbols;
using Microsoft.CodeAnalysis.CSharp.Symbols.Metadata.PE;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.CSharp.Test.Utilities;
using Microsoft.CodeAnalysis.Diagnostics;
using Microsoft.CodeAnalysis.Emit;
using Microsoft.CodeAnalysis.Test.Utilities;
using Microsoft.CodeAnalysis.VisualBasic;
using Roslyn.Test.Utilities;
using Xunit;
using Xunit.Sdk;

namespace Microsoft.CodeAnalysis.CSharp.UnitTests.Semantics;

[CompilerTrait(CompilerFeature.Extensions)]
public partial class ExtensionTests : CompilingTestBase
{
    internal string ExtensionMarkerAttributeIL = """

.class public auto ansi sealed beforefieldinit System.Runtime.CompilerServices.ExtensionMarkerAttribute
    extends [mscorlib]System.Attribute
{
    .custom instance void [mscorlib]System.AttributeUsageAttribute::.ctor(valuetype [mscorlib]System.AttributeTargets) = (
        01 00 ff 7f 00 00 01 00 54 02 09 49 6e 68 65 72
        69 74 65 64 00
    )

    .method public hidebysig specialname rtspecialname 
        instance void .ctor (
            string name
        ) cil managed 
    {
        .maxstack 8

        IL_0000: ldarg.0
        IL_0001: call instance void [mscorlib]System.Attribute::.ctor()
        IL_0006: nop
        IL_0007: nop
        IL_0008: ret
    }
}
""";

    private static string ExpectedOutput(string output)
    {
        return ExecutionConditionUtil.IsMonoOrCoreClr ? output : null;
    }

    private static void AssertEqualAndNoDuplicates(string[] expected, string[] actual)
    {
        Assert.True(expected.All(new HashSet<string>().Add), $"Duplicates were found in '{nameof(expected)}'");
        Assert.True(actual.All(new HashSet<string>().Add), $"Duplicates were found in '{nameof(actual)}'");
        AssertEx.SetEqual(expected, actual);
    }

    private static void AssertExtensionDeclaration(INamedTypeSymbol symbol)
    {
        // Verify things that are common for all extension types
        Assert.Equal(TypeKind.Extension, symbol.TypeKind);
        Assert.True(symbol.IsExtension);
        Assert.False(string.IsNullOrEmpty(symbol.ExtensionGroupingName));
        Assert.False(string.IsNullOrEmpty(symbol.ExtensionMarkerName));
        Assert.Null(symbol.BaseType);
        Assert.Empty(symbol.Interfaces);
        Assert.Empty(symbol.AllInterfaces);
        Assert.True(symbol.IsReferenceType);
        Assert.False(symbol.IsValueType);
        Assert.False(symbol.IsAnonymousType);
        Assert.False(symbol.IsTupleType);
        Assert.False(symbol.IsNativeIntegerType);
        Assert.Equal(SpecialType.None, symbol.SpecialType);
        Assert.False(symbol.IsRefLikeType);
        Assert.False(symbol.IsUnmanagedType);
        Assert.False(symbol.IsReadOnly);
        Assert.False(symbol.IsRecord);
        Assert.Equal(CodeAnalysis.NullableAnnotation.None, symbol.NullableAnnotation);
        Assert.Throws<NotSupportedException>(() => { symbol.WithNullableAnnotation(CodeAnalysis.NullableAnnotation.Annotated); });

        Assert.False(symbol.IsScriptClass);
        Assert.False(symbol.IsImplicitClass);
        Assert.False(symbol.IsComImport);
        Assert.False(symbol.IsFileLocal);
        Assert.Null(symbol.DelegateInvokeMethod);
        Assert.Null(symbol.EnumUnderlyingType);
        Assert.Null(symbol.AssociatedSymbol);
        Assert.False(symbol.MightContainExtensionMethods);
        Assert.Null(symbol.TupleUnderlyingType);
        Assert.True(symbol.TupleElements.IsDefault);
        Assert.False(symbol.IsSerializable);
        Assert.Null(symbol.NativeIntegerUnderlyingType);

        Assert.Equal(SymbolKind.NamedType, symbol.Kind);
        AssertEx.Equal("", symbol.Name);
        Assert.Equal(SpecialType.None, symbol.SpecialType);
        Assert.True(symbol.IsDefinition);
        Assert.False(symbol.IsStatic);
        Assert.False(symbol.IsVirtual);
        Assert.False(symbol.IsOverride);
        Assert.False(symbol.IsAbstract);
        Assert.True(symbol.IsSealed);
        Assert.False(symbol.IsExtern);
        Assert.False(symbol.IsImplicitlyDeclared);
        Assert.False(symbol.CanBeReferencedByName);
        Assert.Equal(Accessibility.Public, symbol.DeclaredAccessibility);

        var namedTypeSymbol = symbol.GetSymbol<NamedTypeSymbol>();
        Assert.True(namedTypeSymbol.HasSpecialName);
        Assert.False(namedTypeSymbol.IsImplicitlyDeclared);
    }

    [Fact]
    public void EmptyExtension()
    {
        var src = """
public static class Extensions
{
    extension(object) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);
        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'::'<Extension>$'
        } // end of class <M>$C43E2675C7BBF9284AF22FB8A9BF0280
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);

        Assert.Empty(symbol.MemberNames);
        Assert.Empty(symbol.InstanceConstructors);
        Assert.Empty(symbol.StaticConstructors);
        Assert.Empty(symbol.Constructors);

        Assert.Equal(0, symbol.Arity);
        Assert.False(symbol.IsGenericType);
        Assert.False(symbol.IsUnboundGenericType);
        Assert.Empty(symbol.TypeParameters);
        Assert.Empty(symbol.TypeArguments);
        Assert.Same(symbol, symbol.OriginalDefinition);
        Assert.Same(symbol, symbol.ConstructedFrom);
        AssertEx.Equal("Extensions", symbol.ContainingSymbol.Name);
        AssertEx.Equal("Extensions", symbol.ContainingType.Name);
        AssertEx.Equal("<M>$C43E2675C7BBF9284AF22FB8A9BF0280", symbol.MetadataName);

        var member = symbol.ContainingType.GetMembers().Single();
        AssertEx.Equal("Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<M>$C43E2675C7BBF9284AF22FB8A9BF0280", member.ToTestDisplayString());
        var underlying = (SourceNamedTypeSymbol)((Symbols.PublicModel.NamedTypeSymbol)member).UnderlyingNamedTypeSymbol;
        AssertEx.Equal("extension(System.Object)", underlying.ComputeExtensionGroupingRawName());
        AssertEx.Equal("extension(System.Object)", underlying.ComputeExtensionMarkerRawName());

        var format = new SymbolDisplayFormat(typeQualificationStyle: SymbolDisplayTypeQualificationStyle.NameAndContainingTypesAndNamespaces);
        AssertEx.Equal("Extensions.extension(System.Object)", symbol.ToDisplayString(format));

        format = new SymbolDisplayFormat(kindOptions: SymbolDisplayKindOptions.IncludeTypeKeyword);
        AssertEx.Equal("extension(Object)", symbol.ToDisplayString(format));

        format = new SymbolDisplayFormat();
        AssertEx.Equal("extension(Object)", symbol.ToDisplayString(format));

        format = new SymbolDisplayFormat(compilerInternalOptions: SymbolDisplayCompilerInternalOptions.UseMetadataMemberNames);
        AssertEx.Equal("<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<M>$C43E2675C7BBF9284AF22FB8A9BF0280", symbol.ToDisplayString(format));

        var comp5 = CreateCompilation(src);
        comp5.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp5.VerifyEmitDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object) { }
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5)
            );
    }

    [Fact]
    public void TypeParameters_01()
    {
        // Unconstrained type parameter
        var src = """
public static class Extensions
{
    extension<T>(T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);
        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$01CE3801593377B4E240F33E20D30D50'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static
                void '<Extension>$' (
                    !T ''
                ) cil managed
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$01CE3801593377B4E240F33E20D30D50'::'<Extension>$'
        } // end of class <M>$01CE3801593377B4E240F33E20D30D50
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);

        Assert.Equal(1, symbol.Arity);
        Assert.True(symbol.IsGenericType);
        Assert.False(symbol.IsUnboundGenericType);
        AssertEx.SequenceEqual(["T"], symbol.TypeParameters.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["T"], symbol.TypeArguments.ToTestDisplayStrings());
        Assert.Same(symbol, symbol.OriginalDefinition);
        Assert.Same(symbol, symbol.ConstructedFrom);
        AssertEx.Equal("Extensions", symbol.ContainingSymbol.Name);
        AssertEx.Equal("Extensions", symbol.ContainingType.Name);
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol.MetadataName);

        var member = symbol.ContainingType.GetMembers().Single();
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", member.ToTestDisplayString());

        var constructed = symbol.Construct(comp.GetSpecialType(SpecialType.System_Int32));
        Assert.True(constructed.IsExtension);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<System.Int32>.<M>$01CE3801593377B4E240F33E20D30D50", constructed.ToTestDisplayString());
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", constructed.MetadataName);
        Assert.NotSame(symbol, constructed);
        Assert.Same(symbol, constructed.OriginalDefinition);
        Assert.Same(symbol, constructed.ConstructedFrom);

        var unbound = symbol.ConstructUnboundGenericType();
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<>.<M>$01CE3801593377B4E240F33E20D30D50", unbound.ToTestDisplayString());
        Assert.True(unbound.IsUnboundGenericType);
        Assert.NotSame(symbol, unbound);
        Assert.Same(symbol, unbound.OriginalDefinition);
        Assert.Same(symbol, unbound.ConstructedFrom);
    }

    [Fact]
    public void TypeParameters_02()
    {
        // Constrained type parameter
        var src = """
public static class Extensions
{
    extension<T>(T) where T : struct { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);
        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$BCF902721DDD961E5243C324D8379E5C`1'<valuetype .ctor ([mscorlib]System.ValueType) $T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$B865B3ED3C68CE2EBBC104FFAF3CFF93'<valuetype .ctor ([mscorlib]System.ValueType) T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    !T ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$B865B3ED3C68CE2EBBC104FFAF3CFF93'::'<Extension>$'
        } // end of class <M>$B865B3ED3C68CE2EBBC104FFAF3CFF93
    } // end of class <G>$BCF902721DDD961E5243C324D8379E5C`1
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);

        Assert.Equal(1, symbol.Arity);
        Assert.True(symbol.IsGenericType);
        AssertEx.SequenceEqual(["T"], symbol.TypeParameters.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["T"], symbol.TypeArguments.ToTestDisplayStrings());
        Assert.True(symbol.TypeParameters.Single().IsValueType);
        Assert.False(symbol.TypeParameters.Single().IsReferenceType);
        Assert.Empty(symbol.TypeParameters.Single().ConstraintTypes);

        var format = new SymbolDisplayFormat(genericsOptions: SymbolDisplayGenericsOptions.IncludeTypeParameters | SymbolDisplayGenericsOptions.IncludeTypeConstraints);
        AssertEx.Equal("extension<T>(T) where T : struct", symbol.ToDisplayString(format));
    }

    [Fact]
    public void TypeParameters_03()
    {
        // Constraint on undefined type parameter
        var src = """
public static class Extensions
{
    extension(object) where T : struct { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,23): error CS0080: Constraints are not allowed on non-generic declarations
            //     extension(object) where T : struct { }
            Diagnostic(ErrorCode.ERR_ConstraintOnlyAllowedOnGenericDecl, "where").WithLocation(3, 23));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        Assert.Equal(0, symbol.Arity);
        Assert.False(symbol.IsGenericType);
        Assert.Empty(symbol.TypeParameters.ToTestDisplayStrings());
        Assert.Empty(symbol.TypeArguments.ToTestDisplayStrings());
    }

    [Fact]
    public void TypeParameters_04()
    {
        // Type parameter variance
        var src = """
public static class Extensions
{
    extension<out T>(T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1960: Invalid variance modifier. Only interface and delegate type parameters can be specified as variant.
            //     extension<out T>(object) { }
            Diagnostic(ErrorCode.ERR_IllegalVarianceSyntax, "out").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        Assert.Equal(1, symbol.Arity);
        Assert.True(symbol.IsGenericType);
        AssertEx.SequenceEqual(["out T"], symbol.TypeParameters.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["out T"], symbol.TypeArguments.ToTestDisplayStrings());
    }

    [Fact]
    public void TypeParameters_05()
    {
        // Duplicate type parameter
        var src = """
public static class Extensions
{
    extension<T, T>(T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,18): error CS0692: Duplicate type parameter 'T'
            //     extension<T, T>(T) { }
            Diagnostic(ErrorCode.ERR_DuplicateTypeParameter, "T").WithArguments("T").WithLocation(3, 18),
            // (3,21): error CS0229: Ambiguity between 'T' and 'T'
            //     extension<T, T>(T) { }
            Diagnostic(ErrorCode.ERR_AmbigMember, "T").WithArguments("T", "T").WithLocation(3, 21));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        Assert.Equal(2, symbol.Arity);
        AssertEx.SequenceEqual(["T", "T"], symbol.TypeParameters.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["T", "T"], symbol.TypeArguments.ToTestDisplayStrings());
    }

    [Fact]
    public void TypeParameters_05_Nested()
    {
        // Duplicate type parameter
        var src = """
public static class Extensions
{
    extension<T, T>(C<T>) { }
}
class C<T> { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,18): error CS0692: Duplicate type parameter 'T'
            //     extension<T, T>(C<T>) { }
            Diagnostic(ErrorCode.ERR_DuplicateTypeParameter, "T").WithArguments("T").WithLocation(3, 18),
            // (3,23): error CS0229: Ambiguity between 'T' and 'T'
            //     extension<T, T>(C<T>) { }
            Diagnostic(ErrorCode.ERR_AmbigMember, "T").WithArguments("T", "T").WithLocation(3, 23));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        Assert.Equal(2, symbol.Arity);
        AssertEx.SequenceEqual(["T", "T"], symbol.TypeParameters.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["T", "T"], symbol.TypeArguments.ToTestDisplayStrings());
    }

    [Fact]
    public void TypeParameters_06()
    {
        // Type parameter same as outer type parameter
        var src = """
public static class Extensions<T>
{
    extension<T>(T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension<T>(object) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5),
            // (3,15): warning CS0693: Type parameter 'T' has the same name as the type parameter from outer type 'Extensions<T>'
            //     extension<T>(object) { }
            Diagnostic(ErrorCode.WRN_TypeParameterSameAsOuterTypeParameter, "T").WithArguments("T", "Extensions<T>").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        Assert.Equal(1, symbol.Arity);
        AssertEx.SequenceEqual(["T"], symbol.TypeParameters.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["T"], symbol.TypeArguments.ToTestDisplayStrings());

        var container = symbol.ContainingType;
        var substitutedExtension = (INamedTypeSymbol)container.Construct(comp.GetSpecialType(SpecialType.System_Int32)).GetMembers().Single();
        AssertEx.Equal("Extensions<System.Int32>.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", substitutedExtension.ToTestDisplayString());
        Assert.True(substitutedExtension.IsExtension);
    }

    [Fact]
    public void TypeParameters_07()
    {
        // Reserved type name for type parameter
        var src = $$"""
public static class Extensions
{
    extension<record>(record) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): warning CS8860: Types and aliases should not be named 'record'.
            //     extension<record>(object) { }
            Diagnostic(ErrorCode.WRN_RecordNamedDisallowed, "record").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["record"], symbol.TypeParameters.ToTestDisplayStrings());
    }

    [Fact]
    public void TypeParameters_08()
    {
        // Reserved type name for type parameter
        var src = $$"""
public static class Extensions
{
    extension<file>(file) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9056: Types and aliases cannot be named 'file'.
            //     extension<file>(object) { }
            Diagnostic(ErrorCode.ERR_FileTypeNameDisallowed, "file").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["file"], symbol.TypeParameters.ToTestDisplayStrings());
    }

    [Fact]
    public void TypeParameters_09()
    {
        // Member name same as type parameter
        var src = $$"""
public static class Extensions
{
    extension<T>(T t)
    {
        void T() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void TypeParameters_10()
    {
        var src = $$"""
#nullable enable
public static class Extensions
{
    extension<T>(T) where T : notnull
    {
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
        var verifier = CompileAndVerify(comp);
        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$C7A07C3975E80DE5DBC93B5392C6C922'<T>
            extends [mscorlib]System.Object
        {
            .param type T
                .custom instance void System.Runtime.CompilerServices.NullableAttribute::.ctor(uint8) = (
                    01 00 01 00 00
                )
            // Methods
            .method private hidebysig specialname static
                void '<Extension>$' (
                    !T ''
                ) cil managed
            {
                .custom instance void System.Runtime.CompilerServices.NullableContextAttribute::.ctor(uint8) = (
                    01 00 01 00 00
                )
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x209d
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C7A07C3975E80DE5DBC93B5392C6C922'::'<Extension>$'
        } // end of class <M>$C7A07C3975E80DE5DBC93B5392C6C922
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));
    }

    [Fact]
    public void BadContainer_Generic()
    {
        var src = """
object.M();

public static class Extensions<T>
{
    extension(object) { public static void M() { } }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'M'
            // object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 8),
            // (5,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object) { public static void M() { } }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(5, 5));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);
        Assert.True(symbol.IsGenericType);
        var members = symbol.ContainingType.GetMembers();
        AssertEx.SequenceEqual([
            "Extensions<T>.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<M>$C43E2675C7BBF9284AF22FB8A9BF0280",
            "void Extensions<T>.M()"
            ], members.ToTestDisplayStrings());
    }

    [Fact]
    public void BadContainer_TopLevel()
    {
        var src = """
object.M();

extension(object) { public static void M() { } }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'M'
            // object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 8),
            // (3,1): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            // extension(object) { public static void M() { } }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 1));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);
        Assert.Null(symbol.ContainingType);
        AssertEx.Equal("<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<M>$C43E2675C7BBF9284AF22FB8A9BF0280", symbol.ToTestDisplayString());
    }

    [Fact]
    public void BadContainer_Nested()
    {
        var src = """
object.M();

public static class Extensions
{
    static void Method()
    {
        object.M();
    }

    public static class Extensions2
    {
        extension(object) { public static void M() { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'M'
            // object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 8),
            // (7,16): error CS0117: 'object' does not contain a definition for 'M'
            //         object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(7, 16),
            // (12,9): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //         extension(object) { public static void M() { } }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(12, 9));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var nestedExtension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();

        var nestedExtensionSymbol = model.GetDeclaredSymbol(nestedExtension);
        AssertExtensionDeclaration(nestedExtensionSymbol);
        AssertEx.Equal("Extensions.Extensions2", nestedExtensionSymbol.ContainingType.ToTestDisplayString());
        var members = nestedExtensionSymbol.ContainingType.GetMembers();
        AssertEx.SequenceEqual([
            "Extensions.Extensions2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<M>$C43E2675C7BBF9284AF22FB8A9BF0280",
            "void Extensions.Extensions2.M()"
            ], members.ToTestDisplayStrings());
    }

    [Fact]
    public void BadContainer_NestedInExtension()
    {
        var src = """
string.M();

public static class Extensions
{
    static void Method2()
    {
        string.M();
    }

    extension(object)
    {
        static void Method()
        {
            string.M();
        }

        extension(string) { public static void M() { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'string' does not contain a definition for 'M'
            // string.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("string", "M").WithLocation(1, 8),
            // (7,16): error CS0117: 'string' does not contain a definition for 'M'
            //         string.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("string", "M").WithLocation(7, 16),
            // (14,20): error CS0117: 'string' does not contain a definition for 'M'
            //             string.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("string", "M").WithLocation(14, 20),
            // (17,9): error CS9282: This member is not allowed in an extension block
            //         extension(string) { public static void M() { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "extension").WithLocation(17, 9));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var nestedExtension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();

        var nestedExtensionSymbol = model.GetDeclaredSymbol(nestedExtension);
        AssertExtensionDeclaration(nestedExtensionSymbol);
        AssertEx.Equal("Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<M>$C43E2675C7BBF9284AF22FB8A9BF0280",
            nestedExtensionSymbol.ContainingType.ToTestDisplayString());
        AssertEx.SequenceEqual([
            "void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method()",
            "Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<G>$34505F560D9EACF86A87F3ED1F85E448.<M>$34505F560D9EACF86A87F3ED1F85E448"
            ], nestedExtensionSymbol.ContainingType.GetMembers().ToTestDisplayStrings());
    }

    [Fact]
    public void BadContainer_TypeKind()
    {
        var src = """
object.M();

public static struct Extensions
{
    extension(object) { public static void M() { } }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'M'
            // object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 8),
            // (3,22): error CS0106: The modifier 'static' is not valid for this item
            // public static struct Extensions
            Diagnostic(ErrorCode.ERR_BadMemberFlag, "Extensions").WithArguments("static").WithLocation(3, 22),
            // (5,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object) { public static void M() { } }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(5, 5));
    }

    [Fact]
    public void BadContainer_NotStatic()
    {
        var src = """
object.M();

public class Extensions
{
    extension(object) { public static void M() { } }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'M'
            // object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 8),
            // (5,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object) { public static void M() { } }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(5, 5));
    }

    [Theory, CombinatorialData]
    public void ExtensionIndex_InPartial(bool reverseOrder)
    {
        var src1 = """
public static partial class Extensions
{
    extension(object) { }
}
""";
        var src2 = """
public static partial class Extensions
{
}
""";

        var src = reverseOrder ? new[] { src2, src1 } : new[] { src1, src2 };
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);
        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'::'<Extension>$'
        } // end of class <M>$C43E2675C7BBF9284AF22FB8A9BF0280
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[reverseOrder ? 1 : 0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);
        AssertEx.Equal("<M>$C43E2675C7BBF9284AF22FB8A9BF0280", symbol.MetadataName);
    }

    [Fact]
    public void ExtensionIndex_InPartial_TwoExtension()
    {
        var src1 = """
public static partial class Extensions
{
    extension<T>(T t) { }
}
""";
        var src2 = """
public static partial class Extensions
{
    extension<T1, T2>((T1, T2) t) { }
}
""";

        var comp = CreateCompilation([src1, src2]);
        comp.VerifyEmitDiagnostics();

        var tree1 = comp.SyntaxTrees[0];
        var model1 = comp.GetSemanticModel(tree1);
        var extension1 = tree1.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol1 = model1.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$D1693D81A12E8DED4ED68FE22D9E856F", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$D1693D81A12E8DED4ED68FE22D9E856F", sourceExtension1.ToTestDisplayString());

        var tree2 = comp.SyntaxTrees[1];
        var extension2 = tree2.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var model2 = comp.GetSemanticModel(tree2);
        var symbol2 = model2.GetDeclaredSymbol(extension2);
        var sourceExtension2 = symbol2.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$38DD3033A2145E0D2274BCCB48D8434F", symbol2.MetadataName);
        AssertEx.Equal("Extensions.<G>$B6FEF98A1719AAFE96009C5CC65441CB<T1, T2>.<M>$38DD3033A2145E0D2274BCCB48D8434F", sourceExtension2.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_InPartial_TwoExtensions()
    {
        var src = """
public static partial class Extensions
{
    extension<T>(T) { }
}
public static partial class Extensions
{
    extension<T1, T2>((T1, T2)) { }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol1.ToTestDisplayString());

        var extension2 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol2 = model.GetDeclaredSymbol(extension2);
        AssertEx.Equal("<M>$0A2F70F0BFFD1BC7F8C8E0A6CD0B0194", symbol2.MetadataName);
        AssertEx.Equal("Extensions.<G>$B6FEF98A1719AAFE96009C5CC65441CB<T1, T2>.<M>$0A2F70F0BFFD1BC7F8C8E0A6CD0B0194", symbol2.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_TwoExtensions_01()
    {
        var src = """
public static class Extensions
{
    extension<T>(T) { }
    extension<T>(T) { }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol1.ToTestDisplayString());

        var extension2 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol2 = model.GetDeclaredSymbol(extension2);
        var sourceExtension2 = symbol2.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol2.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol2.ToTestDisplayString());
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78222")]
    public void ExtensionInInterface()
    {
        var src = """
interface IInterface
{
    extension(object)
    {
    }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5)
        );
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78222")]
    public void ExtensionInClass()
    {
        var src = """
class SomeClass
{
    extension(object)
    {
    }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5)
        );
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78222")]
    public void ExtensionInStruct()
    {
        var src = """
struct SomeStruct
{
    extension(object)
    {
    }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5)
        );
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78222")]
    public void ExtensionInRecordClass()
    {
        var src = """
record class SomeRecord
{
    extension(object)
    {
    }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5)
        );
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78222")]
    public void ExtensionInRecordStruct()
    {
        var src = """
record struct SomeRecord
{
    extension(object)
    {
    }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5)
        );
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78222")]
    public void ExtensionInInterfaceWithVariance()
    {
        var src = """
interface IInterface<in T>
{
    extension(object)
    {
    }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5)
        );
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78135")]
    public void ExtensionWithCapturing()
    {
        var src = """
using System;
using System.Text;

var sb = new StringBuilder("Info: ");
StringBuilder.Inspect(sb);

public static class Extensions
{
    extension(StringBuilder)
    {
        public static StringBuilder Inspect(StringBuilder sb)
        {
            var s = () =>
            {
                foreach (char c in sb.ToString())
                {
                    Console.Write(c);
                }
            };
            s();
            return sb;
        }
    }
}
""";

        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp, expectedOutput: """
            Info: 
            """, expectedReturnCode: 0, trimOutput: false);

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    class [mscorlib]System.Text.StringBuilder ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20de
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99'::'<Extension>$'
        } // end of class <M>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99
        // Methods
        .method public hidebysig static 
            class [mscorlib]System.Text.StringBuilder Inspect (
                class [mscorlib]System.Text.StringBuilder sb
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 44 32 39 45 37 30 45 30
                44 43 41 35 42 42 46 43 46 41 43 37 43 32 42 45
                46 33 43 35 43 39 39 00 00
            )
            // Method begins at RVA 0x20a5
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99'::Inspect
    } // end of class <G>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public class [mscorlib]System.Text.StringBuilder sb
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x2079
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method '<>c__DisplayClass1_0'::.ctor
        .method assembly hidebysig 
            instance void '<Inspect>b__0' () cil managed 
        {
            // Method begins at RVA 0x20a8
            // Code size 42 (0x2a)
            .maxstack 2
            .locals init (
                [0] string,
                [1] int32
            )
            IL_0000: ldarg.0
            IL_0001: ldfld class [mscorlib]System.Text.StringBuilder Extensions/'<>c__DisplayClass1_0'::sb
            IL_0006: callvirt instance string [mscorlib]System.Object::ToString()
            IL_000b: stloc.0
            IL_000c: ldc.i4.0
            IL_000d: stloc.1
            IL_000e: br.s IL_0020
            // loop start (head: IL_0020)
                IL_0010: ldloc.0
                IL_0011: ldloc.1
                IL_0012: callvirt instance char [mscorlib]System.String::get_Chars(int32)
                IL_0017: call void [mscorlib]System.Console::Write(char)
                IL_001c: ldloc.1
                IL_001d: ldc.i4.1
                IL_001e: add
                IL_001f: stloc.1
                IL_0020: ldloc.1
                IL_0021: ldloc.0
                IL_0022: callvirt instance int32 [mscorlib]System.String::get_Length()
                IL_0027: blt.s IL_0010
            // end loop
            IL_0029: ret
        } // end of method '<>c__DisplayClass1_0'::'<Inspect>b__0'
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Text.StringBuilder Inspect (
            class [mscorlib]System.Text.StringBuilder sb
        ) cil managed 
    {
        // Method begins at RVA 0x2081
        // Code size 35 (0x23)
        .maxstack 8
        IL_0000: newobj instance void Extensions/'<>c__DisplayClass1_0'::.ctor()
        IL_0005: dup
        IL_0006: ldarg.0
        IL_0007: stfld class [mscorlib]System.Text.StringBuilder Extensions/'<>c__DisplayClass1_0'::sb
        IL_000c: dup
        IL_000d: ldftn instance void Extensions/'<>c__DisplayClass1_0'::'<Inspect>b__0'()
        IL_0013: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0018: callvirt instance void [mscorlib]System.Action::Invoke()
        IL_001d: ldfld class [mscorlib]System.Text.StringBuilder Extensions/'<>c__DisplayClass1_0'::sb
        IL_0022: ret
    } // end of method Extensions::Inspect
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99.<M>$CD29E70E0DCA5BBFCFAC7C2BEF3C5C99", symbol1.ToTestDisplayString());
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78135")]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78042")]
    public void ExtensionWithCapturing2()
    {
        var src = """
using System;

var a = int.DoSomething();
a();

public static class IntExt
{
    extension(int)
    {
        public static Action DoSomething()
        {
            var b = 7;
            Action a = () =>
            {
                int a = 123;
                Console.WriteLine(a);
                b++;
                Console.WriteLine(b);
            };
            Console.WriteLine("Some data");
            ++b;
            return a;
        }
    }
}
""";

        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp, expectedOutput: """
            Some data
            123
            9

            """, expectedReturnCode: 0, trimOutput: false);

        verifier.VerifyTypeIL("IntExt", """
.class public auto ansi abstract sealed beforefieldinit IntExt
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    int32 ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20ef
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'::'<Extension>$'
        } // end of class <M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69
        // Methods
        .method public hidebysig static 
            class [mscorlib]System.Action DoSomething () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 42 41 34 31 43 46 45 32 42
                35 45 44 41 45 42 38 43 31 42 39 30 36 32 46 35
                39 45 44 34 44 36 39 00 00
            )
            // Method begins at RVA 0x20bc
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'::DoSomething
    } // end of class <G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public int32 b
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x2073
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method '<>c__DisplayClass1_0'::.ctor
        .method assembly hidebysig 
            instance void '<DoSomething>b__0' () cil managed 
        {
            // Method begins at RVA 0x20c0
            // Code size 35 (0x23)
            .maxstack 3
            .locals init (
                [0] int32
            )
            IL_0000: ldc.i4.s 123
            IL_0002: call void [mscorlib]System.Console::WriteLine(int32)
            IL_0007: ldarg.0
            IL_0008: ldfld int32 IntExt/'<>c__DisplayClass1_0'::b
            IL_000d: stloc.0
            IL_000e: ldarg.0
            IL_000f: ldloc.0
            IL_0010: ldc.i4.1
            IL_0011: add
            IL_0012: stfld int32 IntExt/'<>c__DisplayClass1_0'::b
            IL_0017: ldarg.0
            IL_0018: ldfld int32 IntExt/'<>c__DisplayClass1_0'::b
            IL_001d: call void [mscorlib]System.Console::WriteLine(int32)
            IL_0022: ret
        } // end of method '<>c__DisplayClass1_0'::'<DoSomething>b__0'
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Action DoSomething () cil managed 
    {
        // Method begins at RVA 0x207c
        // Code size 52 (0x34)
        .maxstack 3
        .locals init (
            [0] class [mscorlib]System.Action,
            [1] int32
        )
        IL_0000: newobj instance void IntExt/'<>c__DisplayClass1_0'::.ctor()
        IL_0005: dup
        IL_0006: ldc.i4.7
        IL_0007: stfld int32 IntExt/'<>c__DisplayClass1_0'::b
        IL_000c: dup
        IL_000d: ldftn instance void IntExt/'<>c__DisplayClass1_0'::'<DoSomething>b__0'()
        IL_0013: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0018: stloc.0
        IL_0019: ldstr "Some data"
        IL_001e: call void [mscorlib]System.Console::WriteLine(string)
        IL_0023: dup
        IL_0024: ldfld int32 IntExt/'<>c__DisplayClass1_0'::b
        IL_0029: ldc.i4.1
        IL_002a: add
        IL_002b: stloc.1
        IL_002c: ldloc.1
        IL_002d: stfld int32 IntExt/'<>c__DisplayClass1_0'::b
        IL_0032: ldloc.0
        IL_0033: ret
    } // end of method IntExt::DoSomething
} // end of class IntExt
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69", symbol1.MetadataName);
        AssertEx.Equal("IntExt.<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69.<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69", symbol1.ToTestDisplayString());
    }

    [Fact]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78135")]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78042")]
    public void ExtensionWithCapturing_LocalFunction()
    {
        var src = """
using System;

var a = int.DoSomething();
a();

public static class IntExt
{
    extension(int)
    {
        public static Action DoSomething()
        {
            var b = 7;
            Console.WriteLine("Some data");
            ++b;
            return Do;

            void Do(){
                int a = 123;
                Console.WriteLine(a);
                b++;
                Console.WriteLine(b);
            }
        }
    }
}
""";

        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp, expectedOutput: """
            Some data
            123
            9

            """, expectedReturnCode: 0, trimOutput: false);

        verifier.VerifyTypeIL("IntExt", """
.class public auto ansi abstract sealed beforefieldinit IntExt
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    int32 ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20ef
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'::'<Extension>$'
        } // end of class <M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69
        // Methods
        .method public hidebysig static 
            class [mscorlib]System.Action DoSomething () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 42 41 34 31 43 46 45 32 42
                35 45 44 41 45 42 38 43 31 42 39 30 36 32 46 35
                39 45 44 34 44 36 39 00 00
            )
            // Method begins at RVA 0x20ba
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'::DoSomething
    } // end of class <G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public int32 b
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x2073
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method '<>c__DisplayClass1_0'::.ctor
        .method assembly hidebysig 
            instance void '<DoSomething>g__Do|0' () cil managed 
        {
            // Method begins at RVA 0x20c0
            // Code size 35 (0x23)
            .maxstack 3
            .locals init (
                [0] int32
            )
            IL_0000: ldc.i4.s 123
            IL_0002: call void [mscorlib]System.Console::WriteLine(int32)
            IL_0007: ldarg.0
            IL_0008: ldfld int32 IntExt/'<>c__DisplayClass1_0'::b
            IL_000d: stloc.0
            IL_000e: ldarg.0
            IL_000f: ldloc.0
            IL_0010: ldc.i4.1
            IL_0011: add
            IL_0012: stfld int32 IntExt/'<>c__DisplayClass1_0'::b
            IL_0017: ldarg.0
            IL_0018: ldfld int32 IntExt/'<>c__DisplayClass1_0'::b
            IL_001d: call void [mscorlib]System.Console::WriteLine(int32)
            IL_0022: ret
        } // end of method '<>c__DisplayClass1_0'::'<DoSomething>g__Do|0'
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Action DoSomething () cil managed 
    {
        // Method begins at RVA 0x207c
        // Code size 50 (0x32)
        .maxstack 3
        .locals init (
            [0] int32
        )
        IL_0000: newobj instance void IntExt/'<>c__DisplayClass1_0'::.ctor()
        IL_0005: dup
        IL_0006: ldc.i4.7
        IL_0007: stfld int32 IntExt/'<>c__DisplayClass1_0'::b
        IL_000c: ldstr "Some data"
        IL_0011: call void [mscorlib]System.Console::WriteLine(string)
        IL_0016: dup
        IL_0017: ldfld int32 IntExt/'<>c__DisplayClass1_0'::b
        IL_001c: ldc.i4.1
        IL_001d: add
        IL_001e: stloc.0
        IL_001f: dup
        IL_0020: ldloc.0
        IL_0021: stfld int32 IntExt/'<>c__DisplayClass1_0'::b
        IL_0026: ldftn instance void IntExt/'<>c__DisplayClass1_0'::'<DoSomething>g__Do|0'()
        IL_002c: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0031: ret
    } // end of method IntExt::DoSomething
} // end of class IntExt
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69", symbol1.MetadataName);
        AssertEx.Equal("IntExt.<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69.<M>$BA41CFE2B5EDAEB8C1B9062F59ED4D69", symbol1.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_TwoExtensions_02()
    {
        var src = """
public static class Extensions
{
    extension<T>(T) { }
    class C { }
    extension<T>(T) { }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol1.ToTestDisplayString());

        var extension2 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol2 = model.GetDeclaredSymbol(extension2);
        var sourceExtension2 = symbol2.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol2.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol2.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_TwoExtensions_03()
    {
        var src = """
extension<T>(T) { }
class C { }
extension<T>(T) { }
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            // extension<T>(object) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(1, 1),
            // (3,1): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            // extension<T>(object) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 1));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol1.MetadataName);
        AssertEx.Equal("<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol1.ToTestDisplayString());

        var extension2 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol2 = model.GetDeclaredSymbol(extension2);
        var sourceExtension2 = symbol2.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol2.MetadataName);
        AssertEx.Equal("<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol2.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_TwoExtensions_05()
    {
        var src = """
public static class Extensions
{
    extension<T>(T) { }
    extension<T1, T2>((T1, T2) t) { }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol1.ToTestDisplayString());

        var extension2 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol2 = model.GetDeclaredSymbol(extension2);
        var sourceExtension2 = symbol2.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$38DD3033A2145E0D2274BCCB48D8434F", symbol2.MetadataName);
        AssertEx.Equal("Extensions.<G>$B6FEF98A1719AAFE96009C5CC65441CB<T1, T2>.<M>$38DD3033A2145E0D2274BCCB48D8434F", symbol2.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_TwoExtensions_06()
    {
        var src = """
public static class Extensions
{
    extension<T>(T) { }
    extension<T1>(T1) where T1 : struct { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(extension1);
        var sourceExtension1 = symbol1.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$01CE3801593377B4E240F33E20D30D50", symbol1.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T>.<M>$01CE3801593377B4E240F33E20D30D50", symbol1.ToTestDisplayString());

        var extension2 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol2 = model.GetDeclaredSymbol(extension2);
        var sourceExtension2 = symbol2.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$0F0A7F439039332917C923D7DF48FA4C", symbol2.MetadataName);
        AssertEx.Equal("Extensions.<G>$BCF902721DDD961E5243C324D8379E5C<T1>.<M>$0F0A7F439039332917C923D7DF48FA4C", symbol2.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionIndex_ElevenExtensions()
    {
        var src = """
public static class Extensions
{
    extension<T1>(T1 o1) { }
    extension<T2>(T2 o2) { }
    extension<T3>(T3 o3) { }
    extension<T4>(T4 o4) { }
    extension<T5>(T5 o5) { }
    extension<T6>(T6 o6) { }
    extension<T7>(T7 o7) { }
    extension<T8>(T8 o8) { }
    extension<T9>(T9 o9) { }
    extension<T10>(T10 o10) { }
    class C { }
    extension<T11>(T11 o11) { }
}
""";

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Last();
        var symbol = model.GetDeclaredSymbol(extension);
        var sourceExtension = symbol.GetSymbol<SourceNamedTypeSymbol>();
        AssertEx.Equal("<M>$9B08A69343790083B512FC2D1C4929FC", symbol.MetadataName);
        AssertEx.Equal("Extensions.<G>$8048A6C8BE30A622530249B904B537EB<T11>.<M>$9B08A69343790083B512FC2D1C4929FC", symbol.ToTestDisplayString());
    }

    [Fact]
    public void Member_InstanceMethod()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        internal void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method assembly hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method assembly hidebysig 
            instance void M () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x2069
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method assembly hidebysig static 
        void M (
            object o
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["M"], symbol.MemberNames);
        AssertEx.SequenceEqual(["", "M"], symbol.ContainingType.MemberNames);
        AssertEx.Equal("void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", symbol.GetMember("M").ToTestDisplayString());
    }

    [Fact]
    public void Member_ExtensionMethod()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        void M(this int i) { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,14): error CS1109: Extension methods must be defined in a top level static class;  is a nested class
            //         void M(this int i) { }
            Diagnostic(ErrorCode.ERR_ExtensionMethodsDecl, "M").WithArguments("").WithLocation(5, 14));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var method = tree.GetRoot().DescendantNodes().OfType<MethodDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(method);
        AssertEx.Equal("void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(this System.Int32 i)", symbol.ToTestDisplayString());
        Assert.True(symbol.IsExtensionMethod);
    }

    [Fact]
    public void Member_StaticMethod()
    {
        var src = """
public static class Extensions
{
    extension(object)
    {
        static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'::'<Extension>$'
        } // end of class <M>$C43E2675C7BBF9284AF22FB8A9BF0280
        // Methods
        .method private hidebysig static 
            void M () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            // Method begins at RVA 0x2069
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method private hidebysig static 
        void M () cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["M"], symbol.MemberNames);
        AssertEx.Equal("void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", symbol.GetMember("M").ToTestDisplayString());
    }

    [Fact]
    public void Member_InstanceMethod_ExplicitInterfaceImplementation()
    {
        var src = """
public interface I
{
    void M();
}

public static class Extensions
{
    extension(object o)
    {
        void I.M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (10,16): error CS0541: 'Extensions.extension(object).M()': explicit interface declaration can only be declared in a class, record, struct or interface
            //         void I.M() { }
            Diagnostic(ErrorCode.ERR_ExplicitInterfaceImplementationInNonClassOrStruct, "M").WithArguments("Extensions.extension(object).M()").WithLocation(10, 16),
            // (10,16): error CS9282: This member is not allowed in an extension block
            //         void I.M() { }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "M").WithLocation(10, 16));
    }

    [Fact]
    public void Member_InstanceMethod_ShadowingTypeParameter()
    {
        var src = """
public static class Extensions
{
    extension<T>(T o)
    {
        void M<T>() { }
        void M2(int T) { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,16): error CS9289: Type parameter 'T' has the same name as an extension container type parameter
            //         void M<T>() { }
            Diagnostic(ErrorCode.ERR_TypeParameterSameNameAsExtensionTypeParameter, "T").WithArguments("T").WithLocation(5, 16),
            // (6,21): error CS9288: 'T': a parameter, local variable, or local function cannot have the same name as an extension container type parameter
            //         void M2(int T) { }
            Diagnostic(ErrorCode.ERR_LocalSameNameAsExtensionTypeParameter, "T").WithArguments("T").WithLocation(6, 21)
            );
    }

    [Fact]
    public void Member_InstanceProperty()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        int Property { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x206b
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method private hidebysig specialname 
            instance int32 get_Property () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x206d
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_Property
        .method private hidebysig specialname 
            instance void set_Property (
                int32 'value'
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x206d
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::set_Property
        // Properties
        .property instance int32 Property()
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            .get instance int32 Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_Property()
            .set instance void Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::set_Property(int32)
        }
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method private hidebysig static 
        int32 get_Property (
            object o
        ) cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 3 (0x3)
        .maxstack 8
        IL_0000: ldc.i4.s 42
        IL_0002: ret
    } // end of method Extensions::get_Property
    .method private hidebysig static 
        void set_Property (
            object o,
            int32 'value'
        ) cil managed 
    {
        // Method begins at RVA 0x206b
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::set_Property
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["Property"], symbol.MemberNames);
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }", symbol.GetMember("Property").ToTestDisplayString());

        AssertEx.Equal([
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }",
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property.get",
            "void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property.set"],
            symbol.GetMembers().ToTestDisplayStrings());
    }

    [Fact]
    public void Member_InstanceProperty_Auto()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        int Property { get; set; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,13): error CS9282: This member is not allowed in an extension block
            //         int Property { get; set; }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Property").WithLocation(5, 13));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["Property"], symbol.MemberNames);
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }", symbol.GetMember("Property").ToTestDisplayString());

        AssertEx.Equal([
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<Property>k__BackingField",
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }",
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property.get",
            "void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property.set"],
            symbol.GetMembers().ToTestDisplayStrings());
    }

    [Fact]
    public void Member_InstanceProperty_ExplicitInterfaceImplementation()
    {
        var src = """
public interface I
{
    int Property { get; set; }
}
public static class Extensions
{
    extension(object o)
    {
        int I.Property { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (9,15): error CS0541: 'Extensions.extension(object).Property': explicit interface declaration can only be declared in a class, record, struct or interface
            //         int I.Property { get => 42; set { } }
            Diagnostic(ErrorCode.ERR_ExplicitInterfaceImplementationInNonClassOrStruct, "Property").WithArguments("Extensions.extension(object).Property").WithLocation(9, 15));
    }

    [Fact]
    public void Member_StaticProperty()
    {
        var src = """
public static class Extensions
{
    extension(object)
    {
        static int Property { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x206b
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'::'<Extension>$'
        } // end of class <M>$C43E2675C7BBF9284AF22FB8A9BF0280
        // Methods
        .method private hidebysig specialname static 
            int32 get_Property () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            // Method begins at RVA 0x206d
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_Property
        .method private hidebysig specialname static 
            void set_Property (
                int32 'value'
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            // Method begins at RVA 0x206d
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::set_Property
        // Properties
        .property int32 Property()
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            .get int32 Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_Property()
            .set void Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::set_Property(int32)
        }
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method private hidebysig static 
        int32 get_Property () cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 3 (0x3)
        .maxstack 8
        IL_0000: ldc.i4.s 42
        IL_0002: ret
    } // end of method Extensions::get_Property
    .method private hidebysig static 
        void set_Property (
            int32 'value'
        ) cil managed 
    {
        // Method begins at RVA 0x206b
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::set_Property
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["Property"], symbol.MemberNames);
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }", symbol.GetMember("Property").ToTestDisplayString());
    }

    [Fact]
    public void Member_StaticProperty_Auto()
    {
        var src = """
public static class Extensions
{
    extension(object)
    {
        static int Property { get; set; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,20): error CS9282: This member is not allowed in an extension block
            //         static int Property { get; set; }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Property").WithLocation(5, 20));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["Property"], symbol.MemberNames);
        AssertEx.SequenceEqual([
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.<Property>k__BackingField",
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }",
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property.get",
            "void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property.set"],
            symbol.GetMembers().ToTestDisplayStrings());

        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }", symbol.GetMember("Property").ToTestDisplayString());
    }

    [Fact]
    public void Member_InstanceIndexer()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        int this[int i] { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,13): error CS9282: This member is not allowed in an extension block
            //         int this[int i] { get => 42; set { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(5, 13));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertEx.SequenceEqual(["this[]"], symbol.MemberNames);
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.this[System.Int32 i] { get; set; }", symbol.GetMember("this[]").ToTestDisplayString());

        AssertEx.Equal([
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.this[System.Int32 i] { get; set; }",
            "System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.this[System.Int32 i].get",
            "void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.this[System.Int32 i].set"],
            symbol.GetMembers().ToTestDisplayStrings());

        var comp5 = CreateCompilation(src);
        comp5.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp5.VerifyEmitDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object o)
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5),
            // (5,13): error CS9282: This member is not allowed in an extension block
            //         int this[int i] { get => 42; set { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(5, 13)
            );
    }

    [Fact]
    public void Member_StaticIndexer()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        static int this[int i] { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object o)
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5),
            // (5,20): error CS0106: The modifier 'static' is not valid for this item
            //         static int this[int i] { get => 42; set { } }
            Diagnostic(ErrorCode.ERR_BadMemberFlag, "this").WithArguments("static").WithLocation(5, 20),
            // (5,20): error CS9282: This member is not allowed in an extension block
            //         static int this[int i] { get => 42; set { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(5, 20)
            );
    }

    [Fact]
    public void Member_Type_01()
    {
        var src = """
public static class Extensions
{
    extension(object)
    {
        class Nested { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,15): error CS9282: This member is not allowed in an extension block
            //         class Nested { }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Nested").WithLocation(5, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        Assert.Empty(symbol.MemberNames);
        AssertEx.SequenceEqual(["Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Nested"], symbol.GetMembers().ToTestDisplayStrings());
        AssertEx.SequenceEqual(["Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Nested"], symbol.GetTypeMembers().ToTestDisplayStrings());
        AssertEx.Equal("Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Nested", symbol.GetTypeMember("Nested").ToTestDisplayString());
    }

    [Fact]
    public void Member_Type_02()
    {
        var src = """
C.Nested x = null;

public static class Extensions
{
    extension(C)
    {
        class Nested { }
    }
}
class C { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,3): error CS0426: The type name 'Nested' does not exist in the type 'C'
            // C.Nested x = null;
            Diagnostic(ErrorCode.ERR_DottedTypeNameNotFoundInAgg, "Nested").WithArguments("Nested", "C").WithLocation(1, 3),
            // (7,15): error CS9282: This member is not allowed in an extension block
            //         class Nested { }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Nested").WithLocation(7, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var syntax = GetSyntax<QualifiedNameSyntax>(tree, "C.Nested");
        Assert.Null(model.GetSymbolInfo(syntax).Symbol);
    }

    [Fact]
    public void Member_Type_03()
    {
        var src = """
object.Nested x = null;

public static class Extensions
{
    extension(object)
    {
        class Nested { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'Nested'
            // object.Nested x = null;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "Nested").WithArguments("object", "Nested").WithLocation(1, 8),
            // (1,15): error CS1002: ; expected
            // object.Nested x = null;
            Diagnostic(ErrorCode.ERR_SemicolonExpected, "x").WithLocation(1, 15),
            // (1,15): error CS0103: The name 'x' does not exist in the current context
            // object.Nested x = null;
            Diagnostic(ErrorCode.ERR_NameNotInContext, "x").WithArguments("x").WithLocation(1, 15),
            // (7,15): error CS9282: This member is not allowed in an extension block
            //         class Nested { }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Nested").WithLocation(7, 15));
    }

    [Fact]
    public void Member_Constructor()
    {
        var src = """
public static class Extensions
{
    extension(object) { Extensions() { } }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,25): error CS1520: Method must have a return type
            //     extension(object) { Extensions() { } }
            Diagnostic(ErrorCode.ERR_MemberNeedsType, "Extensions").WithLocation(3, 25),
            // (3,25): error CS9282: This member is not allowed in an extension block
            //     extension(object) { Extensions() { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Extensions").WithLocation(3, 25));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);

        AssertEx.SequenceEqual([".ctor"], symbol.MemberNames);
        AssertEx.SequenceEqual(["Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280..ctor()"], symbol.InstanceConstructors.ToTestDisplayStrings());
        Assert.Empty(symbol.StaticConstructors);
        AssertEx.SequenceEqual(["Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280..ctor()"], symbol.Constructors.ToTestDisplayStrings());
    }

    [Fact]
    public void Member_Finalizer()
    {
        var src = """
public static class Extensions
{
    extension(object) { ~Extensions() { } }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,26): error CS9282: This member is not allowed in an extension block
            //     extension(object) { ~Extensions() { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "Extensions").WithLocation(3, 26));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);

        AssertEx.SequenceEqual(["Finalize"], symbol.MemberNames);
        AssertEx.SequenceEqual(["void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Finalize()"], symbol.GetMembers().ToTestDisplayStrings());
    }

    [Fact]
    public void Member_Field()
    {
        var src = """
_ = new object().field;

public static class Extensions
{
    extension(object o) { int field = 0; }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,18): error CS1061: 'object' does not contain a definition for 'field' and no accessible extension method 'field' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // _ = new object().field;
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "field").WithArguments("object", "field").WithLocation(1, 18),
            // (5,31): error CS9282: This member is not allowed in an extension block
            //     extension(object o) { int field = 0; }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "field").WithLocation(5, 31),
            // (5,31): warning CS0169: The field 'Extensions.extension(object).field' is never used
            //     extension(object o) { int field = 0; }
            Diagnostic(ErrorCode.WRN_UnreferencedField, "field").WithArguments("Extensions.extension(object).field").WithLocation(5, 31));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);

        AssertEx.SequenceEqual(["field"], symbol.MemberNames);
        AssertEx.SequenceEqual(["System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.field"], symbol.GetMembers().ToTestDisplayStrings());
    }

    [Fact]
    public void Member_Const()
    {
        var src = """
public static class Extensions
{
    extension(object) { const int i = 0; }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,35): error CS9282: This member is not allowed in an extension block
            //     extension(object) { const int i = 0; }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "i").WithLocation(3, 35));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);

        AssertEx.SequenceEqual(["i"], symbol.MemberNames);
        AssertEx.SequenceEqual(["System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.i"], symbol.GetMembers().ToTestDisplayStrings());
    }

    [Fact]
    public void Member_InstanceEvent_ExplicitInterfaceImplementation()
    {
        var src = """
public interface I
{
    event System.Action E;
}
public static class Extensions
{
    extension(object o)
    {
        event System.Action I.E { add { } remove { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (9,31): error CS0541: 'Extensions.extension(object).E': explicit interface declaration can only be declared in a class, record, struct or interface
            //         event System.Action I.E { add { } remove { } }
            Diagnostic(ErrorCode.ERR_ExplicitInterfaceImplementationInNonClassOrStruct, "E").WithArguments("Extensions.extension(object).E").WithLocation(9, 31),
            // (9,31): error CS9282: This member is not allowed in an extension block
            //         event System.Action I.E { add { } remove { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "E").WithLocation(9, 31),
            // (9,35): error CS9282: This member is not allowed in an extension block
            //         event System.Action I.E { add { } remove { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "add").WithLocation(9, 35),
            // (9,43): error CS9282: This member is not allowed in an extension block
            //         event System.Action I.E { add { } remove { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "remove").WithLocation(9, 43));
    }

    [Theory]
    [InlineData("class")]
    [InlineData("struct")]
    [InlineData("interface")]
    [InlineData("record")]
    [InlineData("record struct")]
    public void IsExtension_MiscTypeKinds(string typeKind)
    {
        var src = $$"""
{{typeKind}} C { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<TypeDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(type);
        Assert.False(symbol.IsExtension);
    }

    [Fact]
    public void IsExtension_Delegate()
    {
        var src = $$"""
delegate void C();
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<DelegateDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        Assert.False(symbol.IsExtension);
    }

    [Fact]
    public void IsExtension_Enum()
    {
        var src = $$"""
enum E { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<EnumDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        Assert.False(symbol.IsExtension);
    }

    [Fact]
    public void Attributes_01()
    {
        var src = """
public static class Extensions
{
    [System.Obsolete]
    extension(object) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,6): error CS0592: Attribute 'System.Obsolete' is not valid on this declaration type. It is only valid on 'class, struct, enum, constructor, method, property, indexer, field, event, interface, delegate' declarations.
            //     [System.Obsolete]
            Diagnostic(ErrorCode.ERR_AttributeOnBadSymbolType, "System.Obsolete").WithArguments("System.Obsolete", "class, struct, enum, constructor, method, property, indexer, field, event, interface, delegate").WithLocation(3, 6));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        AssertEx.SetEqual(["System.ObsoleteAttribute"], symbol.GetAttributes().Select(a => a.ToString()));
    }

    [Fact]
    public void Attributes_02()
    {
        var src = """
public static class Extensions
{
    [My(nameof(o)), My(nameof(Extensions))]
    extension(object o) { }
}

public class MyAttribute : System.Attribute
{
    public MyAttribute(string s) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,6): error CS0592: Attribute 'My' is not valid on this declaration type. It is only valid on 'assembly, module, class, struct, enum, constructor, method, property, indexer, field, event, interface, parameter, delegate, return, type parameter' declarations.
            //     [My(nameof(o)), My(nameof(Extensions))]
            Diagnostic(ErrorCode.ERR_AttributeOnBadSymbolType, "My").WithArguments("My", "assembly, module, class, struct, enum, constructor, method, property, indexer, field, event, interface, parameter, delegate, return, type parameter").WithLocation(3, 6),
            // (3,21): error CS0579: Duplicate 'My' attribute
            //     [My(nameof(o)), My(nameof(Extensions))]
            Diagnostic(ErrorCode.ERR_DuplicateAttribute, "My").WithArguments("My").WithLocation(3, 21)
            );
    }

    [Fact]
    public void ReceiverParameter()
    {
        var src = """
public static class Extensions
{
    extension(object) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        AssertEx.Equal("System.Object", symbol.ExtensionParameter.ToTestDisplayString());
    }

    [Fact]
    public void ReceiverParameter_WithIdentifier()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        public object M() { return o; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        AssertEx.Equal("System.Object o", symbol.ExtensionParameter.ToTestDisplayString());

        var returnStatement = GetSyntax<ReturnStatementSyntax>(tree, "return o;");
        AssertEx.Equal("System.Object o", model.GetSymbolInfo(returnStatement.Expression).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ReceiverParameter_Multiple()
    {
        var src = """
public static class Extensions
{
    extension(int i, int j, C c) { }
}
class C { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,22): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i, int j, C c) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "int j").WithLocation(3, 22),
            // (3,29): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i, int j, C c) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "C c").WithLocation(3, 29));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        var extensionParameter = symbol.ExtensionParameter;

        AssertEx.Equal("System.Int32 i", extensionParameter.ToTestDisplayString());
        Assert.True(extensionParameter.Equals(extensionParameter));

        var parameterSyntaxes = tree.GetRoot().DescendantNodes().OfType<ParameterSyntax>().ToArray();
        AssertEx.Equal("System.Int32 i", model.GetDeclaredSymbol(parameterSyntaxes[0]).ToTestDisplayString());
        Assert.Same(extensionParameter, model.GetDeclaredSymbol(parameterSyntaxes[0]));

        AssertEx.Equal("System.Int32", model.GetTypeInfo(parameterSyntaxes[1].Type).Type.ToTestDisplayString());
        Assert.Null(model.GetDeclaredSymbol(parameterSyntaxes[1]));

        AssertEx.Equal("C", model.GetTypeInfo(parameterSyntaxes[2].Type).Type.ToTestDisplayString());
        Assert.Null(model.GetDeclaredSymbol(parameterSyntaxes[2]));
    }

    [Fact]
    public void ReceiverParameter_Multiple_MissingType()
    {
        var src = """
public static class Extensions
{
    extension(int i, Type) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,22): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i, Type) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "Type").WithLocation(3, 22));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Found()
    {
        var src = """
public static class Extensions
{
    extension<T>(T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        var extensionParameter = symbol.ExtensionParameter;
        AssertEx.Equal("T", extensionParameter.ToTestDisplayString());
        Assert.Same(extensionParameter.Type, symbol.TypeParameters[0]);
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Found_FromContainingType()
    {
        var src = """
public static class Extensions<T>
{
    extension(T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(T) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        var extensionParameter = symbol.ExtensionParameter;
        AssertEx.Equal("T", extensionParameter.ToTestDisplayString());
        Assert.Same(extensionParameter.Type, symbol.ContainingType.TypeParameters[0]);
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Missing()
    {
        var src = """
public static class Extensions
{
    extension(T)
    {
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS0246: The type or namespace name 'T' could not be found (are you missing a using directive or an assembly reference?)
            //     extension(T)
            Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "T").WithArguments("T").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        var parameter = symbol.ExtensionParameter;
        AssertEx.Equal("T", parameter.ToTestDisplayString());
        Assert.True(parameter.Type.IsErrorType());
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78472")]
    public void ReceiverParameter_TypeParameter_Unreferenced_01()
    {
        var src = """
int.M();
int.M<string>();

public static class Extensions
{
    extension<T>(int)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS1061: 'int' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // int.M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("int", "M").WithLocation(1, 5));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_02()
    {
        var src = """
int.M();
int.M<int, string>();

public static class Extensions
{
    extension<T1, T2>(T1)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS1061: 'int' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // int.M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("int", "M").WithLocation(1, 5));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_03()
    {
        var src = """
int.M();

public static class Extensions
{
    extension<T1, T2>(T1) where T1 : class
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS1061: 'int' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // int.M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("int", "M").WithLocation(1, 5));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_04()
    {
        var src = """
public static class E
{
    extension<T>(int i)
    {
        public int P => 0; // 1
    }

    extension<T>(C<T> c)
    {
        public int P => 0;
    }
}
public class C<T> { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,20): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public int P => 0; // 1
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "P").WithArguments("T").WithLocation(5, 20));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_05()
    {
        var src = """
public static class E
{
    extension<T>(int)
    {
        public static int P => 0; // 1
    }
    extension<T1, T2>(int)
    {
        public static int P => 0; // 2, 3
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,27): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public static int P => 0; // 1
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "P").WithArguments("T").WithLocation(5, 27),
            // (9,27): error CS9295: The type parameter `T1` is not referenced by either the extension parameter or a parameter of this member
            //         public static int P => 0; // 2, 3
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "P").WithArguments("T1").WithLocation(9, 27),
            // (9,27): error CS9295: The type parameter `T2` is not referenced by either the extension parameter or a parameter of this member
            //         public static int P => 0; // 2, 3
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "P").WithArguments("T2").WithLocation(9, 27));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_06()
    {
        var src = """
public static class E
{
    extension<T>(int i)
    {
        public int P => 0;
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,20): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public int P => 0;
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "P").WithArguments("T").WithLocation(5, 20));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_07()
    {
        var src = """
public static class E
{
    extension<T>(int i)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_08()
    {
        var src = """
public static class E
{
    extension<T>(int)
    {
        public static int operator +(int i, T t) => 0;
        public static int operator -(int i, int j) => 0;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (6,36): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public static int operator -(int i, int j) => 0;
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "-").WithArguments("T").WithLocation(6, 36));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_09()
    {
        var src = """
public static class E
{
    extension<T>(int i)
    {
        public void operator +=(T t) { }
        public void operator -=(int j) { }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (5,30): error CS9322: Cannot declare instance operator for a struct unless containing extension block receiver parameter is a 'ref' parameter
            //         public void operator +=(T t) { }
            Diagnostic(ErrorCode.ERR_InstanceOperatorStructExtensionWrongReceiverRefKind, "+=").WithLocation(5, 30),
            // (6,30): error CS9322: Cannot declare instance operator for a struct unless containing extension block receiver parameter is a 'ref' parameter
            //         public void operator -=(int j) { }
            Diagnostic(ErrorCode.ERR_InstanceOperatorStructExtensionWrongReceiverRefKind, "-=").WithLocation(6, 30),
            // (6,30): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public void operator -=(int j) { }
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "-=").WithArguments("T").WithLocation(6, 30));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_10()
    {
        var src = """
public static class E
{
    extension<T>(int i)
    {
        public static implicit operator string() => "";
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (5,41): error CS9282: Extension declarations can include only methods or properties
            //         public static implicit operator string() => "";
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "string").WithLocation(5, 41),
            // (5,41): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public static implicit operator string() => "";
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "string").WithArguments("T").WithLocation(5, 41),
            // (5,47): error CS1019: Overloadable unary operator expected
            //         public static implicit operator string() => "";
            Diagnostic(ErrorCode.ERR_OvlUnaryOperatorExpected, "()").WithLocation(5, 47));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_11()
    {
        var src = """
public static class E
{
    extension<T>(int i)
    {
        public int this[int j] => 0;
        public int this[long l, T t] => 0;
    }
    extension<T>(T t)
    {
        public int this[string s] => 0;
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (5,20): error CS9282: Extension declarations can include only methods or properties
            //         public int this[int j] => 0;
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(5, 20),
            // (5,20): error CS9295: The type parameter `T` is not referenced by either the extension parameter or a parameter of this member
            //         public int this[int j] => 0;
            Diagnostic(ErrorCode.ERR_UnderspecifiedExtension, "this").WithArguments("T").WithLocation(5, 20),
            // (6,20): error CS9282: Extension declarations can include only methods or properties
            //         public int this[long l, T t] => 0;
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(6, 20),
            // (10,20): error CS9282: Extension declarations can include only methods or properties
            //         public int this[string s] => 0;
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(10, 20));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_12()
    {
        var src = """
public static class E<T>
{
    extension<U>(T t)
    {
        public int P => 0;
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension<U>(T t)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Unreferenced_13()
    {
        var src = """
#nullable enable

public static class E
{
    extension<T>(T? t)
    {
        public int P => 0;
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_TopLevelExtension()
    {
        var src = """
extension<T>(int i)
{
    public int P => 0;
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            // extension<T>(int)
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(1, 1));
    }

    [Fact]
    public void ReceiverParameter_TypeParameter_Missing_Local()
    {
        var src = """
public static class Extensions
{
    extension(T t)
    {
        void T() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS0246: The type or namespace name 'T' could not be found (are you missing a using directive or an assembly reference?)
            //     extension(T t)
            Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "T").WithArguments("T").WithLocation(3, 15),
            // (5,14): error CS9326: 'T': extension member names cannot be the same as their extended type
            //         void T() { }
            Diagnostic(ErrorCode.ERR_MemberNameSameAsExtendedType, "T").WithArguments("T").WithLocation(5, 14));
    }

    [Fact]
    public void ReceiverParameter_Params()
    {
        var src = """
public static class Extensions
{
    extension(params int[] i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1670: params is not valid in this context
            //     extension(params int[] i) { }
            Diagnostic(ErrorCode.ERR_IllegalParams, "params").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(type1);
        var parameter = symbol1.ExtensionParameter;
        AssertEx.Equal("System.Int32[] i", parameter.ToTestDisplayString());
        Assert.False(parameter.IsParams);
    }

    [Fact]
    public void ReceiverParameter_Params_BadType()
    {
        var src = """
public static class Extensions
{
    extension(params int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1670: params is not valid in this context
            //     extension(params int i) { }
            Diagnostic(ErrorCode.ERR_IllegalParams, "params").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_Params_NotLast()
    {
        var src = """
public static class Extensions
{
    extension(params int[] i, int j) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1670: params is not valid in this context
            //     extension(params int[] i, int j) { }
            Diagnostic(ErrorCode.ERR_IllegalParams, "params").WithLocation(3, 15),
            // (3,31): error CS9285: An extension container can have only one receiver parameter
            //     extension(params int[] i, int j) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "int j").WithLocation(3, 31));
    }

    [Fact]
    public void ReceiverParameter_ParameterTypeViolatesConstraint()
    {
        var src = """
public static class Extensions
{
    extension<T>(C<T>) { }
    extension<T2>(C<T2>) where T2 : struct { }
}

public class C<T> where T : struct { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,18): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'C<T>'
            //     extension<T>(C<T>) { }
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "C<T>").WithArguments("C<T>", "T", "T").WithLocation(3, 18));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(type1);
        AssertEx.Equal("C<T>", symbol1.ExtensionParameter.ToTestDisplayString());
    }

    [Fact]
    public void ReceiverParameter_DefaultValue()
    {
        var src = """
public static class Extensions
{
    extension(int i = 0) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension(int i = 0) { }
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "int i = 0").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        Assert.True(symbol.ExtensionParameter.HasExplicitDefaultValue);
    }

    [Fact]
    public void ReceiverParameter_DefaultValue_BeforeAnotherParameter()
    {
        var src = """
public static class Extensions
{
    extension(int i = 0, object) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension(int i = 0, object) { }
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "int i = 0").WithLocation(3, 15),
            // (3,26): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i = 0, object) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "object").WithLocation(3, 26));
    }

    [Fact]
    public void ReceiverParameter_DefaultValue_BadValue()
    {
        var src = """
public static class Extensions
{
    extension(int i = null) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension(int i = null) { }
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "int i = null").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_DefaultValue_RefReadonly()
    {
        var src = """
public static class Extensions
{
    extension(ref readonly int x = 2) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension(ref readonly int x = 2) { }
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "ref readonly int x = 2").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_Attributes_01()
    {
        var src = """
public static class Extensions
{
    extension([System.Runtime.InteropServices.DefaultParameterValue(1)] int o = 2) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension([System.Runtime.InteropServices.DefaultParameterValue(1)] int o = 2) { }
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "[System.Runtime.InteropServices.DefaultParameterValue(1)] int o = 2").WithLocation(3, 15),
            // (3,16): error CS1745: Cannot specify default parameter value in conjunction with DefaultParameterAttribute or OptionalAttribute
            //     extension([System.Runtime.InteropServices.DefaultParameterValue(1)] int o = 2) { }
            Diagnostic(ErrorCode.ERR_DefaultValueUsedWithAttributes, "System.Runtime.InteropServices.DefaultParameterValue").WithLocation(3, 16));
    }

    [Fact]
    public void ReceiverParameter_Attributes_02()
    {
        var src = """
public static class Extensions
{
    extension([System.Runtime.InteropServices.Optional, System.Runtime.InteropServices.DefaultParameterValue(1)] ref readonly int i) { }
    extension([System.Runtime.InteropServices.Optional, System.Runtime.InteropServices.DefaultParameterValue(2)] ref readonly int) { }
}
""";
        var comp = CreateCompilation(src);
        // Note: we use "" name in the diagnostic for the second parameter
        // Note: these attributes are allowed on the receiver parameter of an extension method
        comp.VerifyEmitDiagnostics(
            // (3,57): warning CS9200: A default value is specified for 'ref readonly' parameter 'i', but 'ref readonly' should be used only for references. Consider declaring the parameter as 'in'.
            //     extension([System.Runtime.InteropServices.Optional, System.Runtime.InteropServices.DefaultParameterValue(1)] ref readonly int i) { }
            Diagnostic(ErrorCode.WRN_RefReadonlyParameterDefaultValue, "System.Runtime.InteropServices.DefaultParameterValue(1)").WithArguments("i").WithLocation(3, 57),
            // (4,57): warning CS9200: A default value is specified for 'ref readonly' parameter '', but 'ref readonly' should be used only for references. Consider declaring the parameter as 'in'.
            //     extension([System.Runtime.InteropServices.Optional, System.Runtime.InteropServices.DefaultParameterValue(2)] ref readonly int) { }
            Diagnostic(ErrorCode.WRN_RefReadonlyParameterDefaultValue, "System.Runtime.InteropServices.DefaultParameterValue(2)").WithArguments("").WithLocation(4, 57),
            // (4,127): error CS9305: Cannot use modifiers on the unnamed receiver parameter of extension block
            //     extension([System.Runtime.InteropServices.Optional, System.Runtime.InteropServices.DefaultParameterValue(2)] ref readonly int) { }
            Diagnostic(ErrorCode.ERR_ModifierOnUnnamedReceiverParameter, "int").WithLocation(4, 127));
    }

    [Fact]
    public void ReceiverParameter_Attributes_03()
    {
        var src = """
public static class Extensions
{
    extension([System.Runtime.CompilerServices.ParamCollectionAttribute] int[] xs) { }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (3,16): error CS0674: Do not use 'System.ParamArrayAttribute'/'System.Runtime.CompilerServices.ParamCollectionAttribute'. Use the 'params' keyword instead.
            //     extension([System.Runtime.CompilerServices.ParamCollectionAttribute] int[] xs) { }
            Diagnostic(ErrorCode.ERR_ExplicitParamArrayOrCollection, "System.Runtime.CompilerServices.ParamCollectionAttribute").WithLocation(3, 16));
    }

    [Fact]
    public void ReceiverParameter_Attributes_04()
    {
        var src = """
public static class Extensions
{
    extension([System.Runtime.CompilerServices.CallerLineNumber] int x = 2) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension([System.Runtime.CompilerServices.CallerLineNumber] int x = 2) { }
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "[System.Runtime.CompilerServices.CallerLineNumber] int x = 2").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_Attributes_54()
    {
        var src = """
public static class Extensions
{
    extension([System.Runtime.CompilerServices.CallerLineNumber] int x) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,16): error CS4020: The CallerLineNumberAttribute may only be applied to parameters with default values
            //     extension([System.Runtime.CompilerServices.CallerLineNumber] int x) { }
            Diagnostic(ErrorCode.ERR_BadCallerLineNumberParamWithoutDefaultValue, "System.Runtime.CompilerServices.CallerLineNumber").WithLocation(3, 16));
    }

    [Fact]
    public void ReceiverParameter_Attributes_06()
    {
        var src = """
public static class Extensions
{
    extension([My] int x) { }
}
public class MyAttribute : System.Attribute { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var parameter = tree.GetRoot().DescendantNodes().OfType<ParameterSyntax>().Single();
        var parameterSymbol = model.GetDeclaredSymbol(parameter);
        AssertEx.Equal("System.Int32 x", parameterSymbol.ToTestDisplayString());
        AssertEx.SetEqual(["MyAttribute"], parameterSymbol.GetAttributes().Select(a => a.ToString()));

        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var extensionSymbol = model.GetDeclaredSymbol(type);
        AssertEx.SetEqual(["MyAttribute"], extensionSymbol.ExtensionParameter.GetAttributes().Select(a => a.ToString()));
    }

    [Fact]
    public void ReceiverParameter_This_01()
    {
        var src = """
public static class Extensions
{
    extension(this int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS0027: Keyword 'this' is not available in the current context
            //     extension(this int i) { }
            Diagnostic(ErrorCode.ERR_ThisInBadContext, "this").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_This_02()
    {
        var src = """
public static class Extensions
{
    extension(int i, this int j) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,22): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i, this int j) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "this int j").WithLocation(3, 22));
    }

    [Fact]
    public void ReceiverParameter_This_03()
    {
        var src = """
public static class Extensions
{
    extension(this this int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,20): error CS1107: A parameter can only have one 'this' modifier
            //     extension(this this int i) { }
            Diagnostic(ErrorCode.ERR_DupParamMod, "this").WithArguments("this").WithLocation(3, 20),
            // (3,20): error CS0027: Keyword 'this' is not available in the current context
            //     extension(this this int i) { }
            Diagnostic(ErrorCode.ERR_ThisInBadContext, "this").WithLocation(3, 20));
    }

    [Fact]
    public void ReceiverParameter_Ref_01()
    {
        var src = """
int i = 42;
i.M();
System.Console.Write(i);

public static class Extensions
{
    extension(ref int i)
    {
        public void M() { System.Console.Write(i); i = 43; }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "4243").VerifyDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_Ref_02()
    {
        var src = """
public static class Extensions
{
    extension(ref ref int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): error CS1107: A parameter can only have one 'ref' modifier
            //     extension(ref ref int i) { }
            Diagnostic(ErrorCode.ERR_DupParamMod, "ref").WithArguments("ref").WithLocation(3, 19));
    }

    [Fact]
    public void ReceiverParameter_Out_01()
    {
        var src = """
public static class Extensions
{
    extension(out int i)
    {
        void M2() { }
        static void M3() { }
    }
    static void M(this out int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS8328:  The parameter modifier 'out' cannot be used with 'extension'
            //     extension(out int i)
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "extension").WithLocation(3, 15),
            // (5,14): error CS0177: The out parameter 'i' must be assigned to before control leaves the current method
            //         void M2() { }
            Diagnostic(ErrorCode.ERR_ParamUnassigned, "M2").WithArguments("i").WithLocation(5, 14),
            // (8,17): error CS0177: The out parameter 'i' must be assigned to before control leaves the current method
            //     static void M(this out int i) { }
            Diagnostic(ErrorCode.ERR_ParamUnassigned, "M").WithArguments("i").WithLocation(8, 17),
            // (8,24): error CS8328:  The parameter modifier 'out' cannot be used with 'this'
            //     static void M(this out int i) { }
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "this").WithLocation(8, 24));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type1 = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().First();
        var symbol1 = model.GetDeclaredSymbol(type1);
        var parameter = symbol1.ExtensionParameter;
        AssertEx.Equal("out System.Int32 i", parameter.ToTestDisplayString());
        Assert.Equal(RefKind.Out, parameter.RefKind);
    }

    [Fact]
    public void ReceiverParameter_Out_02()
    {
        var src = """
public static class Extensions
{
    extension(int i, out int j) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,22): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i, out int j) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "out int j").WithLocation(3, 22));
    }

    [Fact]
    public void ReceiverParameter_Out_03()
    {
        var src = """
public static class Extensions
{
    extension(out out int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS8328:  The parameter modifier 'out' cannot be used with 'extension'
            //     extension(out out int i) { }
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "extension").WithLocation(3, 15),
            // (3,19): error CS8328:  The parameter modifier 'out' cannot be used with 'extension'
            //     extension(out out int i) { }
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "extension").WithLocation(3, 19));
    }

    [Fact]
    public void ReceiverParameter_Out_04()
    {
        var src = """
public static class Extensions
{
    extension(out int i)
    {
        void M2(bool b) { if (b) return; else return; }
    }
    static void M(this out int i, bool b) { if (b) return; else return; }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS8328:  The parameter modifier 'out' cannot be used with 'extension'
            //     extension(out int i)
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "extension").WithLocation(3, 15),
            // (5,34): error CS0177: The out parameter 'i' must be assigned to before control leaves the current method
            //         void M2(bool b) { if (b) return; else return; }
            Diagnostic(ErrorCode.ERR_ParamUnassigned, "return;").WithArguments("i").WithLocation(5, 34),
            // (5,47): error CS0177: The out parameter 'i' must be assigned to before control leaves the current method
            //         void M2(bool b) { if (b) return; else return; }
            Diagnostic(ErrorCode.ERR_ParamUnassigned, "return;").WithArguments("i").WithLocation(5, 47),
            // (7,24): error CS8328:  The parameter modifier 'out' cannot be used with 'this'
            //     static void M(this out int i, bool b) { if (b) return; else return; }
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "this").WithLocation(7, 24),
            // (7,52): error CS0177: The out parameter 'i' must be assigned to before control leaves the current method
            //     static void M(this out int i, bool b) { if (b) return; else return; }
            Diagnostic(ErrorCode.ERR_ParamUnassigned, "return;").WithArguments("i").WithLocation(7, 52),
            // (7,65): error CS0177: The out parameter 'i' must be assigned to before control leaves the current method
            //     static void M(this out int i, bool b) { if (b) return; else return; }
            Diagnostic(ErrorCode.ERR_ParamUnassigned, "return;").WithArguments("i").WithLocation(7, 65));
    }

    [Fact]
    public void ReceiverParameter_Out_05()
    {
        var src = """
public static class Extensions
{
    extension(out int i)
    {
        void M2(bool b) { i = 0; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS8328:  The parameter modifier 'out' cannot be used with 'extension'
            //     extension(out int i)
            Diagnostic(ErrorCode.ERR_BadParameterModifiers, "out").WithArguments("out", "extension").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_In_01()
    {
        var src = """
public static class Extensions
{
    extension(in int i) { }
    static void M(this in int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_In_02()
    {
        var src = """
public static class Extensions
{
    extension(in in int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,18): error CS1107: A parameter can only have one 'in' modifier
            //     extension(in in int i) { }
            Diagnostic(ErrorCode.ERR_DupParamMod, "in").WithArguments("in").WithLocation(3, 18));
    }

    [Fact]
    public void ReceiverParameter_In_03()
    {
        var src = """
public static class Extensions
{
    extension(in int i) { }
    extension(in object o) { }
    extension<T>(in T t) { }
    extension<T>(in T t) where T : struct { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (4,18): error CS9301: The 'in' or 'ref readonly' receiver parameter of extension must be a concrete (non-generic) value type.
            //     extension(in object o) { }
            Diagnostic(ErrorCode.ERR_InExtensionParameterMustBeValueType, "object").WithLocation(4, 18),
            // (5,21): error CS9301: The 'in' or 'ref readonly' receiver parameter of extension must be a concrete (non-generic) value type.
            //     extension<T>(in T t) { }
            Diagnostic(ErrorCode.ERR_InExtensionParameterMustBeValueType, "T").WithLocation(5, 21),
            // (6,21): error CS9301: The 'in' or 'ref readonly' receiver parameter of extension must be a concrete (non-generic) value type.
            //     extension<T>(in T t) where T : struct { }
            Diagnostic(ErrorCode.ERR_InExtensionParameterMustBeValueType, "T").WithLocation(6, 21));
    }

    [Fact]
    public void ReceiverParameter_RefReadonly_01()
    {
        var src = """
int i = 42;
i.M();

public static class Extensions
{
    extension(ref readonly int i)
    {
        public void M() { System.Console.Write(i); }
    }
    static void M2(this ref readonly int i) { }
}
""";
        var comp = CreateCompilation(src);

        CompileAndVerify(comp, symbolValidator: (m) =>
        {
            Assert.Equal(RefKind.RefReadOnlyParameter, m.GlobalNamespace.GetTypeMember("Extensions").GetTypeMembers().Single().ExtensionParameter.RefKind);
        }, expectedOutput: "42").VerifyDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_RefReadonly_02()
    {
        var src = """
public static class E
{
    extension(ref readonly int i) { }
    extension(ref readonly object o) { }
    extension<T>(ref readonly T o) { }
    extension<T>(ref readonly T o) where T : struct { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (4,28): error CS9301: The 'in' or 'ref readonly' receiver parameter of extension must be a concrete (non-generic) value type.
            //     extension(ref readonly object o) { }
            Diagnostic(ErrorCode.ERR_InExtensionParameterMustBeValueType, "object").WithLocation(4, 28),
            // (5,31): error CS9301: The 'in' or 'ref readonly' receiver parameter of extension must be a concrete (non-generic) value type.
            //     extension<T>(ref readonly T o) { }
            Diagnostic(ErrorCode.ERR_InExtensionParameterMustBeValueType, "T").WithLocation(5, 31),
            // (6,31): error CS9301: The 'in' or 'ref readonly' receiver parameter of extension must be a concrete (non-generic) value type.
            //     extension<T>(ref readonly T o) where T : struct { }
            Diagnostic(ErrorCode.ERR_InExtensionParameterMustBeValueType, "T").WithLocation(6, 31));
    }

    [Fact]
    public void ReceiverParameter_ReadonlyRef()
    {
        var src = """
public static class Extensions
{
    extension(readonly ref int i) { }
    static void M(this readonly ref int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9190: 'readonly' modifier must be specified after 'ref'.
            //     extension(readonly ref int i) { }
            Diagnostic(ErrorCode.ERR_RefReadOnlyWrongOrdering, "readonly").WithLocation(3, 15),
            // (4,24): error CS9190: 'readonly' modifier must be specified after 'ref'.
            //     static void M(this readonly ref int i) { }
            Diagnostic(ErrorCode.ERR_RefReadOnlyWrongOrdering, "readonly").WithLocation(4, 24));
    }

    [Fact]
    public void ReceiverParameter_ArgList_01()
    {
        var src = """
_ = object.M();

public static class Extensions
{
    extension(__arglist)
    {
        void M(){}
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,12): error CS0117: 'object' does not contain a definition for 'M'
            // _ = object.M();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 12),
            // (5,15): error CS1669: __arglist is not valid in this context
            //     extension(__arglist)
            Diagnostic(ErrorCode.ERR_IllegalVarArgs, "__arglist").WithLocation(5, 15));

        Assert.Empty(comp.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>());

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var o = ((Compilation)comp).GetSpecialType(SpecialType.System_Object);
        AssertEqualAndNoDuplicates(_objectMembers, model.LookupSymbols(position: 0, o, name: null, includeReducedExtensionMethods: true).ToTestDisplayStrings());
    }

    [Fact]
    public void ReceiverParameter_ArgList_02()
    {
        var src = """
public static class Extensions
{
    extension(__arglist, int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1669: __arglist is not valid in this context
            //     extension(__arglist, int i) { }
            Diagnostic(ErrorCode.ERR_IllegalVarArgs, "__arglist").WithLocation(3, 15),
            // (3,26): error CS9285: An extension container can have only one receiver parameter
            //     extension(__arglist, int i) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "int i").WithLocation(3, 26));
    }

    [Fact]
    public void ReceiverParameter_StaticType_01()
    {
        var src = """
public static class Extensions
{
    extension(Extensions) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_StaticType_02()
    {
        var src = """
public static class Extensions
{
    extension(object o, Extensions e) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,25): error CS9285: An extension container can have only one receiver parameter
            //     extension(object o, Extensions e) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "Extensions e").WithLocation(3, 25));
    }

    [Fact]
    public void ReceiverParameter_StaticType_03()
    {
        var src = """
public static class Extensions
{
    extension(Extensions e) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS0721: 'Extensions': static types cannot be used as parameters
            //     extension(Extensions) { }
            Diagnostic(ErrorCode.ERR_ParameterIsStaticClass, "Extensions").WithArguments("Extensions").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_StaticType_04()
    {
        var src = """
extension(C) { }
extension(C c) { }
public static class C { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            // extension(C) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(1, 1),
            // (2,1): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            // extension(C c) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(2, 1),
            // (2,11): error CS0721: 'C': static types cannot be used as parameters
            // extension(C c) { }
            Diagnostic(ErrorCode.ERR_ParameterIsStaticClass, "C").WithArguments("C").WithLocation(2, 11));
    }

    [Fact]
    public void ReceiverParameter_StaticType_05()
    {
        var src = """
static class Extensions
{
    extension(C)
    {
    }
}

static class C {}
""";
        var comp = CreateCompilation(src);

        CompileAndVerify(comp).VerifyDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_InstanceType_01()
    {
        var src = """
public static class Extensions
{
    extension(C c) { }
    extension(C) { }
}
public class C { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_ShadowingTypeParameter()
    {
        var src = """
public static class Extensions<T>
{
    extension(object T) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,5): error CS9283: Extensions must be declared in a top-level, non-generic, static class
            //     extension(object T) { }
            Diagnostic(ErrorCode.ERR_BadExtensionContainingType, "extension").WithLocation(3, 5));
    }

    [Fact]
    public void ReceiverParameter_Ref()
    {
        var src = """
int i = 42;
i.M(43);

public static class Extensions
{
    extension(ref int i)
    {
        public void M(int j) { System.Console.Write((i, j)); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        src = """
static class E
{
    extension(ref string s) // 1
    {
        public static void M() { }
        public static int P => 0;
    }
    extension<T>(ref T t) // 2
    {
        public static void M2() { }
        public static int P2 => 0;
    }

    extension(ref int i)
    {
        public static void M3() { }
        public static int P3 => 0;
    }
    extension<T>(ref T t) where T : struct
    {
        public static void M4() { }
        public static int P4 => 0;
    }
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): error CS9300: The 'ref' receiver parameter of an extension block must be a value type or a generic type constrained to struct.
            //     extension(ref string s) // 1
            Diagnostic(ErrorCode.ERR_RefExtensionParameterMustBeValueTypeOrConstrainedToOne, "string").WithLocation(3, 19),
            // (8,22): error CS9300: The 'ref' receiver parameter of an extension block must be a value type or a generic type constrained to struct.
            //     extension<T>(ref T t) // 2
            Diagnostic(ErrorCode.ERR_RefExtensionParameterMustBeValueTypeOrConstrainedToOne, "T").WithLocation(8, 22));
    }

    [Fact]
    public void ReceiverParameter_Scoped_01()
    {
        var src = """
public static class Extensions
{
    extension(scoped int i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS9048: The 'scoped' modifier can be used for refs and ref struct values only.
            //     extension(scoped int i) { }
            Diagnostic(ErrorCode.ERR_ScopedRefAndRefStructOnly, "scoped int i").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_Scoped_02()
    {
        var src = """
public static class Extensions
{
    extension(int i, scoped int j) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,22): error CS9285: An extension container can have only one receiver parameter
            //     extension(int i, scoped int j) { }
            Diagnostic(ErrorCode.ERR_ReceiverParameterOnlyOne, "scoped int j").WithLocation(3, 22));
    }

    [Fact]
    public void ReceiverParameter_Scoped_03()
    {
        var src = """
public static class Extensions
{
    extension(scoped System.Span<int> i) { }
    public static void M(this scoped System.Span<int> i) { }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_Nullable_01()
    {
        var src = """
public static class Extensions
{
    extension(string?) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,21): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
            //     extension(string?) { }
            Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(3, 21));
    }

    [Fact]
    public void ReceiverParameter_Nullable_02()
    {
        var src = """
#nullable enable
public static class Extensions
{
    extension(string?) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_RestrictedType()
    {
        var src = """
public static class Extensions
{
    extension(ref System.ArgIterator a) { }
    extension(ref System.Span<int> s) { }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1601: Cannot make reference to variable of type 'ArgIterator'
            //     extension(ref System.ArgIterator a) { }
            Diagnostic(ErrorCode.ERR_MethodArgCantBeRefAny, "ref System.ArgIterator a").WithArguments("System.ArgIterator").WithLocation(3, 15));
    }

    [Fact]
    public void ReceiverParameter_Empty()
    {
        var src = """
public static class Extensions
{
    extension() { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1031: Type expected
            //     extension() { }
            Diagnostic(ErrorCode.ERR_TypeExpected, ")").WithLocation(3, 15));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var type = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var symbol = model.GetDeclaredSymbol(type);
        AssertEx.Equal("?", symbol.ExtensionParameter.ToTestDisplayString());
        Assert.True(symbol.ExtensionParameter.Type.IsErrorType());
    }

    [Fact]
    public void ReceiverParameter_ConstraintsCheck()
    {
        var src = """
static class Extensions
{
    extension(System.Nullable<string> receiver)
    {
    }

    extension(System.Nullable<string>)
    {
    }
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (3,39): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
            //     extension(System.Nullable<string> receiver)
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "receiver").WithArguments("System.Nullable<T>", "T", "string").WithLocation(3, 39),
            // (7,15): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
            //     extension(System.Nullable<string>)
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "System.Nullable<string>").WithArguments("System.Nullable<T>", "T", "string").WithLocation(7, 15)
            );
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78491")]
    public void ReceiverParameter_RefScope_01()
    {
        var src = """
int i = 42;
i.M();

static class Extensions
{
    extension(scoped ref int receiver)
    {
        public void M() => System.Console.Write(receiver);
    }
}
""";
        var comp = CreateCompilation(src);

        CompileAndVerify(comp, expectedOutput: "42", symbolValidator: (m) =>
        {
            AssertEx.Equal(ScopedKind.ScopedRef, m.GlobalNamespace.GetTypeMember("Extensions").GetTypeMembers().Single().ExtensionParameter.EffectiveScope);
        }).VerifyDiagnostics();
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78491")]
    public void ReceiverParameter_RefScope_02()
    {
        var src = """
int i = 42;
i.M();

static class Extensions
{
    extension(scoped ref int receiver)
    {
        public ref int M() => ref receiver;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (8,35): error CS9075: Cannot return a parameter by reference 'receiver' because it is scoped to the current method
            //         public ref int M() => ref receiver;
            Diagnostic(ErrorCode.ERR_RefReturnScopedParameter, "receiver").WithArguments("receiver").WithLocation(8, 35));
    }

    [Fact]
    public void ReceiverParameter_Nullability()
    {
        var src = """
#nullable enable

static class Extensions
{
    extension(string? receiver)
    {
    }

    extension(string?)
    {
    }
}
""";
        var comp = CreateCompilation(src);

        CompileAndVerify(comp, symbolValidator: (m) =>
        {
            var extensions = m.GlobalNamespace.GetTypeMember("Extensions").GetTypeMembers();
            AssertEx.Equal("System.String?", extensions[0].ExtensionParameter.TypeWithAnnotations.ToTestDisplayString());
            AssertEx.Equal("System.String?", extensions[1].ExtensionParameter.TypeWithAnnotations.ToTestDisplayString());
        }).VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var parameters = tree.GetRoot().DescendantNodes().OfType<ParameterSyntax>().ToArray();
        AssertEx.Equal("System.String? receiver", model.GetDeclaredSymbol(parameters[0]).ToTestDisplayString(includeNonNullable: true));
        AssertEx.Equal("System.String?", model.GetDeclaredSymbol(parameters[1]).ToTestDisplayString(includeNonNullable: true));
    }

    [Fact]
    public void ReceiverParameter_NativeInteger()
    {
        var src = """
#nullable enable

static class Extensions
{
    extension(nint receiver)
    {
    }

    extension(nint)
    {
    }
}
""";
        var comp = CreateCompilation(src);

        CompileAndVerify(comp, symbolValidator: (m) =>
        {
            var extensions = m.GlobalNamespace.GetTypeMember("Extensions").GetTypeMembers();
            Assert.True(extensions[0].ExtensionParameter.Type.IsNativeIntegerType);
            Assert.True(extensions[1].ExtensionParameter.Type.IsNativeIntegerType);
        }).VerifyDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_InconsistentTypeAccessibility_01()
    {
        var src = """
public static class Extensions
{
    extension(C x)
    {
        public void M() {}
        public int P { get => 0; set {}}
        public int this[int i] { get => 0; set {}}

        private void M1() {}
        private int P1 { get => 0; set {}}
        private int this[long i] { get => 0; set {}}

        internal void M2() {}
        internal int P2 { get => 0; set {}}
        internal int this[byte i] { get => 0; set {}}
    }
}

class C {}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (5,21): error CS0051: Inconsistent accessibility: parameter type 'C' is less accessible than method 'Extensions.extension(C).M()'
            //         public void M() {}
            Diagnostic(ErrorCode.ERR_BadVisParamType, "M").WithArguments("Extensions.extension(C).M()", "C").WithLocation(5, 21),
            // (6,20): error CS0055: Inconsistent accessibility: parameter type 'C' is less accessible than indexer or property 'Extensions.extension(C).P'
            //         public int P { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "P").WithArguments("Extensions.extension(C).P", "C").WithLocation(6, 20),
            // (7,20): error CS9282: This member is not allowed in an extension block
            //         public int this[int i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(7, 20),
            // (7,20): error CS0055: Inconsistent accessibility: parameter type 'C' is less accessible than indexer or property 'Extensions.extension(C).this[int]'
            //         public int this[int i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "this").WithArguments("Extensions.extension(C).this[int]", "C").WithLocation(7, 20),
            // (11,21): error CS9282: This member is not allowed in an extension block
            //         private int this[long i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(11, 21),
            // (15,22): error CS9282: This member is not allowed in an extension block
            //         internal int this[byte i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(15, 22)
            );
    }

    [Fact]
    public void ReceiverParameter_InconsistentTypeAccessibility_02()
    {
        var src = """
public static class Extensions
{
    extension(C x)
    {
    }

    private class C {}
}
""";
        var comp = CreateCompilation(src);

        CompileAndVerify(comp).VerifyDiagnostics().VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested private auto ansi beforefieldinit C
        extends [mscorlib]System.Object
    {
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x2067
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method C::.ctor
    } // end of class C
    .class nested public auto ansi sealed specialname '<G>$C3CD11E70DE99F353AE602995BB874BF'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$4D270477BCDFAB12B9E9B1A79213B9FB'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    class Extensions/C x
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x206f
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$4D270477BCDFAB12B9E9B1A79213B9FB'::'<Extension>$'
        } // end of class <M>$4D270477BCDFAB12B9E9B1A79213B9FB
    } // end of class <G>$C3CD11E70DE99F353AE602995BB874BF
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));
    }

    [Fact]
    public void ReceiverParameter_InconsistentTypeAccessibility_03()
    {
        var src = """
public static class Extensions
{
    extension(C x)
    {
        public void M() {}
        public int P { get => 0; set {}}
        public int this[int i] { get => 0; set {}}

        private void M1() {}
        private int P1 { get => 0; set {}}
        private int this[long i] { get => 0; set {}}

        internal void M2() {}
        internal int P2 { get => 0; set {}}
        internal int this[byte i] { get => 0; set {}}
    }

    private class C {}
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (5,21): error CS0051: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than method 'Extensions.extension(Extensions.C).M()'
            //         public void M() {}
            Diagnostic(ErrorCode.ERR_BadVisParamType, "M").WithArguments("Extensions.extension(Extensions.C).M()", "Extensions.C").WithLocation(5, 21),
            // (6,20): error CS0055: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than indexer or property 'Extensions.extension(Extensions.C).P'
            //         public int P { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "P").WithArguments("Extensions.extension(Extensions.C).P", "Extensions.C").WithLocation(6, 20),
            // (7,20): error CS9282: This member is not allowed in an extension block
            //         public int this[int i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(7, 20),
            // (7,20): error CS0055: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than indexer or property 'Extensions.extension(Extensions.C).this[int]'
            //         public int this[int i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "this").WithArguments("Extensions.extension(Extensions.C).this[int]", "Extensions.C").WithLocation(7, 20),
            // (11,21): error CS9282: This member is not allowed in an extension block
            //         private int this[long i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(11, 21),
            // (13,23): error CS0051: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than method 'Extensions.extension(Extensions.C).M2()'
            //         internal void M2() {}
            Diagnostic(ErrorCode.ERR_BadVisParamType, "M2").WithArguments("Extensions.extension(Extensions.C).M2()", "Extensions.C").WithLocation(13, 23),
            // (14,22): error CS0055: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than indexer or property 'Extensions.extension(Extensions.C).P2'
            //         internal int P2 { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "P2").WithArguments("Extensions.extension(Extensions.C).P2", "Extensions.C").WithLocation(14, 22),
            // (15,22): error CS9282: This member is not allowed in an extension block
            //         internal int this[byte i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(15, 22),
            // (15,22): error CS0055: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than indexer or property 'Extensions.extension(Extensions.C).this[byte]'
            //         internal int this[byte i] { get => 0; set {}}
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "this").WithArguments("Extensions.extension(Extensions.C).this[byte]", "Extensions.C").WithLocation(15, 22)
            );
    }

    [Fact]
    public void ReceiverParameter_InconsistentTypeAccessibility_04()
    {
        var src = """
public static class Extensions
{
    extension(C x)
    {
        public static void M() { }
        public static int P { get => 0; set { } }
    }

    private class C { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,28): error CS0051: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than method 'Extensions.extension(Extensions.C).M()'
            //         public static void M() { }
            Diagnostic(ErrorCode.ERR_BadVisParamType, "M").WithArguments("Extensions.extension(Extensions.C).M()", "Extensions.C").WithLocation(5, 28),
            // (6,27): error CS0055: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than indexer 'Extensions.extension(Extensions.C).P'
            //         public static int P { get => 0; set { } }
            Diagnostic(ErrorCode.ERR_BadVisIndexerParam, "P").WithArguments("Extensions.extension(Extensions.C).P", "Extensions.C").WithLocation(6, 27));
    }

    [Fact]
    public void InconsistentTypeAccessibility_01()
    {
        var src = """
public static class Extensions
{
    extension(int x)
    {
        public void M1(C c) {}
        private void M2(C c) {}
    }

    extension(long x)
    {
        public static void M3(int y)
        {
            y.M2(new C());
        }
    }

    public static void M4(int x)
    {
        x.M2(new C());
    }

    private class C {}
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (5,21): error CS0051: Inconsistent accessibility: parameter type 'Extensions.C' is less accessible than method 'Extensions.extension(int).M1(Extensions.C)'
            //         public void M1(C c) {}
            Diagnostic(ErrorCode.ERR_BadVisParamType, "M1").WithArguments("Extensions.extension(int).M1(Extensions.C)", "Extensions.C").WithLocation(5, 21)
            );
    }

    [Fact]
    public void InconsistentTypeAccessibility_02()
    {
        var src = """
public static class Extensions
{
    extension(int x)
    {
        public C M1 => null;
        private C M2 => null;
    }

    extension(long x)
    {
        public static void M3(int y)
        {
            _ = y.M2;
        }
    }

    public static void M4(int x)
    {
        _ = x.M2;
    }

    private class C {}
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (5,18): error CS0053: Inconsistent accessibility: property type 'Extensions.C' is less accessible than property 'Extensions.extension(int).M1'
            //         public C M1 => null;
            Diagnostic(ErrorCode.ERR_BadVisPropertyType, "M1").WithArguments("Extensions.extension(int).M1", "Extensions.C").WithLocation(5, 18)
            );
    }

    [Fact]
    public void Inaccessible_01()
    {
        var src = """
static class Extensions
{
    extension(int x)
    {
        private void M2() {}
    }

    private static void M3(this int x) {}
}

class C
{
    void Test(int x)
    {
        x.M2();
        x.M3();
        Extensions.M2(x);
        Extensions.M3(x);
    }
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (15,11): error CS1061: 'int' does not contain a definition for 'M2' and no accessible extension method 'M2' accepting a first argument of type 'int' could be found (are you missing a using directive or an assembly reference?)
            //         x.M2();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M2").WithArguments("int", "M2").WithLocation(15, 11),
            // (16,11): error CS1061: 'int' does not contain a definition for 'M3' and no accessible extension method 'M3' accepting a first argument of type 'int' could be found (are you missing a using directive or an assembly reference?)
            //         x.M3();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M3").WithArguments("int", "M3").WithLocation(16, 11),
            // (17,20): error CS0122: 'Extensions.M2(int)' is inaccessible due to its protection level
            //         Extensions.M2(x);
            Diagnostic(ErrorCode.ERR_BadAccess, "M2").WithArguments("Extensions.M2(int)").WithLocation(17, 20),
            // (18,20): error CS0122: 'Extensions.M3(int)' is inaccessible due to its protection level
            //         Extensions.M3(x);
            Diagnostic(ErrorCode.ERR_BadAccess, "M3").WithArguments("Extensions.M3(int)").WithLocation(18, 20)
            );
    }

    [Fact]
    public void ReceiverParameter_FileType_01()
    {
        var src = """
file class C {}

static class Extensions
{
    extension(C x)
    {
    }

    private static void M3(this C x) {}
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (5,14): error CS9051: File-local type 'C' cannot be used in a member signature in non-file-local type 'Extensions'.
            //     extension(C x)
            Diagnostic(ErrorCode.ERR_FileTypeDisallowedInSignature, "(").WithArguments("C", "Extensions").WithLocation(5, 14),
            // (9,25): error CS9051: File-local type 'C' cannot be used in a member signature in non-file-local type 'Extensions'.
            //     private static void M3(this C x) {}
            Diagnostic(ErrorCode.ERR_FileTypeDisallowedInSignature, "M3").WithArguments("C", "Extensions").WithLocation(9, 25)
            );
    }

    [Fact]
    public void ReceiverParameter_FileType_02()
    {
        var src = """
file class C {}

file static class Extensions
{
    extension(C x)
    {
        public void M1() {}
        public int P => 0;
        public int this[int i] => 0;
    }

    public static void M2(this C x) {}

    private static void M3(this C x) {}
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyEmitDiagnostics(
            // (9,20): error CS9282: This member is not allowed in an extension block
            //         public int this[int i] => 0;
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(9, 20));
    }

    [Fact]
    public void FileType_01()
    {
        var src = """
file class C {}

file static class Extensions
{
    extension(int x)
    {
        public void M1(C c) {}
        public C P => null;
        public C this[int y]  => null;
        public int this[C y]  => 0;
    }

    public static void M2(this int x, C c) {}
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyEmitDiagnostics(
            // (9,18): error CS9282: This member is not allowed in an extension block
            //         public C this[int y]  => null;
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(9, 18),
            // (10,20): error CS9282: This member is not allowed in an extension block
            //         public int this[C y]  => 0;
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(10, 20));
    }

    [Fact]
    public void ReceiverParameter_PointerType()
    {
        string source = """
unsafe static class E
{
    extension(int* i)
    {
        public static void M() { }
        public static int P => 0;
    }
    public static void M2(this int* i) { }
}
""";
        // Note: we disallow even if the extension block only has static members
        var comp = CreateCompilation(source, options: TestOptions.UnsafeDebugDll);
        comp.VerifyEmitDiagnostics(
            // (3,15): error CS1103: The receiver parameter of an extension cannot be of type 'int*'
            //     extension(int* i)
            Diagnostic(ErrorCode.ERR_BadTypeforThis, "int*").WithArguments("int*").WithLocation(3, 15),
            // (8,32): error CS1103: The receiver parameter of an extension cannot be of type 'int*'
            //     public static void M2(this int* i) { }
            Diagnostic(ErrorCode.ERR_BadTypeforThis, "int*").WithArguments("int*").WithLocation(8, 32));

        NamedTypeSymbol e = comp.GlobalNamespace.GetTypeMember("E");
        Assert.IsType<SourceNamedTypeSymbol>(e);
        Assert.False(e.IsExtension);
        Assert.Null(e.ExtensionGroupingName);
        Assert.Null(e.ExtensionMarkerName);
    }

    [Fact]
    public void Skeleton()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var verifier = CompileAndVerify(comp);
        verifier.VerifyIL("Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", """
{
  // Code size        2 (0x2)
  .maxstack  1
  IL_0000:  ldnull
  IL_0001:  throw
}
""");

        verifier.VerifyIL("Extensions.M", """
{
  // Code size       11 (0xb)
  .maxstack  1
  IL_0000:  ldstr      "ran"
  IL_0005:  call       "void System.Console.Write(string)"
  IL_000a:  ret
}
""");
    }

    [Fact]
    public void GetDiagnosticsForSpan_NoReceiverParameter()
    {
        var src = """
public static class Extensions
{
    extension(__arglist)
    {
        public int M()
        {
            return "";
        }
    }
}
""";
        var comp = CreateCompilation(src);

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var ext = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var m = ext.DescendantNodes().OfType<MethodDeclarationSyntax>().Single();

        model.GetDiagnostics(ext.ParameterList.Span).Verify(
            // (3,15): error CS1669: __arglist is not valid in this context
            //     extension(__arglist)
            Diagnostic(ErrorCode.ERR_IllegalVarArgs, "__arglist").WithLocation(3, 15)
            );

        model.GetDiagnostics(m.Body.Span).Verify(
            // (7,20): error CS0029: Cannot implicitly convert type 'string' to 'int'
            //             return "";
            Diagnostic(ErrorCode.ERR_NoImplicitConv, @"""""").WithArguments("string", "int").WithLocation(7, 20)
            );

        comp.VerifyDiagnostics(
            // (7,20): error CS0029: Cannot implicitly convert type 'string' to 'int'
            //             return "";
            Diagnostic(ErrorCode.ERR_NoImplicitConv, @"""""").WithArguments("string", "int").WithLocation(7, 20),
            // (3,15): error CS1669: __arglist is not valid in this context
            //     extension(__arglist)
            Diagnostic(ErrorCode.ERR_IllegalVarArgs, "__arglist").WithLocation(3, 15)
            );
    }

    [Fact]
    public void GetDiagnosticsForSpan_WithReceiverParameter()
    {
        var src = """
public static class Extensions
{
    extension(object o = null)
    {
        public int M()
        {
            return "";
        }
    }
}
""";
        var comp = CreateCompilation(src);

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var ext = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();
        var m = ext.DescendantNodes().OfType<MethodDeclarationSyntax>().Single();

        model.GetDiagnostics(ext.ParameterList.Span).Verify(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension(object o = null)
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "object o = null").WithLocation(3, 15)
            );

        model.GetDiagnostics(m.Body.Span).Verify(
            // (7,20): error CS0029: Cannot implicitly convert type 'string' to 'int'
            //             return "";
            Diagnostic(ErrorCode.ERR_NoImplicitConv, @"""""").WithArguments("string", "int").WithLocation(7, 20)
            );

        comp.VerifyDiagnostics(
            // (3,15): error CS9284: The receiver parameter of an extension cannot have a default value
            //     extension(object o = null)
            Diagnostic(ErrorCode.ERR_ExtensionParameterDisallowsDefaultValue, "object o = null").WithLocation(3, 15),
            // (7,20): error CS0029: Cannot implicitly convert type 'string' to 'int'
            //             return "";
            Diagnostic(ErrorCode.ERR_NoImplicitConv, @"""""").WithArguments("string", "int").WithLocation(7, 20)
            );

        AssertEx.Equal("[System.Object o = null]", model.GetDeclaredSymbol(ext.ParameterList.Parameters[0]).ToTestDisplayString());
    }

    [Fact]
    public void ReceiverInScopeButIllegalInStaticMember()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        static object M1() => o;
        static object M2() { return o; }
        static object P1 => o;
        static object P2 { get { return o; } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,31): error CS9293: Cannot use extension parameter 'object o' in this context.
            //         static object M1() => o;
            Diagnostic(ErrorCode.ERR_InvalidExtensionParameterReference, "o").WithArguments("object o").WithLocation(5, 31),
            // (6,37): error CS9293: Cannot use extension parameter 'object o' in this context.
            //         static object M2() { return o; }
            Diagnostic(ErrorCode.ERR_InvalidExtensionParameterReference, "o").WithArguments("object o").WithLocation(6, 37),
            // (7,29): error CS9293: Cannot use extension parameter 'object o' in this context.
            //         static object P1 => o;
            Diagnostic(ErrorCode.ERR_InvalidExtensionParameterReference, "o").WithArguments("object o").WithLocation(7, 29),
            // (8,41): error CS9293: Cannot use extension parameter 'object o' in this context.
            //         static object P2 { get { return o; } }
            Diagnostic(ErrorCode.ERR_InvalidExtensionParameterReference, "o").WithArguments("object o").WithLocation(8, 41)
            );
    }

    [Fact]
    public void PassingValueForARefReceiver_01()
    {
        var src = """
public class C
{
    static void Main()
    {
        GetInt().M1();
        GetInt().M2();
        _ = GetInt().P;
    }

    static int GetInt() => 0;
}

static class Extensions
{
    extension(ref int receiver)
    {
        public void M1() {}
        public int P => 0;
    }

    public static void M2 (this ref int receiver)
    {
    }
}
""";
        var comp = CreateCompilation(src);

        comp.VerifyDiagnostics(
            // (5,9): error CS1510: A ref or out value must be an assignable variable
            //         GetInt().M1();
            Diagnostic(ErrorCode.ERR_RefLvalueExpected, "GetInt()").WithLocation(5, 9),
            // (6,9): error CS1510: A ref or out value must be an assignable variable
            //         GetInt().M2();
            Diagnostic(ErrorCode.ERR_RefLvalueExpected, "GetInt()").WithLocation(6, 9),
            // (7,13): error CS1510: A ref or out value must be an assignable variable
            //         _ = GetInt().P;
            Diagnostic(ErrorCode.ERR_RefLvalueExpected, "GetInt()").WithLocation(7, 13)
            );
    }

    [Fact]
    public void PassingValueForARefReceiver_02()
    {
        var src = """
public class C
{
    static void Main()
    {
        GetInt().M1();
        GetInt().M2();
        _ = GetInt().P;
    }

    static int GetInt() => 0;
}

static class Extensions
{
    extension(ref readonly int receiver)
    {
        public void M1() { System.Console.Write("ranM1 "); }
        public int P { get { System.Console.Write("ranP"); return 0; } }
    }

    public static void M2 (this ref readonly int receiver)
    {
        System.Console.Write("ranM2 ");
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);

        comp.VerifyEmitDiagnostics(
            // (5,9): warning CS9193: Argument 0 should be a variable because it is passed to a 'ref readonly' parameter
            //         GetInt().M1();
            Diagnostic(ErrorCode.WRN_RefReadonlyNotVariable, "GetInt()").WithArguments("0").WithLocation(5, 9),
            // (6,9): warning CS9193: Argument 0 should be a variable because it is passed to a 'ref readonly' parameter
            //         GetInt().M2();
            Diagnostic(ErrorCode.WRN_RefReadonlyNotVariable, "GetInt()").WithArguments("0").WithLocation(6, 9),
            // (7,13): warning CS9193: Argument 0 should be a variable because it is passed to a 'ref readonly' parameter
            //         _ = GetInt().P;
            Diagnostic(ErrorCode.WRN_RefReadonlyNotVariable, "GetInt()").WithArguments("0").WithLocation(7, 13)
            );

        CompileAndVerify(comp, expectedOutput: "ranM1 ranM2 ranP");
    }

    [Fact]
    public void PassingValueForARefReceiver_03()
    {
        var src = """
public class C
{
    static void Main()
    {
        GetInt().M1();
        GetInt().M2();
        _ = GetInt().P;
    }

    static int GetInt() => 0;
}

static class Extensions
{
    extension(in int receiver)
    {
        public void M1() { System.Console.Write("ranM1 "); }
        public int P { get { System.Console.Write("ranP"); return 0; } }
    }

    public static void M2(this in int receiver)
    {
        System.Console.Write("ranM2 ");
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ranM1 ranM2 ranP").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_01()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        internal void M(string s)
        {
            o.ToString();
            _ = s.Length;
        }
    }
}
""";
        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp).VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method assembly hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x207a
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method assembly hidebysig 
            instance void M (
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x2077
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method assembly hidebysig static 
        void M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 15 (0xf)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0006: pop
        IL_0007: ldarg.1
        IL_0008: callvirt instance int32 [mscorlib]System.String::get_Length()
        IL_000d: pop
        IL_000e: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        CompileAndVerify(
            comp,
            emitOptions: EmitOptions.Default.WithEmitMetadataOnly(true).WithIncludePrivateMembers(false),
            symbolValidator: (m) =>
            {
                AssertEx.Equal("System.Object o", m.GlobalNamespace.GetTypeMember("Extensions").GetTypeMembers().Single().ExtensionParameter.ToTestDisplayString());
            }
            ).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_02()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public string M(string s) => o + s;
    }
}
""";
        var comp1 = CreateCompilation(src1);

        var verifier1 = CompileAndVerify(comp1, sourceSymbolValidator: verifySymbols, symbolValidator: verifySymbols).VerifyDiagnostics();

        static void verifySymbols(ModuleSymbol m)
        {
            NamedTypeSymbol extensions = m.ContainingAssembly.GetTypeByMetadataName("Extensions");
            MethodSymbol implementation = extensions.GetMembers().OfType<MethodSymbol>().Single();
            Assert.True(implementation.IsStatic);
            Assert.Equal(MethodKind.Ordinary, implementation.MethodKind);
            Assert.Equal(2, implementation.ParameterCount);
            AssertEx.Equal("System.String Extensions.M(this System.Object o, System.String s)", implementation.ToTestDisplayString());
            Assert.Equal(m is not PEModuleSymbol, implementation.IsImplicitlyDeclared);
            Assert.True(implementation.IsExtensionMethod);
            Assert.False(implementation.HasSpecialName);
            Assert.False(implementation.HasRuntimeSpecialName);

            Assert.True(implementation.ContainingType.MightContainExtensionMethods);

            Assert.Contains("M", extensions.MemberNames);
            Assert.NotEmpty(extensions.GetSimpleNonTypeMembers("M"));

            if (m is PEModuleSymbol peModuleSymbol)
            {
                Assert.True(peModuleSymbol.Module.HasExtensionAttribute(((PEAssemblySymbol)peModuleSymbol.ContainingAssembly).Assembly.Handle, ignoreCase: false));
            }
        }

        comp1 = CreateCompilation(src1);
        NamedTypeSymbol extensions = comp1.GetTypeByMetadataName("Extensions");
        Assert.Contains("M", extensions.MemberNames);

        comp1 = CreateCompilation(src1);
        extensions = comp1.GetTypeByMetadataName("Extensions");
        Assert.NotEmpty(extensions.GetSimpleNonTypeMembers("M"));

        var expectedTypeIL = """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x207e
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig 
            instance string M (
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x207b
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 19 (0x13)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: brtrue.s IL_0006
        IL_0003: ldnull
        IL_0004: br.s IL_000c
        IL_0006: ldarg.0
        IL_0007: callvirt instance string [mscorlib]System.Object::ToString()
        IL_000c: ldarg.1
        IL_000d: call string [mscorlib]System.String::Concat(string, string)
        IL_0012: ret
    } // end of method Extensions::M
} // end of class Extensions
""";

        verifier1.VerifyTypeIL("Extensions", expectedTypeIL.Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("3".M2("4"));
    }

    static string Test(object o)
    {
        return o.M("2");
    }
}

static class Extensions
{
    extension(object o)
    {
        public string M2(string s) => o.M(s);
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        var verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       17 (0x11)
  .maxstack  2
  .locals init (string V_0)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldstr      ""2""
  IL_0007:  call       ""string Extensions.M(object, string)""
  IL_000c:  stloc.0
  IL_000d:  br.s       IL_000f
  IL_000f:  ldloc.0
  IL_0010:  ret
}
";
        verifier2.VerifyIL("Program.Test", testIL);

        var m2IL =
@"
{
  // Code size        8 (0x8)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  ldarg.1
  IL_0002:  call       ""string Extensions.M(object, string)""
  IL_0007:  ret
}
";
        verifier2.VerifyIL("Extensions.M2", m2IL);

        var comp1ImageReference = comp1.EmitToImageReference();
        comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        verifier2.VerifyIL("Program.Test", testIL);
        verifier2.VerifyIL("Extensions.M2", m2IL);

        comp2 = CreateCompilationWithIL(src2, expectedTypeIL + ExtensionMarkerAttributeIL, options: TestOptions.DebugExe);
        CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        var remove = """
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
""";

        comp2 = CreateCompilationWithIL(src2, expectedTypeIL.Remove(expectedTypeIL.IndexOf(remove), remove.Length) + ExtensionMarkerAttributeIL);
        comp2.VerifyDiagnostics(
            // (11,18): error CS1061: 'object' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         return o.M("2");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("object", "M").WithLocation(11, 18),
            // (19,41): error CS1061: 'object' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         public string M2(string s) => o.M(s);
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("object", "M").WithLocation(19, 41)
            );

        src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("3".M2("4"));
    }

    static string Test(object o)
    {
        return Extensions.M(o, "2");
    }
}

static class Extensions_
{
    extension(object o)
    {
        public string M2(string s) => Extensions.M(o, s);
    }

    public static void NotUsed(this object o) {}
}
""";

        comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        verifier2.VerifyIL("Program.Test", testIL);
        verifier2.VerifyIL("Extensions_.M2", m2IL);

        comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        verifier2.VerifyIL("Program.Test", testIL);
        verifier2.VerifyIL("Extensions_.M2", m2IL);

        var vbComp = CreateVisualBasicCompilation("""
Class Program
    Shared Sub Main()
        System.Console.Write(Test1("1"))
        System.Console.Write(Test2("3"))
    End Sub

    Shared Function Test1(o As String) As String
        return o.M("2")
    End Function
    Shared Function Test2(o As String) As String
        return Extensions.M(o, "4")
    End Function
End Class
""",
            referencedAssemblies: comp2.References, compilationOptions: new VisualBasicCompilationOptions(OutputKind.ConsoleApplication));

        CompileAndVerify(vbComp, expectedOutput: "1234").VerifyDiagnostics();

        if (!CompilationExtensions.EnableVerifyUsedAssemblies) // Tracked by https://github.com/dotnet/roslyn/issues/77542
        {
            var src4 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("3".M2("4"));
    }

    static string Test(object o)
    {
        System.Func<string, string> d = o.M;
        return d("2");
    }
}

static class Extensions
{
    extension(object o)
    {
        public string M2(string s) => new System.Func<string, string>(o.M)(s);
    }
}
""";

            var comp4 = CreateCompilation(src4, references: [comp1MetadataReference], options: TestOptions.DebugExe);
            var verifier4 = CompileAndVerify(comp4, expectedOutput: "1234").VerifyDiagnostics();

            testIL =
    @"
{
  // Code size       30 (0x1e)
  .maxstack  2
  .locals init (System.Func<string, string> V_0, //d
            string V_1)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldftn      ""string Extensions.M(object, string)""
  IL_0008:  newobj     ""System.Func<string, string>..ctor(object, System.IntPtr)""
  IL_000d:  stloc.0
  IL_000e:  ldloc.0
  IL_000f:  ldstr      ""2""
  IL_0014:  callvirt   ""string System.Func<string, string>.Invoke(string)""
  IL_0019:  stloc.1
  IL_001a:  br.s       IL_001c
  IL_001c:  ldloc.1
  IL_001d:  ret
}
";
            verifier4.VerifyIL("Program.Test", testIL);

            m2IL =
    @"
{
  // Code size       19 (0x13)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  ldftn      ""string Extensions.M(object, string)""
  IL_0007:  newobj     ""System.Func<string, string>..ctor(object, System.IntPtr)""
  IL_000c:  ldarg.1
  IL_000d:  callvirt   ""string System.Func<string, string>.Invoke(string)""
  IL_0012:  ret
}
";
            verifier4.VerifyIL("Extensions.M2", m2IL);

            comp4 = CreateCompilation(src4, references: [comp1ImageReference], options: TestOptions.DebugExe);
            verifier4 = CompileAndVerify(comp4, expectedOutput: "1234").VerifyDiagnostics();

            verifier4.VerifyIL("Program.Test", testIL);
            verifier4.VerifyIL("Extensions.M2", m2IL);
        }

        var comp5 = CreateCompilation(src1);
        comp5.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp5.VerifyDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object o)
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5)
            );
    }

    [Fact(Skip = "https://github.com/dotnet/roslyn/issues/77542")]
    public void UseSiteInfoTracking_01()
    {
        var src1 = """
public static class Extensions
{
    public static string M(this object o, string s) => o + s;
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        var src4 = """
class Program
{
    static string Test(object o)
    {
        System.Func<string, string> d = o.M;
        return d("2");
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp4 = CreateCompilation(src4, references: [comp1MetadataReference]);
        comp4.VerifyEmitDiagnostics();
        var refs = comp4.GetUsedAssemblyReferences();
        Assert.Contains(comp1MetadataReference, refs);
    }

    [Fact]
    public void UseSiteInfoTracking_02()
    {
        var src1 = """
public static class Extensions
{
    public static string M(this object o, string s) => o + s;
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        var src4 = """
class Program
{
    static string Test(object o)
    {
        return o.M("2");
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp4 = CreateCompilation(src4, references: [comp1MetadataReference]);
        comp4.VerifyEmitDiagnostics();
        var refs = comp4.GetUsedAssemblyReferences();
        Assert.Contains(comp1MetadataReference, refs);
    }

    [Fact]
    public void Implementation_InstanceMethod_03_WithLocalFunction()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public string M(string s)
        {
            string local() => o + s;
            return local();
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20ae
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig 
            instance string M (
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x20ab
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.ValueType
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public object o
        .field public string s
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2068
        // Code size 24 (0x18)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<>c__DisplayClass1_0'
        )
        IL_0000: ldloca.s 0
        IL_0002: ldarg.0
        IL_0003: stfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_0008: ldloca.s 0
        IL_000a: ldarg.1
        IL_000b: stfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0010: ldloca.s 0
        IL_0012: call string Extensions::'<M>g__local|1_0'(valuetype Extensions/'<>c__DisplayClass1_0'&)
        IL_0017: ret
    } // end of method Extensions::M
    .method assembly hidebysig static 
        string '<M>g__local|1_0' (
            valuetype Extensions/'<>c__DisplayClass1_0'& ''
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x208c
        // Code size 30 (0x1e)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: ldfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_0006: dup
        IL_0007: brtrue.s IL_000d
        IL_0009: pop
        IL_000a: ldnull
        IL_000b: br.s IL_0012
        IL_000d: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0012: ldarg.0
        IL_0013: ldfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0018: call string [mscorlib]System.String::Concat(string, string)
        IL_001d: ret
    } // end of method Extensions::'<M>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        string M(string s)
        {
            string local() => o + s;
            return local();
        }
    }

    extension(object o)
    {
        string M(string s, int x)
        {
            string local() => o + s;
            return local();
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("3".M2("4"));
    }

    static string Test(object o)
    {
        string local() => o.M("2");
        return local();
    }
}

static class Extensions
{
    extension(object o)
    {
        public string M2(string s)
        {
            string local() => o.M(s);
            return local();
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_04_WithLambda()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public string M(string s)
        {
            System.Func<string> local = () => o + s;
            return local();
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20b6
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig 
            instance string M (
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x208c
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public object o
        .field public string s
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x208f
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method '<>c__DisplayClass1_0'::.ctor
        .method assembly hidebysig 
            instance string '<M>b__0' () cil managed 
        {
            // Method begins at RVA 0x2097
            // Code size 30 (0x1e)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld object Extensions/'<>c__DisplayClass1_0'::o
            IL_0006: dup
            IL_0007: brtrue.s IL_000d
            IL_0009: pop
            IL_000a: ldnull
            IL_000b: br.s IL_0012
            IL_000d: callvirt instance string [mscorlib]System.Object::ToString()
            IL_0012: ldarg.0
            IL_0013: ldfld string Extensions/'<>c__DisplayClass1_0'::s
            IL_0018: call string [mscorlib]System.String::Concat(string, string)
            IL_001d: ret
        } // end of method '<>c__DisplayClass1_0'::'<M>b__0'
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 36 (0x24)
        .maxstack 8
        IL_0000: newobj instance void Extensions/'<>c__DisplayClass1_0'::.ctor()
        IL_0005: dup
        IL_0006: ldarg.0
        IL_0007: stfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_000c: dup
        IL_000d: ldarg.1
        IL_000e: stfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0013: ldftn instance string Extensions/'<>c__DisplayClass1_0'::'<M>b__0'()
        IL_0019: newobj instance void class [mscorlib]System.Func`1<string>::.ctor(object, native int)
        IL_001e: callvirt instance !0 class [mscorlib]System.Func`1<string>::Invoke()
        IL_0023: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        string M(string s)
        {
            System.Func<string> local = () => o + s;
            return local();
        }
    }

    extension(object o)
    {
        string M(string s, int x)
        {
            System.Func<string> local = () => o + s;
            return local();
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("3".M2("4"));
    }

    static string Test(object o)
    {
        System.Func<string> local = () => o.M("2");
        return local();
    }
}

static class Extensions
{
    extension(object o)
    {
        public string M2(string s)
        {
            System.Func<string> local = () => o.M(s);
            return local();
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_05_Iterator()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public System.Collections.Generic.IEnumerable<string> M(string s)
        {
            yield return o + s;
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1, symbolValidator: (m) =>
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            AssertEx.Equal("System.Runtime.CompilerServices.IteratorStateMachineAttribute(typeof(Extensions.<M>d__1))", implementation.GetAttributes().Single().ToString());
        }).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", ("""
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2167
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig 
            instance class [mscorlib]System.Collections.Generic.IEnumerable`1<string> M (
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x207e
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<M>d__1'
        extends [mscorlib]System.Object
        implements class [mscorlib]System.Collections.Generic.IEnumerable`1<string>,
                   [mscorlib]System.Collections.IEnumerable,
                   class [mscorlib]System.Collections.Generic.IEnumerator`1<string>,

""" +
        (ExecutionConditionUtil.IsMonoOrCoreClr ?
"""
                   [mscorlib]System.Collections.IEnumerator,
                   [mscorlib]System.IDisposable

""" :
"""
                   [mscorlib]System.IDisposable,
                   [mscorlib]System.Collections.IEnumerator

""") +
"""
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field private int32 '<>1__state'
        .field private string '<>2__current'
        .field private int32 '<>l__initialThreadId'
        .field private object o
        .field public object '<>3__o'
        .field private string s
        .field public string '<>3__s'
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor (
                int32 '<>1__state'
            ) cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            // Method begins at RVA 0x2081
            // Code size 25 (0x19)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ldarg.0
            IL_0007: ldarg.1
            IL_0008: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_000d: ldarg.0
            IL_000e: call int32 [mscorlib]System.Environment::get_CurrentManagedThreadId()
            IL_0013: stfld int32 Extensions/'<M>d__1'::'<>l__initialThreadId'
            IL_0018: ret
        } // end of method '<M>d__1'::.ctor
        .method private final hidebysig newslot virtual 
            instance void System.IDisposable.Dispose () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.IDisposable::Dispose()
            // Method begins at RVA 0x209b
            // Code size 9 (0x9)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldc.i4.s -2
            IL_0003: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0008: ret
        } // end of method '<M>d__1'::System.IDisposable.Dispose
        .method private final hidebysig newslot virtual 
            instance bool MoveNext () cil managed 
        {
            .override method instance bool [mscorlib]System.Collections.IEnumerator::MoveNext()
            // Method begins at RVA 0x20a8
            // Code size 76 (0x4c)
            .maxstack 3
            .locals init (
                [0] int32
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0006: stloc.0
            IL_0007: ldloc.0
            IL_0008: brfalse.s IL_0010
            IL_000a: ldloc.0
            IL_000b: ldc.i4.1
            IL_000c: beq.s IL_0043
            IL_000e: ldc.i4.0
            IL_000f: ret
            IL_0010: ldarg.0
            IL_0011: ldc.i4.m1
            IL_0012: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0017: ldarg.0
            IL_0018: ldarg.0
            IL_0019: ldfld object Extensions/'<M>d__1'::o
            IL_001e: dup
            IL_001f: brtrue.s IL_0025
            IL_0021: pop
            IL_0022: ldnull
            IL_0023: br.s IL_002a
            IL_0025: callvirt instance string [mscorlib]System.Object::ToString()
            IL_002a: ldarg.0
            IL_002b: ldfld string Extensions/'<M>d__1'::s
            IL_0030: call string [mscorlib]System.String::Concat(string, string)
            IL_0035: stfld string Extensions/'<M>d__1'::'<>2__current'
            IL_003a: ldarg.0
            IL_003b: ldc.i4.1
            IL_003c: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0041: ldc.i4.1
            IL_0042: ret
            IL_0043: ldarg.0
            IL_0044: ldc.i4.m1
            IL_0045: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_004a: ldc.i4.0
            IL_004b: ret
        } // end of method '<M>d__1'::MoveNext
        .method private final hidebysig specialname newslot virtual 
            instance string 'System.Collections.Generic.IEnumerator<System.String>.get_Current' () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance !0 class [mscorlib]System.Collections.Generic.IEnumerator`1<string>::get_Current()
            // Method begins at RVA 0x2100
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld string Extensions/'<M>d__1'::'<>2__current'
            IL_0006: ret
        } // end of method '<M>d__1'::'System.Collections.Generic.IEnumerator<System.String>.get_Current'
        .method private final hidebysig newslot virtual 
            instance void System.Collections.IEnumerator.Reset () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.Collections.IEnumerator::Reset()
            // Method begins at RVA 0x2108
            // Code size 6 (0x6)
            .maxstack 8
            IL_0000: newobj instance void [mscorlib]System.NotSupportedException::.ctor()
            IL_0005: throw
        } // end of method '<M>d__1'::System.Collections.IEnumerator.Reset
        .method private final hidebysig specialname newslot virtual 
            instance object System.Collections.IEnumerator.get_Current () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance object [mscorlib]System.Collections.IEnumerator::get_Current()
            // Method begins at RVA 0x2100
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld string Extensions/'<M>d__1'::'<>2__current'
            IL_0006: ret
        } // end of method '<M>d__1'::System.Collections.IEnumerator.get_Current
        .method private final hidebysig newslot virtual 
            instance class [mscorlib]System.Collections.Generic.IEnumerator`1<string> 'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator' () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance class [mscorlib]System.Collections.Generic.IEnumerator`1<!0> class [mscorlib]System.Collections.Generic.IEnumerable`1<string>::GetEnumerator()
            // Method begins at RVA 0x2110
            // Code size 67 (0x43)
            .maxstack 2
            .locals init (
                [0] class Extensions/'<M>d__1'
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0006: ldc.i4.s -2
            IL_0008: bne.un.s IL_0022
            IL_000a: ldarg.0
            IL_000b: ldfld int32 Extensions/'<M>d__1'::'<>l__initialThreadId'
            IL_0010: call int32 [mscorlib]System.Environment::get_CurrentManagedThreadId()
            IL_0015: bne.un.s IL_0022
            IL_0017: ldarg.0
            IL_0018: ldc.i4.0
            IL_0019: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_001e: ldarg.0
            IL_001f: stloc.0
            IL_0020: br.s IL_0029
            IL_0022: ldc.i4.0
            IL_0023: newobj instance void Extensions/'<M>d__1'::.ctor(int32)
            IL_0028: stloc.0
            IL_0029: ldloc.0
            IL_002a: ldarg.0
            IL_002b: ldfld object Extensions/'<M>d__1'::'<>3__o'
            IL_0030: stfld object Extensions/'<M>d__1'::o
            IL_0035: ldloc.0
            IL_0036: ldarg.0
            IL_0037: ldfld string Extensions/'<M>d__1'::'<>3__s'
            IL_003c: stfld string Extensions/'<M>d__1'::s
            IL_0041: ldloc.0
            IL_0042: ret
        } // end of method '<M>d__1'::'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator'
        .method private final hidebysig newslot virtual 
            instance class [mscorlib]System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance class [mscorlib]System.Collections.IEnumerator [mscorlib]System.Collections.IEnumerable::GetEnumerator()
            // Method begins at RVA 0x215f
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance class [mscorlib]System.Collections.Generic.IEnumerator`1<string> Extensions/'<M>d__1'::'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator'()
            IL_0006: ret
        } // end of method '<M>d__1'::System.Collections.IEnumerable.GetEnumerator
        // Properties
        .property instance string 'System.Collections.Generic.IEnumerator<System.String>.Current'()
        {
            .get instance string Extensions/'<M>d__1'::'System.Collections.Generic.IEnumerator<System.String>.get_Current'()
        }
        .property instance object System.Collections.IEnumerator.Current()
        {
            .get instance object Extensions/'<M>d__1'::System.Collections.IEnumerator.get_Current()
        }
    } // end of class <M>d__1
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Collections.Generic.IEnumerable`1<string> M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.IteratorStateMachineAttribute::.ctor(class [mscorlib]System.Type) = (
            01 00 12 45 78 74 65 6e 73 69 6f 6e 73 2b 3c 4d
            3e 64 5f 5f 31 00 00
        )
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 22 (0x16)
        .maxstack 8
        IL_0000: ldc.i4.s -2
        IL_0002: newobj instance void Extensions/'<M>d__1'::.ctor(int32)
        IL_0007: dup
        IL_0008: ldarg.0
        IL_0009: stfld object Extensions/'<M>d__1'::'<>3__o'
        IL_000e: dup
        IL_000f: ldarg.1
        IL_0010: stfld string Extensions/'<M>d__1'::'<>3__s'
        IL_0015: ret
    } // end of method Extensions::M
} // end of class Extensions
""").Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        System.Collections.Generic.IEnumerable<string> M(string s)
        {
            yield return o + s;
        }
    }

    extension(object o)
    {
        System.Collections.Generic.IEnumerable<string> M(string s, int x)
        {
            yield return o + s;
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        foreach (var s in Test("1"))
            System.Console.Write(s);
        foreach (var s in "3".M2("4"))
            System.Console.Write(s);
    }

    static System.Collections.Generic.IEnumerable<string> Test(object o)
    {
        return o.M("2");
    }
}

static class Extensions
{
    extension(object o)
    {
        public System.Collections.Generic.IEnumerable<string> M2(string s) => o.M(s);
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        var verifier3 = CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       17 (0x11)
  .maxstack  2
  .locals init (System.Collections.Generic.IEnumerable<string> V_0)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldstr      ""2""
  IL_0007:  call       ""System.Collections.Generic.IEnumerable<string> Extensions.M(object, string)""
  IL_000c:  stloc.0
  IL_000d:  br.s       IL_000f
  IL_000f:  ldloc.0
  IL_0010:  ret
}
";
        verifier3.VerifyIL("Program.Test", testIL);

        var m2IL =
@"
{
  // Code size        8 (0x8)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  ldarg.1
  IL_0002:  call       ""System.Collections.Generic.IEnumerable<string> Extensions.M(object, string)""
  IL_0007:  ret
}
";
        verifier3.VerifyIL("Extensions.M2", m2IL);

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions.M2", m2IL);
    }

    [Fact]
    public void Implementation_InstanceMethod_06_Async()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public async System.Threading.Tasks.Task<string> M(string s)
        {
            await System.Threading.Tasks.Task.Yield();
            return o + s;
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1, symbolValidator: (m) =>
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            AssertEx.Equal("System.Runtime.CompilerServices.AsyncStateMachineAttribute(typeof(Extensions.<M>d__1))", implementation.GetAttributes().Single().ToString());
        }).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2196
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig 
            instance class [mscorlib]System.Threading.Tasks.Task`1<string> M (
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x20b3
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<M>d__1'
        extends [mscorlib]System.ValueType
        implements [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public int32 '<>1__state'
        .field public valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> '<>t__builder'
        .field public object o
        .field public string s
        .field private valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter '<>u__1'
        // Methods
        .method private final hidebysig newslot virtual 
            instance void MoveNext () cil managed 
        {
            .override method instance void [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine::MoveNext()
            // Method begins at RVA 0x20b8
            // Code size 178 (0xb2)
            .maxstack 3
            .locals init (
                [0] int32,
                [1] string,
                [2] valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter,
                [3] valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable,
                [4] class [mscorlib]System.Exception
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0006: stloc.0
            .try
            {
                IL_0007: ldloc.0
                IL_0008: brfalse.s IL_0041
                IL_000a: call valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable [mscorlib]System.Threading.Tasks.Task::Yield()
                IL_000f: stloc.3
                IL_0010: ldloca.s 3
                IL_0012: call instance valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter [mscorlib]System.Runtime.CompilerServices.YieldAwaitable::GetAwaiter()
                IL_0017: stloc.2
                IL_0018: ldloca.s 2
                IL_001a: call instance bool [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter::get_IsCompleted()
                IL_001f: brtrue.s IL_005d
                IL_0021: ldarg.0
                IL_0022: ldc.i4.0
                IL_0023: dup
                IL_0024: stloc.0
                IL_0025: stfld int32 Extensions/'<M>d__1'::'<>1__state'
                IL_002a: ldarg.0
                IL_002b: ldloc.2
                IL_002c: stfld valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter Extensions/'<M>d__1'::'<>u__1'
                IL_0031: ldarg.0
                IL_0032: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
                IL_0037: ldloca.s 2
                IL_0039: ldarg.0
                IL_003a: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::AwaitUnsafeOnCompleted<valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter, valuetype Extensions/'<M>d__1'>(!!0&, !!1&)
                IL_003f: leave.s IL_00b1
                IL_0041: ldarg.0
                IL_0042: ldfld valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter Extensions/'<M>d__1'::'<>u__1'
                IL_0047: stloc.2
                IL_0048: ldarg.0
                IL_0049: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter Extensions/'<M>d__1'::'<>u__1'
                IL_004e: initobj [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter
                IL_0054: ldarg.0
                IL_0055: ldc.i4.m1
                IL_0056: dup
                IL_0057: stloc.0
                IL_0058: stfld int32 Extensions/'<M>d__1'::'<>1__state'
                IL_005d: ldloca.s 2
                IL_005f: call instance void [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter::GetResult()
                IL_0064: ldarg.0
                IL_0065: ldfld object Extensions/'<M>d__1'::o
                IL_006a: dup
                IL_006b: brtrue.s IL_0071
                IL_006d: pop
                IL_006e: ldnull
                IL_006f: br.s IL_0076
                IL_0071: callvirt instance string [mscorlib]System.Object::ToString()
                IL_0076: ldarg.0
                IL_0077: ldfld string Extensions/'<M>d__1'::s
                IL_007c: call string [mscorlib]System.String::Concat(string, string)
                IL_0081: stloc.1
                IL_0082: leave.s IL_009d
            } // end .try
            catch [mscorlib]System.Exception
            {
                IL_0084: stloc.s 4
                IL_0086: ldarg.0
                IL_0087: ldc.i4.s -2
                IL_0089: stfld int32 Extensions/'<M>d__1'::'<>1__state'
                IL_008e: ldarg.0
                IL_008f: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
                IL_0094: ldloc.s 4
                IL_0096: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetException(class [mscorlib]System.Exception)
                IL_009b: leave.s IL_00b1
            } // end handler
            IL_009d: ldarg.0
            IL_009e: ldc.i4.s -2
            IL_00a0: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_00a5: ldarg.0
            IL_00a6: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
            IL_00ab: ldloc.1
            IL_00ac: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetResult(!0)
            IL_00b1: ret
        } // end of method '<M>d__1'::MoveNext
        .method private final hidebysig newslot virtual 
            instance void SetStateMachine (
                class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine stateMachine
            ) cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine::SetStateMachine(class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine)
            // Method begins at RVA 0x2188
            // Code size 13 (0xd)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
            IL_0006: ldarg.1
            IL_0007: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetStateMachine(class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine)
            IL_000c: ret
        } // end of method '<M>d__1'::SetStateMachine
    } // end of class <M>d__1
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Threading.Tasks.Task`1<string> M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.AsyncStateMachineAttribute::.ctor(class [mscorlib]System.Type) = (
            01 00 12 45 78 74 65 6e 73 69 6f 6e 73 2b 3c 4d
            3e 64 5f 5f 31 00 00
        )
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2068
        // Code size 63 (0x3f)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<M>d__1'
        )
        IL_0000: ldloca.s 0
        IL_0002: call valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<!0> valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::Create()
        IL_0007: stfld valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
        IL_000c: ldloca.s 0
        IL_000e: ldarg.0
        IL_000f: stfld object Extensions/'<M>d__1'::o
        IL_0014: ldloca.s 0
        IL_0016: ldarg.1
        IL_0017: stfld string Extensions/'<M>d__1'::s
        IL_001c: ldloca.s 0
        IL_001e: ldc.i4.m1
        IL_001f: stfld int32 Extensions/'<M>d__1'::'<>1__state'
        IL_0024: ldloca.s 0
        IL_0026: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
        IL_002b: ldloca.s 0
        IL_002d: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::Start<valuetype Extensions/'<M>d__1'>(!!0&)
        IL_0032: ldloca.s 0
        IL_0034: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
        IL_0039: call instance class [mscorlib]System.Threading.Tasks.Task`1<!0> valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::get_Task()
        IL_003e: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        async System.Threading.Tasks.Task<string> M(string s)
        {
            await System.Threading.Tasks.Task.Yield();
            return o + s;
        }
    }

    extension(object o)
    {
        async System.Threading.Tasks.Task<string> M(string s, int x)
        {
            await System.Threading.Tasks.Task.Yield();
            return o + s;
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1").Result);
        System.Console.Write("3".M2("4").Result);
    }

    async static System.Threading.Tasks.Task<string> Test(object o)
    {
        await System.Threading.Tasks.Task.Yield();
        return await o.M("2");
    }
}

static class Extensions
{
    extension(object o)
    {
        async public System.Threading.Tasks.Task<string> M2(string s)
        {
            await System.Threading.Tasks.Task.Yield();
            return await o.M(s);
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_07_Generic()
    {
        var src1 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        public string M<U>(T t, U u)
        {
            return o.GetString() + u.ToString() + t.ToString();
        }
    }
}

public class C<T>(string v)
{
    public string GetString() => v;
}
""";
        var comp1 = CreateCompilation(src1);

        MethodSymbol implementation = comp1.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
        Assert.True(implementation.IsStatic);
        Assert.Equal(MethodKind.Ordinary, implementation.MethodKind);
        Assert.Equal(3, implementation.ParameterCount);
        AssertEx.Equal("System.String Extensions.M<T, U>(this C<T> o, T t, U u)", implementation.ToTestDisplayString());
        Assert.True(implementation.IsImplicitlyDeclared);
        Assert.True(implementation.IsExtensionMethod);
        Assert.False(implementation.HasSpecialName);
        Assert.False(implementation.HasRuntimeSpecialName);

        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D884D1E13988E83801B7574694E1C2C5'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static
                void '<Extension>$' (
                    class C`1<!T> o
                ) cil managed
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20a8
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D884D1E13988E83801B7574694E1C2C5'::'<Extension>$'
        } // end of class <M>$D884D1E13988E83801B7574694E1C2C5
        // Methods
        .method public hidebysig
            instance string M<U> (
                !$T0 t,
                !!U u
            ) cil managed
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 38 38 34 44 31 45 31 33
                39 38 38 45 38 33 38 30 31 42 37 35 37 34 36 39
                34 45 31 43 32 43 35 00 00
            )
            // Method begins at RVA 0x20a5
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'::M
    } // end of class <G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1
    // Methods
    .method public hidebysig static
        string M<T, U> (
            class C`1<!!T> o,
            !!T t,
            !!U u
        ) cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 38 (0x26)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: callvirt instance string class C`1<!!T>::GetString()
        IL_0006: ldarga.s u
        IL_0008: constrained. !!U
        IL_000e: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0013: ldarga.s t
        IL_0015: constrained. !!T
        IL_001b: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0020: call string [mscorlib]System.String::Concat(string, string, string)
        IL_0025: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test(new C<string>("1"), "2", 3));
        System.Console.Write(new C<string>("4").M2("5", 6));
    }

    static string Test<T, U>(C<T> o, T t, U u)
    {
        return o.M(t, u);
    }
}

static class Extensions
{
    extension<T>(C<T> o)
    {
        public string M2<U>(T t, U u) => o.M(t, u);
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        var verifier3 = CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       14 (0xe)
  .maxstack  3
  .locals init (string V_0)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldarg.1
  IL_0003:  ldarg.2
  IL_0004:  call       ""string Extensions.M<T, U>(C<T>, T, U)""
  IL_0009:  stloc.0
  IL_000a:  br.s       IL_000c
  IL_000c:  ldloc.0
  IL_000d:  ret
}
";
        verifier3.VerifyIL("Program.Test<T, U>(C<T>, T, U)", testIL);

        var m2IL =
@"
{
  // Code size        9 (0x9)
  .maxstack  3
  IL_0000:  ldarg.0
  IL_0001:  ldarg.1
  IL_0002:  ldarg.2
  IL_0003:  call       ""string Extensions.M<T, U>(C<T>, T, U)""
  IL_0008:  ret
}
";
        verifier3.VerifyIL("Extensions.M2<T, U>(this C<T>, T, U)", m2IL);

        var comp1ImageReference = comp1.EmitToImageReference();
        comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test<T, U>(C<T>, T, U)", testIL);
        verifier3.VerifyIL("Extensions.M2<T, U>(this C<T>, T, U)", m2IL);

        if (!CompilationExtensions.EnableVerifyUsedAssemblies) // Tracked by https://github.com/dotnet/roslyn/issues/77542
        {
            src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test(new C<string>("1"), "2", 3));
        System.Console.Write(new C<string>("4").M2("5", 6));
    }

    static string Test<T, U>(C<T> o, T t, U u)
    {
        System.Func<T, U, string> d = o.M;
        return d(t, u);
    }
}

static class Extensions
{
    extension<T>(C<T> o)
    {
        public string M2<U>(T t, U u) => new System.Func<T, U, string>(o.M)(t, u);
    }
}
""";

            comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
            verifier3 = CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();

            testIL =
@"
{
  // Code size       27 (0x1b)
  .maxstack  3
  .locals init (System.Func<T, U, string> V_0, //d
            string V_1)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldftn      ""string Extensions.M<T, U>(C<T>, T, U)""
  IL_0008:  newobj     ""System.Func<T, U, string>..ctor(object, System.IntPtr)""
  IL_000d:  stloc.0
  IL_000e:  ldloc.0
  IL_000f:  ldarg.1
  IL_0010:  ldarg.2
  IL_0011:  callvirt   ""string System.Func<T, U, string>.Invoke(T, U)""
  IL_0016:  stloc.1
  IL_0017:  br.s       IL_0019
  IL_0019:  ldloc.1
  IL_001a:  ret
}
";
            verifier3.VerifyIL("Program.Test<T, U>(C<T>, T, U)", testIL);

            m2IL =
@"
{
  // Code size       20 (0x14)
  .maxstack  3
  IL_0000:  ldarg.0
  IL_0001:  ldftn      ""string Extensions.M<T, U>(C<T>, T, U)""
  IL_0007:  newobj     ""System.Func<T, U, string>..ctor(object, System.IntPtr)""
  IL_000c:  ldarg.1
  IL_000d:  ldarg.2
  IL_000e:  callvirt   ""string System.Func<T, U, string>.Invoke(T, U)""
  IL_0013:  ret
}
";
            verifier3.VerifyIL("Extensions.M2<T, U>(this C<T>, T, U)", m2IL);

            comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);

            verifier3 = CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();

            verifier3.VerifyIL("Program.Test<T, U>(C<T>, T, U)", testIL);
            verifier3.VerifyIL("Extensions.M2<T, U>(this C<T>, T, U)", m2IL);
        }
    }

    [Fact]
    public void Implementation_InstanceMethod_08_WithLocalFunction_Generic()
    {
        var src1 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        public C<U> M<U>(T t1, U u1)
        {
            C<U> local<X, Y, Z>(T t2, U u2, X x2, Y y2, Z z2)
            {
                return new C<U>(o.GetString() + u1.ToString() + t1.ToString() + u2.ToString() + t2.ToString() + x2.ToString() + y2.ToString() + z2.ToString());
            };

            return local(t1, u1, 0, t1, u1);
        }
    }
}

public class C<T>(string val)
{
    public string GetString() => val;
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D884D1E13988E83801B7574694E1C2C5'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    class C`1<!T> o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2170
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D884D1E13988E83801B7574694E1C2C5'::'<Extension>$'
        } // end of class <M>$D884D1E13988E83801B7574694E1C2C5
        // Methods
        .method public hidebysig 
            instance class C`1<!!U> M<U> (
                !$T0 t1,
                !!U u1
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 38 38 34 44 31 45 31 33
                39 38 38 45 38 33 38 30 31 42 37 35 37 34 36 39
                34 45 31 43 32 43 35 00 00
            )
            // Method begins at RVA 0x216d
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'::M
    } // end of class <G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0`2'<T, U>
        extends [mscorlib]System.ValueType
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public class C`1<!T> o
        .field public !U u1
        .field public !T t1
    } // end of class <>c__DisplayClass1_0`2
    // Methods
    .method public hidebysig static 
        class C`1<!!U> M<T, U> (
            class C`1<!!T> o,
            !!T t1,
            !!U u1
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2068
        // Code size 57 (0x39)
        .maxstack 6
        .locals init (
            [0] valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>
        )
        IL_0000: ldloca.s 0
        IL_0002: ldarg.0
        IL_0003: stfld class C`1<!0> valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::o
        IL_0008: ldloca.s 0
        IL_000a: ldarg.2
        IL_000b: stfld !1 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::u1
        IL_0010: ldloca.s 0
        IL_0012: ldarg.1
        IL_0013: stfld !0 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::t1
        IL_0018: ldloc.0
        IL_0019: ldfld !0 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::t1
        IL_001e: ldloc.0
        IL_001f: ldfld !1 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::u1
        IL_0024: ldc.i4.0
        IL_0025: ldloc.0
        IL_0026: ldfld !0 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::t1
        IL_002b: ldloc.0
        IL_002c: ldfld !1 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::u1
        IL_0031: ldloca.s 0
        IL_0033: call class C`1<!!1> Extensions::'<M>g__local|1_0'<!!T, !!U, int32, !!T, !!U>(!!0, !!1, !!2, !!3, !!4, valuetype Extensions/'<>c__DisplayClass1_0`2'<!!0, !!1>&)
        IL_0038: ret
    } // end of method Extensions::M
    .method assembly hidebysig static 
        class C`1<!!U> '<M>g__local|1_0'<T, U, X, Y, Z> (
            !!T t2,
            !!U u2,
            !!X x2,
            !!Y y2,
            !!Z z2,
            valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>& ''
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x20b0
        // Code size 154 (0x9a)
        .maxstack 4
        IL_0000: ldc.i4.8
        IL_0001: newarr [mscorlib]System.String
        IL_0006: dup
        IL_0007: ldc.i4.0
        IL_0008: ldarg.s 5
        IL_000a: ldfld class C`1<!0> valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::o
        IL_000f: callvirt instance string class C`1<!!T>::GetString()
        IL_0014: stelem.ref
        IL_0015: dup
        IL_0016: ldc.i4.1
        IL_0017: ldarg.s 5
        IL_0019: ldflda !1 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::u1
        IL_001e: constrained. !!U
        IL_0024: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0029: stelem.ref
        IL_002a: dup
        IL_002b: ldc.i4.2
        IL_002c: ldarg.s 5
        IL_002e: ldflda !0 valuetype Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::t1
        IL_0033: constrained. !!T
        IL_0039: callvirt instance string [mscorlib]System.Object::ToString()
        IL_003e: stelem.ref
        IL_003f: dup
        IL_0040: ldc.i4.3
        IL_0041: ldarga.s u2
        IL_0043: constrained. !!U
        IL_0049: callvirt instance string [mscorlib]System.Object::ToString()
        IL_004e: stelem.ref
        IL_004f: dup
        IL_0050: ldc.i4.4
        IL_0051: ldarga.s t2
        IL_0053: constrained. !!T
        IL_0059: callvirt instance string [mscorlib]System.Object::ToString()
        IL_005e: stelem.ref
        IL_005f: dup
        IL_0060: ldc.i4.5
        IL_0061: ldarga.s x2
        IL_0063: constrained. !!X
        IL_0069: callvirt instance string [mscorlib]System.Object::ToString()
        IL_006e: stelem.ref
        IL_006f: dup
        IL_0070: ldc.i4.6
        IL_0071: ldarga.s y2
        IL_0073: constrained. !!Y
        IL_0079: callvirt instance string [mscorlib]System.Object::ToString()
        IL_007e: stelem.ref
        IL_007f: dup
        IL_0080: ldc.i4.7
        IL_0081: ldarga.s z2
        IL_0083: constrained. !!Z
        IL_0089: callvirt instance string [mscorlib]System.Object::ToString()
        IL_008e: stelem.ref
        IL_008f: call string [mscorlib]System.String::Concat(string[])
        IL_0094: newobj instance void class C`1<!!U>::.ctor(string)
        IL_0099: ret
    } // end of method Extensions::'<M>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        string M<U>(T t1, U u1)
        {
            U local<X, Y, Z>(T t2, U u2, X x2, Y y2, Z z2)
            {
                _ = o.GetString() + u1.ToString() + t1.ToString() + u2.ToString() + t2.ToString() + x2.ToString() + y2.ToString() + z2.ToString();
                return u2;
            };

            return local(t1, u1, 0, t1, u1).ToString();
        }
    }

    extension<T>(C<T> o)
    {
        string M<U>(T t1, U u1, int x)
        {
            U local<X, Y, Z>(T t2, U u2, X x2, Y y2, Z z2)
            {
                _ = o.GetString() + u1.ToString() + t1.ToString() + u2.ToString() + t2.ToString() + x2.ToString() + y2.ToString() + z2.ToString();
                return u2;
            };

            return local(t1, u1, 0, t1, u1).ToString();
        }
    }
}

public class C<T>
{
    public string GetString() => null;
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test(new C<long>("1"), 2, "3").GetString());
        System.Console.Write(new C<long>("4").M2(5, "6").GetString());
    }

    static C<U> Test<T, U>(C<T> o, T t1, U u1)
    {
        C<X> local<X>(T t2, X x2)
        {
            return o.M(t2, x2);
        };

        return local(t1, u1);
    }
}

static class Extensions
{
    extension<T>(C<T> o)
    {
        public C<U> M2<U>(T t1, U u1)
        {
            C<X> local<X>(T t2, X x2)
            {
                return o.M(t2, x2);
            };

            return local(t1, u1);
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1323202346565056").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1323202346565056").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_09_WithLambda_Generic()
    {
        var src1 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        public C<U> M<U>(T t1, U u1)
        {
            System.Func<T, U, C<U>> local = (T t2, U u2) =>
            {
                return new C<U>(o.GetString() + u1.ToString() + t1.ToString() + u2.ToString() + t2.ToString());
            };

            return local(t1, u1);
        }
    }
}

public class C<T>(string val)
{
    public string GetString() => val;
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D884D1E13988E83801B7574694E1C2C5'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    class C`1<!T> o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2143
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D884D1E13988E83801B7574694E1C2C5'::'<Extension>$'
        } // end of class <M>$D884D1E13988E83801B7574694E1C2C5
        // Methods
        .method public hidebysig 
            instance class C`1<!!U> M<U> (
                !$T0 t1,
                !!U u1
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 38 38 34 44 31 45 31 33
                39 38 38 45 38 33 38 30 31 42 37 35 37 34 36 39
                34 45 31 43 32 43 35 00 00
            )
            // Method begins at RVA 0x20c4
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'::M
    } // end of class <G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0`2'<T, U>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public class C`1<!T> o
        .field public !U u1
        .field public !T t1
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x20c7
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method '<>c__DisplayClass1_0`2'::.ctor
        .method assembly hidebysig 
            instance class C`1<!U> '<M>b__0' (
                !T t2,
                !U u2
            ) cil managed 
        {
            // Method begins at RVA 0x20d0
            // Code size 103 (0x67)
            .maxstack 4
            IL_0000: ldc.i4.5
            IL_0001: newarr [mscorlib]System.String
            IL_0006: dup
            IL_0007: ldc.i4.0
            IL_0008: ldarg.0
            IL_0009: ldfld class C`1<!0> class Extensions/'<>c__DisplayClass1_0`2'<!T, !U>::o
            IL_000e: callvirt instance string class C`1<!T>::GetString()
            IL_0013: stelem.ref
            IL_0014: dup
            IL_0015: ldc.i4.1
            IL_0016: ldarg.0
            IL_0017: ldflda !1 class Extensions/'<>c__DisplayClass1_0`2'<!T, !U>::u1
            IL_001c: constrained. !U
            IL_0022: callvirt instance string [mscorlib]System.Object::ToString()
            IL_0027: stelem.ref
            IL_0028: dup
            IL_0029: ldc.i4.2
            IL_002a: ldarg.0
            IL_002b: ldflda !0 class Extensions/'<>c__DisplayClass1_0`2'<!T, !U>::t1
            IL_0030: constrained. !T
            IL_0036: callvirt instance string [mscorlib]System.Object::ToString()
            IL_003b: stelem.ref
            IL_003c: dup
            IL_003d: ldc.i4.3
            IL_003e: ldarga.s u2
            IL_0040: constrained. !U
            IL_0046: callvirt instance string [mscorlib]System.Object::ToString()
            IL_004b: stelem.ref
            IL_004c: dup
            IL_004d: ldc.i4.4
            IL_004e: ldarga.s t2
            IL_0050: constrained. !T
            IL_0056: callvirt instance string [mscorlib]System.Object::ToString()
            IL_005b: stelem.ref
            IL_005c: call string [mscorlib]System.String::Concat(string[])
            IL_0061: newobj instance void class C`1<!U>::.ctor(string)
            IL_0066: ret
        } // end of method '<>c__DisplayClass1_0`2'::'<M>b__0'
    } // end of class <>c__DisplayClass1_0`2
    // Methods
    .method public hidebysig static 
        class C`1<!!U> M<T, U> (
            class C`1<!!T> o,
            !!T t1,
            !!U u1
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2068
        // Code size 57 (0x39)
        .maxstack 3
        .locals init (
            [0] class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>
        )
        IL_0000: newobj instance void class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::.ctor()
        IL_0005: stloc.0
        IL_0006: ldloc.0
        IL_0007: ldarg.0
        IL_0008: stfld class C`1<!0> class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::o
        IL_000d: ldloc.0
        IL_000e: ldarg.2
        IL_000f: stfld !1 class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::u1
        IL_0014: ldloc.0
        IL_0015: ldarg.1
        IL_0016: stfld !0 class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::t1
        IL_001b: ldloc.0
        IL_001c: ldftn instance class C`1<!1> class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::'<M>b__0'(!0, !1)
        IL_0022: newobj instance void class [mscorlib]System.Func`3<!!T, !!U, class C`1<!!U>>::.ctor(object, native int)
        IL_0027: ldloc.0
        IL_0028: ldfld !0 class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::t1
        IL_002d: ldloc.0
        IL_002e: ldfld !1 class Extensions/'<>c__DisplayClass1_0`2'<!!T, !!U>::u1
        IL_0033: callvirt instance !2 class [mscorlib]System.Func`3<!!T, !!U, class C`1<!!U>>::Invoke(!0, !1)
        IL_0038: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        string M<U>(T t1, U u1)
        {
            System.Func<T, U, U> local = (T t2, U u2) =>
            {
                _ = o.GetString() + u1.ToString() + t1.ToString() + u2.ToString() + t2.ToString();
                return u2;
            };

            return local(t1, u1).ToString();
        }
    }

    extension<T>(C<T> o)
    {
        string M<U>(T t1, U u1, int x)
        {
            System.Func<T, U, U> local = (T t2, U u2) =>
            {
                _ = o.GetString() + u1.ToString() + t1.ToString() + u2.ToString() + t2.ToString();
                return u2;
            };

            return local(t1, u1).ToString();
        }
    }
}

public class C<T>
{
    public string GetString() => null;
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test(new C<long>("1"), 2, "3").GetString());
        System.Console.Write(new C<long>("4").M2(5, "6").GetString());
    }

    static C<U> Test<T, U>(C<T> o, T t1, U u1)
    {
        System.Func<T, U, C<U>> local = (T t2, U u2) =>
        {
            return o.M(t2, u2);
        };

        return local(t1, u1);
    }
}

static class Extensions
{
    extension<T>(C<T> o)
    {
        public C<U> M2<U>(T t1, U u1)
        {
            System.Func<T, U, C<U>> local = (T t2, U u2) =>
            {
                return o.M(t2, u2);
            };

            return local(t1, u1);
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1323246565").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1323246565").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_10_Iterator_Generic()
    {
        var src1 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        public System.Collections.Generic.IEnumerable<string> M<U>(T t1, U u1)
        {
            yield return o.GetString() + u1.ToString() + t1.ToString();
        }
    }
}

public class C<T>(string val)
{
    public string GetString() => val;
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1, symbolValidator: (m) =>
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            AssertEx.Equal("System.Runtime.CompilerServices.IteratorStateMachineAttribute(typeof(Extensions.<M>d__1<,>))", implementation.GetAttributes().Single().ToString());
        }).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", ("""
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D884D1E13988E83801B7574694E1C2C5'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    class C`1<!T> o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x21a3
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D884D1E13988E83801B7574694E1C2C5'::'<Extension>$'
        } // end of class <M>$D884D1E13988E83801B7574694E1C2C5
        // Methods
        .method public hidebysig 
            instance class [mscorlib]System.Collections.Generic.IEnumerable`1<string> M<U> (
                !$T0 t1,
                !!U u1
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 38 38 34 44 31 45 31 33
                39 38 38 45 38 33 38 30 31 42 37 35 37 34 36 39
                34 45 31 43 32 43 35 00 00
            )
            // Method begins at RVA 0x209c
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'::M
    } // end of class <G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1
    .class nested private auto ansi sealed beforefieldinit '<M>d__1`2'<T, U>
        extends [mscorlib]System.Object
        implements class [mscorlib]System.Collections.Generic.IEnumerable`1<string>,
                   [mscorlib]System.Collections.IEnumerable,
                   class [mscorlib]System.Collections.Generic.IEnumerator`1<string>,

""" +
        (ExecutionConditionUtil.IsMonoOrCoreClr ?
"""
                   [mscorlib]System.Collections.IEnumerator,
                   [mscorlib]System.IDisposable

""" :
"""
                   [mscorlib]System.IDisposable,
                   [mscorlib]System.Collections.IEnumerator

""") +
"""
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field private int32 '<>1__state'
        .field private string '<>2__current'
        .field private int32 '<>l__initialThreadId'
        .field private class C`1<!T> o
        .field public class C`1<!T> '<>3__o'
        .field private !U u1
        .field public !U '<>3__u1'
        .field private !T t1
        .field public !T '<>3__t1'
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor (
                int32 '<>1__state'
            ) cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            // Method begins at RVA 0x209f
            // Code size 25 (0x19)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ldarg.0
            IL_0007: ldarg.1
            IL_0008: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_000d: ldarg.0
            IL_000e: call int32 [mscorlib]System.Environment::get_CurrentManagedThreadId()
            IL_0013: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>l__initialThreadId'
            IL_0018: ret
        } // end of method '<M>d__1`2'::.ctor
        .method private final hidebysig newslot virtual 
            instance void System.IDisposable.Dispose () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.IDisposable::Dispose()
            // Method begins at RVA 0x20b9
            // Code size 9 (0x9)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldc.i4.s -2
            IL_0003: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_0008: ret
        } // end of method '<M>d__1`2'::System.IDisposable.Dispose
        .method private final hidebysig newslot virtual 
            instance bool MoveNext () cil managed 
        {
            .override method instance bool [mscorlib]System.Collections.IEnumerator::MoveNext()
            // Method begins at RVA 0x20c4
            // Code size 97 (0x61)
            .maxstack 4
            .locals init (
                [0] int32
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_0006: stloc.0
            IL_0007: ldloc.0
            IL_0008: brfalse.s IL_0010
            IL_000a: ldloc.0
            IL_000b: ldc.i4.1
            IL_000c: beq.s IL_0058
            IL_000e: ldc.i4.0
            IL_000f: ret
            IL_0010: ldarg.0
            IL_0011: ldc.i4.m1
            IL_0012: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_0017: ldarg.0
            IL_0018: ldarg.0
            IL_0019: ldfld class C`1<!0> class Extensions/'<M>d__1`2'<!T, !U>::o
            IL_001e: callvirt instance string class C`1<!T>::GetString()
            IL_0023: ldarg.0
            IL_0024: ldflda !1 class Extensions/'<M>d__1`2'<!T, !U>::u1
            IL_0029: constrained. !U
            IL_002f: callvirt instance string [mscorlib]System.Object::ToString()
            IL_0034: ldarg.0
            IL_0035: ldflda !0 class Extensions/'<M>d__1`2'<!T, !U>::t1
            IL_003a: constrained. !T
            IL_0040: callvirt instance string [mscorlib]System.Object::ToString()
            IL_0045: call string [mscorlib]System.String::Concat(string, string, string)
            IL_004a: stfld string class Extensions/'<M>d__1`2'<!T, !U>::'<>2__current'
            IL_004f: ldarg.0
            IL_0050: ldc.i4.1
            IL_0051: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_0056: ldc.i4.1
            IL_0057: ret
            IL_0058: ldarg.0
            IL_0059: ldc.i4.m1
            IL_005a: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_005f: ldc.i4.0
            IL_0060: ret
        } // end of method '<M>d__1`2'::MoveNext
        .method private final hidebysig specialname newslot virtual 
            instance string 'System.Collections.Generic.IEnumerator<System.String>.get_Current' () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance !0 class [mscorlib]System.Collections.Generic.IEnumerator`1<string>::get_Current()
            // Method begins at RVA 0x2131
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld string class Extensions/'<M>d__1`2'<!T, !U>::'<>2__current'
            IL_0006: ret
        } // end of method '<M>d__1`2'::'System.Collections.Generic.IEnumerator<System.String>.get_Current'
        .method private final hidebysig newslot virtual 
            instance void System.Collections.IEnumerator.Reset () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.Collections.IEnumerator::Reset()
            // Method begins at RVA 0x2139
            // Code size 6 (0x6)
            .maxstack 8
            IL_0000: newobj instance void [mscorlib]System.NotSupportedException::.ctor()
            IL_0005: throw
        } // end of method '<M>d__1`2'::System.Collections.IEnumerator.Reset
        .method private final hidebysig specialname newslot virtual 
            instance object System.Collections.IEnumerator.get_Current () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance object [mscorlib]System.Collections.IEnumerator::get_Current()
            // Method begins at RVA 0x2131
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld string class Extensions/'<M>d__1`2'<!T, !U>::'<>2__current'
            IL_0006: ret
        } // end of method '<M>d__1`2'::System.Collections.IEnumerator.get_Current
        .method private final hidebysig newslot virtual 
            instance class [mscorlib]System.Collections.Generic.IEnumerator`1<string> 'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator' () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance class [mscorlib]System.Collections.Generic.IEnumerator`1<!0> class [mscorlib]System.Collections.Generic.IEnumerable`1<string>::GetEnumerator()
            // Method begins at RVA 0x2140
            // Code size 79 (0x4f)
            .maxstack 2
            .locals init (
                [0] class Extensions/'<M>d__1`2'<!T, !U>
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_0006: ldc.i4.s -2
            IL_0008: bne.un.s IL_0022
            IL_000a: ldarg.0
            IL_000b: ldfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>l__initialThreadId'
            IL_0010: call int32 [mscorlib]System.Environment::get_CurrentManagedThreadId()
            IL_0015: bne.un.s IL_0022
            IL_0017: ldarg.0
            IL_0018: ldc.i4.0
            IL_0019: stfld int32 class Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_001e: ldarg.0
            IL_001f: stloc.0
            IL_0020: br.s IL_0029
            IL_0022: ldc.i4.0
            IL_0023: newobj instance void class Extensions/'<M>d__1`2'<!T, !U>::.ctor(int32)
            IL_0028: stloc.0
            IL_0029: ldloc.0
            IL_002a: ldarg.0
            IL_002b: ldfld class C`1<!0> class Extensions/'<M>d__1`2'<!T, !U>::'<>3__o'
            IL_0030: stfld class C`1<!0> class Extensions/'<M>d__1`2'<!T, !U>::o
            IL_0035: ldloc.0
            IL_0036: ldarg.0
            IL_0037: ldfld !0 class Extensions/'<M>d__1`2'<!T, !U>::'<>3__t1'
            IL_003c: stfld !0 class Extensions/'<M>d__1`2'<!T, !U>::t1
            IL_0041: ldloc.0
            IL_0042: ldarg.0
            IL_0043: ldfld !1 class Extensions/'<M>d__1`2'<!T, !U>::'<>3__u1'
            IL_0048: stfld !1 class Extensions/'<M>d__1`2'<!T, !U>::u1
            IL_004d: ldloc.0
            IL_004e: ret
        } // end of method '<M>d__1`2'::'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator'
        .method private final hidebysig newslot virtual 
            instance class [mscorlib]System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance class [mscorlib]System.Collections.IEnumerator [mscorlib]System.Collections.IEnumerable::GetEnumerator()
            // Method begins at RVA 0x219b
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance class [mscorlib]System.Collections.Generic.IEnumerator`1<string> class Extensions/'<M>d__1`2'<!T, !U>::'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator'()
            IL_0006: ret
        } // end of method '<M>d__1`2'::System.Collections.IEnumerable.GetEnumerator
        // Properties
        .property instance string 'System.Collections.Generic.IEnumerator<System.String>.Current'()
        {
            .get instance string Extensions/'<M>d__1`2'::'System.Collections.Generic.IEnumerator<System.String>.get_Current'()
        }
        .property instance object System.Collections.IEnumerator.Current()
        {
            .get instance object Extensions/'<M>d__1`2'::System.Collections.IEnumerator.get_Current()
        }
    } // end of class <M>d__1`2
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Collections.Generic.IEnumerable`1<string> M<T, U> (
            class C`1<!!T> o,
            !!T t1,
            !!U u1
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.IteratorStateMachineAttribute::.ctor(class [mscorlib]System.Type) = (
            01 00 14 45 78 74 65 6e 73 69 6f 6e 73 2b 3c 4d
            3e 64 5f 5f 31 60 32 00 00
        )
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 29 (0x1d)
        .maxstack 8
        IL_0000: ldc.i4.s -2
        IL_0002: newobj instance void class Extensions/'<M>d__1`2'<!!T, !!U>::.ctor(int32)
        IL_0007: dup
        IL_0008: ldarg.0
        IL_0009: stfld class C`1<!0> class Extensions/'<M>d__1`2'<!!T, !!U>::'<>3__o'
        IL_000e: dup
        IL_000f: ldarg.1
        IL_0010: stfld !0 class Extensions/'<M>d__1`2'<!!T, !!U>::'<>3__t1'
        IL_0015: dup
        IL_0016: ldarg.2
        IL_0017: stfld !1 class Extensions/'<M>d__1`2'<!!T, !!U>::'<>3__u1'
        IL_001c: ret
    } // end of method Extensions::M
} // end of class Extensions
""").Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        System.Collections.Generic.IEnumerable<string> M<U>(T t1, U u1)
        {
            yield return o.GetString() + u1.ToString() + t1.ToString();
        }
    }

    extension<T>(C<T> o)
    {
        System.Collections.Generic.IEnumerable<string> M<U>(T t1, U u1, int x)
        {
            yield return o.GetString() + u1.ToString() + t1.ToString();
        }
    }
}

public class C<T>
{
    public string GetString() => null;
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        foreach (var s in Test(new C<long>("1"), 2, "3"))
            System.Console.Write(s);
        foreach (var s in new C<long>("4").M2(5, "6"))
            System.Console.Write(s);
    }

    static System.Collections.Generic.IEnumerable<string> Test<T, U>(C<T> o, T t1, U u1)
    {
        return o.M(t1, u1);
    }
}

static class Extensions
{
    extension<T>(C<T> o)
    {
        public System.Collections.Generic.IEnumerable<string> M2<U>(T t1, U u1) => o.M(t1, u1);
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceMethod_11_Async_Generic()
    {
        var src1 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        public async System.Threading.Tasks.Task<string> M<U>(T t1, U u1)
        {
            await System.Threading.Tasks.Task.Yield();
            return o.GetString() + u1.ToString() + t1.ToString();
        }
    }
}

public class C<T>(string val)
{
    public string GetString() => val;
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1, symbolValidator: (m) =>
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            AssertEx.Equal("System.Runtime.CompilerServices.AsyncStateMachineAttribute(typeof(Extensions.<M>d__1<,>))", implementation.GetAttributes().Single().ToString());
        }).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D884D1E13988E83801B7574694E1C2C5'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    class C`1<!T> o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x21ce
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D884D1E13988E83801B7574694E1C2C5'::'<Extension>$'
        } // end of class <M>$D884D1E13988E83801B7574694E1C2C5
        // Methods
        .method public hidebysig 
            instance class [mscorlib]System.Threading.Tasks.Task`1<string> M<U> (
                !$T0 t1,
                !!U u1
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 38 38 34 44 31 45 31 33
                39 38 38 45 38 33 38 30 31 42 37 35 37 34 36 39
                34 45 31 43 32 43 35 00 00
            )
            // Method begins at RVA 0x20d2
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1'::M
    } // end of class <G>$4A1E373BE5A70EE56E2FA5F469AC30F9`1
    .class nested private auto ansi sealed beforefieldinit '<M>d__1`2'<T, U>
        extends [mscorlib]System.ValueType
        implements [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public int32 '<>1__state'
        .field public valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> '<>t__builder'
        .field public class C`1<!T> o
        .field public !U u1
        .field public !T t1
        .field private valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter '<>u__1'
        // Methods
        .method private final hidebysig newslot virtual 
            instance void MoveNext () cil managed 
        {
            .override method instance void [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine::MoveNext()
            // Method begins at RVA 0x20d8
            // Code size 202 (0xca)
            .maxstack 3
            .locals init (
                [0] int32,
                [1] string,
                [2] valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter,
                [3] valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable,
                [4] class [mscorlib]System.Exception
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_0006: stloc.0
            .try
            {
                IL_0007: ldloc.0
                IL_0008: brfalse.s IL_0044
                IL_000a: call valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable [mscorlib]System.Threading.Tasks.Task::Yield()
                IL_000f: stloc.3
                IL_0010: ldloca.s 3
                IL_0012: call instance valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter [mscorlib]System.Runtime.CompilerServices.YieldAwaitable::GetAwaiter()
                IL_0017: stloc.2
                IL_0018: ldloca.s 2
                IL_001a: call instance bool [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter::get_IsCompleted()
                IL_001f: brtrue.s IL_0060
                IL_0021: ldarg.0
                IL_0022: ldc.i4.0
                IL_0023: dup
                IL_0024: stloc.0
                IL_0025: stfld int32 valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
                IL_002a: ldarg.0
                IL_002b: ldloc.2
                IL_002c: stfld valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>u__1'
                IL_0031: ldarg.0
                IL_0032: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>t__builder'
                IL_0037: ldloca.s 2
                IL_0039: ldarg.0
                IL_003a: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::AwaitUnsafeOnCompleted<valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter, valuetype Extensions/'<M>d__1`2'<!T, !U>>(!!0&, !!1&)
                IL_003f: leave IL_00c9
                IL_0044: ldarg.0
                IL_0045: ldfld valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>u__1'
                IL_004a: stloc.2
                IL_004b: ldarg.0
                IL_004c: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>u__1'
                IL_0051: initobj [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter
                IL_0057: ldarg.0
                IL_0058: ldc.i4.m1
                IL_0059: dup
                IL_005a: stloc.0
                IL_005b: stfld int32 valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
                IL_0060: ldloca.s 2
                IL_0062: call instance void [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter::GetResult()
                IL_0067: ldarg.0
                IL_0068: ldfld class C`1<!0> valuetype Extensions/'<M>d__1`2'<!T, !U>::o
                IL_006d: callvirt instance string class C`1<!T>::GetString()
                IL_0072: ldarg.0
                IL_0073: ldflda !1 valuetype Extensions/'<M>d__1`2'<!T, !U>::u1
                IL_0078: constrained. !U
                IL_007e: callvirt instance string [mscorlib]System.Object::ToString()
                IL_0083: ldarg.0
                IL_0084: ldflda !0 valuetype Extensions/'<M>d__1`2'<!T, !U>::t1
                IL_0089: constrained. !T
                IL_008f: callvirt instance string [mscorlib]System.Object::ToString()
                IL_0094: call string [mscorlib]System.String::Concat(string, string, string)
                IL_0099: stloc.1
                IL_009a: leave.s IL_00b5
            } // end .try
            catch [mscorlib]System.Exception
            {
                IL_009c: stloc.s 4
                IL_009e: ldarg.0
                IL_009f: ldc.i4.s -2
                IL_00a1: stfld int32 valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
                IL_00a6: ldarg.0
                IL_00a7: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>t__builder'
                IL_00ac: ldloc.s 4
                IL_00ae: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetException(class [mscorlib]System.Exception)
                IL_00b3: leave.s IL_00c9
            } // end handler
            IL_00b5: ldarg.0
            IL_00b6: ldc.i4.s -2
            IL_00b8: stfld int32 valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>1__state'
            IL_00bd: ldarg.0
            IL_00be: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>t__builder'
            IL_00c3: ldloc.1
            IL_00c4: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetResult(!0)
            IL_00c9: ret
        } // end of method '<M>d__1`2'::MoveNext
        .method private final hidebysig newslot virtual 
            instance void SetStateMachine (
                class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine stateMachine
            ) cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine::SetStateMachine(class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine)
            // Method begins at RVA 0x21c0
            // Code size 13 (0xd)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!T, !U>::'<>t__builder'
            IL_0006: ldarg.1
            IL_0007: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetStateMachine(class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine)
            IL_000c: ret
        } // end of method '<M>d__1`2'::SetStateMachine
    } // end of class <M>d__1`2
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Threading.Tasks.Task`1<string> M<T, U> (
            class C`1<!!T> o,
            !!T t1,
            !!U u1
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.AsyncStateMachineAttribute::.ctor(class [mscorlib]System.Type) = (
            01 00 14 45 78 74 65 6e 73 69 6f 6e 73 2b 3c 4d
            3e 64 5f 5f 31 60 32 00 00
        )
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2068
        // Code size 71 (0x47)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<M>d__1`2'<!!T, !!U>
        )
        IL_0000: ldloca.s 0
        IL_0002: call valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<!0> valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::Create()
        IL_0007: stfld valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!!T, !!U>::'<>t__builder'
        IL_000c: ldloca.s 0
        IL_000e: ldarg.0
        IL_000f: stfld class C`1<!0> valuetype Extensions/'<M>d__1`2'<!!T, !!U>::o
        IL_0014: ldloca.s 0
        IL_0016: ldarg.1
        IL_0017: stfld !0 valuetype Extensions/'<M>d__1`2'<!!T, !!U>::t1
        IL_001c: ldloca.s 0
        IL_001e: ldarg.2
        IL_001f: stfld !1 valuetype Extensions/'<M>d__1`2'<!!T, !!U>::u1
        IL_0024: ldloca.s 0
        IL_0026: ldc.i4.m1
        IL_0027: stfld int32 valuetype Extensions/'<M>d__1`2'<!!T, !!U>::'<>1__state'
        IL_002c: ldloca.s 0
        IL_002e: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!!T, !!U>::'<>t__builder'
        IL_0033: ldloca.s 0
        IL_0035: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::Start<valuetype Extensions/'<M>d__1`2'<!!T, !!U>>(!!0&)
        IL_003a: ldloca.s 0
        IL_003c: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> valuetype Extensions/'<M>d__1`2'<!!T, !!U>::'<>t__builder'
        IL_0041: call instance class [mscorlib]System.Threading.Tasks.Task`1<!0> valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::get_Task()
        IL_0046: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(C<T> o)
    {
        async System.Threading.Tasks.Task<string> M<U>(T t1, U u1)
        {
            await System.Threading.Tasks.Task.Yield();
            return o.GetString() + u1.ToString() + t1.ToString();
        }
    }

    extension<T>(C<T> o)
    {
        async System.Threading.Tasks.Task<string> M<U>(T t1, U u1, int x)
        {
            await System.Threading.Tasks.Task.Yield();
            return o.GetString() + u1.ToString() + t1.ToString();
        }
    }
}

public class C<T>
{
    public string GetString() => null;
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test(new C<long>("1"), 2, "3").Result);
        System.Console.Write(new C<long>("4").M2(5, "6").Result);
    }

    async static System.Threading.Tasks.Task<string> Test<T, U>(C<T> o, T t1, U u1)
    {
        await System.Threading.Tasks.Task.Yield();
        return await o.M(t1, u1);
    }
}

static class Extensions
{
    extension<T>(C<T> o)
    {
        async public System.Threading.Tasks.Task<string> M2<U>(T t1, U u1)
        {
            await System.Threading.Tasks.Task.Yield();
            return await o.M(t1, u1);
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "132465").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_StaticMethod_01()
    {
        var src1 = """
public static class Extensions
{
    extension(object _)
    {
        public static string M(object o, string s)
        {
            return o + s;
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);

        var verifier1 = CompileAndVerify(comp1, sourceSymbolValidator: verifySymbols, symbolValidator: verifySymbols).VerifyDiagnostics();

        static void verifySymbols(ModuleSymbol m)
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            Assert.True(implementation.IsStatic);
            Assert.Equal(MethodKind.Ordinary, implementation.MethodKind);
            Assert.Equal(2, implementation.ParameterCount);
            AssertEx.Equal("System.String Extensions.M(System.Object o, System.String s)", implementation.ToTestDisplayString());
            Assert.Equal(m is not PEModuleSymbol, implementation.IsImplicitlyDeclared);
            Assert.False(implementation.IsExtensionMethod);
            Assert.False(implementation.HasSpecialName);
            Assert.False(implementation.HasRuntimeSpecialName);

            Assert.True(implementation.ContainingType.MightContainExtensionMethods);

            if (m is PEModuleSymbol peModuleSymbol)
            {
                Assert.True(peModuleSymbol.Module.HasExtensionAttribute(((PEAssemblySymbol)peModuleSymbol.ContainingAssembly).Assembly.Handle, ignoreCase: false));
            }
        }

        var expectedTypeIL = """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$3D34838CB2C73A4E406AE3905787D97D'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object _
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x207e
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$3D34838CB2C73A4E406AE3905787D97D'::'<Extension>$'
        } // end of class <M>$3D34838CB2C73A4E406AE3905787D97D
        // Methods
        .method public hidebysig static 
            string M (
                object o,
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 33 44 33 34 38 33 38 43 42
                32 43 37 33 41 34 45 34 30 36 41 45 33 39 30 35
                37 38 37 44 39 37 44 00 00
            )
            // Method begins at RVA 0x207b
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 19 (0x13)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: brtrue.s IL_0006
        IL_0003: ldnull
        IL_0004: br.s IL_000c
        IL_0006: ldarg.0
        IL_0007: callvirt instance string [mscorlib]System.Object::ToString()
        IL_000c: ldarg.1
        IL_000d: call string [mscorlib]System.String::Concat(string, string)
        IL_0012: ret
    } // end of method Extensions::M
} // end of class Extensions
""";

        verifier1.VerifyTypeIL("Extensions", expectedTypeIL.Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3", "4"));
    }

    static string Test(object o)
    {
        return object.M(o, "2");
    }
}

static class Extensions
{
    extension(object o)
    {
        public static string M2(object o1, string s)
        {
            return object.M(o1, s);
        }
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        var verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       17 (0x11)
  .maxstack  2
  .locals init (string V_0)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldstr      ""2""
  IL_0007:  call       ""string Extensions.M(object, string)""
  IL_000c:  stloc.0
  IL_000d:  br.s       IL_000f
  IL_000f:  ldloc.0
  IL_0010:  ret
}
";
        verifier2.VerifyIL("Program.Test", testIL);

        var m2IL =
@"
{
  // Code size        9 (0x9)
  .maxstack  2
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  ldarg.1
  IL_0003:  call       ""string Extensions.M(object, string)""
  IL_0008:  ret
}
";
        verifier2.VerifyIL("Extensions.M2", m2IL);

        var comp1ImageReference = comp1.EmitToImageReference();
        comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        verifier2.VerifyIL("Program.Test", testIL);
        verifier2.VerifyIL("Extensions.M2", m2IL);

        comp2 = CreateCompilationWithIL(src2, expectedTypeIL + ExtensionMarkerAttributeIL, options: TestOptions.DebugExe);
        CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        var remove = """
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
""";

        comp2 = CreateCompilationWithIL(src2, expectedTypeIL.Remove(expectedTypeIL.IndexOf(remove), remove.Length) + ExtensionMarkerAttributeIL);
        comp2.VerifyDiagnostics(
            // (11,23): error CS0117: 'object' does not contain a definition for 'M'
            //         return object.M(o, "2");
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(11, 23),
            // (21,27): error CS0117: 'object' does not contain a definition for 'M'
            //             return object.M(o, s);
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(21, 27)
            );

        src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3", "4"));
    }

    static string Test(object o)
    {
        return Extensions.M(o, "2");
    }
}

static class Extensions_
{
    extension(object o)
    {
        public static string M2(object o1, string s)
        {
            return Extensions.M(o1, s);
        }
    }
}
""";

        comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        verifier2.VerifyIL("Program.Test", testIL);
        verifier2.VerifyIL("Extensions_.M2", m2IL);

        comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

        verifier2.VerifyIL("Program.Test", testIL);
        verifier2.VerifyIL("Extensions_.M2", m2IL);

        var vbComp = CreateVisualBasicCompilation("""
Class Program
    Shared Sub Main()
        System.Console.Write(Test2("3"))
    End Sub

    Shared Function Test2(o As String) As String
        return Extensions.M(o, "4")
    End Function
End Class
""",
            referencedAssemblies: comp2.References, compilationOptions: new VisualBasicCompilationOptions(OutputKind.ConsoleApplication));

        CompileAndVerify(vbComp, expectedOutput: "34").VerifyDiagnostics();

        if (!CompilationExtensions.EnableVerifyUsedAssemblies) // Tracked by https://github.com/dotnet/roslyn/issues/77542
        {
            src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3", "4"));
    }

    static string Test(object o)
    {
        System.Func<object, string, string> d = object.M;
        return d(o, "2");
    }
}

static class Extensions
{
    extension(object o)
    {
        public static string M2(object o1, string s)
        {
            return new System.Func<object, string, string>(object.M)(o1, s);
        }
    }
}
""";

            comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe);
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

            testIL =
    @"
{
  // Code size       46 (0x2e)
  .maxstack  3
  .locals init (System.Func<object, string, string> V_0, //d
                string V_1)
  IL_0000:  nop
  IL_0001:  ldsfld     ""System.Func<object, string, string> Program.<>O.<0>__M""
  IL_0006:  dup
  IL_0007:  brtrue.s   IL_001c
  IL_0009:  pop
  IL_000a:  ldnull
  IL_000b:  ldftn      ""string Extensions.M(object, string)""
  IL_0011:  newobj     ""System.Func<object, string, string>..ctor(object, System.IntPtr)""
  IL_0016:  dup
  IL_0017:  stsfld     ""System.Func<object, string, string> Program.<>O.<0>__M""
  IL_001c:  stloc.0
  IL_001d:  ldloc.0
  IL_001e:  ldarg.0
  IL_001f:  ldstr      ""2""
  IL_0024:  callvirt   ""string System.Func<object, string, string>.Invoke(object, string)""
  IL_0029:  stloc.1
  IL_002a:  br.s       IL_002c
  IL_002c:  ldloc.1
  IL_002d:  ret
}
";
            verifier2.VerifyIL("Program.Test", testIL);

            m2IL =
    @"
{
  // Code size       21 (0x15)
  .maxstack  3
  IL_0000:  nop
  IL_0001:  ldnull
  IL_0002:  ldftn      ""string Extensions.M(object, string)""
  IL_0008:  newobj     ""System.Func<object, string, string>..ctor(object, System.IntPtr)""
  IL_000d:  ldarg.0
  IL_000e:  ldarg.1
  IL_000f:  callvirt   ""string System.Func<object, string, string>.Invoke(object, string)""
  IL_0014:  ret
}
";
            verifier2.VerifyIL("Extensions.M2", m2IL);

            comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe);
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

            verifier2.VerifyIL("Program.Test", testIL);
            verifier2.VerifyIL("Extensions.M2", m2IL);

            src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3", "4"));
    }

    static string Test(object o)
    {
        System.Func<object, string, string> d = Extensions.M;
        return d(o, "2");
    }
}

static class Extensions_
{
    extension(object o)
    {
        public static string M2(object o1, string s)
        {
            return new System.Func<object, string, string>(Extensions.M)(o1, s);
        }
    }
}
""";

            comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe);
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

            verifier2.VerifyIL("Program.Test", testIL);
            verifier2.VerifyIL("Extensions_.M2", m2IL);

            comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe);
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234").VerifyDiagnostics();

            verifier2.VerifyIL("Program.Test", testIL);
            verifier2.VerifyIL("Extensions_.M2", m2IL);

            src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3", "4"));
    }

    unsafe static string Test(object o)
    {
        delegate*<object, string, string> d = &object.M;
        return d(o, "2");
    }
}

static class Extensions
{
    extension(object o)
    {
        unsafe public static string M2(object o1, string s)
        {
            return ((delegate*<object, string, string>)&object.M)(o1, s);
        }
    }
}
""";

            comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe.WithAllowUnsafe(true));
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234", verify: Verification.Skipped).VerifyDiagnostics();

            testIL =
@"
{
  // Code size       27 (0x1b)
  .maxstack  3
  .locals init (delegate*<object, string, string> V_0, //d
                delegate*<object, string, string> V_1,
                string V_2)
  IL_0000:  nop
  IL_0001:  ldftn      ""string Extensions.M(object, string)""
  IL_0007:  stloc.0
  IL_0008:  ldloc.0
  IL_0009:  stloc.1
  IL_000a:  ldarg.0
  IL_000b:  ldstr      ""2""
  IL_0010:  ldloc.1
  IL_0011:  calli      ""delegate*<object, string, string>""
  IL_0016:  stloc.2
  IL_0017:  br.s       IL_0019
  IL_0019:  ldloc.2
  IL_001a:  ret
}
";
            verifier2.VerifyIL("Program.Test", testIL);

            m2IL =
@"
{
  // Code size       17 (0x11)
  .maxstack  3
  .locals init (delegate*<object, string, string> V_0)
  IL_0000:  nop
  IL_0001:  ldftn      ""string Extensions.M(object, string)""
  IL_0007:  stloc.0
  IL_0008:  ldarg.0
  IL_0009:  ldarg.1
  IL_000a:  ldloc.0
  IL_000b:  calli      ""delegate*<object, string, string>""
  IL_0010:  ret
}
";
            verifier2.VerifyIL("Extensions.M2", m2IL);

            comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe.WithAllowUnsafe(true));
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234", verify: Verification.Skipped).VerifyDiagnostics();

            verifier2.VerifyIL("Program.Test", testIL);
            verifier2.VerifyIL("Extensions.M2", m2IL);

            src2 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3", "4"));
    }

    unsafe static string Test(object o)
    {
        delegate*<object, string, string> d = &Extensions.M;
        return d(o, "2");
    }
}

static class Extensions_
{
    extension(object o)
    {
        unsafe public static string M2(object o1, string s)
        {
            return ((delegate*<object, string, string>)&Extensions.M)(o1, s);
        }
    }
}
""";

            comp2 = CreateCompilation(src2, references: [comp1MetadataReference], options: TestOptions.DebugExe.WithAllowUnsafe(true));
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234", verify: Verification.Skipped).VerifyDiagnostics();

            verifier2.VerifyIL("Program.Test", testIL);
            verifier2.VerifyIL("Extensions_.M2", m2IL);

            comp2 = CreateCompilation(src2, references: [comp1ImageReference], options: TestOptions.DebugExe.WithAllowUnsafe(true));
            verifier2 = CompileAndVerify(comp2, expectedOutput: "1234", verify: Verification.Skipped).VerifyDiagnostics();

            verifier2.VerifyIL("Program.Test", testIL);
            verifier2.VerifyIL("Extensions_.M2", m2IL);
        }

        var comp5 = CreateCompilation(src1);
        comp5.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp5.VerifyEmitDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object _)
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5)
            );
    }

    [Fact]
    public void Implementation_StaticMethod_02_WithLocalFunction()
    {
        var src1 = """
public static class Extensions
{
    extension(object _)
    {
        public static string M(object o, string s)
        {
            string local() => o + s;
            return local();
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$3D34838CB2C73A4E406AE3905787D97D'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object _
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20ae
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$3D34838CB2C73A4E406AE3905787D97D'::'<Extension>$'
        } // end of class <M>$3D34838CB2C73A4E406AE3905787D97D
        // Methods
        .method public hidebysig static 
            string M (
                object o,
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 33 44 33 34 38 33 38 43 42
                32 43 37 33 41 34 45 34 30 36 41 45 33 39 30 35
                37 38 37 44 39 37 44 00 00
            )
            // Method begins at RVA 0x20ab
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.ValueType
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public object o
        .field public string s
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        // Method begins at RVA 0x2068
        // Code size 24 (0x18)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<>c__DisplayClass1_0'
        )
        IL_0000: ldloca.s 0
        IL_0002: ldarg.0
        IL_0003: stfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_0008: ldloca.s 0
        IL_000a: ldarg.1
        IL_000b: stfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0010: ldloca.s 0
        IL_0012: call string Extensions::'<M>g__local|1_0'(valuetype Extensions/'<>c__DisplayClass1_0'&)
        IL_0017: ret
    } // end of method Extensions::M
    .method assembly hidebysig static 
        string '<M>g__local|1_0' (
            valuetype Extensions/'<>c__DisplayClass1_0'& ''
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x208c
        // Code size 30 (0x1e)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: ldfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_0006: dup
        IL_0007: brtrue.s IL_000d
        IL_0009: pop
        IL_000a: ldnull
        IL_000b: br.s IL_0012
        IL_000d: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0012: ldarg.0
        IL_0013: ldfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0018: call string [mscorlib]System.String::Concat(string, string)
        IL_001d: ret
    } // end of method Extensions::'<M>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object _)
    {
        public static string M(object o, string s)
        {
            string local() => o + s;
            return local();
        }
    }

    extension(object)
    {
        public static string M(object o, string s, int x)
        {
            string local() => o + s;
            return local();
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2, symbolValidator: (m) =>
        {
            var container = m.GlobalNamespace.GetTypeMember("Extensions");
            var extensions = container.GetTypeMembers();

            AssertEx.Equal("System.Object _", extensions[0].ExtensionParameter.ToTestDisplayString());
            AssertEx.Equal("<M>$3D34838CB2C73A4E406AE3905787D97D", extensions[0].MetadataName);
            Symbol m1 = extensions[0].GetMembers().Single();
            AssertEx.Equal("Extensions.extension(object).M(object, string)", m1.ToDisplayString());
            AssertEx.Equal([], m1.GetAttributes());

            AssertEx.Equal("System.Object value", extensions[1].ExtensionParameter.ToTestDisplayString());
            AssertEx.Equal("<M>$C43E2675C7BBF9284AF22FB8A9BF0280", extensions[1].MetadataName);
            Symbol m2 = extensions[1].GetMembers().Single();
            AssertEx.Equal("Extensions.extension(object).M(object, string, int)", m2.ToDisplayString());
            AssertEx.Equal([], m2.GetAttributes());
        }).VerifyDiagnostics().
           VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$3D34838CB2C73A4E406AE3905787D97D'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object _
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20f1
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$3D34838CB2C73A4E406AE3905787D97D'::'<Extension>$'
        } // end of class <M>$3D34838CB2C73A4E406AE3905787D97D
        .class nested public auto ansi abstract sealed specialname '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20f1
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'::'<Extension>$'
        } // end of class <M>$C43E2675C7BBF9284AF22FB8A9BF0280
        // Methods
        .method public hidebysig static 
            string M (
                object o,
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 33 44 33 34 38 33 38 43 42
                32 43 37 33 41 34 45 34 30 36 41 45 33 39 30 35
                37 38 37 44 39 37 44 00 00
            )
            // Method begins at RVA 0x20ee
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
        .method public hidebysig static 
            string M (
                object o,
                string s,
                int32 x
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            // Method begins at RVA 0x20ee
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.ValueType
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public object o
        .field public string s
    } // end of class <>c__DisplayClass1_0
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass3_0'
        extends [mscorlib]System.ValueType
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public object o
        .field public string s
    } // end of class <>c__DisplayClass3_0
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        // Method begins at RVA 0x2068
        // Code size 24 (0x18)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<>c__DisplayClass1_0'
        )
        IL_0000: ldloca.s 0
        IL_0002: ldarg.0
        IL_0003: stfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_0008: ldloca.s 0
        IL_000a: ldarg.1
        IL_000b: stfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0010: ldloca.s 0
        IL_0012: call string Extensions::'<M>g__local|1_0'(valuetype Extensions/'<>c__DisplayClass1_0'&)
        IL_0017: ret
    } // end of method Extensions::M
    .method public hidebysig static 
        string M (
            object o,
            string s,
            int32 x
        ) cil managed 
    {
        // Method begins at RVA 0x208c
        // Code size 24 (0x18)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<>c__DisplayClass3_0'
        )
        IL_0000: ldloca.s 0
        IL_0002: ldarg.0
        IL_0003: stfld object Extensions/'<>c__DisplayClass3_0'::o
        IL_0008: ldloca.s 0
        IL_000a: ldarg.1
        IL_000b: stfld string Extensions/'<>c__DisplayClass3_0'::s
        IL_0010: ldloca.s 0
        IL_0012: call string Extensions::'<M>g__local|3_0'(valuetype Extensions/'<>c__DisplayClass3_0'&)
        IL_0017: ret
    } // end of method Extensions::M
    .method assembly hidebysig static 
        string '<M>g__local|1_0' (
            valuetype Extensions/'<>c__DisplayClass1_0'& ''
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x20b0
        // Code size 30 (0x1e)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: ldfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_0006: dup
        IL_0007: brtrue.s IL_000d
        IL_0009: pop
        IL_000a: ldnull
        IL_000b: br.s IL_0012
        IL_000d: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0012: ldarg.0
        IL_0013: ldfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0018: call string [mscorlib]System.String::Concat(string, string)
        IL_001d: ret
    } // end of method Extensions::'<M>g__local|1_0'
    .method assembly hidebysig static 
        string '<M>g__local|3_0' (
            valuetype Extensions/'<>c__DisplayClass3_0'& ''
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x20cf
        // Code size 30 (0x1e)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: ldfld object Extensions/'<>c__DisplayClass3_0'::o
        IL_0006: dup
        IL_0007: brtrue.s IL_000d
        IL_0009: pop
        IL_000a: ldnull
        IL_000b: br.s IL_0012
        IL_000d: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0012: ldarg.0
        IL_0013: ldfld string Extensions/'<>c__DisplayClass3_0'::s
        IL_0018: call string [mscorlib]System.String::Concat(string, string)
        IL_001d: ret
    } // end of method Extensions::'<M>g__local|3_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3","4"));
    }

    static string Test(object o)
    {
        string local() => object.M(o, "2");
        return local();
    }
}

static class Extensions
{
    extension(object _)
    {
        public static string M2(object o, string s)
        {
            string local() => object.M(o, s);
            return local();
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_StaticMethod_03_WithLambda()
    {
        var src1 = """
public static class Extensions
{
    extension(object _)
    {
        public static string M(object o, string s)
        {
            System.Func<string> local = () => o + s;
            return local();
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$3D34838CB2C73A4E406AE3905787D97D'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object _
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20b6
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$3D34838CB2C73A4E406AE3905787D97D'::'<Extension>$'
        } // end of class <M>$3D34838CB2C73A4E406AE3905787D97D
        // Methods
        .method public hidebysig static 
            string M (
                object o,
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 33 44 33 34 38 33 38 43 42
                32 43 37 33 41 34 45 34 30 36 41 45 33 39 30 35
                37 38 37 44 39 37 44 00 00
            )
            // Method begins at RVA 0x208c
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<>c__DisplayClass1_0'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public object o
        .field public string s
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor () cil managed 
        {
            // Method begins at RVA 0x208f
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ret
        } // end of method '<>c__DisplayClass1_0'::.ctor
        .method assembly hidebysig 
            instance string '<M>b__0' () cil managed 
        {
            // Method begins at RVA 0x2097
            // Code size 30 (0x1e)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld object Extensions/'<>c__DisplayClass1_0'::o
            IL_0006: dup
            IL_0007: brtrue.s IL_000d
            IL_0009: pop
            IL_000a: ldnull
            IL_000b: br.s IL_0012
            IL_000d: callvirt instance string [mscorlib]System.Object::ToString()
            IL_0012: ldarg.0
            IL_0013: ldfld string Extensions/'<>c__DisplayClass1_0'::s
            IL_0018: call string [mscorlib]System.String::Concat(string, string)
            IL_001d: ret
        } // end of method '<>c__DisplayClass1_0'::'<M>b__0'
    } // end of class <>c__DisplayClass1_0
    // Methods
    .method public hidebysig static 
        string M (
            object o,
            string s
        ) cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 36 (0x24)
        .maxstack 8
        IL_0000: newobj instance void Extensions/'<>c__DisplayClass1_0'::.ctor()
        IL_0005: dup
        IL_0006: ldarg.0
        IL_0007: stfld object Extensions/'<>c__DisplayClass1_0'::o
        IL_000c: dup
        IL_000d: ldarg.1
        IL_000e: stfld string Extensions/'<>c__DisplayClass1_0'::s
        IL_0013: ldftn instance string Extensions/'<>c__DisplayClass1_0'::'<M>b__0'()
        IL_0019: newobj instance void class [mscorlib]System.Func`1<string>::.ctor(object, native int)
        IL_001e: callvirt instance !0 class [mscorlib]System.Func`1<string>::Invoke()
        IL_0023: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object _)
    {
        static string M(object o, string s)
        {
            System.Func<string> local = () => o + s;
            return local();
        }
    }

    extension(object)
    {
        static string M(object o, string s, int x)
        {
            System.Func<string> local = () => o + s;
            return local();
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write(object.M2("3","4"));
    }

    static string Test(object o)
    {
        System.Func<string> local = () => object.M(o, "2");
        return local();
    }
}

static class Extensions
{
    extension(object _)
    {
        public static string M2(object o, string s)
        {
            System.Func<string> local = () => object.M(o, s);
            return local();
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_StaticMethod_04_Iterator()
    {
        var src1 = """
public static class Extensions
{
    extension(object _)
    {
        public static System.Collections.Generic.IEnumerable<string> M(object o, string s)
        {
            yield return o + s;
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1, symbolValidator: (m) =>
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            AssertEx.Equal("System.Runtime.CompilerServices.IteratorStateMachineAttribute(typeof(Extensions.<M>d__1))", implementation.GetAttributes().Single().ToString());
        }).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", ("""
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$3D34838CB2C73A4E406AE3905787D97D'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object _
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2167
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$3D34838CB2C73A4E406AE3905787D97D'::'<Extension>$'
        } // end of class <M>$3D34838CB2C73A4E406AE3905787D97D
        // Methods
        .method public hidebysig static 
            class [mscorlib]System.Collections.Generic.IEnumerable`1<string> M (
                object o,
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 33 44 33 34 38 33 38 43 42
                32 43 37 33 41 34 45 34 30 36 41 45 33 39 30 35
                37 38 37 44 39 37 44 00 00
            )
            // Method begins at RVA 0x207e
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<M>d__1'
        extends [mscorlib]System.Object
        implements class [mscorlib]System.Collections.Generic.IEnumerable`1<string>,
                   [mscorlib]System.Collections.IEnumerable,
                   class [mscorlib]System.Collections.Generic.IEnumerator`1<string>,

""" +
        (ExecutionConditionUtil.IsMonoOrCoreClr ?
"""
                   [mscorlib]System.Collections.IEnumerator,
                   [mscorlib]System.IDisposable

""" :
"""
                   [mscorlib]System.IDisposable,
                   [mscorlib]System.Collections.IEnumerator

""") +
"""
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field private int32 '<>1__state'
        .field private string '<>2__current'
        .field private int32 '<>l__initialThreadId'
        .field private object o
        .field public object '<>3__o'
        .field private string s
        .field public string '<>3__s'
        // Methods
        .method public hidebysig specialname rtspecialname 
            instance void .ctor (
                int32 '<>1__state'
            ) cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            // Method begins at RVA 0x2081
            // Code size 25 (0x19)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance void [mscorlib]System.Object::.ctor()
            IL_0006: ldarg.0
            IL_0007: ldarg.1
            IL_0008: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_000d: ldarg.0
            IL_000e: call int32 [mscorlib]System.Environment::get_CurrentManagedThreadId()
            IL_0013: stfld int32 Extensions/'<M>d__1'::'<>l__initialThreadId'
            IL_0018: ret
        } // end of method '<M>d__1'::.ctor
        .method private final hidebysig newslot virtual 
            instance void System.IDisposable.Dispose () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.IDisposable::Dispose()
            // Method begins at RVA 0x209b
            // Code size 9 (0x9)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldc.i4.s -2
            IL_0003: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0008: ret
        } // end of method '<M>d__1'::System.IDisposable.Dispose
        .method private final hidebysig newslot virtual 
            instance bool MoveNext () cil managed 
        {
            .override method instance bool [mscorlib]System.Collections.IEnumerator::MoveNext()
            // Method begins at RVA 0x20a8
            // Code size 76 (0x4c)
            .maxstack 3
            .locals init (
                [0] int32
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0006: stloc.0
            IL_0007: ldloc.0
            IL_0008: brfalse.s IL_0010
            IL_000a: ldloc.0
            IL_000b: ldc.i4.1
            IL_000c: beq.s IL_0043
            IL_000e: ldc.i4.0
            IL_000f: ret
            IL_0010: ldarg.0
            IL_0011: ldc.i4.m1
            IL_0012: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0017: ldarg.0
            IL_0018: ldarg.0
            IL_0019: ldfld object Extensions/'<M>d__1'::o
            IL_001e: dup
            IL_001f: brtrue.s IL_0025
            IL_0021: pop
            IL_0022: ldnull
            IL_0023: br.s IL_002a
            IL_0025: callvirt instance string [mscorlib]System.Object::ToString()
            IL_002a: ldarg.0
            IL_002b: ldfld string Extensions/'<M>d__1'::s
            IL_0030: call string [mscorlib]System.String::Concat(string, string)
            IL_0035: stfld string Extensions/'<M>d__1'::'<>2__current'
            IL_003a: ldarg.0
            IL_003b: ldc.i4.1
            IL_003c: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0041: ldc.i4.1
            IL_0042: ret
            IL_0043: ldarg.0
            IL_0044: ldc.i4.m1
            IL_0045: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_004a: ldc.i4.0
            IL_004b: ret
        } // end of method '<M>d__1'::MoveNext
        .method private final hidebysig specialname newslot virtual 
            instance string 'System.Collections.Generic.IEnumerator<System.String>.get_Current' () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance !0 class [mscorlib]System.Collections.Generic.IEnumerator`1<string>::get_Current()
            // Method begins at RVA 0x2100
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld string Extensions/'<M>d__1'::'<>2__current'
            IL_0006: ret
        } // end of method '<M>d__1'::'System.Collections.Generic.IEnumerator<System.String>.get_Current'
        .method private final hidebysig newslot virtual 
            instance void System.Collections.IEnumerator.Reset () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.Collections.IEnumerator::Reset()
            // Method begins at RVA 0x2108
            // Code size 6 (0x6)
            .maxstack 8
            IL_0000: newobj instance void [mscorlib]System.NotSupportedException::.ctor()
            IL_0005: throw
        } // end of method '<M>d__1'::System.Collections.IEnumerator.Reset
        .method private final hidebysig specialname newslot virtual 
            instance object System.Collections.IEnumerator.get_Current () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance object [mscorlib]System.Collections.IEnumerator::get_Current()
            // Method begins at RVA 0x2100
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldfld string Extensions/'<M>d__1'::'<>2__current'
            IL_0006: ret
        } // end of method '<M>d__1'::System.Collections.IEnumerator.get_Current
        .method private final hidebysig newslot virtual 
            instance class [mscorlib]System.Collections.Generic.IEnumerator`1<string> 'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator' () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance class [mscorlib]System.Collections.Generic.IEnumerator`1<!0> class [mscorlib]System.Collections.Generic.IEnumerable`1<string>::GetEnumerator()
            // Method begins at RVA 0x2110
            // Code size 67 (0x43)
            .maxstack 2
            .locals init (
                [0] class Extensions/'<M>d__1'
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0006: ldc.i4.s -2
            IL_0008: bne.un.s IL_0022
            IL_000a: ldarg.0
            IL_000b: ldfld int32 Extensions/'<M>d__1'::'<>l__initialThreadId'
            IL_0010: call int32 [mscorlib]System.Environment::get_CurrentManagedThreadId()
            IL_0015: bne.un.s IL_0022
            IL_0017: ldarg.0
            IL_0018: ldc.i4.0
            IL_0019: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_001e: ldarg.0
            IL_001f: stloc.0
            IL_0020: br.s IL_0029
            IL_0022: ldc.i4.0
            IL_0023: newobj instance void Extensions/'<M>d__1'::.ctor(int32)
            IL_0028: stloc.0
            IL_0029: ldloc.0
            IL_002a: ldarg.0
            IL_002b: ldfld object Extensions/'<M>d__1'::'<>3__o'
            IL_0030: stfld object Extensions/'<M>d__1'::o
            IL_0035: ldloc.0
            IL_0036: ldarg.0
            IL_0037: ldfld string Extensions/'<M>d__1'::'<>3__s'
            IL_003c: stfld string Extensions/'<M>d__1'::s
            IL_0041: ldloc.0
            IL_0042: ret
        } // end of method '<M>d__1'::'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator'
        .method private final hidebysig newslot virtual 
            instance class [mscorlib]System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator () cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance class [mscorlib]System.Collections.IEnumerator [mscorlib]System.Collections.IEnumerable::GetEnumerator()
            // Method begins at RVA 0x215f
            // Code size 7 (0x7)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: call instance class [mscorlib]System.Collections.Generic.IEnumerator`1<string> Extensions/'<M>d__1'::'System.Collections.Generic.IEnumerable<System.String>.GetEnumerator'()
            IL_0006: ret
        } // end of method '<M>d__1'::System.Collections.IEnumerable.GetEnumerator
        // Properties
        .property instance string 'System.Collections.Generic.IEnumerator<System.String>.Current'()
        {
            .get instance string Extensions/'<M>d__1'::'System.Collections.Generic.IEnumerator<System.String>.get_Current'()
        }
        .property instance object System.Collections.IEnumerator.Current()
        {
            .get instance object Extensions/'<M>d__1'::System.Collections.IEnumerator.get_Current()
        }
    } // end of class <M>d__1
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Collections.Generic.IEnumerable`1<string> M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.IteratorStateMachineAttribute::.ctor(class [mscorlib]System.Type) = (
            01 00 12 45 78 74 65 6e 73 69 6f 6e 73 2b 3c 4d
            3e 64 5f 5f 31 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 22 (0x16)
        .maxstack 8
        IL_0000: ldc.i4.s -2
        IL_0002: newobj instance void Extensions/'<M>d__1'::.ctor(int32)
        IL_0007: dup
        IL_0008: ldarg.0
        IL_0009: stfld object Extensions/'<M>d__1'::'<>3__o'
        IL_000e: dup
        IL_000f: ldarg.1
        IL_0010: stfld string Extensions/'<M>d__1'::'<>3__s'
        IL_0015: ret
    } // end of method Extensions::M
} // end of class Extensions
""").Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object _)
    {
        static System.Collections.Generic.IEnumerable<string> M(object o, string s)
        {
            yield return o + s;
        }
    }

    extension(object)
    {
        static System.Collections.Generic.IEnumerable<string> M(object o, string s, int x)
        {
            yield return o + s;
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        foreach (var s in Test("1"))
            System.Console.Write(s);
        foreach (var s in object.M2("3", "4"))
            System.Console.Write(s);
    }

    static System.Collections.Generic.IEnumerable<string> Test(object o)
    {
        return object.M(o, "2");
    }
}

static class Extensions
{
    extension(object _)
    {
        public static System.Collections.Generic.IEnumerable<string> M2(object o, string s) => object.M(o, s);
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_StaticMethod_05_Async()
    {
        var src1 = """
public static class Extensions
{
    extension(object _)
    {
        public static async System.Threading.Tasks.Task<string> M(object o, string s)
        {
            await System.Threading.Tasks.Task.Yield();
            return o + s;
        }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1, symbolValidator: (m) =>
        {
            MethodSymbol implementation = m.ContainingAssembly.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
            AssertEx.Equal("System.Runtime.CompilerServices.AsyncStateMachineAttribute(typeof(Extensions.<M>d__1))", implementation.GetAttributes().Single().ToString());
        }).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$3D34838CB2C73A4E406AE3905787D97D'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object _
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2196
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$3D34838CB2C73A4E406AE3905787D97D'::'<Extension>$'
        } // end of class <M>$3D34838CB2C73A4E406AE3905787D97D
        // Methods
        .method public hidebysig static 
            class [mscorlib]System.Threading.Tasks.Task`1<string> M (
                object o,
                string s
            ) cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 33 44 33 34 38 33 38 43 42
                32 43 37 33 41 34 45 34 30 36 41 45 33 39 30 35
                37 38 37 44 39 37 44 00 00
            )
            // Method begins at RVA 0x20b3
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi sealed beforefieldinit '<M>d__1'
        extends [mscorlib]System.ValueType
        implements [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public int32 '<>1__state'
        .field public valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> '<>t__builder'
        .field public object o
        .field public string s
        .field private valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter '<>u__1'
        // Methods
        .method private final hidebysig newslot virtual 
            instance void MoveNext () cil managed 
        {
            .override method instance void [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine::MoveNext()
            // Method begins at RVA 0x20b8
            // Code size 178 (0xb2)
            .maxstack 3
            .locals init (
                [0] int32,
                [1] string,
                [2] valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter,
                [3] valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable,
                [4] class [mscorlib]System.Exception
            )
            IL_0000: ldarg.0
            IL_0001: ldfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_0006: stloc.0
            .try
            {
                IL_0007: ldloc.0
                IL_0008: brfalse.s IL_0041
                IL_000a: call valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable [mscorlib]System.Threading.Tasks.Task::Yield()
                IL_000f: stloc.3
                IL_0010: ldloca.s 3
                IL_0012: call instance valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter [mscorlib]System.Runtime.CompilerServices.YieldAwaitable::GetAwaiter()
                IL_0017: stloc.2
                IL_0018: ldloca.s 2
                IL_001a: call instance bool [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter::get_IsCompleted()
                IL_001f: brtrue.s IL_005d
                IL_0021: ldarg.0
                IL_0022: ldc.i4.0
                IL_0023: dup
                IL_0024: stloc.0
                IL_0025: stfld int32 Extensions/'<M>d__1'::'<>1__state'
                IL_002a: ldarg.0
                IL_002b: ldloc.2
                IL_002c: stfld valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter Extensions/'<M>d__1'::'<>u__1'
                IL_0031: ldarg.0
                IL_0032: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
                IL_0037: ldloca.s 2
                IL_0039: ldarg.0
                IL_003a: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::AwaitUnsafeOnCompleted<valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter, valuetype Extensions/'<M>d__1'>(!!0&, !!1&)
                IL_003f: leave.s IL_00b1
                IL_0041: ldarg.0
                IL_0042: ldfld valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter Extensions/'<M>d__1'::'<>u__1'
                IL_0047: stloc.2
                IL_0048: ldarg.0
                IL_0049: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter Extensions/'<M>d__1'::'<>u__1'
                IL_004e: initobj [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter
                IL_0054: ldarg.0
                IL_0055: ldc.i4.m1
                IL_0056: dup
                IL_0057: stloc.0
                IL_0058: stfld int32 Extensions/'<M>d__1'::'<>1__state'
                IL_005d: ldloca.s 2
                IL_005f: call instance void [mscorlib]System.Runtime.CompilerServices.YieldAwaitable/YieldAwaiter::GetResult()
                IL_0064: ldarg.0
                IL_0065: ldfld object Extensions/'<M>d__1'::o
                IL_006a: dup
                IL_006b: brtrue.s IL_0071
                IL_006d: pop
                IL_006e: ldnull
                IL_006f: br.s IL_0076
                IL_0071: callvirt instance string [mscorlib]System.Object::ToString()
                IL_0076: ldarg.0
                IL_0077: ldfld string Extensions/'<M>d__1'::s
                IL_007c: call string [mscorlib]System.String::Concat(string, string)
                IL_0081: stloc.1
                IL_0082: leave.s IL_009d
            } // end .try
            catch [mscorlib]System.Exception
            {
                IL_0084: stloc.s 4
                IL_0086: ldarg.0
                IL_0087: ldc.i4.s -2
                IL_0089: stfld int32 Extensions/'<M>d__1'::'<>1__state'
                IL_008e: ldarg.0
                IL_008f: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
                IL_0094: ldloc.s 4
                IL_0096: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetException(class [mscorlib]System.Exception)
                IL_009b: leave.s IL_00b1
            } // end handler
            IL_009d: ldarg.0
            IL_009e: ldc.i4.s -2
            IL_00a0: stfld int32 Extensions/'<M>d__1'::'<>1__state'
            IL_00a5: ldarg.0
            IL_00a6: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
            IL_00ab: ldloc.1
            IL_00ac: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetResult(!0)
            IL_00b1: ret
        } // end of method '<M>d__1'::MoveNext
        .method private final hidebysig newslot virtual 
            instance void SetStateMachine (
                class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine stateMachine
            ) cil managed 
        {
            .custom instance void [mscorlib]System.Diagnostics.DebuggerHiddenAttribute::.ctor() = (
                01 00 00 00
            )
            .override method instance void [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine::SetStateMachine(class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine)
            // Method begins at RVA 0x2188
            // Code size 13 (0xd)
            .maxstack 8
            IL_0000: ldarg.0
            IL_0001: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
            IL_0006: ldarg.1
            IL_0007: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::SetStateMachine(class [mscorlib]System.Runtime.CompilerServices.IAsyncStateMachine)
            IL_000c: ret
        } // end of method '<M>d__1'::SetStateMachine
    } // end of class <M>d__1
    // Methods
    .method public hidebysig static 
        class [mscorlib]System.Threading.Tasks.Task`1<string> M (
            object o,
            string s
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.AsyncStateMachineAttribute::.ctor(class [mscorlib]System.Type) = (
            01 00 12 45 78 74 65 6e 73 69 6f 6e 73 2b 3c 4d
            3e 64 5f 5f 31 00 00
        )
        // Method begins at RVA 0x2068
        // Code size 63 (0x3f)
        .maxstack 2
        .locals init (
            [0] valuetype Extensions/'<M>d__1'
        )
        IL_0000: ldloca.s 0
        IL_0002: call valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<!0> valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::Create()
        IL_0007: stfld valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
        IL_000c: ldloca.s 0
        IL_000e: ldarg.0
        IL_000f: stfld object Extensions/'<M>d__1'::o
        IL_0014: ldloca.s 0
        IL_0016: ldarg.1
        IL_0017: stfld string Extensions/'<M>d__1'::s
        IL_001c: ldloca.s 0
        IL_001e: ldc.i4.m1
        IL_001f: stfld int32 Extensions/'<M>d__1'::'<>1__state'
        IL_0024: ldloca.s 0
        IL_0026: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
        IL_002b: ldloca.s 0
        IL_002d: call instance void valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::Start<valuetype Extensions/'<M>d__1'>(!!0&)
        IL_0032: ldloca.s 0
        IL_0034: ldflda valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string> Extensions/'<M>d__1'::'<>t__builder'
        IL_0039: call instance class [mscorlib]System.Threading.Tasks.Task`1<!0> valuetype [mscorlib]System.Runtime.CompilerServices.AsyncTaskMethodBuilder`1<string>::get_Task()
        IL_003e: ret
    } // end of method Extensions::M
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object _)
    {
        static async System.Threading.Tasks.Task<string> M(object o, string s)
        {
            await System.Threading.Tasks.Task.Yield();
            return o + s;
        }
    }

    extension(object)
    {
        static async System.Threading.Tasks.Task<string> M(object o, string s, int x)
        {
            await System.Threading.Tasks.Task.Yield();
            return o + s;
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1").Result);
        System.Console.Write(object.M2("3", "4").Result);
    }

    async static System.Threading.Tasks.Task<string> Test(object o)
    {
        await System.Threading.Tasks.Task.Yield();
        return await object.M(o, "2");
    }
}

static class Extensions
{
    extension(object _)
    {
        async public static System.Threading.Tasks.Task<string> M2(object o, string s)
        {
            await System.Threading.Tasks.Task.Yield();
            return await object.M(o, s);
        }
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "1234").VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_InstanceProperty_01()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public string P => o.ToString();
    }
}
""";
        var comp1 = CreateCompilation(src1);

        MethodSymbol implementation = comp1.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
        Assert.True(implementation.IsStatic);
        Assert.Equal(MethodKind.Ordinary, implementation.MethodKind);
        Assert.Equal(1, implementation.ParameterCount);
        AssertEx.Equal("System.String Extensions.get_P(System.Object o)", implementation.ToTestDisplayString());
        Assert.True(implementation.IsImplicitlyDeclared);
        Assert.False(implementation.IsExtensionMethod);
        Assert.False(implementation.HasSpecialName);
        Assert.False(implementation.HasRuntimeSpecialName);

        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        var expectedTypeIL = """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2072
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig specialname 
            instance string get_P () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x206f
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_P
        // Properties
        .property instance string P()
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            .get instance string Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_P()
        }
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method public hidebysig static 
        string get_P (
            object o
        ) cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 7 (0x7)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0006: ret
    } // end of method Extensions::get_P
} // end of class Extensions
""";

        verifier1.VerifyTypeIL("Extensions", expectedTypeIL.Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("2".P2);
    }

    static string Test(object o)
    {
        return o.P;
    }
}

static class Extensions
{
    extension(object o)
    {
        public string P2 => o.P;
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        var verifier3 = CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       12 (0xc)
  .maxstack  1
  .locals init (string V_0)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  call       ""string Extensions.get_P(object)""
  IL_0007:  stloc.0
  IL_0008:  br.s       IL_000a
  IL_000a:  ldloc.0
  IL_000b:  ret
}
";
        verifier3.VerifyIL("Program.Test", testIL);

        var m2IL =
@"
{
  // Code size        7 (0x7)
  .maxstack  1
  IL_0000:  ldarg.0
  IL_0001:  call       ""string Extensions.get_P(object)""
  IL_0006:  ret
}
";
        verifier3.VerifyIL("Extensions.get_P2(object)", m2IL);

        var comp1ImageReference = comp1.EmitToImageReference();
        comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions.get_P2(object)", m2IL);

        comp3 = CreateCompilationWithIL(src3, expectedTypeIL + ExtensionMarkerAttributeIL, options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        var remove = """
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
""";

        comp3 = CreateCompilationWithIL(src3, expectedTypeIL.Remove(expectedTypeIL.IndexOf(remove), remove.Length) + ExtensionMarkerAttributeIL);
        comp3.VerifyDiagnostics(
            // (11,18): error CS1061: 'object' does not contain a definition for 'P' and no accessible extension method 'P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         return o.P;
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "P").WithArguments("object", "P").WithLocation(11, 18),
            // (19,31): error CS1061: 'object' does not contain a definition for 'P' and no accessible extension method 'P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         public string P2 => o.P;
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "P").WithArguments("object", "P").WithLocation(19, 31)
            );

        src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("2".P2);
    }

    static string Test(object o)
    {
        return o.get_P();
    }
}

static class Extensions
{
    extension(object o)
    {
        public string P2 => o.get_P();
    }
}
""";

        comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        comp3.VerifyDiagnostics(
            // (11,18): error CS1061: 'object' does not contain a definition for 'get_P' and no accessible extension method 'get_P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         return o.get_P();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "get_P").WithArguments("object", "get_P").WithLocation(11, 18),
            // (19,31): error CS1061: 'object' does not contain a definition for 'get_P' and no accessible extension method 'get_P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         public string P2 => o.get_P();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "get_P").WithArguments("object", "get_P").WithLocation(19, 31)
            );

        comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);
        comp3.VerifyDiagnostics(
            // (11,18): error CS1061: 'object' does not contain a definition for 'get_P' and no accessible extension method 'get_P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         return o.get_P();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "get_P").WithArguments("object", "get_P").WithLocation(11, 18),
            // (19,31): error CS1061: 'object' does not contain a definition for 'get_P' and no accessible extension method 'get_P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         public string P2 => o.get_P();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "get_P").WithArguments("object", "get_P").WithLocation(19, 31)
            );

        src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("2".P2);
    }

    static string Test(object o)
    {
        return Extensions.get_P(o);
    }
}

static class Extensions_
{
    extension(object o)
    {
        public string P2 => Extensions.get_P(o);
    }
}
""";

        comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions_.get_P2(object)", m2IL);

        comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions_.get_P2(object)", m2IL);

        var vbComp = CreateVisualBasicCompilation("""
Class Program
    Shared Sub Main()
        System.Console.Write(Test2("3"))
    End Sub

    Shared Function Test2(o As String) As String
        return Extensions.get_P(o)
    End Function
End Class
""",
            referencedAssemblies: comp3.References, compilationOptions: new VisualBasicCompilationOptions(OutputKind.ConsoleApplication));

        CompileAndVerify(vbComp, expectedOutput: "3").VerifyDiagnostics();

        vbComp = CreateVisualBasicCompilation("""
Class Program
    Shared Sub Main()
        System.Console.Write(Test1("1"))
    End Sub

    Shared Function Test1(o As String) As String
        return o.get_P()
    End Function
End Class
""",
            referencedAssemblies: comp3.References, compilationOptions: new VisualBasicCompilationOptions(OutputKind.ConsoleApplication));

        vbComp.VerifyDiagnostics(
            // error BC30456: 'get_P' is not a member of 'String'.
            Diagnostic(30456 /*ERRID.ERR_NameNotMember2*/, "o.get_P").WithArguments("get_P", "String").WithLocation(7, 16)
            );

        var comp5 = CreateCompilation(src1);
        comp5.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp5.VerifyEmitDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object o)
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5)
            );
    }

    [Fact]
    public void Implementation_StaticProperty_01()
    {
        var src1 = """
public static class Extensions
{
    extension(object)
    {
        public static string P => "P";
    }
}
""";
        var comp1 = CreateCompilation(src1);

        MethodSymbol implementation = comp1.GetTypeByMetadataName("Extensions").GetMembers().OfType<MethodSymbol>().Single();
        Assert.True(implementation.IsStatic);
        Assert.Equal(MethodKind.Ordinary, implementation.MethodKind);
        Assert.Equal(0, implementation.ParameterCount);
        AssertEx.Equal("System.String Extensions.get_P()", implementation.ToTestDisplayString());
        Assert.True(implementation.IsImplicitlyDeclared);
        Assert.False(implementation.IsExtensionMethod);
        Assert.False(implementation.HasSpecialName);
        Assert.False(implementation.HasRuntimeSpecialName);

        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        var expectedTypeIL = """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object ''
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2071
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$C43E2675C7BBF9284AF22FB8A9BF0280'::'<Extension>$'
        } // end of class <M>$C43E2675C7BBF9284AF22FB8A9BF0280
        // Methods
        .method public hidebysig specialname static 
            string get_P () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            // Method begins at RVA 0x206e
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_P
        // Properties
        .property string P()
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 43 34 33 45 32 36 37 35 43
                37 42 42 46 39 32 38 34 41 46 32 32 46 42 38 41
                39 42 46 30 32 38 30 00 00
            )
            .get string Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_P()
        }
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method public hidebysig static 
        string get_P () cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 6 (0x6)
        .maxstack 8
        IL_0000: ldstr "P"
        IL_0005: ret
    } // end of method Extensions::get_P
} // end of class Extensions
""";

        verifier1.VerifyTypeIL("Extensions", expectedTypeIL.Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test());
        System.Console.Write(object.P2);
    }

    static string Test()
    {
        return object.P;
    }
}

static class Extensions
{
    extension(object o)
    {
        public static string P2 => object.P;
    }
}
""";

        var comp1MetadataReference = comp1.ToMetadataReference();
        var comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        var verifier3 = CompileAndVerify(comp3, expectedOutput: "PP").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       11 (0xb)
  .maxstack  1
  .locals init (string V_0)
  IL_0000:  nop
  IL_0001:  call       ""string Extensions.get_P()""
  IL_0006:  stloc.0
  IL_0007:  br.s       IL_0009
  IL_0009:  ldloc.0
  IL_000a:  ret
}
";
        verifier3.VerifyIL("Program.Test", testIL);

        var m2IL =
@"
{
  // Code size        6 (0x6)
  .maxstack  1
  IL_0000:  call       ""string Extensions.get_P()""
  IL_0005:  ret
}
";
        verifier3.VerifyIL("Extensions.get_P2()", m2IL);

        var comp1ImageReference = comp1.EmitToImageReference();
        comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "PP").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions.get_P2()", m2IL);

        comp3 = CreateCompilationWithIL(src3, expectedTypeIL + ExtensionMarkerAttributeIL, options: TestOptions.DebugExe);
        CompileAndVerify(comp3, expectedOutput: "PP").VerifyDiagnostics();

        var remove = """
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
""";

        comp3 = CreateCompilationWithIL(src3, expectedTypeIL.Remove(expectedTypeIL.IndexOf(remove), remove.Length) + ExtensionMarkerAttributeIL);
        comp3.VerifyDiagnostics(
            // (11,23): error CS0117: 'object' does not contain a definition for 'P'
            //         return object.P;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "P").WithArguments("object", "P").WithLocation(11, 23),
            // (19,43): error CS0117: 'object' does not contain a definition for 'P'
            //         public static string P2 => object.P;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "P").WithArguments("object", "P").WithLocation(19, 43)
            );

        src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test());
        System.Console.Write(object.P2);
    }

    static string Test()
    {
        return Extensions.get_P();
    }
}

static class Extensions_
{
    extension(object o)
    {
        public static string P2 => Extensions.get_P();
    }
}
""";

        comp3 = CreateCompilation(src3, references: [comp1MetadataReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "PP").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions_.get_P2()", m2IL);

        comp3 = CreateCompilation(src3, references: [comp1ImageReference], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "PP").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions_.get_P2()", m2IL);

        var vbComp = CreateVisualBasicCompilation("""
Class Program
    Shared Sub Main()
        System.Console.Write(Test2())
    End Sub

    Shared Function Test2() As String
        return Extensions.get_P()
    End Function
End Class
""",
            referencedAssemblies: comp3.References, compilationOptions: new VisualBasicCompilationOptions(OutputKind.ConsoleApplication));

        CompileAndVerify(vbComp, expectedOutput: "P").VerifyDiagnostics();

        var comp5 = CreateCompilation(src1);
        comp5.MakeMemberMissing(WellKnownMember.System_Runtime_CompilerServices_ExtensionAttribute__ctor);
        comp5.VerifyEmitDiagnostics(
            // (3,5): error CS1110: Cannot define a new extension because the compiler required type 'System.Runtime.CompilerServices.ExtensionAttribute' cannot be found. Are you missing a reference to System.Core.dll?
            //     extension(object)
            Diagnostic(ErrorCode.ERR_ExtensionAttrNotFound, "extension").WithArguments("System.Runtime.CompilerServices.ExtensionAttribute").WithLocation(3, 5)
            );
    }

    [Fact]
    public void Implementation_InstanceProperty_02()
    {
        var src1 = """
public static class Extensions
{
    extension(object o)
    {
        public string P { get { return o.ToString(); } }
    }
}
""";
        var comp1 = CreateCompilation(src1);
        var verifier1 = CompileAndVerify(comp1).VerifyDiagnostics();

        verifier1.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2072
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method public hidebysig specialname 
            instance string get_P () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x206f
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_P
        // Properties
        .property instance string P()
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            .get instance string Extensions/'<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::get_P()
        }
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    // Methods
    .method public hidebysig static 
        string get_P (
            object o
        ) cil managed 
    {
        // Method begins at RVA 0x2067
        // Code size 7 (0x7)
        .maxstack 8
        IL_0000: ldarg.0
        IL_0001: callvirt instance string [mscorlib]System.Object::ToString()
        IL_0006: ret
    } // end of method Extensions::get_P
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src3 = """
class Program
{
    static void Main()
    {
        System.Console.Write(Test("1"));
        System.Console.Write("2".P2);
    }

    static string Test(object o)
    {
        return o.P;
    }
}

static class Extensions
{
    extension(object o)
    {
        public string P2 => o.P;
    }
}
""";

        var comp3 = CreateCompilation(src3, references: [comp1.ToMetadataReference()], options: TestOptions.DebugExe);
        var verifier3 = CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        var testIL =
@"
{
  // Code size       12 (0xc)
  .maxstack  1
  .locals init (string V_0)
  IL_0000:  nop
  IL_0001:  ldarg.0
  IL_0002:  call       ""string Extensions.get_P(object)""
  IL_0007:  stloc.0
  IL_0008:  br.s       IL_000a
  IL_000a:  ldloc.0
  IL_000b:  ret
}
";
        verifier3.VerifyIL("Program.Test", testIL);

        var m2IL =
@"
{
  // Code size        7 (0x7)
  .maxstack  1
  IL_0000:  ldarg.0
  IL_0001:  call       ""string Extensions.get_P(object)""
  IL_0006:  ret
}
";
        verifier3.VerifyIL("Extensions.get_P2(object)", m2IL);

        comp3 = CreateCompilation(src3, references: [comp1.EmitToImageReference()], options: TestOptions.DebugExe);
        verifier3 = CompileAndVerify(comp3, expectedOutput: "12").VerifyDiagnostics();

        verifier3.VerifyIL("Program.Test", testIL);
        verifier3.VerifyIL("Extensions.get_P2(object)", m2IL);
    }

    [Fact]
    public void Implementation_DelegateCaching_01()
    {
        var src = """
42.M2<int, string>();

public static class Extensions
{
    extension<T>(T o)
    {
        public void M2<U>()
        {
            local<long>()();

            System.Func<V> local<V>()
            {
                return C1.M1<T, U, V>;
            }
        }
    }
}

class C1
{
    static public V M1<T, U, V>() { System.Console.Write((typeof(T), typeof(U), typeof(V))); return default; }
}
""";
        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp, expectedOutput: "(System.Int32, System.String, System.Int64)").VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D3EAC011D93395A3E50DF069CE627102'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    !T o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20ea
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D3EAC011D93395A3E50DF069CE627102'::'<Extension>$'
        } // end of class <M>$D3EAC011D93395A3E50DF069CE627102
        // Methods
        .method public hidebysig 
            instance void M2<U> () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 33 45 41 43 30 31 31 44
                39 33 33 39 35 41 33 45 35 30 44 46 30 36 39 43
                45 36 32 37 31 30 32 00 00
            )
            // Method begins at RVA 0x20e7
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$8048A6C8BE30A622530249B904B537EB`1'::M2
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
    .class nested private auto ansi abstract sealed beforefieldinit '<local>O__1_0`3'<T, U, V>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [mscorlib]System.Func`1<!V> '<0>__M1'
    } // end of class <local>O__1_0`3
    // Methods
    .method public hidebysig static 
        void M2<T, U> (
            !!T o
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2078
        // Code size 12 (0xc)
        .maxstack 8
        IL_0000: call class [mscorlib]System.Func`1<!!2> Extensions::'<M2>g__local|1_0'<!!T, !!U, int64>()
        IL_0005: callvirt instance !0 class [mscorlib]System.Func`1<int64>::Invoke()
        IL_000a: pop
        IL_000b: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static 
        class [mscorlib]System.Func`1<!!V> '<M2>g__local|1_0'<T, U, V> () cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2085
        // Code size 28 (0x1c)
        .maxstack 8
        IL_0000: ldsfld class [mscorlib]System.Func`1<!2> class Extensions/'<local>O__1_0`3'<!!T, !!U, !!V>::'<0>__M1'
        IL_0005: dup
        IL_0006: brtrue.s IL_001b
        IL_0008: pop
        IL_0009: ldnull
        IL_000a: ldftn !!2 C1::M1<!!T, !!U, !!V>()
        IL_0010: newobj instance void class [mscorlib]System.Func`1<!!V>::.ctor(object, native int)
        IL_0015: dup
        IL_0016: stsfld class [mscorlib]System.Func`1<!2> class Extensions/'<local>O__1_0`3'<!!T, !!U, !!V>::'<0>__M1'
        IL_001b: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(T o)
    {
        void M2<U>()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            System.Func<V> local<V>()
            {
                return C1.M1<T, U, V>;
            }
        }
    }

    extension<T>(T o)
    {
        void M2<U>(int x)
        {
            System.Func<V> local<V>()
            {
                return C1.M1<T, U, V>;
            }
        }
    }
}

class C1
{
    static public V M1<T, U, V>() => default;
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DelegateCaching_02()
    {
        var src = """

42.M2();

public static class Extensions
{
    extension<T>(T o)
    {
        public void M2()
        {
            local()();

            System.Action local()
            {
                return C1.M1<T>;
            }
        }
    }
}

class C1
{
    static public void M1<T>() { System.Console.Write(typeof(T)); }
}
""";
        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp, expectedOutput: "System.Int32").VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D3EAC011D93395A3E50DF069CE627102'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static 
                void '<Extension>$' (
                    !T o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x20b5
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D3EAC011D93395A3E50DF069CE627102'::'<Extension>$'
        } // end of class <M>$D3EAC011D93395A3E50DF069CE627102
        // Methods
        .method public hidebysig 
            instance void M2 () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 33 45 41 43 30 31 31 44
                39 33 33 39 35 41 33 45 35 30 44 46 30 36 39 43
                45 36 32 37 31 30 32 00 00
            )
            // Method begins at RVA 0x20b2
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$8048A6C8BE30A622530249B904B537EB`1'::M2
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
    .class nested private auto ansi abstract sealed beforefieldinit '<>O__1_0`1'<T>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [mscorlib]System.Action '<0>__M1'
    } // end of class <>O__1_0`1
    // Methods
    .method public hidebysig static 
        void M2<T> (
            !!T o
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2078
        // Code size 11 (0xb)
        .maxstack 8
        IL_0000: call class [mscorlib]System.Action Extensions::'<M2>g__local|1_0'<!!T>()
        IL_0005: callvirt instance void [mscorlib]System.Action::Invoke()
        IL_000a: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static 
        class [mscorlib]System.Action '<M2>g__local|1_0'<T> () cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2084
        // Code size 28 (0x1c)
        .maxstack 8
        IL_0000: ldsfld class [mscorlib]System.Action class Extensions/'<>O__1_0`1'<!!T>::'<0>__M1'
        IL_0005: dup
        IL_0006: brtrue.s IL_001b
        IL_0008: pop
        IL_0009: ldnull
        IL_000a: ldftn void C1::M1<!!T>()
        IL_0010: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0015: dup
        IL_0016: stsfld class [mscorlib]System.Action class Extensions/'<>O__1_0`1'<!!T>::'<0>__M1'
        IL_001b: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(T o)
    {
        void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            System.Action local()
            {
                return C1.M1<T>;
            }
        }
    }

    extension<T>(T o)
    {
        void M2(int x)
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            System.Action local()
            {
                return C1.M1<T>;
            }
        }
    }
}

class C1
{
    static public void M1<T>() {}
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DelegateCaching_03()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            System.Action local()
            {
                return C1.M1;
            }
        }
    }
}

class C1
{
    static public void M1() {}
}
""";
        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp).VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method private hidebysig 
            instance void M2 () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x208e
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M2
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi abstract sealed beforefieldinit '<>O'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [mscorlib]System.Action '<0>__M1'
    } // end of class <>O
    // Methods
    .method private hidebysig static 
        void M2 (
            object o
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static 
        class [mscorlib]System.Action '<M2>g__local|1_0' () cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2069
        // Code size 28 (0x1c)
        .maxstack 8
        IL_0000: ldsfld class [mscorlib]System.Action Extensions/'<>O'::'<0>__M1'
        IL_0005: dup
        IL_0006: brtrue.s IL_001b
        IL_0008: pop
        IL_0009: ldnull
        IL_000a: ldftn void C1::M1()
        IL_0010: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0015: dup
        IL_0016: stsfld class [mscorlib]System.Action Extensions/'<>O'::'<0>__M1'
        IL_001b: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            System.Action local()
            {
                return C1.M1;
            }
        }
    }

    extension(object o)
    {
        void M2(int x)
        {
            System.Action local()
            {
                return C1.M1;
            }
        }
    }
}

class C1
{
    static public void M1() {}
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DelegateCaching_04()
    {
        var src = """
public static class Extensions
{
    extension<T>(T o)
    {
        System.Action M2()
        {
            return local;

            static void local()
            {
                typeof(T).ToString();
            }
        }
    }
}
""";
        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp).VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D3EAC011D93395A3E50DF069CE627102'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static
                void '<Extension>$' (
                    !T o
                ) cil managed
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2099
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D3EAC011D93395A3E50DF069CE627102'::'<Extension>$'
        } // end of class <M>$D3EAC011D93395A3E50DF069CE627102
        // Methods
        .method private hidebysig
            instance class [mscorlib]System.Action M2 () cil managed
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 33 45 41 43 30 31 31 44
                39 33 33 39 35 41 33 45 35 30 44 46 30 36 39 43
                45 36 32 37 31 30 32 00 00
            )
            // Method begins at RVA 0x2096
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$8048A6C8BE30A622530249B904B537EB`1'::M2
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
    .class nested private auto ansi abstract sealed beforefieldinit '<>O__1_0`1'<T>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [mscorlib]System.Action '<0>__local'
    } // end of class <>O__1_0`1
    // Methods
    .method private hidebysig static
        class [mscorlib]System.Action M2<T> (
            !!T o
        ) cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 28 (0x1c)
        .maxstack 8
        IL_0000: ldsfld class [mscorlib]System.Action class Extensions/'<>O__1_0`1'<!!T>::'<0>__local'
        IL_0005: dup
        IL_0006: brtrue.s IL_001b
        IL_0008: pop
        IL_0009: ldnull
        IL_000a: ldftn void Extensions::'<M2>g__local|1_0'<!!T>()
        IL_0010: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0015: dup
        IL_0016: stsfld class [mscorlib]System.Action class Extensions/'<>O__1_0`1'<!!T>::'<0>__local'
        IL_001b: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static
        void '<M2>g__local|1_0'<T> () cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2084
        // Code size 17 (0x11)
        .maxstack 8
        IL_0000: ldtoken !!T
        IL_0005: call class [mscorlib]System.Type [mscorlib]System.Type::GetTypeFromHandle(valuetype [mscorlib]System.RuntimeTypeHandle)
        IL_000a: callvirt instance string [mscorlib]System.Object::ToString()
        IL_000f: pop
        IL_0010: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension<T>(T o)
    {
        System.Action M2()
        {
            return local;

            static void local()
            {
                typeof(T).ToString();
            }
        }
    }

    extension<T>(T o)
    {
        System.Action M2(int x)
        {
            return local;

            static void local()
            {
                typeof(T).ToString();
            }
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DelegateCaching_05()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        System.Action M2()
        {
            return local;

            static void local()
            {
                typeof(object).ToString();
            }
        }
    }
}
""";
        var comp = CreateCompilation(src);
        var verifier = CompileAndVerify(comp).VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2099
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method private hidebysig 
            instance class [mscorlib]System.Action M2 () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x2096
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M2
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi abstract sealed beforefieldinit '<>O'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [mscorlib]System.Action '<0>__local'
    } // end of class <>O
    // Methods
    .method private hidebysig static 
        class [mscorlib]System.Action M2 (
            object o
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 28 (0x1c)
        .maxstack 8
        IL_0000: ldsfld class [mscorlib]System.Action Extensions/'<>O'::'<0>__local'
        IL_0005: dup
        IL_0006: brtrue.s IL_001b
        IL_0008: pop
        IL_0009: ldnull
        IL_000a: ldftn void Extensions::'<M2>g__local|1_0'()
        IL_0010: newobj instance void [mscorlib]System.Action::.ctor(object, native int)
        IL_0015: dup
        IL_0016: stsfld class [mscorlib]System.Action Extensions/'<>O'::'<0>__local'
        IL_001b: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static 
        void '<M2>g__local|1_0' () cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2084
        // Code size 17 (0x11)
        .maxstack 8
        IL_0000: ldtoken [mscorlib]System.Object
        IL_0005: call class [mscorlib]System.Type [mscorlib]System.Type::GetTypeFromHandle(valuetype [mscorlib]System.RuntimeTypeHandle)
        IL_000a: callvirt instance string [mscorlib]System.Object::ToString()
        IL_000f: pop
        IL_0010: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        System.Action M2()
        {
            return local;

            static void local()
            {
                typeof(object).ToString();
            }
        }
    }

    extension(object o)
    {
        System.Action M2(int x)
        {
            return local;

            static void local()
            {
                typeof(object).ToString();
            }
        }
    }
}
""";
        var comp2 = CreateCompilation(src2);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DynamicCallSite_01()
    {
        var src = """
42.M2<int, string>();

class D
{
    public void M1<T, U, V>(T t, U u, V v) { System.Console.Write((typeof(T), typeof(U), typeof(V))); }
}

public static class Extensions
{
    extension<T>(T o)
    {
        public void M2<U>()
        {
            local(new D(), default(T), default(U), 42L);

            void local<V>(dynamic d, T t, U u, V v)
            {
                d.M1(t, u, v);
            }
        }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.StandardAndCSharp);
        var verifier = CompileAndVerify(comp, expectedOutput: "(System.Int32, System.String, System.Int64)").VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D3EAC011D93395A3E50DF069CE627102'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static
                void '<Extension>$' (
                    !T o
                ) cil managed
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2155
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D3EAC011D93395A3E50DF069CE627102'::'<Extension>$'
        } // end of class <M>$D3EAC011D93395A3E50DF069CE627102
        // Methods
        .method public hidebysig
            instance void M2<U> () cil managed
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 33 45 41 43 30 31 31 44
                39 33 33 39 35 41 33 45 35 30 44 46 30 36 39 43
                45 36 32 37 31 30 32 00 00
            )
            // Method begins at RVA 0x2152
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$8048A6C8BE30A622530249B904B537EB`1'::M2
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
    .class nested private auto ansi abstract sealed beforefieldinit '<>o__0|1`3'<T, U, V>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !T, !U, !V>> '<>p__0'
    } // end of class <>o__0|1`3
    // Methods
    .method public hidebysig static
        void M2<T, U> (
            !!T o
        ) cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x20a8
        // Code size 32 (0x20)
        .maxstack 4
        .locals init (
            [0] !!T,
            [1] !!U
        )
        IL_0000: newobj instance void D::.ctor()
        IL_0005: ldloca.s 0
        IL_0007: initobj !!T
        IL_000d: ldloc.0
        IL_000e: ldloca.s 1
        IL_0010: initobj !!U
        IL_0016: ldloc.1
        IL_0017: ldc.i4.s 42
        IL_0019: conv.i8
        IL_001a: call void Extensions::'<M2>g__local|1_0'<!!T, !!U, int64>(object, !!0, !!1, !!2)
        IL_001f: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static
        void '<M2>g__local|1_0'<T, U, V> (
            object d,
            !!T t,
            !!U u,
            !!V v
        ) cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        .param [1]
            .custom instance void [System.Core]System.Runtime.CompilerServices.DynamicAttribute::.ctor() = (
                01 00 00 00
            )
        // Method begins at RVA 0x20d4
        // Code size 114 (0x72)
        .maxstack 9
        IL_0000: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0, !1, !2>> class Extensions/'<>o__0|1`3'<!!T, !!U, !!V>::'<>p__0'
        IL_0005: brtrue.s IL_0059
        IL_0007: ldc.i4 256
        IL_000c: ldstr "M1"
        IL_0011: ldnull
        IL_0012: ldtoken Extensions
        IL_0017: call class [mscorlib]System.Type [mscorlib]System.Type::GetTypeFromHandle(valuetype [mscorlib]System.RuntimeTypeHandle)
        IL_001c: ldc.i4.4
        IL_001d: newarr [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo
        IL_0022: dup
        IL_0023: ldc.i4.0
        IL_0024: ldc.i4.0
        IL_0025: ldnull
        IL_0026: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_002b: stelem.ref
        IL_002c: dup
        IL_002d: ldc.i4.1
        IL_002e: ldc.i4.1
        IL_002f: ldnull
        IL_0030: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_0035: stelem.ref
        IL_0036: dup
        IL_0037: ldc.i4.2
        IL_0038: ldc.i4.1
        IL_0039: ldnull
        IL_003a: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_003f: stelem.ref
        IL_0040: dup
        IL_0041: ldc.i4.3
        IL_0042: ldc.i4.1
        IL_0043: ldnull
        IL_0044: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_0049: stelem.ref
        IL_004a: call class [System.Core]System.Runtime.CompilerServices.CallSiteBinder [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.Binder::InvokeMember(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpBinderFlags, string, class [mscorlib]System.Collections.Generic.IEnumerable`1<class [mscorlib]System.Type>, class [mscorlib]System.Type, class [mscorlib]System.Collections.Generic.IEnumerable`1<class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo>)
        IL_004f: call class [System.Core]System.Runtime.CompilerServices.CallSite`1<!0> class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !!T, !!U, !!V>>::Create(class [System.Core]System.Runtime.CompilerServices.CallSiteBinder)
        IL_0054: stsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0, !1, !2>> class Extensions/'<>o__0|1`3'<!!T, !!U, !!V>::'<>p__0'
        IL_0059: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0, !1, !2>> class Extensions/'<>o__0|1`3'<!!T, !!U, !!V>::'<>p__0'
        IL_005e: ldfld !0 class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !!T, !!U, !!V>>::Target
        IL_0063: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0, !1, !2>> class Extensions/'<>o__0|1`3'<!!T, !!U, !!V>::'<>p__0'
        IL_0068: ldarg.0
        IL_0069: ldarg.1
        IL_006a: ldarg.2
        IL_006b: ldarg.3
        IL_006c: callvirt instance void class [mscorlib]System.Action`5<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !!T, !!U, !!V>::Invoke(!0, !1, !2, !3, !4)
        IL_0071: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]").
    Replace("[System.Core]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[System.Core]"));

        var src2 = """
public static class Extensions
{
    extension<T>(T o)
    {
        void M2<U>()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            void local<V>(dynamic d, T t, U u, V v)
            {
                d.M1(t, u, v);
            }
        }
    }

    extension<T>(T o)
    {
        void M2<U>(int x)
        {
            void local<V>(dynamic d, T t, U u, V v)
            {
                d.M1(t, u, v);
            }
        }
    }
}
""";
        var comp2 = CreateCompilation(src2, targetFramework: TargetFramework.StandardAndCSharp);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DynamicCallSite_02()
    {
        var src = """
public static class Extensions
{
    extension<T>(T o)
    {
        public void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            void local(dynamic d, T t)
            {
                d.M1(t);
            }
        }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.StandardAndCSharp);
        var verifier = CompileAndVerify(comp).VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$8048A6C8BE30A622530249B904B537EB`1'<$T0>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$D3EAC011D93395A3E50DF069CE627102'<T>
            extends [mscorlib]System.Object
        {
            // Methods
            .method public hidebysig specialname static
                void '<Extension>$' (
                    !T o
                ) cil managed
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$D3EAC011D93395A3E50DF069CE627102'::'<Extension>$'
        } // end of class <M>$D3EAC011D93395A3E50DF069CE627102
        // Methods
        .method public hidebysig
            instance void M2 () cil managed
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 44 33 45 41 43 30 31 31 44
                39 33 33 39 35 41 33 45 35 30 44 46 30 36 39 43
                45 36 32 37 31 30 32 00 00
            )
            // Method begins at RVA 0x20d4
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$8048A6C8BE30A622530249B904B537EB`1'::M2
    } // end of class <G>$8048A6C8BE30A622530249B904B537EB`1
    .class nested private auto ansi abstract sealed beforefieldinit '<>o__1`1'<T>
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !T>> '<>p__0'
    } // end of class <>o__1`1
    // Methods
    .method public hidebysig static
        void M2<T> (
            !!T o
        ) cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static
        void '<M2>g__local|1_0'<T> (
            object d,
            !!T t
        ) cil managed
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        .param [1]
            .custom instance void [System.Core]System.Runtime.CompilerServices.DynamicAttribute::.ctor() = (
                01 00 00 00
            )
        // Method begins at RVA 0x206c
        // Code size 92 (0x5c)
        .maxstack 9
        IL_0000: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0>> class Extensions/'<>o__1`1'<!!T>::'<>p__0'
        IL_0005: brtrue.s IL_0045
        IL_0007: ldc.i4 256
        IL_000c: ldstr "M1"
        IL_0011: ldnull
        IL_0012: ldtoken Extensions
        IL_0017: call class [mscorlib]System.Type [mscorlib]System.Type::GetTypeFromHandle(valuetype [mscorlib]System.RuntimeTypeHandle)
        IL_001c: ldc.i4.2
        IL_001d: newarr [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo
        IL_0022: dup
        IL_0023: ldc.i4.0
        IL_0024: ldc.i4.0
        IL_0025: ldnull
        IL_0026: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_002b: stelem.ref
        IL_002c: dup
        IL_002d: ldc.i4.1
        IL_002e: ldc.i4.1
        IL_002f: ldnull
        IL_0030: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_0035: stelem.ref
        IL_0036: call class [System.Core]System.Runtime.CompilerServices.CallSiteBinder [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.Binder::InvokeMember(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpBinderFlags, string, class [mscorlib]System.Collections.Generic.IEnumerable`1<class [mscorlib]System.Type>, class [mscorlib]System.Type, class [mscorlib]System.Collections.Generic.IEnumerable`1<class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo>)
        IL_003b: call class [System.Core]System.Runtime.CompilerServices.CallSite`1<!0> class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !!T>>::Create(class [System.Core]System.Runtime.CompilerServices.CallSiteBinder)
        IL_0040: stsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0>> class Extensions/'<>o__1`1'<!!T>::'<>p__0'
        IL_0045: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0>> class Extensions/'<>o__1`1'<!!T>::'<>p__0'
        IL_004a: ldfld !0 class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !!T>>::Target
        IL_004f: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !0>> class Extensions/'<>o__1`1'<!!T>::'<>p__0'
        IL_0054: ldarg.0
        IL_0055: ldarg.1
        IL_0056: callvirt instance void class [mscorlib]System.Action`3<class [System.Core]System.Runtime.CompilerServices.CallSite, object, !!T>::Invoke(!0, !1, !2)
        IL_005b: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]").
    Replace("[System.Core]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[System.Core]"));

        var src2 = """
public static class Extensions
{
    extension<T>(T o)
    {
        void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            void local(dynamic d, T t)
            {
                d.M1(t);
            }
        }
    }

    extension<T>(T o)
    {
        void M2(int x)
        {
            void local(dynamic d, T t)
            {
                d.M1(t);
            }
        }
    }
}
""";
        var comp2 = CreateCompilation(src2, targetFramework: TargetFramework.StandardAndCSharp);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void Implementation_DynamicCallSite_03()
    {
        var src = """
public static class Extensions
{
    extension(object o)
    {
        void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            void local(dynamic d)
            {
                d.M1();
            }
        }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.StandardAndCSharp);
        var verifier = CompileAndVerify(comp).VerifyDiagnostics();

        verifier.VerifyTypeIL("Extensions", """
.class public auto ansi abstract sealed beforefieldinit Extensions
    extends [mscorlib]System.Object
{
    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
        01 00 00 00
    )
    // Nested Types
    .class nested public auto ansi sealed specialname '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Nested Types
        .class nested public auto ansi abstract sealed specialname '<M>$119AA281C143547563250CAF89B48A76'
            extends [mscorlib]System.Object
        {
            // Methods
            .method private hidebysig specialname static 
                void '<Extension>$' (
                    object o
                ) cil managed 
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
                    01 00 00 00
                )
                // Method begins at RVA 0x2067
                // Code size 1 (0x1)
                .maxstack 8
                IL_0000: ret
            } // end of method '<M>$119AA281C143547563250CAF89B48A76'::'<Extension>$'
        } // end of class <M>$119AA281C143547563250CAF89B48A76
        // Methods
        .method private hidebysig 
            instance void M2 () cil managed 
        {
            .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                01 00 24 3c 4d 3e 24 31 31 39 41 41 32 38 31 43
                31 34 33 35 34 37 35 36 33 32 35 30 43 41 46 38
                39 42 34 38 41 37 36 00 00
            )
            // Method begins at RVA 0x20c9
            // Code size 2 (0x2)
            .maxstack 8
            IL_0000: ldnull
            IL_0001: throw
        } // end of method '<G>$C43E2675C7BBF9284AF22FB8A9BF0280'::M2
    } // end of class <G>$C43E2675C7BBF9284AF22FB8A9BF0280
    .class nested private auto ansi abstract sealed beforefieldinit '<>o__1'
        extends [mscorlib]System.Object
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        // Fields
        .field public static class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>> '<>p__0'
    } // end of class <>o__1
    // Methods
    .method private hidebysig static 
        void M2 (
            object o
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = (
            01 00 00 00
        )
        // Method begins at RVA 0x2067
        // Code size 1 (0x1)
        .maxstack 8
        IL_0000: ret
    } // end of method Extensions::M2
    .method assembly hidebysig static 
        void '<M2>g__local|1_0' (
            object d
        ) cil managed 
    {
        .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = (
            01 00 00 00
        )
        .param [1]
            .custom instance void [System.Core]System.Runtime.CompilerServices.DynamicAttribute::.ctor() = (
                01 00 00 00
            )
        // Method begins at RVA 0x206c
        // Code size 81 (0x51)
        .maxstack 9
        IL_0000: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>> Extensions/'<>o__1'::'<>p__0'
        IL_0005: brtrue.s IL_003b
        IL_0007: ldc.i4 256
        IL_000c: ldstr "M1"
        IL_0011: ldnull
        IL_0012: ldtoken Extensions
        IL_0017: call class [mscorlib]System.Type [mscorlib]System.Type::GetTypeFromHandle(valuetype [mscorlib]System.RuntimeTypeHandle)
        IL_001c: ldc.i4.1
        IL_001d: newarr [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo
        IL_0022: dup
        IL_0023: ldc.i4.0
        IL_0024: ldc.i4.0
        IL_0025: ldnull
        IL_0026: call class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo::Create(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfoFlags, string)
        IL_002b: stelem.ref
        IL_002c: call class [System.Core]System.Runtime.CompilerServices.CallSiteBinder [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.Binder::InvokeMember(valuetype [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpBinderFlags, string, class [mscorlib]System.Collections.Generic.IEnumerable`1<class [mscorlib]System.Type>, class [mscorlib]System.Type, class [mscorlib]System.Collections.Generic.IEnumerable`1<class [Microsoft.CSharp]Microsoft.CSharp.RuntimeBinder.CSharpArgumentInfo>)
        IL_0031: call class [System.Core]System.Runtime.CompilerServices.CallSite`1<!0> class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>>::Create(class [System.Core]System.Runtime.CompilerServices.CallSiteBinder)
        IL_0036: stsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>> Extensions/'<>o__1'::'<>p__0'
        IL_003b: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>> Extensions/'<>o__1'::'<>p__0'
        IL_0040: ldfld !0 class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>>::Target
        IL_0045: ldsfld class [System.Core]System.Runtime.CompilerServices.CallSite`1<class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>> Extensions/'<>o__1'::'<>p__0'
        IL_004a: ldarg.0
        IL_004b: callvirt instance void class [mscorlib]System.Action`2<class [System.Core]System.Runtime.CompilerServices.CallSite, object>::Invoke(!0, !1)
        IL_0050: ret
    } // end of method Extensions::'<M2>g__local|1_0'
} // end of class Extensions
""".Replace("[mscorlib]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[mscorlib]").
    Replace("[System.Core]", ExecutionConditionUtil.IsMonoOrCoreClr ? "[netstandard]" : "[System.Core]"));

        var src2 = """
public static class Extensions
{
    extension(object o)
    {
        void M2()
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            void local(dynamic d)
            {
                d.M1();
            }
        }
    }

    extension(object o)
    {
        void M2(int x)
        {
            #pragma warning disable CS8321 // The local function 'local' is declared but never used
            void local(dynamic d)
            {
                d.M1();
            }
        }
    }
}
""";
        var comp2 = CreateCompilation(src2, targetFramework: TargetFramework.StandardAndCSharp);
        CompileAndVerify(comp2).VerifyDiagnostics();
    }

    [Fact]
    public void InstanceMethodInvocation_Simple()
    {
        var src = """
new object().M();

public static class Extensions
{
    extension(object o)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().M()");
        AssertEx.Equal("void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Inaccessible()
    {
        var src = """
new object().M();

public static class Extensions
{
    extension(object o)
    {
        void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS1061: 'object' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // new object().M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("object", "M").WithLocation(1, 14));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().M()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetSymbolInfo(invocation).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        src = """
new object().M();

public static class Extensions
{
    private static void M(this object o) { }
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS1061: 'object' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // new object().M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("object", "M").WithLocation(1, 14));

        tree = comp.SyntaxTrees[0];
        model = comp.GetSemanticModel(tree);
        invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().M()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetSymbolInfo(invocation).CandidateSymbols.ToTestDisplayStrings());

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void System.Object.M()"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void System.Object.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_Inaccessible_02()
    {
        var src = """
new object().M();

public static class E1
{
    extension(object o)
    {
        void M() { }
    }
}
public static class E2
{
    extension(object o)
    {
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().M()");
        AssertEx.Equal("void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        src = """
new object().M();

public static class E1
{
    private static void M(this object o) { }
}
public static class E2
{
    public static void M(this object o) { }
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        tree = comp.SyntaxTrees[0];
        model = comp.GetSemanticModel(tree);
        invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().M()");
        AssertEx.Equal("void System.Object.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void System.Object.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void System.Object.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_GenericReceiverParameter()
    {
        var src = """
new object().M();

public static class Extensions
{
    extension<T>(T t)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().M()");
        AssertEx.Equal("void Extensions.<G>$8048A6C8BE30A622530249B904B537EB<System.Object>.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.SequenceEqual(["void Extensions.<G>$8048A6C8BE30A622530249B904B537EB<System.Object>.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void StaticMethodInvocation_GenericReceiverParameter_Constrained()
    {
        var src = """
object.M();
int.M();
new object().M2();

public static class Extensions
{
    extension<T>(T) where T : struct
    {
        public static void M() { }
    }
    public static void M2<T>(this T t) where T : struct { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0453: The type 'object' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Extensions.extension<T>(T)'
            // object.M();
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M").WithArguments("Extensions.extension<T>(T)", "T", "object").WithLocation(1, 8),
            // (3,14): error CS0453: The type 'object' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Extensions.M2<T>(T)'
            // new object().M2();
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M2").WithArguments("Extensions.M2<T>(T)", "T", "object").WithLocation(3, 14));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "object.M()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);

        invocation = GetSyntax<InvocationExpressionSyntax>(tree, "int.M()");
        AssertEx.Equal("void Extensions.<G>$BCF902721DDD961E5243C324D8379E5C<System.Int32>.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "int.M");
        AssertEx.SequenceEqual(["void Extensions.<G>$BCF902721DDD961E5243C324D8379E5C<System.Int32>.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void PropertyAccess_GenericReceiverParameter_Constrained()
    {
        var src = """
_ = object.P;
_ = int.P;

public static class E
{
    extension<T>(T) where T : struct
    {
        public static int P => 0;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS9286: 'object' does not contain a definition for 'P' and no accessible extension member 'P' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // _ = object.P;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "object.P").WithArguments("object", "P").WithLocation(1, 5));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.P");
        AssertEx.Equal("System.Int32 E.<G>$BCF902721DDD961E5243C324D8379E5C<T>.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "int.P");
        AssertEx.Equal("System.Int32 E.<G>$BCF902721DDD961E5243C324D8379E5C<System.Int32>.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void ReceiverParameter_TypeWithUseSiteError()
    {
        var lib1_cs = "public class MissingBase { }";
        var comp1 = CreateCompilation(lib1_cs, assemblyName: "missing");
        comp1.VerifyDiagnostics();

        var lib2_cs = "public class UseSiteError : MissingBase { }";
        var comp2 = CreateCompilation(lib2_cs, [comp1.EmitToImageReference()]);
        comp2.VerifyDiagnostics();

        var src = """
class C<T> { }
static class Extensions
{
    extension(UseSiteError) { }
    extension(C<UseSiteError>) { }
}

class C1
{
    void M(UseSiteError x) { }
    void M(C<UseSiteError> x) { }
}
""";
        var comp = CreateCompilation(src, [comp2.EmitToImageReference()]);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ReceiverParameter_SuppressConstraintChecksInitially()
    {
        var text = @"
public class C1<T> where T : struct { }

public static class Extensions
{
    extension<T>(C1<T>) { }
}
";
        var comp = CreateCompilation(text);
        comp.VerifyEmitDiagnostics(
            // (6,18): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'C1<T>'
            //     extension<T>(C1<T>) { }
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "C1<T>").WithArguments("C1<T>", "T", "T").WithLocation(6, 18));
    }

    [Fact]
    public void ReceiverParameter_SuppressConstraintChecksInitially_PointerAsTypeArgument()
    {
        var text = @"
public class C<T> { }

unsafe static class Extensions
{
    extension(C<int*>) { }
}
";
        var comp = CreateCompilation(text, options: TestOptions.UnsafeDebugDll);
        comp.VerifyEmitDiagnostics(
            // (6,15): error CS0306: The type 'int*' may not be used as a type argument
            //     extension(C<int*>) { }
            Diagnostic(ErrorCode.ERR_BadTypeArgument, "C<int*>").WithArguments("int*").WithLocation(6, 15));
    }

    [Fact]
    public void InstanceMethodInvocation_VariousScopes_Errors()
    {
        var cSrc = """
class C
{
    public static void Main()
    {
        new object().Method();
        _ = new object().Property;
    }
}
""";

        var eSrc = """
static class Extensions
{
    extension(object o)
    {
        public void Method() => throw null;
        public int Property => throw null;
    }
}
""";

        var src1 = $$"""
namespace N
{
    {{cSrc}}
    namespace N2
    {
        {{eSrc}}
    }
}
""";

        verify(src1,
            // (7,22): error CS1061: 'object' does not contain a definition for 'Method' and no accessible extension method 'Method' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         new object().Method();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Method").WithArguments("object", "Method").WithLocation(7, 22),
            // (8,26): error CS1061: 'object' does not contain a definition for 'Property' and no accessible extension method 'Property' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         _ = new object().Property;
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Property").WithArguments("object", "Property").WithLocation(8, 26));

        var src2 = $$"""
file {{eSrc}}
""";

        verify(new[] { cSrc, src2 },
            // 0.cs(5,22): error CS1061: 'object' does not contain a definition for 'Method' and no accessible extension method 'Method' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         new object().Method();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Method").WithArguments("object", "Method").WithLocation(5, 22),
            // 0.cs(6,26): error CS1061: 'object' does not contain a definition for 'Property' and no accessible extension method 'Property' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            //         _ = new object().Property;
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Property").WithArguments("object", "Property").WithLocation(6, 26));

        static void verify(CSharpTestSource src, params DiagnosticDescription[] expected)
        {
            var comp = CreateCompilation(src, options: TestOptions.DebugExe);
            comp.VerifyEmitDiagnostics(expected);

            var tree = comp.SyntaxTrees.First();
            var model = comp.GetSemanticModel(tree);

            var method = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Method");
            Assert.Null(model.GetSymbolInfo(method).Symbol);
            Assert.Empty(model.GetMemberGroup(method));

            var property = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Property");
            Assert.Null(model.GetSymbolInfo(property).Symbol);
            Assert.Empty(model.GetMemberGroup(property));
        }
    }

    [Fact]
    public void InstanceMethodInvocation_FromUsingNamespace()
    {
        var cSrc = """
class C
{
    public static void Main()
    {
        new object().Method();
    }
}
""";

        var eSrc = """
namespace N2
{
    static class E
    {
        extension(object o)
        {
            public void Method() { System.Console.Write("ran"); }
        }
    }
}
""";

        var src1 = $$"""
using N2;
{{cSrc}}

{{eSrc}}
""";
        verify(src1, "N2.E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280");

        var src2 = $$"""
using N2;
using N2; // 1, 2
{{cSrc}}

{{eSrc}}
""";

        var comp = CreateCompilation(src2, options: TestOptions.DebugExe);
        comp.VerifyEmitDiagnostics(
            // (2,1): hidden CS8019: Unnecessary using directive.
            // using N2; // 1, 2
            Diagnostic(ErrorCode.HDN_UnusedUsingDirective, "using N2;").WithLocation(2, 1),
            // (2,7): warning CS0105: The using directive for 'N2' appeared previously in this namespace
            // using N2; // 1, 2
            Diagnostic(ErrorCode.WRN_DuplicateUsing, "N2").WithArguments("N2").WithLocation(2, 7)
            );

        var src3 = $$"""
namespace N3
{
    using N2;

    namespace N4
    {
        {{cSrc}}
    }

    {{eSrc}}
}
""";
        verify(src3, "N3.N2.E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280");

        void verify(string src, string extensionName)
        {
            var comp = CreateCompilation(src, options: TestOptions.DebugExe);
            CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var invocation = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Method");
            AssertEx.Equal($$"""void {{extensionName}}.Method()""", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
            AssertEx.Equal([$$"""void {{extensionName}}.Method()"""], model.GetMemberGroup(invocation).ToTestDisplayStrings());
        }
    }

    [Fact]
    public void InstanceMethodInvocation_UsingNamespaceNecessity()
    {
        var src = """
using N;

class C
{
    public static void Main()
    {
        new object().Method();
    }
}

""";
        var eSrc = """
namespace N
{
    public static class E
    {
        extension(object o)
        {
            public void Method() { System.Console.Write("method"); }
        }
    }
}
""";

        var comp = CreateCompilation([src, eSrc], options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "method").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);

        var invocation = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Method");
        AssertEx.Equal("void N.E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void N.E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method()"], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        src = """
using N;

class C
{
    public static void Main() { }
}

namespace N
{
    public static class Extensions
    {
        extension(object o)
        {
            public void Method() { }
        }
    }
}
""";

        comp = CreateCompilation([src, eSrc]);
        comp.VerifyEmitDiagnostics(
            // (1,1): hidden CS8019: Unnecessary using directive.
            // using N;
            Diagnostic(ErrorCode.HDN_UnusedUsingDirective, "using N;").WithLocation(1, 1));
    }

    [Theory, CombinatorialData]
    public void InstanceMethodInvocation_Ambiguity(bool e1BeforeE2)
    {
        var e1 = """
static class E1
{
    extension(object o)
    {
        public void Method() => throw null;
    }
}
""";

        var e2 = """
static class E2
{
    extension(object o)
    {
        public void Method() => throw null;
    }
}
""";

        var src = $$"""
new object().Method();

{{(e1BeforeE2 ? e1 : e2)}}
{{(e1BeforeE2 ? e2 : e1)}}
""";
        var comp = CreateCompilation(src);
        if (!e1BeforeE2)
        {
            comp.VerifyEmitDiagnostics(
                // (1,14): error CS0121: The call is ambiguous between the following methods or properties: 'E2.extension(object).Method()' and 'E1.extension(object).Method()'
                // new object().Method();
                Diagnostic(ErrorCode.ERR_AmbigCall, "Method").WithArguments("E2.extension(object).Method()", "E1.extension(object).Method()").WithLocation(1, 14));
        }
        else
        {
            comp.VerifyEmitDiagnostics(
                // (1,14): error CS0121: The call is ambiguous between the following methods or properties: 'E1.extension(object).Method()' and 'E2.extension(object).Method()'
                // new object().Method();
                Diagnostic(ErrorCode.ERR_AmbigCall, "Method").WithArguments("E1.extension(object).Method()", "E2.extension(object).Method()").WithLocation(1, 14));
        }
    }

    [Fact]
    public void InstanceMethodInvocation_Overloads()
    {
        var src = """
new object().Method(42);
new object().Method("hello");

static class E1
{
    extension(object o)
    {
        public void Method(int i) { System.Console.Write($"E1.Method({i}) "); }
    }
}

static class E2
{
    extension(object o)
    {
        public void Method(string s) { System.Console.Write($"E2.Method({s}) "); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "E1.Method(42) E2.Method(hello)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var invocation1 = GetSyntax<InvocationExpressionSyntax>(tree, "new object().Method(42)");
        AssertEx.Equal("void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", model.GetSymbolInfo(invocation1).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation1));

        var invocation2 = GetSyntax<InvocationExpressionSyntax>(tree, """new object().Method("hello")""");
        AssertEx.Equal("void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)", model.GetSymbolInfo(invocation2).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation2));

        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new object().Method").First();
        AssertEx.SequenceEqual(["void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", "void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)"], model.GetMemberGroup(memberAccess1).ToTestDisplayStrings());

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new object().Method").Last();
        AssertEx.SequenceEqual(["void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", "void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)"], model.GetMemberGroup(memberAccess2).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_Overloads_DifferentScopes_NestedNamespace()
    {
        var src = """
namespace N1
{
    static class E1
    {
        extension(object o)
        {
            public void Method(int i) { System.Console.Write($"E1.Method({i}) "); }
        }
    }

    namespace N2
    {
        static class E2
        {
            extension(object o)
            {
                public void Method(string s) { System.Console.Write($"E2.Method({s}) "); }
            }
        }

        class C
        {
            public static void Main()
            {
                new object().Method(42);
                new object().Method("hello");
            }
        }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "E1.Method(42) E2.Method(hello)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var invocation1 = GetSyntax<InvocationExpressionSyntax>(tree, "new object().Method(42)");
        AssertEx.Equal("void N1.E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", model.GetSymbolInfo(invocation1).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation1));

        var invocation2 = GetSyntax<InvocationExpressionSyntax>(tree, """new object().Method("hello")""");
        AssertEx.Equal("void N1.N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)", model.GetSymbolInfo(invocation2).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation2));

        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new object().Method").First();
        AssertEx.SequenceEqual(["void N1.N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)", "void N1.E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)"], model.GetMemberGroup(memberAccess1).ToTestDisplayStrings());

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new object().Method").Last();
        AssertEx.SequenceEqual(["void N1.N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)", "void N1.E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)"], model.GetMemberGroup(memberAccess2).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_NamespaceVsUsing_FromNamespace()
    {
        var src = """
using N2;

new object().Method(42);
new object().Method("hello");
new object().Method(default);

static class E1
{
    extension(object o)
    {
        public void Method(int i) { System.Console.Write("E1.Method "); }
    }
}

namespace N2
{
    static class E2
    {
        extension(object o)
        {
            public void Method(string s) { System.Console.Write("E2.Method "); }
        }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "E1.Method E2.Method E1.Method").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var invocation1 = GetSyntax<InvocationExpressionSyntax>(tree, "new object().Method(42)");
        AssertEx.Equal("void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", model.GetSymbolInfo(invocation1).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation1));
        AssertEx.SequenceEqual(["void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", "void N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)"], model.GetMemberGroup(invocation1.Expression).ToTestDisplayStrings());

        var invocation2 = GetSyntax<InvocationExpressionSyntax>(tree, """new object().Method("hello")""");
        AssertEx.Equal("void N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)", model.GetSymbolInfo(invocation2).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation2));
        AssertEx.SequenceEqual(["void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", "void N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)"], model.GetMemberGroup(invocation2.Expression).ToTestDisplayStrings());

        var invocation3 = GetSyntax<InvocationExpressionSyntax>(tree, "new object().Method(default)");
        AssertEx.Equal("void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", model.GetSymbolInfo(invocation3).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation3));
        AssertEx.SequenceEqual(["void E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", "void N2.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.String s)"], model.GetMemberGroup(invocation3.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_DerivedDerivedType()
    {
        var src = """
new Derived().M();

class Base { }
class Derived : Base { }

static class E
{
    extension(object o)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new Derived().M()");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation));

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new Derived().M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_ImplementedInterface()
    {
        var src = """
new C().M();

interface I { }
class C : I { }

static class E
{
    extension(I i)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C().M()");
        AssertEx.Equal("void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation));

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_IndirectlyImplementedInterface()
    {
        var src = """
new C().M();

interface I { }
interface Indirect : I { }
class C : Indirect { }

static class E
{
    extension(I i)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C().M()");
        AssertEx.Equal("void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(invocation));

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_TypeParameterImplementedInterface()
    {
        var src = """
class C
{
    void M<T>(T t) where T : I
    {
        t.M();
    }
}

interface I { }

static class E
{
    extension(I i)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "t.M()");
        AssertEx.Equal("void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "t.M");
        AssertEx.Equal("void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$3EADBD08A82F6ABA9495623CB335729C.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void StaticMethodInvocation_MatchingExtendedType_TypeParameterImplementedInterface()
    {
        var src = """
class C
{
    void M<T>() where T : I
    {
        T.M();
    }
}

interface I { }

static class E
{
    extension(I)
    {
        public static void M() => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS0704: Cannot do non-virtual member lookup in 'T' because it is a type parameter
            //         T.M();
            Diagnostic(ErrorCode.ERR_LookupInTypeVariable, "T").WithArguments("T").WithLocation(5, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "T.M");
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_TypeParameterWithBaseClass()
    {
        var src = """
D.M(new D());

class C<T> { }

class D : C<D>
{
    public static void M<T>(T t) where T : C<T>
    {
        t.M2();
    }
}

static class E
{
    extension<T>(C<T> c)
    {
        public void M2() { System.Console.Write(typeof(C<T>)); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "C`1[D]").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "t.M2()");
        AssertEx.Equal("void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<T>.M2()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "t.M2");
        AssertEx.SequenceEqual(["void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<T>.M2()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_ConstrainedTypeParameter()
    {
        var src = $$"""
D.M<string>("");

class D
{
    public static void M<T>(T t) where T : class
    {
        t.M2();
    }
}

static class E1
{
    extension<T>(T t) where T : struct
    {
        public void M2() { }
    }
}

static class E2
{
    extension<T>(T t) where T : class
    {
        public void M2() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "t.M2()");
        AssertEx.Equal("void E2.<G>$66F77D1E46F965A5B22D4932892FA78B<T>.M2()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "t.M2");
        AssertEx.SequenceEqual(["void E2.<G>$66F77D1E46F965A5B22D4932892FA78B<T>.M2()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_BaseType()
    {
        var src = """
new object().M();
new object().M2();

static class E
{
    extension(string s)
    {
        public void M() => throw null;
    }
    public static void M2(this string s) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS1929: 'object' does not contain a definition for 'M' and the best extension method overload 'E.extension(string).M()' requires a receiver of type 'string'
            // new object().M();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, "new object()").WithArguments("object", "M", "E.extension(string).M()", "string").WithLocation(1, 1),
            // (2,1): error CS1929: 'object' does not contain a definition for 'M2' and the best extension method overload 'E.M2(string)' requires a receiver of type 'string'
            // new object().M2();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, "new object()").WithArguments("object", "M2", "E.M2(string)", "string").WithLocation(2, 1)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Empty(model.GetMemberGroup(memberAccess));
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_GenericType()
    {
        var src = """
new C<int>().M();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C<int>().M()");
        AssertEx.Equal("void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<System.Int32>.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());

        AssertEx.SequenceEqual(["void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<System.Int32>.M()"], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_GenericType_GenericMember_01()
    {
        var src = """
new C<int>().M<string>();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public void M<U>() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS1061: 'C<int>' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C<int>' could be found (are you missing a using directive or an assembly reference?)
            // new C<int>().M<string>();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M<string>").WithArguments("C<int>", "M").WithLocation(1, 14));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C<int>().M<string>()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<int>().M<string>");
        AssertEx.SequenceEqual(["void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<System.Int32>.M<U>()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_GenericType_GenericMember_02()
    {
        var src = """
new C<int>().M<int, string>();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public void M<U>() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C<int>().M<int, string>()");
        AssertEx.Equal("void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<System.Int32>.M<System.String>()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        AssertEx.SequenceEqual(["void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<System.Int32>.M<System.String>()"], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_GenericType_GenericMember_OmittedTypeArgument_01()
    {
        var src = """
new C<int>().M<,>();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public void M<U, V>() => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS8389: Omitting the type argument is not allowed in the current context
            // new C<int>().M<,>();
            Diagnostic(ErrorCode.ERR_OmittedTypeArgument, "new C<int>().M<,>").WithLocation(1, 1),
            // (1,14): error CS1061: 'C<int>' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C<int>' could be found (are you missing a using directive or an assembly reference?)
            // new C<int>().M<,>();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M<,>").WithArguments("C<int>", "M").WithLocation(1, 14));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C<int>().M<,>()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        AssertEx.SequenceEqual(["void E.<G>$4A1E373BE5A70EE56E2FA5F469AC30F9<System.Int32>.M<U, V>()"], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_GenericType_GenericMember_OmittedTypeArgument_02()
    {
        var src = """
new C<int>().M<,,>();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public void M<U, V>() => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS8389: Omitting the type argument is not allowed in the current context
            // new C<int>().M<,,>();
            Diagnostic(ErrorCode.ERR_OmittedTypeArgument, "new C<int>().M<,,>").WithLocation(1, 1),
            // (1,1): error CS1929: 'C<int>' does not contain a definition for 'M' and the best extension method overload 'E.extension<?>(C<?>).M<?, ?>()' requires a receiver of type 'C<?>'
            // new C<int>().M<,,>();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, "new C<int>()").WithArguments("C<int>", "M", "E.extension<?>(C<?>).M<?, ?>()", "C<?>").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C<int>().M<,,>()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());

        Assert.Equal([], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_GenericType_GenericMember_BrokenConstraint()
    {
        var src = """
new C<int>().M<int, string>();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public void M<U>() where U : struct => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'U' in the generic type or method 'E.extension<int>(C<int>).M<U>()'
            // new C<int>().M<int, string>();
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M<int, string>").WithArguments("E.extension<int>(C<int>).M<U>()", "U", "string").WithLocation(1, 14));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new C<int>().M<int, string>()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetMemberGroup(invocation).ToTestDisplayStrings());
        Assert.Equal([], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_BrokenConstraint()
    {
        var source = """
new object().Method();
new object().Method2();

static class E
{
    extension<T>(T t) where T : struct
    {
        public void Method() { }
    }
    public static void Method2<T>(this T t) where T : struct { }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS0453: The type 'object' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'E.extension<T>(T)'
            // new object().Method();
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "Method").WithArguments("E.extension<T>(T)", "T", "object").WithLocation(1, 14),
            // (2,14): error CS0453: The type 'object' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'E.Method2<T>(T)'
            // new object().Method2();
            Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "Method2").WithArguments("E.Method2<T>(T)", "T", "object").WithLocation(2, 14)
            );
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_BrokenConstraint_Nullability()
    {
        var source = """
#nullable enable
bool b = true;
var o = b ? null : new object();
o.Method();

static class E
{
    extension<T>(T t) where T : notnull
    {
        public void Method() { System.Console.Write(t is null); }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (4,1): warning CS8714: The type 'object?' cannot be used as type parameter 'T' in the generic type or method 'E.extension<T>(T)'. Nullability of type argument 'object?' doesn't match 'notnull' constraint.
            // o.Method();
            Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterNotNullConstraint, "o.Method").WithArguments("E.extension<T>(T)", "T", "object?").WithLocation(4, 1));
        CompileAndVerify(comp, expectedOutput: "True");

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "o.Method");
        AssertEx.Equal("void E.extension<System.Object?>(System.Object?).Method()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString(includeNonNullable: true));
    }

    [Fact]
    public void ReceiverParameter_AliasType()
    {
        var source = """
using Alias = C;

new Alias().M();

class C { }

static class E
{
    extension(Alias a)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new Alias().M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_DynamicArgument()
    {
        // No extension members in dynamic invocation
        var src = """
dynamic d = null;
new object().M(d);
new object().M2(d);

static class E
{
    extension(object o)
    {
        public void M(object o1) => throw null;
    }
    public static void M2(this object o, object o2) => throw null;
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (2,1): error CS1973: 'object' has no applicable method named 'M' but appears to have an extension method by that name. Extension methods cannot be dynamically dispatched. Consider casting the dynamic arguments or calling the extension method without the extension method syntax.
            // new object().M(d);
            Diagnostic(ErrorCode.ERR_BadArgTypeDynamicExtension, "new object().M(d)").WithArguments("object", "M").WithLocation(2, 1),
            // (3,1): error CS1973: 'object' has no applicable method named 'M2' but appears to have an extension method by that name. Extension methods cannot be dynamically dispatched. Consider casting the dynamic arguments or calling the extension method without the extension method syntax.
            // new object().M2(d);
            Diagnostic(ErrorCode.ERR_BadArgTypeDynamicExtension, "new object().M2(d)").WithArguments("object", "M2").WithLocation(3, 1));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(System.Object o1)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_DynamicDifference_Nested()
    {
        var src = """
new C<dynamic>().M();

class C<T> { }

static class E
{
    extension(C<object> c)
    {
        public void M()
        {
            System.Console.Write("M");
        }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "M").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<dynamic>().M");
        AssertEx.Equal("void E.<G>$BCD00C90E683E728071BA88912DD74BD.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$BCD00C90E683E728071BA88912DD74BD.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_DynamicDifference_InBase()
    {
        var src = """
new D().M();

class C<T> { }
class D : C<dynamic> { }

static class E
{
    extension(C<object> c)
    {
        public void M()
        {
            System.Console.Write("M");
        }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "M").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new D().M");
        AssertEx.Equal("void E.<G>$BCD00C90E683E728071BA88912DD74BD.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$BCD00C90E683E728071BA88912DD74BD.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_DynamicDifference_InInterface()
    {
        var src = """
new D().M();

interface I<T> { }
class D : I<dynamic> { }

static class E
{
    extension(I<object> i)
    {
        public void M() { System.Console.Write("M"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (4,11): error CS1966: 'D': cannot implement a dynamic interface 'I<dynamic>'
            // class D : I<dynamic> { }
            Diagnostic(ErrorCode.ERR_DeriveFromConstructedDynamic, "I<dynamic>").WithArguments("D", "I<dynamic>").WithLocation(4, 11));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new D().M");
        AssertEx.SequenceEqual(["void E.<G>$2B406085AC5EBECC11B16BCD2A24DF4E.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_TupleNamesDifference()
    {
        var src = """
new C<(int a, int b)>().M();
new C<(int, int)>().M();
new C<(int other, int)>().M();

class C<T> { }

static class E
{
    extension(C<(int a, int b)> c)
    {
        public void M() { System.Console.Write("M"); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "MMM").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<(int a, int b)>().M");
        AssertEx.SequenceEqual(["void E.<G>$F9AFEE2D1546C3A2A4599051616A8F6D.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<(int, int)>().M");
        AssertEx.SequenceEqual(["void E.<G>$F9AFEE2D1546C3A2A4599051616A8F6D.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<(int other, int)>().M");
        AssertEx.SequenceEqual(["void E.<G>$F9AFEE2D1546C3A2A4599051616A8F6D.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        src = """
new C<(int a, int b)>().M();
new C<(int, int)>().M();
new C<(int other, int)>().M();

class C<T> { }

static class E
{
    public static void M(this C<(int a, int b)> c) { System.Console.Write("M"); }
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_TupleNamesDifference_InBase()
    {
        var src = """
new D1().M();
new D2().M();
new D3().M();

class C<T> { }
class D1 : C<(int a, int b)> { }
class D2 : C<(int, int)> { }
class D3 : C<(int other, int)> { }

static class E
{
    extension(C<(int a, int b)> c)
    {
        public void M() { System.Console.Write("M"); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "MMM").VerifyDiagnostics();
    }

    [Fact]
    public void InstanceMethodInvocation_MatchingExtendedType_TupleNamesDifference_InInterface()
    {
        var src = """
new D1().M();
new D2().M();
new D3().M();

class I<T> { }
class D1 : I<(int a, int b)> { }
class D2 : I<(int, int)> { }
class D3 : I<(int other, int)> { }

static class E
{
    extension(I<(int a, int b)> i)
    {
        public void M() { System.Console.Write("M"); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "MMM").VerifyDiagnostics();
    }

    [Fact]
    public void InstanceMethodInvocation_Nameof()
    {
        var src = """
object o = null;
System.Console.Write($"{nameof(o.M)} ");

static class E
{
    extension(object o)
    {
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (2,32): error CS8093: Extension method groups are not allowed as an argument to 'nameof'.
            // System.Console.Write($"{nameof(o.M)} ");
            Diagnostic(ErrorCode.ERR_NameofExtensionMethod, "o.M").WithLocation(2, 32));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "o.M");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_Nameof_ViaType()
    {
        var src = """
System.Console.Write($"{nameof(E.M)} ");

static class E
{
    extension(object o)
    {
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "M").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "E.M");
        AssertEx.SequenceEqual(["void E.M(this System.Object o)", "void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_Nameof_Overloads()
    {
        var src = """
object o = null;
System.Console.Write($"{nameof(o.M)} ");

static class E
{
    extension(object o)
    {
        public void M() { }
        public void M(int i) { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (2,32): error CS8093: Extension method groups are not allowed as an argument to 'nameof'.
            // System.Console.Write($"{nameof(o.M)} ");
            Diagnostic(ErrorCode.ERR_NameofExtensionMethod, "o.M").WithLocation(2, 32));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "o.M");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", "void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_Nameof_SimpleName()
    {
        var src = """
class C
{
    void M()
    {
        _ = nameof(Method);
    }
}

static class E
{
    extension(object o)
    {
        public void Method() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (5,20): error CS0103: The name 'Method' does not exist in the current context
            //         _ = nameof(Method);
            Diagnostic(ErrorCode.ERR_NameNotInContext, "Method").WithArguments("Method").WithLocation(5, 20));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var identifier = GetSyntax<IdentifierNameSyntax>(tree, "Method");
        Assert.Equal([], model.GetMemberGroup(identifier).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_Null_Method()
    {
        var src = """
#nullable enable

object? o = null;
o.Method();

static class E
{
    extension(object o)
    {
        public void Method() { System.Console.Write("Method"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (4,1): warning CS8604: Possible null reference argument for parameter 'o' in 'extension(object)'.
            // o.Method();
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "o").WithArguments("o", "extension(object)").WithLocation(4, 1));

        CompileAndVerify(comp, expectedOutput: "Method");
    }

    [Fact]
    public void InstanceMethodInvocation_ColorColor_Method()
    {
        var src = """
C.M(new C());

class C
{
    public static void M(C C)
    {
        C.Method();
    }
}

static class E
{
    extension(C c)
    {
        public void Method() { System.Console.Write("Method "); }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "Method").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.Method");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Method()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Method()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_ColorColor_Static_Method()
    {
        var src = """
C.M(null);

class C
{
    public static void M(C C)
    {
        C.Method();
    }
}

static class E
{
    extension(C c)
    {
        public void Method() { System.Console.Write("Method"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
        CompileAndVerify(comp, expectedOutput: "Method").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.Method");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Method()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Method()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_PatternBased_ForEach_MoveNext()
    {
        var src = """
foreach (var x in new C())
{
}

class C { }
class D
{
    public int Current => 42;
}

static class E
{
    extension(C c)
    {
        public D GetEnumerator() => new D();
    }

    extension(D d)
    {
        public bool MoveNext() => true;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,19): error CS0117: 'D' does not contain a definition for 'MoveNext'
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_NoSuchMember, "new C()").WithArguments("D", "MoveNext").WithLocation(1, 19),
            // (1,19): error CS0202: foreach requires that the return type 'D' of 'E.extension(C).GetEnumerator()' must have a suitable public 'MoveNext' method and public 'Current' property
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_BadGetEnumerator, "new C()").WithArguments("D", "E.extension(C).GetEnumerator()").WithLocation(1, 19)
            );

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).MoveNextMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).CurrentProperty);
    }

    [Fact]
    public void InstanceMethodInvocation_PatternBased_ForEach_Current()
    {
        var src = """
foreach (var x in new C())
{
}

class C { }
class D
{
    public bool MoveNext() => true;
}

static class E
{
    extension(C c)
    {
        public D GetEnumerator() => new D();
    }

    extension(D d)
    {
        public int Current => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,19): error CS0117: 'D' does not contain a definition for 'Current'
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_NoSuchMember, "new C()").WithArguments("D", "Current").WithLocation(1, 19),
            // (1,19): error CS0202: foreach requires that the return type 'D' of 'E.extension(C).GetEnumerator()' must have a suitable public 'MoveNext' method and public 'Current' property
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_BadGetEnumerator, "new C()").WithArguments("D", "E.extension(C).GetEnumerator()").WithLocation(1, 19)
            );

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).MoveNextMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).CurrentProperty);
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78828")]
    public void InstanceMethodInvocation_PatternBased_ForEach_GetEnumerator_Conversion_Classic()
    {
        var src = """
foreach (var x in new C())
{
    System.Console.Write(x);
    break;
}

class C { }
class D
{
    public bool MoveNext() => true;
    public int Current => 42;
}

static class E
{
    public static D GetEnumerator(this object obj) => new D();
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78828")]
    public void Foreach_GetEnumerator_ReceiverNullableWarning()
    {
        var src = """
#nullable enable

foreach (var x in ((C?)null).Id()) // 1
{
}

class C
{
    public C Id() => this;
}

static class E
{
    extension(C c)
    {
        public System.Collections.Generic.IEnumerator<int> GetEnumerator() => throw null!;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,20): warning CS8602: Dereference of a possibly null reference.
            // foreach (var x in ((C?)null).Id()) // 1
            Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "(C?)null").WithLocation(3, 20));
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78828")]
    public void InstanceMethodInvocation_PatternBased_ForEach_GetEnumerator_Conversion()
    {
        var src = """
foreach (var x in new C())
{
    System.Console.Write(x);
    break;
}

class C { }
class D
{
    public bool MoveNext() => true;
    public int Current => 42;
}

static class E
{
    extension(object o)
    {
        public D GetEnumerator() => new D();
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78828")]
    public void ExtensionGetEnumerator_NullReceiver()
    {
        var src = """
            #nullable enable
            C? c = null;
            foreach (var x in c) // 1
            {
                System.Console.Write(x);
                break;
            }

            class C
            {
            }

            static class E
            {
                extension(C c)
                {
                    public System.Collections.Generic.IEnumerator<int> GetEnumerator() => throw null!;
                }
            }
            """;
        var comp = CreateCompilation(src);
        // Tracked by https://github.com/dotnet/roslyn/issues/78830 : diagnostic quality consider reporting a better containing symbol
        comp.VerifyEmitDiagnostics(
            // (3,19): warning CS8604: Possible null reference argument for parameter 'c' in 'extension(C)'.
            // foreach (var x in c) // 1
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "c").WithArguments("c", "extension(C)").WithLocation(3, 19));
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78828")]
    public void ExtensionGetEnumerator_NullReceiver_Classic()
    {
        var src = """
            #nullable enable
            C? c = null;
            foreach (var x in c) // 1
            {
                System.Console.Write(x);
                break;
            }

            class C
            {
            }

            static class E
            {
                public static System.Collections.Generic.IEnumerator<int> GetEnumerator(this C c) => throw null!;
            }
            """;
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): warning CS8604: Possible null reference argument for parameter 'c' in 'IEnumerator<int> E.GetEnumerator(C c)'.
            // foreach (var x in c) // 1
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "c").WithArguments("c", "IEnumerator<int> E.GetEnumerator(C c)").WithLocation(3, 19));
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78828")]
    public void ExtensionGetEnumerator_Reinference()
    {
        var src = """
            #nullable enable
            var s = "a";
            if (string.Empty == "")
                s = null;

            var c = M(s); // 'C<string?>'
            foreach (var x in c)
            {
                x.ToString(); // 1
            }

            var d = M("a"); // 'C<string>'
            foreach (var x in d)
            {
                x.ToString(); // ok
            }

            C<T> M<T>(T item) => throw null!;

            class C<T>
            {
            }

            static class E
            {
                extension<T>(C<T> c)
                {
                    public System.Collections.Generic.IEnumerator<T> GetEnumerator() => throw null!;
                }
            }
            """;
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (9,5): warning CS8602: Dereference of a possibly null reference.
            //     x.ToString(); // 1
            Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(9, 5));
    }

    [Fact]
    public void InstanceMethodInvocation_NameOf_SingleParameter()
    {
        var src = """
class C
{
    public static void Main()
    {
        string x = "";
        System.Console.Write(nameof(x));
    }
}


static class E
{
    extension(C c)
    {
        public string nameof(string s) => throw null;
    }
}
""";

        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "x").VerifyDiagnostics();
    }

    [Fact]
    public void InstanceMethodInvocation_Simple_ExpressionTree()
    {
        var source = """
using System.Linq.Expressions;
Expression<System.Action> x = () => new C().M(42);
System.Console.Write(x.Dump());
System.Action a = x.Compile();
a();

class C
{
    public void M() => throw null;
}

static class E
{
    extension(C c)
    {
        public void M(int i) { System.Console.Write(" ran"); }
    }
}
""";
        var comp = CreateCompilation([source, ExpressionTestLibrary]);
        CompileAndVerify(comp, expectedOutput: "Call(null.[Void M(C, Int32)](New([Void .ctor()]() Type:C), Constant(42 Type:System.Int32)) Type:System.Void) ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void C.M()", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_NextScope()
    {
        // If overload resolution on extension type methods yields no applicable candidates,
        // we look in the next scope.
        var source = """
using N;

new C().M(42);

class C
{
    public void M() => throw null;
}

static class E1
{
    extension(C c)
    {
        public void M(string s) => throw null;
    }
}

namespace N
{
    static class E2
    {
        extension(C c)
        {
            public void M(int i) { System.Console.Write($"E2.M({i})"); }
        }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "E2.M(42)");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void N.E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void C.M()", "void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)", "void N.E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_NewExtensionPriority()
    {
        var source = """
new C().M(42);

class C
{
    public void M() => throw null;
}

static class E1
{
    extension(C c)
    {
        public void M(int i) => throw null;
    }
}

static class E2
{
    public static void M(this C c, int i) => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,9): error CS0121: The call is ambiguous between the following methods or properties: 'E1.extension(C).M(int)' and 'E2.M(C, int)'
            // new C().M(42);
            Diagnostic(ErrorCode.ERR_AmbigCall, "M").WithArguments("E1.extension(C).M(int)", "E2.M(C, int)").WithLocation(1, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void C.M()", "void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void C.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_NewExtensionPriority_02()
    {
        var source = """
new C().M(42);

class C
{
    public void M() => throw null;
}

static class E
{
    extension(C c)
    {
        public void M(int i) => throw null;
    }
    public static void M(this C c, int i) => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,9): error CS0121: The call is ambiguous between the following methods or properties: 'E.extension(C).M(int)' and 'E.M(C, int)'
            // new C().M(42);
            Diagnostic(ErrorCode.ERR_AmbigCall, "M").WithArguments("E.extension(C).M(int)", "E.M(C, int)").WithLocation(1, 9),
            // (12,21): error CS0111: Type 'E' already defines a member called 'M' with the same parameter types
            //         public void M(int i) => throw null;
            Diagnostic(ErrorCode.ERR_MemberAlreadyExists, "M").WithArguments("M", "E").WithLocation(12, 21)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void C.M()", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void C.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_FallbackToExtensionMethod()
    {
        // The extension method is picked up if extension declaration candidates were not applicable
        var source = """
new C().M(42);

class C
{
    public static void M() => throw null;
}

static class E1
{
    extension(C c)
    {
        public void M(string s) => throw null;
        public void M(char c1) => throw null;
    }
}

static class E2
{
    public static void M(this C c, int i) { System.Console.Write($"E2.M({i})"); }
}
""";
        var comp = CreateCompilation(source);

        CompileAndVerify(comp, expectedOutput: "E2.M(42)");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void C.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void C.M()", "void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)", "void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Char c1)", "void C.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_SimpleName()
    {
        // Extension invocation comes into play on an invocation on a member access but not an invocation on a simple name
        var source = """
class C
{
    public void M() => throw null;

    void M2()
    {
        M(42); // 1
    }
}

static class E
{
    extension(C c)
    {
        public void M(int i) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // 0.cs(7,9): error CS1501: No overload for method 'M' takes 1 arguments
            //         M(42); // 1
            Diagnostic(ErrorCode.ERR_BadArgCount, "M").WithArguments("M", "1").WithLocation(7, 9)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "M(42)");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Empty(model.GetMemberGroup(invocation));
        AssertEx.SequenceEqual(["void C.M()"], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_ArgumentName()
    {
        // Instance method with incompatible parameter name is skipped in favor of extension declaration method
        var source = """
new C().M(b: 42);

class C
{
    public void M(int a) => throw null;
}

static class E1
{
    extension(C c)
    {
        public void M(int b) { System.Console.Write($"E1.M({b})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics();
        CompileAndVerify(comp, expectedOutput: "E1.M(42)");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 b)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void C.M(System.Int32 a)", "void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 b)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_ArgumentName_02()
    {
        // Extension declaration method with incompatible parameter name is skipped in favor of extension method
        var source = """
new C().M(c: 42);

public class C
{
    public static void M(int a) => throw null;
}

static class E1
{
    extension(C c)
    {
        public void M(int b) => throw null;
    }
}

public static class E2
{
    public static void M(this C self, int c)
    {
        System.Console.Write($"E2.M({c})");
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "E2.M(42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void C.M(System.Int32 c)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        AssertEx.SequenceEqual(["void C.M(System.Int32 a)", "void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 b)", "void C.M(System.Int32 c)"],
            model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_ArgumentName_03()
    {
        var source = """
new object().M(c: 43, b: 42);

static class E
{
    extension(object o)
    {
        public void M(int b, int c) { System.Console.Write($"E.M({b}, {c})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "E.M(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(System.Int32 b, System.Int32 c)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_ArgumentName_04()
    {
        var source = """
new object().M(o: new object());

static class E
{
    extension(object o)
    {
        public void M(object o2) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,16): error CS1744: Named argument 'o' specifies a parameter for which a positional argument has already been given
            // new object().M(o: new object());
            Diagnostic(ErrorCode.ERR_NamedArgumentUsedInPositional, "o").WithArguments("o").WithLocation(1, 16));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
    }

    [Fact]
    public void InstanceMethodInvocation_RefKind()
    {
        var source = """
int i = 42;
int j;

new object().M(ref i, out j);

static class E
{
    extension(object o)
    {
        public void M(ref int b, out int c) { c = 43; System.Console.Write($"E.M({b}, {c})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "E.M(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(ref System.Int32 b, out System.Int32 c)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_AmbiguityWithExtensionOnBaseType_PreferMoreSpecific()
    {
        var source = """
System.Console.Write(new C().M(42));

class Base { }

class C : Base { }

static class E1
{
    extension(Base b)
    {
        public int M(int i) => throw null;
    }
}

static class E2
{
    extension(C c)
    {
        public int M(int i) => i;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("System.Int32 E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["System.Int32 E1.<G>$76A32DFFBBF61DFEA0C27B13F12F6EFB.M(System.Int32 i)", "System.Int32 E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        source = """
System.Console.Write(new C().M(42));

public class Base { }

public class C : Base { }

public static class E1
{
    public static int M(this Base b, int i) => throw null;
}

public static class E2
{
    public static int M(this C c, int i) => i;
}
""";
        comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();
    }

    [Fact]
    public void InstanceMethodInvocation_TypeArguments()
    {
        var source = """
new C().M<object>(42);

class C { }

static class E
{
    extension(C c)
    {
        public void M(int i) => throw null;
        public void M<T>(int i)
        {
            System.Console.Write("ran");
        }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M<object>");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<System.Object>(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<System.Object>(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_TypeArguments_WrongNumber()
    {
        var source = """
new C().M<object, object>(42);

class C { }

static class E
{
    extension(C c)
    {
        public void M(int i) => throw null;
        public void M<T>(int i) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // 0.cs(1,9): error CS1061: 'C' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // new C().M<object, object>(42);
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M<object, object>").WithArguments("C", "M").WithLocation(1, 9)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M<object, object>");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<T>(System.Int32 i)"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<T>(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_TypeArguments_Omitted()
    {
        var source = """
new C().M<>(42);

class C { }

static class E
{
    extension(C c)
    {
        public void M<T>(int i) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS8389: Omitting the type argument is not allowed in the current context
            // new C().M<>(42);
            Diagnostic(ErrorCode.ERR_OmittedTypeArgument, "new C().M<>").WithLocation(1, 1)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M<>");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<?>(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<?>(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_TypeArguments_Inferred()
    {
        // No type arguments passed, but the extension declaration method is found and the type parameter inferred
        var source = """
new C().M(42);

class C { }

static class E
{
    extension(C c)
    {
        public void M<T>(T t)
        {
            System.Console.Write($"M({t})");
        }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "M(42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<System.Int32>(System.Int32 t)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<T>(T t)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void StaticMethodInvocation_InstanceExtensionMethod()
    {
        // The extension method is not static, but the receiver is a type
        var source = """
C.M();

class C { }

static class E
{
    extension(C c)
    {
        public void M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0120: An object reference is required for the non-static field, method, or property 'E.extension(C).M()'
            // C.M();
            Diagnostic(ErrorCode.ERR_ObjectRequired, "C.M").WithArguments("E.extension(C).M()").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        source = """
C.Method();

public class C { }

public static class E
{
    public static void Method(this C c) { }
}
""";
        comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // 0.cs(1,1): error CS0120: An object reference is required for the non-static field, method, or property 'E.Method(C)'
            // C.Method();
            Diagnostic(ErrorCode.ERR_ObjectRequired, "C.Method").WithArguments("E.Method(C)").WithLocation(1, 1));
    }

    [Fact]
    public void InstanceMethodInvocation_StaticExtensionMethod()
    {
        // The extension method is static but the receiver is a value
        var source = """
new C().M();

class C { }

static class E
{
    extension(C c)
    {
        public static void M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0176: Member 'E.extension(C).M()' cannot be accessed with an instance reference; qualify it with a type name instead
            // new C().M();
            Diagnostic(ErrorCode.ERR_ObjectProhibited, "new C().M").WithArguments("E.extension(C).M()").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_GenericType()
    {
        var src = """
new C<int>().StaticType<string>();

class C<T> { }

static class E
{
    extension<T>(C<T> c)
    {
        public static class StaticType<U> { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS1061: 'C<int>' does not contain a definition for 'StaticType' and no accessible extension method 'StaticType' accepting a first argument of type 'C<int>' could be found (are you missing a using directive or an assembly reference?)
            // new C<int>().StaticType<string>();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "StaticType<string>").WithArguments("C<int>", "StaticType").WithLocation(1, 14),
            // (9,29): error CS9282: This member is not allowed in an extension block
            //         public static class StaticType<U> { }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "StaticType").WithLocation(9, 29));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<int>().StaticType<string>");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess).CandidateReason);
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_RefOmittedComCall()
    {
        // For COM import type, omitting the ref is allowed
        string source = @"
using System;
using System.Runtime.InteropServices;

[ComImport, Guid(""1234C65D-1234-447A-B786-64682CBEF136"")]
class C { }

static class E
{
    extension(C c)
    {
        public void M(ref short p) { }
        public void M(sbyte p) { }
        public void I(ref int p) { }
    }
}

class X
{
    public static void Goo()
    {
        short x = 123;
        C c = new C();
        c.M(x);
        c.I(123);
    }
}
";
        var comp = CreateCompilation(source);
        comp.VerifyDiagnostics();
    }

    [Fact]
    public void ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_ExtensionDeclarationMethods()
    {
        // See ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_Method
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.M1(this);
    }
}

class Color { }

static class E
{
    extension(Color c)
    {
        public void M1(S1 x, int y = 0) { System.Console.WriteLine("instance"); }

        public static void M1<T>(T x) where T : unmanaged { System.Console.WriteLine("static"); }
    }
}
""";
        var comp = CreateCompilation(source, options: TestOptions.ReleaseDll);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.M1(this);
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9)
            );

        Assert.NotEmpty(comp.GetTypeByMetadataName("S1").InstanceConstructors.OfType<SynthesizedPrimaryConstructor>().Single().GetCapturedParameters());

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.M1");
        AssertEx.Equal("void E.<G>$2404CFB602D7DEE90BDDEF217EC37C58.M1(S1 x, [System.Int32 y = 0])", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_ExtensionDeclarationMembersVsExtensionMethod()
    {
        var source = """
public struct S1(Color Color)
{
    public void Test()
    {
        Color.M1(this);
    }
}

public class Color { }

public static class E1
{
    public static void M1(this Color c, S1 x, int y = 0) { System.Console.WriteLine("instance"); }
}

static class E2
{
    extension(Color c)
    {
        public static void M1<T>(T x) where T : unmanaged { System.Console.WriteLine("static"); }
    }
}
""";
        var comp = CreateCompilation(source, options: TestOptions.ReleaseDll);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.M1(this);
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9)
            );

        Assert.NotEmpty(comp.GetTypeByMetadataName("S1").InstanceConstructors.OfType<SynthesizedPrimaryConstructor>().Single().GetCapturedParameters());

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.M1");
        AssertEx.Equal("void Color.M1(S1 x, [System.Int32 y = 0])", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_NotOnBase()
    {
        // Unlike `this`, `base` is not an expression in itself.
        // "Extension invocation" and "extension member lookup" do not apply to `base_access` syntax.
        var src = """
class Base { }

class Derived : Base
{
    void Main()
    {
        M(); // 1
        base.M(); // 2
    }
}

static class E
{
    extension(Base b)
    {
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (7,9): error CS0103: The name 'M' does not exist in the current context
            //         M(); // 1
            Diagnostic(ErrorCode.ERR_NameNotInContext, "M").WithArguments("M").WithLocation(7, 9),
            // (8,14): error CS0117: 'Base' does not contain a definition for 'M'
            //         base.M(); // 2
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("Base", "M").WithLocation(8, 14));
    }

    [Fact]
    public void LookupKind_Invocation()
    {
        // Non-invocable extension member in inner scope is skipped in favor of invocable one from outer scope
        var src = """
using N;

new object().Member();

static class E
{
    extension(object o)
    {
        public int Member => 0;
    }
}

namespace N
{
    static class E2
    {
        extension(object o)
        {
            public void Member() { System.Console.Write("ran "); }
        }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Member");
        AssertEx.Equal("void N.E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_NotUnique()
    {
        var src = """
new C<object, dynamic>().M();
new C<dynamic, object>().M();

new C<object, dynamic>().M2();
new C<dynamic, object>().M2();

class C<T, U> { }

static class E
{
    extension<T>(C<T, T> c)
    {
        public string M() => "hi";
    }
    public static string M2<T>(this C<T, T> c) => "hi";
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,26): error CS1061: 'C<object, dynamic>' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C<object, dynamic>' could be found (are you missing a using directive or an assembly reference?)
            // new C<object, dynamic>().M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("C<object, dynamic>", "M").WithLocation(1, 26),
            // (2,26): error CS1061: 'C<dynamic, object>' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C<dynamic, object>' could be found (are you missing a using directive or an assembly reference?)
            // new C<dynamic, object>().M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("C<dynamic, object>", "M").WithLocation(2, 26),
            // (4,26): error CS1061: 'C<object, dynamic>' does not contain a definition for 'M2' and no accessible extension method 'M2' accepting a first argument of type 'C<object, dynamic>' could be found (are you missing a using directive or an assembly reference?)
            // new C<object, dynamic>().M2();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M2").WithArguments("C<object, dynamic>", "M2").WithLocation(4, 26),
            // (5,26): error CS1061: 'C<dynamic, object>' does not contain a definition for 'M2' and no accessible extension method 'M2' accepting a first argument of type 'C<dynamic, object>' could be found (are you missing a using directive or an assembly reference?)
            // new C<dynamic, object>().M2();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M2").WithArguments("C<dynamic, object>", "M2").WithLocation(5, 26));
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_NestedTuples()
    {
        var src = """
var s = new C<(string, string)>.Nested<(int, int)>().M();
System.Console.Write(s);

class C<T>
{
    internal class Nested<U> { }
}

static class E
{
    extension<T1, T2>(C<(T1, T1)>.Nested<(T2, T2)> cn)
    {
        public string M() => "hi";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "hi").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<(string, string)>.Nested<(int, int)>().M");
        AssertEx.Equal("System.String E.<G>$FD79C355D693194B747A629F6876929C<System.String, System.Int32>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_PointerArray()
    {
        var src = """
unsafe
{
    string s = new C<long*[]>.Nested<int*[]>().M();
    System.Console.Write(s);
}

unsafe class C<T>
{
    internal class Nested<U> { }
}

unsafe static class E
{
    extension<T1, T2>(C<T1*[]>.Nested<T2*[]> cn)
        where T1 : unmanaged
        where T2 : unmanaged
    {
        public string M() => "hi";
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.UnsafeDebugExe);
        CompileAndVerify(comp, expectedOutput: "hi").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<long*[]>.Nested<int*[]>().M");
        AssertEx.Equal("System.String E.<G>$C781704B647A2CCC8FD47AE9790BA08B<System.Int64, System.Int32>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_Pointer()
    {
        var src = """
unsafe
{
    new C<long*[]>.Nested<int*[]>().M();
    new C<long*[]>.Nested<int*[]>().M2();
}

unsafe class C<T>
{
    internal class Nested<U> { }
}

static class E
{
    extension<T1, T2>(C<T1[]>.Nested<T2[]> cn)
    {
        public string M() => null;
    }
    public static string M2<T1, T2>(this C<T1[]>.Nested<T2[]> cn) => null;
}
""";
        var comp = CreateCompilation(src, options: TestOptions.UnsafeDebugExe);
        comp.VerifyEmitDiagnostics(
            // (3,37): error CS0306: The type 'long*' may not be used as a type argument
            //     new C<long*[]>.Nested<int*[]>().M();
            Diagnostic(ErrorCode.ERR_BadTypeArgument, "M").WithArguments("long*").WithLocation(3, 37),
            // (3,37): error CS0306: The type 'int*' may not be used as a type argument
            //     new C<long*[]>.Nested<int*[]>().M();
            Diagnostic(ErrorCode.ERR_BadTypeArgument, "M").WithArguments("int*").WithLocation(3, 37),
            // (4,37): error CS0306: The type 'long*' may not be used as a type argument
            //     new C<long*[]>.Nested<int*[]>().M2();
            Diagnostic(ErrorCode.ERR_BadTypeArgument, "M2").WithArguments("long*").WithLocation(4, 37),
            // (4,37): error CS0306: The type 'int*' may not be used as a type argument
            //     new C<long*[]>.Nested<int*[]>().M2();
            Diagnostic(ErrorCode.ERR_BadTypeArgument, "M2").WithArguments("int*").WithLocation(4, 37)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<long*[]>.Nested<int*[]>().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_FunctionPointer()
    {
        var src = """
unsafe
{
    string s = new C<delegate*<int>[]>.Nested<delegate*<long>[]>().M();
    System.Console.Write(s);
}

unsafe class C<T>
{
    internal class Nested<U> { }
}

unsafe static class E
{
    extension<T1, T2>(C<delegate*<T1>[]>.Nested<delegate*<T2>[]> cn)
    {
        public string M() => "hi";
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.UnsafeDebugExe);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<delegate*<int>[]>.Nested<delegate*<long>[]>().M");
        AssertEx.Equal("System.String E.<G>$5F3142482E98DE8C6B0C70A682DD0496<System.Int32, System.Int64>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_ForInterface()
    {
        var src = """
string s = new C<int>().M();
System.Console.Write(s);

class C<T> : I<T> { }
interface I<T> { }

static class E
{
    extension<T>(I<T> i)
    {
        public string M() => "hi";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "hi").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<int>().M");
        AssertEx.Equal("System.String E.<G>$74EBC78B2187AB07A25EEFC1322000B0<System.Int32>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_ForBaseInterface()
    {
        var src = """
string s = new C<int>().M();
System.Console.Write(s);

class C<T> : I<T> { }
interface I<T> : I2<T> { }
interface I2<T> { }

static class E
{
    extension<T>(I2<T> i)
    {
        public string M() => "hi";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "hi").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<int>().M");
        AssertEx.Equal("System.String E.<G>$5D7EC0FD2C9001515B0ADE0CEE121AB0<System.Int32>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Generic_ForBase()
    {
        var src = """
string s = new C<int, string>().M();
System.Console.Write(s);

class Base<T, U> { }
class C<T, U> : Base<U, T> { }

static class E
{
    extension<T, U>(Base<T, U> b)
    {
        public string M() => "hi";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "hi").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C<int, string>().M");
        AssertEx.Equal("System.String E.<G>$414BE9969A3DFDFF167B842681736663<System.String, System.Int32>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void InstanceMethodInvocation_Obsolete()
    {
        var src = """
new object().Method();

static class E
{
    extension(object o)
    {
        [System.Obsolete("Method is obsolete", true)]
        public void Method() => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0619: 'E.extension(object).Method()' is obsolete: 'Method is obsolete'
            // new object().Method();
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "new object().Method()").WithArguments("E.extension(object).Method()", "Method is obsolete").WithLocation(1, 1));
    }

    [Fact]
    public void InstanceMethodInvocation_BrokenConstraintMethodOuterExtension()
    {
        var src = """
static class E2
{
    extension(object o)
    {
        public void M<T>() => throw null;
    }
}

namespace Inner
{
    class C
    {
        public static void Main()
        {
            new C().M<object>();
        }
    }

    static class E1
    {
        extension(C c)
        {
            public string M<T>() where T : struct => throw null;
        }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M<object>");
        AssertEx.Equal("void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M<System.Object>()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M<System.Object>()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethodInvocation_MultipleSubstitutions()
    {
        var src = """
new C().M();

interface I<T> { }
class C : I<int>, I<string> { }

static class E
{
    extension<T>(I<T> i)
    {
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,9): error CS1061: 'C' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // new C().M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("C", "M").WithLocation(1, 9));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Empty(model.GetMemberGroup(memberAccess));
    }

    [Theory, CombinatorialData]
    public void InstanceMethodInvocation_MultipleExtensions(bool e1BeforeE2)
    {
        var e1 = """
static class E1
{
    extension(object o)
    {
        public string M() => throw null;
    }
}
""";

        var e2 = """
static class E2
{
    extension(object o)
    {
        public string M() => throw null;
    }
}
""";

        var src = $$"""
new object().M();
{{(e1BeforeE2 ? e1 : e2)}}
{{(e1BeforeE2 ? e2 : e1)}}
""";
        var comp = CreateCompilation(src);
        if (!e1BeforeE2)
        {
            comp.VerifyEmitDiagnostics(
                // (1,14): error CS0121: The call is ambiguous between the following methods or properties: 'E2.extension(object).M()' and 'E1.extension(object).M()'
                // new object().M();
                Diagnostic(ErrorCode.ERR_AmbigCall, "M").WithArguments("E2.extension(object).M()", "E1.extension(object).M()").WithLocation(1, 14));
        }
        else
        {
            comp.VerifyEmitDiagnostics(
                // (1,14): error CS0121: The call is ambiguous between the following methods or properties: 'E1.extension(object).M()' and 'E2.extension(object).M()'
                // new object().M();
                Diagnostic(ErrorCode.ERR_AmbigCall, "M").WithArguments("E1.extension(object).M()", "E2.extension(object).M()").WithLocation(1, 14));
        }

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        if (e1BeforeE2)
        {
            AssertEx.SequenceEqual(["System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", "System.String E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        }
        else
        {
            AssertEx.SequenceEqual(["System.String E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", "System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        }
    }

    public class ThreePermutationGenerator : IEnumerable<object[]>
    {
        private readonly List<object[]> _data = [
            [0, 1, 2],
            [0, 2, 1],
            [1, 0, 2],
            [1, 2, 0],
            [2, 0, 1],
            [2, 1, 0]];

        public IEnumerator<object[]> GetEnumerator() => _data.GetEnumerator();

        IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();
    }

    [Theory, ClassData(typeof(ThreePermutationGenerator))]
    public void InstanceMethodInvocation_InterfaceAppearsTwice(int first, int second, int third)
    {
        string[] segments = [
            """
            static class E1
            {
                extension<T>(I1<T> i)
                {
                    public string M() => null;
                }
            }
            """,
            """
            static class E2
            {
                extension(I2 i) { }
            }
            """,
            """
            static class E3
            {
                extension(C c) { }
            }
            """];

        var src = $$"""
System.Console.Write(new C().M());

interface I1<T> { }
interface I2 : I1<string> { }

class C : I1<int>, I2 { }

{{segments[first]}}

{{segments[second]}}

{{segments[third]}}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,30): error CS1061: 'C' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // System.Console.Write(new C().M());
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("C", "M").WithLocation(1, 30));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Empty(model.GetMemberGroup(memberAccess));
    }

    [Fact]
    public void InstanceMethodInvocation_SingleStageInference()
    {
        var src = """
public class C
{
    public void M(I<string> i, out object o)
    {
        i.M(out o);
        i.M2(out o);
    }
}

public static class E
{
   public static void M<T>(this I<T> i, out T t) { t = default; }
}

static class E2
{
    extension<T>(I<T> i)
    {
       public void M2(out T t) { t = default; }
    }
}

public interface I<out T> { }
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "i.M");
        AssertEx.Equal("void I<System.Object>.M<System.Object>(out System.Object t)", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void I<System.String>.M<System.String>(out System.String t)"], model.GetMemberGroup(memberAccess1).ToTestDisplayStrings());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "i.M2");
        AssertEx.Equal("void E2.<G>$74EBC78B2187AB07A25EEFC1322000B0<System.Object>.M2(out System.Object t)", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void E2.<G>$74EBC78B2187AB07A25EEFC1322000B0<System.String>.M2(out System.String t)"], model.GetMemberGroup(memberAccess2).ToTestDisplayStrings());
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_01()
    {
        var src = """
using System.Collections.Generic;

IEnumerable<string> i = null;
i.M();
_ = i.P;

static class E
{
    extension(IEnumerable<object> o)
    {
        public void M() { System.Console.Write(o is null); }
        public int P { get { System.Console.Write(o is null); return 0; } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "TrueTrue").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "i.M");
        AssertEx.Equal("void E.<G>$977919F21861BE18BA139544085CA0BD.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$977919F21861BE18BA139544085CA0BD.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "i.P");
        AssertEx.Equal("System.Int32 E.<G>$977919F21861BE18BA139544085CA0BD.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        src = """
using System.Collections.Generic;

IEnumerable<object> i = null;
i.M();
_ = i.P;

static class E
{
    extension(IEnumerable<string> o)
    {
        public static void M() { }
        public static int P => throw null;
    }
}
""";
        comp = CreateCompilation(src);

        comp.VerifyEmitDiagnostics(
            // (4,1): error CS1929: 'IEnumerable<object>' does not contain a definition for 'M' and the best extension method overload 'E.extension(IEnumerable<string>).M()' requires a receiver of type 'System.Collections.Generic.IEnumerable<string>'
            // i.M();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, "i").WithArguments("System.Collections.Generic.IEnumerable<object>", "M", "E.extension(System.Collections.Generic.IEnumerable<string>).M()", "System.Collections.Generic.IEnumerable<string>").WithLocation(4, 1),
            // (5,5): error CS9286: 'IEnumerable<object>' does not contain a definition for 'P' and no accessible extension member 'P' for receiver of type 'IEnumerable<object>' could be found (are you missing a using directive or an assembly reference?)
            // _ = i.P;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "i.P").WithArguments("System.Collections.Generic.IEnumerable<object>", "P").WithLocation(5, 5)
            );
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_02()
    {
        var src = """
string.M();
_ = string.P;

static class E
{
    extension(object)
    {
        public static void M() { System.Console.Write("ran "); }
        public static int P { get { System.Console.Write("ran2"); return 0; } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran ran2").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "string.M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_03()
    {
        var src = """
int.M();
_ = int.P;

static class E
{
    extension(object)
    {
        public static void M() { System.Console.Write("ran "); }
        public static int P { get { System.Console.Write("ran2"); return 0; } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran ran2").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "int.M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_04()
    {
        var src = """
int.M();
42.M2();

_ = int.P;
_ = 42.P2;

static class E
{
    extension(int? i)
    {
        public static void M() { }
        public static int P => 0;
        public int P2 => 0;
    }
    public static void M2(this int? i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS1929: 'int' does not contain a definition for 'M' and the best extension method overload 'E.extension(int?).M()' requires a receiver of type 'int?'
            // int.M();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, "int").WithArguments("int", "M", "E.extension(int?).M()", "int?").WithLocation(1, 1),
            // (2,1): error CS1929: 'int' does not contain a definition for 'M2' and the best extension method overload 'E.M2(int?)' requires a receiver of type 'int?'
            // 42.M2();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, "42").WithArguments("int", "M2", "E.M2(int?)", "int?").WithLocation(2, 1),
            // (4,5): error CS9286: 'int' does not contain a definition for 'P' and no accessible extension member 'P' for receiver of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // _ = int.P;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "int.P").WithArguments("int", "P").WithLocation(4, 5),
            // (5,5): error CS9286: 'int' does not contain a definition for 'P2' and no accessible extension member 'P2' for receiver of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // _ = 42.P2;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "42.P2").WithArguments("int", "P2").WithLocation(5, 5));
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_05()
    {
        var src = """
MyEnum.Zero.M();

enum MyEnum { Zero }

static class E
{
    extension(System.Enum e)
    {
        public void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_06()
    {
        var src = """
dynamic d = new C();
d.M();
d.M2();

static class E
{
    extension(object o)
    {
        public void M() => throw null;
    }

    public static void M2(this object o) => throw null;
}

class C
{
    public void M() { System.Console.Write("ran "); }
    public void M2() { System.Console.Write("ran2"); }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        CompileAndVerify(comp, expectedOutput: ExpectedOutput("ran ran2"), verify: Verification.FailsPEVerify).VerifyDiagnostics();
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_07()
    {
        var src = """
object o = null;
o.M();
o.M2();

static class E
{
    extension(dynamic d)
    {
        public void M() { }
    }

    public static void M2(this dynamic d) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (7,15): error CS1103: The receiver parameter of an extension cannot be of type 'dynamic'
            //     extension(dynamic d)
            Diagnostic(ErrorCode.ERR_BadTypeforThis, "dynamic").WithArguments("dynamic").WithLocation(7, 15),
            // (12,32): error CS1103: The receiver parameter of an extension cannot be of type 'dynamic'
            //     public static void M2(this dynamic d) { }
            Diagnostic(ErrorCode.ERR_BadTypeforThis, "dynamic").WithArguments("dynamic").WithLocation(12, 32));
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_08()
    {
        var src = """
(int a, int b) t = default;
t.M();
t.M2();

static class E
{
    extension((int c, int d) t)
    {
        public void M() { }
    }

    public static void M2(this (int c, int d) t) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_09()
    {
        var src = """
int[] i = default;
i.M();
i.M2();

static class E
{
    extension(System.ReadOnlySpan<int> ros)
    {
        public void M() { }
    }

    public static void M2(this System.ReadOnlySpan<int> ros) { }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_10()
    {
        var missingSrc = """
public class Missing { }
""";
        var missingRef = CreateCompilation(missingSrc, assemblyName: "missing").EmitToImageReference();

        var derivedSrc = """
public class Derived : Missing { }
""";
        var derivedRef = CreateCompilation(derivedSrc, references: [missingRef]).EmitToImageReference();

        var src = """
new Derived().M();
new Derived().M2();

class Other { }

static class E
{
    extension(Other o)
    {
        public void M() { }
    }

    public static void M2(this Other o) { }
}
""";
        var comp = CreateCompilation(src, references: [derivedRef]);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Derived().M();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "new Derived().M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 1),
            // (1,15): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Derived().M();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 15),
            // (2,1): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Derived().M2();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "new Derived().M2").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(2, 1),
            // (2,15): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Derived().M2();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M2").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(2, 15));
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_11()
    {
        var missingSrc = """
public class Missing { }
""";
        var missingRef = CreateCompilation(missingSrc, assemblyName: "missing").EmitToImageReference();

        var derivedSrc = """
public class Derived : Missing { }
""";
        var derivedRef = CreateCompilation(derivedSrc, references: [missingRef]).EmitToImageReference();

        var src = """
new Derived().M();
new Derived().M2();

static class E
{
    extension(Derived d)
    {
        public void M() { }
    }

    public static void M2(this Derived d) { }
}
""";
        var comp = CreateCompilation(src, references: [derivedRef]);
        comp.VerifyEmitDiagnostics(
            // (1,15): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Derived().M();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 15),
            // (2,15): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Derived().M2();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M2").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(2, 15));
    }

    [Fact]
    public void GetCompatibleExtension_Conversion_12()
    {
        var missingSrc = """
public class Missing { }
""";
        var missingRef = CreateCompilation(missingSrc, assemblyName: "missing").EmitToImageReference();

        var derivedSrc = """
public class I<T> { }
public class Derived : I<Missing> { }
""";
        var derivedRef = CreateCompilation(derivedSrc, references: [missingRef]).EmitToImageReference();

        var src = """
new Derived().M();
new Derived().M2();

static class E
{
    extension(I<object> i)
    {
        public void M() { }
    }

    public static void M2(this I<object> i) { }
}
""";
        var comp = CreateCompilation(src, references: [derivedRef]);
        comp.VerifyEmitDiagnostics(
                // (1,1): error CS1929: 'Derived' does not contain a definition for 'M' and the best extension method overload 'E.extension(I<object>).M()' requires a receiver of type 'I<object>'
                // new Derived().M();
                Diagnostic(ErrorCode.ERR_BadInstanceArgType, "new Derived()").WithArguments("Derived", "M", "E.extension(I<object>).M()", "I<object>").WithLocation(1, 1),
                // (2,1): error CS1929: 'Derived' does not contain a definition for 'M2' and the best extension method overload 'E.M2(I<object>)' requires a receiver of type 'I<object>'
                // new Derived().M2();
                Diagnostic(ErrorCode.ERR_BadInstanceArgType, "new Derived()").WithArguments("Derived", "M2", "E.M2(I<object>)", "I<object>").WithLocation(2, 1));
    }

    [Fact]
    public void GetCompatibleExtension_TypeInference_01()
    {
        var src = """
I<object, string>.M();

interface I<out T1, out T2> { }

static class E
{
    extension<T>(I<T, T>)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "I<object, string>.M");
        AssertEx.Equal("void E.<G>$B135BA58FDFC6D88E9886008265BE41B<System.Object>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void GetCompatibleExtension_TypeInference_02()
    {
        var src = """
I<object, string>.M();

interface I<in T1, in T2> { }

static class E
{
    extension<T>(I<T, T>)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "I<object, string>.M");
        AssertEx.Equal("void E.<G>$B135BA58FDFC6D88E9886008265BE41B<System.String>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void GetCompatibleExtension_TypeInference_03()
    {
        var src = """
I<object, string>.M();

interface I<T1, T2> { }

static class E
{
    extension<T>(I<T, T>)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,19): error CS1061: 'I<object, string>' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'I<object, string>' could be found (are you missing a using directive or an assembly reference?)
            // I<object, string>.M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("I<object, string>", "M").WithLocation(1, 19));
    }

    [Fact]
    public void GetCompatibleExtension_Constraint_UseSiteInfo_01()
    {
        var missingSrc = """
public struct Missing { public int i; }
""";
        var missingRef = CreateCompilation(missingSrc, assemblyName: "missing").EmitToImageReference();

        var containerSrc = """
public struct Container { public Missing field; }
""";
        var containerRef = CreateCompilation(containerSrc, references: [missingRef]).EmitToImageReference();

        var src = """
Container.M();

static class E
{
    extension<T>(T t) where T : unmanaged
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics(
            // (1,11): error CS8377: The type 'Container' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'E.extension<T>(T)'
            // Container.M();
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("E.extension<T>(T)", "T", "Container").WithLocation(1, 11),
            // (1,11): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // Container.M();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 11));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Container.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);

        src = """
new Container().M();

static class E
{
    public static void M<T>(this T t) where T : unmanaged { }
}
""";
        comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics(
            // (1,17): error CS8377: The type 'Container' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'E.M<T>(T)'
            // new Container().M();
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("E.M<T>(T)", "T", "Container").WithLocation(1, 17),
            // (1,17): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new Container().M();
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 17));

        src = """
new object().M(new Container());

static class E
{
    public static void M<T>(this object o, T t) where T : unmanaged { }
}
""";
        comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS8377: The type 'Container' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'E.M<T>(object, T)'
            // new object().M(new Container());
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("E.M<T>(object, T)", "T", "Container").WithLocation(1, 14),
            // (1,14): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // new object().M(new Container());
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 14));
    }

    [Fact]
    public void GetCompatibleExtension_Constraint_UseSiteInfo_02()
    {
        var missingSrc = """
public struct Missing { public int i; }
""";
        var missingRef = CreateCompilation(missingSrc, assemblyName: "missing").EmitToImageReference();

        var containerSrc = """
public struct Container { public Missing field; }
""";
        var containerRef = CreateCompilation(containerSrc, references: [missingRef]).EmitToImageReference();

        var src = """
using N;

Container.M();

static class E
{
    extension<T>(T t) where T : unmanaged
    {
        public static void M(int inapplicable) => throw null;
    }
}

namespace N
{
    static class E2
    {
        extension<T>(T t)
        {
            public static void M() { }
        }
    }
}
""";
        var comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics(); // The inapplicable candidate gets rejected before we get to check its constraints

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Container.M");
        AssertEx.Equal("void N.E2.<G>$8048A6C8BE30A622530249B904B537EB<Container>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());

        src = """
Container.M();

static class E
{
    extension<T>(T t) where T : unmanaged
    {
        public static void M(int inapplicable) => throw null;
    }
}
""";
        comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics(
            // (1,11): error CS7036: There is no argument given that corresponds to the required parameter 'inapplicable' of 'E.extension<T>(T).M(int)'
            // Container.M();
            Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, "M").WithArguments("inapplicable", "E.extension<T>(T).M(int)").WithLocation(1, 11));

        src = """
using N;

Container.M();

static class E
{
    extension<T>(T t) where T : unmanaged
    {
        public static void M() => throw null; // applicable to arguments
    }
}

namespace N
{
    static class E2
    {
        extension<T>(T t)
        {
            public static void M() => throw null;
        }
    }
}
""";
        comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics();

        tree = comp.SyntaxTrees.Single();
        model = comp.GetSemanticModel(tree);
        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Container.M");
        AssertEx.Equal("void N.E2.<G>$8048A6C8BE30A622530249B904B537EB<Container>.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());

        src = """
using N;

new Container().M();

static class E
{
    public static void M<T>(this T t) where T : unmanaged { }
}

namespace N
{
    static class E2
    {
        public static void M<T>(this T t) { }
    }
}
""";
        // Expecting an error to be reported since we're not able to check whether the constraint is violated
        // Tracked by https://github.com/dotnet/roslyn/issues/77407
        comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void GetCompatibleExtension_Constraint_UseSiteInfo_03()
    {
        var missingSrc = """
public struct Missing { public int i; }
""";
        var missingRef = CreateCompilation(missingSrc, assemblyName: "missing").EmitToImageReference();

        var containerSrc = """
public struct Container { public Missing field; }
""";
        var containerRef = CreateCompilation(containerSrc, references: [missingRef]).EmitToImageReference();

        var src = """
int.M(new Container());

static class E
{
    extension(int)
    {
        public static void M<T>(T t) where T : unmanaged { }
    }
}
""";
        var comp = CreateCompilation(src, references: [containerRef]);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS8377: The type 'Container' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'E.extension(int).M<T>(T)'
            // int.M(new Container());
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("E.extension(int).M<T>(T)", "T", "Container").WithLocation(1, 5),
            // (1,5): error CS0012: The type 'Missing' is defined in an assembly that is not referenced. You must add a reference to assembly 'missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
            // int.M(new Container());
            Diagnostic(ErrorCode.ERR_NoTypeDef, "M").WithArguments("Missing", "missing, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(1, 5));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "int.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
    }

    [Fact]
    public void InstancePropertyAccess_Simple()
    {
        var src = """
System.Console.Write(new object().P);

public static class Extensions
{
    extension(object o)
    {
        public int P => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().P");
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
    }

    [Fact]
    public void InstancePropertyAccess_StaticExtensionProperty()
    {
        var src = """
System.Console.Write(new object().P);

public static class Extensions
{
    extension(object o)
    {
        public static int P => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,22): error CS9286: 'object' does not contain a definition for 'P' and no accessible extension member 'P' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // System.Console.Write(new object().P);
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "new object().P").WithArguments("object", "P").WithLocation(1, 22));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().P");
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
    }

    [Fact]
    public void InstancePropertyAccess_Invoked()
    {
        var src = """
new object().P();

public static class Extensions
{
    extension(object o)
    {
        public int P => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS1061: 'object' does not contain a definition for 'P' and no accessible extension method 'P' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // new object().P();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "P").WithArguments("object", "P").WithLocation(1, 14));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "new object().P()");
        Assert.Null(model.GetSymbolInfo(invocation).Symbol);
        Assert.Equal([], model.GetSymbolInfo(invocation).CandidateSymbols.ToTestDisplayStrings());

        AssertEx.SequenceEqual(["System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.P { get; }"], model.GetMemberGroup(invocation.Expression).ToTestDisplayStrings());
    }

    [Fact]
    public void InstancePropertyAccess_Invoked_Invocable()
    {
        var src = """
new object().P();

public static class Extensions
{
    extension(object o)
    {
        public System.Action P { get { return () => { System.Console.Write("ran"); }; } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().P");
        AssertEx.Equal("System.Action Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void StaticMethodInvocation_Simple()
    {
        var src = """
object.M();

public static class Extensions
{
    extension(object)
    {
        public static int M() => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics();

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var invocation = GetSyntax<InvocationExpressionSyntax>(tree, "object.M()");
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(invocation).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(invocation).CandidateSymbols.ToTestDisplayStrings());
    }

    [Fact]
    public void StaticMethodInvocation_TypeArguments()
    {
        var source = """
C.M<object>(42);

class C { }

static class E
{
    extension(C)
    {
        public static void M(int i) => throw null;
        public static void M<T>(int i) { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.M<object>");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<System.Object>(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M<System.Object>(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Theory, CombinatorialData]
    public void StaticMethodInvocation_Ambiguity_Method(bool e1BeforeE2)
    {
        var e1 = """
static class E1
{
    extension(object)
    {
        public static void Method() => throw null;
    }
}
""";

        var e2 = """
static class E2
{
    extension(object)
    {
        public static void Method() => throw null;
    }
}
""";

        var src = $$"""
object.Method();

{{(e1BeforeE2 ? e1 : e2)}}
{{(e1BeforeE2 ? e2 : e1)}}
""";
        var comp = CreateCompilation(src);
        if (!e1BeforeE2)
        {
            comp.VerifyEmitDiagnostics(
                // (1,8): error CS0121: The call is ambiguous between the following methods or properties: 'E2.extension(object).Method()' and 'E1.extension(object).Method()'
                // object.Method();
                Diagnostic(ErrorCode.ERR_AmbigCall, "Method").WithArguments("E2.extension(object).Method()", "E1.extension(object).Method()").WithLocation(1, 8));
        }
        else
        {
            comp.VerifyEmitDiagnostics(
                // (1,8): error CS0121: The call is ambiguous between the following methods or properties: 'E1.extension(object).Method()' and 'E2.extension(object).Method()'
                // object.Method();
                Diagnostic(ErrorCode.ERR_AmbigCall, "Method").WithArguments("E1.extension(object).Method()", "E2.extension(object).Method()").WithLocation(1, 8));
        }
    }

    [Fact]
    public void StaticPropertyAccess_InstanceExtensionProperty()
    {
        var src = """
System.Console.Write(new object().P);

public static class Extensions
{
    extension(object o)
    {
        public static int P => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,22): error CS9286: 'object' does not contain a definition for 'P' and no accessible extension member 'P' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // System.Console.Write(new object().P);
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "new object().P").WithArguments("object", "P").WithLocation(1, 22));

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().P");
        AssertEx.Equal("System.Int32 Extensions.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.P { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_ExtensionMethod()
    {
        var source = """
bool b = true;
var x = b ? object.M : object.M;

static class E
{
    extension(object o)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (2,9): error CS0173: Type of conditional expression cannot be determined because there is no implicit conversion between 'method group' and 'method group'
            // var x = b ? object.M : object.M;
            Diagnostic(ErrorCode.ERR_InvalidQM, "b ? object.M : object.M").WithArguments("method group", "method group").WithLocation(2, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").ToArray();
        Assert.Null(model.GetSymbolInfo(memberAccess[0]).Symbol);
        Assert.Null(model.GetSymbolInfo(memberAccess[1]).Symbol);

        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess[0]).ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess[1]).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_ExtensionProperty()
    {
        var source = """
bool b = true;
var x = b ? object.StaticProperty : object.StaticProperty;
System.Console.Write(x);

static class E
{
    extension(object o)
    {
        public static int StaticProperty => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.StaticProperty").ToArray();
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess[0]).Symbol.ToTestDisplayString());
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess[1]).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_DifferentTypes()
    {
        var source = """
bool b = true;
var x = b ? object.StaticProperty : object.StaticProperty2;
System.Console.Write(x.ToString());

static class E
{
    extension(object o)
    {
        public static int StaticProperty => 42;
        public static long StaticProperty2 => 43;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.StaticProperty");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess).Type.ToTestDisplayString());
        AssertEx.Equal("System.Int64", model.GetTypeInfo(memberAccess).ConvertedType.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_WithTargetType()
    {
        var source = """
bool b = true;
long x = b ? object.StaticProperty : object.StaticProperty;
System.Console.Write(x.ToString());

static class E
{
    extension(object o)
    {
        public static int StaticProperty => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.StaticProperty").ToArray();
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess[0]).Symbol.ToTestDisplayString());
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess[1]).Symbol.ToTestDisplayString());

        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess[0]).Type.ToTestDisplayString());
        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess[0]).ConvertedType.ToTestDisplayString());

        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess[1]).Type.ToTestDisplayString());
        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess[1]).ConvertedType.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_TwoExtensions_WithTargetType()
    {
        var source = """
bool b = true;
string x = b ? D.f : D.f;
System.Console.Write(x);

class D { }

static class E1
{
    extension(D)
    {
        public static string f => "ran";
    }
}

static class E2
{
    extension(object o)
    {
        public static void f() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (2,16): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // string x = b ? D.f : D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(2, 16),
            // (2,22): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // string x = b ? D.f : D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(2, 22));
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_TwoExtensions_WithTargetDelegateType()
    {
        var source = """
bool b = true;
System.Action x = b ? D.f : D.f;
System.Console.Write(x);

class D { }

static class E
{
    extension(D)
    {
        public static string f => null;
    }
}

static class E2
{
    extension(object o)
    {
        public static void f() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (2,23): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // System.Action x = b ? D.f : D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(2, 23),
            // (2,29): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // System.Action x = b ? D.f : D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(2, 29));
    }

    [Fact]
    public void ResolveAll_Cast_Static_Operand()
    {
        var source = """
var x = (long)object.StaticProperty;
System.Console.Write(x.ToString());

static class E
{
    extension(object o)
    {
        public static int StaticProperty => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.StaticProperty");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess).Type.ToTestDisplayString());
        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess).ConvertedType.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Cast_Static_Operand_TwoExtensions()
    {
        var source = """
var x = (string)D.f;
System.Console.Write(x);

class D { }

static class E1
{
    extension(D)
    {
        public static string f => "ran";
    }
}

static class E2
{
    extension(object)
    {
        public static void f() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // var x = (string)D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(1, 17));
    }

    [Fact]
    public void ResolveAll_Cast_Static_Operand_TwoExtensions_DelegateType()
    {
        var source = """
var x = (System.Action)D.f;
System.Action a = D.f;

class D { }

static class E1
{
    extension(D)
    {
        public static string f => null;
    }
}

static class E2
{
    extension(object)
    {
        public static void f() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,24): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // var x = (System.Action)D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(1, 24),
            // (2,19): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // System.Action a = D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(2, 19));

        // Note: a conversion to a delegate type does not provide invocation context for resolving the member access
        source = """
var x = (System.Action)D.f;
System.Action a = D.f;

class C
{
    public static void f() { }
}

class D : C
{
    public static new string f => null!;
}
""";
        comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,9): error CS0030: Cannot convert type 'string' to 'System.Action'
            // var x = (System.Action)D.f;
            Diagnostic(ErrorCode.ERR_NoExplicitConv, "(System.Action)D.f").WithArguments("string", "System.Action").WithLocation(1, 9),
            // (2,19): error CS0029: Cannot implicitly convert type 'string' to 'System.Action'
            // System.Action a = D.f;
            Diagnostic(ErrorCode.ERR_NoImplicitConv, "D.f").WithArguments("string", "System.Action").WithLocation(2, 19));
    }

    [Fact]
    public void ResolveAll_MethodTypeInference()
    {
        var source = """
write(object.M);
void write<T>(T t) { System.Console.Write(t.ToString()); }

static class E
{
    extension(object)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess).Type.ToTestDisplayString());
        AssertEx.Equal("System.Int32", model.GetTypeInfo(memberAccess).ConvertedType.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ArrayCreation_Initializer_Static()
    {
        var source = """
var x = new[] { object.StaticProperty, object.StaticProperty };
System.Console.Write((x[0], x[1]));

static class E
{
    extension(object)
    {
        public static int StaticProperty => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "(42, 42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.StaticProperty").ToArray();
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess[0]).Symbol.ToTestDisplayString());
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess[1]).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ArrayCreation_Rank()
    {
        var source = """
var x = new object[object.StaticProperty];
System.Console.Write(x.Length.ToString());

static class E
{
    extension(object o)
    {
        public static int StaticProperty => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.StaticProperty");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.StaticProperty { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Deconstruction_Declaration()
    {
        var source = """
var (x, y) = object.M;
System.Console.Write((x, y));

static class E
{
    extension(object)
    {
        public static (int, int) M => (42, 43);
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("(System.Int32, System.Int32) E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Deconstruction_Assignment()
    {
        var source = """
int x, y;
(x, y) = object.M;
System.Console.Write((x, y));

static class E
{
    extension(object o)
    {
        public static (int, int) M => (42, 43);
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("(System.Int32, System.Int32) E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_TupleExpression()
    {
        var source = """
System.Console.Write((object.M, object.M));

static class E
{
    extension(object o)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "(42, 42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").ToArray();
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess[0]).Symbol.ToTestDisplayString());
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess[1]).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_CollectionExpression()
    {
        var source = """
int[] x = [object.M];
System.Console.Write(x[0].ToString());

static class E
{
    extension(object o)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_CollectionExpression_ExtensionAddMethod()
    {
        var source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

static class E
{
    extension(MyCollection c)
    {
        public void Add(int i) { System.Console.Write("ran"); }
    }
}

public class MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();
    }

    [Fact]
    public void ResolveAll_CollectionExpression_ExtensionAddMethod_RefReceiverParameter()
    {
        // The receiver argument gets an implicit `ref` when the parameter is `ref`
        var source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];
System.Console.Write(c.field);

static class E
{
    extension(ref MyCollection c)
    {
        public void Add(int i) { System.Console.Write("ran "); c = new MyCollection() { field = i }; }
    }
}

public struct MyCollection : IEnumerable<int>
{
    public int field;
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran 42").VerifyDiagnostics();
    }

    [Fact]
    public void ResolveAll_CollectionExpression_ExtensionAddMethod_InReceiverParameter()
    {
        var source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

static class E
{
    extension(in MyCollection c)
    {
        public void Add(int i) { System.Console.Write("ran"); }
    }
}

public struct MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();
    }

    [Fact]
    public void ResolveAll_CollectionExpression_ExtensionAddMethod_RefParameter()
    {
        var source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

static class E
{
    extension(MyCollection c)
    {
        public void Add(ref int i) { }
    }
}

public class MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (4,18): error CS1954: The best overloaded method match 'E.extension(MyCollection).Add(ref int)' for the collection initializer element cannot be used. Collection initializer 'Add' methods cannot have ref or out parameters.
            // MyCollection c = [42];
            Diagnostic(ErrorCode.ERR_InitializerAddHasParamModifiers, "[42]").WithArguments("E.extension(MyCollection).Add(ref int)").WithLocation(4, 18));
    }

    [Fact]
    public void ResolveAll_CollectionExpression_ExtensionAdd_DelegateTypeProperty()
    {
        var source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

static class E
{
    extension(MyCollection c)
    {
        public System.Action<int> Add => (int i) => { };
    }
}

public class MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (4,18): error CS1061: 'MyCollection' does not contain a definition for 'Add' and no accessible extension method 'Add' accepting a first argument of type 'MyCollection' could be found (are you missing a using directive or an assembly reference?)
            // MyCollection c = [42];
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "[42]").WithArguments("MyCollection", "Add").WithLocation(4, 18)
            );

        source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

public class MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
    public System.Action<int> Add => (int i) => { };
}
""";
        comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (4,18): error CS0118: 'Add' is a property but is used like a method
            // MyCollection c = [42];
            Diagnostic(ErrorCode.ERR_BadSKknown, "[42]").WithArguments("Add", "property", "method").WithLocation(4, 18));
    }

    [Fact]
    public void ResolveAll_CollectionExpression_ExtensionAdd_DynamicTypeProperty()
    {
        var source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

static class E
{
    extension(MyCollection c)
    {
        public dynamic Add => throw null;
    }
}

public class MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (4,18): error CS1061: 'MyCollection' does not contain a definition for 'Add' and no accessible extension method 'Add' accepting a first argument of type 'MyCollection' could be found (are you missing a using directive or an assembly reference?)
            // MyCollection c = [42];
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "[42]").WithArguments("MyCollection", "Add").WithLocation(4, 18));

        source = """
using System.Collections;
using System.Collections.Generic;

MyCollection c = [42];

public class MyCollection : IEnumerable<int>
{
    IEnumerator<int> IEnumerable<int>.GetEnumerator() => throw null;
    IEnumerator IEnumerable.GetEnumerator() => throw null;
    public dynamic Add => throw null;
}
""";
        comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (4,18): error CS0118: 'Add' is a property but is used like a method
            // MyCollection c = [42];
            Diagnostic(ErrorCode.ERR_BadSKknown, "[42]").WithArguments("Add", "property", "method").WithLocation(4, 18));
    }

    [Fact]
    public void ResolveAll_Initializer_Property()
    {
        var source = """
var x = new System.Collections.Generic.List<int>() { object.M };
System.Console.Write(x[0]);

static class E
{
    extension(object o)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Initializer_Method()
    {
        var source = """
var x = new System.Collections.Generic.List<int>() { object.M };

static class E
{
    extension(object o)
    {
        public static void M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,54): error CS1950: The best overloaded Add method 'List<int>.Add(int)' for the collection initializer has some invalid arguments
            // var x = new System.Collections.Generic.List<int>() { object.M };
            Diagnostic(ErrorCode.ERR_BadArgTypesForCollectionAdd, "object.M").WithArguments("System.Collections.Generic.List<int>.Add(int)").WithLocation(1, 54),
            // (1,54): error CS1503: Argument 1: cannot convert from 'method group' to 'int'
            // var x = new System.Collections.Generic.List<int>() { object.M };
            Diagnostic(ErrorCode.ERR_BadArgType, "object.M").WithArguments("1", "method group", "int").WithLocation(1, 54));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.OverloadResolutionFailure, model.GetSymbolInfo(memberAccess).CandidateReason);
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Initializer_ObjectInitializer()
    {
        var source = """
var x = new C() { f = object.M };
System.Console.Write(x.f.ToString());

class C
{
    public int f;
}

static class E
{
    extension(object o)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ConditionalAccess_Receiver()
    {
        var source = """
System.Console.Write(object.M?.ToString());

static class E
{
    extension(object o)
    {
        public static string M => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ConditionalAccess_WhenNotNull_Property()
    {
        var source = """
var x = new object()?.M;
System.Console.Write(x.ToString());

static class E
{
    extension(object o)
    {
        public string M => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberBinding = GetSyntax<MemberBindingExpressionSyntax>(tree, ".M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberBinding).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ConditionalAccess_WhenNotNull_Invocation()
    {
        var source = """
var x = new object()?.M();
System.Console.Write(x.ToString());

static class E
{
    extension(object o)
    {
        public string M() => "ran";
        public string M(int i) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberBinding = GetSyntax<MemberBindingExpressionSyntax>(tree, ".M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberBinding).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_CompoundAssignment_Left()
    {
        var source = """
object.M += 41;
System.Console.Write(E.M.ToString());

static class E
{
    extension(object o)
    {
        public static int M { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; set; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_CompoundAssignment_Right()
    {
        var source = """
int x = 1;
x += object.M;
System.Console.Write(x.ToString());

static class E
{
    extension(object o)
    {
        public static int M => 41;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_BinaryOperator_UserDefinedOperator()
    {
        var source = """
var x = object.M + object.M;
System.Console.Write(x.ToString());

public class C
{
    public static int operator+(C c1, C c2) => 42;
}

static class E
{
    extension(object o)
    {
        public static C M => new C();
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").ToArray();
        AssertEx.Equal("C E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess[0]).Symbol.ToTestDisplayString());
        AssertEx.Equal("C E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess[1]).Symbol.ToTestDisplayString());

        var binaryOp = GetSyntax<BinaryExpressionSyntax>(tree, "object.M + object.M");
        AssertEx.Equal("System.Int32 C.op_Addition(C c1, C c2)", model.GetSymbolInfo(binaryOp).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_BinaryOperator_NoUserDefinedOperator()
    {
        var source = """
var x = object.M + object.M;
System.Console.Write(x.ToString());

public class C { }

static class E
{
    extension(object o)
    {
        public static C M => new C();
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,9): error CS0019: Operator '+' cannot be applied to operands of type 'C' and 'C'
            // var x = object.M + object.M;
            Diagnostic(ErrorCode.ERR_BadBinaryOps, "object.M + object.M").WithArguments("+", "C", "C").WithLocation(1, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").ToArray();
        AssertEx.Equal("C E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess[0]).Symbol.ToTestDisplayString());
        AssertEx.Equal("C E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess[1]).Symbol.ToTestDisplayString());

        var binaryOp = GetSyntax<BinaryExpressionSyntax>(tree, "object.M + object.M");
        Assert.Null(model.GetSymbolInfo(binaryOp).Symbol);
    }

    [Fact]
    public void ResolveAll_IncrementOperator()
    {
        var source = """
object.M++;

public class C { }

static class E
{
    extension(object o)
    {
        public static int M { get { System.Console.Write("get "); return 41; } set { System.Console.Write($"set({value}) "); } }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "get set(42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; set; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        var unaryOp = GetSyntax<PostfixUnaryExpressionSyntax>(tree, "object.M++");
        AssertEx.Equal("System.Int32 System.Int32.op_Increment(System.Int32 value)", model.GetSymbolInfo(unaryOp).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_UnaryOperator()
    {
        var source = """
_ = !object.M;

public class C { }

static class E
{
    extension(object o)
    {
        public static bool M { get { System.Console.Write("ran"); return true; } }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Boolean E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        var unaryOp = GetSyntax<PrefixUnaryExpressionSyntax>(tree, "!object.M");
        AssertEx.Equal("System.Boolean System.Boolean.op_LogicalNot(System.Boolean value)",
            model.GetSymbolInfo(unaryOp).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_NullCoalescingOperator()
    {
        var source = """
var x = object.M ?? object.M2;
System.Console.Write(x);

static class E
{
    extension(object o)
    {
        public static string M => null;
        public static string M2 => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_NullCoalescingAssignmentOperator()
    {
        var source = """
object.M ??= object.M2;

static class E
{
    extension(object o)
    {
        public static string M { get { System.Console.Write("get "); return null; }  set { System.Console.Write($"set({value}) "); } }
        public static string M2 => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "get set(ran)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Query_Select()
    {
        var source = """
using System.Linq;

int[] array = [1];
var r = from int i in array select object.M;
foreach (var x in r)
{
    System.Console.Write(x.ToString());
}

static class E
{
    extension(object o)
    {
        public static string M => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Query_Where_DelegateTypeProperty()
    {
        var src = """
var x = from i in new C()
        where i is not null
        select i;

System.Console.Write(x);

public class C { }

public static class E
{
    extension(C c)
    {
        public System.Func<System.Func<C, bool>, C> Where => (System.Func<C, bool> f) => { System.Console.Write(f(c)); return c; };
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "TrueC").VerifyDiagnostics();

        src = """
var x = from i in new C()
        where i is not null
        select i;

System.Console.Write(x);

public class C
{
    public System.Func<System.Func<C, bool>, C> Where => (System.Func<C, bool> f) => { System.Console.Write(f(this)); return this; };
}
""";
        comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "TrueC").VerifyDiagnostics();
    }

    [Fact]
    public void ResolveAll_Query_Where_DynamicTypeProperty()
    {
        var src = """
var x = from i in new C()
        where i is not null
        select i;

public class C { }

public static class E
{
    extension(C c)
    {
        public dynamic Where => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (2,15): error CS1977: Cannot use a lambda expression as an argument to a dynamically dispatched operation without first casting it to a delegate or expression tree type.
            //         where i is not null
            Diagnostic(ErrorCode.ERR_BadDynamicMethodArgLambda, "i is not null").WithLocation(2, 15));

        src = """
var x = from i in new C()
        where i is not null
        select i;

public class C
{
    public dynamic Where => throw null;
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (2,9): error CS1979: Query expressions over source type 'dynamic' or with a join sequence of type 'dynamic' are not allowed
            //         where i is not null
            Diagnostic(ErrorCode.ERR_BadDynamicQuery, "where i is not null").WithLocation(2, 9));
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/80008")]
    public void ResolveAll_Query_Cast_StaticProperty()
    {
        var source = """
using System.Linq;

var r = from string s in object.M from string s2 in object.M2 select s.ToString();
foreach (var x in r)
{
    System.Console.Write(x.ToString());
}

static class E
{
    extension(object o)
    {
        public static object[] M => ["ran"];
        public static object[] M2 => [""];
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Object[] E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }",
            model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.Equal("System.Object[] E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }",
            model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/80008")]
    public void ResolveAll_Query_Cast_InstanceProperty()
    {
        var source = """
using System.Linq;

var o = new object();
var r = from string s in o.M from string s2 in o.M2 select s.ToString();
foreach (var x in r)
{
    System.Console.Write(x.ToString());
}

static class E
{
    extension(object o)
    {
        public object[] M => ["ran"];
        public object[] M2 => [""];
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "o.M");
        AssertEx.Equal("System.Object[] E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }",
            model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "o.M2");
        AssertEx.Equal("System.Object[] E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }",
            model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Return_Lambda()
    {
        var source = """
var x = () =>
    {
        bool b = true;
        if (b)
            return object.M;
        else
            return object.M2;
    };
System.Console.Write(x().ToString());

static class E
{
    extension(object o)
    {
        public static int M => 42;
        public static int M2 => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_ExpressionBodiedLambda()
    {
        var source = """
var x = () => object.M;
System.Console.Write(x().ToString());

static class E
{
    extension(object o)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_YieldReturn()
    {
        var source = """
foreach (var y in local())
{
    System.Console.Write(y.ToString());
}

System.Collections.Generic.IEnumerable<int> local()
{
    bool b = true;
    if (b)
        yield return object.M;
    else
        yield return object.M2;
}

static class E
{
    extension(object o)
    {
        public static int M => 42;
        public static int M2 => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_YieldReturn_Lambda()
    {
        var source = """
var x = System.Collections.Generic.IEnumerable<int> () =>
    {
        bool b = true;
        if (b)
            yield return object.M;
        else
            yield return object.M2;
    };

foreach (var y in x())
{
    System.Console.Write(y.ToString());
}

static class E
{
    extension(object o)
    {
        public static int M => 42;
        public static int M2 => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,56): error CS1643: Not all code paths return a value in lambda expression of type 'Func<IEnumerable<int>>'
            // var x = System.Collections.Generic.IEnumerable<int> () =>
            Diagnostic(ErrorCode.ERR_AnonymousReturnExpected, "=>").WithArguments("lambda expression", "System.Func<System.Collections.Generic.IEnumerable<int>>").WithLocation(1, 56),
            // (5,13): error CS1621: The yield statement cannot be used inside an anonymous method or lambda expression
            //             yield return object.M;
            Diagnostic(ErrorCode.ERR_YieldInAnonMeth, "yield").WithLocation(5, 13),
            // (7,13): error CS1621: The yield statement cannot be used inside an anonymous method or lambda expression
            //             yield return object.M2;
            Diagnostic(ErrorCode.ERR_YieldInAnonMeth, "yield").WithLocation(7, 13));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2 { get; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Throw()
    {
        var source = """
try
{
    throw object.M;
}
catch (System.Exception e)
{
    System.Console.Write(e.Message);
}

static class E
{
    extension(object o)
    {
        public static System.Exception M => new System.Exception("ran");
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.Exception E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_FieldInitializer()
    {
        var source = """
System.Console.Write(C.field.ToString());

class C
{
    public static string field = object.M;
}

static class E
{
    extension(object o)
    {
        public static string M => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Invocation_Static()
    {
        var src = """
local(object.M);

void local(string s)
{
    System.Console.Write(s);
}

static class E
{
    extension(object o)
    {
        public static string M => "ran";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").First();
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Invocation_Static_DelegateTypeParameter()
    {
        var src = """
local(object.M);

void local(System.Func<string> d)
{
    System.Console.Write(d());
}

static class E
{
    extension(object o)
    {
        public static string M() => "ran";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").First();
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Invocation_Static_Inferred()
    {
        var src = """
System.Console.Write(local(object.M));

T local<T>(T t)
{
    return t;
}

static class E
{
    extension(object o)
    {
        public static string M => "ran";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.M").First();
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Invocation_Static_DelegateTypeParameter_InapplicableInstanceMember()
    {
        var src = """
local(object.ToString);

void local(System.Func<int, string> d)
{
    System.Console.Write(d(42));
}

static class E
{
    extension(object o)
    {
        public static string ToString(int i) => "ran";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.ToString").First();
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.ToString(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Invocation_Static_DelegateTypeParameter_PropertyAndMethod()
    {
        var src = """
var o = new object();
C.M(o.Member);

class C
{
    public static void M(System.Action a) { a(); }
}

static class E1
{
    extension(object o)
    {
        public string Member => throw null;
    }
}

public static class E2
{
    public static void Member(this object o) => throw null;
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (2,5): error CS9286: 'object' does not contain a definition for 'Member' and no accessible extension member 'Member' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // C.M(o.Member);
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "o.Member").WithArguments("object", "Member").WithLocation(2, 5));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "o.Member");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        AssertEx.SequenceEqual(["System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Member { get; }", "void E2.Member(this System.Object o)"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_ObjectCreation_Static()
    {
        var source = """
new C(object.M);

class C
{
    public C(string s) { System.Console.Write(s); }
}

static class E
{
    extension(object o)
    {
        public static string M => "ran";
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_BinaryOperator_Static_TwoExtensions()
    {
        var src = """
bool b = D.f + D.f;

class C
{
    public static bool operator +(C c, System.Action a) => true;
}

class D { }

static class E1
{
    extension(D d)
    {
        public static C f => null;
    }
}

static class E2
{
    extension(object o)
    {
        public static void f() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,10): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // bool b = D.f + D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(1, 10),
            // (1,16): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // bool b = D.f + D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(1, 16));
    }

    [Fact]
    public void ResolveAll_Lambda_Static_TwoAsGoodExtensions_LambdaConverted()
    {
        var src = """
System.Func<System.Action> l = () => object.f;

static class E1
{
    extension(object o)
    {
        public static string f => null;
    }
}

static class E2
{
    extension(object o)
    {
        public static void f() { System.Console.Write("ran"); }
    }
}
""";
        // Tracked by https://github.com/dotnet/roslyn/issues/78830 : diagnostic quality, the diagnostic should describe what went wrong
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,38): error CS9286: 'object' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // System.Func<System.Action> l = () => object.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "object.f").WithArguments("object", "f").WithLocation(1, 38)
            );

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.f").First();
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.f { get; }", "void E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.f()"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Lambda_Instance_ExtensionMethodVsExtensionMember()
    {
        var src = """
System.Func<System.Action> lambda = () => new object().Member;

static class E
{
    extension(object o)
    {
        public string Member => throw null;
    }

    public static void Member(this object o) => throw null;
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,43): error CS9286: 'object' does not contain a definition for 'Member' and no accessible extension member 'Member' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // System.Func<System.Action> lambda = () => new object().Member;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "new object().Member").WithArguments("object", "Member").WithLocation(1, 43));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Member");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
    }

    [Fact]
    public void ResolveAll_Lambda_Instance_MethodGroupWithMultipleOverloads()
    {
        var src = """
System.Func<System.Action> lambda = () => new object().Member;
lambda()();

static class E
{
    extension(object o)
    {
        public void Member() { System.Console.Write("ran"); }
        public void Member(int i) => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().Member");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Lambda_Static_TwoExtensions_ConversionToDelegateType_ExplicitReturnType()
    {
        var src = """
var l = System.Action () => D.f;

class D { }

static class E1
{
    extension(object o)
    {
        public static string f => null;
    }
}
static class E2
{
    extension(object o)
    {
        public static void f() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,29): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // var l = System.Action () => D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(1, 29)
            );
    }

    [Fact]
    public void ResolveAll_Lambda_Static_TwoExtensions_ConversionToDelegateType()
    {
        var src = """
System.Func<System.Action> l = () => D.f;

class D { }

static class E1
{
    extension(D)
    {
        public static string f => null;
    }
}

static class E2
{
    extension(object)
    {
        public static void f() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,38): error CS9286: 'D' does not contain a definition for 'f' and no accessible extension member 'f' for receiver of type 'D' could be found (are you missing a using directive or an assembly reference?)
            // System.Func<System.Action> l = () => D.f;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "D.f").WithArguments("D", "f").WithLocation(1, 38)
            );
    }

    [Fact]
    public void ResolveAll_SwitchExpression_Static_Default()
    {
        var src = """
bool b = true;
var s = b switch { true => object.f, false => default };
System.Console.Write(s);

static class E
{
    extension(object)
    {
        public static string f => "hi";
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "hi").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.f").First();
        AssertEx.Equal("System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.f { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());

        var defaultExpr = GetSyntax<LiteralExpressionSyntax>(tree, "default");
        AssertEx.Equal("System.String", model.GetTypeInfo(defaultExpr).Type.ToTestDisplayString());
        AssertEx.Equal("System.String", model.GetTypeInfo(defaultExpr).ConvertedType.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_RefTernary()
    {
        var src = """
bool b = true;
string s1 = "ran";
string s2 = null;

var x = b ? ref s1.f : ref s2.f;
System.Console.Write(x);

static class E
{
    extension(ref string s)
    {
        public ref string f => ref s;
    }

    public static ref string M(this ref string s) => ref s;
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (10,19): error CS9300: The 'ref' receiver parameter of an extension block must be a value type or a generic type constrained to struct.
            //     extension(ref string s)
            Diagnostic(ErrorCode.ERR_RefExtensionParameterMustBeValueTypeOrConstrainedToOne, "string").WithLocation(10, 19),
            // (15,30): error CS8337: The first parameter of a 'ref' extension method 'M' must be a value type or a generic type constrained to struct.
            //     public static ref string M(this ref string s) => ref s;
            Diagnostic(ErrorCode.ERR_RefExtensionMustBeValueTypeOrConstrainedToOne, "M").WithArguments("M").WithLocation(15, 30));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "s1.f");
        AssertEx.Equal("ref System.String E.<G>$34505F560D9EACF86A87F3ED1F85E448.f { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ResolveAll_Query_Static_InstanceMethodGroup()
    {
        var src = """
string query = from x in object.ToString select x;

static class E
{
    extension(object)
    {
        public static string ToString() => null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,33): error CS0119: 'object.ToString()' is a method, which is not valid in the given context
            // string query = from x in object.ToString select x;
            Diagnostic(ErrorCode.ERR_BadSKunknown, "ToString").WithArguments("object.ToString()", "method").WithLocation(1, 33));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.ToString");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["System.String System.Object.ToString()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Query_Static_ExtensionMethodGroup()
    {
        var src = """
string query = from x in object.M select x;

static class E
{
    extension(object o)
    {
        public static string M() => null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,33): error CS0119: 'E.extension(object).M()' is a method, which is not valid in the given context
            // string query = from x in object.M select x;
            Diagnostic(ErrorCode.ERR_BadSKunknown, "M").WithArguments("E.extension(object).M()", "method").WithLocation(1, 33));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Empty(model.GetMemberGroup(memberAccess)); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : consider handling BoundBadExpression better
    }

    [Fact]
    public void ResolveAll_Instance_Invocation_InnerInapplicableExtensionMethodVsOuterInvocableExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            new object().M();
        }
    }

    public static class Extension
    {
        public static void M(this object o, int i) { } // not applicable because of second parameter
    }
}

static class E
{
    extension(object o)
    {
        public System.Action M => () => { System.Console.Write("ran"); };
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("System.Action E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void ResolveAll_Instance_Invocation_InnerIrrelevantExtensionMethodVsOuterInvocableExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            new object().M();
        }
    }

    public static class Extension
    {
        public static void M(this string o, int i) { } // not eligible because of `this` parameter
    }
}

static class E
{
    extension(object o)
    {
        public System.Action M => () => { System.Console.Write("ran"); };
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("System.Action E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void ResolveAll_Instance_InferredVariable_InnerExtensionMethodVsOuterInvocableExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            var x = new object().M;
            x(42);
        }
    }

    public static class Extension
    {
        public static void M(this object o, int i) { System.Console.Write("ran"); }
    }
}

static class E
{
    extension(object o)
    {
        public System.Action M => () => throw null;
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void System.Object.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void System.Object.M(System.Int32 i)", "System.Action E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Instance_LocalDeclaration_InnerExtensionMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            int x = new object().M;
        }
    }

    public static class Extension
    {
        public static void M(this object o, int i) { }
    }
}

static class E
{
    extension(object o)
    {
        public int M => 42;
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        comp.VerifyEmitDiagnostics(
            // (7,34): error CS0428: Cannot convert method group 'M' to non-delegate type 'int'. Did you intend to invoke the method?
            //             int x = new object().M;
            Diagnostic(ErrorCode.ERR_MethGrpToNonDel, "M").WithArguments("M", "int").WithLocation(7, 34));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void System.Object.M(System.Int32 i)", "System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Static_LocalDeclaration_InnerExtensionMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            int x = object.M;
        }
    }

    public static class Extension
    {
        public static void M(this object o, int i) { }
    }
}

static class E
{
    extension(object o)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        comp.VerifyEmitDiagnostics(
            // (7,28): error CS0428: Cannot convert method group 'M' to non-delegate type 'int'. Did you intend to invoke the method?
            //             int x = object.M;
            Diagnostic(ErrorCode.ERR_MethGrpToNonDel, "M").WithArguments("M", "int").WithLocation(7, 28));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void System.Object.M(System.Int32 i)", "System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Static_LocalDeclaration_InstanceInnerExtensionTypeMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            int x = object.M;
        }
    }

    static class E1
    {
        extension(object o)
        {
            public void M(int i) { }
        }
    }
}

static class E2
{
    extension(object)
    {
        public static int M => 42;
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        comp.VerifyEmitDiagnostics(
            // (7,28): error CS0428: Cannot convert method group 'M' to non-delegate type 'int'. Did you intend to invoke the method?
            //             int x = object.M;
            Diagnostic(ErrorCode.ERR_MethGrpToNonDel, "M").WithArguments("M", "int").WithLocation(7, 28));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void N.E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(System.Int32 i)", "System.Int32 E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Instance_LocalDeclaration_StaticInnerExtensionTypeMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            int x = new object().M;
        }
    }

    static class E1
    {
        extension(object)
        {
            public static void M(int i) => throw null;
        }
    }
}

static class E2
{
    extension(object o)
    {
        public int M => 42;
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        comp.VerifyEmitDiagnostics(
            // (7,34): error CS0428: Cannot convert method group 'M' to non-delegate type 'int'. Did you intend to invoke the method?
            //             int x = new object().M;
            Diagnostic(ErrorCode.ERR_MethGrpToNonDel, "M").WithArguments("M", "int").WithLocation(7, 34));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void N.E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(System.Int32 i)", "System.Int32 E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Instance_LocalDeclaration_InnerIrrelevantExtensionMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            int x = new object().M;
            System.Console.Write(x);
        }
    }

    public static class Extension
    {
        public static void M(this string o, int i) { } // not eligible because of `this` parameter
    }
}

static class E
{
    extension(object o)
    {
        public int M => 42;
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void ResolveAll_Instance_LocalDeclaration_DelegateType_InnerInapplicableExtensionMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            System.Action x = new object().M;
            x();
        }
    }

    public static class Extension
    {
        public static void M(this object o, int i) { }
    }
}

static class E
{
    extension(object o)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void System.Object.M(System.Int32 i)", "void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_Instance_LocalDeclaration_DelegateType_InnerIrrelevantExtensionMethodVsOuterExtensionProperty()
    {
        var src = """
namespace N
{
    public class C
    {
        public static void Main()
        {
            System.Action x = new object().M;
            x();
        }
    }

    public static class Extension
    {
        public static void M(this string o, int i) { } // not eligible because of `this` parameter
    }
}

static class E
{
    extension(object o)
    {
        public void M() { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(src, options: TestOptions.DebugExe);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void ResolveAll_ConditionalOperator_Static_TwoAsGoodExtensions_Property()
    {
        var source = """
bool b = true;
var x = b ? object.StaticProperty : object.StaticProperty;

static class E1
{
    extension(object)
    {
        public static int StaticProperty => 42;
    }
}
static class E2
{
    extension(object)
    {
        public static int StaticProperty => 42;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (2,13): error CS9286: 'object' does not contain a definition for 'StaticProperty' and no accessible extension member 'StaticProperty' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // var x = b ? object.StaticProperty : object.StaticProperty;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "object.StaticProperty").WithArguments("object", "StaticProperty").WithLocation(2, 13),
            // (2,37): error CS9286: 'object' does not contain a definition for 'StaticProperty' and no accessible extension member 'StaticProperty' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // var x = b ? object.StaticProperty : object.StaticProperty;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "object.StaticProperty").WithArguments("object", "StaticProperty").WithLocation(2, 37));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.StaticProperty").ToArray();
        Assert.Null(model.GetSymbolInfo(memberAccess[0]).Symbol);
        Assert.Null(model.GetSymbolInfo(memberAccess[1]).Symbol);
    }

    [Fact]
    public void DelegateConversion_TypeReceiver()
    {
        var source = """
D d = C.M;
d(42);

delegate void D(int i);

class C { }

static class E
{
    extension(C)
    {
        public static void M(int i) { System.Console.Write($"E.M({i})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        var verifier = CompileAndVerify(comp, expectedOutput: "E.M(42)").VerifyDiagnostics();
        verifier.VerifyIL("<top-level-statements-entry-point>", """
{
  // Code size       35 (0x23)
  .maxstack  2
  IL_0000:  ldsfld     "D Program.<>O.<0>__M"
  IL_0005:  dup
  IL_0006:  brtrue.s   IL_001b
  IL_0008:  pop
  IL_0009:  ldnull
  IL_000a:  ldftn      "void E.M(int)"
  IL_0010:  newobj     "D..ctor(object, System.IntPtr)"
  IL_0015:  dup
  IL_0016:  stsfld     "D Program.<>O.<0>__M"
  IL_001b:  ldc.i4.s   42
  IL_001d:  callvirt   "void D.Invoke(int)"
  IL_0022:  ret
}
""");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Null(model.GetTypeInfo(memberAccess).Type);
        AssertEx.Equal("D", model.GetTypeInfo(memberAccess).ConvertedType.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        Assert.Equal(ConversionKind.MethodGroup, model.GetConversion(memberAccess).Kind);
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Creation()
    {
        var source = """
D d = new D(C.M);
d(42);

delegate void D(int i);

class C { }

static class E
{
    extension(C)
    {
        public static void M(int i) { System.Console.Write($"E.M({i})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "E.M(42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Null(model.GetTypeInfo(memberAccess).Type);
        AssertEx.Equal("D", model.GetTypeInfo(memberAccess).ConvertedType.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        Assert.Equal(ConversionKind.MethodGroup, model.GetConversion(memberAccess).Kind);
    }

    [Fact]
    public void DelegateConversion_InstanceReceiver_Creation()
    {
        var source = """
D d = new D(new C().M);
d(42);

delegate void D(int i);

class C { }

static class E
{
    extension(C c)
    {
        public void M(int i) { System.Console.Write($"E.M({i})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "E.M(42)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Null(model.GetTypeInfo(memberAccess).Type);
        AssertEx.Equal("D", model.GetTypeInfo(memberAccess).ConvertedType.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        Assert.Equal(ConversionKind.MethodGroup, model.GetConversion(memberAccess).Kind);
    }

    [Fact]
    public void DelegateConversion_InstanceReceiver()
    {
        var source = """
D d = new C(42).M;
d(43);

delegate void D(int i);

class C(int x)
{
    public int x = x;
    public void M() => throw null;
}

static class E
{
    extension(C c)
    {
        public void M(int i) { System.Console.Write($"{c.x}.M({i})"); }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics();
        var verifier = CompileAndVerify(comp, expectedOutput: "42.M(43)").VerifyDiagnostics();
        verifier.VerifyIL("<top-level-statements-entry-point>", """
{
  // Code size       26 (0x1a)
  .maxstack  2
  IL_0000:  ldc.i4.s   42
  IL_0002:  newobj     "C..ctor(int)"
  IL_0007:  ldftn      "void E.M(C, int)"
  IL_000d:  newobj     "D..ctor(object, System.IntPtr)"
  IL_0012:  ldc.i4.s   43
  IL_0014:  callvirt   "void D.Invoke(int)"
  IL_0019:  ret
}
""");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C(42).M");
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void C.M()", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
        Assert.Equal(ConversionKind.MethodGroup, model.GetConversion(memberAccess).Kind);
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Overloads()
    {
        var source = """
D d = C.M;
d(42);

C.M(42);

delegate void D(int i);

class C { }

static class E
{
    extension(C)
    {
        public static void M(int i) { System.Console.Write("ran "); }

        public static void M(string s) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "C.M").First();
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_ValueReceiver_Overloads()
    {
        var source = """
D d = new C().M;
d(42);

new C().M(42);

delegate void D(int i);

class C { }

static class E
{
    extension(C c)
    {
        public void M(int i) { System.Console.Write("ran "); }
        public void M(string s) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new C().M").First();
        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Overloads_DifferentExtensions()
    {
        var source = """
D d = C.M;
d(42);

C.M(42);

delegate void D(int i);

class C { }

static class E1
{
    extension(C)
    {
        public static void M(int i) { System.Console.Write("ran "); }
    }
}
static class E2
{
    extension(C)
    {
        public static void M(string s) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "C.M").First();
        AssertEx.Equal("void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_WrongSignature()
    {
        var source = """
D d = new C().M;
d(42);

new C().M(42);

delegate void D(int i);

class C { }

static class E
{
    extension(C c)
    {
        public void M(string s) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,15): error CS0123: No overload for 'M' matches delegate 'D'
            // D d = new C().M;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M").WithArguments("M", "D").WithLocation(1, 15),
            // (4,11): error CS1503: Argument 2: cannot convert from 'int' to 'string'
            // new C().M(42);
            Diagnostic(ErrorCode.ERR_BadArgType, "42").WithArguments("2", "int", "string").WithLocation(4, 11));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new C().M").First();
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_ZeroArityMatchesAny()
    {
        var source = """
D d = object.Method;
d("");

d = object.Method<string>;
d("");

delegate void D(string s);

static class E
{
    extension(object)
    {
        public static void Method(int i) => throw null;
        public static void Method<T>(T t) { System.Console.Write("Method "); }
        public static void Method<T1, T2>(T1 t1, T2 t2) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "Method Method").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.Method");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method<System.String>(System.String t)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method(System.Int32 i)", "void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method<T>(T t)", "void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method<T1, T2>(T1 t1, T2 t2)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_ValueReceiver_Overloads_OuterScope_WithInapplicableInstanceMember()
    {
        var source = """
using N;

D d = new C().M;
d(42);

new C().M(42);

delegate void D(int i);

class C
{
    public void M(char c) { }
}

namespace N
{
    static class E1
    {
        extension(C c)
        {
            public void M(int i)
            {
                System.Console.Write("ran ");
            }
        }
    }
}

static class E2
{
    extension(C c)
    {
        public void M(string s) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "new C().M").First();
        AssertEx.Equal("void N.E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void C.M(System.Char c)", "void E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.String s)", "void N.E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Overloads_InnerScope()
    {
        var source = """
using N;

D d = C.M;
d(42);

delegate void D(int i);

class C { }

static class E1
{
    extension(C)
    {
        public static void M(int i) { System.Console.Write("ran"); }
    }
}

namespace N
{
    static class E2
    {
        extension(C)
        {
            public static void M(int i) => throw null;
        }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // 0.cs(1,1): hidden CS8019: Unnecessary using directive.
            // using N;
            Diagnostic(ErrorCode.HDN_UnusedUsingDirective, "using N;").WithLocation(1, 1));

        CompileAndVerify(comp, expectedOutput: "ran");

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.M");
        AssertEx.Equal("void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E1.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)", "void N.E2.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_TypeArguments_01()
    {
        var source = """
D d = object.M<object>;
d(42);

delegate void D(int i);

static class E
{
    extension(object)
    {
        public static void M(int i) => throw null;
        public static void M<T>(int i) { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M<object>");
        AssertEx.Equal("void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M<System.Object>(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M<System.Object>(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_TypeArguments_02()
    {
        var source = """
D d = object.M<object, int>;
d(42);

delegate void D(int i);

static class E
{
    extension(object)
    {
        public static void M<T>(T t) { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS0117: 'object' does not contain a definition for 'M'
            // D d = object.M<object, int>;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M<object, int>").WithArguments("object", "M").WithLocation(1, 14));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M<object, int>");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Empty(model.GetMemberGroup(memberAccess));
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_TypeArguments_03()
    {
        var source = """
D d = object.M<object, int>;
d(42);

delegate void D(int i);

static class E
{
    extension<T>(T t)
    {
        public static void M<U>(U u) { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M<object, int>");
        AssertEx.Equal("void E.<G>$8048A6C8BE30A622530249B904B537EB<System.Object>.M<System.Int32>(System.Int32 u)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$8048A6C8BE30A622530249B904B537EB<System.Object>.M<System.Int32>(System.Int32 u)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_TypeArguments_04()
    {
        var source = """
D d = object.M<object>;
d(42);

delegate void D(int i);

static class E
{
    extension<T>(T)
    {
        public static void M(int i) { System.Console.Write("ran"); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M<object>");
        AssertEx.Equal("void E.<G>$8048A6C8BE30A622530249B904B537EB<System.Object>.M(System.Int32 i)", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        AssertEx.SequenceEqual(["void E.<G>$8048A6C8BE30A622530249B904B537EB<System.Object>.M(System.Int32 i)"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_OptionalParameter()
    {
        var source = """
System.Action a = object.M;
System.Action a2 = E.M2;

static class E
{
    extension(object)
    {
        public static void M(int i = 0) => throw null;
    }

    public static void M2(int i = 0) => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,26): error CS0123: No overload for 'M' matches delegate 'Action'
            // System.Action a = object.M;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M").WithArguments("M", "System.Action").WithLocation(1, 26),
            // (2,22): error CS0123: No overload for 'M2' matches delegate 'Action'
            // System.Action a2 = E.M2;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M2").WithArguments("M2", "System.Action").WithLocation(2, 22));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M([System.Int32 i = 0])"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        source = """
static class E
{
    static void Main()
    {
        System.Action a2 = E.M2;
    }

    public static void M2(int i = 0) => throw null;
}
""";
        comp = CreateCompilation(source, parseOptions: TestOptions.Regular7);
        comp.VerifyEmitDiagnostics(
            // (5,30): error CS0123: No overload for 'M2' matches delegate 'Action'
            //         System.Action a2 = E.M2;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M2").WithArguments("M2", "System.Action").WithLocation(5, 30));
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_ReturnRefKindMismatch()
    {
        var source = """
D d = object.M;

delegate int D();

static class E
{
    extension(object)
    {
        public static ref int M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,7): error CS8189: Ref mismatch between 'E.extension(object).M()' and delegate 'D'
            // D d = object.M;
            Diagnostic(ErrorCode.ERR_DelegateRefMismatch, "object.M").WithArguments("E.extension(object).M()", "D").WithLocation(1, 7));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.SequenceEqual(["ref System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_ReturnTypeMismatch()
    {
        var source = """
D d = object.M;

delegate int D();

static class E
{
    extension(object)
    {
        public static string M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,7): error CS0407: 'string E.extension(object).M()' has the wrong return type
            // D d = object.M;
            Diagnostic(ErrorCode.ERR_BadRetType, "object.M").WithArguments("E.extension(object).M()", "string").WithLocation(1, 7));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.SequenceEqual(["System.String E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_BadParameterConversion()
    {
        var source = """
D d = object.M;
D2 d2 = object.M2;

delegate void D(long l);
delegate void D2(int i);

static class E
{
    extension(object)
    {
        public static void M(int i) => throw null;
        public static void M2(long l) => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS0123: No overload for 'M' matches delegate 'D'
            // D d = object.M;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M").WithArguments("M", "D").WithLocation(1, 14),
            // (2,9): error CS0123: No overload for 'E.extension(object).M2(long)' matches delegate 'D2'
            // D2 d2 = object.M2;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "object.M2").WithArguments("E.extension(object).M2(long)", "D2").WithLocation(2, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M(System.Int32 i)"], model.GetMemberGroup(memberAccess1).ToTestDisplayStrings());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M2");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M2(System.Int64 l)"], model.GetMemberGroup(memberAccess2).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Conditional()
    {
        var source = """
System.Action a = object.M;

static class E
{
    extension(object)
    {
        [System.Diagnostics.Conditional("DEBUG")]
        public static void M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,19): error CS1618: Cannot create delegate with 'E.extension(object).M()' because it or a method it overrides has a Conditional attribute
            // System.Action a = object.M;
            Diagnostic(ErrorCode.ERR_DelegateOnConditional, "object.M").WithArguments("E.extension(object).M()").WithLocation(1, 19));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Partial()
    {
        var source = """
System.Action a = object.M;

static partial class E
{
    extension(object)
    {
        static partial void M();
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,26): error CS0117: 'object' does not contain a definition for 'M'
            // System.Action a = object.M;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("object", "M").WithLocation(1, 26),
            // (7,29): error CS0751: A partial member must be declared within a partial type
            //         static partial void M();
            Diagnostic(ErrorCode.ERR_PartialMemberOnlyInPartialClass, "M").WithLocation(7, 29));
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Pointer()
    {
        var source = """
D d = object.M;

unsafe delegate int* D();

unsafe static class E
{
    extension(object)
    {
        public static int* M() => throw null;
    }
}
""";
        var comp = CreateCompilation(source, options: TestOptions.UnsafeDebugExe);
        comp.VerifyEmitDiagnostics(
            // (1,7): error CS0214: Pointers and fixed size buffers may only be used in an unsafe context
            // D d = object.M;
            Diagnostic(ErrorCode.ERR_UnsafeNeeded, "object.M").WithLocation(1, 7));
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_RefReadonlyMismatch()
    {
        var source = """
D d = object.M;
D2 d2 = object.M2;

D d3 = E.M3;
D2 d4 = E.M4;

delegate void D(ref int i);
delegate void D2(ref readonly int i);

static class E
{
    extension(object)
    {
        public static void M(ref readonly int i) => throw null;
        public static void M2(ref int i) => throw null;
    }

    public static void M3(ref readonly int i) => throw null;
    public static void M4(ref int i) => throw null;
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,7): warning CS9198: Reference kind modifier of parameter 'ref readonly int i' doesn't match the corresponding parameter 'ref int i' in target.
            // D d = object.M;
            Diagnostic(ErrorCode.WRN_TargetDifferentRefness, "object.M").WithArguments("ref readonly int i", "ref int i").WithLocation(1, 7),
            // (2,16): error CS0123: No overload for 'M2' matches delegate 'D2'
            // D2 d2 = object.M2;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M2").WithArguments("M2", "D2").WithLocation(2, 16),
            // (4,8): warning CS9198: Reference kind modifier of parameter 'ref readonly int i' doesn't match the corresponding parameter 'ref int i' in target.
            // D d3 = E.M3;
            Diagnostic(ErrorCode.WRN_TargetDifferentRefness, "E.M3").WithArguments("ref readonly int i", "ref int i").WithLocation(4, 8),
            // (5,11): error CS0123: No overload for 'M4' matches delegate 'D2'
            // D2 d4 = E.M4;
            Diagnostic(ErrorCode.ERR_MethDelegateMismatch, "M4").WithArguments("M4", "D2").WithLocation(5, 11));
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_Obsolete()
    {
        var source = """
System.Action a = object.M;

static class E
{
    extension(object)
    {
        [System.Obsolete("obsolete", true)]
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,19): error CS0619: 'E.extension(object).M()' is obsolete: 'obsolete'
            // System.Action a = object.M;
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "object.M").WithArguments("E.extension(object).M()", "obsolete").WithLocation(1, 19));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_ReceiverTypeKind_01()
    {
        var source = """
System.Action a = object.M;
System.Action a2 = int.M;
a();
a2();

static class E
{
    extension(object)
    {
        public static void M() { System.Console.Write("ran "); }
    }
}
""";
        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran ran").VerifyDiagnostics();
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_ReceiverTypeKind_02()
    {
        var source = """
System.Action a2 = int.M;
a2();

static class E
{
    extension(int)
    {
        public static void M() { System.Console.Write("ran"); }
    }
}
""";

        var comp = CreateCompilation(source);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();
    }

    [Fact]
    public void DelegateConversion_InstanceReceiver_ReceiverTypeKind()
    {
        var source = """
object o = null;
int i = 0;

System.Action a = o.M;
System.Action a2 = i.M;

static class E
{
    extension(object o)
    {
        public void M() => throw null;
    }

    extension(int i)
    {
        public void M() => throw null;
    }
}
""";

        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (5,20): error CS1113: Extension method 'E.extension(int).M()' defined on value type 'int' cannot be used to create delegates
            // System.Action a2 = i.M;
            Diagnostic(ErrorCode.ERR_ValueTypeExtDelegate, "i.M").WithArguments("E.extension(int).M()", "int").WithLocation(5, 20));
    }

    [Fact]
    public void DelegateConversion_InstanceReceiver_RefStructReceiver()
    {
        var source = """
System.Span<int> s = default;

System.Action a = s.M;

static class E
{
    extension(object o)
    {
        public void M() => throw null;
    }
}
""";

        var comp = CreateCompilation(source, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (3,21): error CS1061: 'Span<int>' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'Span<int>' could be found (are you missing a using directive or an assembly reference?)
            // System.Action a = s.M;
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("System.Span<int>", "M").WithLocation(3, 21));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "s.M");
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void DelegateConversion_TypeReceiver_RefStructReceiver()
    {
        var source = """
System.Action a = System.Span<int>.M;

static class E
{
    extension(object)
    {
        public static void M() => throw null;
    }
}
""";

        // Note: we apply the same conversion requirements even though no conversion on the receiver
        //   is needed in a static scenario.
        var comp = CreateCompilation(source, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (1,36): error CS0117: 'Span<int>' does not contain a definition for 'M'
            // System.Action a = System.Span<int>.M;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "M").WithArguments("System.Span<int>", "M").WithLocation(1, 36));
    }

    [Fact]
    public void InstancePropertyAccess_Obsolete()
    {
        var src = """
_ = new object().Property;
new object().Property = 43;

static class E
{
    extension(object o)
    {
        [System.Obsolete("Property is obsolete", true)]
        public int Property { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS0619: 'E.extension(object).Property' is obsolete: 'Property is obsolete'
            // _ = new object().Property;
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "new object().Property").WithArguments("E.extension(object).Property", "Property is obsolete").WithLocation(1, 5),
            // (2,1): error CS0619: 'E.extension(object).Property' is obsolete: 'Property is obsolete'
            // new object().Property = 43;
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "new object().Property").WithArguments("E.extension(object).Property", "Property is obsolete").WithLocation(2, 1));
    }

    [Fact]
    public void StaticPropertyAccess_Obsolete()
    {
        var src = """
_ = object.Property;
object.Property = 43;

static class E
{
    extension(object)
    {
        [System.Obsolete("Property is obsolete", true)]
        public static int Property { get => 42; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,5): error CS0619: 'E.extension(object).Property' is obsolete: 'Property is obsolete'
            // _ = object.Property;
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "object.Property").WithArguments("E.extension(object).Property", "Property is obsolete").WithLocation(1, 5),
            // (2,1): error CS0619: 'E.extension(object).Property' is obsolete: 'Property is obsolete'
            // object.Property = 43;
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "object.Property").WithArguments("E.extension(object).Property", "Property is obsolete").WithLocation(2, 1));
    }

    [Fact]
    public void InstancePropertyAccess_Obsolete_InInvocation()
    {
        var src = """
new object().Property();

static class E
{
    extension(object o)
    {
        [System.Obsolete("Property is obsolete", true)]
        public System.Action Property { get => throw null; set { } }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0619: 'E.extension(object).Property' is obsolete: 'Property is obsolete'
            // new object().Property();
            Diagnostic(ErrorCode.ERR_DeprecatedSymbolStr, "new object().Property").WithArguments("E.extension(object).Property", "Property is obsolete").WithLocation(1, 1));
    }

    [Fact]
    public void InstancePropertyAccess_ColorColor()
    {
        var src = """
C.M(new C());

class C
{
    public static void M(C C)
    {
        C.Property = 42;
    }
}

static class E
{
    extension(C c)
    {
        public int Property { set { System.Console.Write(value); } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var property = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.Property");
        AssertEx.Equal("System.Int32 E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Property { set; }", model.GetSymbolInfo(property).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(property)); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void StaticPropertyAccess_ColorColor()
    {
        var src = """
C.M(null);

class C
{
    public static void M(C C)
    {
        C.Property = 42;
    }
}

static class E
{
    extension(C c)
    {
        public static int Property { set { System.Console.Write("Property"); } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "Property").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var property = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.Property");
        AssertEx.Equal("System.Int32 E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Property { set; }", model.GetSymbolInfo(property).Symbol.ToTestDisplayString());
        Assert.Empty(model.GetMemberGroup(property)); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void ConditionalReceiver_Property_MemberAccess()
    {
        var src = """
bool b = true;
System.Console.Write((b ? "" : null).Property.ToString());

static class E
{
    extension(string s)
    {
        public int Property => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "42").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, """(b ? "" : null).Property""");
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PropertyAccess_ReturnNotLValue()
    {
        var src = """
object.Property.field = 1;

public struct S
{
    public int field;
}
static class E
{
    extension(object)
    {
        public static S Property { get => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS1612: Cannot modify the return value of 'E.extension(object).Property' because it is not a variable
            // object.Property.field = 1;
            Diagnostic(ErrorCode.ERR_ReturnNotLValue, "object.Property").WithArguments("E.extension(object).Property").WithLocation(1, 1));
    }

    [Fact]
    public void StaticPropertyAccess_RefProperty_01()
    {
        var src = """
localFuncRef(ref object.Property);
localFuncOut(out object.Property);

void localFuncRef(ref int i) => throw null;
void localFuncOut(out int i) => throw null;

static class E
{
    extension(object)
    {
        public static int Property { get => throw null; set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,18): error CS0206: A non ref-returning property or indexer may not be used as an out or ref value
            // localFuncRef(ref object.Property);
            Diagnostic(ErrorCode.ERR_RefProperty, "object.Property").WithLocation(1, 18),
            // (2,18): error CS0206: A non ref-returning property or indexer may not be used as an out or ref value
            // localFuncOut(out object.Property);
            Diagnostic(ErrorCode.ERR_RefProperty, "object.Property").WithLocation(2, 18));
    }

    [Fact]
    public void StaticPropertyAccess_RefProperty_02()
    {
        var src = """
localFuncRef(ref object.Property);
System.Console.Write(E.field);

void localFuncRef(ref int i) { i++; }

static class E
{
    public static int field = 42;
    extension(object)
    {
        public static ref int Property { get => ref E.field; }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "43").VerifyDiagnostics();
    }

    [Fact]
    public void StaticPropertyAccess_AssignReadonlyNotField()
    {
        var src = """
object.Property = 1;

static class E
{
    extension(object)
    {
        public static ref readonly int Property { get => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS8331: Cannot assign to property 'Property' or use it as the right hand side of a ref assignment because it is a readonly variable
            // object.Property = 1;
            Diagnostic(ErrorCode.ERR_AssignReadonlyNotField, "object.Property").WithArguments("property", "Property").WithLocation(1, 1));
    }

    [Fact]
    public void StaticPropertyAccess_AssgReadonlyProp()
    {
        var src = """
object.Property = 1;

static class E
{
    extension(object)
    {
        public static int Property { get => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0200: Property or indexer 'E.extension(object).Property' cannot be assigned to -- it is read only
            // object.Property = 1;
            Diagnostic(ErrorCode.ERR_AssgReadonlyProp, "object.Property").WithArguments("E.extension(object).Property").WithLocation(1, 1));
    }

    [Fact]
    public void StaticPropertyAccess_InitOnlyProperty()
    {
        var src = """
object.Property = 1;

static class E
{
    extension(object)
    {
        public static int Property { init => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (7,38): error CS8856: The 'init' accessor is not valid on static members
            //         public static int Property { init => throw null; }
            Diagnostic(ErrorCode.ERR_BadInitAccessor, "init").WithLocation(7, 38));
    }

    [Fact]
    public void InstancePropertyAccess_InitOnlyProperty()
    {
        var src = """
new object().Property = 1;

static class E
{
    extension(object o)
    {
        public int Property { init => throw null; }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS8852: Init-only property or indexer 'E.extension(object).Property' can only be assigned in an object initializer, or on 'this' or 'base' in an instance constructor or an 'init' accessor.
            // new object().Property = 1;
            Diagnostic(ErrorCode.ERR_AssignmentInitOnly, "new object().Property").WithArguments("E.extension(object).Property").WithLocation(1, 1),
            // (7,31): error CS9304: 'E.extension(object).Property': cannot declare init-only accessors in an extension block
            //         public int Property { init => throw null; }
            Diagnostic(ErrorCode.ERR_InitInExtension, "init").WithArguments("E.extension(object).Property").WithLocation(7, 31));
    }

    [Fact]
    public void InstancePropertyAccess_InitOnlyProperty_ObjectInitializer()
    {
        var src = """
_ = new object() { Property = 1 };

static class E
{
    extension(object o)
    {
        public int Property { init => throw null; }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (7,31): error CS9304: 'E.extension(object).Property': cannot declare init-only accessors in an extension block
            //         public int Property { init => throw null; }
            Diagnostic(ErrorCode.ERR_InitInExtension, "init").WithArguments("E.extension(object).Property").WithLocation(7, 31));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var assignment = GetSyntax<AssignmentExpressionSyntax>(tree, "Property = 1");
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { init; }", model.GetSymbolInfo(assignment.Left).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void StaticPropertyAccess_InaccessibleSetter()
    {
        var src = """
object.Property = 1;

static class E
{
    extension(object)
    {
        public static int Property { get => throw null; private set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0272: The property or indexer 'E.extension(object).Property' cannot be used in this context because the set accessor is inaccessible
            // object.Property = 1;
            Diagnostic(ErrorCode.ERR_InaccessibleSetter, "object.Property").WithArguments("E.extension(object).Property").WithLocation(1, 1));
    }

    [Fact]
    public void ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_ExtensionTypeMethods()
    {
        // See ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_Method
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.M1(this);
    }
}

class Color { }

static class E
{
    extension(Color c)
    {
        public void M1(S1 x, int y = 0)
        {
            System.Console.WriteLine("instance");
        }

        public static void M1<T>(T x) where T : unmanaged
        {
            System.Console.WriteLine("static");
        }
    }
}
""";
        var comp = CreateCompilation(source, options: TestOptions.ReleaseDll);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.M1(this);
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9)
            );

        Assert.NotEmpty(comp.GetTypeByMetadataName("S1").InstanceConstructors.OfType<SynthesizedPrimaryConstructor>().Single().GetCapturedParameters());

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.M1");
        AssertEx.Equal("void E.<G>$2404CFB602D7DEE90BDDEF217EC37C58.M1(S1 x, [System.Int32 y = 0])", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_ExtensionTypeProperties()
    {
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        _ = Color.P1;
    }
}

class Color { }

static class E1
{
    extension(Color c)
    {
        public int P1 => 0;
    }
}

static class E2
{
    extension(Color)
    {
        public static int P1 => 0;
    }
}
""";
        var comp = CreateCompilation(source, options: TestOptions.ReleaseDll);
        comp.VerifyEmitDiagnostics(
            // (5,13): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         _ = Color.P1;
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 13));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.P1");
        AssertEx.Equal("System.Int32 E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.P1 { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ParameterCapturing_023_ColorColor_MemberAccess_InstanceAndStatic_ExtensionTypeMembersVsExtensionMethod()
    {
        var source = """
public struct S1(Color Color)
{
    public void Test()
    {
        Color.M1(this);
    }
}

public class Color { }

public static class E1
{
    public static void M1(this Color c, S1 x, int y = 0)
    {
        System.Console.WriteLine("instance");
    }
}

static class E
{
    extension(Color)
    {
        public static void M1<T>(T x) where T : unmanaged
        {
            System.Console.WriteLine("static");
        }
    }
}
""";
        var comp = CreateCompilation(source, options: TestOptions.ReleaseDll);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.M1(this);
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9)
            );

        Assert.NotEmpty(comp.GetTypeByMetadataName("S1").InstanceConstructors.OfType<SynthesizedPrimaryConstructor>().Single().GetCapturedParameters());

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.M1");
        AssertEx.Equal("void Color.M1(S1 x, [System.Int32 y = 0])", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_01()
    {
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color { }

static class E1
{
    extension(Color c)
    {
        public System.Action Member => null;
    }
}

static class E2
{
    extension(Color)
    {
        public static int Member => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("System.Action E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_02()
    {
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
    public static int Member => 0;
}

static class E1
{
    public static void Member(this Color c) { }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void Color.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_03()
    {
        // Non-invocable candidate is out of the picture, so we're left with only the instance candidate
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E1
{
    extension(Color c)
    {
        public void Member() { }
    }
}

static class E2
{
    extension(Color)
    {
        public static int Member => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_04()
    {
        // Non-invocable candidate is out of the picture, so we're left with only the static candidate
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E1
{
    extension(Color)
    {
        public static void Member() { }
    }
}

static class E2
{
    extension(Color c)
    {
        public int Member => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(1, 17));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_05()
    {
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E1
{
    extension(Color c)
    {
        public void Member() { }
    }
}

static class E2
{
    extension(Color)
    {
        public static System.Action Member => null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.Member();
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9),
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_06()
    {
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E1
{
    extension(Color c)
    {
        public void Member() { }
    }
}

static class E2
{
    extension(Color)
    {
        public static void Member(int i) { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.Member();
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9),
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_07()
    {
        // instance extension method in inner scope, static extension method in outer scope
        var source = """
namespace N
{
    struct S1(Color Color)
    {
        public void Test()
        {
            Color.Member();
            M(this);
        }

        public static void M<T>(T x) where T : unmanaged { }
    }

    static class E1
    {
        extension(Color c)
        {
            public void Member() { }
        }
    }
}

class Color
{
}

static class E2
{
    extension(Color)
    {
        public static void Member() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (7,13): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //             Color.Member();
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(7, 13),
            // (8,13): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //             M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("N.S1.M<T>(T)", "T", "N.S1").WithLocation(8, 13));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void N.E1.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_08()
    {
        // static extension method in inner scope, instance extension method in outer scope
        var source = """
namespace N
{
    struct S1(Color Color)
    {
        public void Test()
        {
            Color.Member();
            M(this);
        }

        public static void M<T>(T x) where T : unmanaged { }
    }

    static class E1
    {
        extension(Color)
        {
            public static void Member() { }
        }
    }
}

class Color
{
}

static class E2
{
    extension(Color c)
    {
        public void Member() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (7,13): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //             Color.Member();
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(7, 13),
            // (8,13): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //             M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("N.S1.M<T>(T)", "T", "N.S1").WithLocation(8, 13));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E2.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_09()
    {
        // static extension property in inner scope, instance extension method in outer scope
        var source = """
namespace N
{
    struct S1(Color Color)
    {
        public void Test()
        {
            Color.Member();
            M(this);
        }

        public static void M<T>(T x) where T : unmanaged { }
    }

    static class E1
    {
        extension(Color)
        {
            public static System.Action Member => throw null;
        }
    }
}

class Color
{
}

static class E2
{
    extension(Color c)
    {
        public void Member() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (7,13): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //             Color.Member();
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(7, 13),
            // (8,13): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //             M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("N.S1.M<T>(T)", "T", "N.S1").WithLocation(8, 13));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E2.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_10()
    {
        // inapplicable candidate
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E1
{
    extension<T>(T t) where T : struct
    {
        public void Member() { }
    }
}

static class E2
{
    extension(Color)
    {
        public static void Member() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(1, 17));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E2.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_11()
    {
        // inapplicable candidate
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E1
{
    extension<T>(T) where T : struct
    {
        public static void Member() { }
    }
}

static class E2
{
    extension(Color c)
    {
        public void Member() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E2.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_12()
    {
        // only static candidate method
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E
{
    extension(Color)
    {
        public static void Member() { }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(1, 17));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("void E.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_13()
    {
        // only static candidate property, invocable
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E
{
    extension(Color)
    {
        public static System.Action Member => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(1, 17));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("System.Action E.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_14()
    {
        // only static candidate property, non-invocable
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E
{
    extension(Color)
    {
        public static int Member => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(1, 17),
            // (5,15): error CS1061: 'Color' does not contain a definition for 'Member' and no accessible extension method 'Member' accepting a first argument of type 'Color' could be found (are you missing a using directive or an assembly reference?)
            //         Color.Member();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Member").WithArguments("Color", "Member").WithLocation(5, 15));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_15()
    {
        // only instance candidate property, invocable
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E
{
    extension(Color c)
    {
        public System.Action Member => throw null;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (6,9): error CS8377: The type 'S1' must be a non-nullable value type, along with all fields at any level of nesting, in order to use it as parameter 'T' in the generic type or method 'S1.M<T>(T)'
            //         M(this);
            Diagnostic(ErrorCode.ERR_UnmanagedConstraintNotSatisfied, "M").WithArguments("S1.M<T>(T)", "T", "S1").WithLocation(6, 9));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        AssertEx.Equal("System.Action E.<G>$2404CFB602D7DEE90BDDEF217EC37C58.Member { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_16()
    {
        // only instance candidate property, non-invocable
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member();
        M(this);
    }

    public static void M<T>(T x) where T : unmanaged { }
}

class Color
{
}

static class E
{
    extension(Color c)
    {
        public int Member => 0;
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (1,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(1, 17),
            // (5,15): error CS1061: 'Color' does not contain a definition for 'Member' and no accessible extension method 'Member' accepting a first argument of type 'Color' could be found (are you missing a using directive or an assembly reference?)
            //         Color.Member();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Member").WithArguments("Color", "Member").WithLocation(5, 15));

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Color.Member");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_17()
    {
        // non-extension method not applicable due to arity
        var source = """
new S1(new Color()).Test();

struct S1(Color Color)
{
    public void Test()
    {
        Color.Member<int>(0);
    }
}

class Color
{
    public static void Member(int x) => throw null;
}

static class E
{
    extension(Color c)
    {
        public void Member<T>(T x) { System.Console.WriteLine("extension"); }
    }
}
""";
        var comp = CreateCompilation(source);

        CompileAndVerify(comp, expectedOutput: "extension").VerifyDiagnostics();

        Assert.NotEmpty(comp.GetTypeByMetadataName("S1").InstanceConstructors.OfType<SynthesizedPrimaryConstructor>().Single().GetCapturedParameters());
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_18()
    {
        // non-extension method not applicable due to arity, and non-extension method applicable, and instance extension method
        var source = """
struct S1(Color Color)
{
    public void Test()
    {
        Color.Member<int>(0);
    }
}

class Color
{
    public static void Member(int x) => throw null;
    public static void Member<T>(T x) => throw null;
}

static class E
{
    extension(Color c)
    {
        public void Member<T>(T x) { System.Console.WriteLine("extension"); }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (5,9): error CS9106: Identifier 'Color' is ambiguous between type 'Color' and parameter 'Color Color' in this context.
            //         Color.Member<int>(0);
            Diagnostic(ErrorCode.ERR_AmbiguousPrimaryConstructorParameterAsColorColorReceiver, "Color").WithArguments("Color", "Color", "Color Color").WithLocation(5, 9));
    }

    [Fact]
    public void PrimaryCtorParameterCapturing_19()
    {
        // non-extension method not applicable due to arity, and non-extension method applicable, and static extension method
        var source = """
new S1(new Color()).Test();

struct S1(Color Color)
{
    public void Test()
    {
        Color.Member<int>(0);
    }
}

class Color
{
    public static void Member(int x) => throw null;
    public static void Member<T>(T x) => throw null;
}

static class E
{
    extension(Color)
    {
        public static void Member<T>(T x) { System.Console.WriteLine("extension"); }
    }
}
""";
        var comp = CreateCompilation(source);
        comp.VerifyEmitDiagnostics(
            // (3,17): warning CS9113: Parameter 'Color' is unread.
            // struct S1(Color Color)
            Diagnostic(ErrorCode.WRN_UnreadPrimaryConstructorParameter, "Color").WithArguments("Color").WithLocation(3, 17));
    }

    [Fact]
    public void InstanceMethod_MemberAccess()
    {
        var src = """
new object().M.ToString();

static class E
{
    extension(object o)
    {
        public int M() => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS0119: 'E.extension(object).M()' is a method, which is not valid in the given context
            // new object().M.ToString();
            Diagnostic(ErrorCode.ERR_BadSKunknown, "M").WithArguments("E.extension(object).M()", "method").WithLocation(1, 14));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        src = """
new object().M.ToString();

static class E
{
    public static int M(this object o) => 42;
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS0119: 'E.M(object)' is a method, which is not valid in the given context
            // new object().M.ToString();
            Diagnostic(ErrorCode.ERR_BadSKunknown, "M").WithArguments("E.M(object)", "method").WithLocation(1, 14));

        tree = comp.SyntaxTrees.Single();
        model = comp.GetSemanticModel(tree);
        memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "new object().M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void InstanceMethod_MemberAccess_Missing()
    {
        var src = """
new object().M.ToString();
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,14): error CS1061: 'object' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // new object().M.ToString();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("object", "M").WithLocation(1, 14));
    }

    [Fact]
    public void CheckValueKind_AssignToMethodGroup()
    {
        var src = """
object.M = null;

static class E
{
    extension(object)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS1656: Cannot assign to 'M' because it is a 'method group'
            // object.M = null;
            Diagnostic(ErrorCode.ERR_AssgReadonlyLocalCause, "object.M").WithArguments("M", "method group").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.NotAVariable, model.GetSymbolInfo(memberAccess).CandidateReason);
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void AccessOnVoid_Invocation()
    {
        var src = """
object.M().ToString();

static class E
{
    extension(object)
    {
        public static void M() { }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,11): error CS0023: Operator '.' cannot be applied to operand of type 'void'
            // object.M().ToString();
            Diagnostic(ErrorCode.ERR_BadUnaryOp, ".").WithArguments(".", "void").WithLocation(1, 11));
    }

    [Fact]
    public void ExtensionMemberLookup_InaccessibleMembers_01()
    {
        var src = """
object.Method();
_ = object.Property;

static class E
{
    extension(object o)
    {
        private static void Method() => throw null;
        private static int Property => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,8): error CS0117: 'object' does not contain a definition for 'Method'
            // object.Method();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "Method").WithArguments("object", "Method").WithLocation(1, 8),
            // (2,12): error CS0117: 'object' does not contain a definition for 'Property'
            // _ = object.Property;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "Property").WithArguments("object", "Property").WithLocation(2, 12));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.Method");
        AssertEx.SequenceEqual(["void E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Method()"], model.GetMemberGroup(memberAccess1).ToTestDisplayStrings());

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.Property");
        Assert.Equal([], model.GetMemberGroup(memberAccess2).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void ExtensionMemberLookup_InaccessibleMembers_02()
    {
        var src = """
/*<bind>*/
object.Member = 42;
/*</bind>*/

object.Member.ToString();
object.Member++;

public static class E
{
    extension(object)
    {
        private static int Member { get => 0; set { } }
    }
}
""";
        DiagnosticDescription[] expectedDiagnostics = [
            // (2,8): error CS0117: 'object' does not contain a definition for 'Member'
            // object.Member = 42;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "Member").WithArguments("object", "Member").WithLocation(2, 8),
            // (5,8): error CS0117: 'object' does not contain a definition for 'Member'
            // object.Member.ToString();
            Diagnostic(ErrorCode.ERR_NoSuchMember, "Member").WithArguments("object", "Member").WithLocation(5, 8),
            // (6,8): error CS0117: 'object' does not contain a definition for 'Member'
            // object.Member++;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "Member").WithArguments("object", "Member").WithLocation(6, 8)];

        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(expectedDiagnostics);

        var tree = comp.SyntaxTrees.First();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "object.Member").First();
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());

        string expectedOperationTree = """
IExpressionStatementOperation (OperationKind.ExpressionStatement, Type: null, IsInvalid) (Syntax: 'object.Member = 42;')
Expression:
  ISimpleAssignmentOperation (OperationKind.SimpleAssignment, Type: ?, IsInvalid) (Syntax: 'object.Member = 42')
    Left:
      IInvalidOperation (OperationKind.Invalid, Type: ?, IsInvalid) (Syntax: 'object.Member')
        Children(1):
            IOperation:  (OperationKind.None, Type: System.Object) (Syntax: 'object')
    Right:
      ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 42) (Syntax: '42')
""";

        VerifyOperationTreeAndDiagnosticsForTest<ExpressionStatementSyntax>(src, expectedOperationTree, expectedDiagnostics);
    }

    [Fact]
    public void ExtensionMemberLookup_InterpolationHandler_Simple()
    {
        var src = """
_ = f($"{(object)1} {f2()}");

static int f(InterpolationHandler s) => 0;
static string f2() => "hello";

[System.Runtime.CompilerServices.InterpolatedStringHandler]
public struct InterpolationHandler
{
    public InterpolationHandler(int literalLength, int formattedCount) => throw null;
    public void AppendLiteral(string value) { }
    public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
}
""";

        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics();
    }

    [Fact]
    public void ExtensionMemberLookup_InterpolationHandler_AppendLiteralExtensionMethod()
    {
        var src = """
_ = f($"{(object)1} {f2()}");

static int f(InterpolationHandler s) => 0;
static string f2() => "hello";

[System.Runtime.CompilerServices.InterpolatedStringHandler]
public struct InterpolationHandler
{
    public InterpolationHandler(int literalLength, int formattedCount) => throw null;
    public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
}

public static class Extensions
{
    public static void AppendLiteral(this InterpolationHandler ih, string value) { }
}
""";

        // Interpolation handlers don't allow extension methods
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (1,20): error CS1061: 'InterpolationHandler' does not contain a definition for 'AppendLiteral' and no accessible extension method 'AppendLiteral' accepting a first argument of type 'InterpolationHandler' could be found (are you missing a using directive or an assembly reference?)
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, " ").WithArguments("InterpolationHandler", "AppendLiteral").WithLocation(1, 20),
            // (1,20): error CS8941: Interpolated string handler method '?.()' is malformed. It does not return 'void' or 'bool'.
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerMethodReturnMalformed, " ").WithArguments("?.()").WithLocation(1, 20));
    }

    [Fact]
    public void ExtensionMemberLookup_InterpolationHandler_AppendLiteralExtensionDeclarationMethod()
    {
        var src = """
_ = f($"{(object)1} {f2()}");

static int f(InterpolationHandler s) => 0;
static string f2() => "hello";

[System.Runtime.CompilerServices.InterpolatedStringHandler]
public struct InterpolationHandler
{
    public InterpolationHandler(int literalLength, int formattedCount) => throw null;
    public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
}

static class E
{
    extension(InterpolationHandler i)
    {
        public void AppendLiteral(string value) { }
    }
}
""";

        // Interpolation handlers don't allow extension methods
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (1,20): error CS1061: 'InterpolationHandler' does not contain a definition for 'AppendLiteral' and no accessible extension method 'AppendLiteral' accepting a first argument of type 'InterpolationHandler' could be found (are you missing a using directive or an assembly reference?)
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, " ").WithArguments("InterpolationHandler", "AppendLiteral").WithLocation(1, 20),
            // (1,20): error CS8941: Interpolated string handler method '?.()' is malformed. It does not return 'void' or 'bool'.
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerMethodReturnMalformed, " ").WithArguments("?.()").WithLocation(1, 20));
    }

    [Fact]
    public void ExtensionMemberLookup_InterpolationHandler_AppendFormattedExtensionMethod()
    {
        var src = """
/*<bind>*/
_ = f($"{(object)1} {f2()}");
/*</bind>*/

static int f(InterpolationHandler s) => 0;
static string f2() => "hello";

[System.Runtime.CompilerServices.InterpolatedStringHandler]
public struct InterpolationHandler
{
    public InterpolationHandler(int literalLength, int formattedCount) => throw null;
    public void AppendLiteral(string value) { }
}

public static class Extensions
{
    public static void AppendFormatted<T>(this InterpolationHandler ih, T hole, int alignment = 0, string format = null) { }
}
""";

        // Interpolation handlers don't allow extension methods
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (2,9): error CS1061: 'InterpolationHandler' does not contain a definition for 'AppendFormatted' and no accessible extension method 'AppendFormatted' accepting a first argument of type 'InterpolationHandler' could be found (are you missing a using directive or an assembly reference?)
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "{(object)1}").WithArguments("InterpolationHandler", "AppendFormatted").WithLocation(2, 9),
            // (2,9): error CS8941: Interpolated string handler method '?.()' is malformed. It does not return 'void' or 'bool'.
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerMethodReturnMalformed, "{(object)1}").WithArguments("?.()").WithLocation(2, 9),
            // (2,21): error CS1061: 'InterpolationHandler' does not contain a definition for 'AppendFormatted' and no accessible extension method 'AppendFormatted' accepting a first argument of type 'InterpolationHandler' could be found (are you missing a using directive or an assembly reference?)
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "{f2()}").WithArguments("InterpolationHandler", "AppendFormatted").WithLocation(2, 21),
            // (2,21): error CS8941: Interpolated string handler method '?.()' is malformed. It does not return 'void' or 'bool'.
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerMethodReturnMalformed, "{f2()}").WithArguments("?.()").WithLocation(2, 21)
            );
    }

    [Fact]
    public void ExtensionMemberLookup_InterpolationHandler_AppendFormattedExtensionTypeMethod()
    {
        var src = """
_ = f($"{(object)1} {f2()}");

static int f(InterpolationHandler s) => 0;
static string f2() => "hello";

[System.Runtime.CompilerServices.InterpolatedStringHandler]
public struct InterpolationHandler
{
    public InterpolationHandler(int literalLength, int formattedCount) => throw null;
    public void AppendLiteral(string value) { }
}

static class E
{
    extension(InterpolationHandler i)
    {
        public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) { }
    }
}
""";

        // Interpolation handlers don't allow extension methods
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (1,9): error CS1061: 'InterpolationHandler' does not contain a definition for 'AppendFormatted' and no accessible extension method 'AppendFormatted' accepting a first argument of type 'InterpolationHandler' could be found (are you missing a using directive or an assembly reference?)
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "{(object)1}").WithArguments("InterpolationHandler", "AppendFormatted").WithLocation(1, 9),
            // (1,9): error CS8941: Interpolated string handler method '?.()' is malformed. It does not return 'void' or 'bool'.
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerMethodReturnMalformed, "{(object)1}").WithArguments("?.()").WithLocation(1, 9),
            // (1,21): error CS1061: 'InterpolationHandler' does not contain a definition for 'AppendFormatted' and no accessible extension method 'AppendFormatted' accepting a first argument of type 'InterpolationHandler' could be found (are you missing a using directive or an assembly reference?)
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "{f2()}").WithArguments("InterpolationHandler", "AppendFormatted").WithLocation(1, 21),
            // (1,21): error CS8941: Interpolated string handler method '?.()' is malformed. It does not return 'void' or 'bool'.
            // _ = f($"{(object)1} {f2()}");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerMethodReturnMalformed, "{f2()}").WithArguments("?.()").WithLocation(1, 21));
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_Identity(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, string s)
                {
                    System.Console.Write(s);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(string s)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            "1".M($"");
            E.M("2", $"");
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "12" : null;
        CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        var verifier = CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();

        verifier.VerifyIL("<top-level-statements-entry-point>", """
            {
              // Code size       41 (0x29)
              .maxstack  4
              .locals init (string V_0)
              IL_0000:  ldstr      "1"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  ldc.i4.0
              IL_0008:  ldc.i4.0
              IL_0009:  ldloc.0
              IL_000a:  newobj     "InterpolationHandler..ctor(int, int, string)"
              IL_000f:  call       "void E.M(string, InterpolationHandler)"
              IL_0014:  ldstr      "2"
              IL_0019:  stloc.0
              IL_001a:  ldloc.0
              IL_001b:  ldc.i4.0
              IL_001c:  ldc.i4.0
              IL_001d:  ldloc.0
              IL_001e:  newobj     "InterpolationHandler..ctor(int, int, string)"
              IL_0023:  call       "void E.M(string, InterpolationHandler)"
              IL_0028:  ret
            }
            """);

        var comp = (CSharpCompilation)verifier.Compilation;
        var tree = comp.SyntaxTrees[0];
        var compRoot = tree.GetCompilationUnitRoot();
        var model = comp.GetSemanticModel(tree);
        var opRoot = model.GetOperation(compRoot);
        VerifyOperationTree(comp, opRoot, expectedOperationTree: """
            IMethodBodyOperation (OperationKind.MethodBody, Type: null) (Syntax: '"1".M($""); ... ("2", $"");')
              BlockBody:
                IBlockOperation (2 statements) (OperationKind.Block, Type: null, IsImplicit) (Syntax: '"1".M($""); ... ("2", $"");')
                  IExpressionStatementOperation (OperationKind.ExpressionStatement, Type: null) (Syntax: '"1".M($"");')
                    Expression:
                      IInvocationOperation ( void E.<G>$34505F560D9EACF86A87F3ED1F85E448.M(InterpolationHandler h)) (OperationKind.Invocation, Type: System.Void) (Syntax: '"1".M($"")')
                        Instance Receiver:
                          ILiteralOperation (OperationKind.Literal, Type: System.String, Constant: "1") (Syntax: '"1"')
                        Arguments(1):
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: h) (OperationKind.Argument, Type: null) (Syntax: '$""')
                              IInterpolatedStringHandlerCreationOperation (HandlerAppendCallsReturnBool: False, HandlerCreationHasSuccessParameter: False) (OperationKind.InterpolatedStringHandlerCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                Creation:
                                  IObjectCreationOperation (Constructor: InterpolationHandler..ctor(System.Int32 literalLength, System.Int32 formattedCount, System.String s)) (OperationKind.ObjectCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                    Arguments(3):
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: literalLength) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: formattedCount) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '"1"')
                                          IInterpolatedStringHandlerArgumentPlaceholderOperation (CallsiteReceiver) (OperationKind.InterpolatedStringHandlerArgumentPlaceholder, Type: null, IsImplicit) (Syntax: '"1"')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                    Initializer:
                                      null
                                Content:
                                  IInterpolatedStringOperation (OperationKind.InterpolatedString, Type: System.String, Constant: "") (Syntax: '$""')
                                    Parts(0)
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                  IExpressionStatementOperation (OperationKind.ExpressionStatement, Type: null) (Syntax: 'E.M("2", $"");')
                    Expression:
                      IInvocationOperation (void E.M(this System.String s, InterpolationHandler h)) (OperationKind.Invocation, Type: System.Void) (Syntax: 'E.M("2", $"")')
                        Instance Receiver:
                          null
                        Arguments(2):
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s) (OperationKind.Argument, Type: null) (Syntax: '"2"')
                              ILiteralOperation (OperationKind.Literal, Type: System.String, Constant: "2") (Syntax: '"2"')
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: h) (OperationKind.Argument, Type: null) (Syntax: '$""')
                              IInterpolatedStringHandlerCreationOperation (HandlerAppendCallsReturnBool: False, HandlerCreationHasSuccessParameter: False) (OperationKind.InterpolatedStringHandlerCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                Creation:
                                  IObjectCreationOperation (Constructor: InterpolationHandler..ctor(System.Int32 literalLength, System.Int32 formattedCount, System.String s)) (OperationKind.ObjectCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                    Arguments(3):
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: literalLength) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: formattedCount) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '"2"')
                                          IInterpolatedStringHandlerArgumentPlaceholderOperation (ArgumentIndex: 0) (OperationKind.InterpolatedStringHandlerArgumentPlaceholder, Type: null, IsImplicit) (Syntax: '"2"')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                    Initializer:
                                      null
                                Content:
                                  IInterpolatedStringOperation (OperationKind.InterpolatedString, Type: System.String, Constant: "") (Syntax: '$""')
                                    Parts(0)
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
              ExpressionBody:
                null
            """);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_MultipleEvaluation(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, string s)
                {
                    System.Console.Write(s);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(string s)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            Get("1").M($"");
            E.M(Get("2"), $"");

            static T Get<T>(T t)
            {
                System.Console.Write("Get");
                return t;
            }
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "Get1Get2" : null;
        CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_WithOtherParameters(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)
                {
                    System.Console.Write(s1);
                    System.Console.Write(s2);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(string s1)
                {
                    public void M(string s2, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1", "s2")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            "1".M("2", $"");
            E.M("3", "4", $"");
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "1234" : null;
        CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        var verifier = CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
        var comp = (CSharpCompilation)verifier.Compilation;
        var tree = comp.SyntaxTrees[0];
        var compRoot = tree.GetCompilationUnitRoot();
        var model = comp.GetSemanticModel(tree);
        var opRoot = model.GetOperation(compRoot);
        VerifyOperationTree(comp, opRoot, expectedOperationTree: """
            IMethodBodyOperation (OperationKind.MethodBody, Type: null) (Syntax: '"1".M("2",  ...  "4", $"");')
              BlockBody:
                IBlockOperation (2 statements) (OperationKind.Block, Type: null, IsImplicit) (Syntax: '"1".M("2",  ...  "4", $"");')
                  IExpressionStatementOperation (OperationKind.ExpressionStatement, Type: null) (Syntax: '"1".M("2", $"");')
                    Expression:
                      IInvocationOperation ( void E.<G>$34505F560D9EACF86A87F3ED1F85E448.M(System.String s2, InterpolationHandler h)) (OperationKind.Invocation, Type: System.Void) (Syntax: '"1".M("2", $"")')
                        Instance Receiver:
                          ILiteralOperation (OperationKind.Literal, Type: System.String, Constant: "1") (Syntax: '"1"')
                        Arguments(2):
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s2) (OperationKind.Argument, Type: null) (Syntax: '"2"')
                              ILiteralOperation (OperationKind.Literal, Type: System.String, Constant: "2") (Syntax: '"2"')
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: h) (OperationKind.Argument, Type: null) (Syntax: '$""')
                              IInterpolatedStringHandlerCreationOperation (HandlerAppendCallsReturnBool: False, HandlerCreationHasSuccessParameter: False) (OperationKind.InterpolatedStringHandlerCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                Creation:
                                  IObjectCreationOperation (Constructor: InterpolationHandler..ctor(System.Int32 literalLength, System.Int32 formattedCount, System.String s1, System.String s2)) (OperationKind.ObjectCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                    Arguments(4):
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: literalLength) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: formattedCount) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s1) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '"1"')
                                          IInterpolatedStringHandlerArgumentPlaceholderOperation (CallsiteReceiver) (OperationKind.InterpolatedStringHandlerArgumentPlaceholder, Type: null, IsImplicit) (Syntax: '"1"')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s2) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '"2"')
                                          IInterpolatedStringHandlerArgumentPlaceholderOperation (ArgumentIndex: 0) (OperationKind.InterpolatedStringHandlerArgumentPlaceholder, Type: null, IsImplicit) (Syntax: '"2"')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                    Initializer:
                                      null
                                Content:
                                  IInterpolatedStringOperation (OperationKind.InterpolatedString, Type: System.String, Constant: "") (Syntax: '$""')
                                    Parts(0)
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                  IExpressionStatementOperation (OperationKind.ExpressionStatement, Type: null) (Syntax: 'E.M("3", "4", $"");')
                    Expression:
                      IInvocationOperation (void E.M(this System.String s1, System.String s2, InterpolationHandler h)) (OperationKind.Invocation, Type: System.Void) (Syntax: 'E.M("3", "4", $"")')
                        Instance Receiver:
                          null
                        Arguments(3):
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s1) (OperationKind.Argument, Type: null) (Syntax: '"3"')
                              ILiteralOperation (OperationKind.Literal, Type: System.String, Constant: "3") (Syntax: '"3"')
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s2) (OperationKind.Argument, Type: null) (Syntax: '"4"')
                              ILiteralOperation (OperationKind.Literal, Type: System.String, Constant: "4") (Syntax: '"4"')
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                            IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: h) (OperationKind.Argument, Type: null) (Syntax: '$""')
                              IInterpolatedStringHandlerCreationOperation (HandlerAppendCallsReturnBool: False, HandlerCreationHasSuccessParameter: False) (OperationKind.InterpolatedStringHandlerCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                Creation:
                                  IObjectCreationOperation (Constructor: InterpolationHandler..ctor(System.Int32 literalLength, System.Int32 formattedCount, System.String s1, System.String s2)) (OperationKind.ObjectCreation, Type: InterpolationHandler, IsImplicit) (Syntax: '$""')
                                    Arguments(4):
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: literalLength) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: formattedCount) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '$""')
                                          ILiteralOperation (OperationKind.Literal, Type: System.Int32, Constant: 0, IsImplicit) (Syntax: '$""')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s1) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '"3"')
                                          IInterpolatedStringHandlerArgumentPlaceholderOperation (ArgumentIndex: 0) (OperationKind.InterpolatedStringHandlerArgumentPlaceholder, Type: null, IsImplicit) (Syntax: '"3"')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                        IArgumentOperation (ArgumentKind.Explicit, Matching Parameter: s2) (OperationKind.Argument, Type: null, IsImplicit) (Syntax: '"4"')
                                          IInterpolatedStringHandlerArgumentPlaceholderOperation (ArgumentIndex: 1) (OperationKind.InterpolatedStringHandlerArgumentPlaceholder, Type: null, IsImplicit) (Syntax: '"4"')
                                          InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                          OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                                    Initializer:
                                      null
                                Content:
                                  IInterpolatedStringOperation (OperationKind.InterpolatedString, Type: System.String, Constant: "") (Syntax: '$""')
                                    Parts(0)
                              InConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
                              OutConversion: CommonConversion (Exists: True, IsIdentity: True, IsNumeric: False, IsReference: False, IsUserDefined: False) (MethodSymbol: null)
              ExpressionBody:
                null
            """);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_WithConversion(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, object o)
                {
                    System.Console.Write(o);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(object o)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("o")] InterpolationHandler h) {}
                }
            }
            """;

        // The receiver is a string, so there's a BoundConversion to object as the receiver of the extension method
        var exeSource = """
            "1".M($"");
            E.M("2", $"");
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "12" : null;
        var verifier = CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        verifier.VerifyIL("<top-level-statements-entry-point>", """
            {
              // Code size       41 (0x29)
              .maxstack  4
              .locals init (object V_0)
              IL_0000:  ldstr      "1"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  ldc.i4.0
              IL_0008:  ldc.i4.0
              IL_0009:  ldloc.0
              IL_000a:  newobj     "InterpolationHandler..ctor(int, int, object)"
              IL_000f:  call       "void E.M(object, InterpolationHandler)"
              IL_0014:  ldstr      "2"
              IL_0019:  stloc.0
              IL_001a:  ldloc.0
              IL_001b:  ldc.i4.0
              IL_001c:  ldc.i4.0
              IL_001d:  ldloc.0
              IL_001e:  newobj     "InterpolationHandler..ctor(int, int, object)"
              IL_0023:  call       "void E.M(object, InterpolationHandler)"
              IL_0028:  ret
            }
            """);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_WithConversion_ExtensionParameterNarrowerThanConstructor(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, object o)
                {
                    System.Console.Write(o);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(string o)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("o")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            "1".M($"");
            E.M("2", $"");
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "12" : null;
        CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_WithConversion_ExtensionParameterWiderThanConstructor(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, string s)
                {
                    System.Console.WriteLine(s);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(object o)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("o")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            "1".M($"");
            E.M("2", $"");
            """;

        var expectedDiagnostics = new[] {
            // (1,1): error CS1503: Argument 3: cannot convert from 'object' to 'string'
            // "1".M($"");
            Diagnostic(ErrorCode.ERR_BadArgType, @"""1""").WithArguments("3", "object", "string").WithLocation(1, 1),
            // (2,5): error CS1503: Argument 3: cannot convert from 'object' to 'string'
            // E.M("2", $"");
            Diagnostic(ErrorCode.ERR_BadArgType, @"""2""").WithArguments("3", "object", "string").WithLocation(2, 5)
        };
        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CreateCompilation(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByRef(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, ref int i)
                {
                    System.Console.Write(i);
                    i = 2;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(ref int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h)
                    {
                        System.Console.Write(i);
                        i = 3;
                    }
                }
            }
            """;

        var exeSource = """
            class Program
            {
                static void Main()
                {
                    int i = 1;
                    Test1(ref i);
                    System.Console.Write(i);
                    i = 4;
                    Test2(ref i);
                    System.Console.Write(i);
                }

                static void Test1(ref int i)
                {
                    i.M($"");
                }
            
                static void Test2(ref int i)
                {
                    E.M(ref i, $"");
                }
            }
            """;

        var expectedOutput = "123423";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput).VerifyDiagnostics();

        verifier.VerifyIL("Program.Test1", """
            {
              // Code size       17 (0x11)
              .maxstack  4
              .locals init (int& V_0)
              IL_0000:  ldarg.0
              IL_0001:  stloc.0
              IL_0002:  ldloc.0
              IL_0003:  ldc.i4.0
              IL_0004:  ldc.i4.0
              IL_0005:  ldloc.0
              IL_0006:  newobj     "InterpolationHandler..ctor(int, int, ref int)"
              IL_000b:  call       "void E.M(ref int, InterpolationHandler)"
              IL_0010:  ret
            }
            """);

        verifier.VerifyIL("Program.Test2", """
            {
              // Code size       17 (0x11)
              .maxstack  4
              .locals init (int& V_0)
              IL_0000:  ldarg.0
              IL_0001:  stloc.0
              IL_0002:  ldloc.0
              IL_0003:  ldc.i4.0
              IL_0004:  ldc.i4.0
              IL_0005:  ldloc.0
              IL_0006:  newobj     "InterpolationHandler..ctor(int, int, ref int)"
              IL_000b:  call       "void E.M(ref int, InterpolationHandler)"
              IL_0010:  ret
            }
            """);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78433"), WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByRef_WithConstantReceiver(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, ref int i)
                {
                    System.Console.Write(i);
                    System.Runtime.CompilerServices.Unsafe.AsRef(in i)++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(ref int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h)
                    {
                        System.Console.Write(i);
                    }
                }
            }
            """;

        var exeSource = """
            1.M($"");
            E.M(3, $"");
            """;

        var expectedDiagnostic = new DiagnosticDescription[] {
            // (1,1): error CS1510: A ref or out value must be an assignable variable
            // 1.M($"");
            Diagnostic(ErrorCode.ERR_RefLvalueExpected, "1").WithLocation(1, 1),
            // (2,5): error CS1620: Argument 1 must be passed with the 'ref' keyword
            // E.M(3, $"");
            Diagnostic(ErrorCode.ERR_BadArgRef, "3").WithArguments("1", "ref").WithLocation(2, 5)
        };
        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostic);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CreateCompilation(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostic);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_Generic_ByRef(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler<TR>
            {

                public InterpolationHandler(int literalLength, int formattedCount, ref TR i)
                {
                    System.Console.Write(i);
                    i = (TR)(object)2;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension<T>(ref T i) where T : struct
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler<T> h)
                    {
                        System.Console.Write(i);
                        i = (T)(object)3;
                    }
                }
            }
            """;

        var exeSource = """
            class Program
            {
                static void Main()
                {
                    int i = 1;
                    Test1(ref i);
                    System.Console.Write(i);
                    i = 4;
                    Test2(ref i);
                    System.Console.Write(i);
                }

                static void Test1<T>(ref T i) where T : struct
                {
                    i.M($"");
                }
            
                static void Test2<T>(ref T i) where T : struct
                {
                    E.M(ref i, $"");
                }
            }
            """;

        var expectedOutput = "123423";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput).VerifyDiagnostics();

        verifier.VerifyIL("Program.Test1<T>", """
            {
              // Code size       17 (0x11)
              .maxstack  4
              .locals init (T& V_0)
              IL_0000:  ldarg.0
              IL_0001:  stloc.0
              IL_0002:  ldloc.0
              IL_0003:  ldc.i4.0
              IL_0004:  ldc.i4.0
              IL_0005:  ldloc.0
              IL_0006:  newobj     "InterpolationHandler<T>..ctor(int, int, ref T)"
              IL_000b:  call       "void E.M<T>(ref T, InterpolationHandler<T>)"
              IL_0010:  ret
            }
            """);

        verifier.VerifyIL("Program.Test2<T>", """
            {
              // Code size       17 (0x11)
              .maxstack  4
              .locals init (T& V_0)
              IL_0000:  ldarg.0
              IL_0001:  stloc.0
              IL_0002:  ldloc.0
              IL_0003:  ldc.i4.0
              IL_0004:  ldc.i4.0
              IL_0005:  ldloc.0
              IL_0006:  newobj     "InterpolationHandler<T>..ctor(int, int, ref T)"
              IL_000b:  call       "void E.M<T>(ref T, InterpolationHandler<T>)"
              IL_0010:  ret
            }
            """);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByIn_WithConstantReceiver(bool useMetadataRef, [CombinatorialValues("ref readonly", "in")] string refkind)
    {
        var src = $$$"""
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, {{{refkind}}} int i)
                {
                    System.Console.Write(i);
                    System.Runtime.CompilerServices.Unsafe.AsRef(in i)++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension({{{refkind}}} int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h)
                    {
                        System.Console.Write(i);
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS9193 // Argument 0 should be a variable because it is passed to a 'ref readonly' parameter

            class Program
            {
                static void Main()
                {
                    Test1();
                    Test2();
                }
            
                static void Test1()
                {
                    1.M($"");
                }
            
                static void Test2()
                {
                    E.M(3, $"");
                } 
            }
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "1234" : null;
        var verifier = CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        verifier.VerifyIL("Program.Test1", $$$"""
            {
              // Code size       19 (0x13)
              .maxstack  4
              .locals init (int V_0)
              IL_0000:  ldc.i4.1
              IL_0001:  stloc.0
              IL_0002:  ldloca.s   V_0
              IL_0004:  ldc.i4.0
              IL_0005:  ldc.i4.0
              IL_0006:  ldloca.s   V_0
              IL_0008:  newobj     "InterpolationHandler..ctor(int, int, {{{refkind}}} int)"
              IL_000d:  call       "void E.M({{{refkind}}} int, InterpolationHandler)"
              IL_0012:  ret
            }
            """);

        verifier.VerifyIL("Program.Test2", $$$"""
            {
              // Code size       19 (0x13)
              .maxstack  4
              .locals init (int V_0)
              IL_0000:  ldc.i4.3
              IL_0001:  stloc.0
              IL_0002:  ldloca.s   V_0
              IL_0004:  ldc.i4.0
              IL_0005:  ldc.i4.0
              IL_0006:  ldloca.s   V_0
              IL_0008:  newobj     "InterpolationHandler..ctor(int, int, {{{refkind}}} int)"
              IL_000d:  call       "void E.M({{{refkind}}} int, InterpolationHandler)"
              IL_0012:  ret
            }
            """);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByIn_WithLocalReciever(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, in int i)
                {
                    System.Console.Write(i);
                    System.Runtime.CompilerServices.Unsafe.AsRef(in i)++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(in int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h)
                    {
                        System.Console.Write(i);
                        System.Runtime.CompilerServices.Unsafe.AsRef(in i)++;
                    }
                }
            }
            """;

        var exeSource = """
            int i = 1;
            i.M($"");
            System.Console.Write(i);
            E.M(i, $"");
            System.Console.Write(i);
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "123345" : null;
        var verifier = CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();
        verifier.VerifyIL("<top-level-statements-entry-point>", """
            {
              // Code size       49 (0x31)
              .maxstack  4
              .locals init (int V_0, //i
                            int& V_1)
              IL_0000:  ldc.i4.1
              IL_0001:  stloc.0
              IL_0002:  ldloca.s   V_0
              IL_0004:  stloc.1
              IL_0005:  ldloc.1
              IL_0006:  ldc.i4.0
              IL_0007:  ldc.i4.0
              IL_0008:  ldloc.1
              IL_0009:  newobj     "InterpolationHandler..ctor(int, int, in int)"
              IL_000e:  call       "void E.M(in int, InterpolationHandler)"
              IL_0013:  ldloc.0
              IL_0014:  call       "void System.Console.Write(int)"
              IL_0019:  ldloca.s   V_0
              IL_001b:  stloc.1
              IL_001c:  ldloc.1
              IL_001d:  ldc.i4.0
              IL_001e:  ldc.i4.0
              IL_001f:  ldloc.1
              IL_0020:  newobj     "InterpolationHandler..ctor(int, int, in int)"
              IL_0025:  call       "void E.M(in int, InterpolationHandler)"
              IL_002a:  ldloc.0
              IL_002b:  call       "void System.Console.Write(int)"
              IL_0030:  ret
            }
            """);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByRefMismatch_01(bool useMetadataRef, [CombinatorialValues("ref readonly", "in", "")] string refkind)
    {
        var src = $$"""
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, ref int i)
                {
                    System.Console.Write(i);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension({{refkind}} int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = $$"""
            int i = 1;
            i.M($"");
            E.M({{(refkind == "" ? "" : "in ")}}i, $"");
            """;

        var expectedDiagnostic = new[] {
            // (2,1): error CS1620: Argument 3 must be passed with the 'ref' keyword
            // i.M($"");
            Diagnostic(ErrorCode.ERR_BadArgRef, "i").WithArguments("3", "ref").WithLocation(2, 1),
            // (3,5): error CS1620: Argument 3 must be passed with the 'ref' keyword
            // E.M(i, $"");
            Diagnostic(ErrorCode.ERR_BadArgRef, "i").WithArguments("3", "ref").WithLocation(3, 5 + (refkind == "" ? 0 : 3))

        };
        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostic);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CreateCompilation(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostic);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByRefMismatch_02(bool useMetadataRef, [CombinatorialValues("ref readonly", "in")] string refkind)
    {
        var src = $$"""
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, {{refkind}} int i)
                {
                    System.Console.Write(i);
                    System.Runtime.CompilerServices.Unsafe.AsRef(in i)++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(ref int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h)
                    {
                        System.Console.Write(i);
                        i++;
                    }
                }
            }
            """;

        var exeSource = """
            int i = 1;
            i.M($"");
            System.Console.Write(i);
            E.M(ref i, $"");
            System.Console.Write(i);
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "123345" : null;
        CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_ByRefMismatch_03(bool useMetadataRef)
    {
        var src = $$"""
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, int i)
                {
                    System.Console.WriteLine(i);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(ref int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            int i = 1;
            i.M($"");
            E.M(ref i, $"");
            """;

        var expectedDiagnostics = new[] {
            // (2,1): error CS1615: Argument 3 may not be passed with the 'ref' keyword
            // i.M($"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, "i").WithArguments("3", "ref").WithLocation(2, 1),
            // (3,9): error CS1615: Argument 3 may not be passed with the 'ref' keyword
            // E.M(ref i, $"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, "i").WithArguments("3", "ref").WithLocation(3, 9)
         };

        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CreateCompilation(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);
    }

    [Theory]
    [CombinatorialData]
    public void InterpolationHandler_StructReceiverParameter_ByValue(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s)
                {
                    System.Console.Write(s.i);
                    s.i++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public struct MyStruct
            {
                public int i;
            }

            public static class E
            {
                extension(MyStruct s)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler h)
                    {
                        System.Console.Write(s.i);
                        s.i++;
                    }
                }
            }
            """;

        var exeSource = """
            new MyStruct().M($"");
            E.M(new MyStruct(), $"");
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "0000" : null;
        var verifier = CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();

        verifier.VerifyIL("<top-level-statements-entry-point>", """
            {
              // Code size       45 (0x2d)
              .maxstack  4
              .locals init (MyStruct V_0)
              IL_0000:  ldloca.s   V_0
              IL_0002:  initobj    "MyStruct"
              IL_0008:  ldloc.0
              IL_0009:  ldc.i4.0
              IL_000a:  ldc.i4.0
              IL_000b:  ldloc.0
              IL_000c:  newobj     "InterpolationHandler..ctor(int, int, MyStruct)"
              IL_0011:  call       "void E.M(MyStruct, InterpolationHandler)"
              IL_0016:  ldloca.s   V_0
              IL_0018:  initobj    "MyStruct"
              IL_001e:  ldloc.0
              IL_001f:  ldc.i4.0
              IL_0020:  ldc.i4.0
              IL_0021:  ldloc.0
              IL_0022:  newobj     "InterpolationHandler..ctor(int, int, MyStruct)"
              IL_0027:  call       "void E.M(MyStruct, InterpolationHandler)"
              IL_002c:  ret
            }
            """);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Theory]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_StructReceiverParameter_ByValueThroughField(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s)
                {
                    System.Console.Write(s.i);
                    E.field.i++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public struct MyStruct
            {
                public int i;
            }

            public static class E
            {
                extension(MyStruct s)
                {
                    public void M(int i, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler h)
                    {
                        System.Console.Write(s.i);
                        E.field.i++;
                    }
                }

                public static MyStruct field;
            }
            """;

        var exeSource = """
            E.field.M(Increment(), $"");
            E.M(E.field, Increment(), $"");

            int Increment() => E.field.i++;
            """;

        var expectedOutput = "0033";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput)
            .VerifyDiagnostics();

        verifier.VerifyIL("<top-level-statements-entry-point>", """
            {
              // Code size       51 (0x33)
              .maxstack  5
              .locals init (MyStruct V_0)
              IL_0000:  ldsfld     "MyStruct E.field"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  call       "int Program.<<Main>$>g__Increment|0_0()"
              IL_000c:  ldc.i4.0
              IL_000d:  ldc.i4.0
              IL_000e:  ldloc.0
              IL_000f:  newobj     "InterpolationHandler..ctor(int, int, MyStruct)"
              IL_0014:  call       "void E.M(MyStruct, int, InterpolationHandler)"
              IL_0019:  ldsfld     "MyStruct E.field"
              IL_001e:  stloc.0
              IL_001f:  ldloc.0
              IL_0020:  call       "int Program.<<Main>$>g__Increment|0_0()"
              IL_0025:  ldc.i4.0
              IL_0026:  ldc.i4.0
              IL_0027:  ldloc.0
              IL_0028:  newobj     "InterpolationHandler..ctor(int, int, MyStruct)"
              IL_002d:  call       "void E.M(MyStruct, int, InterpolationHandler)"
              IL_0032:  ret
            }
            """);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_StructReceiverParameter_Generic_ByValueThroughField(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler<TR>
            {

                public InterpolationHandler(int literalLength, int formattedCount, TR s)
                {
                    System.Console.Write(((MyStruct)(object)s).i);
                    E<MyStruct>.field.i++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public struct MyStruct
            {
                public int i;
            }

            public static class E
            {
                extension<T>(T s)
                {
                    public void M(int i, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler<T> h)
                    {
                        System.Console.Write(((MyStruct)(object)s).i);
                        E<MyStruct>.field.i++;
                    }
                }
            }

            public static class E<T>
            {
                public static T field;
            }            
            """;

        var exeSource = """
            class Porgram
            {
                static void Main()
                {
                    Test1<MyStruct>();
                    Test2<MyStruct>();
                    Test3<MyStruct>();
                    Test4<MyStruct>();
                }

                static void Test1<T>()
                {
                    E<T>.field.M(Increment(), $"");
                }

                static void Test2<T>()
                {
                    E.M(E<T>.field, Increment(), $"");
                }
            
                static void Test3<T>() where T : struct
                {
                    E<T>.field.M(Increment(), $"");
                }
            
                static void Test4<T>() where T : struct
                {
                    E.M(E<T>.field, Increment(), $"");
                }
                            
                static int Increment() => E<MyStruct>.field.i++;
            }
            """;

        var expectedOutput = "00336699";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput)
            .VerifyDiagnostics();

        var expectedIL = """
            {
              // Code size       26 (0x1a)
              .maxstack  5
              .locals init (T V_0)
              IL_0000:  ldsfld     "T E<T>.field"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  call       "int Porgram.Increment()"
              IL_000c:  ldc.i4.0
              IL_000d:  ldc.i4.0
              IL_000e:  ldloc.0
              IL_000f:  newobj     "InterpolationHandler<T>..ctor(int, int, T)"
              IL_0014:  call       "void E.M<T>(T, int, InterpolationHandler<T>)"
              IL_0019:  ret
            }
            """;

        verifier.VerifyIL("Porgram.Test1<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test2<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test3<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test4<T>", expectedIL);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_StructReceiverParameter_GenericStruct_ByValueThroughField(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler<TR>
            {

                public InterpolationHandler(int literalLength, int formattedCount, TR s)
                {
                    System.Console.Write(((MyStruct)(object)s).i);
                    E<MyStruct>.field.i++;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public struct MyStruct
            {
                public int i;
            }

            public static class E
            {
                extension<T>(T s) where T : struct
                {
                    public void M(int i, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler<T> h)
                    {
                        System.Console.Write(((MyStruct)(object)s).i);
                        E<MyStruct>.field.i++;
                    }
                }
            }

            public static class E<T>
            {
                public static T field;
            }            
            """;

        var exeSource = """
            class Porgram
            {
                static void Main()
                {
                    Test3<MyStruct>();
                    Test4<MyStruct>();
                }

                static void Test3<T>() where T : struct
                {
                    E<T>.field.M(Increment(), $"");
                }
            
                static void Test4<T>() where T : struct
                {
                    E.M(E<T>.field, Increment(), $"");
                }
                            
                static int Increment() => E<MyStruct>.field.i++;
            }
            """;

        var expectedOutput = "0033";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput)
            .VerifyDiagnostics();

        var expectedIL = """
            {
              // Code size       26 (0x1a)
              .maxstack  5
              .locals init (T V_0)
              IL_0000:  ldsfld     "T E<T>.field"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  call       "int Porgram.Increment()"
              IL_000c:  ldc.i4.0
              IL_000d:  ldc.i4.0
              IL_000e:  ldloc.0
              IL_000f:  newobj     "InterpolationHandler<T>..ctor(int, int, T)"
              IL_0014:  call       "void E.M<T>(T, int, InterpolationHandler<T>)"
              IL_0019:  ret
            }
            """;

        verifier.VerifyIL("Porgram.Test3<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test4<T>", expectedIL);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_ClassReceiverParameter_GenericClass_ByValueThroughField(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler<TR>
            {

                public InterpolationHandler(int literalLength, int formattedCount, TR s)
                {
                    System.Console.Write(((MyClass)(object)s).i);
                    E<MyClass>.field = new MyClass() { i = E<MyClass>.field.i + 1 };
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public class MyClass
            {
                public int i;
            }

            public static class E
            {
                extension<T>(T s)
                {
                    public void M(int i, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler<T> h)
                    {
                        System.Console.Write(((MyClass)(object)s).i);
                        E<MyClass>.field = new MyClass() { i = E<MyClass>.field.i + 1 };
                    }
                }
            }

            public static class E<T>
            {
                public static T field;
            }            
            """;

        var exeSource = """
            class Porgram
            {
                static void Main()
                {
                    E<MyClass>.field = new MyClass();
                    Test1<MyClass>();
                    Test2<MyClass>();
                    Test3<MyClass>();
                    Test4<MyClass>();
                }

                static void Test1<T>()
                {
                    E<T>.field.M(Increment(), $"");
                }

                static void Test2<T>()
                {
                    E.M(E<T>.field, Increment(), $"");
                }
            
                static void Test3<T>() where T : class
                {
                    E<T>.field.M(Increment(), $"");
                }
            
                static void Test4<T>() where T : class
                {
                    E.M(E<T>.field, Increment(), $"");
                }
                            
                static int Increment() => (E<MyClass>.field = new MyClass() { i = E<MyClass>.field.i + 1 }).i;
            }
            """;

        var expectedOutput = "00336699";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput)
            .VerifyDiagnostics();

        var expectedIL = """
            {
              // Code size       26 (0x1a)
              .maxstack  5
              .locals init (T V_0)
              IL_0000:  ldsfld     "T E<T>.field"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  call       "int Porgram.Increment()"
              IL_000c:  ldc.i4.0
              IL_000d:  ldc.i4.0
              IL_000e:  ldloc.0
              IL_000f:  newobj     "InterpolationHandler<T>..ctor(int, int, T)"
              IL_0014:  call       "void E.M<T>(T, int, InterpolationHandler<T>)"
              IL_0019:  ret
            }
            """;

        verifier.VerifyIL("Porgram.Test1<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test2<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test3<T>", expectedIL);

        verifier.VerifyIL("Porgram.Test4<T>", expectedIL);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory]
    [WorkItem("https://github.com/dotnet/roslyn/issues/79379")]
    [CombinatorialData]
    public void InterpolationHandler_ClassReceiverParameter_Generic_ByValueThroughField(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler<TR>
            {

                public InterpolationHandler(int literalLength, int formattedCount, TR s)
                {
                    System.Console.Write(((MyClass)(object)s).i);
                    E<MyClass>.field = new MyClass() { i = E<MyClass>.field.i + 1 };
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public class MyClass
            {
                public int i;
            }

            public static class E
            {
                extension<T>(T s) where T : class
                {
                    public void M(int i, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler<T> h)
                    {
                        System.Console.Write(((MyClass)(object)s).i);
                        E<MyClass>.field = new MyClass() { i = E<MyClass>.field.i + 1 };
                    }
                }
            }

            public static class E<T>
            {
                public static T field;
            }            
            """;

        var exeSource = """
            class Porgram
            {
                static void Main()
                {
                    E<MyClass>.field = new MyClass();
                    Test3<MyClass>();
                    Test4<MyClass>();
                }

                static void Test3<T>() where T : class
                {
                    E<T>.field.M(Increment(), $"");
                }
            
                static void Test4<T>() where T : class
                {
                    E.M(E<T>.field, Increment(), $"");
                }
                            
                static int Increment() => (E<MyClass>.field = new MyClass() { i = E<MyClass>.field.i + 1 }).i;
            }
            """;

        var expectedOutput = "0033";
        var verifier = CompileAndVerify([exeSource, src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute], expectedOutput: expectedOutput)
            .VerifyDiagnostics();

        verifier.VerifyIL("Porgram.Test3<T>", """
            {
              // Code size       26 (0x1a)
              .maxstack  5
              .locals init (T V_0)
              IL_0000:  ldsfld     "T E<T>.field"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  call       "int Porgram.Increment()"
              IL_000c:  ldc.i4.0
              IL_000d:  ldc.i4.0
              IL_000e:  ldloc.0
              IL_000f:  newobj     "InterpolationHandler<T>..ctor(int, int, T)"
              IL_0014:  call       "void E.M<T>(T, int, InterpolationHandler<T>)"
              IL_0019:  ret
            }
            """);

        verifier.VerifyIL("Porgram.Test4<T>", """
            {
              // Code size       26 (0x1a)
              .maxstack  5
              .locals init (T V_0)
              IL_0000:  ldsfld     "T E<T>.field"
              IL_0005:  stloc.0
              IL_0006:  ldloc.0
              IL_0007:  call       "int Porgram.Increment()"
              IL_000c:  ldc.i4.0
              IL_000d:  ldc.i4.0
              IL_000e:  ldloc.0
              IL_000f:  newobj     "InterpolationHandler<T>..ctor(int, int, T)"
              IL_0014:  call       "void E.M<T>(T, int, InterpolationHandler<T>)"
              IL_0019:  ret
            }
            """);

        var comp1 = CreateCompilation([src, InterpolatedStringHandlerAttribute, InterpolatedStringHandlerArgumentAttribute]);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], expectedOutput: expectedOutput)
            .VerifyDiagnostics();
    }

    [Theory]
    [CombinatorialData]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_01(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
            #pragma warning disable CS0169 // The field 'InterpolationHandler.i' is never used
                private ref int i;
            #pragma warning restore CS0169 // The field 'InterpolationHandler.i' is never used

                public InterpolationHandler(int literalLength, int formattedCount, scoped MyStruct s)
                {
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(MyStruct s)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s")] InterpolationHandler h)
                    {
                        return new();
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                return new MyStruct().M($"");
            }
            """;

        var verifier = CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, verify: Verification.Fails).VerifyDiagnostics();

        verifier.VerifyIL("Program.<<Main>$>g__localFunc|0_0()", """
            {
              // Code size       23 (0x17)
              .maxstack  4
              .locals init (MyStruct V_0)
              IL_0000:  ldloca.s   V_0
              IL_0002:  initobj    "MyStruct"
              IL_0008:  ldloc.0
              IL_0009:  ldc.i4.0
              IL_000a:  ldc.i4.0
              IL_000b:  ldloc.0
              IL_000c:  newobj     "InterpolationHandler..ctor(int, int, scoped MyStruct)"
              IL_0011:  call       "MyStruct E.M(MyStruct, InterpolationHandler)"
              IL_0016:  ret
            }
            """);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, verify: Verification.Fails)
            .VerifyDiagnostics();
    }

    [Fact]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_02()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
                public ref int i;

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s2)
                {
                    i = ref s2.i;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(MyStruct s1)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1")] InterpolationHandler h)
                    {
                        return new() { i = ref h.i };
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                int i = 0;
                return new MyStruct() { i = ref i }.M($"");
            }
            """;

        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (6,27): error CS8352: Cannot use variable 'i = ref i' in this context because it may expose referenced variables outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeVariable, "{ i = ref i }").WithArguments("i = ref i").WithLocation(6, 27),
            // (6,12): error CS8347: Cannot use a result of 'E.extension(MyStruct).M(InterpolationHandler)' in this context because it may expose variables referenced by parameter 's1' outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeCall, @"new MyStruct() { i = ref i }.M($"""")").WithArguments("E.extension(MyStruct).M(InterpolationHandler)", "s1").WithLocation(6, 12)
        );
    }

    [Fact]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_03()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
                public ref int i;

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s2)
                {
                    i = ref s2.i;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(MyStruct s1)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1")] ref InterpolationHandler h)
                    {
                        return new() { i = ref h.i };
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                var myStruct = new MyStruct();
                return myStruct.M($"");
            }
            """;

        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (6,12): error CS8347: Cannot use a result of 'E.extension(MyStruct).M(ref InterpolationHandler)' in this context because it may expose variables referenced by parameter 'h' outside of their declaration scope
            //     return myStruct.M($"");
            Diagnostic(ErrorCode.ERR_EscapeCall, @"myStruct.M($"""")").WithArguments("E.extension(MyStruct).M(ref InterpolationHandler)", "h").WithLocation(6, 12),
            // (6,23): error CS8156: An expression cannot be used in this context because it may not be passed or returned by reference
            //     return myStruct.M($"");
            Diagnostic(ErrorCode.ERR_RefReturnLvalueExpected, @"$""""").WithLocation(6, 23)
        );
    }

    [Fact]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_04()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
                public ref int i;

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s2)
                {
                    i = ref s2.i;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(MyStruct s1)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1"), System.Diagnostics.CodeAnalysis.UnscopedRef] ref InterpolationHandler h)
                    {
                        return new() { i = ref h.i };
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                int i = 0;
                return new MyStruct() { i = ref i }.M($"");
            }
            """;

        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (6,12): error CS8347: Cannot use a result of 'E.extension(MyStruct).M(ref InterpolationHandler)' in this context because it may expose variables referenced by parameter 's1' outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeCall, @"new MyStruct() { i = ref i }.M($"""")").WithArguments("E.extension(MyStruct).M(ref InterpolationHandler)", "s1").WithLocation(6, 12),
            // (6,27): error CS8352: Cannot use variable 'i = ref i' in this context because it may expose referenced variables outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeVariable, "{ i = ref i }").WithArguments("i = ref i").WithLocation(6, 27)
        );
    }

    [Fact]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_05()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
                public ref int i;

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s2)
                {
                    i = ref s2.i;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(scoped MyStruct s1)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1")] InterpolationHandler h)
                    {
                        return new() { i = ref h.i };
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                int i = 0;
                return new MyStruct() { i = ref i }.M($"");
            }
            """;

        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (6,12): error CS8352: Cannot use variable 'new MyStruct() { i = ref i }' in this context because it may expose referenced variables outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeVariable, "new MyStruct() { i = ref i }").WithArguments("new MyStruct() { i = ref i }").WithLocation(6, 12),
            // (6,12): error CS8347: Cannot use a result of 'E.extension(scoped MyStruct).M(InterpolationHandler)' in this context because it may expose variables referenced by parameter 'h' outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeCall, @"new MyStruct() { i = ref i }.M($"""")").WithArguments("E.extension(scoped MyStruct).M(InterpolationHandler)", "h").WithLocation(6, 12),
            // (6,43): error CS8347: Cannot use a result of 'InterpolationHandler.InterpolationHandler(int, int, MyStruct)' in this context because it may expose variables referenced by parameter 's2' outside of their declaration scope
            //     return new MyStruct() { i = ref i }.M($"");
            Diagnostic(ErrorCode.ERR_EscapeCall, @"$""""").WithArguments("InterpolationHandler.InterpolationHandler(int, int, MyStruct)", "s2").WithLocation(6, 43)
        );
    }

    [Fact]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_06()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
                public ref int i;

                public InterpolationHandler(int literalLength, int formattedCount, MyStruct s2)
                {
                    i = ref s2.i;
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(scoped MyStruct s1)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1")] scoped InterpolationHandler h)
                    {
                        return new() { i = ref h.i };
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                int i = 0;
                return new MyStruct() { i = ref i }.M($"");
            }
            """;

        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (25,26): error CS8352: Cannot use variable 'i = ref h.i' in this context because it may expose referenced variables outside of their declaration scope
            //             return new() { i = ref h.i };
            Diagnostic(ErrorCode.ERR_EscapeVariable, "{ i = ref h.i }").WithArguments("i = ref h.i").WithLocation(25, 26)
        );
    }

    [Theory, CombinatorialData]
    public void InterpolationHandler_RefStructReceiverParameter_EscapeScopes_07(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public ref struct InterpolationHandler
            {
            #pragma warning disable CS0169 // The field 'InterpolationHandler.i' is never used
                private ref int i;
            #pragma warning restore CS0169 // The field 'InterpolationHandler.i' is never used

                public InterpolationHandler(int literalLength, int formattedCount, scoped MyStruct s2)
                {
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public ref struct MyStruct
            {
                public ref int i;
            }

            public static class E
            {
                extension(scoped MyStruct s1)
                {
                    public MyStruct M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1")] scoped InterpolationHandler h)
                    {
                        return new();
                    }
                }
            }
            """;

        var exeSource = """
            #pragma warning disable CS8321 // The local function 'localFunc' is declared but never used
            MyStruct localFunc()
            #pragma warning restore CS8321 // The local function 'localFunc' is declared but never used
            {
                int i = 0;
                return new MyStruct() { i = ref i }.M($"");
            }
            """;

        var verifier = CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, verify: Verification.Fails).VerifyDiagnostics();

        verifier.VerifyIL("Program.<<Main>$>g__localFunc|0_0()", """
            {
              // Code size       36 (0x24)
              .maxstack  4
              .locals init (int V_0, //i
                            MyStruct V_1,
                            MyStruct V_2)
              IL_0000:  ldc.i4.0
              IL_0001:  stloc.0
              IL_0002:  ldloca.s   V_2
              IL_0004:  initobj    "MyStruct"
              IL_000a:  ldloca.s   V_2
              IL_000c:  ldloca.s   V_0
              IL_000e:  stfld      "ref int MyStruct.i"
              IL_0013:  ldloc.2
              IL_0014:  stloc.1
              IL_0015:  ldloc.1
              IL_0016:  ldc.i4.0
              IL_0017:  ldc.i4.0
              IL_0018:  ldloc.1
              IL_0019:  newobj     "InterpolationHandler..ctor(int, int, scoped MyStruct)"
              IL_001e:  call       "MyStruct E.M(scoped MyStruct, scoped InterpolationHandler)"
              IL_0023:  ret
            }
            """);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, verify: Verification.Fails)
            .VerifyDiagnostics();
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_NullableMismatch_01(bool useMetadataRef, bool useOut)
    {
        string outParam = useOut ? ", out bool valid" : "";
        var src = $$"""
            #nullable enable
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{{outParam}})
                {
                    System.Console.Write(s1);
                    System.Console.Write(s2);
                    {{(useOut ? "valid = true;" : "")}}
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string? format = null) => throw null!;
            }

            public static class E
            {
                extension(string? s1)
                {
                    public void M(string? s2, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s1", "s2")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            #nullable enable
            ((string?)null).M(null, $"");
            ((string?)null).M("", $"");
            "".M(null, $"");
            "".M("", $"");
            E.M(null, null, $"");
            E.M("", null, $"");
            E.M(null, "", $"");
            E.M("", "", $"");
            """;

        var expectedDiagnostics = new[] {
            // (2,2): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // ((string?)null).M(null, $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "(string?)null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(2, 2),
            // (2,19): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // ((string?)null).M(null, $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(2, 19),
            // (3,2): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // ((string?)null).M("", $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "(string?)null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(3, 2),
            // (4,6): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // "".M(null, $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(4, 6),
            // (6,5): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // E.M(null, null, $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(6, 5),
            // (6,11): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // E.M(null, null, $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(6, 11),
            // (7,9): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // E.M("", null, $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(7, 9),
            // (8,5): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
            // E.M(null, "", $"");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(8, 5)
        };
        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CreateCompilation(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_ReceiverParameter_NullableMismatch_02(bool useMetadataRef, bool useOut)
    {
        string outParam = useOut ? ", out bool valid" : "";
        var src = $$"""
            #nullable enable
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{{outParam}})
                {
                    System.Console.Write(s1);
                    System.Console.Write(s2);
                    {{(useOut ? "valid = true;" : "")}}
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string? format = null) => throw null!;
            }

            public static class E
            {
                extension(string? s1)
                {
                    public void M(string? s2, string? s3, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s2", "s3")] InterpolationHandler h) {}
                }
            }
            """;

        var exeSource = """
            #nullable enable
            "".M(null, null, $"");
            "".M(null, "", $"");
            "".M("", null, $"");
            "".M("", "", $"");
            E.M("", null, null, $"");
            E.M("", "", null, $"");
            E.M("", null, "", $"");
            E.M("", "", "", $"");
            """;

        var expectedDiagnostics = new[] {
                // (2,6): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // "".M(null, null, $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(2, 6),
                // (2,12): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // "".M(null, null, $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(2, 12),
                // (3,6): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // "".M(null, "", $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(3, 6),
                // (4,10): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // "".M("", null, $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(4, 10),
                // (6,9): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // E.M("", null, null, $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(6, 9),
                // (6,15): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // E.M("", null, null, $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(6, 15),
                // (7,13): warning CS8604: Possible null reference argument for parameter 's2' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // E.M("", "", null, $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s2", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(7, 13),
                // (8,9): warning CS8604: Possible null reference argument for parameter 's1' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2)'.
                // E.M("", null, "", $"");
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("s1", $"InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string s1, string s2{outParam})").WithLocation(8, 9)

        };
        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CreateCompilation(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90).VerifyDiagnostics(expectedDiagnostics);
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    public void InterpolationHandler_ParameterErrors_MappedCorrectly_01()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, int i)
                {
                    System.Console.WriteLine(i);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("nonexistent")] InterpolationHandler h) {}
                }
            }
            """;

        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyDiagnostics(
            // (17,24): error CS8945: 'nonexistent' is not a valid parameter name from 'E.extension(int).M(InterpolationHandler)'.
            //         public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("nonexistent")] InterpolationHandler h) {}
            Diagnostic(ErrorCode.ERR_InvalidInterpolatedStringHandlerArgumentName, @"System.Runtime.CompilerServices.InterpolatedStringHandlerArgument(""nonexistent"")").WithArguments("nonexistent", "E.extension(int).M(InterpolationHandler)").WithLocation(17, 24)
        );

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);
        var underlying = symbol.GetSymbol<NamedTypeSymbol>();
        var m = underlying.GetMember<MethodSymbol>("M");
        Assert.False(underlying.ExtensionParameter.HasInterpolatedStringHandlerArgumentError);
        Assert.True(underlying.ExtensionParameter.InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        Assert.True(m.Parameters[0].HasInterpolatedStringHandlerArgumentError);
        Assert.True(m.Parameters[0].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        var implM = underlying.ContainingType.GetMember<MethodSymbol>("M");
        Assert.False(implM.Parameters[0].HasInterpolatedStringHandlerArgumentError);
        Assert.True(implM.Parameters[0].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        Assert.True(implM.Parameters[1].HasInterpolatedStringHandlerArgumentError);
        Assert.True(implM.Parameters[1].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [InlineData("i")]
    [InlineData("")]
    [InlineData("nonexistent")]
    public void InterpolationHandler_ParameterErrors_MappedCorrectly_02(string attributeValue)
    {
        var src = $$"""
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, InterpolationHandler i)
                {
                    System.Console.WriteLine(i);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("{{attributeValue}}")] InterpolationHandler i)
                {
                    public void M() {}
                }
            }
            """;

        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyDiagnostics(
            // (15,16): error CS9325: Interpolated string handler arguments are not allowed in this context.
            //     extension([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("")] InterpolationHandler i)
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerArgumentDisallowed, $@"System.Runtime.CompilerServices.InterpolatedStringHandlerArgument(""{attributeValue}"")").WithLocation(15, 16)
        );

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);
        var underlying = symbol.GetSymbol<NamedTypeSymbol>();
        Assert.True(underlying.ExtensionParameter.HasInterpolatedStringHandlerArgumentError);
        Assert.True(underlying.ExtensionParameter.InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        var implM = underlying.ContainingType.GetMember<MethodSymbol>("M");
        Assert.True(implM.Parameters[0].HasInterpolatedStringHandlerArgumentError);
        Assert.True(implM.Parameters[0].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [InlineData("01 00 01 69 00 00")] // "i"
    [InlineData("01 00 00 00 00")] // ""
    [InlineData("01 00 0b 6e 6f 6e 65 78 69 73 74 65 6e 74 00 00")] // "nonexistent"
    public void InterpolationHandler_ParameterErrors_MappedCorrectly_02_FromMetadata(string attributeValue)
    {
        // Equivalent to:
        // [System.Runtime.CompilerServices.InterpolatedStringHandler]
        // public struct InterpolationHandler
        // {
        //     public InterpolationHandler(int literalLength, int formattedCount, InterpolationHandler i)
        //     {
        //         System.Console.WriteLine(i);
        //     }
        //     public void AppendLiteral(string value) { }
        //     public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
        // }

        // public static class E
        // {
        //     extension([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("{attributeValue}")] InterpolationHandler i)
        //     {
        //         public void M() {}
        //     }
        // }
        // Note: the grouping and marker types and attributes use a previous naming convention (which doesn't affect metadata loading)
        var il = $$"""
            .class public sequential ansi sealed beforefieldinit InterpolationHandler
                extends [mscorlib]System.ValueType
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerAttribute::.ctor() = (01 00 00 00)
                .pack 0
                .size 1

                // Methods
                .method public hidebysig specialname rtspecialname 
                    instance void .ctor (int32 literalLength, int32 formattedCount, valuetype InterpolationHandler param) cil managed 
                {
                    nop
                    ret
                } // end of method InterpolationHandler::.ctor

                .method public hidebysig instance void AppendLiteral (string 'value') cil managed 
                {
                    nop
                    ret
                } // end of method InterpolationHandler::AppendLiteral

                .method public hidebysig instance void AppendFormatted<T> (!!T hole, [opt] int32 'alignment', [opt] string format) cil managed 
                {
                    .param [2] = int32(0)
                    .param [3] = nullref
                    nop
                    ret
                } // end of method InterpolationHandler::AppendFormatted

            } // end of class InterpolationHandler

            .class public auto ansi abstract sealed beforefieldinit E
                extends System.Object
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                .class nested public auto ansi sealed specialname '<Extension>$E159F66A155642BDC88178F886EFBCA4'
                    extends System.Object
                {
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                    .class nested public auto ansi abstract sealed specialname '<Marker>$4325EE76DDCC76651231F283DA59D9E9'
                        extends System.Object
                    {
                        .method public hidebysig specialname static void '<Extension>$' ( valuetype InterpolationHandler i ) cil managed 
                        {
                            .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = ( 01 00 00 00 )
                            .param [1]
                            .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerArgumentAttribute::.ctor(string) = ({{attributeValue}})
                            IL_0000: ret
                        }
                    }

                    .method public hidebysig instance void M () cil managed 
                    {
                        .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                            01 00 29 3c 4d 61 72 6b 65 72 3e 24 34 33 32 35
                            45 45 37 36 44 44 43 43 37 36 36 35 31 32 33 31
                            46 32 38 33 44 41 35 39 44 39 45 39 00 00
                        )
                        IL_0000: ldnull
                        IL_0001: throw
                    }
                }

                .method public hidebysig static void M ( valuetype InterpolationHandler i ) cil managed 
                {
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                    .param [1]
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerArgumentAttribute::.ctor(string) = ({{attributeValue}})
                    IL_0000: ret
                }
            }
            """ + ExtensionMarkerAttributeIL;

        var src = """
            $"".M();
            E.M($"");
            """;

        var comp = CreateCompilationWithIL(src, ilSource: il, targetFramework: TargetFramework.Net90);
        comp.VerifyDiagnostics(
            // (1,1): error CS1929: 'string' does not contain a definition for 'M' and the best extension method overload 'E.extension(InterpolationHandler).M()' requires a receiver of type 'InterpolationHandler'
            // $"".M();
            Diagnostic(ErrorCode.ERR_BadInstanceArgType, @"$""""").WithArguments("string", "M", "E.extension(InterpolationHandler).M()", "InterpolationHandler").WithLocation(1, 1),
            // (2,5): error CS8949: The InterpolatedStringHandlerArgumentAttribute applied to parameter 'InterpolationHandler i' is malformed and cannot be interpreted. Construct an instance of 'InterpolationHandler' manually.
            // E.M($"");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerArgumentAttributeMalformed, @"$""""").WithArguments("InterpolationHandler i", "InterpolationHandler").WithLocation(2, 5),
            // (2,5): error CS7036: There is no argument given that corresponds to the required parameter 'param' of 'InterpolationHandler.InterpolationHandler(int, int, InterpolationHandler)'
            // E.M($"");
            Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, @"$""""").WithArguments("param", "InterpolationHandler.InterpolationHandler(int, int, InterpolationHandler)").WithLocation(2, 5),
            // (2,5): error CS1615: Argument 3 may not be passed with the 'out' keyword
            // E.M($"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, @"$""""").WithArguments("3", "out").WithLocation(2, 5)
        );

        var e = comp.GetTypeByMetadataName("E");
        var symbol = e.GetTypeMembers().Single();

        Assert.True(symbol.ExtensionParameter.HasInterpolatedStringHandlerArgumentError);
        Assert.True(symbol.ExtensionParameter.InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        var implM = symbol.ContainingType.GetMember<MethodSymbol>("M");
        Assert.True(implM.Parameters[0].HasInterpolatedStringHandlerArgumentError);
        Assert.True(implM.Parameters[0].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    public void InterpolationHandler_ParameterErrors_MappedCorrectly_03()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {

                public InterpolationHandler(int literalLength, int formattedCount, int i)
                {
                    System.Console.WriteLine(i);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) => throw null;
            }

            public static class E
            {
                extension(int i)
                {
                    public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("")] InterpolationHandler h) {}
                }
            }
            """;

        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyDiagnostics(
            // (17,24): error CS8944: 'E.extension(int).M(InterpolationHandler)' is not an instance method, the receiver or extension receiver parameter cannot be an interpolated string handler argument.
            //         public void M([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("")] InterpolationHandler h) {}
            Diagnostic(ErrorCode.ERR_NotInstanceInvalidInterpolatedStringHandlerArgumentName, @"System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("""")").WithArguments("E.extension(int).M(InterpolationHandler)").WithLocation(17, 24)
        );

        var tree = comp.SyntaxTrees[0];
        var model = comp.GetSemanticModel(tree);
        var extension = tree.GetRoot().DescendantNodes().OfType<ExtensionBlockDeclarationSyntax>().Single();

        var symbol = model.GetDeclaredSymbol(extension);
        AssertExtensionDeclaration(symbol);
        var underlying = symbol.GetSymbol<NamedTypeSymbol>();
        var m = underlying.GetMember<MethodSymbol>("M");
        Assert.False(underlying.ExtensionParameter.HasInterpolatedStringHandlerArgumentError);
        Assert.True(underlying.ExtensionParameter.InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        Assert.True(m.Parameters[0].HasInterpolatedStringHandlerArgumentError);
        Assert.True(m.Parameters[0].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        var implM = underlying.ContainingType.GetMember<MethodSymbol>("M");
        Assert.False(implM.Parameters[0].HasInterpolatedStringHandlerArgumentError);
        Assert.True(implM.Parameters[0].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
        Assert.True(implM.Parameters[1].HasInterpolatedStringHandlerArgumentError);
        Assert.True(implM.Parameters[1].InterpolatedStringHandlerArgumentIndexes.IsEmpty);
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    public void InterpolationHandler_ExtensionIndexer_Basic()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {
                public InterpolationHandler(int literalLength, int formattedCount, int receiver, int index)
                {
                    System.Console.Write(receiver);
                    System.Console.Write(index);
                }
                public void AppendLiteral(string value) { System.Console.Write(value); }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) { System.Console.Write(hole); }
            }

            public static class E
            {
                extension(int i)
                {
                    public int this[int j, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i", "j")] InterpolationHandler h]
                    {
                        get => 1;
                        set { }
                    }
                }
            }
            """;

        var exeSource = """
            1[2, $""] = 2;
            _ = 3[4, $""];
            E.set_Item(5, 6, $"", 0);
            E.get_Item(7, 8, $"");
            """;

        // Tracked by https://github.com/dotnet/roslyn/issues/76130 : add full support for indexers or disallow them
        // var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "12345678" : null;
        CreateCompilation([exeSource, src], targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (1,1): error CS0021: Cannot apply indexing with [] to an expression of type 'int'
            // 1[2, $""] = 2;
            Diagnostic(ErrorCode.ERR_BadIndexLHS, @"1[2, $""""]").WithArguments("int").WithLocation(1, 1),
            // (2,5): error CS0021: Cannot apply indexing with [] to an expression of type 'int'
            // _ = 3[4, $""];
            Diagnostic(ErrorCode.ERR_BadIndexLHS, @"3[4, $""""]").WithArguments("int").WithLocation(2, 5),
            // (17,20): error CS9282: Extension declarations can include only methods or properties
            //         public int this[int j, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i", "j")] InterpolationHandler h]
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(17, 20)
        );
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    public void InterpolationHandler_ExtensionIndexer_NullableWarnings()
    {
        var src = """
            #nullable enable
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {
                public InterpolationHandler(int literalLength, int formattedCount, string receiver, string key)
                {
                    System.Console.Write($"receiver: {receiver}, key: {key}");
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string? format = null) { }
            }

            public static class E
            {
                extension(string? s)
                {
                    public string this[string key, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s", "key")] InterpolationHandler h] { set { } }
                }
            }
            """;

        var exeSource = """
            #nullable enable
            ((string?)null)["key", $""] = "";
            "test"[null, $""] = "";
            E.set_Item((string?)null, "key", $"", "");
            E.set_Item("test", null, $"", "");
            """;

        // Tracked by https://github.com/dotnet/roslyn/issues/76130 : add full support for indexers or disallow them
        CreateCompilation([src, exeSource], parseOptions: TestOptions.RegularPreview, targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (2,1): error CS1501: No overload for method 'this' takes 2 arguments
            // ((string?)null)["key", $""] = "";
            Diagnostic(ErrorCode.ERR_BadArgCount, @"((string?)null)[""key"", $""""]").WithArguments("this", "2").WithLocation(2, 1),
            // (2,2): warning CS8602: Dereference of a possibly null reference.
            // ((string?)null)["key", $""] = "";
            Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "(string?)null").WithLocation(2, 2),
            // (3,1): error CS1501: No overload for method 'this' takes 2 arguments
            // "test"[null, $""] = "";
            Diagnostic(ErrorCode.ERR_BadArgCount, @"""test""[null, $""""]").WithArguments("this", "2").WithLocation(3, 1),
            // (4,12): warning CS8604: Possible null reference argument for parameter 'receiver' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string receiver, string key)'.
            // E.set_Item((string?)null, "key", $"", "");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "(string?)null").WithArguments("receiver", "InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string receiver, string key)").WithLocation(4, 12),
            // (5,20): warning CS8625: Cannot convert null literal to non-nullable reference type.
            // E.set_Item("test", null, $"", "");
            Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(5, 20),
            // (5,20): warning CS8604: Possible null reference argument for parameter 'key' in 'InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string receiver, string key)'.
            // E.set_Item("test", null, $"", "");
            Diagnostic(ErrorCode.WRN_NullReferenceArgument, "null").WithArguments("key", "InterpolationHandler.InterpolationHandler(int literalLength, int formattedCount, string receiver, string key)").WithLocation(5, 20),
            // (17,23): error CS9282: Extension declarations can include only methods or properties
            //         public string this[string key, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("s", "key")] InterpolationHandler h] { set { } }
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(17, 23)
        );
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [CombinatorialData]
    public void InterpolationHandler_StaticExtensionMethod_Basic(bool useMetadataRef)
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {
                public InterpolationHandler(int literalLength, int formattedCount, int param1)
                {
                    System.Console.Write(param1);
                }
                public void AppendLiteral(string value) { System.Console.Write(value); }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) { System.Console.Write(hole); }
            }

            public static class E
            {
                extension(int)
                {
                    public static void StaticMethod(int param1, [System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("param1")] InterpolationHandler h)
                    {
                    }
                }
            }
            """;

        var exeSource = """
            int.StaticMethod(1, $"");
            E.StaticMethod(2, $"");
            """;

        var expectedOutput = ExecutionConditionUtil.IsCoreClr ? "12" : null;
        CompileAndVerify([exeSource, src], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var comp1 = CreateCompilation(src, targetFramework: TargetFramework.Net90);

        CompileAndVerify(exeSource, references: [useMetadataRef ? comp1.ToMetadataReference() : comp1.EmitToImageReference()], targetFramework: TargetFramework.Net90, expectedOutput: expectedOutput, verify: Verification.FailsPEVerify)
            .VerifyDiagnostics();
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    public void InterpolationHandler_StaticExtensionMethod_ParameterMismatch()
    {
        var src = """
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {
                public InterpolationHandler(int literalLength, int formattedCount, string param)
                {
                    System.Console.WriteLine(param);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) { }
            }

            public static class E
            {
                extension(int)
                {
                    public static void StaticMethod([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("nonexistent")] InterpolationHandler h)
                    {
                    }
                }
            }
            """;

        CreateCompilation(src, targetFramework: TargetFramework.Net90, parseOptions: TestOptions.RegularPreview).VerifyDiagnostics(
            // (16,42): error CS8945: 'nonexistent' is not a valid parameter name from 'E.extension(int).StaticMethod(InterpolationHandler)'.
            //         public static void StaticMethod([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("nonexistent")] InterpolationHandler h)
            Diagnostic(ErrorCode.ERR_InvalidInterpolatedStringHandlerArgumentName, @"System.Runtime.CompilerServices.InterpolatedStringHandlerArgument(""nonexistent"")").WithArguments("nonexistent", "E.extension(int).StaticMethod(InterpolationHandler)").WithLocation(16, 42)
        );
    }

    [Theory, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [InlineData("i")]
    [InlineData("")]
    public void InterpolationHandler_StaticExtensionMethod_ReferencesExtensionParameter(string argument)
    {
        var src = $$"""
            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct InterpolationHandler
            {
                public InterpolationHandler(int literalLength, int formattedCount, string param)
                {
                    System.Console.WriteLine(param);
                }
                public void AppendLiteral(string value) { }
                public void AppendFormatted<T>(T hole, int alignment = 0, string format = null) { }
            }

            public static class E
            {
                extension(int i)
                {
                    public static void StaticMethod([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("{{argument}}")] InterpolationHandler h)
                    {
                    }
                }
            }
            """;

        CreateCompilation(src, parseOptions: TestOptions.RegularPreview, targetFramework: TargetFramework.Net90).VerifyDiagnostics(
            // (16,42): error CS8944: 'E.extension(int).StaticMethod(InterpolationHandler)' is not an instance method, the receiver or extension receiver parameter cannot be an interpolated string handler argument.
            //         public static void StaticMethod([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("i")] InterpolationHandler h)
            Diagnostic(ErrorCode.ERR_NotInstanceInvalidInterpolatedStringHandlerArgumentName, @$"System.Runtime.CompilerServices.InterpolatedStringHandlerArgument(""{argument}"")").WithArguments("E.extension(int).StaticMethod(InterpolationHandler)").WithLocation(16, 42)
        );
    }

    [Theory]
    [WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    [InlineData("01 00 01 69 00 00")] // "i"
    [InlineData("01 00 00 00 00")] // ""
    public void InterpolationHandler_StaticExtensionMethod_ReferencesExtensionParameter_FromMetadata(string argument)
    {
        // Equivalent to:
        // [System.Runtime.CompilerServices.InterpolatedStringHandler]
        // public struct InterpolationHandler
        // {
        //     public InterpolationHandler(int literalLength, int formattedCount, string param)
        //     {
        //     }
        //     public void AppendLiteral(string value) { }
        //     public void AppendFormatted<T>(T hole, int alignment = 0, string? format = null) { }
        // }

        // public static class E
        // {
        //     extension(int i)
        //     {
        //         public static void StaticMethod([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument({argument})] InterpolationHandler h)
        //         {
        //         }
        //     }
        // }
        // Note: the grouping and marker types and attributes use a previous naming convention (which doesn't affect metadata loading)

        var il = $$"""
            .class public sequential ansi sealed beforefieldinit InterpolationHandler
                extends [mscorlib]System.ValueType
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerAttribute::.ctor() = (01 00 00 00)
                .pack 0
                .size 1

                // Methods
                .method public hidebysig specialname rtspecialname 
                    instance void .ctor (int32 literalLength, int32 formattedCount, int32 param) cil managed 
                {
                    nop
                    ret
                } // end of method InterpolationHandler::.ctor

                .method public hidebysig instance void AppendLiteral (string 'value') cil managed 
                {
                    nop
                    ret
                } // end of method InterpolationHandler::AppendLiteral

                .method public hidebysig instance void AppendFormatted<T> (!!T hole, [opt] int32 'alignment', [opt] string format) cil managed 
                {
                    .param [2] = int32(0)
                    .param [3] = nullref
                    nop
                    ret
                } // end of method InterpolationHandler::AppendFormatted

            } // end of class InterpolationHandler

            .class public auto ansi abstract sealed beforefieldinit E
                extends System.Object
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                .class nested public auto ansi sealed specialname '<Extension>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'
                    extends System.Object
                {
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                    .class nested public auto ansi abstract sealed specialname '<Marker>$F4B4FFE41AB49E80A4ECF390CF6EB372'
                        extends System.Object
                    {
                        .method public hidebysig specialname static void '<Extension>$' ( int32 i ) cil managed 
                        {
                            .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = ( 01 00 00 00 )
                            ret
                        }
                    }

                    .method public hidebysig static void StaticMethod ( valuetype InterpolationHandler h ) cil managed 
                    {
                        .custom instance void System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                            01 00 29 3c 4d 61 72 6b 65 72 3e 24 46 34 42 34
                            46 46 45 34 31 41 42 34 39 45 38 30 41 34 45 43
                            46 33 39 30 43 46 36 45 42 33 37 32 00 00
                        )
                        .param [1]
                            .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerArgumentAttribute::.ctor(string) = ({{argument}})
                        ldnull
                        throw
                    }
                }

                .method public hidebysig static void StaticMethod ( valuetype InterpolationHandler h ) cil managed 
                {
                    .param [1]
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerArgumentAttribute::.ctor(string) = ({{argument}})
                    ret
                }
            }
            """ + ExtensionMarkerAttributeIL;

        var src = """
            int.StaticMethod($"");
            E.StaticMethod($"");
            """;

        CreateCompilationWithIL(src, ilSource: il, parseOptions: TestOptions.RegularPreview).VerifyDiagnostics(
            // (1,18): error CS8949: The InterpolatedStringHandlerArgumentAttribute applied to parameter 'InterpolationHandler h' is malformed and cannot be interpreted. Construct an instance of 'InterpolationHandler' manually.
            // int.StaticMethod($"");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerArgumentAttributeMalformed, @"$""""").WithArguments("InterpolationHandler h", "InterpolationHandler").WithLocation(1, 18),
            // (1,18): error CS7036: There is no argument given that corresponds to the required parameter 'param' of 'InterpolationHandler.InterpolationHandler(int, int, int)'
            // int.StaticMethod($"");
            Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, @"$""""").WithArguments("param", "InterpolationHandler.InterpolationHandler(int, int, int)").WithLocation(1, 18),
            // (1,18): error CS1615: Argument 3 may not be passed with the 'out' keyword
            // int.StaticMethod($"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, @"$""""").WithArguments("3", "out").WithLocation(1, 18),
            // (2,16): error CS8949: The InterpolatedStringHandlerArgumentAttribute applied to parameter 'InterpolationHandler h' is malformed and cannot be interpreted. Construct an instance of 'InterpolationHandler' manually.
            // E.StaticMethod($"");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerArgumentAttributeMalformed, @"$""""").WithArguments("InterpolationHandler h", "InterpolationHandler").WithLocation(2, 16),
            // (2,16): error CS7036: There is no argument given that corresponds to the required parameter 'param' of 'InterpolationHandler.InterpolationHandler(int, int, int)'
            // E.StaticMethod($"");
            Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, @"$""""").WithArguments("param", "InterpolationHandler.InterpolationHandler(int, int, int)").WithLocation(2, 16),
            // (2,16): error CS1615: Argument 3 may not be passed with the 'out' keyword
            // E.StaticMethod($"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, @"$""""").WithArguments("3", "out").WithLocation(2, 16)
        );
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78137")]
    public void InterpolationHandler_InstanceExtensionMethod_ReferencesInstanceParameter_FromMetadata()
    {
        // Equivalent to:
        // [System.Runtime.CompilerServices.InterpolatedStringHandler]
        // public struct InterpolationHandler
        // {
        //     public InterpolationHandler(int literalLength, int formattedCount, string param)
        //     {
        //     }
        //     public void AppendLiteral(string value) { }
        //     public void AppendFormatted<T>(T hole, int alignment = 0, string? format = null) { }
        // }

        // public static class E
        // {
        //     extension(int i)
        //     {
        //         public void InstanceMethod([System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("")] InterpolationHandler h)
        //         {
        //         }
        //     }
        // }
        // Note: the grouping and marker types and attributes use a previous naming convention (which doesn't affect metadata loading)

        var il = """
            .class public sequential ansi sealed beforefieldinit InterpolationHandler
                extends [mscorlib]System.ValueType
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerAttribute::.ctor() = (01 00 00 00)
                .pack 0
                .size 1

                // Methods
                .method public hidebysig specialname rtspecialname 
                    instance void .ctor (int32 literalLength, int32 formattedCount, int32 param) cil managed 
                {
                    nop
                    ret
                } // end of method InterpolationHandler::.ctor

                .method public hidebysig instance void AppendLiteral (string 'value') cil managed 
                {
                    nop
                    ret
                } // end of method InterpolationHandler::AppendLiteral

                .method public hidebysig instance void AppendFormatted<T> (!!T hole, [opt] int32 'alignment', [opt] string format) cil managed 
                {
                    .param [2] = int32(0)
                    .param [3] = nullref
                    nop
                    ret
                } // end of method InterpolationHandler::AppendFormatted

            } // end of class InterpolationHandler

            .class public auto ansi abstract sealed beforefieldinit E
                extends System.Object
            {
                .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                .class nested public auto ansi sealed specialname '<Extension>$BA41CFE2B5EDAEB8C1B9062F59ED4D69'
                    extends System.Object
                {
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                    .class nested public auto ansi abstract sealed specialname '<Marker>$F4B4FFE41AB49E80A4ECF390CF6EB372'
                        extends System.Object
                    {
                        .method public hidebysig specialname static void '<Extension>$' ( int32 i ) cil managed 
                        {
                            .custom instance void [mscorlib]System.Runtime.CompilerServices.CompilerGeneratedAttribute::.ctor() = ( 01 00 00 00 )
                            ret
                        }
                    }

                    .method public hidebysig instance void InstanceMethod ( valuetype InterpolationHandler h ) cil managed 
                    {
                        .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionMarkerAttribute::.ctor(string) = (
                            01 00 29 3c 4d 61 72 6b 65 72 3e 24 46 34 42 34
                            46 46 45 34 31 41 42 34 39 45 38 30 41 34 45 43
                            46 33 39 30 43 46 36 45 42 33 37 32 00 00
                        )
                        .param [1]
                        .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerArgumentAttribute::.ctor(string) = (01 00 00 00 00)
                        ldnull
                        throw
                    }
                }

                .method public hidebysig static void InstanceMethod ( int32 i, valuetype InterpolationHandler h ) cil managed 
                {
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.ExtensionAttribute::.ctor() = ( 01 00 00 00 )
                    .param [2]
                    .custom instance void [mscorlib]System.Runtime.CompilerServices.InterpolatedStringHandlerArgumentAttribute::.ctor(string) = (01 00 00 00 00)
                    ret
                }
            }
            """ + ExtensionMarkerAttributeIL;

        var src = """
            1.InstanceMethod($"");
            E.InstanceMethod(1, $"");
            """;

        CreateCompilationWithIL(src, ilSource: il, parseOptions: TestOptions.RegularPreview).VerifyDiagnostics(
            // (1,18): error CS8949: The InterpolatedStringHandlerArgumentAttribute applied to parameter 'InterpolationHandler h' is malformed and cannot be interpreted. Construct an instance of 'InterpolationHandler' manually.
            // 1.InstanceMethod($"");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerArgumentAttributeMalformed, @"$""""").WithArguments("InterpolationHandler h", "InterpolationHandler").WithLocation(1, 18),
            // (1,18): error CS7036: There is no argument given that corresponds to the required parameter 'param' of 'InterpolationHandler.InterpolationHandler(int, int, int)'
            // 1.InstanceMethod($"");
            Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, @"$""""").WithArguments("param", "InterpolationHandler.InterpolationHandler(int, int, int)").WithLocation(1, 18),
            // (1,18): error CS1615: Argument 3 may not be passed with the 'out' keyword
            // 1.InstanceMethod($"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, @"$""""").WithArguments("3", "out").WithLocation(1, 18),
            // (2,21): error CS8949: The InterpolatedStringHandlerArgumentAttribute applied to parameter 'InterpolationHandler h' is malformed and cannot be interpreted. Construct an instance of 'InterpolationHandler' manually.
            // E.InstanceMethod(1, $"");
            Diagnostic(ErrorCode.ERR_InterpolatedStringHandlerArgumentAttributeMalformed, @"$""""").WithArguments("InterpolationHandler h", "InterpolationHandler").WithLocation(2, 21),
            // (2,21): error CS7036: There is no argument given that corresponds to the required parameter 'param' of 'InterpolationHandler.InterpolationHandler(int, int, int)'
            // E.InstanceMethod(1, $"");
            Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, @"$""""").WithArguments("param", "InterpolationHandler.InterpolationHandler(int, int, int)").WithLocation(2, 21),
            // (2,21): error CS1615: Argument 3 may not be passed with the 'out' keyword
            // E.InstanceMethod(1, $"");
            Diagnostic(ErrorCode.ERR_BadArgExtraRef, @"$""""").WithArguments("3", "out").WithLocation(2, 21)
        );
    }

    [Fact]
    public void InterpolationHandler_Indexer_InObjectInitializer()
    {
        var code = """
            _ = new C() { [$"{1}"] = 1 };

            public class C
            {
            }

            [System.Runtime.CompilerServices.InterpolatedStringHandler]
            public struct CustomHandler
            {
                public CustomHandler(int literalLength, int formattedCount, C c)
                {
                }

                public void AppendFormatted(int i) {}
            }

            public static class CExt
            {
                extension(C c)
                {
                    public int this[[System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("c")] CustomHandler h]
                    {
                        get => throw null!;
                    }
                }
            }
            """;

        var comp = CreateCompilation(code, targetFramework: TargetFramework.NetCoreApp);

        // When extension indexers are supported, this should start reporting ERR_InterpolatedStringsReferencingInstanceCannotBeInObjectInitializers,
        // instead of the current errors.
        comp.VerifyDiagnostics(
            // (1,15): error CS0021: Cannot apply indexing with [] to an expression of type 'C'
            // _ = new C() { [$"{1}"] = 1 };
            Diagnostic(ErrorCode.ERR_BadIndexLHS, @"[$""{1}""]").WithArguments("C").WithLocation(1, 15),
            // (21,20): error CS9282: This member is not allowed in an extension block
            //         public int this[[System.Runtime.CompilerServices.InterpolatedStringHandlerArgument("c")] CustomHandler h]
            Diagnostic(ErrorCode.ERR_ExtensionDisallowsMember, "this").WithLocation(21, 20)
        );
    }

    [Fact]
    public void LiteralReceiver_Property_Enum_Set()
    {
        var src = """
Enum.Zero.Property = 1;

enum Enum
{
    Zero
}

static class E
{
    extension(Enum e)
    {
        public int Property { set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        // Consider improving the error message
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0131: The left-hand side of an assignment must be a variable, property or indexer
            // Enum.Zero.Property = 1;
            Diagnostic(ErrorCode.ERR_AssgLvalueExpected, "Enum.Zero.Property").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "Enum.Zero.Property");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["System.Int32 E.<G>$5BDAAC939B0896D4F1349316F7C8CE0F.Property { set; }"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.NotAVariable, model.GetSymbolInfo(memberAccess).CandidateReason);
        Assert.Equal([], model.GetMemberGroup(memberAccess).ToTestDisplayStrings()); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : handle GetMemberGroup on a property access
    }

    [Fact]
    public void LiteralReceiver_Property_Integer_ForLong()
    {
        var src = """
1.Property = 42;
_ = 2.Property;

static class E
{
    extension(long l)
    {
        public int Property { get { System.Console.Write("get "); return 42; } set { System.Console.Write("set "); }  }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS9286: 'int' does not contain a definition for 'Property' and no accessible extension member 'Property' for receiver of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // 1.Property = 42;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "1.Property").WithArguments("int", "Property").WithLocation(1, 1),
            // (2,5): error CS9286: 'int' does not contain a definition for 'Property' and no accessible extension member 'Property' for receiver of type 'int' could be found (are you missing a using directive or an assembly reference?)
            // _ = 2.Property;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "2.Property").WithArguments("int", "Property").WithLocation(2, 5));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "1.Property");
        AssertEx.Equal("System.Int32 E.<G>$E8CA98ACBCAEE63BB261A3FD4AF31675.Property { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess1).CandidateReason);

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "2.Property");
        AssertEx.Equal("System.Int32 E.<G>$E8CA98ACBCAEE63BB261A3FD4AF31675.Property { get; set; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess2).CandidateReason);
    }

    [Fact]
    public void LiteralReceiver_Property_String()
    {
        var src = """
"".Property = 42;
_ = "".Property;

static class E
{
    extension(string s)
    {
        public int Property { get { System.Console.Write("get "); return 42; } set { System.Console.Write("set "); }  }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "set get").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "\"\".Property").First();
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "\"\".Property").Last();
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; set; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void SwitchReceiver_Property_String()
    {
        var src = """
bool b = true;
(b switch { true => "", _ => "" }).Property = 42;
_ = (b switch { true => "", _ => "" }).Property;

static class E
{
    extension(string s)
    {
        public int Property { get { System.Console.Write("get "); return 42; } set { System.Console.Write("set "); }  }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "set get").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, """(b switch { true => "", _ => "" }).Property""").First();
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, """(b switch { true => "", _ => "" }).Property""").Last();
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; set; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void ConditionalReceiver_Property_String()
    {
        var src = """
bool b = true;
(b ? "" : null).Property = 42;
_ = (b ? "" : null).Property;

static class E
{
    extension(string s)
    {
        public int Property { get { System.Console.Write("get "); return 42; } set { System.Console.Write("set "); }  }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "set get").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, """(b ? "" : null).Property""").First();
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, """(b ? "" : null).Property""").Last();
        AssertEx.Equal("System.Int32 E.<G>$34505F560D9EACF86A87F3ED1F85E448.Property { get; set; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void LiteralReceiver_Property_Integer_Get()
    {
        var src = """
_ = 1.Property;

static class E
{
    extension(int i)
    {
        public int Property { get { System.Console.Write("get"); return 42; } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "get").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "1.Property");
        AssertEx.Equal("System.Int32 E.<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69.Property { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void LiteralReceiver_Property_Integer_Set()
    {
        var src = """
1.Property = 1;

static class E
{
    extension(int i)
    {
        public int Property { set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        // Consider improving the error message
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0131: The left-hand side of an assignment must be a variable, property or indexer
            // 1.Property = 1;
            Diagnostic(ErrorCode.ERR_AssgLvalueExpected, "1.Property").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "1.Property");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["System.Int32 E.<G>$BA41CFE2B5EDAEB8C1B9062F59ED4D69.Property { set; }"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.NotAVariable, model.GetSymbolInfo(memberAccess).CandidateReason);
    }

    [Fact]
    public void LiteralReceiver_Property_Null()
    {
        var src = """
null.Property = 1;
_ = null.Property;

static class E
{
    extension(object o)
    {
        public int Property { get => throw null; set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0023: Operator '.' cannot be applied to operand of type '<null>'
            // null.Property = 1;
            Diagnostic(ErrorCode.ERR_BadUnaryOp, "null.Property").WithArguments(".", "<null>").WithLocation(1, 1),
            // (2,5): error CS0023: Operator '.' cannot be applied to operand of type '<null>'
            // _ = null.Property;
            Diagnostic(ErrorCode.ERR_BadUnaryOp, "null.Property").WithArguments(".", "<null>").WithLocation(2, 5));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "null.Property").First();
        Assert.Null(model.GetSymbolInfo(memberAccess1).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess1).CandidateReason);

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "null.Property").Last();
        Assert.Null(model.GetSymbolInfo(memberAccess2).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess2).CandidateReason);
    }

    [Fact]
    public void LiteralReceiver_Property_Default()
    {
        var src = """
default.Property = 1;
_ = default.Property;

static class E
{
    extension(object o)
    {
        public int Property { get => throw null; set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS8716: There is no target type for the default literal.
            // default.Property = 1;
            Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(1, 1),
            // (2,5): error CS8716: There is no target type for the default literal.
            // _ = default.Property;
            Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(2, 5));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "default.Property").First();
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess1).CandidateReason);

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "default.Property").Last();
        AssertEx.Equal("System.Int32 E.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.Property { get; set; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess2).CandidateReason);
    }

    [Fact]
    public void LiteralReceiver_Property_Tuple_Get()
    {
        var src = """
_ = (1, 2).Property;

static class E
{
    extension((int, int) t)
    {
        public int Property { get { System.Console.Write("get "); return 42; } }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "get").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "(1, 2).Property");
        AssertEx.Equal("System.Int32 E.<G>$49AAF2D3C1326E88AED3848611C299DA.Property { get; }", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
    }

    [Fact]
    public void LiteralReceiver_Property_Tuple_Set()
    {
        var src = """
(1, 2).Property = 1;

static class E
{
    extension((int, int) t)
    {
        public int Property { set { System.Console.Write($"set(value)"); }}
    }
}
""";
        var comp = CreateCompilation(src);
        // Consider improving the error message
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0131: The left-hand side of an assignment must be a variable, property or indexer
            // (1, 2).Property = 1;
            Diagnostic(ErrorCode.ERR_AssgLvalueExpected, "(1, 2).Property").WithLocation(1, 1));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "(1, 2).Property");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["System.Int32 E.<G>$49AAF2D3C1326E88AED3848611C299DA.Property { set; }"], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.NotAVariable, model.GetSymbolInfo(memberAccess).CandidateReason);
    }

    [Fact]
    public void LiteralReceiver_Property_Tuple_Default()
    {
        var src = """
(default, default).Property = 1;
_ = (default, default).Property;

static class E
{
    extension((object, object) t)
    {
        public int Property { get => throw null; set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,2): error CS8716: There is no target type for the default literal.
            // (default, default).Property = 1;
            Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(1, 2),
            // (1,11): error CS8716: There is no target type for the default literal.
            // (default, default).Property = 1;
            Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(1, 11),
            // (2,6): error CS8716: There is no target type for the default literal.
            // _ = (default, default).Property;
            Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(2, 6),
            // (2,15): error CS8716: There is no target type for the default literal.
            // _ = (default, default).Property;
            Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(2, 15));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "(default, default).Property").First();
        Assert.Null(model.GetSymbolInfo(memberAccess1).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess1).CandidateReason);

        var memberAccess2 = GetSyntaxes<MemberAccessExpressionSyntax>(tree, "(default, default).Property").Last();
        Assert.Null(model.GetSymbolInfo(memberAccess2).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess2).CandidateReason);
    }

    [Fact]
    public void LiteralReceiver_Property_Tuple_Integer_ForLong()
    {
        var src = """
(1, 1).Property = 1;
_ = (2, 2).Property;

static class E
{
    extension((long, long) t)
    {
        public int Property { get => throw null; set => throw null; }
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS9286: '(int, int)' does not contain a definition for 'Property' and no accessible extension member 'Property' for receiver of type '(int, int)' could be found (are you missing a using directive or an assembly reference?)
            // (1, 1).Property = 1;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "(1, 1).Property").WithArguments("(int, int)", "Property").WithLocation(1, 1),
            // (2,5): error CS9286: '(int, int)' does not contain a definition for 'Property' and no accessible extension member 'Property' for receiver of type '(int, int)' could be found (are you missing a using directive or an assembly reference?)
            // _ = (2, 2).Property;
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "(2, 2).Property").WithArguments("(int, int)", "Property").WithLocation(2, 5));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "(1, 1).Property");
        AssertEx.Equal("System.Int32 E.<G>$8477960720B8106C28CEADF5CDF3A674.Property { get; set; }", model.GetSymbolInfo(memberAccess1).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess1).CandidateReason);

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "(2, 2).Property");
        AssertEx.Equal("System.Int32 E.<G>$8477960720B8106C28CEADF5CDF3A674.Property { get; set; }", model.GetSymbolInfo(memberAccess2).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        Assert.Equal(CandidateReason.None, model.GetSymbolInfo(memberAccess2).CandidateReason);
    }

    [Fact]
    public void PreferMoreSpecific_Static_MethodAndProperty()
    {
        var src = """
System.Console.Write(object.M);

static class E1
{
    extension(object)
    {
        public static string M() => throw null;
    }
}

static class E2
{
    extension(object)
    {
        public static string M => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,22): error CS9286: 'object' does not contain a definition for 'M' and no accessible extension member 'M' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // System.Console.Write(object.M);
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "object.M").WithArguments("object", "M").WithLocation(1, 22));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);
        AssertEx.SequenceEqual(["System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", "System.String E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"],
            model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Empty(model.GetMemberGroup(memberAccess)); // Tracked by https://github.com/dotnet/roslyn/issues/78957 : public API, consider handling BoundBadExpression better
    }

    [Fact]
    public void PreferMoreSpecific_Static_MethodAndProperty_Generic()
    {
        var src = """
System.Console.Write(object.M);

static class E1
{
    extension<T>(T)
    {
        public static string M() => throw null;
    }
}

static class E2
{
    extension<T>(T)
    {
        public static string M => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,22): error CS9286: 'object' does not contain a definition for 'M' and no accessible extension member 'M' for receiver of type 'object' could be found (are you missing a using directive or an assembly reference?)
            // System.Console.Write(object.M);
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "object.M").WithArguments("object", "M").WithLocation(1, 22));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        Assert.Null(model.GetSymbolInfo(memberAccess).Symbol);

        // Tracked by https://github.com/dotnet/roslyn/issues/78957 : public API, consider handling BoundBadExpression better
        AssertEx.SequenceEqual(["System.String E1.<G>$8048A6C8BE30A622530249B904B537EB<T>.M()", "System.String E2.<G>$8048A6C8BE30A622530249B904B537EB<T>.M { get; }"],
            model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        Assert.Empty(model.GetMemberGroup(memberAccess));
    }

    [Fact]
    public void PreferMoreSpecific_Static_MethodAndProperty_Invocation()
    {
        var src = """
System.Console.Write(object.M());

static class E1
{
    extension(object)
    {
        public static string M() => "ran";
    }
}

static class E2
{
    extension(object)
    {
        public static string M => throw null; // not invocable
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "ran").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var memberAccess = GetSyntax<MemberAccessExpressionSyntax>(tree, "object.M");
        AssertEx.Equal("System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", model.GetSymbolInfo(memberAccess).Symbol.ToTestDisplayString());
        Assert.Equal([], model.GetSymbolInfo(memberAccess).CandidateSymbols.ToTestDisplayStrings());
        AssertEx.SequenceEqual(["System.String E1.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M()", "System.String E2.<G>$C43E2675C7BBF9284AF22FB8A9BF0280.M { get; }"], model.GetMemberGroup(memberAccess).ToTestDisplayStrings());
    }

    [Fact]
    public void GetCompatibleExtensions_TwoSubstitutions()
    {
        var src = """
C.M();
new C().M2();

interface I<T> { }
class C : I<int>, I<string> { }

static class E
{
    extension<T>(I<T>)
    {
        public static void M() { }
    }

    public static void M2<T>(this I<T> i) { }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyDiagnostics(
            // (1,3): error CS1061: 'C' does not contain a definition for 'M' and no accessible extension method 'M' accepting a first argument of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // C.M();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M").WithArguments("C", "M").WithLocation(1, 3),
            // (2,9): error CS1061: 'C' does not contain a definition for 'M2' and no accessible extension method 'M2' accepting a first argument of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // new C().M2();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "M2").WithArguments("C", "M2").WithLocation(2, 9));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);

        var memberAccess1 = GetSyntax<MemberAccessExpressionSyntax>(tree, "C.M");
        Assert.Null(model.GetSymbolInfo(memberAccess1).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess1).CandidateSymbols.ToTestDisplayStrings());
        Assert.Empty(model.GetMemberGroup(memberAccess1));

        var memberAccess2 = GetSyntax<MemberAccessExpressionSyntax>(tree, "new C().M2");
        Assert.Null(model.GetSymbolInfo(memberAccess2).Symbol);
        Assert.Equal([], model.GetSymbolInfo(memberAccess2).CandidateSymbols.ToTestDisplayStrings());
        Assert.Empty(model.GetMemberGroup(memberAccess2));
    }

    [Theory, ClassData(typeof(ThreePermutationGenerator))]
    public void PreferMoreSpecific_Static_MethodAndMoreSpecificInvocablePropertyAndMoreSpecificMethod(int first, int second, int third)
    {
        string[] segments = [
            """
            static class E1
            {
                extension(object)
                {
                    public static string M() => throw null;
                }
            }
            """,
            """
            static class E2
            {
                extension(C)
                {
                    public static System.Func<string> M => null;
                }
            }
            """,
            """
            static class E3
            {
                extension(C)
                {
                    public static string M() => throw null;
                }
            }
            """];

        var src = $$"""
System.Console.Write(C.M());

class C { }

{{segments[first]}}

{{segments[second]}}

{{segments[third]}}
""";
        var comp = CreateCompilation(src);

        comp.VerifyEmitDiagnostics(
            // (1,22): error CS9286: 'C' does not contain a definition for 'M' and no accessible extension member 'M' for receiver of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // System.Console.Write(C.M());
            Diagnostic(ErrorCode.ERR_ExtensionResolutionFailed, "C.M").WithArguments("C", "M").WithLocation(1, 22));
    }

    [Fact]
    public void AmbiguousCallOnInterface()
    {
        var src = """
I2.M();

interface I<T>
{
    public static void M() { }
}

interface I2 : I<int>, I<string> { }
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        // Tracked by https://github.com/dotnet/roslyn/issues/78830 : diagnostic quality, consider improving the symbols in this error message
        comp.VerifyEmitDiagnostics(
            // (1,4): error CS0121: The call is ambiguous between the following methods or properties: 'I<T>.M()' and 'I<T>.M()'
            // I2.M();
            Diagnostic(ErrorCode.ERR_AmbigCall, "M").WithArguments("I<T>.M()", "I<T>.M()").WithLocation(1, 4));
    }

    [Fact]
    public void AmbiguousCallOnInterface_Generic()
    {
        var src = """
I2.M<int>();

interface I<T>
{
    public static void M<U>() { }
}

interface I2 : I<int>, I<string> { }
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (1,4): error CS0121: The call is ambiguous between the following methods or properties: 'I<T>.M<U>()' and 'I<T>.M<U>()'
            // I2.M<int>();
            Diagnostic(ErrorCode.ERR_AmbigCall, "M<int>").WithArguments("I<T>.M<U>()", "I<T>.M<U>()").WithLocation(1, 4));
    }

    [Fact]
    public void OmittedTypeArguments()
    {
        var src = """
object.P<int>;
object.P<>;

static class E
{
    extension(object)
    {
        public static int P => 42;
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net70);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
            // object.P<int>;
            Diagnostic(ErrorCode.ERR_IllegalStatement, "object.P<int>").WithLocation(1, 1),
            // (1,8): error CS0117: 'object' does not contain a definition for 'P'
            // object.P<int>;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "P<int>").WithArguments("object", "P").WithLocation(1, 8),
            // (2,1): error CS8389: Omitting the type argument is not allowed in the current context
            // object.P<>;
            Diagnostic(ErrorCode.ERR_OmittedTypeArgument, "object.P<>").WithLocation(2, 1),
            // (2,1): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
            // object.P<>;
            Diagnostic(ErrorCode.ERR_IllegalStatement, "object.P<>").WithLocation(2, 1),
            // (2,8): error CS0117: 'object' does not contain a definition for 'P'
            // object.P<>;
            Diagnostic(ErrorCode.ERR_NoSuchMember, "P<>").WithArguments("object", "P").WithLocation(2, 8));
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_NoMethod()
    {
        var src = """
foreach (var x in new C())
{
    System.Console.Write(x);
    break;
}

class C { }
class D { }

static class E
{
    extension(C c)
    {
        public D GetEnumerator() => new D();
    }
    extension(D d)
    {
        public bool MoveNext() => true;
        public int Current => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyDiagnostics(
            // (1,19): error CS0117: 'D' does not contain a definition for 'Current'
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_NoSuchMember, "new C()").WithArguments("D", "Current").WithLocation(1, 19),
            // (1,19): error CS0202: foreach requires that the return type 'D' of 'E.extension(C).GetEnumerator()' must have a suitable public 'MoveNext' method and public 'Current' property
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_BadGetEnumerator, "new C()").WithArguments("D", "E.extension(C).GetEnumerator()").WithLocation(1, 19)
            );

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).MoveNextMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).CurrentProperty);
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_NoApplicableMethod()
    {
        var src = """
foreach (var x in new C())
{
    System.Console.Write(x);
    break;
}

class C
{
    public void GetEnumerator(int notApplicable) { } // not applicable
}
class D { }

static class E
{
    extension(C c)
    {
        public D GetEnumerator() => new D();
    }
    extension(D d)
    {
        public bool MoveNext() => true;
        public int Current => 42;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,19): error CS0117: 'D' does not contain a definition for 'Current'
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_NoSuchMember, "new C()").WithArguments("D", "Current").WithLocation(1, 19),
            // (1,19): error CS0202: foreach requires that the return type 'D' of 'E.extension(C).GetEnumerator()' must have a suitable public 'MoveNext' method and public 'Current' property
            // foreach (var x in new C())
            Diagnostic(ErrorCode.ERR_BadGetEnumerator, "new C()").WithArguments("D", "E.extension(C).GetEnumerator()").WithLocation(1, 19)
            );

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).MoveNextMethod);
        Assert.Null(model.GetForEachStatementInfo(loop).CurrentProperty);
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_WrongArity()
    {
        var src = """
using System.Collections;

foreach (var x in new C()) { }

class C { }

static class E
{
    extension(C c)
    {
        public IEnumerator GetEnumerator<T>() => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): error CS0411: The type arguments for method 'E.extension(C).GetEnumerator<T>()' cannot be inferred from the usage. Try specifying the type arguments explicitly.
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_CantInferMethTypeArgs, "new C()").WithArguments("E.extension(C).GetEnumerator<T>()").WithLocation(3, 19),
            // (3,19): error CS1579: foreach statement cannot operate on variables of type 'C' because 'C' does not contain a public instance or extension definition for 'GetEnumerator'
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_ForEachMissingMember, "new C()").WithArguments("C", "GetEnumerator").WithLocation(3, 19));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_NonInvocable()
    {
        var src = """
using System.Collections;

foreach (var x in new C()) { }

class C { }

static class E
{
    extension(C c)
    {
        public IEnumerator GetEnumerator => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): error CS1579: foreach statement cannot operate on variables of type 'C' because 'C' does not contain a public instance or extension definition for 'GetEnumerator'
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_ForEachMissingMember, "new C()").WithArguments("C", "GetEnumerator").WithLocation(3, 19));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_DelegateTypeProperty()
    {
        var src = """
using System.Collections;

foreach (var x in new C()) { }

class C { }

static class E
{
    extension(C c)
    {
        public System.Func<IEnumerator> GetEnumerator => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): error CS1579: foreach statement cannot operate on variables of type 'C' because 'C' does not contain a public instance or extension definition for 'GetEnumerator'
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_ForEachMissingMember, "new C()").WithArguments("C", "GetEnumerator").WithLocation(3, 19));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);

        src = """
using System.Collections;

foreach (var x in new C()) { }

class C
{
    public System.Func<IEnumerator> GetEnumerator => throw null;
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): warning CS0280: 'C' does not implement the 'collection' pattern. 'C.GetEnumerator' has the wrong signature.
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.WRN_PatternBadSignature, "new C()").WithArguments("C", "collection", "C.GetEnumerator").WithLocation(3, 19),
            // (3,19): error CS1579: foreach statement cannot operate on variables of type 'C' because 'C' does not contain a public instance or extension definition for 'GetEnumerator'
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_ForEachMissingMember, "new C()").WithArguments("C", "GetEnumerator").WithLocation(3, 19));
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_GetEnumerator_DynamicTypeProperty()
    {
        var src = """
using System.Collections;

foreach (var x in new C()) { }

class C { }

static class E
{
    extension(C c)
    {
        public dynamic GetEnumerator => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,19): error CS1579: foreach statement cannot operate on variables of type 'C' because 'C' does not contain a public instance or extension definition for 'GetEnumerator'
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_ForEachMissingMember, "new C()").WithArguments("C", "GetEnumerator").WithLocation(3, 19));

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        Assert.Null(model.GetForEachStatementInfo(loop).GetEnumeratorMethod);

        src = """
foreach (var x in new C()) { }

class C
{
    public dynamic GetEnumerator => throw null;
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,19): warning CS0280: 'C' does not implement the 'collection' pattern. 'C.GetEnumerator' has the wrong signature.
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.WRN_PatternBadSignature, "new C()").WithArguments("C", "collection", "C.GetEnumerator").WithLocation(1, 19),
            // (1,19): error CS1579: foreach statement cannot operate on variables of type 'C' because 'C' does not contain a public instance or extension definition for 'GetEnumerator'
            // foreach (var x in new C()) { }
            Diagnostic(ErrorCode.ERR_ForEachMissingMember, "new C()").WithArguments("C", "GetEnumerator").WithLocation(1, 19));
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_ForEach_GetEnumerator_Generic()
    {
        var src = """
using System.Collections.Generic;

foreach (var x in new C()) { System.Console.Write(x); }

class C { }

static class E
{
    extension<T>(T t)
    {
        public IEnumerator<T> GetEnumerator()
        {
            yield return t;
        }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "C").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        AssertEx.Equal("System.Collections.Generic.IEnumerator<C> E.<G>$8048A6C8BE30A622530249B904B537EB<C>.GetEnumerator()",
            model.GetForEachStatementInfo(loop).GetEnumeratorMethod.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_AwaitForEach_GetAsyncEnumerator()
    {
        var src = """
using System.Collections.Generic;

await foreach (var x in new C()) { System.Console.Write(x); }

class C { }

static class E
{
    extension<T>(T t)
    {
        public async IAsyncEnumerator<T> GetAsyncEnumerator()
        {
            await System.Threading.Tasks.Task.Yield();
            yield return t;
        }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        CompileAndVerify(comp, expectedOutput: ExpectedOutput("C"), verify: Verification.FailsPEVerify).VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var loop = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().Single();
        AssertEx.Equal("System.Collections.Generic.IAsyncEnumerator<C> E.<G>$8048A6C8BE30A622530249B904B537EB<C>.GetAsyncEnumerator()",
            model.GetForEachStatementInfo(loop).GetEnumeratorMethod.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_NoMethod()
    {
        var src = """
var (x, y) = new C();
System.Console.Write((x, y));

class C { }

static class E
{
    extension(C c)
    {
        public void Deconstruct(out int i, out int j) { i = 42; j = 43; }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var deconstruction = tree.GetRoot().DescendantNodes().OfType<AssignmentExpressionSyntax>().First();

        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Deconstruct(out System.Int32 i, out System.Int32 j)",
            model.GetDeconstructionInfo(deconstruction).Method.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_Conversion_01()
    {
        var src = """
var (x, y) = new C();
System.Console.Write((x, y));

class C { }

static class E
{
    extension(object o)
    {
        public void Deconstruct(out int i, out int j) { i = 42; j = 43; }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78682")]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_Conversion_02()
    {
        // array to Span
        var src = """
var (x, y) = new int[] { 42 };
System.Console.Write((x, y));

class C { }

static class E
{
    extension(System.Span<int> s)
    {
        public void Deconstruct(out int i, out int j) { i = 42; j = 43; }
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        CompileAndVerify(comp, expectedOutput: ExpectedOutput("(42, 43)"), verify: Verification.Skipped).VerifyDiagnostics();

        src = """
var (x, y) = new int[] { 42 };
System.Console.Write((x, y));

static class E
{
    public static void Deconstruct(this System.Span<int> s, out int i, out int j) { i = 42; j = 43; }
}
""";
        comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        CompileAndVerify(comp, expectedOutput: ExpectedOutput("(42, 43)"), verify: Verification.Skipped).VerifyDiagnostics();
    }

    [Fact, WorkItem("https://github.com/dotnet/roslyn/issues/78682")]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_Conversion_03()
    {
        // We check conversion during initial binding
        var src = """
var (x, y) = new int[] { 42 };
System.Console.Write((x, y));

class C { }

static class E
{
    extension(System.Span<int> s)
    {
        public void Deconstruct(out int i, out int j) => throw null;
    }
}

namespace System
{
    public ref struct Span<T>
    {
    }
}
""";
        var comp = CreateCompilation(src, targetFramework: TargetFramework.Net90);
        comp.VerifyEmitDiagnostics(
            // (1,1): error CS0656: Missing compiler required member 'Span<T>.op_Implicit'
            // var (x, y) = new int[] { 42 };
            Diagnostic(ErrorCode.ERR_MissingPredefinedMember, "var (x, y) = new int[] { 42 }").WithArguments("System.Span<T>", "op_Implicit").WithLocation(1, 1),
            // (8,22): warning CS0436: The type 'Span<T>' in '' conflicts with the imported type 'Span<T>' in 'System.Runtime, Version=9.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a'. Using the type defined in ''.
            //     extension(System.Span<int> s)
            Diagnostic(ErrorCode.WRN_SameFullNameThisAggAgg, "Span<int>").WithArguments("", "System.Span<T>", "System.Runtime, Version=9.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", "System.Span<T>").WithLocation(8, 22));
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_Generic()
    {
        var src = """
var (x, y) = new C();
System.Console.Write((x, y));

class C { }

static class E
{
    extension<T>(T t)
    {
        public void Deconstruct(out int i, out int j) { i = 42; j = 43; }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var deconstruction = tree.GetRoot().DescendantNodes().OfType<AssignmentExpressionSyntax>().First();

        AssertEx.Equal("void E.<G>$8048A6C8BE30A622530249B904B537EB<C>.Deconstruct(out System.Int32 i, out System.Int32 j)",
            model.GetDeconstructionInfo(deconstruction).Method.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_FallbackToExtensionMethod()
    {
        // If the method from the extension type is not applicable, we fall back
        // to a Deconstruct extension method
        var src = """
var (x, y) = new C();
System.Console.Write((x, y));

public class C { }

static class E
{
    extension(C c)
    {
        public void Deconstruct(int inapplicable) => throw null;
    }
}

public static class E2
{
    public static void Deconstruct(this C c, out int i, out int j) { i = 42; j = 43; }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var deconstruction = tree.GetRoot().DescendantNodes().OfType<AssignmentExpressionSyntax>().First();

        AssertEx.Equal("void E2.Deconstruct(this C c, out System.Int32 i, out System.Int32 j)",
            model.GetDeconstructionInfo(deconstruction).Method.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_DelegateTypeProperty()
    {
        var src = """
var (x1, y1) = new C1();

var (x2, y2) = new C2();

class C1 { }

class C2
{
    public D Deconstruct => (out int i, out int j) => { i = 42; j = 43; };
}

delegate void D(out int i, out int j);

static class E
{
    extension(C1 c)
    {
        public D Deconstruct => (out int i, out int j) => { i = 42; j = 43; };
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyDiagnostics(
            // (1,6): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'x1'.
            // var (x1, y1) = new C1();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "x1").WithArguments("x1").WithLocation(1, 6),
            // (1,10): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'y1'.
            // var (x1, y1) = new C1();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "y1").WithArguments("y1").WithLocation(1, 10),
            // (1,16): error CS1061: 'C1' does not contain a definition for 'Deconstruct' and no accessible extension method 'Deconstruct' accepting a first argument of type 'C1' could be found (are you missing a using directive or an assembly reference?)
            // var (x1, y1) = new C1();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "new C1()").WithArguments("C1", "Deconstruct").WithLocation(1, 16),
            // (1,16): error CS8129: No suitable 'Deconstruct' instance or extension method was found for type 'C1', with 2 out parameters and a void return type.
            // var (x1, y1) = new C1();
            Diagnostic(ErrorCode.ERR_MissingDeconstruct, "new C1()").WithArguments("C1", "2").WithLocation(1, 16),
            // (3,6): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'x2'.
            // var (x2, y2) = new C2();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "x2").WithArguments("x2").WithLocation(3, 6),
            // (3,10): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'y2'.
            // var (x2, y2) = new C2();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "y2").WithArguments("y2").WithLocation(3, 10),
            // (3,16): error CS8129: No suitable 'Deconstruct' instance or extension method was found for type 'C2', with 2 out parameters and a void return type.
            // var (x2, y2) = new C2();
            Diagnostic(ErrorCode.ERR_MissingDeconstruct, "new C2()").WithArguments("C2", "2").WithLocation(3, 16)
            );

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var deconstruction = tree.GetRoot().DescendantNodes().OfType<AssignmentExpressionSyntax>().First();

        Assert.Null(model.GetDeconstructionInfo(deconstruction).Method);
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_DynamicProperty()
    {
        var src = """
var (x, y) = new C();

class C { }

static class E
{
    extension(C c)
    {
        public dynamic Deconstruct => throw null;
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,6): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'x'.
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "x").WithArguments("x").WithLocation(1, 6),
            // (1,9): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'y'.
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "y").WithArguments("y").WithLocation(1, 9),
            // (1,14): error CS1061: 'C' does not contain a definition for 'Deconstruct' and no accessible extension method 'Deconstruct' accepting a first argument of type 'C' could be found (are you missing a using directive or an assembly reference?)
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "new C()").WithArguments("C", "Deconstruct").WithLocation(1, 14),
            // (1,14): error CS8129: No suitable 'Deconstruct' instance or extension method was found for type 'C', with 2 out parameters and a void return type.
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_MissingDeconstruct, "new C()").WithArguments("C", "2").WithLocation(1, 14)
            );

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var deconstruction = tree.GetRoot().DescendantNodes().OfType<AssignmentExpressionSyntax>().First();

        Assert.Null(model.GetDeconstructionInfo(deconstruction).Method);

        src = """
var (x, y) = new C();

class C
{
    public dynamic Deconstruct => throw null;
}
""";
        comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (1,6): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'x'.
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "x").WithArguments("x").WithLocation(1, 6),
            // (1,9): error CS8130: Cannot infer the type of implicitly-typed deconstruction variable 'y'.
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_TypeInferenceFailedForImplicitlyTypedDeconstructionVariable, "y").WithArguments("y").WithLocation(1, 9),
            // (1,14): error CS8129: No suitable 'Deconstruct' instance or extension method was found for type 'C', with 2 out parameters and a void return type.
            // var (x, y) = new C();
            Diagnostic(ErrorCode.ERR_MissingDeconstruct, "new C()").WithArguments("C", "2").WithLocation(1, 14));
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Deconstruct_NoApplicableMethod()
    {
        var src = """
var (x, y) = new C();
System.Console.Write((x, y));

class C
{
    public void Deconstruct() { } // not applicable
}

static class E
{
    extension(C c)
    {
        public void Deconstruct(out int i, out int j) { i = 42; j = 43; }
    }
}
""";
        var comp = CreateCompilation(src);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        var tree = comp.SyntaxTrees.Single();
        var model = comp.GetSemanticModel(tree);
        var deconstruction = tree.GetRoot().DescendantNodes().OfType<AssignmentExpressionSyntax>().First();

        AssertEx.Equal("void E.<G>$9794DAFCCB9E752B29BFD6350ADA77F2.Deconstruct(out System.Int32 i, out System.Int32 j)",
            model.GetDeconstructionInfo(deconstruction).Method.ToTestDisplayString());
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_PositionalPattern()
    {
        var libSrc = """
public static class E
{
    extension<T>(T t)
    {
        public void Deconstruct(out int i, out int j) { i = 42; j = 43; }
    }
}
""";
        var libRef = CreateCompilation(libSrc).EmitToImageReference();

        var src = """
var c = new C();
if (c is var (x, y))
    System.Console.Write((x, y));

class C { }

""";
        var comp = CreateCompilation(src, references: [libRef]);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        comp = CreateCompilation(src, references: [libRef], parseOptions: TestOptions.Regular13);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();

        comp = CreateCompilation(src, references: [libRef], parseOptions: TestOptions.Regular14);
        CompileAndVerify(comp, expectedOutput: "(42, 43)").VerifyDiagnostics();
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_DisposeAsync_NoMethod()
    {
        var src = """
using System.Threading.Tasks;

/*<bind>*/
await using var x1 = new C1();
/*</bind>*/

await using var x2 = new C2();

class C1 { }
class C2 { }

static class E
{
    extension(C1 c)
    {
        public Task DisposeAsync() => throw null;
    }

    public static Task DisposeAsync(this C2 c) => throw null;
}
""";
        var comp = CreateCompilation(src);

        var expectedDiagnostics = new[] {
            // (4,1): error CS8410: 'C1': type used in an asynchronous using statement must implement 'System.IAsyncDisposable' or implement a suitable 'DisposeAsync' method.
            // await using var x1 = new C1();
            Diagnostic(ErrorCode.ERR_NoConvToIAsyncDisp, "await using var x1 = new C1();").WithArguments("C1").WithLocation(4, 1),
            // (7,1): error CS8410: 'C2': type used in an asynchronous using statement must implement 'System.IAsyncDisposable' or implement a suitable 'DisposeAsync' method.
            // await using var x2 = new C2();
            Diagnostic(ErrorCode.ERR_NoConvToIAsyncDisp, "await using var x2 = new C2();").WithArguments("C2").WithLocation(7, 1)
            };

        comp.VerifyDiagnostics(expectedDiagnostics);

        string expectedOperationTree = """
IUsingDeclarationOperation(IsAsynchronous: True) (OperationKind.UsingDeclaration, Type: null, IsInvalid) (Syntax: 'await using ... = new C1();')
  DeclarationGroup:
    IVariableDeclarationGroupOperation (1 declarations) (OperationKind.VariableDeclarationGroup, Type: null, IsInvalid, IsImplicit) (Syntax: 'await using ... = new C1();')
      IVariableDeclarationOperation (1 declarators) (OperationKind.VariableDeclaration, Type: null, IsInvalid) (Syntax: 'var x1 = new C1()')
        Declarators:
            IVariableDeclaratorOperation (Symbol: C1 x1) (OperationKind.VariableDeclarator, Type: null, IsInvalid) (Syntax: 'x1 = new C1()')
              Initializer:
                IVariableInitializerOperation (OperationKind.VariableInitializer, Type: null, IsInvalid) (Syntax: '= new C1()')
                  IObjectCreationOperation (Constructor: C1..ctor()) (OperationKind.ObjectCreation, Type: C1, IsInvalid) (Syntax: 'new C1()')
                    Arguments(0)
                    Initializer:
                      null
        Initializer:
          null
""";

        VerifyOperationTreeAndDiagnosticsForTest<LocalDeclarationStatementSyntax>(src, expectedOperationTree, expectedDiagnostics);
    }

    [Fact]
    public void TestPatternBasedDisposal_ExtensionMethod()
    {
        string source = @"
public class C
{
    public static async System.Threading.Tasks.Task<int> Main()
    {
        await using (var x = new C())
        {
        }

        return 1;
    }
}
public static class Extensions
{
    extension (C c)
    {
        public System.Threading.Tasks.ValueTask DisposeAsync()
            => throw null;
    }
}
";
        // extension methods do not contribute to pattern-based disposal
        var comp = CreateCompilationWithTasksExtensions(new[] { source, IAsyncDisposableDefinition }, options: TestOptions.DebugExe);
        comp.VerifyDiagnostics(
            // 0.cs(6,22): error CS8410: 'C': type used in an asynchronous using statement must implement 'System.IAsyncDisposable' or implement a suitable 'DisposeAsync' method.
            //         await using (var x = new C())
            Diagnostic(ErrorCode.ERR_NoConvToIAsyncDisp, "var x = new C()").WithArguments("C").WithLocation(6, 22)
            );
    }

    [Fact]
    public void PatternBased_Dispose_Async_DelegateTypeProperty()
    {
        var src = """
using System.Threading.Tasks;

await using var x = new C();

class C
{
    public System.Func<Task> DisposeAsync => async () => { System.Console.Write("ran2"); await Task.Yield(); };
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,1): error CS8410: 'C': type used in an asynchronous using statement must implement 'System.IAsyncDisposable' or implement a suitable 'DisposeAsync' method.
            // await using var x = new C();
            Diagnostic(ErrorCode.ERR_NoConvToIAsyncDisp, "await using var x = new C();").WithArguments("C").WithLocation(3, 1)
            );
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Dispose_Async_DelegateTypeProperty()
    {
        var src = """
using System.Threading.Tasks;

await using var x = new C();

class C { }

static class E
{
    extension(C c)
    {
        public System.Func<Task> DisposeAsync => async () => { await Task.Yield(); };
    }
}
""";
        var comp = CreateCompilation(src);
        comp.VerifyEmitDiagnostics(
            // (3,1): error CS8410: 'C': type used in an asynchronous using statement must implement 'System.IAsyncDisposable' or implement a suitable 'DisposeAsync' method.
            // await using var x = new C();
            Diagnostic(ErrorCode.ERR_NoConvToIAsyncDisp, "await using var x = new C();").WithArguments("C").WithLocation(3, 1)
            );
    }

    [Fact]
    public void ExtensionMemberLookup_PatternBased_Dispose_Async_NoApplicableMethod()
    {
        var src = """
using System.Threading.Tasks;

/*<bind>*/
await using var x = new C();
/*</bind>*/

class C
{
    public Task DisposeAsync(int notApplicable) => throw null; // not applicable
}

static class E
{
    extension(C c)
    {
        public async Task DisposeAsync()
        {
            System.Console.Write("RAN");
            await Task.Yield();
        }
    }
}
""";
        DiagnosticDescription[] expectedDiagnostics = [
            // (4,1): error CS8410: 'C': type used in an asynchronous using statement must implement 'System.IAsyncDisposable' or implement a suitable 'DisposeAsync' method.
            // await using var x = new C();
            Diagnostic(ErrorCode.ERR_NoConvToIAsyncDisp, "await using var x = new C();").WithArguments("C").WithLocation(4, 1)];

        var comp = CreateCompilation(src).VerifyEmitDiagnostics(expectedDiagnostics);

        string expectedOperationTree = """
IUsingDeclarationOperation(IsAsynchronous: True) (OperationKind.UsingDeclaration, Type: null, IsInvalid) (Syntax: 'await using ...  = new C();')
  DeclarationGroup:
    IVariableDeclarationGroupOperation (1 declarations) (OperationKind.VariableDeclarationGroup, Type: null, IsInvalid, IsImplicit) (Syntax: 'await using ...  = new C