/* 
  Copyright (C) 2008 Kai Hertel

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#ifdef __APPLE__
#include <sys/sysctl.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <netdb.h>
#include <pwd.h>
#include <assert.h>
#include <pthread.h>
#include <signal.h>
#include <errno.h>
#include <netinet/tcp.h>
#include <netinet/ip.h>
#include <fcntl.h>
#include <sys/resource.h>
#include "addrtable.h"
#include "thread.h"
#include "lib/message.h"
#include "lib/game.h"
#include "config.h"

#define BCAST_INTERVAL_MIN 10000
#define DEFAULT_BACKLOG 128

static struct config {
	// (-1) == unlimited
	in_addr_t addr;
	unsigned short port;
	char device[IF_NAMESIZE];
	int n_threads;
	int max_connect; 	// concurrent connections
	int max_perip; 		// prevent flooding
	int max_payload; 	// kbytes/sec
	int client_timeout; 	// msec
	int min_rttm_interval; 	// minimum interval between synchronous two-way application-layer communication (`ACK`) (msec)
	short daemonize;
	short verbose;
	enum gamemode gmode;
	enum gamepadprofile gpads;
} conf= {
	.addr= 0x00000000l, .port= 1212,
	.device= { '\x0' },
	.n_threads= 1,
	.max_connect= (-1),
	.max_perip= 5,
	.max_payload= (-1),
	.min_rttm_interval= 8000,
	.client_timeout= 5000,
	.daemonize= 0,
	.verbose= 0,
	.gmode= gamemode_linear,
	.gpads= gamepadprofile_flat
};


// communicate with worker threads
short volatile signal_exiting= 0;
unsigned bcast_interval= BCAST_INTERVAL_MIN;
unsigned rttm_min= 0;
struct gameplay_public game;
pthread_rwlock_t gsync;
int *insert_socket= NULL;
in_addr_t *insert_peer= NULL;
pthread_mutex_t *isync= NULL;
unsigned * volatile n_connects= NULL;
float * volatile sums= NULL;
float * volatile variances= NULL;
pthread_rwlock_t statsync;
short n_teams;
short verbosity; 	// not quite satisfied with this one yet


// prototypes
void sigterm_handler(int);
static void parse_options(int, char **, struct config *);
static int setup_listening_socket(const struct config *);
static int setup_peer_socket(int, const struct config *);
static int drop_privileges(void);
static int collect_worker_threads(pthread_t *);
static void free_globals(void);
static void destroy_rwlocks(void);
static void destroy_mutexes(void);
static unsigned calc_bcast(int, unsigned, int); 	// returns appropriate interval measured in microseconds
static int update_players(struct gameplay *, int, unsigned *, float *, float *);
static int peer_cnt_increase(unsigned *);



int main(argc, argv)
int argc;
char **argv;
{
	parse_options(argc, argv, &conf);

	if (conf.daemonize) {
		pid_t pid= fork();
		if (pid != 0) { 	// parent
			if (pid == (-1)) {
				perror("fork()");
				exit (-1);
			}
			else {
				printf("Forked to background (pid == %d).\n", pid);
				exit(0);
			}
		}
	}

	if (!geteuid()) { 	// super-user -> set socket limit
		struct rlimit rlim= {
			.rlim_cur= RLIM_INFINITY,
			.rlim_max= RLIM_INFINITY
		};
		if (conf.max_connect >0) {
			rlim.rlim_cur= conf.max_connect +5; 	// managing socket, stdin, stdout, stderr + one to spare
			rlim.rlim_max= rlim.rlim_cur +2;
		}
		if (setrlimit(RLIMIT_NOFILE, &rlim) == (-1)) {
			perror("setrlimit()");
			if (errno != EPERM) exit(-1);
		}
	}
	int sock= setup_listening_socket(&conf);
	if (sock <0) {
		exit (sock);
	}

	int condition= drop_privileges();
	if (condition <0) {
		close(sock);
		exit(condition);
	}
	if ((condition == 0) && (conf.verbose))
		printf("Dropped user privileges.\n");

	// set signal handler
	struct sigaction termact= { .sa_handler= sigterm_handler, .sa_flags= SA_RESETHAND };
	sigemptyset(&termact.sa_mask);
	if ( (sigaction(SIGTERM, &termact, NULL)) ||
		 (termact.sa_flags= SA_RESTART, sigaction(SIGPIPE, &termact, NULL)) ) {
		fprintf(stderr, "Error initiating signal handler.\n");
		close(sock);
		exit(-1);
	}

	// spawn
	if (conf.n_threads < 0) { 	// auto
		int ncpu=0;
#ifdef __APPLE__
		int mib[2] = { CTL_HW, HW_NCPU };
		size_t len=sizeof(ncpu);
		sysctl(mib, 2, &ncpu, &len, NULL, 0);
#else
		ncpu = sysconf(_SC_NPROCESSORS_CONF);
#endif
		conf.n_threads = (ncpu > 0 ? ncpu : 1);
		if (conf.verbose) printf("Utilizing %d threads.\n", conf.n_threads);
	}
	assert(conf.n_threads >0);

	if (pthread_rwlock_init(&gsync, NULL) ||
	    pthread_rwlock_init(&statsync, NULL)) {
		close(sock);
		fprintf(stderr, "Error initializing rwlock objects.\n");
		exit(-1);
	}

	if (atexit(destroy_rwlocks)) {
		close(sock);
		fprintf(stderr, "Error adding an exit() handler.\n");
		if ( (pthread_rwlock_destroy(&gsync)) ||
		     (pthread_rwlock_destroy(&statsync)) )
			fprintf(stderr, "Error destroying rwlock objects.\n");
		exit(-1);
	}

	if (atexit(free_globals)) {
		close(sock);
		fprintf(stderr, "Error adding an exit() handler.\n");
		exit(-2);
	}

	n_teams= 2; 	// fixed value for now
	verbosity= conf.verbose;
	if (conf.min_rttm_interval >0) rttm_min= (unsigned)conf.min_rttm_interval;
	insert_socket= calloc(conf.n_threads, sizeof(int));
	if (conf.max_perip > 0) insert_peer= calloc(conf.n_threads, sizeof(in_addr_t));
	isync= calloc(conf.n_threads, sizeof(pthread_mutex_t));
	n_connects= calloc(conf.n_threads * n_teams, sizeof(unsigned));
	sums= calloc(conf.n_threads * n_teams, sizeof(unsigned));
	variances= calloc(conf.n_threads * n_teams, sizeof(unsigned));
	pthread_t *threads= calloc(conf.n_threads +1, sizeof(pthread_t)); 	// NULL-terminated list
	assert(n_connects && insert_socket && ((conf.max_perip <= 0) || insert_peer) && isync && sums && variances && threads);

	pthread_mutexattr_t mutexattr;
	if (pthread_mutexattr_init(&mutexattr)) {
		free(threads);
		close(sock);
		fprintf(stderr, "Error initializing mutex attributes.\n");
		exit(-1);
	}

	for (int tnum= 0; tnum< conf.n_threads; tnum++) {
		short half= 0;
		if ( (pthread_mutex_init(isync+tnum, &mutexattr) && 	// init mutex
		      fprintf(stderr, "Error creating mutex objects.\n")) ||
			  (half++) ||
		     (pthread_create(threads+tnum, NULL, (void *(*)(void *))worker_thread, insert_socket + tnum) && 	// start thread
		      fprintf(stderr, "Error spawning worker threads.\n")) ) {
			memset(threads+tnum, 0, sizeof(pthread_t));
			collect_worker_threads(threads);
			free(threads);
			close(sock);
			if (pthread_mutexattr_destroy(&mutexattr))
				fprintf(stderr, "Error destroying mutex attributes.\n");
			for (int idx= 0; idx< tnum +half; idx++)
				if (pthread_mutex_destroy(isync+idx))
					fprintf(stderr, "Error destroying mutex object.\n");
			exit(-1);
		}
	}

	if (atexit(destroy_mutexes)) {
		collect_worker_threads(threads);
		free(threads);
		close(sock);
		fprintf(stderr, "Error adding an exit() handler.\n");
		exit(-3);
	}

	if ( (conf.max_perip >0) &&
		 (!conf.verbose || printf("Initializing Flood Protection.\n")) &&
		 addrtable_init(conf.max_connect, &mutexattr))
		fprintf(stderr, "Error allocating hash map, dropping flood protection.\n");

	if (pthread_mutexattr_destroy(&mutexattr)) {
		collect_worker_threads(threads);
		free(threads);
		close(sock);
		if (addrtable_is_active()) addrtable_destroy(1);
		fprintf(stderr, "Error destroying mutex attributes.\n");
		exit(-2);
	}

	// listen
	if (listen(sock, (conf.max_connect >0)? (conf.max_connect+19)/20 : DEFAULT_BACKLOG) == (-1)) {
		collect_worker_threads(threads);
		free(threads);
		close(sock);
		if (addrtable_is_active()) addrtable_destroy(1);
		perror("listen()");
		exit(-1);
	}

	// begin game
	struct timeval cur_time;
	gettimeofday(&cur_time, NULL);
	srand(cur_time.tv_sec);
	struct gameplay igamelocal;
	if (gameplay_init(&igamelocal, sizeof(struct gameplay), conf.gmode, conf.gpads) != PONG_SUCCESS) {
		fprintf(stderr, "Compile server against new lib version headers.\n");
		collect_worker_threads(threads);
		free(threads);
		close(sock);
		if (addrtable_is_active()) addrtable_destroy(1);
		exit(-1);
	}
	netmessage_scrambler_init();

	// accept loop
	condition= 0;
	for (;!signal_exiting;) {
		fd_set rfds;
		FD_ZERO(&rfds);
		FD_SET(sock, &rfds);
		struct timeval tv_select= { .tv_sec= 0, .tv_usec= 200 };

		// remain responsive to signals
		// note: don't rely on the value of tv_select after the call
		int retselect = select(sock+1, &rfds, NULL, NULL, &tv_select);

		// update game progress and broadcast interval
		unsigned cur_connects= 0, emptyteam= 0;
		for (int teamidx= 0; teamidx< n_teams; teamidx++) {
			unsigned teamcount= 0;
			for (int threadidx= 0; threadidx< conf.n_threads; threadidx++)
				teamcount+= n_connects[threadidx * n_teams + teamidx];
			if (teamcount)
				cur_connects+= teamcount;
			else
				emptyteam++;
		}
		bcast_interval= calc_bcast(conf.n_threads, cur_connects, conf.max_payload);
		if (emptyteam) {
			// wait for players
			if (game.status != gamestatus_onhold) {
				if (pthread_rwlock_wrlock(&gsync))
					fprintf(stderr, "Cannot acquire write lock.\n");

				game.status= gamestatus_onhold;
				if (pthread_rwlock_unlock(&gsync))
					fprintf(stderr, "Cannot release write lock.\n");
			}
		}
		if (cur_connects) {
			// semantics on this lock may seem strange, but make perfect sense, since write operations are non-overlapping
			if (pthread_rwlock_wrlock(&statsync))
				fprintf(stderr, "Cannot acquire write lock.\n");

			update_players(&igamelocal, conf.n_threads, n_connects, sums, variances);
			if (pthread_rwlock_unlock(&statsync))
				fprintf(stderr, "Cannot release write lock.\n");
		}
		if (!emptyteam) {
			gettimeofday(&cur_time, NULL);
			if ((condition= gameplay_update(&igamelocal, &cur_time)) != PONG_SUCCESS)
				fprintf(stderr, "Failed to update the game state (%d).\n", condition);
			condition= 0;
		}
		if ((cur_connects)||(igamelocal.status != game.status)) {
			// publish updated state
			struct gameplay_public gamelocal;
			gameplay_internal_to_public(&igamelocal, &gamelocal);

			if (pthread_rwlock_wrlock(&gsync))
				fprintf(stderr, "Cannot acquire write lock.\n");

			memcpy(&game, &gamelocal, sizeof(struct gameplay_public));

			if (pthread_rwlock_unlock(&gsync))
				fprintf(stderr, "Cannot release write lock.\n");
		}

		if (!retselect) continue; 	// timeout condition
		if (retselect == (-1)) {
			perror("select()");
			if (errno == EINTR) continue; 	// signal caught
			condition= 1;
			goto leave;
		}
		// FD_ISSET(sock, &rfds) will be true

		struct sockaddr_in peer;
		unsigned peerlen= sizeof(peer);
		if (verbosity) printf("Client knocking.\n");
		int intro= accept(sock, (struct sockaddr *) &peer, &peerlen);
		assert(peerlen == sizeof(peer));
		if (intro == (-1)) {
			if (errno == EINTR) continue;
			perror("accept()");
			if ((errno == EMFILE) || (errno == ENFILE)) continue; 	// too many open files
			condition= 1;
			goto leave;
		}
		if (verbosity >1) printf("Considering client [%d]\n", intro);

		// flood checks
		int flood= ((conf.max_connect >0) && (cur_connects >= conf.max_connect));

		if ((flood == 0) && addrtable_is_active() && (getpeername(intro, (struct sockaddr *) &peer, &peerlen) == (-1))) {
			perror("getpeername()");
			fprintf(stderr, "Shutting down flood protection.\n");
			addrtable_destroy(0);
		}

		if ((flood == 0) && addrtable_is_active()) {
			int atomic= addrtable_atomic(peer, peer_cnt_increase, NULL);
			if (atomic< 0) {
				fprintf(stderr, "Error with address map, disabling flood protection.\n");
				addrtable_destroy(0);
			}
			if (atomic> 0) flood= 1;
		}
		if (flood) {
			char *msg= "No more connections";
			netmessage_send(intro, NETMSG_KICK, msg, strlen(msg) +1, NULL);
			close(intro);
			continue;
		}

		if (verbosity) {
			char namebuf[INET_ADDRSTRLEN];
			const char *name= inet_ntop(AF_INET, &peer.sin_addr, &namebuf[0], INET_ADDRSTRLEN);
			printf("Accepted client%s%s\n", (name)?" ":".", (name)?name:"");
		}
		if (setup_peer_socket(intro, &conf)) {
			close(intro);
			continue;
		}

		// assign a worker thread to the client
		int load= 0; 	// determine the best-suited one (measured by client load)
		unsigned lcon= 0;
		for (int idx= 0; idx< conf.n_threads; idx++) {
			if (idx * n_teams > cur_connects) break; 	// resolves team assignment issues for very small numbers of peers
			unsigned ccon= 0;
			for (int tm= 0; tm< n_teams; tm++)
				ccon+= n_connects[idx * n_teams + tm];
			if (!idx) lcon= ccon;
			if (ccon < lcon) {
				load= idx;
				lcon= ccon;
			}
		}
		for (int greed= 0; greed< 3; greed++) {
			for (int idx= 0; idx< conf.n_threads; idx++) 	// round-robin
				if ((insert_socket[(idx + load) % conf.n_threads] == 0)||(greed == 2)) {
					if (greed) {
						if (greed == 2)
							while (insert_socket[ (idx + load) % conf.n_threads ] != 0)
								usleep(5000);
						if (pthread_mutex_lock( isync+ ((idx + load) % conf.n_threads) )) {
							fprintf(stderr, "Error acquiring mutex lock.\n");
							condition= 1;
							goto leave;
						}
					}
					else if (pthread_mutex_trylock( isync+ ((idx + load) % conf.n_threads) )) continue; 	// minimize blocking occurrences
					if (verbosity >1) printf("Inserting client [%d] into thread %d:%d\n", intro,
						(idx + load) % conf.n_threads, insert_socket[ (idx + load) % conf.n_threads ]);
					insert_socket[ (idx + load) % conf.n_threads ]= intro;
					if (insert_peer) insert_peer[ (idx + load) % conf.n_threads ]= peer.sin_addr.s_addr;
					if (pthread_mutex_unlock( isync+((idx + load) % conf.n_threads) ))
						fprintf(stderr, "Error releasing mutex lock.\n");
					intro= (-1);
					break;
				}
			if (intro == (-1)) break;
		}
		if (intro != (-1)) {
			fprintf(stderr, "Fault: No worker threads available.\n");
			condition= 1;
			goto leave;
		}
	}

	if (verbosity) printf("Exiting on behalf of SIGTERM signal.\n");
leave:
	collect_worker_threads(threads);
	free(threads);
	close(sock);
	addrtable_destroy(1);
	return condition;
}



void sigterm_handler(signal)
int signal;
{
	if ((signal == SIGPIPE)&&(conf.verbose)) printf("Caught SIGPIPE.\n");
	if (signal == SIGTERM)
		signal_exiting= 1;
}



static void parse_options(argc, argv, conf)
int argc;
char **argv;
struct config *conf;
{
	const char *short_options= "a:bc:d:hi:l:m:n:p:r:t:Vv";
	const struct option long_options[]= {
		{ "model", required_argument, NULL, 'm' },
		{ "paddles", required_argument, NULL, 'r' },
		{ "address", required_argument, NULL, 'a' },
		{ "port", required_argument, NULL, 'p' },
		{ "dev", required_argument, NULL, 'd' },
		{ "background", no_argument, NULL, 'b' },
		{ "nthreads", required_argument, NULL, 'n' },
		{ "max-connect", required_argument, NULL, 'c' },
		{ "max-per-ip", required_argument, NULL, 'i' },
		{ "max-payload", required_argument, NULL, 'l' },
		{ "min-rttm-interval", required_argument, NULL, 's' },
		{ "client-timeout", required_argument, NULL, 't' },
		{ "verbose", no_argument, NULL, 'V' },
		{ "help", no_argument, NULL, 'h' },
		{ "version", no_argument, NULL, 'v' },
		{ NULL, 0, NULL, '\0' }
	};
	const char *options_desc[]= {
		"Game model to play (defaults to \"linear\")",
		"Paddle profile to be used by all teams (defaults to \"flat\")",
		"Local hostname or IP address to bind to",
		"Local port to listen on",
		"Network device to bind the socket to",
		"Daemonize and return",
		"Number of threads to run simultaneously (-1 == auto)",
		"Maximum number of clients to serve simultaneously (-1 == unlimited)",
		"Maximum number of clients to allow to connect from one host (-1 == unlimited)",
		"Restrict traffic to specified amount of kBytes/sec (-1 == unlimited)",
		"Minimal interval between near-synchronous packet transfers (-1 == unrestricted)",
		"Timeout between status updates before client session is terminated, measured in msecs (-1 == no timeout)",
		"Log more messages to the console",
		"Show this help screen",
		"Show version information",
		NULL
	};
	const char *options_int= "pncilt";

	extern char *optarg;
	extern int optind;

	struct {
		char key;
		int *val;
	} map[]= {
		{ 'n', &conf->n_threads },
		{ 'c', &conf->max_connect },
		{ 'i', &conf->max_perip },
		{ 'l', &conf->max_payload },
		{ 't', &conf->client_timeout },
		{ 's', &conf->min_rttm_interval },
		{ '\x0', NULL }
	};

	int opt, idx, showhelp= 0, doexit= (-1);
	while (( opt= getopt_long(argc, argv, short_options, long_options, NULL) )) {
		if (opt == (-1)) break;

		switch(opt) {
		case 'm':
			if (gameplay_parse(optarg, &conf->gmode, NULL) != PONG_SUCCESS) {
				fprintf(stderr, "Model not implemented (%s).\n", optarg);
				doexit= 1;
			}
			break;
		case 'r':
			if (gameplay_parse(optarg, NULL, &conf->gpads) != PONG_SUCCESS) {
				fprintf(stderr, "Paddle profile not implemented (%s).\n", optarg);
				doexit= 1;
			}
			break;
		case 'a':
			conf->addr= inet_addr(optarg);
			if (conf->addr == INADDR_NONE) {
				struct hostent *hostptr= gethostbyname(optarg);
				if ((!hostptr)||((hostptr->h_addrtype & AF_INET) == 0)||(!hostptr->h_length)) {
					fprintf(stderr, "Invalid IP Address (%s).\n", optarg);
					doexit= 2;
				}
				else {
					assert(sizeof(conf->addr) == sizeof(hostptr->h_addr_list[0]));
					memcpy(&conf->addr, hostptr->h_addr_list[0], sizeof(conf->addr));
				}
				if (conf->addr == INADDR_NONE) {
					fprintf(stderr, "Cannot determine IP Address of host %s.\n", hostptr->h_addr_list[0]);
					doexit= 3;
				}
			}
			break;
		case 'p':
			conf->port= atoi(optarg);
			break;
		case 'd':
			strncpy(conf->device, optarg, IF_NAMESIZE);
			conf->device[IF_NAMESIZE -1]= '\x0';
			break;
		case 'v':
			printf("mmpongd %d.%d (r%s.c%d.l%d)\nReleased under the terms of the GPL (http://www.gnu.org/licenses/gpl.html).\n", VER_MAJ, VER_MIN, VER_SVN, (int)sizeof(struct gameplay), gameplay_getversion());
			doexit= 0;
			break;
		case 'b':
			conf->daemonize= 1;
			break;
		case 'h':
			showhelp= 1;
			break;
		case 'V':
			conf->verbose++;
			break;
		default:
			for (idx= 0; map[idx].val; idx++)
				if (map[idx].key == opt) {
					*map[idx].val= atoi(optarg);
					break;
				}
			if (!map[idx].val) {
				fprintf(stderr, "Parameter not recognized (%d).\n", opt);
				exit(-1);
			}
		}
	}

	if ((optind < argc) || (showhelp)) {
		printf("Usage: %s", argv[0]);
		for (idx= 0; long_options[idx].name; idx++) {
			printf(" [--%s", long_options[idx].name);
			if (long_options[idx].has_arg == required_argument)
				printf("=%s", (strchr(options_int, long_options[idx].val))? "%d":"%s");
			printf("]");
		}
		printf("\n");
		for (idx= 0; options_desc[idx] && long_options[idx].name; idx++) {
			printf("\t-%c%s, --%s%s :\t%s\n",
				long_options[idx].val,
				(long_options[idx].has_arg != required_argument)? "":
					((strchr(options_int, long_options[idx].val))?" <int>":" <string>"),
				long_options[idx].name,
				(long_options[idx].has_arg != required_argument)? "":
					((strchr(options_int, long_options[idx].val))?"=<int>":" <string>"),
				options_desc[idx]);
		}
		doexit= (optind < argc)*99;
	}

	struct {
		char *name;
		int *limit;
	} limits[]= {
		{ "Number of threads", &conf->n_threads },
		{ "Connection limit", &conf->max_connect },
		{ "Connections per IP", &conf->max_perip },
		{ "Payload limit", &conf->max_payload },
		{ "Client timeout", &conf->client_timeout },
		{ "RTTM Interval", &conf->min_rttm_interval },
		{ NULL, NULL }
	};

	for (idx= 0; limits[idx].limit; idx++)
		if ((*limits[idx].limit< (-1)) || (*limits[idx].limit == 0)) {
			fprintf(stderr, "%s requested is invalid (%d). Use numbers >0, or (-1) for unlimited.\n",
				limits[idx].name, *limits[idx].limit);
			doexit= idx+1;
		}
	if (doexit != (-1)) exit(doexit);
}



static int setup_listening_socket(conf)
const struct config *conf;
{
	int sock;
	if ((sock= socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == (-1)) { 	// let's start out with TCP
		perror("socket()");
		return (-1);
	}
#ifndef __APPLE__
	if ((strlen(conf->device)) && (setsockopt(sock, SOL_SOCKET, SO_BINDTODEVICE, conf->device, strlen(conf->device) +1) == (-1))) {
		close(sock);
		perror("setsockopt()");
		return (-2);
	}
#endif

	int retcode= -2, keepalive= 1, reuse= 1, rcvlow= 2;
	int maxseg= sizeof(struct netmessage);
	struct timeval timeout;
	memset(&timeout, 0, sizeof(timeout));
	timeout.tv_sec= conf->client_timeout / 1000;
	timeout.tv_usec= ((suseconds_t)(conf->client_timeout % 1000L)) * 1000L;
	if ( (--retcode, setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &keepalive, sizeof(keepalive)) == (-1)) ||
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) == (-1)) ||
		 // we expect to receive short integers
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_RCVLOWAT, &rcvlow, sizeof(rcvlow)) == (-1)) ||
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout)) == (-1)) ||
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout)) == (-1)) ||
		 // ideally, we don't send or receive more data than a netmessage contains
		 (--retcode, setsockopt(sock, IPPROTO_TCP, TCP_MAXSEG, &maxseg, sizeof(maxseg)) == (-1)) 	)
	{
		close(sock);
		perror("setsockopt()");
		return retcode;
	}

	// bind to port
	struct sockaddr_in addr;
	memset(&addr, 0, sizeof(addr));
	addr.sin_family= AF_INET;
	addr.sin_addr.s_addr= conf->addr;
	addr.sin_port= htons(conf->port);
	if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) == (-1)) {
		close(sock);
		perror("bind()");
		return (-9);
	}

	return sock;
}



// set up socket for close-to-realtime data transfers
static int setup_peer_socket(sock, conf)
int sock;
const struct config *conf;
{
	int retcode= 0, yes= 1, rcvlow= 2;
	int maxseg= sizeof(struct netmessage);
#ifdef __APPLE__
	int tos= IPTOS_LOWDELAY;
#else
	uint8_t tos= IPTOS_LOWDELAY;
#endif
	struct timeval timeout;
	memset(&timeout, 0, sizeof(timeout));
	timeout.tv_sec= conf->client_timeout / 1000;
	timeout.tv_usec= ((suseconds_t)(conf->client_timeout % 1000L)) * 1000L;
	if ( (--retcode, fcntl(sock, F_SETFL, O_NONBLOCK) == (-1)) ||
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &yes, sizeof(yes)) == (-1)) ||
		 // we expect to receive short integers
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_RCVLOWAT, &rcvlow, sizeof(rcvlow)) == (-1)) ||
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout)) == (-1)) ||
		 (--retcode, setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout)) == (-1)) ||
		 (--retcode, setsockopt(sock, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)) == (-1)) ||
		 // ideally, we don't send or receive more data than a netmessage contains
		 (--retcode, setsockopt(sock, IPPROTO_TCP, TCP_MAXSEG, &maxseg, sizeof(maxseg)) == (-1)) ||
		 (--retcode, setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, &yes, sizeof(yes)) == (-1)) 
//		 (--retcode, setsockopt(sock, IPPROTO_TCP, TCP_CORK, &yes, sizeof(yes)) == (-1)) 	// this last one is a possible portability issue (remove if required)
		 )
	{
		perror((retcode == (-1))?"fcntl()":"setsockopt()");
		return retcode;
	}
	return 0;
}



static int drop_privileges(void)
{
	if (geteuid()) return 1; 	// not super-user

	const struct passwd *pwd= getpwnam("nobody");
	if (!pwd) {
		perror("getpwnam()");
		return (-1);
	}

#if defined __USE_BSD || (defined __USE_XOPEN && !defined __USE_UNIX98)
	const char *shell;
	while (( shell= getusershell() ))
		if (!strcmp(pwd->pw_shell, shell) && (strcmp(pwd->pw_shell, "/sbin/nologin"))) break;
	endusershell();
	if (shell) {
		fprintf(stderr, "User nobody is not safe\n");
		return (-2);
	}
#endif

	if (setgid(pwd->pw_gid)) {
		perror("setgid()");
		return (-3);
	}
	if (setuid(pwd->pw_uid)) {
		perror("setuid()");
		return (-4);
	}
	return 0;
}



static int peer_cnt_increase(cnt)
unsigned *cnt;
{
	if (conf.verbose >2) printf("Peer count == %u (@%p).\n", *cnt, cnt);
	if (*cnt >= conf.max_perip)
		return 1; 	// limit reached
	(*cnt)++;
	return 0; 	// success
}



static int collect_worker_threads(threads)
pthread_t *threads;
{
	signal_exiting= 1; 	// signal to exit
	int threadval, retval= 0;
	for (int idx= 0; threads[idx]; idx++)
		if (pthread_join(threads[idx], (void *)&threadval)) {
			fprintf(stderr, "Error joining worker thread.\n");
			retval= (-1);
		}
	return retval;
}



// atexit() handler
static void free_globals(void)
{
	if (n_connects) free(n_connects);
	if (insert_socket) free(insert_socket);
	if (insert_peer) free(insert_peer);
	if (isync) free(isync);
	if (sums) free(sums);
	if (variances) free(variances);
	n_connects = NULL; 
	sums= variances= NULL;
	insert_socket= NULL;
	insert_peer= NULL;
	isync= NULL;
}



// atexit() handler
static void destroy_mutexes(void)
{
	for (int idx= 0; idx< conf.n_threads; idx++)
		if (pthread_mutex_destroy(isync + idx))
			fprintf(stderr, "Error destroying mutex object.\n");
}



// atexit() handler
static void destroy_rwlocks(void)
{
	pthread_rwlock_t *locks[]= { &gsync, &statsync, NULL };
	for (int idx= 0; locks[idx]; idx++)
		if (pthread_rwlock_destroy(locks[idx]))
			fprintf(stderr, "Error destroying rwlock object.\n");
}



// return value is of dimension microseconds
static unsigned calc_bcast(n_threads, cur_connects, max_payload)
int n_threads, max_payload; 	// measured in kBytes/sec
unsigned cur_connects;
{
	if ((max_payload <= 0) || (n_threads <= 0)) return BCAST_INTERVAL_MIN;
	unsigned interval= (unsigned) (((unsigned long)cur_connects) * n_threads * (NETMESSAGE_MAXLEN /2) *1000 *1000 /max_payload /1024);
	if (interval < BCAST_INTERVAL_MIN) interval= BCAST_INTERVAL_MIN;
	return interval;
}



static int update_players(igame, n_threads, n_connects, sums, variances)
struct gameplay *igame;
int n_threads;
unsigned *n_connects;
float *sums, *variances;
{
	float mean, var;
	unsigned tcons;
	for (int team= 0; team< n_teams; team++) {
		mean= var= 0;
		tcons= 0;
		for (int idx= 0; idx< n_threads; idx++) {
			tcons+= n_connects[idx * n_teams + team];
			mean+= sums[idx * n_teams + team];
			var+= variances[idx * n_teams + team];
		}
		if (tcons >0) {
			mean/= tcons;
			var/= tcons;
		}
		if ((mean < 0.f)||(mean > 1.f)||(var < 0.f)) return (-1);

		// I'm not sure baby-sitting our user-base is the right thing to do, but here we go:
		if (mean > 1.f - igame->pad[team].size / 2)
			mean = 1.f - igame->pad[team].size / 2;
		if (mean < igame->pad[team].size / 2)
			mean = igame->pad[team].size / 2;

		igame->pad[team].mean= mean;
		igame->pad[team].var= var;
		igame->pad_attr[team].peers= tcons;
	}
	return 0;
}

