///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file ElidedTextLabel.h
 * \brief Contains the definition of the Core::ElidedTextLabel class.
 */

#ifndef __OVITO_ELIDED_TEXT_LABEL_H
#define __OVITO_ELIDED_TEXT_LABEL_H

#include <core/Core.h>

namespace Core {

/**
 * \brief A QLabel-like widgets that display a line of text, which is shortened if necessary to fit the available space.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ElidedTextLabel : public QLabel
{
	Q_OBJECT
	
public:
	
	/// \brief Constructs an empty label.
	/// \param parent The parent widget for the new widget.
	/// \param f Passed to the QFrame constructor.
	ElidedTextLabel(QWidget* parent = NULL, Qt::WindowFlags f = 0) : QLabel(parent, f) {}

	/// \brief Constructs a label with text.
	/// \param text The text string to display.
	/// \param parent The parent widget for the new widget.
	/// \param f Passed to the QFrame constructor.
	ElidedTextLabel(const QString& string, QWidget* parent = NULL, Qt::WindowFlags f = 0) : QLabel(string, parent, f) {}
	
protected:

	// Returns the rect that is available for us to draw the document
	QRect documentRect() const {
		QRect cr = contentsRect();
		int m = margin();
		cr.adjust(m, m, -m, -m);
		m = indent();
	    if(m < 0 && frameWidth()) // no indent, but we do have a frame
	    	m = fontMetrics().width(QLatin1Char('x')) / 2 - margin();
	    int align = QStyle::visualAlignment(layoutDirection(), alignment());
	    if(m > 0) {
			if (align & Qt::AlignLeft)
	            cr.setLeft(cr.left() + m);
	        if (align & Qt::AlignRight)
	            cr.setRight(cr.right() - m);
	        if (align & Qt::AlignTop)
	            cr.setTop(cr.top() + m);
	        if (align & Qt::AlignBottom)
	            cr.setBottom(cr.bottom() - m);
	    }
	    return cr;
	}

	void paintEvent(QPaintEvent *) {
	    QStyle *style = QWidget::style();
	    QPainter painter(this);
	    QRect cr = documentRect();
	    int flags = QStyle::visualAlignment(layoutDirection(), alignment());
	    QString elidedText = painter.fontMetrics().elidedText(text(), Qt::ElideLeft, cr.width(), flags);
	    style->drawItemText(&painter, cr, flags, palette(), isEnabled(), elidedText, foregroundRole());
	}
};

};

#endif // __OVITO_ELIDED_TEXT_LABEL_H
