///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file AnimManager.h 
 * \brief Contains the definition of the Core::AnimManager class. 
 */
 
#ifndef __OVITO_ANIM_MANAGER_H
#define __OVITO_ANIM_MANAGER_H

#include <core/Core.h>
#include "AnimationSettings.h"

namespace Core {
	
class SpinnerWidget;		// defined in SpinnerWidget.h 

/// \def ANIM_MANAGER
/// \brief The predefined instance of the Core::AnimManager class.
/// 
/// Always use this macro to access the Core::AnimManager class instance.
#define ANIM_MANAGER		(*AnimManager::getSingletonInstance())

/**
 * \brief Manages the global animation settings.
 * 
 * This is a singleton class with only one predefined instance of this class. 
 * You can access the instance of this class using the ANIM_MANAGER macro.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT AnimManager : public QObject
{
	Q_OBJECT
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the AnimManager singleton class.
	/// \note You should use the ANIM_MANAGER macro to access the AnimManager instance instead
	///       of this method.
	inline static AnimManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "AnimManager::getSingletonInstance", "AnimManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns whether animation is active and animation keys should be automatically generated. 
	/// \return \c true if animating is currently turned on and not suspended; \c false otherwise.
	/// 
	/// When animating is turned on, controllers should automatically set keys when their values is changed.
	///
	/// \sa animationModeEnabled()
	bool isAnimating() const { return _animationMode && _animSuspendCount == 0; }
	
	/// \brief Returns whether animation mode has been activated.
	/// \return \c true if the automatic generation of keys has been enabled. 
	/// \note The automatic generation of animation keys may be suspended by a call to suspendAnim().
	///       This overrides the animation mode. Controllers should use isAnimating() to check whether
	///       have to generate a key on value change.
	///
	/// \sa isAnimating()
	/// \sa setAnimationMode()
	bool animationMode() const { return _animationMode; }

	/// \brief Suspends the animation mode temporaly.
	/// 
	/// Automatic generation of animation keys is suspended by this method until a call to resumeAnim().
	/// If suspendAnim() is called multiple times then resumeAnim() must be called the same number of
	/// times until animation mode is enabled again.
	///
	/// It is recommended to use the AnimationSuspender helper class to suspend animation mode because
	/// this is more exception save than the suspendAnim()/resumeAnim() combination.
	///
	/// \sa resumeAnim()
	/// \sa AnimationSuspender
	void suspendAnim() { _animSuspendCount++; }

	/// \brief Resumes the automatic generation of animation keys.
	/// 
	/// This re-enables animation mode after it had 
	/// been suspended by a call to suspendAnim().
	///
	/// \sa suspendAnim()
	/// \sa AnimationSuspender
	void resumeAnim() { 
		OVITO_ASSERT_MSG(_animSuspendCount > 0, "AnimManager::resumeAnim()", "resumeAnim() has been called more often than suspendAnim()."); 
		_animSuspendCount--; 
	}

	/// \brief Gets the current animation time.
	/// \return The current time.
	/// 
	/// The state of the scene at this time is shown in the viewports.
	/// \sa setTime()
	/// \sa AnimationSettings::time()
	TimeTicks time() const { 
		if(!_settings) return 0; 
		return _settings->time(); 
	}

	/// \brief Sets the current animation time.
	/// \param time The new animation time.
	///
	/// The state of the scene at the given time will be shown in the viewports.
	/// \undoable
	/// \sa time()
	/// \sa AnimationSettings::setTime()
	void setTime(TimeTicks time) { 
		if(_settings) _settings->setTime(time); 
	}

	/// \brief Gets the animation interval.
	/// \return The time interval of the animation.
	/// \sa setAnimationInterval() 
	const TimeInterval& animationInterval() const { 
		if(!_settings) return nullInterval; 
		return _settings->animationInterval(); 
	}

	/// \brief Sets the animation interval.
	/// \param interval The new animation interval for the current scene.
	/// \undoable
	/// \sa animationInterval()
	void setAnimationInterval(const TimeInterval& interval) { 
		if(_settings) _settings->setAnimationInterval(interval); 
	}
	
	/// \brief Returns the number of frames per second.
	/// \return The number of frames per second.
	/// 
	/// This setting controls the playback speed of the animation.
	///
	/// \sa setFramesPerSecond()
	int framesPerSecond() const { 
		if(!_settings) return 1;
		return _settings->framesPerSecond(); 
	}

	/// \brief Sets the number of frames per second.
	/// \param fps The number of frames per second. Please note that not all
	///            values are allowed here because time is measured in integer ticks units.
	/// \undoable
	/// 
	/// This setting controls the playback speed of the animation.
	///
	/// \sa setFramesPerSecond()
	/// \sa framesPerSecond()
	/// \sa setTicksPerFrame()
	void setFramesPerSecond(int fps) { 
		if(_settings) _settings->setFramesPerSecond(fps); 
	}

	/// \brief Returns the number of time ticks per frame.
	/// \return The number of time ticks per animation frame. One tick is 1/4800 of a second.
	/// 
	/// This setting controls the playback speed of the animation.
	///
	/// \sa setTicksPerFrame()
	int ticksPerFrame() const { 
		if(!_settings) return TICKS_PER_SECOND;
		return _settings->ticksPerFrame(); 
	}

	/// \brief Sets the number of time ticks per frame.
	/// \param ticksPerFrame The number of time tick units per animation frame.
	///                      Thsi must be a positive value.
	/// \undoable
	/// 
	/// This setting controls the playback speed of the animation.
	/// \sa ticksPerFrame()
	void setTicksPerFrame(int ticksPerFrame) {
		if(_settings) _settings->setTicksPerFrame(ticksPerFrame);
	}

	/// \brief Returns the playback speed factor that is used for animation playback in the viewports.
	/// \return The playback speed factor. A value greater than 1 means that the animation is played at a speed higher
	///         than realtime whereas a value smaller than -1 means that the animation is played at a speed lower than realtime.
	/// \sa setPlaybackSpeed()
	int playbackSpeed() const { 
		if(!_settings) return 1;
		return _settings->playbackSpeed(); 
	}

	/// \brief Sets the playback speed factor that is used for animation playback in the viewport.
	/// \param factor A value greater than 1 means that the animation is played at a speed higher
	///               than realtime. A value smaller than -1 that the animation is played at a speed lower than realtime.
	/// \undoable
	/// \sa playbackSpeed()
	void setPlaybackSpeed(int factor) {
		if(_settings) _settings->setPlaybackSpeed(factor);
	}
	
	/// \brief Converts an animation frame number to a time value.
	/// \param frame A frame number starting at 0.
	/// \return The animation frame at which the animation frame begins.
	/// \sa timeToFrame() 	
	TimeTicks frameToTime(int frame) const { return Core::frameToTime(frame, ticksPerFrame()); }
	
	/// \brief Converts a time value to an animation frame number.
	/// \param time A time in ticks units.
	/// \return The animation frame that corresponds to the given time.
	/// \sa frameToTime()	
	int timeToFrame(TimeTicks time) const { return Core::timeToFrame(time, ticksPerFrame()); }

	/// \brief Converts a time value to its string representation.
	/// \param time Some time value.
	/// \return A human-readable representation of the time value.
	/// \sa stringToTime()
	QString timeToString(TimeTicks time);

	/// \brief Converts a string to a time value.
	/// \param stringValue The  human-readable representation of a time value.
	/// \return The parsed time value.
	/// \throw Exception when a parsing error occurs.
	/// \sa timeToString()
	TimeTicks stringToTime(const QString& stringValue);
	
	/// \brief Creates a spinner widget that lets the user control the current animation time.
	/// \return A spinner control that can be inserted into a window. When the user changes the value of the
	///         spinner control than the current animation time of the scene will be changed accordingly.
	/// \sa AnimationTimeSpinner
	SpinnerWidget* createCurrentTimeSpinner();

public Q_SLOTS:

	/// \brief Enables or disables animation mode.
	/// \param on Controls the state of the animation mode.
	/// \note The automatic generation of animation keys may be suspended by a call to suspendAnim().
	///       This overrides the animation mode. Controllers should use isAnimating() to check whether
	///       have to generate a key on value change.
	/// \sa animationMode() 
	void setAnimationMode(bool on);

	/// \brief Resets the animation manager.
	/// 
	/// This turns animation mode off.
	void reset();

Q_SIGNALS:

	/// This signal is emmited by the AnimManager when the current animation time has changed.
	void timeChanged(TimeTicks newTime);

	/// This signal is emmited by the AnimManager when the animation interval has changed.
	void intervalChanged(TimeInterval newAnimationInterval);

	/// This signal is emmited by the AnimManager when the animation speed has changed.
	void speedChanged(int ticksPerFrame);

	/// This signal is emmited by the AnimManager when the time to string conversion format has changed.
	void timeFormatChanged();
	
	/// This signal is emitted when the animation mode has been activated or deactivated.
	void animationModeChanged(bool active);
	
private:

	/// Counts the number of times the animation modes has been suspended.
	int _animSuspendCount;

	/// Points to the animation settings of the current DataSet.
	AnimationSettings::SmartPtr _settings;
	
	/// This is just a dummy interval.
	TimeInterval nullInterval;	
	
	/// Indicates whether animation mode is active.
	bool _animationMode;

	/////////////////////////// Maintainance ////////////////////////////////
    
	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	AnimManager();

	/// Initializes the AnimManager.
	static void initialize() { 
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new AnimManager();
	}
	
	/// AnimManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}
	
	/// The singleton instance of this class.
	static AnimManager* _singletonInstance;
	
	friend class ApplicationManager;
	friend class DataSet;
	friend class DataSetManager;
};

/**
 * \brief A small helper object that suspends generation of animation keys while it
 *        exists. It can be used to make your code exception-safe.
 * 
 * The constructor of this class calls AnimManager::suspendAnim() and
 * the destructor calls AnimManager::resumeAnim().
 * 
 * Just create an instance of this class on the stack to suspend the automatic generation of animation keys on the undo stack
 * during the lifetime of the class instance.
 * 
 * \author Alexander Stukowski
 * \sa AnimManager
 */
struct AnimationSuspender {
	AnimationSuspender() { ANIM_MANAGER.suspendAnim(); }
	~AnimationSuspender() { ANIM_MANAGER.resumeAnim(); }
};

};

#endif // __OVITO_ANIM_MANAGER_H
